const express = require('express');
const { mongoose } = require('../dbConnection'); // Use shared connection
const cors = require('cors');
const nodemailer = require('nodemailer');
const http = require('http');

const app = express();

// Middleware
app.use(cors());
app.use(express.json());

// MongoDB configuration
const MONGO_HOST = process.env.MONGO_HOST || 'api.titandrillingzm.com';
const MONGO_PORT = process.env.MONGO_PORT || '27017';
const MONGO_USER = process.env.MONGO_USER || 'titandrillingadminuser';
const MONGO_PASSWORD = process.env.MONGO_PASSWORD || 'Titandrilling_@2024';
const MONGO_DB = process.env.MONGO_DB || 'titan_drilling';

// Using shared MongoDB connection from dbConnection.js
console.log('✅ [Logbook Notifications] Using shared MongoDB connection');

// GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: String,
  name: String,
  email: String,
  role: String,
  countries: [String],
  country: String,
  projects: [String],
  project: String,
  fcmTokens: [{
    token: String,
    device: String,
    updatedAt: Date
  }],
  fcmToken: String,
}, { strict: false, collection: 'GlobalUsers' });

// Check if model exists to prevent OverwriteModelError
let GlobalUser;
try {
  GlobalUser = mongoose.model('GlobalUser');
  console.log('✅ [LogbookNotifications] Using existing GlobalUser model');
} catch (error) {
  GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
  console.log('✅ [LogbookNotifications] Created new GlobalUser model');
}

// Email transporter configuration
const emailTransporter = nodemailer.createTransport({
  service: 'gmail',
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'sonia.owninfotech@gmail.com',
    pass: 'wvpv dmsv ytxa vdgd',
  },

});

// Helper function to send push notifications
const sendPushNotifications = async (logbookData, createdByName, recipients) => {
  try {
    console.log('📱 Sending push notifications for logbook entry...');
    
    const notificationData = {
      userIds: recipients.map(r => r._id),
      title: '📔 New Logbook Entry',
      body: `${logbookData.entryNumber}\nEquipment: ${logbookData.selectedEquipment || 'N/A'}\nStatus: ${logbookData.equipmentStatus || 'N/A'}\nProject: ${logbookData.project || 'N/A'}\n\nLogged by: ${createdByName}`,
      data: {
        type: 'logbook_entry_created',
        action: 'view_logbook',
        logbookId: logbookData._id || logbookData.id,
        entryNumber: logbookData.entryNumber,
        selectedEquipment: logbookData.selectedEquipment || '',
        equipmentStatus: logbookData.equipmentStatus || '',
        project: logbookData.project || '',
        country: logbookData.country || '',
        creator: createdByName,
        timestamp: new Date().toISOString(),
      },
      // iOS specific settings
      apns: {
        payload: {
          aps: {
            alert: {
              title: '📔 New Logbook Entry',
              body: `${logbookData.entryNumber}\nEquipment: ${logbookData.selectedEquipment || 'N/A'}\n\nLogged by: ${createdByName}`
            },
            sound: 'default',
            badge: 1,
            'content-available': 1,
          }
        }
      },
      // Android specific settings
      android: {
        priority: 'high',
        notification: {
          sound: 'default',
          channelId: 'logbook_notifications',
        }
      }
    };
    
    // Send push via PushNotificationHandler (port 5016)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify(notificationData);
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5016,
        path: '/push-notifications/send-to-users',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from push service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Push notifications sent to ${result.successCount || 0} device(s)`);
    return result;
  } catch (error) {
    console.error('❌ Error sending push notifications:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to save notifications to MongoDB
const saveNotificationsToMongoDB = async (logbookData, createdByName, recipients) => {
  try {
    console.log('💾 Saving logbook notifications to MongoDB...');
    
    const notifications = recipients.map(recipient => ({
      notificationId: `logbook_${Date.now()}_${Math.random().toString(36).substr(2, 9)}_${recipient._id}`,
      userId: recipient._id,
      title: '📔 New Logbook Entry',
      body: `${logbookData.entryNumber}\nEquipment: ${logbookData.selectedEquipment || 'N/A'}\nStatus: ${logbookData.equipmentStatus || 'N/A'}\nProject: ${logbookData.project || 'N/A'}\n\nLogged by: ${createdByName}`,
      type: 'logbook_entry_created',
      timestamp: Date.now(),
      isRead: false,
      data: {
        type: 'logbook_entry_created',
        action: 'view_logbook',
        logbookId: logbookData._id || logbookData.id,
        entryNumber: logbookData.entryNumber,
        selectedEquipment: logbookData.selectedEquipment || '',
        equipmentStatus: logbookData.equipmentStatus || '',
        workType: logbookData.workType || '',
        operatorName: logbookData.operatorName || '',
        project: logbookData.project || '',
        country: logbookData.country || '',
        creator: createdByName,
        timestamp: new Date().toISOString(),
      }
    }));
    
    // Save to MongoDB via Notifications API (port 5017)
    const result = await new Promise((resolve, reject) => {
      const postData = JSON.stringify({ notifications });
      const options = {
        hostname: 'api.titandrillingzm.com',
        port: 5017,
        path: '/api/notifications/bulk',
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Content-Length': Buffer.byteLength(postData),
        },
      };
      
      const req = http.request(options, (res) => {
        let data = '';
        res.on('data', (chunk) => { data += chunk; });
        res.on('end', () => {
          try {
            const response = JSON.parse(data);
            resolve(response);
          } catch (e) {
            reject(new Error('Invalid JSON response from notifications service'));
          }
        });
      });
      
      req.on('error', (error) => {
        reject(error);
      });
      
      req.write(postData);
      req.end();
    });
    
    console.log(`✅ Saved ${result.count || notifications.length} notification(s) to MongoDB`);
    return result;
  } catch (error) {
    console.error('❌ Error saving notifications to MongoDB:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send email to creator
const sendCreatorEmail = async (logbookData, createdByName, creatorEmail) => {
  try {
    if (!creatorEmail) {
      console.log('⚠️ No email address for creator, skipping creator email');
      return { success: false, error: 'No email address' };
    }
    
    console.log(`📧 Sending confirmation email to creator: ${createdByName} (${creatorEmail})`);
    
    const emailSubject = `Logbook Entry Created: ${logbookData.entryNumber}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>Logbook Entry Created - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">✅ Logbook Entry Created Successfully</h2>

      <p style="line-height: 1.6; margin-bottom: 10px;">Dear ${createdByName},</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">Your logbook entry has been successfully created and saved in the system.</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Entry Number:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.entryNumber}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Equipment:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.selectedEquipment || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Equipment Status:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.equipmentStatus || 'N/A'}</td>
          </tr>
          ${logbookData.operatorName ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Operator:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.operatorName}</td>
          </tr>` : ''}
          ${logbookData.workType ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Work Type:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.workType}</td>
          </tr>` : ''}
          ${logbookData.workLocation ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Work Location:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.workLocation}</td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Start/End Reading:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.startReading} - ${logbookData.endReading}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Country:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.country || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Project:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.project || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Logged By:</th>
            <td style="padding: 10px; text-align: left;">${createdByName}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        The logbook entry has been sent to all relevant supervisors and managers.
      </p>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    const mailOptions = {
      from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
      to: creatorEmail,
      subject: emailSubject,
      html: emailHtml,
    };
    
    await emailTransporter.sendMail(mailOptions);
    
    console.log(`✅ Confirmation email sent to creator ${createdByName} (${creatorEmail})`);
    return { success: true };
  } catch (error) {
    console.error('❌ Error sending creator email:', error);
    return { success: false, error: error.message };
  }
};

// Helper function to send emails to recipients
const sendEmails = async (logbookData, createdByName, recipients) => {
  try {
    console.log('📧 Sending email notifications for logbook entry...');
    
    let successCount = 0;
    let failCount = 0;
    
    // Email template
    const emailSubject = `New Logbook Entry: ${logbookData.entryNumber}`;
    const emailHtml = `
<!DOCTYPE html>
<html>
<head>
  <meta charset="UTF-8">
  <title>New Logbook Entry - Titan Drilling</title>
</head>
<body style="font-family: Arial, sans-serif; background-color: #f5f6f8; margin: 0; padding: 0;">
  <div style="max-width: 700px; margin: 40px auto; background: #ffffff; box-shadow: 0 3px 8px rgba(0,0,0,0.05); overflow: hidden; border: 1px solid #e5e5e5;">
    
    <!-- Header -->
    <div style="background-color: #004a93; padding: 20px; text-align: center; border-bottom: 3px solid #003366;">
      <img src="https://titandrillingzm.com/images/Sidebar-logo.png" alt="Titan Drilling Logo" style="width: 120px;">
    </div>

    <!-- Content -->
    <div style="padding: 30px; color: #333;">
      <h2 style="color: #003366; margin-bottom: 20px; font-size: 22px;">📔 New Logbook Entry</h2>

      <p style="line-height: 1.6; margin-bottom: 10px;">Dear Recipient,</p>
      <p style="line-height: 1.6; margin-bottom: 20px;">A new logbook entry has been recorded in the Titan Drilling system.</p>
      
      <!-- Table -->
      <div style="margin-top: 20px; background: #f9fafc; border: 1px solid #e5e5e5; border-radius: 8px; padding: 15px;">
        <table style="width: 100%; border-collapse: collapse; margin-top: 10px;">
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Entry Number:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.entryNumber}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Equipment:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.selectedEquipment || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Equipment Status:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.equipmentStatus || 'N/A'}</td>
          </tr>
          ${logbookData.operatorName ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Operator:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.operatorName}</td>
          </tr>` : ''}
          ${logbookData.workType ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Work Type:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.workType}</td>
          </tr>` : ''}
          ${logbookData.workLocation ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Work Location:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.workLocation}</td>
          </tr>` : ''}
          ${logbookData.workDescription ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Work Description:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.workDescription}</td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Start/End Reading:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.startReading} - ${logbookData.endReading}</td>
          </tr>
          ${logbookData.fuelConsumption ? `
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Fuel Consumption:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.fuelConsumption} ${logbookData.fuelUnit || ''}</td>
          </tr>` : ''}
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Country:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.country || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Project:</th>
            <td style="padding: 10px; text-align: left;">${logbookData.project || 'N/A'}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Logged By:</th>
            <td style="padding: 10px; text-align: left;">${createdByName}</td>
          </tr>
          <tr>
            <th style="color: #003366; width: 35%; text-align: left; padding: 10px;">Date:</th>
            <td style="padding: 10px; text-align: left;">${new Date().toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric', hour: '2-digit', minute: '2-digit' })}</td>
          </tr>
        </table>
      </div>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        Please review the logbook entry in the Titan Drilling application.
      </p>

      <p style="line-height: 1.6; margin-top: 20px; margin-bottom: 10px;">
        If you need assistance, please contact our support team at 
        <a href="mailto:support@titandrillingzm.com" style="text-decoration: none; color: #003366; font-weight: bold;">support@titandrillingzm.com</a>.
      </p>

      <p style="line-height: 1.6; margin-bottom: 10px;">Best regards,</p>
      <p style="line-height: 1.6; margin-bottom: 10px;"><strong>Titan Drilling Team</strong></p>
    </div>

    <!-- Footer -->
    <div style="background-color: #f4f4f4; text-align: center; padding: 20px; font-size: 13px; color: #555;">
      © ${new Date().getFullYear()} Titan Drilling. All Rights Reserved.
    </div>
  </div>
</body>
</html>
    `;
    
    // Send email to each recipient
    for (const recipient of recipients) {
      if (recipient.email) {
        try {
          const mailOptions = {
            from: '"Titan Drilling" <mandeep.owninfotech@gmail.com>',
            to: recipient.email,
            subject: emailSubject,
            html: emailHtml,
          };
          
          await emailTransporter.sendMail(mailOptions);
          
          console.log(`✅ Email sent to ${recipient.name} (${recipient.email})`);
          successCount++;
        } catch (emailError) {
          console.error(`❌ Failed to send email to ${recipient.name}:`, emailError.message);
          failCount++;
        }
      }
    }
    
    console.log(`📧 Email summary: ${successCount} sent, ${failCount} failed`);
    return { success: true, sent: successCount, failed: failCount, total: recipients.length };
  } catch (error) {
    console.error('❌ Error sending emails:', error);
    return { success: false, error: error.message };
  }
};

/**
 * POST /api/logbook-notifications/new-entry
 * Handle notifications when a new logbook entry is created
 */
app.post('/api/logbook-notifications/new-entry', async (req, res) => {
  try {
    const { logbookData, createdByName, creatorEmail } = req.body;
    
    if (!logbookData || !logbookData.entryNumber) {
      return res.status(400).json({
        success: false,
        error: 'Logbook data is required'
      });
    }
    
    console.log(`\n🔔 Processing notifications for logbook entry: ${logbookData.entryNumber}`);
    console.log(`📋 Country: ${logbookData.country}, Project: ${logbookData.project}`);
    console.log(`👤 Creator: ${createdByName}`);
    console.log(`🚜 Equipment: ${logbookData.selectedEquipment}`);
    
    // Manager roles to notify
    const managerRoles = [
      'HrManager',
      'Workshop Manager',
      'Maintenance Manager',
      'FinanceManager',
      'Project Manager',
      'Site Manager'
    ];
    
    // Get all ProAdmins (no filtering)
    const proAdmins = await GlobalUser.find({
      role: 'ProAdmin'
    }).select('_id name email fcmToken fcmTokens');
    
    console.log(`📋 Found ${proAdmins.length} ProAdmin(s)`);
    
    // Get Supervisors filtered by country AND project
    const supervisors = await GlobalUser.find({
      role: 'Supervisor',
      countries: logbookData.country,
      projects: logbookData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${supervisors.length} Supervisor(s) for ${logbookData.country} - ${logbookData.project}`);
    
    // Get Managers filtered by country AND project
    const managers = await GlobalUser.find({
      role: { $in: managerRoles },
      countries: logbookData.country,
      projects: logbookData.project
    }).select('_id name email fcmToken fcmTokens role countries projects');
    
    console.log(`📋 Found ${managers.length} Manager(s) for ${logbookData.country} - ${logbookData.project}`);
    
    // Combine all recipients
    const allRecipients = [...proAdmins, ...supervisors, ...managers];
    
    // Remove duplicates based on _id
    const uniqueRecipients = allRecipients.filter((recipient, index, self) =>
      index === self.findIndex((r) => r._id === recipient._id)
    );
    
    console.log(`📋 Total unique recipients: ${uniqueRecipients.length}`);
    
    if (uniqueRecipients.length === 0) {
      console.log('⚠️ No recipients found');
      return res.json({
        success: true,
        message: 'No recipients to notify',
        pushSent: 0,
        emailsSent: 0,
        notificationsSaved: 0,
      });
    }
    
    // Execute all notification tasks in parallel
    const [mongoResult, pushResult, emailResult, creatorEmailResult] = await Promise.all([
      saveNotificationsToMongoDB(logbookData, createdByName, uniqueRecipients),
      sendPushNotifications(logbookData, createdByName, uniqueRecipients),
      sendEmails(logbookData, createdByName, uniqueRecipients),
      sendCreatorEmail(logbookData, createdByName, creatorEmail),
    ]);
    
    // Log results
    console.log('\n📊 Notification Results:');
    console.log(`  💾 MongoDB: ${mongoResult.success ? '✅' : '❌'} (${mongoResult.count || 0} notifications)`);
    console.log(`  📱 Push: ${pushResult.success ? '✅' : '❌'} (${pushResult.successCount || 0} devices)`);
    console.log(`  📧 Emails to Recipients: ${emailResult.success ? '✅' : '❌'} (${emailResult.sent || 0} sent)`);
    console.log(`  📧 Creator Email: ${creatorEmailResult.success ? '✅' : '❌'}\n`);
    
    res.json({
      success: true,
      message: `Notifications sent for logbook entry: ${logbookData.entryNumber}`,
      results: {
        mongodb: {
          success: mongoResult.success,
          saved: mongoResult.count || 0,
          error: mongoResult.error || null,
        },
        push: {
          success: pushResult.success,
          sent: pushResult.successCount || 0,
          failed: pushResult.failureCount || 0,
          error: pushResult.error || null,
        },
        emails: {
          success: emailResult.success,
          sent: emailResult.sent || 0,
          failed: emailResult.failed || 0,
          total: emailResult.total || 0,
          error: emailResult.error || null,
        },
        creatorEmail: {
          success: creatorEmailResult.success,
          error: creatorEmailResult.error || null,
        },
      },
      recipients: {
        proAdmins: proAdmins.length,
        supervisors: supervisors.length,
        managers: managers.length,
        total: uniqueRecipients.length,
      }
    });
  } catch (error) {
    console.error('❌ Error processing logbook notifications:', error);
    res.status(500).json({
      success: false,
      error: error.message,
    });
  }
});

// Health check
app.get('/health', (req, res) => {
  res.json({
    status: 'OK',
    service: 'Logbook Notifications Service',
    mongodb: mongoose.connection.readyState === 1 ? 'Connected' : 'Disconnected',
    timestamp: new Date().toISOString(),
  });
});

// Start server
const PORT = process.env.LOGBOOK_NOTIFICATIONS_PORT || 5026;
app.listen(PORT, () => {
  console.log(`\n🚀 Logbook Notifications Service running on port ${PORT}`);
  console.log(`📊 MongoDB: titan_drilling`);
  console.log(`🔗 Health check: http://api.titandrillingzm.com:${PORT}/health\n`);
});

// Graceful shutdown
process.on('SIGTERM', () => {
  console.log('SIGTERM signal received: closing HTTP server');
  app.close(() => {
    console.log('HTTP server closed');
    mongoose.connection.close(false, () => {
      console.log('MongoDB connection closed');
      process.exit(0);
    });
  });
});

module.exports = app;

