<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class UserFirebaseService
{
    private $databaseUrl;
    private $apiKey;
    private $firebaseAuthService;
    
    public function __construct(FirebaseAuthService $firebaseAuthService)
    {
        $this->databaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
        $this->apiKey = 'AIzaSyDsA1JohORkg-EdXWTHlFEea8zCqfvig8w';
        $this->firebaseAuthService = $firebaseAuthService;
    }
    
    /**
     * Save new user to Firebase Authentication and Realtime Database
     */
    public function saveNewUser($userData, $adminId = 'cahj3jJqe5Z16pQdLGSWrNgm84o1')
    {
        try {
            // Step 1: Create user in Firebase Authentication
            $authResult = $this->firebaseAuthService->createUser(
                $userData['email'],
                $userData['password'],
                $userData['name']
            );
            
            if (!$authResult['success']) {
                Log::error('Failed to create Firebase Auth user', [
                    'error' => $authResult['message'],
                    'email' => $userData['email']
                ]);
                
                return [
                    'success' => false,
                    'message' => 'Failed to create user in Firebase Authentication: ' . $authResult['message']
                ];
            }
            
            $uid = $authResult['uid'];
            Log::info('Firebase Auth user created successfully', [
                'uid' => $uid,
                'email' => $userData['email']
            ]);
            
            // Step 2: Prepare user data for Realtime Database in the exact format requested
            $firebaseUserData = [
                // Basic user info
                'name' => $userData['name'] ?? '',
                'email' => $userData['email'] ?? '',
                'password' => $userData['password'] ?? '',
                'employeeNumber' => $userData['employeeNumber'] ?? '',
                'role' => $userData['role'] ?? '',
                'nrcIdNumber' => $userData['nrcIdNumber'] ?? '',
                'engagementDate' => $userData['engagementDate'] ?? '',
                'loginStatus' => $userData['loginStatus'] ?? false,
                
                // Countries and projects as arrays
                'countries' => $userData['countries'] ?? [],
                'projects' => $userData['projects'] ?? [],
                
                // Modules with permissions
                'modules' => $userData['modules'] ?? [],
                
                // Medical information
                'dateExamined' => $userData['dateExamined'] ?? null,
                'expiryMedicals' => $userData['expiryMedicals'] ?? null,
                
                // Driver's license
                'driverLicenseNumber' => $userData['driverLicenseNumber'] ?? null,
                'driverLicenseCode' => $userData['driverLicenseCode'] ?? null,
                'driverLicenseExpiry' => $userData['driverLicenseExpiry'] ?? null,
                
                // Training dates
                'clientDrivingDate' => $userData['clientDrivingDate'] ?? null,
                'titanInductionDate' => $userData['titanInductionDate'] ?? null,
                'dateCompletedFirstAid' => $userData['dateCompletedFirstAid'] ?? null,
                
                // Notifications (empty object for new user)
                'notifications' => $userData['notifications'] ?? [],
                
                // Source key for tracking
                'sourceKey' => $userData['sourceKey'] ?? null,
                
                // Additional metadata
                'uid' => $uid,
                'createdAt' => now()->toISOString(),
                'updatedAt' => now()->toISOString(),
                'createdBy' => $adminId,
                'status' => 'active',
                'userType' => 'New User',
                'authProvider' => 'email/password',
                'emailVerified' => false
            ];
            
            // Remove null values to clean up the data
            $firebaseUserData = array_filter($firebaseUserData, function($value) {
                return $value !== null && $value !== '';
            });
            
            // Step 3: Save to Firebase Realtime Database using UID
            $path = "Globalusers/admins/{$uid}.json";
            $url = $this->databaseUrl . $path;
            
            Log::info('Attempting to save user data to Firebase Realtime Database', [
                'uid' => $uid,
                'path' => $path,
                'url' => $url,
                'adminId' => $adminId,
                'countries' => $userData['countries'] ?? [],
                'projects' => $userData['projects'] ?? [],
                'modules' => $userData['modules'] ?? []
            ]);
            
            $response = Http::put($url, $firebaseUserData);
            
            if ($response->successful()) {
                Log::info('User data saved to Firebase Realtime Database successfully', [
                    'uid' => $uid,
                    'path' => $path,
                    'adminId' => $adminId,
                    'response' => $response->json()
                ]);
                
                return [
                    'success' => true,
                    'message' => 'User created successfully in Firebase Authentication and data saved to Realtime Database',
                    'userId' => $uid,
                    'uid' => $uid,
                    'email' => $userData['email'],
                    'path' => $path,
                    'authData' => $authResult,
                    'data' => $response->json()
                ];
            } else {
                Log::error('Failed to save user data to Firebase Realtime Database', [
                    'status' => $response->status(),
                    'response' => $response->body(),
                    'uid' => $uid,
                    'url' => $url
                ]);
                
                // If Realtime Database save fails, we should clean up the Auth user
                $this->firebaseAuthService->deleteUser($authResult['idToken']);
                
                return [
                    'success' => false,
                    'message' => 'Failed to save user data to Firebase Realtime Database: ' . $response->body(),
                    'error' => $response->body(),
                    'status' => $response->status()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while saving new user to Firebase', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
                'userData' => $userData
            ]);
            
            return [
                'success' => false,
                'message' => 'Error saving user to Firebase: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get all users for an admin
     */
    public function getAllUsers($adminId = 'cahj3jJqe5Z16pQdLGSWrNgm84o1')
    {
        try {
            $path = "Globalusers/admins/{$adminId}/New User.json";
            $url = $this->databaseUrl . $path;
            
            $response = Http::get($url);
            
            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to retrieve users from Firebase'
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while retrieving users from Firebase', [
                'message' => $e->getMessage()
            ]);
            
            return [
                'success' => false,
                'message' => 'Error retrieving users from Firebase: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Get specific user by ID
     */
    public function getUserById($userId, $adminId = 'cahj3jJqe5Z16pQdLGSWrNgm84o1')
    {
        try {
            $path = "Globalusers/admins/{$adminId}/New User/{$userId}.json";
            $url = $this->databaseUrl . $path;
            
            $response = Http::get($url);
            
            if ($response->successful()) {
                return [
                    'success' => true,
                    'data' => $response->json()
                ];
            } else {
                return [
                    'success' => false,
                    'message' => 'Failed to retrieve user from Firebase'
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while retrieving user from Firebase', [
                'message' => $e->getMessage(),
                'userId' => $userId
            ]);
            
            return [
                'success' => false,
                'message' => 'Error retrieving user from Firebase: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Update user in Firebase
     */
    public function updateUser($userId, $userData, $adminId = 'cahj3jJqe5Z16pQdLGSWrNgm84o1')
    {
        try {
            // Add update timestamp
            $userData['updatedAt'] = now()->toISOString();
            
            $path = "Globalusers/admins/{$adminId}/New User/{$userId}.json";
            $url = $this->databaseUrl . $path;
            
            $response = Http::patch($url, $userData);
            
            if ($response->successful()) {
                Log::info('User updated in Firebase successfully', [
                    'userId' => $userId,
                    'path' => $path
                ]);
                
                return [
                    'success' => true,
                    'message' => 'User updated successfully',
                    'data' => $response->json()
                ];
            } else {
                Log::error('Failed to update user in Firebase', [
                    'status' => $response->status(),
                    'response' => $response->body(),
                    'userId' => $userId
                ]);
                
                return [
                    'success' => false,
                    'message' => 'Failed to update user in Firebase',
                    'error' => $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while updating user in Firebase', [
                'message' => $e->getMessage(),
                'userId' => $userId
            ]);
            
            return [
                'success' => false,
                'message' => 'Error updating user in Firebase: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Delete user from Firebase
     */
    public function deleteUser($userId, $adminId = 'cahj3jJqe5Z16pQdLGSWrNgm84o1')
    {
        try {
            $path = "Globalusers/admins/{$adminId}/New User/{$userId}.json";
            $url = $this->databaseUrl . $path;
            
            $response = Http::delete($url);
            
            if ($response->successful()) {
                Log::info('User deleted from Firebase successfully', [
                    'userId' => $userId,
                    'path' => $path
                ]);
                
                return [
                    'success' => true,
                    'message' => 'User deleted successfully'
                ];
            } else {
                Log::error('Failed to delete user from Firebase', [
                    'status' => $response->status(),
                    'response' => $response->body(),
                    'userId' => $userId
                ]);
                
                return [
                    'success' => false,
                    'message' => 'Failed to delete user from Firebase',
                    'error' => $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while deleting user from Firebase', [
                'message' => $e->getMessage(),
                'userId' => $userId
            ]);
            
            return [
                'success' => false,
                'message' => 'Error deleting user from Firebase: ' . $e->getMessage()
            ];
        }
    }
}
