<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class EquipmentFirebaseService
{
    private $databaseUrl;
    private $apiKey;

    public function __construct()
    {
        $this->databaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
        $this->apiKey = 'AIzaSyDsA1JohORkg-EdXWTHlFEea8zCqfvig8w';
    }

    /**
     * Fetch all equipment data from Firebase
     */
    public function fetchAllEquipment()
    {
        try {
            $equipmentPath = '/EquipmentsDataJSON/GlobalEquipmentJsonData.json';
            $url = $this->databaseUrl . $equipmentPath;

            Log::info('Fetching equipment data from Firebase', ['url' => $url]);

            $response = Http::get($url);

            if ($response->successful()) {
                $equipmentData = $response->json();

                // Process the data to ensure it's in the expected format
                $processedEquipment = [];

                if (is_array($equipmentData)) {
                    foreach ($equipmentData as $index => $item) {
                        if (is_array($item) || is_object($item)) {
                            $item = (array) $item;

                            // Add firebaseKey for reference
                            $item['firebaseKey'] = $index;

                            // Ensure required fields exist
                            $processedEquipment[] = [
                                'id' => $item['id'] ?? $item['firebaseKey'] ?? $index,
                                'firebaseKey' => $index,
                                'equipmentName' => $item['equipmentName'] ?? $item['vehicleNumber'] ?? 'Unknown Equipment',
                                'vehicleNumber' => $item['vehicleNumber'] ?? $item['id'] ?? $index,
                                'model' => $item['model'] ?? 'N/A',
                                'mainCategory' => $item['mainCategory'] ?? $item['equipmentCategory'] ?? 'N/A',
                                'equipmentCategory' => $item['equipmentCategory'] ?? $item['mainCategory'] ?? 'N/A',
                                'status' => $item['status'] ?? 'Available',
                                'meterReading' => $item['meterReading'] ?? 0,
                                'meterUnit' => $item['meterUnit'] ?? 'KM',
                                'createdAt' => $item['createdAt'] ?? now()->toISOString(),
                                'updatedAt' => $item['updatedAt'] ?? now()->toISOString(),
                            ];
                        }
                    }
                }

                Log::info('Successfully fetched equipment data', ['count' => count($processedEquipment)]);

                return [
                    'success' => true,
                    'equipment' => $processedEquipment,
                    'count' => count($processedEquipment)
                ];
            } else {
                Log::error('Failed to fetch equipment data from Firebase', [
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);

                return [
                    'success' => false,
                    'message' => 'Failed to fetch equipment data from Firebase. Status: ' . $response->status()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Error fetching equipment from Firebase', [
                'message' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'message' => 'Error fetching equipment data: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get specific equipment by ID
     */
    public function getEquipmentById($equipmentId)
    {
        try {
            $equipmentPath = "/EquipmentsDataJSON/GlobalEquipmentJsonData/{$equipmentId}.json";
            $url = $this->databaseUrl . $equipmentPath;

            $response = Http::get($url);

            if ($response->successful()) {
                $equipmentData = $response->json();

                if ($equipmentData && (is_array($equipmentData) || is_object($equipmentData))) {
                    $equipmentData = (array) $equipmentData;
                    $equipmentData['firebaseKey'] = $equipmentId;

                    return [
                        'success' => true,
                        'equipment' => [
                            'id' => $equipmentData['id'] ?? $equipmentId,
                            'firebaseKey' => $equipmentId,
                            'equipmentName' => $equipmentData['equipmentName'] ?? $equipmentData['vehicleNumber'] ?? 'Unknown Equipment',
                            'vehicleNumber' => $equipmentData['vehicleNumber'] ?? $equipmentData['id'] ?? $equipmentId,
                            'model' => $equipmentData['model'] ?? 'N/A',
                            'mainCategory' => $equipmentData['mainCategory'] ?? $equipmentData['equipmentCategory'] ?? 'N/A',
                            'equipmentCategory' => $equipmentData['equipmentCategory'] ?? $equipmentData['mainCategory'] ?? 'N/A',
                            'status' => $equipmentData['status'] ?? 'Available',
                            'meterReading' => $equipmentData['meterReading'] ?? 0,
                            'meterUnit' => $equipmentData['meterUnit'] ?? 'KM',
                        ]
                    ];
                }
            }

            return [
                'success' => false,
                'message' => 'Equipment not found'
            ];
        } catch (\Exception $e) {
            Log::error('Error fetching equipment by ID from Firebase', [
                'message' => $e->getMessage(),
                'equipmentId' => $equipmentId
            ]);

            return [
                'success' => false,
                'message' => 'Error fetching equipment data: ' . $e->getMessage()
            ];
        }
    }
}
