<?php

namespace App\Services;

use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class DvirFirebaseService
{
    private $databaseUrl;
    private $storageUrl;

    public function __construct()
    {
        $this->databaseUrl = 'https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/';
        $this->storageUrl = 'https://firebasestorage.googleapis.com/v0/b/titan-drilling-1f8e9.firebasestorage.app/o/';
    }

    /**
     * Save DVIR report to Firebase Realtime Database
     */
    public function saveDvirReport($userId, $reportData)
    {
        try {
            $url = $this->databaseUrl . "GlobalDvirReports/admins/{$userId}/inspectionReports/{$reportData['id']}.json";
            
            Log::info('Saving DVIR report to Firebase', [
                'userId' => $userId,
                'reportId' => $reportData['id'],
                'url' => $url
            ]);

            $response = Http::put($url, $reportData);

            if ($response->successful()) {
                Log::info('DVIR report saved to Firebase successfully', [
                    'userId' => $userId,
                    'reportId' => $reportData['id'],
                    'status' => $response->status()
                ]);

                return [
                    'success' => true,
                    'message' => 'DVIR report saved successfully',
                    'data' => $response->json()
                ];
            } else {
                Log::error('Failed to save DVIR report to Firebase', [
                    'userId' => $userId,
                    'reportId' => $reportData['id'],
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);

                return [
                    'success' => false,
                    'message' => 'Failed to save DVIR report to Firebase',
                    'error' => $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while saving DVIR report to Firebase', [
                'userId' => $userId,
                'reportId' => $reportData['id'] ?? 'unknown',
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString()
            ]);

            return [
                'success' => false,
                'message' => 'Error saving DVIR report to Firebase: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get DVIR reports for a user
     */
    public function getDvirReports($userId)
    {
        try {
            $url = $this->databaseUrl . "GlobalDvirReports/admins/{$userId}/inspectionReports.json";
            
            $response = Http::get($url);

            if ($response->successful()) {
                $data = $response->json();
                return [
                    'success' => true,
                    'data' => $data ?: []
                ];
            } else {
                Log::error('Failed to retrieve DVIR reports from Firebase', [
                    'userId' => $userId,
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);

                return [
                    'success' => false,
                    'message' => 'Failed to retrieve DVIR reports from Firebase',
                    'error' => $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while retrieving DVIR reports from Firebase', [
                'userId' => $userId,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Error retrieving DVIR reports from Firebase: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Get a specific DVIR report
     */
    public function getDvirReport($userId, $reportId)
    {
        try {
            $url = $this->databaseUrl . "GlobalDvirReports/admins/{$userId}/inspectionReports/{$reportId}.json";
            
            $response = Http::get($url);

            if ($response->successful()) {
                $data = $response->json();
                return [
                    'success' => true,
                    'data' => $data
                ];
            } else {
                Log::error('Failed to retrieve DVIR report from Firebase', [
                    'userId' => $userId,
                    'reportId' => $reportId,
                    'status' => $response->status()
                ]);

                return [
                    'success' => false,
                    'message' => 'Failed to retrieve DVIR report from Firebase',
                    'error' => $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while retrieving DVIR report from Firebase', [
                'userId' => $userId,
                'reportId' => $reportId,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Error retrieving DVIR report from Firebase: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Update DVIR report in Firebase
     */
    public function updateDvirReport($userId, $reportId, $reportData)
    {
        try {
            $url = $this->databaseUrl . "GlobalDvirReports/admins/{$userId}/inspectionReports/{$reportId}.json";
            
            $response = Http::put($url, $reportData);

            if ($response->successful()) {
                Log::info('DVIR report updated in Firebase successfully', [
                    'userId' => $userId,
                    'reportId' => $reportId
                ]);

                return [
                    'success' => true,
                    'message' => 'DVIR report updated successfully'
                ];
            } else {
                Log::error('Failed to update DVIR report in Firebase', [
                    'userId' => $userId,
                    'reportId' => $reportId,
                    'status' => $response->status()
                ]);

                return [
                    'success' => false,
                    'message' => 'Failed to update DVIR report in Firebase',
                    'error' => $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while updating DVIR report in Firebase', [
                'userId' => $userId,
                'reportId' => $reportId,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Error updating DVIR report in Firebase: ' . $e->getMessage()
            ];
        }
    }

    /**
     * Delete DVIR report from Firebase
     */
    public function deleteDvirReport($userId, $reportId)
    {
        try {
            $url = $this->databaseUrl . "GlobalDvirReports/admins/{$userId}/inspectionReports/{$reportId}.json";
            
            $response = Http::delete($url);

            if ($response->successful()) {
                Log::info('DVIR report deleted from Firebase successfully', [
                    'userId' => $userId,
                    'reportId' => $reportId
                ]);

                return [
                    'success' => true,
                    'message' => 'DVIR report deleted successfully'
                ];
            } else {
                Log::error('Failed to delete DVIR report from Firebase', [
                    'userId' => $userId,
                    'reportId' => $reportId,
                    'status' => $response->status()
                ]);

                return [
                    'success' => false,
                    'message' => 'Failed to delete DVIR report from Firebase',
                    'error' => $response->body()
                ];
            }
        } catch (\Exception $e) {
            Log::error('Exception while deleting DVIR report from Firebase', [
                'userId' => $userId,
                'reportId' => $reportId,
                'error' => $e->getMessage()
            ]);

            return [
                'success' => false,
                'message' => 'Error deleting DVIR report from Firebase: ' . $e->getMessage()
            ];
        }
    }
}
