<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Services\DvirFirebaseService;
use Illuminate\Support\Facades\Validator;

class DvirReportController extends Controller
{
    private $dvirFirebaseService;

    public function __construct(DvirFirebaseService $dvirFirebaseService)
    {
        $this->dvirFirebaseService = $dvirFirebaseService;
    }

    /**
     * Save DVIR report to Firebase
     */
    public function saveReport(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'userId' => 'required|string',
            'reportData' => 'required|array',
            'reportData.id' => 'required|string',
            'reportData.dvirNumber' => 'required|string',
            'reportData.step1' => 'required|array',
            'reportData.step2' => 'required|array',
            'reportData.step3' => 'required|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        $userId = $request->input('userId');
        $reportData = $request->input('reportData');

        // Add timestamp if not present
        if (!isset($reportData['createdAt'])) {
            $reportData['createdAt'] = now()->toISOString();
        }

        $result = $this->dvirFirebaseService->saveDvirReport($userId, $reportData);

        if ($result['success']) {
            return response()->json($result, 200);
        } else {
            return response()->json($result, 500);
        }
    }

    /**
     * Get DVIR reports for a user
     */
    public function getReports(Request $request, $userId)
    {
        $result = $this->dvirFirebaseService->getDvirReports($userId);

        if ($result['success']) {
            return response()->json($result, 200);
        } else {
            return response()->json($result, 500);
        }
    }

    /**
     * Get a specific DVIR report
     */
    public function getReport(Request $request, $userId, $reportId)
    {
        $result = $this->dvirFirebaseService->getDvirReport($userId, $reportId);

        if ($result['success']) {
            return response()->json($result, 200);
        } else {
            return response()->json($result, 500);
        }
    }

    /**
     * Update DVIR report
     */
    public function updateReport(Request $request, $userId, $reportId)
    {
        $validator = Validator::make($request->all(), [
            'reportData' => 'required|array',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'success' => false,
                'message' => 'Validation failed',
                'errors' => $validator->errors()
            ], 400);
        }

        $reportData = $request->input('reportData');
        $result = $this->dvirFirebaseService->updateDvirReport($userId, $reportId, $reportData);

        if ($result['success']) {
            return response()->json($result, 200);
        } else {
            return response()->json($result, 500);
        }
    }

    /**
     * Delete DVIR report
     */
    public function deleteReport(Request $request, $userId, $reportId)
    {
        $result = $this->dvirFirebaseService->deleteDvirReport($userId, $reportId);

        if ($result['success']) {
            return response()->json($result, 200);
        } else {
            return response()->json($result, 500);
        }
    }
}
