@extends('welcome')

@section('title', 'Add New Vehicle')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/sweetalert2@11/dist/sweetalert2.min.css" rel="stylesheet">
<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.vehicle-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
}

.section-title i {
    color: var(--primary-color);
}

/* Form Groups */
.form-group {
    margin-bottom: 25px;
}

.form-group:last-child {
    margin-bottom: 0;
}

.form-label {
    display: block;
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
}

.required {
    color: var(--danger-color);
    font-weight: bold;
}

.form-input {
    width: 100%;
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-family: var(--font-family);
    transition: var(--transition-base);
    background: white;
}

.form-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-input.error {
    border-color: var(--danger-color);
}

.form-textarea {
    min-height: 100px;
    resize: vertical;
}

.error-message {
    color: var(--danger-color);
    font-size: 12px;
    margin-top: 5px;
    display: none;
}

.error-message.show {
    display: block;
}

/* Form Row Layout */
.form-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 20px;
}

@media (max-width: 768px) {
    .form-row {
        grid-template-columns: 1fr;
        gap: 15px;
    }
}

/* Input with Select */
.input-with-select {
    display: flex;
    gap: 10px;
}

.input-with-select .form-input {
    flex: 1;
}

.input-with-select .form-select {
    width: 120px;
    min-width: 120px;
}

/* Collapsible Sections */
.collapsible-section {
    margin-top: 20px;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    overflow: hidden;
}

.collapsible-header {
    background: var(--bg-light);
    padding: 15px 20px;
    cursor: pointer;
    display: flex;
    justify-content: space-between;
    align-items: center;
    transition: var(--transition-base);
    border-bottom: 1px solid var(--border-light);
}

.collapsible-header:hover {
    background: var(--bg-secondary);
}

.collapsible-title {
    font-weight: 600;
    color: var(--primary-color);
    font-size: 16px;
}

.collapsible-arrow {
    color: var(--primary-color);
    font-size: 14px;
    transition: transform 0.3s ease;
}

.collapsible-arrow.rotated {
    transform: rotate(180deg);
}

.collapsible-content {
    padding: 20px;
    background: white;
    display: none;
    animation: slideDown 0.3s ease;
}

.collapsible-content.expanded {
    display: block;
}

@keyframes slideDown {
    from {
        opacity: 0;
        max-height: 0;
    }
    to {
        opacity: 1;
        max-height: 1000px;
    }
}

/* Button Styles */
.btn {
    padding: 10px 20px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 8px;
}

.btn-primary {
    background: var(--primary-color);
    color: white;
}

.btn-primary:hover {
    background: var(--primary-hover);
}

.btn-secondary {
    background: var(--secondary-color);
    color: white;
}

.btn-secondary:hover {
    background: #5a6268;
}

.btn-outline {
    background: transparent;
    color: var(--primary-color);
    border: 1px solid var(--primary-color);
}

.btn-outline:hover {
    background: var(--primary-color);
    color: white;
}

/* Form Actions */
.form-actions {
    display: flex;
    gap: 16px;
    justify-content: flex-end;
    padding-top: 30px;
    border-top: 1px solid var(--border-light);
}

/* Notification Styles */
.notification {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 10000;
    background: white;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    transform: translateX(100%);
    transition: transform 0.3s ease;
    max-width: 400px;
}

.notification.show {
    transform: translateX(0);
}

.notification-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    padding: 16px 20px;
}

.notification-message {
    flex: 1;
    font-size: 0.9rem;
    font-weight: 500;
}

.notification-close {
    background: none;
    border: none;
    font-size: 1.2rem;
    cursor: pointer;
    color: #6c757d;
    margin-left: 12px;
}

.notification-success {
    border-left: 4px solid #28a745;
}

.notification-error {
    border-left: 4px solid #dc3545;
}

.notification-info {
    border-left: 4px solid #0055a5;
}

.notification-success .notification-message {
    color: #28a745;
}

.notification-error .notification-message {
    color: #dc3545;
}

.notification-info .notification-message {
    color: #0055a5;
}

/* Success Modal Styles */
.success-modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10000;
    opacity: 0;
    transition: opacity 0.3s ease;
}

.success-modal-container {
    background: white;
    border-radius: 12px;
    box-shadow: 0 10px 30px rgba(0, 0, 0, 0.3);
    max-width: 500px;
    width: 90%;
    max-height: 80vh;
    overflow: hidden;
    position: relative;
}

.success-modal-header {
    background: #0055A5;
    padding: 20px 30px;
    text-align: center;
    border-radius: 12px 12px 0 0;
}

.success-title {
    color: white;
    font-size: 24px;
    font-weight: 700;
    margin: 0;
    font-family: var(--font-family);
}

.success-modal-body {
    padding: 40px 30px;
    text-align: center;
}

.success-message {
    color: #333;
    font-size: 18px;
    font-weight: 500;
    margin: 0 0 30px 0;
    font-family: var(--font-family);
}

.success-buttons {
    display: flex;
    gap: 15px;
    justify-content: center;
    margin-top: 20px;
}

.success-btn {
    background: #0055A5;
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 8px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.3s ease;
    font-family: var(--font-family);
    min-width: 120px;
}

.success-btn:hover {
    background: #003d7a;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.success-btn:active {
    transform: translateY(0);
}

.back-btn {
    background: #0055A5;
}

.add-another-btn {
    background: #0055A5;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .main-content {
        padding: 20px 15px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .form-actions {
        flex-direction: column;
    }
    
    .success-buttons {
        flex-direction: column;
    }
    
    .success-modal-container {
        width: 95%;
        margin: 10% auto;
    }
    
    .success-modal-body {
        padding: 30px 20px;
    }
    
    .success-title {
        font-size: 20px;
    }
    
    .success-message {
        font-size: 16px;
    }
}

@media (max-width: 480px) {
    .form-group {
        margin-bottom: 20px;
    }
    
    .input-with-select {
        flex-direction: column;
    }
    
    .input-with-select .form-select {
        width: 100%;
    }
}
</style>
@endsection

@section('content')
<div class="vehicle-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('vehicles.inventory') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Add New Vehicle</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" onclick="submitVehicleForm()">
                    <i class="fas fa-plus"></i>
                    Add Vehicle
                </button>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <div class="form-container">
            <form id="addVehicleForm" class="vehicle-form">
                <!-- Basic Information Section -->
                <div class="form-section">
                    <h2 class="section-title">
                        <i class="fas fa-info-circle"></i>
                        Basic Information
                    </h2>

                    <div class="form-row">
                        <div class="form-group">
                            <label class="form-label">
                                Country
                                <span class="required">*</span>
                            </label>
                            <select class="form-input" id="country" name="country" required>
                                <option value="">Select Country</option>
                                <option value="Zambia">Zambia</option>
                                <option value="DRC">DRC</option>
                            </select>
                            <div class="error-message" id="countryError">Country is required</div>
                        </div>
                        <div class="form-group">
                            <label class="form-label">
                                Main Category
                                <span class="required">*</span>
                            </label>
                            <select class="form-input" id="mainCategory" name="mainCategory" required onchange="handleCategoryChange()">
                                <option value="">Select Main Category</option>
                                <option value="Light Vehicles">Light Vehicles</option>
                                <option value="Heavy Vehicles">Heavy Vehicles</option>
                                <option value="Rigs">Rigs</option>
                                <option value="Heavy Plant Equipment">Heavy Plant Equipment</option>
                                <option value="Auxiliary Equipment">Auxiliary Equipment</option>
                                <option value="Trailers">Trailers</option>
                                <option value="Lathes">Lathes</option>
                            </select>
                            <div class="error-message" id="categoryError">Main Category is required</div>
                        </div>
                    </div>
                    
                    <!-- Dynamic Fields Container -->
                    <div id="dynamicFieldsContainer" style="display: none;">
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label" id="dynamicField1Label">Field 1 <span class="required">*</span></label>
                                <input type="text" class="form-input" id="dynamicField1" name="dynamicField1" placeholder="Enter field 1" required>
                                <div class="error-message" id="dynamicField1Error">This field is required</div>
                            </div>
                            <div class="form-group">
                                <label class="form-label" id="dynamicField2Label">Field 2 <span class="required">*</span></label>
                                <input type="text" class="form-input" id="dynamicField2" name="dynamicField2" placeholder="Enter field 2" required>
                                <div class="error-message" id="dynamicField2Error">This field is required</div>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label" id="dynamicField3Label">Field 3 (Optional)</label>
                                <input type="text" class="form-input" id="dynamicField3" name="dynamicField3" placeholder="Enter field 3">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Vehicle Number <span class="required">*</span></label>
                                <input type="text" class="form-input" id="vehicleNumber" name="vehicleNumber" placeholder="Enter vehicle number" required>
                                <div class="error-message" id="vehicleNumberError">Vehicle Number is required</div>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Vehicle Name <span class="required">*</span></label>
                                <input type="text" class="form-input" id="vehicleName" name="vehicleName" placeholder="Enter vehicle name" required>
                                <div class="error-message" id="vehicleNameError">Vehicle Name is required</div>
                            </div>
                            <div class="form-group">
                                <label class="form-label">Meter Reading</label>
                                <div class="input-with-select">
                                    <input type="text" class="form-input" id="meterReading" name="meterReading" placeholder="Enter meter reading">
                                    <select class="form-input" id="meterUnit" name="meterUnit">
                                        <option value="Miles">Miles</option>
                                        <option value="KM">KM</option>
                                        <option value="Hours">Hours</option>
                                    </select>
                                </div>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Model</label>
                                <input type="text" class="form-input" id="model" name="model" placeholder="Enter model">
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Optional Fields Section -->
                <div class="collapsible-section">
                    <div class="collapsible-header" onclick="toggleOptionalFields()">
                        <span class="collapsible-title">Optional Fields</span>
                        <i class="fas fa-chevron-down collapsible-arrow" id="optionalToggleIcon"></i>
                    </div>
                    <div class="collapsible-content" id="optionalFieldsContainer">
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Last Service Reading</label>
                                <input type="text" class="form-input" id="lastServiceReading" name="lastServiceReading" placeholder="Enter last service reading">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Last Service Date</label>
                                <input type="date" class="form-input" id="lastServiceDate" name="lastServiceDate">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">VIN Number</label>
                                <input type="text" class="form-input" id="vin" name="vin" placeholder="Enter VIN number">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Fleet Value</label>
                                <input type="text" class="form-input" id="fleetValue" name="fleetValue" placeholder="Enter fleet value">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Purchase Date</label>
                                <input type="date" class="form-input" id="purchaseDate" name="purchaseDate">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Warranty Exp Date</label>
                                <input type="date" class="form-input" id="warrantyDate" name="warrantyDate">
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Operator Name</label>
                                <input type="text" class="form-input" id="operatorName" name="operatorName" placeholder="Enter operator name">
                            </div>
                            <div class="form-group">
                                <label class="form-label">Maintenance Priority</label>
                                <select class="form-input" id="maintenancePriority" name="maintenancePriority">
                                    <option value="1">1</option>
                                    <option value="2">2</option>
                                    <option value="3">3</option>
                                </select>
                            </div>
                        </div>
                        <div class="form-row">
                            <div class="form-group">
                                <label class="form-label">Ownership Mode</label>
                                <select class="form-input" id="ownershipMode" name="ownershipMode">
                                    <option value="Owned Asset">Owned Asset</option>
                                    <option value="Leased">Leased</option>
                                    <option value="Rented">Rented</option>
                                    <option value="Other">Other</option>
                                </select>
                            </div>
                        </div>
                    </div>
                </div>


            </form>
        </div>
    </div>
</div>

<!-- Success Modal -->
<div id="successModal" class="success-modal-overlay" style="display: none;">
    <div class="success-modal-container">
        <div class="success-modal-header">
            <h2 class="success-title">Success</h2>
        </div>
        <div class="success-modal-body">
            <p class="success-message">Vehicle Added Successfully</p>
            <div class="success-buttons">
                <button type="button" class="success-btn back-btn" onclick="goBackToInventory()">
                    Back to Inventory
                </button>
                <button type="button" class="success-btn add-another-btn" onclick="addAnotherVehicle()">
                    Add Another Vehicle
                </button>
            </div>
        </div>
    </div>
</div>

<script>
// Field mappings based on category
const fieldMappings = {
    'Light Vehicles': {
        field1Key: 'vehicleNumber',
        field2Key: 'registrationNumber',
        field3Key: 'vinNumber',
        field1Label: 'Vehicle Number',
        field2Label: 'Registration Number',
        field3Label: 'VIN Number',
    },
    'Heavy Vehicles': {
        field1Key: 'vehicleNumber',
        field2Key: 'registrationNumber',
        field3Key: 'vinNumber',
        field1Label: 'Vehicle Number',
        field2Label: 'Registration Number',
        field3Label: 'VIN Number',
    },
    'Rigs': {
        field1Key: 'rigType',
        field2Key: 'chassisNumber',
        field3Key: 'serialNumber',
        field1Label: 'Rig Type',
        field2Label: 'Chassis Number',
        field3Label: 'Serial Number',
    },
    'Heavy Plant Equipment': {
        field1Key: 'equipmentNumberDynamic',
        field2Key: 'equipmentNameDynamic',
        field3Key: 'vinNumber',
        field1Label: 'Equipment Number',
        field2Label: 'Equipment Name',
        field3Label: 'VIN Number',
    },
    'Auxiliary Equipment': {
        field1Key: 'type',
        field2Key: 'modelDynamic',
        field3Key: 'serialNumber',
        field1Label: 'Type',
        field2Label: 'Model',
        field3Label: 'Serial Number',
    },
    'Trailers': {
        field1Key: 'type',
        field2Key: 'chassisNumber',
        field3Key: 'chassisNumberDuplicate',
        field1Label: 'Type',
        field2Label: 'Chassis Number',
        field3Label: 'Chassis Number',
    },
    'Lathes': {
        field1Key: 'type',
        field2Key: 'modelDynamic',
        field3Key: 'serialNumber',
        field1Label: 'Type',
        field2Label: 'Model',
        field3Label: 'Serial Number',
    }
};

// Rig type options
const rigTypeOptions = [
    { value: 'Track Mounted', label: 'Track Mounted' },
    { value: 'Trailer Mounted', label: 'Trailer Mounted' },
    { value: 'Truck Mounted', label: 'Truck Mounted' }
];

document.addEventListener('DOMContentLoaded', function() {
    // Initialize form handlers
    initFormHandlers();
    
    // Check if in edit mode
    checkEditMode();
    
    // Generate vehicle number only if not in edit mode
    const urlParams = new URLSearchParams(window.location.search);
    const isEditMode = urlParams.get('edit') === 'true';
    if (!isEditMode) {
        generateUniqueVehicleNumber();
    }
});

function initFormHandlers() {
    const form = document.getElementById('addVehicleForm');
    const requiredFields = form.querySelectorAll('[required]');
    
    // Add real-time validation
    requiredFields.forEach(field => {
        field.addEventListener('blur', function() {
            validateField(this);
        });
    });
}

function generateUniqueVehicleNumber() {
    const timestamp = Math.floor(Date.now() / 1000);
    const vehicleNumber = `EQ-${timestamp}`;
    
    // Set the generated number in the vehicle number field
    const vehicleNumberField = document.getElementById('vehicleNumber');
    if (vehicleNumberField) {
        vehicleNumberField.value = vehicleNumber;
        console.log('✅ Generated Vehicle Number:', vehicleNumber);
    }
}

function handleCategoryChange() {
    const category = document.getElementById('mainCategory').value;
    const dynamicContainer = document.getElementById('dynamicFieldsContainer');
    
    if (category && fieldMappings[category]) {
        const mapping = fieldMappings[category];
        
        // Always restore the original field structure first
        restoreOriginalFieldStructure();
        
        // Update field labels
        document.getElementById('dynamicField1Label').innerHTML = mapping.field1Label + ' <span class="required">*</span>';
        document.getElementById('dynamicField2Label').innerHTML = mapping.field2Label + ' <span class="required">*</span>';
        document.getElementById('dynamicField3Label').textContent = mapping.field3Label + ' (Optional)';
        
        // Update placeholders
        document.getElementById('dynamicField1').placeholder = `Enter ${mapping.field1Label}`;
        document.getElementById('dynamicField2').placeholder = `Enter ${mapping.field2Label}`;
        document.getElementById('dynamicField3').placeholder = `Enter ${mapping.field3Label}`;
        
        // Handle special case for Rigs - replace input with select
        if (category === 'Rigs') {
            const field1Input = document.getElementById('dynamicField1');
            const field1Container = field1Input.parentElement;
            
            // Store the current value if any
            const currentValue = field1Input.value;
            
            // Replace input with select
            field1Container.innerHTML = `
                <label class="form-label">${mapping.field1Label} <span class="required">*</span></label>
                <select class="form-input" id="rigType" name="rigType" required>
                    <option value="">Select ${mapping.field1Label}</option>
                    ${rigTypeOptions.map(option => `<option value="${option.value}" ${option.value === currentValue ? 'selected' : ''}>${option.label}</option>`).join('')}
                </select>
                <div class="error-message" id="rigTypeError">This field is required</div>
            `;
        }
        
        // Show dynamic fields
        dynamicContainer.style.display = 'block';
        
        // Update required field validation
        updateRequiredFields(category);
    } else {
        // Hide dynamic fields
        dynamicContainer.style.display = 'none';
    }
}

function restoreOriginalFieldStructure() {
    // Find the first form group in dynamic fields container
    const dynamicContainer = document.getElementById('dynamicFieldsContainer');
    const firstFormGroup = dynamicContainer.querySelector('.form-group');
    
    // If the first field is a select (rigType), restore it to input
    if (firstFormGroup && firstFormGroup.querySelector('select#rigType')) {
        const currentValue = firstFormGroup.querySelector('select#rigType').value;
        firstFormGroup.innerHTML = `
            <label class="form-label" id="dynamicField1Label">Field 1 <span class="required">*</span></label>
            <input type="text" class="form-input" id="dynamicField1" name="dynamicField1" placeholder="Enter field 1" required value="${currentValue}">
            <div class="error-message" id="dynamicField1Error">This field is required</div>
        `;
    }
}

function updateRequiredFields(category) {
    const nonMandatoryCategories = ['Light Vehicles', 'Heavy Vehicles', 'Rigs'];
    const vehicleNumberField = document.getElementById('vehicleNumber');
    const vehicleNameField = document.getElementById('vehicleName');
    
    if (nonMandatoryCategories.includes(category)) {
        vehicleNumberField.removeAttribute('required');
        vehicleNameField.removeAttribute('required');
    } else {
        vehicleNumberField.setAttribute('required', 'required');
        vehicleNameField.setAttribute('required', 'required');
    }
}

function toggleOptionalFields() {
    const container = document.getElementById('optionalFieldsContainer');
    const toggleIcon = document.getElementById('optionalToggleIcon');
    
    if (container.classList.contains('expanded')) {
        container.classList.remove('expanded');
        toggleIcon.classList.remove('rotated');
    } else {
        container.classList.add('expanded');
        toggleIcon.classList.add('rotated');
    }
}

function validateField(field) {
    const value = field.value.trim();
    const isValid = value !== '';
    
    if (isValid) {
        field.style.borderColor = '#28a745';
    } else {
        field.style.borderColor = '#dc3545';
    }
    
    return isValid;
}

function validateForm() {
    const form = document.getElementById('addVehicleForm');
    const requiredFields = form.querySelectorAll('[required]');
    let isValid = true;
    let errors = [];
    
    // Check country
    const country = document.getElementById('country').value;
    if (!country) {
        errors.push('Please select a country');
        isValid = false;
    }
    
    // Check main category
    const mainCategory = document.getElementById('mainCategory').value;
    if (!mainCategory) {
        errors.push('Please select a main category');
        isValid = false;
    }
    
    // Check dynamic fields based on category
    if (mainCategory && fieldMappings[mainCategory]) {
        const mapping = fieldMappings[mainCategory];
        
        if (mainCategory === 'Rigs') {
            const rigTypeElement = document.getElementById('rigType');
            if (rigTypeElement) {
                const rigType = rigTypeElement.value;
                if (!rigType) {
                    errors.push(`Please select ${mapping.field1Label}`);
                    isValid = false;
                }
            } else {
                errors.push(`Please select ${mapping.field1Label}`);
                isValid = false;
            }
        } else {
            const dynamicField1 = document.getElementById('dynamicField1');
            if (dynamicField1 && !dynamicField1.value) {
                errors.push(`Please enter ${mapping.field1Label}`);
                isValid = false;
            }
        }
        
        const dynamicField2 = document.getElementById('dynamicField2');
        if (dynamicField2 && !dynamicField2.value) {
            errors.push(`Please enter ${mapping.field2Label}`);
            isValid = false;
        }
        
        // Check vehicle number and name for non-mandatory categories
        const nonMandatoryCategories = ['Light Vehicles', 'Heavy Vehicles', 'Rigs'];
        if (!nonMandatoryCategories.includes(mainCategory)) {
            const vehicleNumber = document.getElementById('vehicleNumber').value;
            const vehicleName = document.getElementById('vehicleName').value;
            
            if (!vehicleNumber) {
                errors.push('Please enter Vehicle Number');
                isValid = false;
            }
            if (!vehicleName) {
                errors.push('Please enter Vehicle Name');
                isValid = false;
            }
        }
    }
    
    // Validate all required fields
    requiredFields.forEach(field => {
        if (!validateField(field)) {
            isValid = false;
        }
    });
    
    if (!isValid) {
        const errorMessage = errors.length > 0 ? errors.join(', ') : 'Please fill in all required fields';
        showNotification(errorMessage, 'error');
    }
    
    return isValid;
}

async function saveVehicleData() {
    if (!validateForm()) {
        return;
    }
    
    // Show loading state
    const saveBtn = document.querySelector('.btn-secondary');
    const originalText = saveBtn.innerHTML;
    saveBtn.innerHTML = '<i class="fas fa-spinner fa-spin"></i> Saving...';
    saveBtn.disabled = true;
    
    try {
        // Get current equipment count to determine next index
        const countResponse = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData.json');
        const existingData = await countResponse.json();
        const nextIndex = existingData ? Object.keys(existingData).length : 0;
        
        // Collect form data
        const formData = new FormData(document.getElementById('addVehicleForm'));
        const vehicleData = {};
        
        for (let [key, value] of formData.entries()) {
            vehicleData[key] = value;
        }
        
        // Get current category for proper field mapping
    const mainCategory = document.getElementById('mainCategory').value;
    const mapping = mainCategory && fieldMappings[mainCategory] ? fieldMappings[mainCategory] : null;
    
        // Prepare data in the exact format required
        const equipmentData = {
            "category": mainCategory,
            "comment": "",
            "country": vehicleData.country,
            "createdBy": "cahj3jJqe5Z16pQdLGSWrNgm84o1",
            "engineNumber": "",
            "equipmentName": vehicleData.vehicleName || "",
            "firebaseKey": `equip_${nextIndex}`,
            "id": nextIndex.toString(),
            "index": nextIndex,
            "isActive": true,
            "make": "",
            "meterReading": parseFloat(vehicleData.meterReading) || 0,
            "meterType": vehicleData.meterUnit || "",
            "model": vehicleData.model || "",
            "odometerReading": "",
            "registrationNumber": "",
            "status": "Available",
            "timestamp": new Date().toISOString(),
            "updatedAt": new Date().toISOString(),
            "vehicleNumber": vehicleData.vehicleNumber || "",
            "vehicleType": mainCategory,
            "vinNumber": vehicleData.vin || ""
        };
        
        // Add category-specific fields
        if (mapping) {
            if (mainCategory === 'Rigs') {
                const rigTypeElement = document.getElementById('rigType');
                if (rigTypeElement) {
                    equipmentData.rigType = rigTypeElement.value;
                }
                const dynamicField2 = document.getElementById('dynamicField2');
                const dynamicField3 = document.getElementById('dynamicField3');
                equipmentData.chassisNumber = dynamicField2 ? dynamicField2.value : '';
                equipmentData.serialNumber = dynamicField3 ? dynamicField3.value : '';
            } else if (mainCategory === 'Light Vehicles' || mainCategory === 'Heavy Vehicles') {
                const dynamicField2 = document.getElementById('dynamicField2');
                const dynamicField3 = document.getElementById('dynamicField3');
                equipmentData.registrationNumber = dynamicField2 ? dynamicField2.value : '';
                equipmentData.vinNumber = (dynamicField3 ? dynamicField3.value : '') || vehicleData.vin || "";
            } else if (mainCategory === 'Heavy Plant Equipment') {
                const dynamicField1 = document.getElementById('dynamicField1');
                const dynamicField2 = document.getElementById('dynamicField2');
                const dynamicField3 = document.getElementById('dynamicField3');
                equipmentData.equipmentNumber = dynamicField1 ? dynamicField1.value : '';
                equipmentData.equipmentName = dynamicField2 ? dynamicField2.value : '';
                equipmentData.vinNumber = (dynamicField3 ? dynamicField3.value : '') || vehicleData.vin || "";
            } else if (mainCategory === 'Auxiliary Equipment' || mainCategory === 'Lathes') {
                const dynamicField1 = document.getElementById('dynamicField1');
                const dynamicField2 = document.getElementById('dynamicField2');
                const dynamicField3 = document.getElementById('dynamicField3');
                equipmentData.type = dynamicField1 ? dynamicField1.value : '';
                equipmentData.model = dynamicField2 ? dynamicField2.value : '';
                equipmentData.serialNumber = dynamicField3 ? dynamicField3.value : '';
            } else if (mainCategory === 'Trailers') {
                const dynamicField1 = document.getElementById('dynamicField1');
                const dynamicField2 = document.getElementById('dynamicField2');
                equipmentData.type = dynamicField1 ? dynamicField1.value : '';
                equipmentData.chassisNumber = dynamicField2 ? dynamicField2.value : '';
            }
        }
        
        // Add optional fields if provided
        if (vehicleData.lastServiceReading) {
            equipmentData.lastServiceReading = vehicleData.lastServiceReading;
        }
        if (vehicleData.lastServiceDate) {
            equipmentData.lastServiceDate = vehicleData.lastServiceDate;
        }
        if (vehicleData.fleetValue) {
            equipmentData.fleetValue = vehicleData.fleetValue;
        }
        if (vehicleData.purchaseDate) {
            equipmentData.purchaseDate = vehicleData.purchaseDate;
        }
        if (vehicleData.warrantyDate) {
            equipmentData.warrantyDate = vehicleData.warrantyDate;
        }
        if (vehicleData.operatorName) {
            equipmentData.operatorName = vehicleData.operatorName;
        }
        if (vehicleData.maintenancePriority) {
            equipmentData.maintenancePriority = vehicleData.maintenancePriority;
        }
        if (vehicleData.ownershipMode) {
            equipmentData.ownershipMode = vehicleData.ownershipMode;
        }
        
        console.log('Equipment Data to Save:', equipmentData);
        
        // Save to Firebase
        const response = await fetch(`https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/EquipmentsDataJSON/GlobalEquipmentJsonData/${nextIndex}.json`, {
            method: 'PUT',
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(equipmentData)
        });
        
        if (response.ok) {
            showNotification('Vehicle saved successfully!', 'success');
            console.log('Vehicle saved to Firebase:', equipmentData);
        } else {
            throw new Error('Failed to save to Firebase');
        }
        
    } catch (error) {
        console.error('Error saving vehicle:', error);
        showNotification('Error saving vehicle: ' + error.message, 'error');
    } finally {
        // Restore button state
        saveBtn.innerHTML = originalText;
        saveBtn.disabled = false;
    }
}

// Check if page is in Edit Mode
async function checkEditMode() {
    const urlParams = new URLSearchParams(window.location.search);
    const isEditMode = urlParams.get('edit') === 'true';
    const equipmentId = urlParams.get('equipmentId');
    
    if (isEditMode && equipmentId) {
        console.log('🔧 Edit mode detected for equipment:', equipmentId);
        
        // Update page title and button
        const headerTitle = document.querySelector('.header-title');
        if (headerTitle) {
            headerTitle.innerHTML = '<i class="fas fa-edit" style="margin-right: 10px;"></i>Update Vehicle';
        }
        
        document.title = 'Update Vehicle - Titan Drilling';
        
        const submitBtn = document.querySelector('.create-btn');
        if (submitBtn) {
            submitBtn.innerHTML = '<i class="fas fa-save"></i> Update Vehicle';
        }
        
        // Load equipment data from MongoDB
        await loadEquipmentDataForEdit(equipmentId);
    }
}

// Load equipment data for editing
async function loadEquipmentDataForEdit(equipmentId) {
    try {
        console.log('📥 Loading equipment data for editing:', equipmentId);
        
        const response = await fetch(`http://168.231.113.219:5001/equipment/${equipmentId}`);
        const equipment = await response.json();
        
        if (equipment) {
            console.log('✅ Equipment data loaded:', equipment);
            
            // Store equipment ID for update operation
            window.editingEquipmentId = equipmentId;
            
            // Fill form fields
            const form = document.getElementById('addVehicleForm');
            if (!form) return;
            
            // Fill all fields
            const fillField = (name, value) => {
                const input = form.querySelector(`[name="${name}"]`);
                if (input && value) {
                    input.value = value;
                    console.log(`✓ Filled ${name}:`, value);
                }
            };
            
            // Basic fields
            fillField('country', equipment.country);
            fillField('mainCategory', equipment.category || equipment.mainCategory);
            
            // Trigger category change to show dynamic fields
            const mainCategory = equipment.category || equipment.mainCategory;
            if (mainCategory) {
                console.log('🔄 Triggering category change for:', mainCategory);
                handleCategoryChange();
                
                // Wait a bit for DOM to update, then fill category-specific fields
                setTimeout(() => {
                    // Fill category-specific fields based on category type
                    if (mainCategory === 'Rigs') {
                        const rigTypeSelect = document.getElementById('dynamicField1');
                        if (rigTypeSelect && equipment.rigType) {
                            rigTypeSelect.value = equipment.rigType;
                            console.log('✓ Filled rigType:', equipment.rigType);
                        }
                        const chassisField = document.getElementById('dynamicField2');
                        if (chassisField && equipment.chassisNumber) {
                            chassisField.value = equipment.chassisNumber;
                            console.log('✓ Filled chassisNumber:', equipment.chassisNumber);
                        }
                        const serialField = document.getElementById('dynamicField3');
                        if (serialField && equipment.serialNumber) {
                            serialField.value = equipment.serialNumber;
                            console.log('✓ Filled serialNumber:', equipment.serialNumber);
                        }
                    } else if (mainCategory === 'Light Vehicles' || mainCategory === 'Heavy Vehicles') {
                        const regField = document.getElementById('dynamicField2');
                        if (regField && (equipment.registrationNumber || equipment.vehicleNumber)) {
                            regField.value = equipment.registrationNumber || equipment.vehicleNumber;
                            console.log('✓ Filled registrationNumber');
                        }
                        const vinField = document.getElementById('dynamicField3');
                        if (vinField && equipment.vinNumber) {
                            vinField.value = equipment.vinNumber;
                            console.log('✓ Filled vinNumber');
                        }
                    } else if (mainCategory === 'Heavy Plant Equipment') {
                        const eqNumField = document.getElementById('dynamicField1');
                        if (eqNumField && equipment.equipmentNumber) {
                            eqNumField.value = equipment.equipmentNumber;
                            console.log('✓ Filled equipmentNumber');
                        }
                        const eqNameField = document.getElementById('dynamicField2');
                        if (eqNameField && equipment.equipmentName) {
                            eqNameField.value = equipment.equipmentName;
                            console.log('✓ Filled equipmentName');
                        }
                        const vinField = document.getElementById('dynamicField3');
                        if (vinField && equipment.vinNumber) {
                            vinField.value = equipment.vinNumber;
                            console.log('✓ Filled vinNumber');
                        }
                    } else if (mainCategory === 'Auxiliary Equipment' || mainCategory === 'Lathes') {
                        const typeField = document.getElementById('dynamicField1');
                        if (typeField && equipment.type) {
                            typeField.value = equipment.type;
                            console.log('✓ Filled type');
                        }
                        const modelField = document.getElementById('dynamicField2');
                        if (modelField && equipment.model) {
                            modelField.value = equipment.model;
                            console.log('✓ Filled model');
                        }
                        const serialField = document.getElementById('dynamicField3');
                        if (serialField && equipment.serialNumber) {
                            serialField.value = equipment.serialNumber;
                            console.log('✓ Filled serialNumber');
                        }
                    } else if (mainCategory === 'Trailers') {
                        const typeField = document.getElementById('dynamicField1');
                        if (typeField && equipment.type) {
                            typeField.value = equipment.type;
                            console.log('✓ Filled type');
                        }
                        const chassisField = document.getElementById('dynamicField2');
                        if (chassisField && equipment.chassisNumber) {
                            chassisField.value = equipment.chassisNumber;
                            console.log('✓ Filled chassisNumber');
                        }
                    }
                }, 100);
            }
            
            // Fill remaining basic fields
            fillField('vehicleNumber', equipment.vehicleNumber || equipment.registrationNumber);
            fillField('vehicleName', equipment.vehicleName || equipment.equipmentName);
            fillField('model', equipment.model);
            fillField('make', equipment.make);
            fillField('year', equipment.year);
            fillField('serialNumber', equipment.serialNumber);
            fillField('status', equipment.status);
            fillField('meterReading', equipment.meterReading);
            fillField('meterUnit', equipment.meterUnit);
            fillField('operator', equipment.operator);
            fillField('site', equipment.site || equipment.location);
            fillField('description', equipment.description);
            
            // Fill optional fields if they exist
            if (equipment.lastServiceReading) fillField('lastServiceReading', equipment.lastServiceReading);
            if (equipment.lastServiceDate) fillField('lastServiceDate', equipment.lastServiceDate);
            if (equipment.vinNumber) fillField('vin', equipment.vinNumber);
            if (equipment.fleetValue) fillField('fleetValue', equipment.fleetValue);
            if (equipment.purchaseDate) fillField('purchaseDate', equipment.purchaseDate);
            if (equipment.warrantyDate) fillField('warrantyDate', equipment.warrantyDate);
            if (equipment.operatorName) fillField('operatorName', equipment.operatorName);
            if (equipment.maintenancePriority) fillField('maintenancePriority', equipment.maintenancePriority);
            if (equipment.ownershipMode) fillField('ownershipMode', equipment.ownershipMode);
            
            console.log('✅ Form populated with equipment data including category-specific fields');
        } else {
            console.error('❌ Failed to load equipment data');
            alert('Failed to load equipment data for editing');
        }
        
    } catch (error) {
        console.error('❌ Error loading equipment data:', error);
        alert('Error loading equipment data: ' + error.message);
    }
}

async function submitVehicleForm() {
    if (!validateForm()) {
        return;
    }
    
    // Check if in Edit Mode
    const isEditMode = window.editingEquipmentId ? true : false;
    const equipmentId = window.editingEquipmentId;
    
    // Show loading state
    const submitBtn = document.querySelector('.create-btn');
    const originalText = submitBtn.innerHTML;
    submitBtn.innerHTML = `<i class="fas fa-spinner fa-spin"></i> ${isEditMode ? 'Updating' : 'Adding'} Vehicle...`;
    submitBtn.disabled = true;
    
    try {
        // Get current equipment count to determine next index (only for new equipment)
        let nextIndex = 0;
        if (!isEditMode) {
            const countResponse = await fetch('http://168.231.113.219:5001/equipment/all?page=1&limit=1');
            const countData = await countResponse.json();
            nextIndex = countData.total || 0;
        }
    
    // Collect form data
    const formData = new FormData(document.getElementById('addVehicleForm'));
    const vehicleData = {};
    
    for (let [key, value] of formData.entries()) {
        vehicleData[key] = value;
    }
    
        // Get current category for proper field mapping
    const mainCategory = document.getElementById('mainCategory').value;
        const mapping = mainCategory && fieldMappings[mainCategory] ? fieldMappings[mainCategory] : null;
        
        // Prepare data in the exact format required
        const equipmentData = {
            "category": mainCategory,
            "mainCategory": mainCategory,
            "comment": vehicleData.description || "",
            "description": vehicleData.description || "",
            "country": vehicleData.country || "",
            "createdBy": "cahj3jJqe5Z16pQdLGSWrNgm84o1",
            "engineNumber": "",
            "equipmentName": vehicleData.vehicleName || "",
            "vehicleName": vehicleData.vehicleName || "",
            "firebaseKey": isEditMode ? (equipmentId || `equip_${nextIndex}`) : `equip_${nextIndex}`,
            "id": isEditMode ? equipmentId : `EQ-${Date.now()}`,
            "index": nextIndex,
            "isActive": true,
            "make": vehicleData.make || "",
            "meterReading": parseFloat(vehicleData.meterReading) || 0,
            "meterType": vehicleData.meterUnit || "",
            "meterUnit": vehicleData.meterUnit || "",
            "model": vehicleData.model || "",
            "year": vehicleData.year || "",
            "odometerReading": "",
            "registrationNumber": vehicleData.registrationNumber || "",
            "status": vehicleData.status || "Available",
            "timestamp": new Date().toISOString(),
            "updatedAt": new Date().toISOString(),
            "vehicleNumber": vehicleData.vehicleNumber || "",
            "vehicleType": mainCategory,
            "vinNumber": vehicleData.vin || "",
            "operator": vehicleData.operator || "",
            "operatorName": vehicleData.operatorName || vehicleData.operator || "",
            "site": vehicleData.site || "",
            "location": vehicleData.site || "",
            "serialNumber": vehicleData.serialNumber || ""
        };
        
        // Add category-specific fields
        if (mapping) {
            if (mainCategory === 'Rigs') {
                const rigTypeElement = document.getElementById('rigType');
                if (rigTypeElement) {
                    equipmentData.rigType = rigTypeElement.value;
                }
                const dynamicField2 = document.getElementById('dynamicField2');
                const dynamicField3 = document.getElementById('dynamicField3');
                equipmentData.chassisNumber = dynamicField2 ? dynamicField2.value : '';
                equipmentData.serialNumber = dynamicField3 ? dynamicField3.value : '';
            } else if (mainCategory === 'Light Vehicles' || mainCategory === 'Heavy Vehicles') {
                const dynamicField2 = document.getElementById('dynamicField2');
                const dynamicField3 = document.getElementById('dynamicField3');
                equipmentData.registrationNumber = dynamicField2 ? dynamicField2.value : '';
                equipmentData.vinNumber = (dynamicField3 ? dynamicField3.value : '') || vehicleData.vin || "";
            } else if (mainCategory === 'Heavy Plant Equipment') {
                const dynamicField1 = document.getElementById('dynamicField1');
                const dynamicField2 = document.getElementById('dynamicField2');
                const dynamicField3 = document.getElementById('dynamicField3');
                equipmentData.equipmentNumber = dynamicField1 ? dynamicField1.value : '';
                equipmentData.equipmentName = dynamicField2 ? dynamicField2.value : '';
                equipmentData.vinNumber = (dynamicField3 ? dynamicField3.value : '') || vehicleData.vin || "";
            } else if (mainCategory === 'Auxiliary Equipment' || mainCategory === 'Lathes') {
                const dynamicField1 = document.getElementById('dynamicField1');
                const dynamicField2 = document.getElementById('dynamicField2');
                const dynamicField3 = document.getElementById('dynamicField3');
                equipmentData.type = dynamicField1 ? dynamicField1.value : '';
                equipmentData.model = dynamicField2 ? dynamicField2.value : '';
                equipmentData.serialNumber = dynamicField3 ? dynamicField3.value : '';
            } else if (mainCategory === 'Trailers') {
                const dynamicField1 = document.getElementById('dynamicField1');
                const dynamicField2 = document.getElementById('dynamicField2');
                equipmentData.type = dynamicField1 ? dynamicField1.value : '';
                equipmentData.chassisNumber = dynamicField2 ? dynamicField2.value : '';
            }
        }
        
        // Add optional fields if provided
        if (vehicleData.lastServiceReading) {
            equipmentData.lastServiceReading = vehicleData.lastServiceReading;
        }
        if (vehicleData.lastServiceDate) {
            equipmentData.lastServiceDate = vehicleData.lastServiceDate;
        }
        if (vehicleData.fleetValue) {
            equipmentData.fleetValue = vehicleData.fleetValue;
        }
        if (vehicleData.purchaseDate) {
            equipmentData.purchaseDate = vehicleData.purchaseDate;
        }
        if (vehicleData.warrantyDate) {
            equipmentData.warrantyDate = vehicleData.warrantyDate;
        }
        if (vehicleData.operatorName) {
            equipmentData.operatorName = vehicleData.operatorName;
        }
        if (vehicleData.maintenancePriority) {
            equipmentData.maintenancePriority = vehicleData.maintenancePriority;
        }
        if (vehicleData.ownershipMode) {
            equipmentData.ownershipMode = vehicleData.ownershipMode;
        }
        
        console.log('Equipment Data to Submit:', equipmentData);
        
        // Save to MongoDB (create or update)
        let apiUrl, method;
        if (isEditMode) {
            apiUrl = `http://168.231.113.219:5001/equipment/${equipmentId}`;
            method = 'PUT';
            console.log('🔄 Updating equipment:', equipmentId);
        } else {
            apiUrl = 'http://168.231.113.219:5001/equipment/add';
            method = 'POST';
            console.log('➕ Creating new equipment');
        }
        
        const response = await fetch(apiUrl, {
            method: method,
            headers: {
                'Content-Type': 'application/json',
            },
            body: JSON.stringify(equipmentData)
        });
        
        const result = await response.json();
        
        if (response.ok) {
            console.log(`✅ Vehicle ${isEditMode ? 'updated' : 'added'} successfully:`, result);
            
            // Show success notification with SweetAlert2
            Swal.fire({
                icon: 'success',
                title: isEditMode ? 'Vehicle Updated!' : 'Vehicle Added!',
                text: isEditMode ? 'Equipment has been updated successfully.' : 'New equipment has been added successfully.',
                confirmButtonColor: '#0055a5',
                confirmButtonText: 'Go to Inventory'
            }).then((result) => {
                if (result.isConfirmed) {
                    window.location.href = '/vehicles';
                }
            });
        } else {
            throw new Error(result.message || `Failed to ${isEditMode ? 'update' : 'save'} equipment`);
        }
        
    } catch (error) {
        console.error(`❌ Error ${isEditMode ? 'updating' : 'adding'} vehicle:`, error);
        
        Swal.fire({
            icon: 'error',
            title: 'Error!',
            text: `Error ${isEditMode ? 'updating' : 'adding'} vehicle: ${error.message}`,
            confirmButtonColor: '#dc3545'
        });
    } finally {
        // Restore button state
        submitBtn.innerHTML = originalText;
        submitBtn.disabled = false;
    }
}

function uploadImage() {
    showNotification('Image upload functionality will be implemented', 'info');
}

function selectChecklist() {
    showNotification('Checklist selection functionality will be implemented', 'info');
}

function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span class="notification-message">${message}</span>
            <button class="notification-close" onclick="this.parentElement.parentElement.remove()">×</button>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.classList.add('show');
    }, 100);
    
    setTimeout(() => {
        notification.classList.remove('show');
        setTimeout(() => {
            if (notification.parentElement) {
                notification.remove();
            }
        }, 300);
    }, 3000);
}

// Success Modal Functions
function showSuccessModal() {
    const modal = document.getElementById('successModal');
    if (modal) {
        modal.style.display = 'flex';
        // Add fade-in animation
        setTimeout(() => {
            modal.style.opacity = '1';
        }, 10);
    }
}

function hideSuccessModal() {
    const modal = document.getElementById('successModal');
    if (modal) {
        modal.style.opacity = '0';
        setTimeout(() => {
            modal.style.display = 'none';
        }, 300);
    }
}

function goBackToInventory() {
    hideSuccessModal();
    window.location.href = '{{ route("vehicles.inventory") }}';
}

function addAnotherVehicle() {
    hideSuccessModal();
    // Reset the form
    document.getElementById('addVehicleForm').reset();
    // Hide dynamic fields
    document.getElementById('dynamicFieldsContainer').style.display = 'none';
    // Reset optional fields
    const optionalContainer = document.getElementById('optionalFieldsContainer');
    if (optionalContainer) {
        optionalContainer.classList.remove('expanded');
        document.getElementById('optionalToggleIcon').classList.remove('rotated');
    }
    // Scroll to top
    window.scrollTo({ top: 0, behavior: 'smooth' });
}
</script>

<!-- SweetAlert2 JS -->
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

@endsection
