@extends('welcome')

@section('title', 'Service Schedule')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

<meta name="csrf-token" content="{{ csrf_token() }}">

@php
    function getStatusColor($status) {
        $status = strtolower($status);
        switch ($status) {
            case 'completed':
            case 'closed':
            case 'done':
                return 'success';
            case 'in progress':
            case 'processing':
            case 'pending':
                return 'primary';
            case 'on hold':
            case 'overdue':
                return 'danger';
            case 'initiated':
            case 'scheduled':
                return 'info';
            case 'cancelled':
                return 'secondary';
            default:
                return 'info';
        }
    }

    function getDueStatusColor($dueDate) {
        if ($dueDate === 'N/A') return 'secondary';
        
        $due = \Carbon\Carbon::parse($dueDate);
        $now = \Carbon\Carbon::now();
        
        if ($due->isPast()) return 'danger'; // Overdue
        if ($due->diffInDays($now) <= 7) return 'warning'; // Due soon
        return 'success'; // On time
    }
@endphp

@section('content')
<div class="service-schedule-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="javascript:history.back()" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Service Schedule</h1>
            </div>
            <div class="header-right">
                <div class="search-container">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="search-icon">
                        <circle cx="11" cy="11" r="8"/>
                        <path d="M21 21l-4.35-4.35"/>
                    </svg>
                    <input type="text" class="search-input" placeholder="Search service schedules..." id="scheduleSearch">
                </div>
                <a href="{{ route('maintenance.create_schedule') }}" class="create-btn">
                    <i class="fas fa-plus"></i>
                    Create Schedule
                </a>
            </div>
        </div>
    </div>
    
    <!-- Main Content -->
    <div class="main-content">
        <div class="service-schedule-content">
        <div class="table-section">
            <div class="table-container">
                <table class="service-schedule-table">
                    <thead class="table-header">
                        <tr>
                            <th>Action</th>
                            <th>Schedule ID</th>
                            <th>Vehicle Number</th>
                            <th>Due Date</th>
                            <th>Status</th>
                            <th>Workorder</th>
                            <th>Schedule Title</th>
                            <th>Last Service Date</th>
                        </tr>
                    </thead>
                    <tbody class="table-body" id="serviceScheduleTableBody">
                        @if(isset($serviceScheduleData) && count($serviceScheduleData) > 0)
                            @php
                                $currentPage = request()->get('page', 1);
                                $perPage = 10;
                                $totalSchedules = count($serviceScheduleData);
                                $totalPages = ceil($totalSchedules / $perPage);
                                $startIndex = ($currentPage - 1) * $perPage;
                                $endIndex = min($startIndex + $perPage, $totalSchedules);
                                $paginatedSchedules = array_slice($serviceScheduleData, $startIndex, $perPage);
                            @endphp
                            
                            @if(count($paginatedSchedules) > 0)
                                @foreach($paginatedSchedules as $schedule)
                                    <tr class="table-row">
                                        <td>
                                            <button class="btn-view" onclick="viewSchedule('{{ $schedule['id'] }}')">
                                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                                    <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"/>
                                                    <circle cx="12" cy="12" r="3"/>
                                                </svg>
                                                View
                                            </button>
                                        </td>
                                        <td class="schedule-id">{{ $schedule['scheduleId'] }}</td>
                                        <td>{{ $schedule['vehicleNumber'] }}</td>
                                        <td>
                                            <span class="due-date-badge due-{{ getDueStatusColor($schedule['dueDate']) }}">{{ $schedule['dueDate'] }}</span>
                                        </td>
                                        <td>
                                            <span class="status-badge status-{{ getStatusColor($schedule['status']) }}">{{ $schedule['status'] }}</span>
                                        </td>
                                        <td>{{ $schedule['workorder'] }}</td>
                                        <td class="schedule-title">{{ $schedule['scheduleTitle'] }}</td>
                                        <td>{{ $schedule['lastServiceDate'] }}</td>
                                    </tr>
                                @endforeach
                            @else
                                <tr class="empty-row">
                                    <td colspan="8" class="empty-state">
                                        <div class="empty-content">
                                            <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                                <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                                                <line x1="16" y1="2" x2="16" y2="6"/>
                                                <line x1="8" y1="2" x2="8" y2="6"/>
                                                <line x1="3" y1="10" x2="21" y2="10"/>
                                            </svg>
                                            <h3>No Service Schedules</h3>
                                            <p>There are currently no service schedules to display.</p>
                                        </div>
                                    </td>
                                </tr>
                            @endif
                        @else
                            <tr class="empty-row">
                                <td colspan="8" class="empty-state">
                                    <div class="empty-content">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                                            <line x1="16" y1="2" x2="16" y2="6"/>
                                            <line x1="8" y1="2" x2="8" y2="6"/>
                                            <line x1="3" y1="10" x2="21" y2="10"/>
                                        </svg>
                                        <h3>No Service Schedules</h3>
                                        <p>There are currently no service schedules to display.</p>
                                    </div>
                                </td>
                            </tr>
                        @endif
                    </tbody>
                </table>
            </div>
            
            <!-- Pagination Controls -->
            @if(isset($serviceScheduleData) && count($serviceScheduleData) > 0 && $totalPages > 1)
                <div class="pagination-container">
                    <div class="pagination-info">
                        <span>Showing {{ $startIndex + 1 }}-{{ $endIndex }} of {{ $totalSchedules }} service schedules</span>
                    </div>
                    <div class="pagination-controls">
                        @if($currentPage > 1)
                            <a href="?page={{ $currentPage - 1 }}" class="pagination-btn">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="15,18 9,12 15,6"/>
                                </svg>
                                Previous
                            </a>
                        @else
                            <span class="pagination-btn disabled">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="15,18 9,12 15,6"/>
                                </svg>
                                Previous
                            </span>
                        @endif
                        
                        <div class="pagination-pages">
                            @php
                                $startPage = max(1, $currentPage - 2);
                                $endPage = min($totalPages, $currentPage + 2);
                            @endphp
                            
                            @if($startPage > 1)
                                <a href="?page=1" class="page-number">1</a>
                                @if($startPage > 2)
                                    <span class="page-ellipsis">...</span>
                                @endif
                            @endif
                            
                            @for($i = $startPage; $i <= $endPage; $i++)
                                @if($i == $currentPage)
                                    <span class="page-number active">{{ $i }}</span>
                                @else
                                    <a href="?page={{ $i }}" class="page-number">{{ $i }}</a>
                                @endif
                            @endfor
                            
                            @if($endPage < $totalPages)
                                @if($endPage < $totalPages - 1)
                                    <span class="page-ellipsis">...</span>
                                @endif
                                <a href="?page={{ $totalPages }}" class="page-number">{{ $totalPages }}</a>
                            @endif
                        </div>
                        
                        @if($currentPage < $totalPages)
                            <a href="?page={{ $currentPage + 1 }}" class="pagination-btn">
                                Next
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="9,18 15,12 9,6"/>
                                </svg>
                            </a>
                        @else
                            <span class="pagination-btn disabled">
                                Next
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <polyline points="9,18 15,12 9,6"/>
                                </svg>
                            </span>
                        @endif
                    </div>
                </div>
            @endif
        </div>
    </div>
</div>


    <script>
        function toggleSidebar() {
            const sidebar = document.querySelector('.sidebar');
            sidebar.classList.toggle('collapsed');
        }

        function viewSchedule(scheduleId) {
            showNotification(`Viewing schedule: ${scheduleId}`, 'info');
            console.log('Viewing schedule:', scheduleId);
            // Implement view functionality
        }

        // Search functionality
        document.addEventListener('DOMContentLoaded', function() {
            const scheduleSearch = document.getElementById('scheduleSearch');
            
            if (scheduleSearch) {
                scheduleSearch.addEventListener('input', function() {
                    filterTable(this.value);
                });
            }
        });

        function filterTable(searchTerm) {
            const tableBody = document.getElementById('serviceScheduleTableBody');
            if (!tableBody) return;
            
            const rows = tableBody.getElementsByTagName('tr');
            
            for (let row of rows) {
                const cells = row.getElementsByTagName('td');
                let found = false;
                
                for (let cell of cells) {
                    if (cell.textContent.toLowerCase().includes(searchTerm.toLowerCase())) {
                        found = true;
                        break;
                    }
                }
                
                row.style.display = found ? '' : 'none';
            }
        }


        // Notification system
        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `service-notification service-notification-${type}`;
            notification.textContent = message;
            
            notification.style.cssText = `
                position: fixed;
                top: 80px;
                right: 20px;
                padding: 12px 20px;
                border-radius: 0;
                color: white;
                font-weight: 500;
                font-size: 14px;
                z-index: 10001;
                opacity: 0;
                transform: translateX(100%);
                transition: all 0.3s ease;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                font-family: 'Inter', sans-serif;
            `;
            
            if (type === 'success') {
                notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
            } else if (type === 'error') {
                notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
            } else if (type === 'warning') {
                notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
            } else {
                notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
            }
            
            document.body.appendChild(notification);
            
            setTimeout(() => {
                notification.style.opacity = '1';
                notification.style.transform = 'translateX(0)';
            }, 100);
            
            setTimeout(() => {
                notification.style.opacity = '0';
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        document.body.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }
</script>
<style>
/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== SERVICE SCHEDULE CONTAINER ===== */
.service-schedule-container {
    min-height: 100vh;
    background: #f8fafc;
}

/* ===== HEADER STYLES ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: all 0.2s ease;
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: #0055a5;
    border: none;
    padding: 12px 24px;
    border-radius: 6px;
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    color: #0055a5;
    text-decoration: none;
}

/* Search Container */
.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #9ca3af;
    z-index: 1;
}

.search-input {
    padding: 10px 12px 10px 40px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    background: white;
    width: 300px;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: #9ca3af;
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* ===== CONTENT SECTION ===== */
.service-schedule-content {
    background: white;
    border-radius: 0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    overflow: hidden;
}

/* ===== ACTION BAR ===== */
.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.action-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 20px;
}

.search-section {
    flex: 1;
    max-width: 400px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.action-buttons {
    display: flex;
    align-items: center;
    gap: 12px;
}

.btn-create {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: #0055a5;
    color: white;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    text-decoration: none;
}

.btn-create:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-2px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    text-decoration: none;
    color: black !important;
}

/* ===== TABLE SECTION ===== */
.table-section {
    background: white;
    border-radius: 8px;
    box-shadow: 0 4px 6px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
    overflow: hidden;
    margin-bottom: 24px;
}

.table-container {
    overflow-x: auto;
}

.service-schedule-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
}

.table-header {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.table-header th {
    padding: 16px 12px;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e5e7eb;
    border-radius: 0;
}

.table-body {
    background: white;
}

.table-row {
    border-bottom: 1px solid #f3f4f6;
    transition: all 0.2s ease;
}

.table-row:hover {
    background: #f8fafc;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.table-row td {
    padding: 16px 12px;
    font-size: 14px;
    color: #374151;
    vertical-align: middle;
}

/* ===== BUTTON STYLES ===== */
.btn-view {
    display: flex;
    align-items: center;
    gap: 6px;
    padding: 6px 12px;
    background: #0055a5;
    color: white;
    border: none;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(0, 85, 165, 0.2);
}

.btn-view:hover {
    background: linear-gradient(135deg, #003f7d 0%, #002a5c 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
}

/* ===== STATUS BADGES ===== */
.status-badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.status-success {
    color: green;
}

.status-primary {
    /* background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); */
    /* color: white; */
}

.status-warning {
    /* background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); */
    color: red;
}

.status-info {
    /* background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%); */
    color: #0055a5;
}

.status-danger {
    /* background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); */
    color: red;
}

.status-secondary {
    /* background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); */
    /* color: white; */
}

/* ===== DUE DATE BADGES ===== */
.due-date-badge {
    display: inline-block;
    padding: 4px 12px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 500;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.due-success {
    color: green;
}

.due-warning {
    color: red;
}

.due-danger {
    color: red;
}

.due-secondary {
    /* background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); */
    /* color: white; */
}

/* ===== EMPTY STATE ===== */
.empty-row {
    background: white;
}

.empty-state {
    text-align: center;
    padding: 48px 24px;
    background: white;
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-content svg {
    color: #9ca3af;
}

.empty-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.empty-content p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

/* ===== PAGINATION STYLES ===== */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 24px;
    background: #f8fafc;
    border-top: 1px solid #e2e8f0;
}

.pagination-info {
    color: #64748b;
    font-size: 14px;
    font-weight: 500;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 8px;
}

.pagination-btn {
    display: flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    text-decoration: none;
}

.pagination-btn:hover:not(.disabled) {
    background: #f3f4f6;
    border-color: #9ca3af;
    color: #374151;
    text-decoration: none;
}

.pagination-btn.disabled {
    opacity: 0.5;
    cursor: not-allowed;
    background: #f9fafb;
    color: #9ca3af;
}

.pagination-pages {
    display: flex;
    align-items: center;
    gap: 4px;
}

.page-number {
    padding: 8px 12px;
    border: 1px solid #d1d5db;
    border-radius: 6px;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
    min-width: 40px;
    text-align: center;
    text-decoration: none;
    display: inline-block;
}

.page-number:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    color: #374151;
    text-decoration: none;
}

.page-number.active {
    background: #0055a5;
    border-color: #0055a5;
    color: white;
}

.page-number.active:hover {
    background: #003f7d;
    border-color: #003f7d;
    color: white;
}

.page-ellipsis {
    padding: 8px 4px;
    color: #9ca3af;
    font-size: 14px;
    font-weight: 500;
}


/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .main-content {
        padding: 20px;
    }
    
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .search-input {
        width: 250px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .header-left {
        justify-content: center;
    }
    
    .header-right {
        justify-content: center;
        flex-direction: column;
        gap: 10px;
    }
    
    .search-input {
        width: 100%;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
    
    .pagination-container {
        flex-direction: column;
        gap: 15px;
        padding: 15px 20px;
    }
    
    .pagination-controls {
        flex-wrap: wrap;
        justify-content: center;
        gap: 6px;
    }
    
    .pagination-btn {
        padding: 6px 12px;
        font-size: 13px;
    }
    
    .page-number {
        padding: 6px 10px;
        font-size: 13px;
        min-width: 35px;
    }
}

@media (max-width: 992px) {
    .service-schedule-container {
        padding: 16px;
    }
    
    .service-schedule-header {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .page-title h1 {
        font-size: 26px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-bar {
        padding: 16px 20px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .btn-create {
        padding: 10px 16px;
        font-size: 13px;
        flex: 1;
        min-width: 150px;
    }
    
    .table-container {
        overflow-x: auto;
    }
    
    .service-schedule-table {
        min-width: 800px;
    }
    
}

@media (max-width: 768px) {
    .service-schedule-container {
        padding: 12px;
    }
    
    .service-schedule-header {
        padding: 16px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 22px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-bar {
        padding: 12px 16px;
    }
    
    .action-content {
        flex-direction: column;
        gap: 12px;
        align-items: stretch;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-create {
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
        justify-content: center;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .service-schedule-table {
        min-width: 700px;
    }
    
    .table-header th,
    .table-row td {
        padding: 10px 8px;
        font-size: 13px;
    }
    
    .table-header th {
        font-size: 12px;
        font-weight: 600;
    }
    
    .status-badge,
    .due-date-badge {
        padding: 4px 8px;
        font-size: 11px;
    }
    
    .btn-view {
        padding: 6px 10px;
        font-size: 11px;
    }
    
    .schedule-title {
        max-width: 150px;
        overflow: hidden;
        text-overflow: ellipsis;
        white-space: nowrap;
    }
    
}

@media (max-width: 576px) {
    .service-schedule-container {
        padding: 8px;
    }
    
    .service-schedule-header {
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .btn-create {
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .search-input {
        padding: 12px 12px 12px 40px;
        font-size: 14px;
    }
    
    .search-icon {
        left: 12px;
        width: 16px;
        height: 16px;
    }
    
    .table-container {
        margin-bottom: 16px;
    }
    
    .service-schedule-table {
        min-width: 600px;
    }
    
    .table-header th,
    .table-row td {
        padding: 8px 6px;
        font-size: 12px;
    }
    
    .table-header th {
        font-size: 11px;
        padding: 10px 6px;
    }
    
    .status-badge,
    .due-date-badge {
        padding: 3px 6px;
        font-size: 10px;
    }
    
    .btn-view {
        padding: 5px 8px;
        font-size: 10px;
    }
    
    .schedule-title {
        max-width: 120px;
    }
    
    .empty-state {
        padding: 32px 16px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
    
}

@media (max-width: 480px) {
    .service-schedule-container {
        padding: 4px;
    }
    
    .service-schedule-header {
        padding: 8px;
        margin-bottom: 8px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-bar {
        padding: 6px 8px;
    }
    
    .btn-create {
        padding: 16px 12px;
        font-size: 16px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .search-input {
        padding: 14px 14px 14px 44px;
        font-size: 15px;
        min-height: 48px;
    }
    
    .search-icon {
        left: 14px;
        width: 18px;
        height: 18px;
    }
    
    .service-schedule-table {
        min-width: 500px;
    }
    
    .table-header th,
    .table-row td {
        padding: 6px 4px;
        font-size: 11px;
    }
    
    .table-header th {
        font-size: 10px;
        padding: 8px 4px;
    }
    
    .status-badge,
    .due-date-badge {
        padding: 2px 4px;
        font-size: 9px;
    }
    
    .btn-view {
        padding: 4px 6px;
        font-size: 9px;
    }
    
    .schedule-title {
        max-width: 100px;
    }
    
    .empty-state {
        padding: 24px 12px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
    
}

/* ===== TABLET LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 1024px) and (orientation: landscape) {
    .service-schedule-container {
        padding: 16px;
    }
    
    .action-content {
        flex-direction: row;
        align-items: center;
    }
    
    .search-section {
        max-width: 300px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 6px;
    }
    
    .btn-create {
        padding: 8px 12px;
        font-size: 12px;
    }
}

/* ===== MOBILE LANDSCAPE OPTIMIZATION ===== */
@media (max-width: 896px) and (orientation: landscape) {
    .service-schedule-header {
        padding: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .btn-create {
        padding: 10px 12px;
        font-size: 13px;
    }
}

/* ===== TABLE RESPONSIVE IMPROVEMENTS ===== */
@media (max-width: 768px) {
    .service-schedule-table {
        font-size: 12px;
    }
    
    .table-header th:first-child,
    .table-row td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 10;
        border-right: 1px solid #e2e8f0;
    }
    
    .table-header th:first-child {
        background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    }
    
    .schedule-id {
        font-weight: 600;
        color: #0055a5;
        font-size: 12px;
    }
}


/* ===== ACCESSIBILITY IMPROVEMENTS ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        transition: none !important;
        animation: none !important;
    }
}

/* ===== HIGH DPI DISPLAYS ===== */
@media (-webkit-min-device-pixel-ratio: 2), (min-resolution: 192dpi) {
    .search-icon,
    .btn-create svg,
    .btn-view svg,
    .add-vehicle-btn svg,
    .cancel-btn svg {
        image-rendering: -webkit-optimize-contrast;
        image-rendering: crisp-edges;
    }
}

/* ===== FOCUS STATES ===== */
.search-input:focus,
.btn-create:focus,
.btn-view:focus {
    outline: 2px solid rgba(0, 85, 165, 0.5);
    outline-offset: 2px;
}

/* ===== SMOOTH ANIMATIONS ===== */
* {
    transition: all 0.2s ease;
}

/* ===== HOVER EFFECTS ===== */
.btn-create:active,
.btn-view:active {
    transform: translateY(0);
}

/* ===== LOADING STATES ===== */
.btn-create:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

.btn-create:disabled:hover {
    background: #0055a5;
    transform: none;
}
</style>

@endsection