@extends('welcome')

@section('title', 'Operator Daily Reporting')

@section('content')
<div class="odr-container">
    <!-- Header Section -->
    <div class="odr-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Operator Daily Reporting</h1>
                <p>Track and manage daily operator reports and schedules</p>
            </div>
            <div class="header-actions">
                <div class="search-container">
                    <div class="search-input-wrapper">
                        <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="11" cy="11" r="8"></circle>
                            <path d="m21 21-4.35-4.35"></path>
                        </svg>
                        <input type="text" class="search-input" placeholder="Search reports..." id="searchInput">
                    </div>
                </div>
                <div class="action-buttons">
                    <button class="btn-secondary" data-bs-toggle="modal" data-bs-target="#sorting">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M3 6h18M7 12h10M10 18h4"/>
                        </svg>
                        Sorting
                </button>
                    <button class="btn-primary">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                            <polyline points="14,2 14,8 20,8"></polyline>
                            <line x1="16" y1="13" x2="8" y2="13"></line>
                            <line x1="16" y1="17" x2="8" y2="17"></line>
                            <polyline points="10,9 9,9 8,9"></polyline>
                        </svg>
                        Excel Export
                </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Table Section -->
    <div class="table-section">
        <div class="table-container">
            <table class="odr-table" id="odrTable">
                <thead>
                    <tr>
                        <th>Action</th>
                        <th>Report Number</th>
                        <th>Start Date</th>
                        <th>Vehicle</th>
                        <th>Mon</th>
                        <th>Tue</th>
                        <th>Wed</th>
                        <th>Thu</th>
                        <th>Fri</th>
                        <th>Sat</th>
                        <th>Sun</th>
                        <th>Submitted Report</th>
                    </tr>
                </thead>
                <tbody id="odrTableBody">
                    <tr class="no-data-row">
                        <td colspan="12" class="no-data-cell">
                            <div class="no-data-content">
                                <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="12" cy="12" r="10"></circle>
                                    <line x1="12" y1="8" x2="12" y2="12"></line>
                                    <line x1="12" y1="16" x2="12.01" y2="16"></line>
                                </svg>
                                <h3>No Data Available</h3>
                                <p>No operator daily reports found. Start by creating a new report.</p>
                            </div>
                        </td>
                    </tr>
                </tbody>
            </table>
        </div>
    </div>
</div>
<!-- Sorting Modal -->
<div class="modal fade" id="sorting" tabindex="-1" aria-labelledby="sortingModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="sortingModalLabel">Sorting Options</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <form id="sortingForm">
                    <div class="sort-section">
                        <h6 class="section-title">Sort Order</h6>
                        <div class="radio-group">
                            <div class="radio-item">
                                <input type="radio" id="ascending" name="sortOrder" value="ascending" checked>
                                <label for="ascending">
                                    <span class="radio-custom"></span>
                                    Ascending
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="descending" name="sortOrder" value="descending">
                                <label for="descending">
                                    <span class="radio-custom"></span>
                                    Descending
                                </label>
                        </div>
                        </div>
                    </div>
                    
                    <div class="sort-section">
                        <h6 class="section-title">Sort By</h6>
                        <div class="radio-group">
                            <div class="radio-item">
                                <input type="radio" id="reportNumber" name="sortBy" value="reportNumber" checked>
                                <label for="reportNumber">
                                    <span class="radio-custom"></span>
                                    Report Number
                                </label>
                            </div>
                            <div class="radio-item">
                                <input type="radio" id="startDate" name="sortBy" value="startDate">
                                <label for="startDate">
                                    <span class="radio-custom"></span>
                                    Start Date
                                </label>
                    </div>
                            <div class="radio-item">
                                <input type="radio" id="vehicleNumber" name="sortBy" value="vehicleNumber">
                                <label for="vehicleNumber">
                                    <span class="radio-custom"></span>
                                    Vehicle Number
                                </label>
                    </div>
                            <div class="radio-item">
                                <input type="radio" id="vehicleName" name="sortBy" value="vehicleName">
                                <label for="vehicleName">
                                    <span class="radio-custom"></span>
                                    Vehicle Name
                                </label>
                    </div>
                        </div>
                    </div>
                </form>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn-cancel" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn-reset" onclick="resetSorting()">Reset</button>
                <button type="button" class="btn-apply" onclick="applySorting()">Apply</button>
            </div>
        </div>
    </div>
</div>
<script>
// Search functionality
document.getElementById('searchInput').addEventListener('input', function() {
    const searchTerm = this.value.toLowerCase();
    const tableRows = document.querySelectorAll('#odrTableBody tr');
    
    tableRows.forEach(row => {
        const text = row.textContent.toLowerCase();
        if (text.includes(searchTerm)) {
            row.style.display = '';
            } else {
            row.style.display = 'none';
        }
    });
});

// Global sorting state
let currentSortBy = 'reportNumber';
let currentSortOrder = 'ascending';
let tableData = []; // This will hold data from backend

// Sorting functionality
function applySorting() {
    const sortOrder = document.querySelector('input[name="sortOrder"]:checked').value;
    const sortBy = document.querySelector('input[name="sortBy"]:checked').value;
    
    console.log('Applying sort:', sortBy, sortOrder);
    
    // Update global sorting state
    currentSortBy = sortBy;
    currentSortOrder = sortOrder;
    
    // Apply sorting to table data
    applySortingToTable();
    
    // Close modal
    const modal = bootstrap.Modal.getInstance(document.getElementById('sorting'));
    modal.hide();
    
    // Show notification
    showNotification(`Table sorted by ${getSortByLabel(sortBy)} (${sortOrder})`, 'success');
}

function resetSorting() {
    // Reset to default values
    document.getElementById('ascending').checked = true;
    document.getElementById('reportNumber').checked = true;
    
    // Reset global state
    currentSortBy = 'reportNumber';
    currentSortOrder = 'ascending';
    
    // Apply default sorting
    applySortingToTable();
    
    showNotification('Sorting options reset to default', 'info');
}

function getSortByLabel(sortBy) {
    const labels = {
        'reportNumber': 'Report Number',
        'startDate': 'Start Date',
        'vehicleNumber': 'Vehicle Number',
        'vehicleName': 'Vehicle Name'
    };
    return labels[sortBy] || sortBy;
}

function applySortingToTable() {
    // This function will sort the table data based on current sorting settings
    if (tableData.length === 0) {
        console.log('No data to sort yet');
        return;
    }
    
    // Sort the data array
    tableData.sort((a, b) => {
        let aValue, bValue;
        
        switch (currentSortBy) {
            case 'reportNumber':
                aValue = a.reportNumber || '';
                bValue = b.reportNumber || '';
                break;
            case 'startDate':
                aValue = new Date(a.startDate || '');
                bValue = new Date(b.startDate || '');
                break;
            case 'vehicleNumber':
                aValue = a.vehicleNumber || '';
                bValue = b.vehicleNumber || '';
                break;
            case 'vehicleName':
                aValue = a.vehicleName || '';
                bValue = b.vehicleName || '';
                break;
            default:
                aValue = a.reportNumber || '';
                bValue = b.reportNumber || '';
        }
        
        // Handle comparison
        let comparison = 0;
        if (aValue > bValue) {
            comparison = 1;
        } else if (aValue < bValue) {
            comparison = -1;
        }
        
        // Apply sort order
        return currentSortOrder === 'descending' ? comparison * -1 : comparison;
    });
    
    // Re-render the table with sorted data
    renderTableData();
}

function renderTableData() {
    const tbody = document.getElementById('odrTableBody');
    
    if (tableData.length === 0) {
        // Show no data message
        tbody.innerHTML = `
            <tr class="no-data-row">
                <td colspan="12" class="no-data-cell">
                    <div class="no-data-content">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="12" cy="12" r="10"></circle>
                            <line x1="12" y1="8" x2="12" y2="12"></line>
                            <line x1="12" y1="16" x2="12.01" y2="16"></line>
                        </svg>
                        <h3>No Data Available</h3>
                        <p>No operator daily reports found. Start by creating a new report.</p>
                    </div>
                </td>
            </tr>
        `;
        return;
    }
    
    // Render table rows with data
    tbody.innerHTML = tableData.map((row, index) => `
        <tr>
            <td>
                <div class="action-buttons">
                    <button class="btn-action btn-view" onclick="viewReport('${row.id || index}')" title="View Report">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                            <circle cx="12" cy="12" r="3"></circle>
                        </svg>
                    </button>
                    <button class="btn-action btn-edit" onclick="editReport('${row.id || index}')" title="Edit Report">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path>
                            <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path>
                        </svg>
                    </button>
                </div>
            </td>
            <td>${row.reportNumber || 'N/A'}</td>
            <td>${row.startDate || 'N/A'}</td>
            <td>${row.vehicleName || row.vehicleNumber || 'N/A'}</td>
            <td>${getDayStatus(row.mon)}</td>
            <td>${getDayStatus(row.tue)}</td>
            <td>${getDayStatus(row.wed)}</td>
            <td>${getDayStatus(row.thu)}</td>
            <td>${getDayStatus(row.fri)}</td>
            <td>${getDayStatus(row.sat)}</td>
            <td>${getDayStatus(row.sun)}</td>
            <td>${getSubmittedStatus(row.submitted)}</td>
        </tr>
    `).join('');
}

function getDayStatus(day) {
    if (day === 'completed') return '<span class="status-badge status-success">✓</span>';
    if (day === 'pending') return '<span class="status-badge status-warning">⏳</span>';
    if (day === 'missed') return '<span class="status-badge status-danger">✗</span>';
    return '<span class="status-badge status-secondary">-</span>';
}

function getSubmittedStatus(submitted) {
    if (submitted) return '<span class="status-badge status-success">Submitted</span>';
    return '<span class="status-badge status-warning">Pending</span>';
}

// Placeholder functions for future backend integration
function viewReport(id) {
    showNotification(`Viewing report ${id}`, 'info');
    // TODO: Implement view report functionality
}

function editReport(id) {
    showNotification(`Editing report ${id}`, 'info');
    // TODO: Implement edit report functionality
}

// Function to load data from backend
async function loadTableData() {
    try {
        showNotification('Loading operator daily reports...', 'info');
        
        const response = await fetch('/inspection/get-operator-daily-reports');
        const result = await response.json();
        
        if (result.success) {
            tableData = result.data || [];
            showNotification(`Loaded ${tableData.length} operator daily reports`, 'success');
        } else {
            console.error('Error loading data:', result.message);
            showNotification('Error loading data: ' + result.message, 'error');
            
            // Fallback to sample data if backend fails
            loadSampleData();
        }
        
        // Apply current sorting and render
        applySortingToTable();
        
    } catch (error) {
        console.error('Error fetching operator daily reports:', error);
        showNotification('Error connecting to server. Using sample data.', 'warning');
        
        // Fallback to sample data
        loadSampleData();
    }
}

// Fallback function with sample data
function loadSampleData() {
    tableData = [
        {
            id: 1,
            reportNumber: 'ODR-001',
            startDate: '2024-01-15',
            vehicleNumber: 'VH-001',
            vehicleName: 'Truck Alpha',
            mon: 'completed',
            tue: 'completed',
            wed: 'pending',
            thu: 'pending',
            fri: 'pending',
            sat: 'missed',
            sun: 'missed',
            submitted: false
        },
        {
            id: 2,
            reportNumber: 'ODR-002',
            startDate: '2024-01-14',
            vehicleNumber: 'VH-002',
            vehicleName: 'Truck Beta',
            mon: 'completed',
            tue: 'completed',
            wed: 'completed',
            thu: 'completed',
            fri: 'completed',
            sat: 'completed',
            sun: 'completed',
            submitted: true
        },
        {
            id: 3,
            reportNumber: 'ODR-003',
            startDate: '2024-01-13',
            vehicleNumber: 'VH-003',
            vehicleName: 'Truck Gamma',
            mon: 'pending',
            tue: 'pending',
            wed: 'pending',
            thu: 'pending',
            fri: 'pending',
            sat: 'pending',
            sun: 'pending',
            submitted: false
        }
    ];
    
    // Apply current sorting and render
    applySortingToTable();
}

// Excel Export functionality
document.querySelector('.btn-primary').addEventListener('click', function() {
    showNotification('Exporting data to Excel...', 'info');
    // Add actual export functionality here
});

// Add refresh functionality
function refreshData() {
    showNotification('Refreshing data...', 'info');
    loadTableData();
}

// Add keyboard shortcut for refresh (Ctrl+R or F5)
document.addEventListener('keydown', function(event) {
    if ((event.ctrlKey && event.key === 'r') || event.key === 'F5') {
        event.preventDefault();
        refreshData();
    }
});

// Notification system
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
    notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
    notification.innerHTML = `
        ${message}
        <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => {
        if (notification.parentNode) {
            notification.remove();
        }
    }, 5000);
}

// Initialize page
document.addEventListener('DOMContentLoaded', function() {
    console.log('Operator Daily Reporting page loaded');
    
    // Add focus to search input on page load
    document.getElementById('searchInput').focus();
    
    // Check if we have server-side data first
    @if(isset($operatorReports) && !empty($operatorReports))
        // Use server-side data
        tableData = @json($operatorReports);
        console.log('Using server-side data:', tableData.length, 'reports');
        applySortingToTable();
    @else
        // Load data from API
        loadTableData();
    @endif
});
</script>
<style>
/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* Main Container */
.odr-container {
    min-height: 100vh;
}

/* Header Section */
.odr-header {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border-radius: 0;
    margin-bottom: 24px;
}

.header-content {
    padding: 32px;
    display: flex;
    justify-content: space-between;
    align-items: flex-start;
    gap: 24px;
}

.page-title h1 {
    color: #0f172a;
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0 0 8px 0;
}

.page-title p {
    color: #64748b;
    font-weight: 400;
    font-size: 16px;
    line-height: 1.5;
    margin: 0;
}

/* .sort-section{
    z-index: 211211121;
} */

.header-actions {
    display: flex;
    align-items: center;
    gap: 20px;
    flex-wrap: wrap;
}

/* Search Container */
.search-container {
    position: relative;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 16px;
    z-index: 2;
    color: #64748b;
    width: 16px;
    height: 16px;
}

.search-input {
    padding: 12px 16px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    height: 44px;
    width: 300px;
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    background: white;
    transition: all 0.2s ease;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: #9ca3af;
}

/* Action Buttons */
.action-buttons {
    display: flex;
    gap: 12px;
    align-items: center;
}

.btn-primary, .btn-secondary {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    text-decoration: none;
    transition: all 0.2s ease;
    cursor: pointer;
    box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05);
}

.btn-primary {
    background: #0055a5;
    color: white;
}

.btn-primary:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black;
}

.btn-secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
}

.btn-secondary:hover {
    background: linear-gradient(135deg, #4b5563 0%, #374151 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: white;
}

/* Table Section */
.table-section {
    background: white;
    border: 1px solid #e2e8f0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border-radius: 0;
    overflow: hidden;
}

.table-container {
    overflow-x: auto;
}

.odr-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.odr-table thead {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.odr-table th {
    padding: 16px 12px;
    text-align: left;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e2e8f0;
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    white-space: nowrap;
}

.odr-table td {
    padding: 16px 12px;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
    white-space: nowrap;
}

.odr-table tbody tr:hover {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

/* No Data Styling */
.no-data-cell {
    padding: 64px 24px !important;
    text-align: center;
}

.no-data-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.no-data-content svg {
    color: #9ca3af;
}

.no-data-content h3 {
    color: #374151;
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.no-data-content p {
    color: #6b7280;
    font-weight: 400;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

/* Modal Styling */
.modal-content {
    border: 1px solid #e2e8f0;
    border-radius: 0;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
    z-index: 1000222;
}

.modal-header {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.modal-title {
    color: #0f172a;
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.btn-close {
    background: none;
    border: none;
    font-size: 24px;
    color: #64748b;
    opacity: 1;
    transition: all 0.2s ease;
}

.btn-close:hover {
    color: #374151;
    transform: scale(1.1);
}

.modal-body {
    padding: 24px;
}

.sort-section {
    margin-bottom: 24px;
}

.section-title {
    color: #374151;
    font-weight: 600;
    font-size: 14px;
    margin-bottom: 12px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.radio-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.radio-item {
    display: flex;
    align-items: center;
    gap: 12px;
}

.radio-item input[type="radio"] {
    display: none;
}

.radio-item label {
    display: flex;
    align-items: center;
    gap: 12px;
    cursor: pointer;
    font-size: 14px;
    font-weight: 400;
    color: #374151;
    transition: all 0.2s ease;
}

.radio-custom {
    width: 18px;
    height: 18px;
    border: 2px solid #d1d5db;
    border-radius: 50%;
    position: relative;
    transition: all 0.2s ease;
}

.radio-item input[type="radio"]:checked + label .radio-custom {
    border-color: #3b82f6;
    background: #3b82f6;
}

.radio-item input[type="radio"]:checked + label .radio-custom::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 6px;
    height: 6px;
    background: white;
    border-radius: 50%;
}

.radio-item label:hover {
    color: #1e293b;
}

.radio-item label:hover .radio-custom {
    border-color: #3b82f6;
}

.modal-footer {
    background: #f8fafc;
    border-top: 1px solid #e2e8f0;
    padding: 16px 24px;
    display: flex;
    gap: 12px;
    justify-content: flex-end;
}

.btn-cancel, .btn-reset, .btn-apply {
    padding: 10px 20px;
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel {
    background: #f3f4f6;
    color: #374151;
}

.btn-cancel:hover {
    background: #e5e7eb;
    color: #1f2937;
}

.btn-reset {
    background: #fef3c7;
    color: #92400e;
}

.btn-reset:hover {
    background: #fde68a;
    color: #78350f;
}

.btn-apply {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: white;
}

.btn-apply:hover {
    background: linear-gradient(135deg, #2563eb 0%, #1e40af 100%);
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .odr-container {
        padding: 20px;
    }
    
    .header-content {
        padding: 24px;
    }
    
    .search-input {
        width: 250px;
    }
    
    .table-section {
        margin-top: 20px;
    }
}

@media (max-width: 992px) {
    .odr-container {
        padding: 16px;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
        padding: 20px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .header-actions {
        justify-content: space-between;
        flex-wrap: wrap;
        gap: 16px;
    }
    
    .search-input {
        width: 100%;
        max-width: 300px;
    }
    
    .action-buttons {
        flex-wrap: wrap;
        gap: 12px;
    }
    
    .btn-primary, .btn-secondary {
        flex: 1;
        min-width: 140px;
        justify-content: center;
    }
    
    .table-section {
        margin-top: 16px;
    }
    
    .odr-table {
        font-size: 13px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 14px 10px;
    }
}

@media (max-width: 768px) {
    .odr-container {
        padding: 0;
    }
    
    .odr-header {
        margin: 0 16px 16px 16px;
    }
    
    .header-content {
        padding: 16px;
        gap: 16px;
    }
    
    .page-title h1 {
        font-size: 24px;
        margin-bottom: 6px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .header-actions {
        flex-direction: column;
        align-items: stretch;
        gap: 16px;
    }
    
    .search-container {
        width: 100%;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px 12px 44px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .action-buttons {
        justify-content: center;
        flex-wrap: wrap;
        gap: 12px;
    }
    
    .btn-primary, .btn-secondary {
        flex: 1;
        justify-content: center;
        min-width: 140px;
        padding: 14px 20px;
        font-size: 14px;
        min-height: 48px;
    }
    
    .table-section {
        margin: 0 16px 16px 16px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .odr-table {
        font-size: 12px;
        min-width: 800px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 12px 8px;
        white-space: nowrap;
    }
    
    .odr-table th {
        font-size: 11px;
        text-transform: uppercase;
        letter-spacing: 0.05em;
    }
    
    .odr-table td {
        font-size: 12px;
    }
    
    .action-buttons {
        gap: 6px;
    }
    
    .btn-action {
        width: 28px;
        height: 28px;
        min-width: 28px;
    }
    
    .btn-action svg {
        width: 14px;
        height: 14px;
    }
    
    .status-badge {
        padding: 3px 6px;
        font-size: 10px;
        min-width: 20px;
        height: 20px;
    }
    
    .no-data-cell {
        padding: 48px 16px !important;
    }
    
    .no-data-content h3 {
        font-size: 16px;
    }
    
    .no-data-content p {
        font-size: 13px;
    }
    
    .no-data-content svg {
        width: 40px;
        height: 40px;
    }
    
    /* Modal Responsive */
    .modal-dialog {
        margin: 10px;
        max-width: calc(100% - 20px);
    }
    
    .modal-content {
        border-radius: 8px;
    }
    
    .modal-header {
        padding: 16px 20px;
    }
    
    .modal-title {
        font-size: 16px;
    }
    
    .modal-body {
        padding: 20px;
    }
    
    .sort-section {
        margin-bottom: 20px;
    }
    
    .section-title {
        font-size: 13px;
        margin-bottom: 10px;
    }
    
    .radio-item {
        gap: 10px;
    }
    
    .radio-item label {
        font-size: 13px;
    }
    
    .radio-custom {
        width: 16px;
        height: 16px;
    }
    
    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 5px;
        height: 5px;
    }
    
    .modal-footer {
        padding: 16px 20px;
        flex-direction: column;
        gap: 12px;
    }
    
    .btn-cancel, .btn-reset, .btn-apply {
        width: 100%;
        padding: 12px 20px;
        font-size: 14px;
        min-height: 44px;
    }
}

@media (max-width: 576px) {
    .odr-container {
        padding: 0;
    }
    
    .odr-header {
        margin: 0 12px 12px 12px;
    }
    
    .header-content {
        padding: 12px;
        gap: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .header-actions {
        gap: 12px;
    }
    
    .search-input {
        padding: 10px 14px 10px 40px;
        font-size: 16px;
        min-height: 44px;
    }
    
    .search-icon {
        width: 14px;
        height: 14px;
        left: 14px;
    }
    
    .action-buttons {
        gap: 8px;
    }
    
    .btn-primary, .btn-secondary {
        padding: 12px 16px;
        font-size: 13px;
        min-height: 44px;
        min-width: 120px;
    }
    
    .table-section {
        margin: 0 12px 12px 12px;
    }
    
    .odr-table {
        font-size: 11px;
        min-width: 700px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 10px 6px;
    }
    
    .odr-table th {
        font-size: 10px;
    }
    
    .odr-table td {
        font-size: 11px;
    }
    
    .btn-action {
        width: 24px;
        height: 24px;
        min-width: 24px;
    }
    
    .btn-action svg {
        width: 12px;
        height: 12px;
    }
    
    .status-badge {
        padding: 2px 4px;
        font-size: 9px;
        min-width: 18px;
        height: 18px;
    }
    
    .no-data-cell {
        padding: 40px 12px !important;
    }
    
    .no-data-content h3 {
        font-size: 15px;
    }
    
    .no-data-content p {
        font-size: 12px;
    }
    
    .no-data-content svg {
        width: 36px;
        height: 36px;
    }
    
    /* Modal Responsive */
    .modal-dialog {
        margin: 8px;
        max-width: calc(100% - 16px);
    }
    
    .modal-header {
        padding: 12px 16px;
    }
    
    .modal-title {
        font-size: 15px;
    }
    
    .modal-body {
        padding: 16px;
    }
    
    .sort-section {
        margin-bottom: 16px;
    }
    
    .section-title {
        font-size: 12px;
        margin-bottom: 8px;
    }
    
    .radio-item {
        gap: 8px;
    }
    
    .radio-item label {
        font-size: 12px;
    }
    
    .radio-custom {
        width: 14px;
        height: 14px;
    }
    
    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 4px;
        height: 4px;
    }
    
    .modal-footer {
        padding: 12px 16px;
        gap: 8px;
    }
    
    .btn-cancel, .btn-reset, .btn-apply {
        padding: 10px 16px;
        font-size: 13px;
        min-height: 40px;
    }
}

@media (max-width: 480px) {
    .odr-header {
        margin: 0 8px 8px 8px;
    }
    
    .header-content {
        padding: 10px;
        gap: 10px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 3px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .header-actions {
        gap: 10px;
    }
    
    .search-input {
        padding: 8px 12px 8px 36px;
        font-size: 16px;
        min-height: 40px;
    }
    
    .search-icon {
        width: 12px;
        height: 12px;
        left: 12px;
    }
    
    .action-buttons {
        gap: 6px;
    }
    
    .btn-primary, .btn-secondary {
        padding: 10px 14px;
        font-size: 12px;
        min-height: 40px;
        min-width: 100px;
    }
    
    .table-section {
        margin: 0 8px 8px 8px;
    }
    
    .odr-table {
        font-size: 10px;
        min-width: 600px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 8px 4px;
    }
    
    .odr-table th {
        font-size: 9px;
    }
    
    .odr-table td {
        font-size: 10px;
    }
    
    .btn-action {
        width: 22px;
        height: 22px;
        min-width: 22px;
    }
    
    .btn-action svg {
        width: 10px;
        height: 10px;
    }
    
    .status-badge {
        padding: 1px 3px;
        font-size: 8px;
        min-width: 16px;
        height: 16px;
    }
    
    .no-data-cell {
        padding: 32px 8px !important;
    }
    
    .no-data-content h3 {
        font-size: 14px;
    }
    
    .no-data-content p {
        font-size: 11px;
    }
    
    .no-data-content svg {
        width: 32px;
        height: 32px;
    }
    
    /* Modal Responsive */
    .modal-dialog {
        margin: 4px;
        max-width: calc(100% - 8px);
    }
    
    .modal-header {
        padding: 10px 12px;
    }
    
    .modal-title {
        font-size: 14px;
    }
    
    .modal-body {
        padding: 12px;
    }
    
    .sort-section {
        margin-bottom: 12px;
    }
    
    .section-title {
        font-size: 11px;
        margin-bottom: 6px;
    }
    
    .radio-item {
        gap: 6px;
    }
    
    .radio-item label {
        font-size: 11px;
    }
    
    .radio-custom {
        width: 12px;
        height: 12px;
    }
    
    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 3px;
        height: 3px;
    }
    
    .modal-footer {
        padding: 10px 12px;
        gap: 6px;
    }
    
    .btn-cancel, .btn-reset, .btn-apply {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 36px;
    }
}

@media (max-width: 360px) {
    .odr-header {
        margin: 0 6px 6px 6px;
    }
    
    .header-content {
        padding: 8px;
        gap: 8px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .header-actions {
        gap: 8px;
    }
    
    .search-input {
        padding: 6px 10px 6px 32px;
        font-size: 16px;
        min-height: 36px;
    }
    
    .search-icon {
        width: 10px;
        height: 10px;
        left: 10px;
    }
    
    .action-buttons {
        gap: 4px;
    }
    
    .btn-primary, .btn-secondary {
        padding: 8px 12px;
        font-size: 11px;
        min-height: 36px;
        min-width: 80px;
    }
    
    .table-section {
        margin: 0 6px 6px 6px;
    }
    
    .odr-table {
        font-size: 9px;
        min-width: 500px;
    }
    
    .odr-table th,
    .odr-table td {
        padding: 6px 3px;
    }
    
    .odr-table th {
        font-size: 8px;
    }
    
    .odr-table td {
        font-size: 9px;
    }
    
    .btn-action {
        width: 20px;
        height: 20px;
        min-width: 20px;
    }
    
    .btn-action svg {
        width: 8px;
        height: 8px;
    }
    
    .status-badge {
        padding: 1px 2px;
        font-size: 7px;
        min-width: 14px;
        height: 14px;
    }
    
    .no-data-cell {
        padding: 24px 6px !important;
    }
    
    .no-data-content h3 {
        font-size: 13px;
    }
    
    .no-data-content p {
        font-size: 10px;
    }
    
    .no-data-content svg {
        width: 28px;
        height: 28px;
    }
    
    /* Modal Responsive */
    .modal-dialog {
        margin: 2px;
        max-width: calc(100% - 4px);
    }
    
    .modal-header {
        padding: 8px 10px;
    }
    
    .modal-title {
        font-size: 13px;
    }
    
    .modal-body {
        padding: 10px;
    }
    
    .sort-section {
        margin-bottom: 10px;
    }
    
    .section-title {
        font-size: 10px;
        margin-bottom: 4px;
    }
    
    .radio-item {
        gap: 4px;
    }
    
    .radio-item label {
        font-size: 10px;
    }
    
    .radio-custom {
        width: 10px;
        height: 10px;
    }
    
    .radio-item input[type="radio"]:checked + label .radio-custom::after {
        width: 2px;
        height: 2px;
    }
    
    .modal-footer {
        padding: 8px 10px;
        gap: 4px;
    }
    
    .btn-cancel, .btn-reset, .btn-apply {
        padding: 6px 10px;
        font-size: 11px;
        min-height: 32px;
    }
}

/* Focus states for accessibility */
.btn-primary:focus,
.btn-secondary:focus,
.search-input:focus,
.btn-cancel:focus,
.btn-reset:focus,
.btn-apply:focus {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* Smooth animations */
* {
    transition: all 0.2s ease;
}

/* Icon improvements */
.btn-primary svg,
.btn-secondary svg {
    flex-shrink: 0;
}

/* ===== STATUS BADGES ===== */
.status-badge {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 4px 8px;
    border-radius: 0;
    font-size: 12px;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    min-width: 24px;
    height: 24px;
}

.status-success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: white;
}

.status-warning {
    /* background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); */
    color: black;
}

.status-danger {
    background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%);
    color: white;
}

.status-secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
    color: white;
}

/* ===== ACTION BUTTONS ===== */
.action-buttons {
    display: flex;
    gap: 8px;
    align-items: center;
}

.btn-action {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 32px;
    height: 32px;
    border: none;
    border-radius: 0;
    cursor: pointer;
    transition: all 0.2s ease;
    /* box-shadow: 0 1px 2px 0 rgba(0, 0, 0, 0.05); */
}

.btn-view {
    background: transparent;
    color: black;
}

/* .btn-view:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black;
} */

.btn-edit {
    background: transparent;
    color: black;
}

/* .btn-edit:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black;
} */
</style>
@endsection