@extends('welcome')

@section('title', 'Inspection')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="inspection-container">
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('inspection.inspection2') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Inspection Reports</h1>
            </div>
            <div class="header-right">
                <div class="search-container">
                    <div class="search-input-wrapper">
                        <i class="fas fa-search search-icon"></i>
                        <input type="text" class="search-input" placeholder="Search inspections..." id="searchInput" aria-label="Search inspections" autocomplete="off">
                    </div>
                </div>
                <button type="button" class="create-btn" onclick="refreshFromMongoDB()" style="background: #00B894; margin-right: 10px;" title="Refresh data from MongoDB">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
                <button type="button" class="create-btn" onclick="window.location.href='{{ route('inspection.inspection2') }}'">
                    <i class="fas fa-plus"></i>
                    New Inspection
                </button>
            </div>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Report Summary -->
        <div class="form-section" id="reportSummary" style="display: none;">
            <div class="request-number-display">
                <span class="request-number-label">Report Summary</span>
                <div class="summary-stats">
                    <span class="stat-item">
                        <span class="stat-number" id="totalReports">0</span>
                        <span class="stat-label">Total Reports</span>
                    </span>
                    <span class="stat-item">
                        <span class="stat-number" id="myReports" style="color: #28a745;">0</span>
                        <span class="stat-label">My Reports</span>
                    </span>
                    <span class="stat-item">
                        <span class="stat-number" id="otherReports">0</span>
                        <span class="stat-label">Other Reports</span>
                    </span>
                </div>
            </div>
        </div>

        <!-- Table Section -->
        <div class="form-container">
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-list"></i>
                    Inspection Reports
                    <button type="button" onclick="refreshFromMongoDB()" style="background: #00B894; color: white; border: none; padding: 8px 16px; border-radius: 5px; cursor: pointer; font-size: 14px; display: flex; align-items: center; gap: 6px;">
                        <i class="fas fa-sync-alt"></i>
                        Refresh Data
                    </button>
                </h2>
                
                <div class="table-container">
                    <table class="inspection-table" id="inspectionTable" role="table">
                        <thead>
                            <tr role="row">
                                <th scope="col" role="columnheader">S.No</th>
                                <th scope="col" role="columnheader">Action</th>
                                <th scope="col" role="columnheader">Report Number</th>
                                <th scope="col" role="columnheader">Date</th>
                                <th scope="col" role="columnheader">Vehicle</th>
                                <th scope="col" role="columnheader">Location</th>
                                <th scope="col" role="columnheader">Inspector</th>
                                <th scope="col" role="columnheader">Fault</th>
                                <th scope="col" role="columnheader">Corrective Action</th>
                                <th scope="col" role="columnheader">Linked Wo</th>
                            </tr>
                        </thead>
                        <tbody id="inspectionTableBody" role="rowgroup">
                            <!-- Data will be loaded dynamically via JavaScript -->
                        </tbody>
                    </table>
                </div>
            </div>
        </div>

        <!-- Pagination Section -->
        <div class="form-container" id="paginationSection" style="display: none;">
            <div class="form-section">
                <div class="pagination-container">
                    <div class="pagination-info">
                        <span id="paginationInfo">Showing 1-10 of 0 records</span>
                    </div>
                    <div class="pagination-controls">
                        <button class="pagination-btn" id="prevPageBtn" onclick="changePage(-1)" disabled>
                            <i class="fas fa-chevron-left"></i>
                            Previous
                        </button>
                        <div class="pagination-pages" id="paginationPages">
                            <!-- Page numbers will be generated here -->
                        </div>
                        <button class="pagination-btn" id="nextPageBtn" onclick="changePage(1)" disabled>
                            Next
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    </div>
                </div>
            </div>
        </div>

        <!-- Loading Spinner -->
        <div class="form-container" id="loadingSpinner" style="display: none;">
            <div class="form-section text-center">
                <div class="loading-content">
                    <div class="spinner-border text-primary" role="status">
                        <span class="visually-hidden">Loading...</span>
                    </div>
                    <p class="mt-2">Fetching inspection data...</p>
                </div>
            </div>
        </div>

        <!-- Error Message -->
        <div class="form-container" id="errorMessage" style="display: none;">
            <div class="form-section">
                <div class="error-content">
                    <i class="fas fa-exclamation-circle"></i>
                    <span id="errorText"></span>
                    <button type="button" class="btn-close" onclick="hideError()">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            </div>
        </div>

        <!-- Debug Section (Hidden by default) -->
        <div class="form-container" id="debugSection" style="display: none;">
            <div class="form-section">
                <h2 class="section-title">
                    <i class="fas fa-bug"></i>
                    Debug Information
                    <button type="button" class="btn-close" onclick="toggleDebug()">
                        <i class="fas fa-times"></i>
                    </button>
                </h2>
                <div class="debug-content">
                    <div class="debug-row">
                        <div class="debug-column">
                            <h6>Data Summary</h6>
                            <ul class="debug-list">
                                <li><strong>Total Reports:</strong> <span id="debugTotalReports">-</span></li>
                                <li><strong>Data Source:</strong> Firebase GlobalInspectionReport</li>
                                <li><strong>Last Updated:</strong> <span id="debugLastUpdated">-</span></li>
                            </ul>
                        </div>
                        <div class="debug-column">
                            <h6>Connection Status</h6>
                            <ul class="debug-list">
                                <li><strong>Firebase URL:</strong> <code>{{ url('/') }}/test-inspection</code></li>
                                <li><strong>Inspection Test:</strong> <code>{{ url('/inspection/test-data') }}</code></li>
                                <li><strong>Status:</strong> <span id="debugStatus" class="status-badge">Connected</span></li>
                            </ul>
                        </div>
                    </div>
                    <div class="debug-actions">
                        <button class="btn btn-sm btn-outline-info" onclick="testConnection()">Test Connection</button>
                        <button class="btn btn-sm btn-outline-warning" onclick="clearCache()">Clear Cache</button>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script>
(function() {
    'use strict';
    
    // Configuration object
    const CONFIG = {
        selectors: {
            searchInput: '#searchInput',
            tableBody: '#inspectionTableBody',
            debugSection: '#debugSection',
            debugTotalReports: '#debugTotalReports',
            debugLastUpdated: '#debugLastUpdated',
            debugStatus: '#debugStatus',
            excelExportBtn: '#excelExportBtn'
        },
        messages: {
            loading: 'Loading inspection reports...',
            error: 'Failed to load inspection reports. Please check your connection and try refreshing.',
            excelExport: 'Excel export functionality would open here. This will allow you to export inspection reports data in Excel format (.xlsx).'
        },
        timing: {
            loadingTimeout: 3000,
            notificationTimeout: 5000
        },
        endpoints: {
            testData: '/inspection/test-data'
        }
    };
    
    // State management
    let state = {
        isLoading: false,
        hasData: false
    };
    
    // DOM element caching
    let elements = {};
    
    // Initialize DOM elements
    function initElements() {
        elements = {
            searchInput: document.querySelector(CONFIG.selectors.searchInput),
            tableBody: document.querySelector(CONFIG.selectors.tableBody),
            debugSection: document.querySelector(CONFIG.selectors.debugSection),
            debugTotalReports: document.querySelector(CONFIG.selectors.debugTotalReports),
            debugLastUpdated: document.querySelector(CONFIG.selectors.debugLastUpdated),
            debugStatus: document.querySelector(CONFIG.selectors.debugStatus),
            excelExportBtn: document.querySelector(CONFIG.selectors.excelExportBtn)
        };
    }
    
    // Main initialization
    function init() {
        initElements();
        checkDataAvailability();
        initEventListeners();
        initializeDropdowns();
    }
    
    // Check if we have data to display
    function checkDataAvailability() {
        state.hasData = elements.tableBody && elements.tableBody.children.length > 0;
        
        if (!state.hasData) {
            showLoading();
            setTimeout(() => {
                if (!state.hasData) {
                    showError(CONFIG.messages.error);
                }
            }, CONFIG.timing.loadingTimeout);
        }
    }
    
    // Initialize event listeners
    function initEventListeners() {
        if (elements.searchInput) {
            elements.searchInput.addEventListener('input', handleSearch);
            elements.searchInput.addEventListener('focus', function() {
                this.select();
            });
        }
        
        // Keyboard shortcut for debug (Ctrl+Shift+D)
        document.addEventListener('keydown', function(e) {
            if (e.ctrlKey && e.shiftKey && e.key === 'D') {
                e.preventDefault();
                toggleDebug();
            }
        });
        
        // Refresh button
        const refreshBtn = document.getElementById('refreshBtn');
        if (refreshBtn) {
            refreshBtn.addEventListener('click', function() {
                console.log('Refreshing inspection data...');
                loadInspectionData();
            });
        }
        
        // Excel export button
        if (elements.excelExportBtn) {
            elements.excelExportBtn.addEventListener('click', handleExcelExport);
        }
    }
    
    // Handle search functionality
    function handleSearch() {
        const searchTerm = this.value.toLowerCase();
        const tableRows = document.querySelectorAll('#inspectionTableBody tr');
        
        tableRows.forEach(row => {
            const text = row.textContent.toLowerCase();
            if (text.includes(searchTerm)) {
                row.style.display = '';
            } else {
                row.style.display = 'none';
            }
        });
    }
    
    // Handle Excel export
    function handleExcelExport(e) {
        e.preventDefault();
        e.stopPropagation();
        
        showSuccessAlert(
            'Excel Export',
            CONFIG.messages.excelExport,
            function() {
                console.log('Excel export alert confirmed');
            }
        );
    }
    
    // Refresh data function
    function refreshData() {
        showLoading();
        location.reload();
    }
    
    // Wait for DOM to be fully loaded
    document.addEventListener('DOMContentLoaded', init);
    
    // Helper function to safely parse dates
    function safeParseDate(dateString, source = 'unknown') {
        if (!dateString) return null;
        
        try {
            // Handle different date formats
            let date;
            
            if (typeof dateString === 'string') {
                // Check if it's a numeric timestamp string
                if (/^\d+$/.test(dateString)) {
                    date = new Date(parseInt(dateString));
                } else {
                    // It's an ISO string or other format
                    date = new Date(dateString);
                }
            } else if (typeof dateString === 'number') {
                date = new Date(dateString);
            } else {
                return null;
            }
            
            // Validate the date
            if (isNaN(date.getTime()) || date.getTime() <= 0) {
                console.warn(`Invalid date from ${source}:`, dateString);
                return null;
            }
            
            return date;
        } catch (error) {
            console.error(`Error parsing date from ${source}:`, error, dateString);
            return null;
        }
    }

    // Load inspection data from MongoDB via Laravel API
    async function loadInspectionData() {
        try {
            showLoading();
            console.log('═══════════════════════════════════════════════════════');
            console.log('📥 LOADING INSPECTION DATA FROM MONGODB');
            console.log('═══════════════════════════════════════════════════════');
            
            // Add cache busting parameter to ensure fresh data
            const timestamp = new Date().getTime();
            const response = await fetch(`/api/inspections/all?_=${timestamp}`);
            const result = await response.json();
            
            console.log('✅ MongoDB API Response Status:', response.status);
            console.log('✅ MongoDB API Response:', result);
            
            if (result.success && result.data && Array.isArray(result.data)) {
                let allInspections = result.data;
                
                console.log(`📊 Found ${allInspections.length} total inspections from MongoDB`);
                console.log('📊 Raw inspections data:', allInspections.slice(0, 3)); // Log first 3 for debugging
                
                // Validate and clean inspections data
                allInspections = allInspections.filter(inspection => {
                    // Must have _id field
                    if (!inspection._id) {
                        console.warn('⚠️ Skipping inspection without _id:', inspection);
                        return false;
                    }
                    return true;
                });
                
                console.log(`📊 After validation: ${allInspections.length} valid inspections`);
                
                // Ensure each inspection has an 'id' field for compatibility
                allInspections = allInspections.map(inspection => {
                    // Always prefer _id for consistency
                    inspection.id = inspection._id;
                    return inspection;
                });
                
                // Log all inspection IDs for debugging
                console.log('📊 All inspection IDs in database:');
                allInspections.forEach((ins, idx) => {
                    console.log(`  [${idx + 1}] _id="${ins._id}" requestNumber="${ins.requestNumber || 'N/A'}"`);
                });
                
                // Sort by createdAt (newest first)
                allInspections.sort((a, b) => {
                    const dateA = new Date(a.createdAt || 0).getTime();
                    const dateB = new Date(b.createdAt || 0).getTime();
                    return dateB - dateA;
                });
                
                state.allData = allInspections;
                state.filteredData = [...allInspections];
                state.totalRecords = allInspections.length;
                state.totalPages = Math.ceil(state.totalRecords / 10);
                state.currentPage = 1;
                
                console.log(`✅ State updated with ${state.totalRecords} inspection reports`);
                console.log('═══════════════════════════════════════════════════════');
                
                if (state.totalRecords > 0) {
                    state.hasData = true;
                    renderTable();
                    renderPagination();
                    updateReportSummary(allInspections);
                    hideLoading();
                } else {
                    showNoData();
                }
            } else {
                console.log('❌ No data found in MongoDB response');
                showNoData();
            }
        } catch (error) {
            console.error('❌ Error loading inspection data:', error);
            showError('Failed to load inspection reports. Please check your connection and try refreshing.');
        }
    }
    
    // Render table with current page data
    function renderTable() {
        if (!elements.tableBody) return;
        
        const startIndex = (state.currentPage - 1) * 10;
        const endIndex = startIndex + 10;
        const pageData = state.filteredData.slice(startIndex, endIndex);
        
        if (pageData.length === 0) {
            showNoData();
            return;
        }
        
        elements.tableBody.innerHTML = pageData.map((inspection, index) => {
            console.log('Rendering inspection:', inspection);
            
            // Calculate S.No based on newest first (highest numbers first)
            // Total records - (current page - 1) * 10 - index
            const serialNumber = state.totalRecords - ((state.currentPage - 1) * 10 + index);
            
            // Extract data from the correct structure
            const createdAt = inspection.step1?.createdAt || inspection.createdAt || (inspection.timestamp ? new Date(parseInt(inspection.timestamp)).toISOString() : new Date().toISOString());
            const requestNumber = inspection.step1?.requestNumber || inspection.requestNumber || 'N/A';
            
            // Fix equipment data extraction - handle both object and string formats
            let equipmentName = 'N/A';
            let equipmentStatus = 'N/A';
            
            // Extract equipment name
            if (inspection.step1?.selectedEquipment) {
                if (typeof inspection.step1.selectedEquipment === 'object') {
                    equipmentName = inspection.step1.selectedEquipment.equipmentName || 
                                   inspection.step1.selectedEquipment.name || 
                                   inspection.step1.selectedEquipment.vehicleNumber ||
                                   'N/A';
                } else {
                    equipmentName = inspection.step1.selectedEquipment;
                }
            } else if (inspection.equipmentName) {
                if (typeof inspection.equipmentName === 'object') {
                    equipmentName = inspection.equipmentName.name || inspection.equipmentName.equipmentName || 'N/A';
                } else {
                    equipmentName = inspection.equipmentName;
                }
            }
            
            // Extract equipment status
            if (inspection.step3?.equipmentStatus) {
                equipmentStatus = inspection.step3.equipmentStatus;
            } else if (inspection.step1?.selectedEquipment && typeof inspection.step1.selectedEquipment === 'object') {
                equipmentStatus = inspection.step1.selectedEquipment.status || 'N/A';
            } else if (inspection.step1?.selectedEquipment_type) {
                if (typeof inspection.step1.selectedEquipment_type === 'object') {
                    equipmentStatus = inspection.step1.selectedEquipment_type.type || inspection.step1.selectedEquipment_type.status || 'N/A';
                } else {
                    equipmentStatus = inspection.step1.selectedEquipment_type;
                }
            } else if (inspection.equipmentStatus) {
                if (typeof inspection.equipmentStatus === 'object') {
                    equipmentStatus = inspection.equipmentStatus.type || inspection.equipmentStatus.status || 'N/A';
                } else {
                    equipmentStatus = inspection.equipmentStatus;
                }
            }
            
            // Extract location - combine country and project if available
            let location = 'N/A';
            if (inspection.step1?.country && inspection.step1?.project) {
                location = `${inspection.step1.country} - ${inspection.step1.project}`;
            } else if (inspection.step1?.location) {
                location = inspection.step1.location;
            } else if (inspection.location) {
                location = inspection.location;
            }
            
            const inspector = inspection.step1?.inspector || inspection.inspector || 'N/A';
            // Fix faulty items count extraction - check multiple possible locations
            let faultyItems = 0;
            
            // Check step2.faultyItems first
            if (inspection.step2?.faultyItems !== undefined) {
                faultyItems = inspection.step2.faultyItems;
            }
            // Check step2.faultyItemsCount
            else if (inspection.step2?.faultyItemsCount !== undefined) {
                faultyItems = inspection.step2.faultyItemsCount;
            }
            // Check root level faultyItems
            else if (inspection.faultyItems !== undefined) {
                faultyItems = inspection.faultyItems;
            }
            // Check root level faultyItemsCount
            else if (inspection.faultyItemsCount !== undefined) {
                faultyItems = inspection.faultyItemsCount;
            }
            // Check if it's in the main data object
            else if (inspection[inspection.requestNumber]?.step2?.faultyItems !== undefined) {
                faultyItems = inspection[inspection.requestNumber].step2.faultyItems;
            }
            else if (inspection[inspection.requestNumber]?.step2?.faultyItemsCount !== undefined) {
                faultyItems = inspection[inspection.requestNumber].step2.faultyItemsCount;
            }
            
            // Ensure it's a number
            faultyItems = parseInt(faultyItems) || 0;
            
            // Debug logging for faulty items count (moved after date formatting)
            if (inspection.requestNumber && inspection.requestNumber.includes('ZAM')) {
                console.log(`Debug data for ${inspection.requestNumber}:`, {
                    step2_faultyItems: inspection.step2?.faultyItems,
                    step2_faultyItemsCount: inspection.step2?.faultyItemsCount,
                    root_faultyItems: inspection.faultyItems,
                    root_faultyItemsCount: inspection.faultyItemsCount,
                    final_count: faultyItems,
                    inspection_keys: Object.keys(inspection),
                    step2_keys: inspection.step2 ? Object.keys(inspection.step2) : 'No step2',
                    step3_keys: inspection.step3 ? Object.keys(inspection.step3) : 'No step3'
                });
            }
            const fault = inspection.step2?.fault || inspection.fault || 'No faults found';
            
            // Extract corrective action - handle both object and string formats
            let correctiveAction = 'No actions specified';
            if (inspection.correctiveAction) {
                if (typeof inspection.correctiveAction === 'object') {
                    correctiveAction = inspection.correctiveAction.exists ? 'Action Required' : 'No actions specified';
                } else {
                    correctiveAction = inspection.correctiveAction;
                }
            } else if (inspection.step2?.correctiveAction) {
                if (typeof inspection.step2.correctiveAction === 'object') {
                    correctiveAction = inspection.step2.correctiveAction.exists ? 'Action Required' : 'No actions specified';
                } else {
                    correctiveAction = inspection.step2.correctiveAction;
                }
            }
            
            const linkedWo = inspection.step2?.linkedWo || inspection.linkedWo || 'N/A';
            
            // Format date from multiple possible timestamp sources
            let formattedDate = 'Invalid Date';
            let dateToFormat = null;
            let dateSource = 'none';
            
            // Check multiple timestamp sources in order of preference using safe parsing
            if (inspection.step2?.timestamp) {
                dateToFormat = safeParseDate(inspection.step2.timestamp, 'step2.timestamp');
                dateSource = 'step2.timestamp';
            } else if (inspection.step3?.createdAt) {
                dateToFormat = safeParseDate(inspection.step3.createdAt, 'step3.createdAt');
                dateSource = 'step3.createdAt';
            } else if (inspection.timestamp) {
                dateToFormat = safeParseDate(inspection.timestamp, 'root.timestamp');
                dateSource = 'root.timestamp';
            } else if (inspection.step1?.createdAt) {
                dateToFormat = safeParseDate(inspection.step1.createdAt, 'step1.createdAt');
                dateSource = 'step1.createdAt';
            } else if (inspection.createdAt) {
                dateToFormat = safeParseDate(inspection.createdAt, 'root.createdAt');
                dateSource = 'root.createdAt';
            }
            
            // Format the date if we have a valid date (using day/month/year format with time as per user preference)
            if (dateToFormat) {
                try {
                    // Format as DD/MM/YYYY HH:MM
                    const day = dateToFormat.getDate().toString().padStart(2, '0');
                    const month = (dateToFormat.getMonth() + 1).toString().padStart(2, '0');
                    const year = dateToFormat.getFullYear();
                    const hours = dateToFormat.getHours().toString().padStart(2, '0');
                    const minutes = dateToFormat.getMinutes().toString().padStart(2, '0');
                    
                    formattedDate = `${day}/${month}/${year} ${hours}:${minutes}`;
                } catch (error) {
                    console.error(`Error formatting date from ${dateSource}:`, error, dateToFormat);
                    formattedDate = 'Invalid Date';
                }
            }
            
            // Debug logging for date issues
            if (formattedDate === 'Invalid Date' && inspection.requestNumber && inspection.requestNumber.includes('ZAM')) {
                console.log(`Date parsing failed for ${inspection.requestNumber}:`, {
                    step2_timestamp: inspection.step2?.timestamp,
                    step3_createdAt: inspection.step3?.createdAt,
                    root_timestamp: inspection.timestamp,
                    step1_createdAt: inspection.step1?.createdAt,
                    root_createdAt: inspection.createdAt,
                    dateSource: dateSource,
                    dateToFormat: dateToFormat,
                    inspection_keys: Object.keys(inspection)
                });
            }
            
            // Additional debug logging for successful date formatting
            if (inspection.requestNumber && inspection.requestNumber.includes('ZAM')) {
                console.log(`Date formatting success for ${inspection.requestNumber}:`, {
                    dateSource: dateSource,
                    formattedDate: formattedDate,
                    dateToFormat: dateToFormat
                });
            }
            
            // Check if this is the user's report for highlighting
            const isUserReport = inspection.adminUID === 'OKNMRqRKmrPz96Izvl9OYEzeTFn2';
            const rowClass = isUserReport ? 'table-row user-report' : 'table-row';
            
            return `
                <tr class="${rowClass}" role="row">
                    <td class="serial-number-cell" role="cell">
                        <span class="serial-number">${serialNumber}</span>
                    </td>
                    <td class="action-cell" role="cell">
                        <div class="dropdown">
                            <button class="action-btn" type="button" id="actionDropdown${inspection.id}" data-bs-toggle="dropdown" aria-expanded="false" aria-label="Actions for inspection ${inspection.id}">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                    <circle cx="12" cy="12" r="3"></circle>
                                    <path d="M12 1v6m0 6v6m11-7h-6m-6 0H1"></path>
                                </svg>
                                Actions
                            </button>
                            <ul class="dropdown-menu" aria-labelledby="actionDropdown${inspection.id}" role="menu">
                                <li role="none">
                                    <a class="dropdown-item" href="javascript:void(0)" onclick="event.preventDefault(); viewInspection('${inspection.id}'); return false;" role="menuitem" aria-label="View inspection ${inspection.id}">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                            <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                                            <circle cx="12" cy="12" r="3"></circle>
                                        </svg>
                                        View
                                    </a>
                                </li>
                                <li role="none">
                                    <a class="dropdown-item" href="javascript:void(0)" onclick="event.preventDefault(); shareInspection('${inspection.id}'); return false;" role="menuitem" aria-label="Share inspection ${inspection.id}">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                            <circle cx="18" cy="5" r="3"></circle>
                                            <circle cx="6" cy="12" r="3"></circle>
                                            <circle cx="18" cy="19" r="3"></circle>
                                            <line x1="8.59" y1="13.51" x2="15.42" y2="17.49"></line>
                                            <line x1="15.41" y1="6.51" x2="8.59" y2="10.49"></line>
                                        </svg>
                                        Share
                                    </a>
                                </li>
                                <li role="none">
                                    <a class="dropdown-item" href="javascript:void(0)" onclick="event.preventDefault(); createWorkorder('${inspection.id}'); return false;" role="menuitem" aria-label="Create workorder for inspection ${inspection.id}">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                            <path d="M14.7 6.3a1 1 0 0 0 0 1.4l1.6 1.6a1 1 0 0 0 1.4 0l3.77-3.77a6 6 0 0 1-7.94 7.94l-6.91 6.91a2.12 2.12 0 0 1-3-3l6.91-6.91a6 6 0 0 1 7.94-7.94l-3.76 3.76z"></path>
                                        </svg>
                                        Create Workorder
                                    </a>
                                </li>
                                <li role="none">
                                    <a class="dropdown-item" href="javascript:void(0)" onclick="event.preventDefault(); correctiveAction('${inspection.id}'); return false;" role="menuitem" aria-label="Corrective action for inspection ${inspection.id}">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                                            <polyline points="20,6 9,17 4,12"></polyline>
                                        </svg>
                                        Corrective Action
                                    </a>
                                </li>
                                <li>
                                    <a class="dropdown-item" href="javascript:void(0)" onclick="event.preventDefault(); rootCauseAnalysis('${inspection.id}'); return false;">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M9 11H5a2 2 0 0 0-2 2v3c0 1.1.9 2 2 2h4m0-7v7m0-7h10a2 2 0 0 1 2 2v3c0 1.1-.9 2-2 2H9m0-7V9a2 2 0 0 1 2-2h2a2 2 0 0 1 2 2v2"></path>
                                        </svg>
                                        Root Cause Analysis
                                    </a>
                                </li>
                                <li>
                                    <a class="dropdown-item" href="javascript:void(0)" onclick="event.preventDefault(); viewPdf('${inspection.id}'); return false;">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                                            <polyline points="14,2 14,8 20,8"></polyline>
                                        </svg>
                                        View PDF
                                    </a>
                                </li>
                                <li>
                                    <a class="dropdown-item" href="javascript:void(0)" onclick="event.preventDefault(); archiveInspection('${inspection.id}'); return false;">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <polyline points="21,8 21,21 3,21 3,8"></polyline>
                                            <rect x="1" y="3" width="22" height="5"></rect>
                                            <line x1="10" y1="12" x2="14" y2="12"></line>
                                        </svg>
                                        Archive
                                    </a>
                                </li>
                                <li class="dropdown-divider"></li>
                                <li>
                                    <a class="dropdown-item danger" href="javascript:void(0)" onclick="event.preventDefault(); deleteInspection('${inspection._id || inspection.id}', '${requestNumber}'); return false;">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <polyline points="3,6 5,6 21,6"></polyline>
                                            <path d="M19,6v14a2,2,0,0,1-2,2H7a2,2,0,0,1-2-2V6M8,6V4a2,2,0,0,1,2-2h4a2,2,0,0,1,2,2V6"></path>
                                        </svg>
                                        Delete
                                    </a>
                                </li>
                            </ul>
                        </div>
                    </td>
                    <td class="report-number">
                        <span class="report-id">${requestNumber}</span>
                        ${isUserReport ? '<br><small style="color: #16a34a; font-weight: bold;">✓ MY REPORT</small>' : ''}
                    </td>
                    <td class="date-cell">${formattedDate}</td>
                    <td class="vehicle-cell">
                        <div class="vehicle-info">
                            <span class="vehicle-name">${equipmentName}</span>
                            <span class="vehicle-status">${equipmentStatus}</span>
                        </div>
                    </td>
                    <td class="location-cell">${location}</td>
                    <td class="inspector-cell">${inspector}</td>
                    <td class="fault-cell">
                        ${faultyItems > 0 ? `
                            <div class="fault-info">
                                <span class="fault-badge">${faultyItems} faults</span>
                                ${fault !== 'No faults found' ? `<span class="fault-details">${fault.length > 50 ? fault.substring(0, 50) + '...' : fault}</span>` : ''}
                            </div>
                        ` : '<span class="no-faults">No faults</span>'}
                    </td>
                    <td class="action-cell">
                        ${correctiveAction !== 'No actions specified' ? 
                            (inspection.correctiveAction?.pdfDownloadUrl ? 
                                `<a href="${inspection.correctiveAction.pdfDownloadUrl}" target="_blank" class="corrective-action-link">
                                    <span class="corrective-action">Action Required</span>
                                    <i class="fas fa-external-link-alt" style="margin-left: 5px; font-size: 10px;"></i>
                                </a>` : 
                                `<span class="corrective-action">${correctiveAction.length > 50 ? correctiveAction.substring(0, 50) + '...' : correctiveAction}</span>`) : 
                            '<span class="no-action">No actions</span>'}
                    </td>
                    <td class="linked-cell">
                        ${linkedWo !== 'N/A' ? 
                            `<span class="linked-badge">${linkedWo}</span>` : 
                            '<span class="not-linked">Not linked</span>'}
                    </td>
                </tr>
            `;
        }).join('');
        
        // Show pagination if we have data
        if (state.totalRecords > 10) {
            document.getElementById('paginationSection').style.display = 'block';
        } else {
            document.getElementById('paginationSection').style.display = 'none';
        }
    }
    
    // Render pagination controls
    function renderPagination() {
        const paginationInfo = document.getElementById('paginationInfo');
        const paginationPages = document.getElementById('paginationPages');
        const prevPageBtn = document.getElementById('prevPageBtn');
        const nextPageBtn = document.getElementById('nextPageBtn');
        
        if (!paginationInfo || !paginationPages || !prevPageBtn || !nextPageBtn) return;
        
        const startRecord = (state.currentPage - 1) * 10 + 1;
        const endRecord = Math.min(state.currentPage * 10, state.totalRecords);
        
        paginationInfo.textContent = `Showing ${startRecord}-${endRecord} of ${state.totalRecords} records`;
        
        // Update previous/next buttons
        prevPageBtn.disabled = state.currentPage === 1;
        nextPageBtn.disabled = state.currentPage === state.totalPages;
        
        // Generate page numbers
        const pages = [];
        const maxVisible = 5;
        let startPage = Math.max(1, state.currentPage - Math.floor(maxVisible / 2));
        let endPage = Math.min(state.totalPages, startPage + maxVisible - 1);
        
        if (endPage - startPage + 1 < maxVisible) {
            startPage = Math.max(1, endPage - maxVisible + 1);
        }
        
        for (let i = startPage; i <= endPage; i++) {
            pages.push(`
                <button class="pagination-page ${i === state.currentPage ? 'active' : ''}" 
                        onclick="goToPage(${i})" 
                        ${i === state.currentPage ? 'disabled' : ''}>
                    ${i}
                </button>
            `);
        }
        
        paginationPages.innerHTML = pages.join('');
    }
    
    // Update report summary
    function updateReportSummary(allInspections) {
        const reportSummary = document.getElementById('reportSummary');
        const totalReports = document.getElementById('totalReports');
        const myReports = document.getElementById('myReports');
        const otherReports = document.getElementById('otherReports');
        
        if (reportSummary && totalReports && myReports && otherReports) {
            const targetUID = 'OKNMRqRKmrPz96Izvl9OYEzeTFn2';
            const myReportsCount = allInspections.filter(inspection => inspection.adminUID === targetUID).length;
            const otherReportsCount = allInspections.length - myReportsCount;
            
            totalReports.textContent = allInspections.length;
            myReports.textContent = myReportsCount;
            otherReports.textContent = otherReportsCount;
            
            reportSummary.style.display = 'block';
        }
    }
    
    // Show no data message
    function showNoData() {
        if (!elements.tableBody) return;
        
        elements.tableBody.innerHTML = `
            <tr class="no-data-row">
                <td colspan="10" class="no-data-cell">
                    <div class="no-data-content">
                        <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="12" cy="12" r="10"></circle>
                            <line x1="12" y1="8" x2="12" y2="12"></line>
                            <line x1="12" y1="16" x2="12.01" y2="16"></line>
                        </svg>
                        <h3>No Data Available</h3>
                        <p>No inspection reports found. Start by creating a new inspection.</p>
                    </div>
                </td>
            </tr>
        `;
        
        document.getElementById('paginationSection').style.display = 'none';
        hideLoading();
    }
    
    // Pagination functions
    function changePage(direction) {
        const newPage = state.currentPage + direction;
        if (newPage >= 1 && newPage <= state.totalPages) {
            state.currentPage = newPage;
            renderTable();
            renderPagination();
        }
    }
    
    function goToPage(page) {
        if (page >= 1 && page <= state.totalPages) {
            state.currentPage = page;
            renderTable();
            renderPagination();
        }
    }
    
    // Update state management
    state = {
        isLoading: false,
        hasData: false,
        allData: [],
        filteredData: [],
        currentPage: 1,
        totalPages: 0,
        totalRecords: 0
    };
    
    // Update init function to load data
    function init() {
        initElements();
        initEventListeners();
        initializeDropdowns();
        loadInspectionData();
    }

    // Helper function to open PDF for inspection (MongoDB version)
    function openInspectionPdf(inspectionId, actionName = 'viewing') {
        console.log(`📄 ${actionName} inspection:`, inspectionId);
        
        // Find the inspection data from MongoDB
        const inspection = state.allData.find(ins => ins._id === inspectionId || ins.id === inspectionId);
        if (!inspection) {
            showNotification('Inspection not found', 'error');
            return;
        }
        
        // Check if PDF URL is available in the inspection data from MongoDB
        const pdfUrl = inspection.step3?.pdfDownloadUrl;
        
        console.log('📋 Inspection data:', inspection);
        console.log('📎 PDF URL:', pdfUrl);
        
        if (pdfUrl) {
            // Use Laravel proxy endpoint for inline PDF viewing
            // This fixes the "download instead of view" issue by serving with proper headers
            const proxyUrl = `/api/inspections/view-pdf?path=${encodeURIComponent(pdfUrl)}`;
            
            console.log('📎 Original PDF URL:', pdfUrl);
            console.log('🔗 Using Laravel proxy URL:', proxyUrl);
            console.log('✅ Opening PDF in modal via proxy');
            
            openPdfModal(proxyUrl, inspection.requestNumber || inspectionId);
        } else {
            console.log('⚠️ No PDF URL found in inspection data');
            showNotification('PDF not available for this inspection yet', 'warning');
        }
    }
    
    // Function to open PDF in modal popup
    function openPdfModal(pdfUrl, reportNumber) {
        // Create modal if it doesn't exist
        let pdfModal = document.getElementById('pdfViewerModal');
        if (!pdfModal) {
            pdfModal = document.createElement('div');
            pdfModal.id = 'pdfViewerModal';
            pdfModal.className = 'pdf-modal';
            pdfModal.innerHTML = `
                <div class="pdf-modal-content">
                    <div class="pdf-modal-header">
                        <h3>📄 Inspection Report PDF - ${reportNumber}</h3>
                        <button onclick="closePdfModal()" class="pdf-close-btn">&times;</button>
                    </div>
                    <div class="pdf-modal-body">
                        <iframe id="pdfIframe" style="width: 100%; height: 100%; border: none;"></iframe>
                    </div>
                    <div class="pdf-modal-footer">
                        <button onclick="downloadPdf('${pdfUrl}', '${reportNumber}')" class="btn-download-pdf">
                            <i class="fas fa-download"></i> Download PDF
                        </button>
                        <button onclick="closePdfModal()" class="btn-close-pdf">Close</button>
                    </div>
                </div>
            `;
            document.body.appendChild(pdfModal);
        }
        
        // Set PDF URL in iframe
        const iframe = document.getElementById('pdfIframe');
        if (iframe) {
            iframe.src = pdfUrl;
            console.log('✅ PDF loaded in iframe:', pdfUrl);
        }
        
        // Show modal
        pdfModal.style.display = 'flex';
        document.body.style.overflow = 'hidden';
    }
    
    // Function to close PDF modal
    function closePdfModal() {
        const pdfModal = document.getElementById('pdfViewerModal');
        if (pdfModal) {
            pdfModal.style.display = 'none';
            document.body.style.overflow = 'auto';
        }
    }
    
    // Function to download PDF
    function downloadPdf(pdfUrl, reportNumber) {
        console.log('📥 Downloading PDF:', pdfUrl);
        
        const link = document.createElement('a');
        link.href = pdfUrl;
        link.download = `Inspection_Report_${reportNumber}.pdf`;
        link.target = '_blank';
        document.body.appendChild(link);
        link.click();
        document.body.removeChild(link);
        
        showNotification('PDF download started', 'success');
    }

// View inspection details
    function viewInspection(inspectionId) {
        openInspectionPdf(inspectionId, 'Viewing');
    }
    
    // Share inspection
    function shareInspection(inspectionId) {
        console.log('Sharing inspection:', inspectionId);
        showNotification(`Sharing inspection report ${inspectionId}`, 'info');
        // You can implement sharing functionality (email, link, etc.)
    }
    
    // Create workorder
    function createWorkorder(inspectionId) {
        console.log('Creating workorder for inspection:', inspectionId);
        showNotification(`Creating workorder for inspection ${inspectionId}`, 'success');
        // You can redirect to workorder creation page or show modal
    }
    
    // Corrective action
    function correctiveAction(inspectionId) {
        console.log('Corrective action for inspection:', inspectionId);
        showNotification(`Opening corrective action for inspection ${inspectionId}`, 'info');
        // You can redirect to corrective action page or show modal
    }
    
    // Root cause analysis
    function rootCauseAnalysis(inspectionId) {
        console.log('Root cause analysis for inspection:', inspectionId);
        showNotification(`Opening root cause analysis for inspection ${inspectionId}`, 'info');
        // You can redirect to root cause analysis page or show modal
    }
    
    // View PDF
    function viewPdf(inspectionId) {
        openInspectionPdf(inspectionId, 'Viewing PDF for');
    }
    
    // Archive inspection
    function archiveInspection(inspectionId) {
        console.log('Archiving inspection:', inspectionId);
        if (confirm('Are you sure you want to archive this inspection?')) {
            showNotification(`Inspection ${inspectionId} has been archived`, 'success');
            // You can implement archiving functionality
    }
}

    // Delete inspection from MongoDB with better confirmation modal
    async function deleteInspection(inspectionId, requestNumber) {
        console.log('═══════════════════════════════════════════════════════');
        console.log('🗑️ DELETE FUNCTION CALLED');
        console.log('═══════════════════════════════════════════════════════');
        console.log('📋 Received inspectionId:', inspectionId);
        console.log('📋 Received inspectionId type:', typeof inspectionId);
        console.log('📋 Received requestNumber:', requestNumber);
        console.log('📊 Current state.allData count:', state.allData.length);
        
        // Log all available inspections for comparison
        console.log('📊 All available inspections in state:');
        state.allData.forEach((ins, idx) => {
            console.log(`  [${idx}] _id="${ins._id}" (type: ${typeof ins._id}), requestNumber="${ins.requestNumber}"`);
        });
        
        // Find the inspection to verify it exists
        const inspection = state.allData.find(ins => {
            const matches = (ins._id === inspectionId || ins.id === inspectionId || String(ins._id) === String(inspectionId));
            if (matches) {
                console.log('✅ Found matching inspection:', ins._id);
            }
            return matches;
        });
        
        if (!inspection) {
            console.error('❌❌❌ INSPECTION NOT FOUND IN LOCAL STATE ❌❌❌');
            console.error('❌ Searched for ID:', inspectionId);
            console.error('❌ Available IDs:', state.allData.map(ins => ins._id));
            showNotification('⚠️ Inspection not found in local data. Reloading...', 'warning');
            loadInspectionData();
            return;
        }
        
        const actualRequestNumber = requestNumber || inspection?.requestNumber || inspection?.step1?.requestNumber || inspectionId;
        
        console.log('═══════════════════════════════════════════════════════');
        console.log('📋 INSPECTION TO DELETE:');
        console.log('═══════════════════════════════════════════════════════');
        console.log('  _id:', inspection._id, '(type:', typeof inspection._id, ')');
        console.log('  id:', inspection.id);
        console.log('  requestNumber:', actualRequestNumber);
        console.log('  userId:', inspection.userId);
        console.log('═══════════════════════════════════════════════════════');
        
        // Create and show custom confirmation modal
        const modalHtml = `
            <div class="delete-confirmation-modal" id="deleteConfirmModal" style="display: flex;">
                <div class="delete-modal-overlay" onclick="closeDeleteModal()"></div>
                <div class="delete-modal-content">
                    <div class="delete-modal-header">
                        <i class="fas fa-exclamation-triangle" style="color: #dc3545; font-size: 48px;"></i>
                        <h3 style="margin: 15px 0 5px 0; color: #212529; font-size: 24px; font-weight: 600;">Delete Inspection Report?</h3>
                        <p style="color: #6c757d; margin: 0;">This action cannot be undone</p>
                    </div>
                    <div class="delete-modal-body">
                        <div style="background: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px;">
                            <p style="margin: 0; color: #495057; font-size: 14px;">
                                <strong>Report Number:</strong> <span style="color: #0055a5; font-weight: 600; font-family: monospace;">${actualRequestNumber}</span>
                            </p>
                            <p style="margin: 10px 0 0 0; color: #495057; font-size: 14px;">
                                <strong>Inspector:</strong> ${inspection.step1?.inspector || 'N/A'}
                            </p>
                            <p style="margin: 10px 0 0 0; color: #495057; font-size: 14px;">
                                <strong>Vehicle:</strong> ${inspection.step1?.selectedEquipment?.equipmentName || 'N/A'}
                            </p>
                        </div>
                        <p style="color: #dc3545; font-size: 13px; margin: 0; text-align: center;">
                            <i class="fas fa-info-circle"></i> This report will be permanently deleted from the database
                        </p>
                    </div>
                    <div class="delete-modal-footer">
                        <button onclick="closeDeleteModal()" class="btn-cancel-delete">
                            <i class="fas fa-times"></i> Cancel
                        </button>
                        <button onclick="confirmDeleteInspection('${inspectionId}', '${actualRequestNumber}')" class="btn-confirm-delete">
                            <i class="fas fa-trash"></i> Yes, Delete
                        </button>
                    </div>
                </div>
            </div>
        `;
        
        // Remove existing modal if present
        const existingModal = document.getElementById('deleteConfirmModal');
        if (existingModal) {
            existingModal.remove();
        }
        
        // Add modal to page
        document.body.insertAdjacentHTML('beforeend', modalHtml);
        document.body.style.overflow = 'hidden';
    }
    
    // Close delete confirmation modal
    function closeDeleteModal() {
        const modal = document.getElementById('deleteConfirmModal');
        if (modal) {
            modal.style.display = 'none';
            modal.remove();
            document.body.style.overflow = 'auto';
        }
    }
    
    // Refresh data from MongoDB
    function refreshFromMongoDB() {
        console.log('═══════════════════════════════════════════════════════');
        console.log('🔄 MANUAL REFRESH TRIGGERED');
        console.log('═══════════════════════════════════════════════════════');
        
        // Clear current state
        state.allData = [];
        state.filteredData = [];
        state.totalRecords = 0;
        state.totalPages = 0;
        state.currentPage = 1;
        
        showNotification('🔄 Refreshing data from MongoDB...', 'info');
        loadInspectionData();
    }
    
    // Verify inspection exists in MongoDB before deleting
    async function verifyInspectionExists(inspectionId) {
        try {
            console.log('🔍 Verifying inspection exists in MongoDB:', inspectionId);
            const response = await fetch(`http://168.231.113.219:5004/inspections/report/${inspectionId}`);
            const result = await response.json();
            
            if (result.success && result.data) {
                console.log('✅ Inspection verified in MongoDB');
                return true;
            } else {
                console.warn('⚠️ Inspection not found in MongoDB:', inspectionId);
                return false;
            }
        } catch (error) {
            console.error('❌ Error verifying inspection:', error);
            return false;
        }
    }
    
    // Confirm and execute deletion
    async function confirmDeleteInspection(inspectionId, requestNumber) {
        console.log('═══════════════════════════════════════════════════════');
        console.log('✅ USER CONFIRMED DELETION');
        console.log('═══════════════════════════════════════════════════════');
        console.log('🔍 Inspection ID to delete:', inspectionId);
        console.log('🔍 Inspection ID type:', typeof inspectionId);
        
        // Find the full inspection object for debugging
        const inspection = state.allData.find(ins => ins._id === inspectionId || ins.id === inspectionId);
        if (inspection) {
            console.log('🔍 Found inspection in local state:', {
                _id: inspection._id,
                id: inspection.id,
                requestNumber: inspection.requestNumber,
                userId: inspection.userId
            });
        } else {
            console.error('❌ Inspection not found in local state!');
            console.log('📊 Available IDs in state:', state.allData.map(ins => ins._id));
        }
        
        closeDeleteModal();
        
        // First verify the inspection exists in MongoDB
        showLoading();
        const exists = await verifyInspectionExists(inspectionId);
        
        if (!exists) {
            hideLoading();
            showNotification('⚠️ Report already deleted or not found in MongoDB. Refreshing list...', 'warning');
            setTimeout(() => loadInspectionData(), 1500);
            return;
        }
        
        try {
            showLoading();
            console.log('📤 Sending delete request to Laravel API...');
            console.log('🔗 Full URL: /api/inspections/delete/' + inspectionId);
            console.log('📋 Request Number:', requestNumber);
            
            // Send delete request to Laravel API which forwards to MongoDB
            const response = await fetch(`/api/inspections/delete/${encodeURIComponent(inspectionId)}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]')?.content || '',
                    'Accept': 'application/json'
                }
            });
            
            console.log('📊 Response status:', response.status);
            console.log('📊 Response ok:', response.ok);
            console.log('📊 Response headers:', [...response.headers.entries()]);
            
            const result = await response.json();
            
            console.log('📊 Delete API response:', result);
            console.log('📊 Response data:', JSON.stringify(result, null, 2));
            
            if (result.success) {
                console.log('✅ Inspection deleted successfully from MongoDB');
                console.log('📋 Deleted report:', result.data);
                
                // Remove from local state using multiple matching strategies
                state.allData = state.allData.filter(ins => {
                    const match = (ins._id !== inspectionId && ins.id !== inspectionId && ins._id !== String(inspectionId));
                    if (!match) {
                        console.log('🗑️ Removing from state:', ins._id, ins.requestNumber);
                    }
                    return match;
                });
                state.filteredData = state.filteredData.filter(ins => 
                    (ins._id !== inspectionId && ins.id !== inspectionId && ins._id !== String(inspectionId))
                );
                state.totalRecords = state.allData.length;
                state.totalPages = Math.ceil(state.totalRecords / 10);
                
                // Adjust current page if needed
                if (state.currentPage > state.totalPages && state.totalPages > 0) {
                    state.currentPage = state.totalPages;
                }
                
                console.log('📊 Updated state - remaining reports:', state.totalRecords);
                
                // Re-render table
                renderTable();
                renderPagination();
                updateReportSummary(state.allData);
                
                showNotification(`✅ Report ${requestNumber} deleted successfully`, 'success');
                
                hideLoading();
            } else {
                console.error('❌ MongoDB returned error:', result);
                console.error('❌ Full error response:', JSON.stringify(result, null, 2));
                
                // Show detailed error information
                let errorMessage = result.message || result.error || 'Unknown error';
                if (result.debug) {
                    console.error('❌ Debug info:', result.debug);
                    if (result.debug.availableIds) {
                        console.error('📋 Available IDs in MongoDB:', result.debug.availableIds);
                    }
                }
                
                hideLoading();
                showNotification(`❌ Failed: ${errorMessage}`, 'error');
                
                // If report not found, reload data to sync with MongoDB
                if (errorMessage.includes('not found')) {
                    console.log('🔄 Report not found - reloading data from MongoDB...');
                    setTimeout(() => loadInspectionData(), 2000);
                }
            }
            
        } catch (error) {
            console.error('❌ Error deleting inspection:', error);
            console.error('❌ Error details:', {
                message: error.message,
                stack: error.stack,
                name: error.name
            });
            hideLoading();
            showNotification(`❌ Error: ${error.message}`, 'error');
        }
    }
    
    // Hide error message
    function hideError() {
        const errorMessage = document.getElementById('errorMessage');
        if (errorMessage) {
            errorMessage.classList.add('d-none');
        }
    }
    
    // Show loading spinner
    function showLoading() {
        const loadingSpinner = document.getElementById('loadingSpinner');
        const inspectionTable = document.getElementById('inspectionTable');
        const errorMessage = document.getElementById('errorMessage');
        
        if (loadingSpinner) loadingSpinner.classList.remove('d-none');
        if (inspectionTable) inspectionTable.classList.add('d-none');
        if (errorMessage) errorMessage.classList.add('d-none');
    }
    
    // Hide loading spinner
    function hideLoading() {
        const loadingSpinner = document.getElementById('loadingSpinner');
        const inspectionTable = document.getElementById('inspectionTable');
        
        if (loadingSpinner) loadingSpinner.classList.add('d-none');
        if (inspectionTable) inspectionTable.classList.remove('d-none');
    }
    
    // Show error message
    function showError(message) {
        const errorText = document.getElementById('errorText');
        const errorMessage = document.getElementById('errorMessage');
        
        if (errorText) errorText.textContent = message;
        if (errorMessage) errorMessage.classList.remove('d-none');
        hideLoading();
    }
    
    // Auto-refresh data every 30 seconds
    setInterval(function() {
        // You can implement auto-refresh here if needed
        console.log('Auto-refresh check...');
    }, 30000);

    // Initialize dropdown functionality
    function initializeDropdowns() {
        // Close dropdowns when clicking outside
        document.addEventListener('click', function(e) {
            if (!e.target.closest('.dropdown')) {
                const dropdowns = document.querySelectorAll('.dropdown-menu.show');
                dropdowns.forEach(dropdown => {
                    dropdown.classList.remove('show');
                });
            }
        });

        // Add keyboard navigation to dropdowns
        const dropdownButtons = document.querySelectorAll('.dropdown-toggle');
        dropdownButtons.forEach(button => {
            button.addEventListener('keydown', function(e) {
                if (e.key === 'Enter' || e.key === ' ') {
                    e.preventDefault();
                    const dropdown = this.nextElementSibling;
                    dropdown.classList.toggle('show');
                }
            });
        });
    }
    
    // Enhanced dropdown functions with better user feedback
    function showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `alert alert-${type} alert-dismissible fade show position-fixed`;
        notification.style.cssText = 'top: 20px; right: 20px; z-index: 9999; min-width: 300px;';
        notification.innerHTML = `
            ${message}
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        `;
        
        document.body.appendChild(notification);
        
        // Auto-remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.remove();
            }
        }, CONFIG.timing.notificationTimeout);
    }
    
    // Toggle debug section
    function toggleDebug() {
        if (elements.debugSection) {
            if (elements.debugSection.classList.contains('d-none')) {
                elements.debugSection.classList.remove('d-none');
                updateDebugInfo();
            } else {
                elements.debugSection.classList.add('d-none');
            }
        }
    }
    
    // Update debug information
    function updateDebugInfo() {
        if (elements.tableBody && elements.debugTotalReports && elements.debugLastUpdated) {
            const totalReports = elements.tableBody.querySelectorAll('tr:not(.tbl-body-no-data-td)').length;
            
            elements.debugTotalReports.textContent = totalReports;
            elements.debugLastUpdated.textContent = new Date().toLocaleString();
        }
    }
    
    // Test Firebase connection
    function testConnection() {
        if (elements.debugStatus) {
            elements.debugStatus.textContent = 'Testing...';
            elements.debugStatus.className = 'badge bg-warning';
            
            // Test inspection data specifically
            fetch(CONFIG.endpoints.testData)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        elements.debugStatus.textContent = 'Connected';
                        elements.debugStatus.className = 'badge bg-success';
                        
                        // Show detailed path analysis
                        let message = 'Connection successful! Found ' + data.total_records + ' records.\n\n';
                        if (data.path_analysis) {
                            message += 'Path Analysis:\n';
                            Object.keys(data.path_analysis).forEach(reportId => {
                                const analysis = data.path_analysis[reportId];
                                message += `\nReport ${reportId}:\n`;
                                message += `- Request Number: ${analysis.step1_requestNumber}\n`;
                                message += `- Created At: ${analysis.step1_createdAt}\n`;
                                message += `- Country: ${analysis.step1_country}\n`;
                                message += `- Selected Equipment: ${analysis.step1_selectedEquipment}\n`;
                                message += `- Selected Equipment Type: ${analysis.step1_selectedEquipment_type}\n`;
                                message += `- Faulty Items: ${analysis.step2_faultyItems}\n`;
                                message += `- Equipment: ${analysis.equipmentName}\n`;
                                message += `- Status: ${analysis.equipmentStatus}\n`;
                            });
                        }
                        alert(message);
                    } else {
                        elements.debugStatus.textContent = 'Failed';
                        elements.debugStatus.className = 'badge bg-danger';
                        alert('Connection failed: ' + data.message);
                    }
                })
                .catch(error => {
                    elements.debugStatus.textContent = 'Error';
                    elements.debugStatus.className = 'badge bg-danger';
                    alert('Connection error: ' + error.message);
                });
        }
    }
    
    // Clear cache (reload page)
    function clearCache() {
        if (confirm('This will refresh the page and clear any cached data. Continue?')) {
            location.reload();
        }
    }
    
    // Make functions globally accessible
    window.refreshData = refreshData;
    window.refreshFromMongoDB = refreshFromMongoDB;
    window.verifyInspectionExists = verifyInspectionExists;
    window.viewInspection = viewInspection;
    window.shareInspection = shareInspection;
    window.createWorkorder = createWorkorder;
    window.correctiveAction = correctiveAction;
    window.rootCauseAnalysis = rootCauseAnalysis;
    window.viewPdf = viewPdf;
    window.archiveInspection = archiveInspection;
    window.deleteInspection = deleteInspection;
    window.confirmDeleteInspection = confirmDeleteInspection;
    window.closeDeleteModal = closeDeleteModal;
    window.hideError = hideError;
    window.showLoading = showLoading;
    window.hideLoading = hideLoading;
    window.showError = showError;
    window.showNotification = showNotification;
    window.toggleDebug = toggleDebug;
    window.updateDebugInfo = updateDebugInfo;
    window.testConnection = testConnection;
    window.closePdfModal = closePdfModal;
    window.downloadPdf = downloadPdf;
    window.clearCache = clearCache;
    window.changePage = changePage;
    window.goToPage = goToPage;
    
})();
</script>

<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.inspection-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* Main Content */
.main-content {
    width: 100%;
    padding: 30px 20px;
}

/* Form Styles */
.form-container {
    background: white;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
    margin-bottom: 20px;
}

.form-section {
    padding: 30px;
    border-bottom: 1px solid var(--border-light);
}

.form-section:last-child {
    border-bottom: none;
}

.section-title {
    font-size: 20px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 20px;
    display: flex;
    align-items: center;
    gap: 10px;
    justify-content: space-between;
}

.section-title i {
    color: var(--primary-color);
}

/* Request Number Display */
.request-number-display {
    background: var(--bg-secondary);
    border: 1px solid var(--border-light);
    border-radius: var(--radius-md);
    padding: 15px 20px;
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 25px;
}

.request-number-label {
    font-weight: 500;
    color: var(--text-primary);
}

.request-number-value {
    font-weight: 700;
    color: var(--primary-color);
    font-size: 16px;
}

/* Summary Stats */
.summary-stats {
    display: flex;
    gap: 30px;
    align-items: center;
}

.stat-item {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 5px;
}

.stat-number {
    font-size: 24px;
    font-weight: 700;
    color: var(--primary-color);
}

.stat-label {
    font-size: 12px;
    color: var(--text-secondary);
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* Search Container */
.search-container {
    position: relative;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 16px;
    z-index: 2;
    color: var(--text-secondary);
    width: 16px;
    height: 16px;
}

.search-input {
    padding: 12px 16px 12px 44px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    height: 44px;
    width: 300px;
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.search-input::placeholder {
    color: var(--text-secondary);
}

/* Table Styles */
.table-container {
    overflow-x: auto;
    border-radius: var(--radius-md);
    border: 1px solid var(--border-light);
}

.inspection-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
    background: white;
}

.inspection-table thead {
    background: var(--bg-secondary);
}

.inspection-table th {
    padding: 16px 12px;
    text-align: left;
    font-weight: 600;
    color: var(--text-primary);
    border-bottom: 1px solid var(--border-color);
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.inspection-table td {
    padding: 16px 12px;
    border-bottom: 1px solid var(--border-light);
    vertical-align: middle;
}

.table-row:hover {
    background: var(--bg-light);
}

/* User report highlighting */
.table-row.user-report {
    background: #f0fdf4;
    border-left: 4px solid #22c55e;
}

.table-row.user-report:hover {
    background: #ecfdf5;
}

/* Action Button */
.dropdown .action-btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 12px;
    background: var(--success-color);
    color: white;
    border: none;
    border-radius: var(--radius-sm);
    font-size: 12px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.dropdown .action-btn:hover {
    background: #218838;
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
    color: white !important;
}

/* Dropdown Menu */
.dropdown-menu {
    background: white;
    border: 1px solid var(--border-color);
    box-shadow: var(--shadow-md);
    border-radius: var(--radius-sm);
    padding: 8px 0;
    min-width: 200px;
    z-index: 1000;
}

.dropdown-item {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    color: var(--text-primary);
    text-decoration: none;
    font-size: 13px;
    font-weight: 400;
    transition: var(--transition-base);
    border-radius: var(--radius-sm);
}

.dropdown-item:hover {
    background: var(--bg-light);
    color: var(--text-primary);
    transform: translateX(4px);
}

.dropdown-item.danger {
    color: var(--danger-color);
}

.dropdown-item.danger:hover {
    background: #f8d7da;
    color: #721c24;
}

.dropdown-divider {
    height: 1px;
    background: var(--border-color);
    margin: 8px 0;
}

/* Table Cell Styling */
.serial-number-cell {
    text-align: center;
    width: 60px;
    min-width: 60px;
}

.serial-number {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 28px;
    height: 28px;
    background: var(--primary-color);
    color: white;
    border-radius: 50%;
    font-weight: 600;
    font-size: 12px;
    font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
}

.report-id {
    font-weight: 600;
    color: var(--text-primary);
    font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
    font-size: 13px;
}

.vehicle-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.vehicle-name {
    font-weight: 600;
    color: var(--text-primary);
    font-size: 14px;
}

.vehicle-status {
    font-size: 12px;
    color: var(--text-secondary);
    font-weight: 400;
}

.fault-info {
    display: flex;
    flex-direction: column;
    gap: 4px;
}

.fault-badge {
    display: inline-block;
    padding: 4px 8px;
    background: #f8d7da;
    color: var(--danger-color);
    font-size: 11px;
    font-weight: 600;
    border-radius: var(--radius-sm);
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.fault-details {
    font-size: 12px;
    color: var(--danger-color);
    font-weight: 400;
}

.no-faults {
    color: var(--success-color);
    font-weight: 500;
    font-size: 13px;
}

.corrective-action {
    color: var(--info-color);
    font-weight: 400;
    font-size: 13px;
}

.corrective-action-link {
    color: var(--info-color);
    text-decoration: none;
    display: inline-flex;
    align-items: center;
    gap: 4px;
    transition: var(--transition-base);
}

.corrective-action-link:hover {
    color: var(--primary-color);
    text-decoration: underline;
}

.no-action {
    color: var(--text-secondary);
    font-weight: 400;
    font-size: 13px;
}

.linked-badge {
    display: inline-block;
    padding: 4px 8px;
    background: #d1ecf1;
    color: var(--info-color);
    font-size: 11px;
    font-weight: 600;
    border-radius: var(--radius-sm);
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.not-linked {
    color: var(--text-secondary);
    font-weight: 400;
    font-size: 13px;
}

/* No Data Styling */
.no-data-cell {
    padding: 64px 24px !important;
    text-align: center;
}

.no-data-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.no-data-content svg {
    color: var(--text-secondary);
}

.no-data-content h3 {
    color: var(--text-primary);
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.no-data-content p {
    color: var(--text-secondary);
    font-weight: 400;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

/* Loading and Error States */
.loading-content {
    padding: 48px;
    text-align: center;
}

.error-content {
    padding: 16px;
    background: #f8d7da;
    border: 1px solid #f5c6cb;
    border-radius: var(--radius-sm);
    color: var(--danger-color);
    display: flex;
    align-items: center;
    gap: 10px;
}

.btn-close {
    background: none;
    border: none;
    cursor: pointer;
    padding: 0;
    margin-left: auto;
    color: var(--danger-color);
}

.spinner-border {
    width: 3rem;
    height: 3rem;
    color: var(--primary-color);
}

/* Debug Section */
.debug-content {
    margin-top: 20px;
}

.debug-row {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 30px;
    margin-bottom: 20px;
}

.debug-column h6 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 15px;
}

.debug-list {
    list-style: none;
    padding: 0;
    margin: 0;
}

.debug-list li {
    padding: 8px 0;
    border-bottom: 1px solid var(--border-light);
}

.debug-list li:last-child {
    border-bottom: none;
}

.status-badge {
    background: var(--success-color);
    color: white;
    padding: 4px 8px;
    border-radius: var(--radius-sm);
    font-size: 11px;
    font-weight: 500;
}

.debug-actions {
    display: flex;
    gap: 10px;
    margin-top: 20px;
}

.btn {
    padding: 8px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    text-decoration: none;
    display: inline-block;
}

.btn-sm {
    padding: 6px 12px;
    font-size: 12px;
}

.btn-outline-info {
    color: var(--info-color);
    border-color: var(--info-color);
    background: transparent;
}

.btn-outline-info:hover {
    background: var(--info-color);
    color: white;
}

.btn-outline-warning {
    color: var(--warning-color);
    border-color: var(--warning-color);
    background: transparent;
}

.btn-outline-warning:hover {
    background: var(--warning-color);
    color: white;
}

/* Pagination Styles */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 20px;
}

.pagination-info {
    color: var(--text-secondary);
    font-size: 14px;
    font-weight: 500;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 12px;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    background: white;
    color: var(--text-secondary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    text-decoration: none;
}

.pagination-btn:hover:not(:disabled) {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

.pagination-pages {
    display: flex;
    gap: 4px;
    align-items: center;
}

.pagination-page {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 40px;
    height: 40px;
    background: white;
    color: var(--text-secondary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-sm);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    text-decoration: none;
}

.pagination-page:hover:not(:disabled) {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
    box-shadow: var(--shadow-sm);
}

.pagination-page.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    cursor: default;
}

.pagination-page.active:hover {
    background: var(--primary-color);
    color: white;
    transform: none;
    box-shadow: none;
}

.pagination-page:disabled {
    opacity: 0.5;
    cursor: not-allowed;
    transform: none;
    box-shadow: none;
}

/* Responsive Design */
@media (max-width: 1200px) {
    .main-content {
        padding: 20px;
    }
    
    .search-input {
        width: 250px;
    }
}

@media (max-width: 992px) {
    .main-content {
        padding: 16px;
    }
    
    .header-content {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .header-right {
        justify-content: space-between;
        flex-wrap: wrap;
        gap: 16px;
    }
    
    .search-input {
        width: 100%;
        max-width: 300px;
    }
    
    .form-section {
        padding: 20px;
    }
    
    .inspection-table {
        font-size: 13px;
    }
    
    .inspection-table th,
    .inspection-table td {
        padding: 14px 10px;
    }
}

@media (max-width: 768px) {
    .main-content {
        padding: 0;
    }
    
    .form-container {
        margin: 0 16px 16px 16px;
    }
    
    .form-section {
        padding: 16px;
    }
    
    .header-content {
        padding: 0 16px;
        gap: 16px;
    }
    
    .header-title {
        font-size: 20px;
    }
    
    .header-right {
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .search-container {
        width: 100%;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px 12px 44px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .create-btn {
        flex: 1;
        justify-content: center;
        min-width: 140px;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 48px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .inspection-table {
        min-width: 900px;
        font-size: 12px;
    }
    
    .inspection-table th,
    .inspection-table td {
        padding: 12px 8px;
        white-space: nowrap;
    }
    
    .inspection-table th {
        font-size: 11px;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .pagination-container {
        flex-direction: column;
        gap: 16px;
        align-items: stretch;
    }
    
    .pagination-info {
        text-align: center;
        font-size: 13px;
    }
    
    .pagination-controls {
        justify-content: center;
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .debug-row {
        grid-template-columns: 1fr;
        gap: 20px;
    }
}

@media (max-width: 480px) {
    .form-container {
        margin: 0 12px 12px 12px;
    }
    
    .form-section {
        padding: 12px;
    }
    
    .header-content {
        padding: 0 12px;
    }
    
    .header-title {
        font-size: 18px;
    }
    
    .search-input {
        padding: 10px 14px 10px 40px;
        font-size: 16px;
        min-height: 44px;
    }
    
    .create-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 44px;
        min-width: 120px;
    }
    
    .inspection-table {
        min-width: 800px;
        font-size: 11px;
    }
    
    .inspection-table th,
    .inspection-table td {
        padding: 10px 6px;
    }
    
    .inspection-table th {
        font-size: 10px;
    }
    
    .pagination-btn {
        padding: 8px 12px;
        font-size: 13px;
    }
    
    .pagination-page {
        width: 36px;
        height: 36px;
        font-size: 13px;
    }
}

/* Utility Classes */
.hidden {
    display: none !important;
}

.text-center {
    text-align: center;
}

.text-success {
    color: var(--success-color);
}

.text-danger {
    color: var(--danger-color);
}

.text-warning {
    color: var(--warning-color);
}

.text-info {
    color: var(--info-color);
}

.mt-2 {
    margin-top: 0.5rem;
}

/* Accessibility and Focus States */
.create-btn:focus,
.pagination-btn:focus,
.dropdown .action-btn:focus,
.search-input:focus {
    outline: 2px solid var(--primary-color);
    outline-offset: 2px;
}

/* Reduced motion for accessibility */
@media (prefers-reduced-motion: reduce) {
    * {
        transition: none !important;
        animation: none !important;
    }
}

/* ========================================
   PDF VIEWER MODAL STYLES
   ======================================== */

.pdf-modal {
    display: none;
    position: fixed;
    z-index: 10000;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.8);
    justify-content: center;
    align-items: center;
    animation: fadeIn 0.3s ease;
}

@keyframes fadeIn {
    from { opacity: 0; }
    to { opacity: 1; }
}

.pdf-modal-content {
    background-color: #fff;
    width: 90%;
    height: 90%;
    max-width: 1200px;
    max-height: 90vh;
    border-radius: 12px;
    display: flex;
    flex-direction: column;
    box-shadow: 0 10px 40px rgba(0, 0, 0, 0.3);
    animation: slideUp 0.3s ease;
}

@keyframes slideUp {
    from {
        transform: translateY(50px);
        opacity: 0;
    }
    to {
        transform: translateY(0);
        opacity: 1;
    }
}

.pdf-modal-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px 25px;
    background: linear-gradient(135deg, #0055a5, #003d7a);
    color: white;
    border-radius: 12px 12px 0 0;
}

.pdf-modal-header h3 {
    margin: 0;
    font-size: 1.3rem;
    font-weight: 600;
    display: flex;
    align-items: center;
    gap: 10px;
}

.pdf-close-btn {
    background: rgba(255, 255, 255, 0.2);
    border: none;
    color: white;
    font-size: 2rem;
    cursor: pointer;
    width: 40px;
    height: 40px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.3s ease;
    line-height: 1;
}

.pdf-close-btn:hover {
    background: rgba(255, 255, 255, 0.3);
    transform: rotate(90deg);
}

.pdf-modal-body {
    flex: 1;
    padding: 0;
    overflow: hidden;
    background-color: #f5f5f5;
}

.pdf-modal-footer {
    display: flex;
    justify-content: center;
    gap: 15px;
    padding: 20px;
    background-color: #f8f9fa;
    border-radius: 0 0 12px 12px;
    border-top: 1px solid #e0e0e0;
}

.btn-download-pdf, .btn-close-pdf {
    padding: 12px 30px;
    font-size: 1rem;
    font-weight: 600;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 8px;
}

.btn-download-pdf {
    background: linear-gradient(135deg, #00B894, #00A082);
    color: white;
}

.btn-download-pdf:hover {
    background: linear-gradient(135deg, #00A082, #008f72);
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(0, 184, 148, 0.3);
}

.btn-close-pdf {
    background: #545d6b;
    color: white;
}

.btn-close-pdf:hover {
    background: #3d444f;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(84, 93, 107, 0.3);
}

/* Responsive Design */
@media (max-width: 768px) {
    .pdf-modal-content {
        width: 95%;
        height: 95%;
    }
    
    .pdf-modal-header h3 {
        font-size: 1.1rem;
    }
    
    .pdf-modal-footer {
        flex-direction: column;
    }
    
    .btn-download-pdf, .btn-close-pdf {
        width: 100%;
        justify-content: center;
    }
}

/* Print Styles */
@media print {
    .pdf-modal {
        display: none !important;
    }
}

/* ========================================
   DELETE CONFIRMATION MODAL STYLES
   ======================================== */

.delete-confirmation-modal {
    display: none;
    position: fixed;
    z-index: 10001;
    left: 0;
    top: 0;
    width: 100%;
    height: 100%;
    justify-content: center;
    align-items: center;
    animation: fadeIn 0.3s ease;
}

.delete-modal-overlay {
    position: absolute;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.6);
    backdrop-filter: blur(4px);
}

.delete-modal-content {
    position: relative;
    background-color: #fff;
    width: 90%;
    max-width: 500px;
    border-radius: 16px;
    box-shadow: 0 20px 60px rgba(0, 0, 0, 0.3);
    animation: slideUp 0.3s ease;
    z-index: 1;
}

.delete-modal-header {
    padding: 30px 30px 20px 30px;
    text-align: center;
    border-bottom: 1px solid #e9ecef;
}

.delete-modal-body {
    padding: 25px 30px;
}

.delete-modal-footer {
    display: flex;
    justify-content: center;
    gap: 15px;
    padding: 20px 30px 30px 30px;
}

.btn-cancel-delete,
.btn-confirm-delete {
    padding: 12px 30px;
    font-size: 15px;
    font-weight: 600;
    border: none;
    border-radius: 8px;
    cursor: pointer;
    transition: all 0.3s ease;
    display: flex;
    align-items: center;
    gap: 8px;
    min-width: 140px;
    justify-content: center;
}

.btn-cancel-delete {
    background: #6c757d;
    color: white;
}

.btn-cancel-delete:hover {
    background: #5a6268;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(108, 117, 125, 0.3);
}

.btn-confirm-delete {
    background: #dc3545;
    color: white;
}

.btn-confirm-delete:hover {
    background: #c82333;
    transform: translateY(-2px);
    box-shadow: 0 4px 12px rgba(220, 53, 69, 0.4);
}

/* Alert notification styles */
.alert {
    padding: 15px 20px;
    border-radius: 8px;
    font-size: 14px;
    font-weight: 500;
    display: flex;
    align-items: center;
    gap: 10px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    animation: slideInRight 0.3s ease;
}

@keyframes slideInRight {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

.alert-success {
    background-color: #d4edda;
    border: 1px solid #c3e6cb;
    color: #155724;
}

.alert-error,
.alert-danger {
    background-color: #f8d7da;
    border: 1px solid #f5c6cb;
    color: #721c24;
}

.alert-info {
    background-color: #d1ecf1;
    border: 1px solid #bee5eb;
    color: #0c5460;
}

.alert-warning {
    background-color: #fff3cd;
    border: 1px solid #ffeaa7;
    color: #856404;
}

/* Mobile responsive */
@media (max-width: 768px) {
    .delete-modal-content {
        width: 95%;
        margin: 0 10px;
    }
    
    .delete-modal-header,
    .delete-modal-body,
    .delete-modal-footer {
        padding: 20px;
    }
    
    .delete-modal-footer {
        flex-direction: column;
    }
    
    .btn-cancel-delete,
    .btn-confirm-delete {
        width: 100%;
    }
}

</style>
@endsection
