@extends('welcome')

@section('title', 'Fuel Logger')

@section('content')
<div class="fuel-logger-container">
    <!-- Header Section - matching Create New Workorder style -->
    <header class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <div>
                    <h1 class="header-title">Fuel Logger</h1>
                    <p class="header-subtitle">Track and manage fuel consumption, costs, and receipts efficiently</p>
                </div>
            </div>
            <div class="header-right">
                <div class="search-container">
                    <div class="search-input-wrapper">
                        <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <circle cx="11" cy="11" r="8"></circle>
                            <path d="m21 21-4.35-4.35"></path>
                        </svg>
                        <input type="text" class="search-input" placeholder="Search fuel entries..." id="fuelSearch" aria-label="Search fuel entries" autocomplete="off">
                    </div>
                </div>
                <div class="action-buttons">
                    <a href="{{ route('fuel.new-entry') }}" class="btn-primary">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <line x1="12" y1="5" x2="12" y2="19"></line>
                            <line x1="5" y1="12" x2="19" y2="12"></line>
                        </svg>
                        New Fuel Entry
                    </a>
                    <button class="btn-secondary" type="button" id="sortFuelBtn" aria-label="Open sorting options">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <path d="M3 6h18M7 12h10M10 18h4"/>
                        </svg>
                        Sorting
                    </button>
                    <button class="btn-success" type="button" id="exportFuelBtn" aria-label="Export fuel entries to Excel">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" aria-hidden="true">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"></path>
                            <polyline points="14,2 14,8 20,8"></polyline>
                            <line x1="16" y1="13" x2="8" y2="13"></line>
                            <line x1="16" y1="17" x2="8" y2="17"></line>
                        </svg>
                        Excel Export
                    </button>
                </div>
            </div>
        </div>
    </header>
    
    <!-- Table Section -->
    <section class="table-section">

        <div class="table-container">
            <table class="fuel-logger-table" id="fuelTable" role="table" aria-label="Fuel Entries">
                <thead>
                    <tr role="row">
                        <th scope="col">Action</th>
                        <th scope="col">Vehicle</th>
                        <th scope="col">Meter Reading</th>
                        <th scope="col">Filling Date</th>
                        <th scope="col">Filling Source</th>
                        <th scope="col">Fuel QTY</th>
                        <th scope="col">Fuel Cost</th>
                        <th scope="col">Receipt</th>
                    </tr>
                </thead>
                <tbody id="fuelTableBody" role="rowgroup">
                    <!-- Data will be populated here -->
                </tbody>
            </table>
            
            <!-- Pagination -->
            <div class="pagination-container" id="paginationContainer" style="display: none;">
                <div class="pagination-info">
                    Showing <span id="showingStart">0</span> to <span id="showingEnd">0</span> of <span id="totalEntries">0</span> entries
                </div>
                <div class="pagination-controls" id="paginationControls">
                    <!-- Pagination buttons will be populated here -->
                </div>
            </div>
            
                    <!-- Empty State -->
            <div id="emptyState" style="display: none;">
                <div class="no-data-cell">
                            <div class="no-data-content">
                                <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9L18.7 8.3c-.3-.8-1-1.3-1.9-1.3H7.2c-.9 0-1.6.5-1.9 1.3L3.5 11.1C2.7 11.3 2 12.1 2 13v3c0 .6.4 1 1 1h2m14 0H5m14 0v3c0 .6-.4 1-1 1H6c-.6 0-1-.4-1-1v-3"/>
                                </svg>
                                <h3>No Fuel Entries Found</h3>
                        <p>No fuel entries match your search criteria. Try adjusting your search or create a new entry.</p>
                            </div>
                </div>
            </div>
        </div>
    </section>
    <script>
        function toggleSidebar() {
            const sidebar = document.querySelector('.sidebar');
            sidebar.classList.toggle('collapsed');
        }
    </script>
</div>
<!-- Sorting Modal -->
<div class="modal fade" id="sorting" tabindex="-1" aria-labelledby="sortingModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title" id="sortingModalLabel">Sorting</h4>
                <button type="button" class="modal-close" data-bs-dismiss="modal" aria-label="Close">&times;</button>
            </div>
            <div class="modal-body">
                <form class="sorting-form" id="sortingForm">
                    <div class="form-section">
                        <h3 class="section-title">Sort Order</h3>
                        <div class="form-group">
                            <label class="radio-label">
                                <input type="radio" name="sortOrder" value="ascending" checked>
                                <span class="radio-custom"></span>
                                Ascending
                            </label>
                        </div>
                        <div class="form-group">
                            <label class="radio-label">
                                <input type="radio" name="sortOrder" value="descending">
                                <span class="radio-custom"></span>
                                Descending
                            </label>
                        </div>
                    </div>
                    
                    <div class="form-section">
                        <h3 class="section-title">Sort By</h3>
                        <div class="form-group">
                            <label class="radio-label">
                                <input type="radio" name="sortBy" value="vehicle" checked>
                                <span class="radio-custom"></span>
                                Vehicle Number
                            </label>
                    </div>
                        <div class="form-group">
                            <label class="radio-label">
                                <input type="radio" name="sortBy" value="vehicleName">
                                <span class="radio-custom"></span>
                                Vehicle Name
                            </label>
                    </div>
                        <div class="form-group">
                            <label class="radio-label">
                                <input type="radio" name="sortBy" value="fillingDate">
                                <span class="radio-custom"></span>
                                Filling Date
                            </label>
                    </div>
                        <div class="form-group">
                            <label class="radio-label">
                                <input type="radio" name="sortBy" value="meterReading">
                                <span class="radio-custom"></span>
                                Meter Reading
                            </label>
                    </div>
                        <div class="form-group">
                            <label class="radio-label">
                                <input type="radio" name="sortBy" value="fuelQty">
                                <span class="radio-custom"></span>
                                Fuel QTY
                            </label>
                    </div>
                        <div class="form-group">
                            <label class="radio-label">
                                <input type="radio" name="sortBy" value="fuelCost">
                                <span class="radio-custom"></span>
                                Fuel Cost
                            </label>
                        </div>
                    </div>
                    
                    <div class="modal-actions">
                        <button type="submit" class="btn-save">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M20 6L9 17l-5-5"/>
                            </svg>
                            Apply
                        </button>
                        <button type="button" class="btn-reset">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M3 12a9 9 0 0 1 9-9 9.75 9.75 0 0 1 6.74 2.74L21 8"/>
                                <path d="M21 3v5h-5"/>
                                <path d="M21 12a9 9 0 0 1-9 9 9.75 9.75 0 0 1-6.74-2.74L3 16"/>
                                <path d="M3 21v-5h5"/>
                            </svg>
                            Reset
                        </button>
                        <button type="button" class="btn-cancel" data-bs-dismiss="modal">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M18 6L6 18M6 6l12 12"/>
                            </svg>
                            Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<script>
let allFuelData = [];
let filteredData = [];
let currentPage = 1;
const itemsPerPage = 10;

// Format date and time
function formatDateTime(dateString) {
    if (!dateString) return '-';
    const date = new Date(dateString);
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    return `${day}/${month}/${year} ${hours}:${minutes}`;
}

// Fetch data from Firebase
async function fetchFuelData() {
    const tableBody = document.getElementById('fuelTableBody');
    const emptyState = document.getElementById('emptyState');
    
    emptyState.style.display = 'none';
    
    try {
        const response = await fetch('https://titan-drilling-1f8e9-default-rtdb.firebaseio.com/GlobalFuelLogs/admins.json');
        const data = await response.json();
        
        console.log('Firebase fuel data:', data);
        
        allFuelData = [];
        
        if (data && typeof data === 'object') {
            // Iterate through each admin UID folder
            Object.keys(data).forEach(adminUID => {
                const adminData = data[adminUID];
                
                if (adminData && typeof adminData === 'object') {
                    // Iterate through each fuel entry inside the admin folder
                    Object.keys(adminData).forEach(entryKey => {
                        const entry = adminData[entryKey];
                        
                        if (entry && typeof entry === 'object' && entry.id) {
                            allFuelData.push({
                                ...entry,
                                adminUID: adminUID,
                                entryKey: entryKey
                            });
                        }
                    });
                }
            });
        }
        
        // Sort by created date (newest first)
        allFuelData.sort((a, b) => {
            const dateA = new Date(a.createdAt || 0);
            const dateB = new Date(b.createdAt || 0);
            return dateB - dateA;
        });
        
        console.log('Processed fuel data:', allFuelData);
        
        // Display data
        filteredData = [...allFuelData];
        displayData(filteredData);
        
    } catch (error) {
        console.error('Error fetching fuel data:', error);
        emptyState.style.display = 'block';
    }
}

// Display data in table with pagination
function displayData(data) {
    const tableBody = document.getElementById('fuelTableBody');
    const emptyState = document.getElementById('emptyState');
    const table = document.getElementById('fuelTable');
    const paginationContainer = document.getElementById('paginationContainer');
    
    tableBody.innerHTML = '';
    
    if (data.length === 0) {
        table.style.display = 'none';
        paginationContainer.style.display = 'none';
        emptyState.style.display = 'block';
        return;
    }
    
    table.style.display = 'table';
    emptyState.style.display = 'none';
    paginationContainer.style.display = 'flex';
    
    // Calculate pagination
    const totalPages = Math.ceil(data.length / itemsPerPage);
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = Math.min(startIndex + itemsPerPage, data.length);
    const pageData = data.slice(startIndex, endIndex);
    
    // Display data for current page
    pageData.forEach(entry => {
        const row = document.createElement('tr');
        row.innerHTML = `
            <td>
                <button class="action-btn view" onclick="viewFuelEntry('${entry.id}')" title="View Details">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                        <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                        <circle cx="12" cy="12" r="3"></circle>
                    </svg>
                </button>
            </td>
            <td>
                <div style="font-weight: 600;">${entry.equipmentName || '-'}</div>
                <div style="font-size: 12px; color: #6c757d;">${entry.equipmentNo || '-'}</div>
            </td>
            <td>${entry.meterReading ? parseFloat(entry.meterReading).toFixed(2) : '-'} ${entry.meterUnit || ''}</td>
            <td>${formatDateTime(entry.createdAt)}</td>
            <td>${entry.fillingSource || entry.selectedBunker || '-'}</td>
            <td>${entry.fuelQuantity || '-'} ${entry.fuelUnit || ''}</td>
            <td>$${entry.fuelCost || '0'}</td>
            <td>
                ${entry.receiptUrl ? `<a href="${entry.receiptUrl}" target="_blank" class="receipt-link">View</a>` : '-'}
            </td>
        `;
        tableBody.appendChild(row);
    });
    
    // Update pagination info and controls
    updatePaginationInfo(startIndex + 1, endIndex, data.length);
    updatePaginationControls(totalPages);
}

// Update pagination info text
function updatePaginationInfo(start, end, total) {
    document.getElementById('showingStart').textContent = start;
    document.getElementById('showingEnd').textContent = end;
    document.getElementById('totalEntries').textContent = total;
}

// Update pagination controls
function updatePaginationControls(totalPages) {
    const paginationControls = document.getElementById('paginationControls');
    paginationControls.innerHTML = '';
    
    // Previous button
    const prevBtn = document.createElement('button');
    prevBtn.className = 'pagination-btn';
    prevBtn.textContent = 'Previous';
    prevBtn.disabled = currentPage === 1;
    prevBtn.onclick = () => changePage(currentPage - 1);
    paginationControls.appendChild(prevBtn);
    
    // Page numbers
    const maxVisiblePages = 5;
    let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
    
    if (endPage - startPage < maxVisiblePages - 1) {
        startPage = Math.max(1, endPage - maxVisiblePages + 1);
    }
    
    // First page
    if (startPage > 1) {
        const firstBtn = document.createElement('button');
        firstBtn.className = 'pagination-btn';
        firstBtn.textContent = '1';
        firstBtn.onclick = () => changePage(1);
        paginationControls.appendChild(firstBtn);
        
        if (startPage > 2) {
            const ellipsis = document.createElement('span');
            ellipsis.className = 'pagination-ellipsis';
            ellipsis.textContent = '...';
            paginationControls.appendChild(ellipsis);
        }
    }
    
    // Page number buttons
    for (let i = startPage; i <= endPage; i++) {
        const pageBtn = document.createElement('button');
        pageBtn.className = 'pagination-btn' + (i === currentPage ? ' active' : '');
        pageBtn.textContent = i;
        pageBtn.onclick = () => changePage(i);
        paginationControls.appendChild(pageBtn);
    }
    
    // Last page
    if (endPage < totalPages) {
        if (endPage < totalPages - 1) {
            const ellipsis = document.createElement('span');
            ellipsis.className = 'pagination-ellipsis';
            ellipsis.textContent = '...';
            paginationControls.appendChild(ellipsis);
        }
        
        const lastBtn = document.createElement('button');
        lastBtn.className = 'pagination-btn';
        lastBtn.textContent = totalPages;
        lastBtn.onclick = () => changePage(totalPages);
        paginationControls.appendChild(lastBtn);
    }
    
    // Next button
    const nextBtn = document.createElement('button');
    nextBtn.className = 'pagination-btn';
    nextBtn.textContent = 'Next';
    nextBtn.disabled = currentPage === totalPages;
    nextBtn.onclick = () => changePage(currentPage + 1);
    paginationControls.appendChild(nextBtn);
}

// Change page
function changePage(page) {
    currentPage = page;
    displayData(filteredData);
    document.getElementById('fuelTable').scrollIntoView({ behavior: 'smooth', block: 'start' });
}

// Filter data
function filterFuelData() {
    const searchTerm = document.getElementById('fuelSearch').value.toLowerCase();
    
    filteredData = allFuelData.filter(entry => {
        return !searchTerm || 
            (entry.equipmentName && entry.equipmentName.toLowerCase().includes(searchTerm)) ||
            (entry.equipmentNo && entry.equipmentNo.toLowerCase().includes(searchTerm)) ||
            (entry.requestNumber && entry.requestNumber.toLowerCase().includes(searchTerm)) ||
            (entry.fillingSource && entry.fillingSource.toLowerCase().includes(searchTerm)) ||
            (entry.fuelType && entry.fuelType.toLowerCase().includes(searchTerm)) ||
            (entry.receiverName && entry.receiverName.toLowerCase().includes(searchTerm)) ||
            (entry.country && entry.country.toLowerCase().includes(searchTerm)) ||
            (entry.project && entry.project.toLowerCase().includes(searchTerm));
    });
    
    currentPage = 1;
    displayData(filteredData);
}

// View fuel entry details
function viewFuelEntry(entryId) {
    const entry = allFuelData.find(item => item.id === entryId);
    if (!entry) return;
    
    alert(`Fuel Entry Details:\n\n` +
        `Request Number: ${entry.requestNumber || '-'}\n` +
        `Equipment: ${entry.equipmentName || '-'} (${entry.equipmentNo || '-'})\n` +
        `Country: ${entry.country || '-'}\n` +
        `Project: ${entry.project || '-'}\n` +
        `Fuel Type: ${entry.fuelType || '-'}\n` +
        `Quantity: ${entry.fuelQuantity || '-'} ${entry.fuelUnit || ''}\n` +
        `Cost: $${entry.fuelCost || '0'}\n` +
        `Meter Reading: ${entry.meterReading || '-'} ${entry.meterUnit || ''}\n` +
        `Filling Source: ${entry.fillingSource || entry.selectedBunker || '-'}\n` +
        `Receiver: ${entry.receiverName || '-'}\n` +
        `Note: ${entry.note || '-'}\n` +
        `Created: ${formatDateTime(entry.createdAt)}`
    );
}

// Export to Excel
function exportFuelToExcel() {
    if (filteredData.length === 0) {
        alert('No data to export');
        return;
    }
    
    const headers = ['Request Number', 'Equipment Name', 'Equipment No', 'Country', 'Project', 'Fuel Type', 'Quantity', 'Unit', 'Cost', 'Meter Reading', 'Filling Source', 'Receiver Name', 'Date', 'Note'];
    const rows = filteredData.map(entry => [
        entry.requestNumber || '',
        entry.equipmentName || '',
        entry.equipmentNo || '',
        entry.country || '',
        entry.project || '',
        entry.fuelType || '',
        entry.fuelQuantity || '',
        entry.fuelUnit || '',
        entry.fuelCost || '',
        entry.meterReading || '',
        entry.fillingSource || entry.selectedBunker || '',
        entry.receiverName || '',
        formatDateTime(entry.createdAt),
        entry.note || ''
    ]);
    
    let csvContent = headers.join(',') + '\n';
    rows.forEach(row => {
        csvContent += row.map(cell => `"${cell}"`).join(',') + '\n';
    });
    
    const blob = new Blob([csvContent], { type: 'text/csv;charset=utf-8;' });
    const link = document.createElement('a');
    const url = URL.createObjectURL(blob);
    link.setAttribute('href', url);
    link.setAttribute('download', `fuel_entries_${new Date().toISOString().split('T')[0]}.csv`);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
}

// Event listeners
document.addEventListener('DOMContentLoaded', function() {
    fetchFuelData();
    
    const fuelSearch = document.getElementById('fuelSearch');
    if (fuelSearch) {
        fuelSearch.addEventListener('input', filterFuelData);
    }
    
    const exportBtn = document.getElementById('exportFuelBtn');
    if (exportBtn) {
        exportBtn.addEventListener('click', exportFuelToExcel);
    }
});
</script>

<style>
/* CSS Variables for consistent theming - matching Create New Workorder */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
}

/* Reset and base styles */
* {
    margin: 0;
    padding: 0;
    box-sizing: border-box;
}

body {
    font-family: var(--font-family);
    background-color: var(--bg-light);
    color: var(--text-primary);
}

/* Main Container */
.fuel-logger-container {
    min-height: 100vh;
    background: var(--bg-light);
}

/* Header Styles - matching Create New Workorder */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
    margin-bottom: 30px;
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-subtitle {
    color: rgba(255, 255, 255, 0.9);
    font-size: 16px;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 20px;
    flex-wrap: wrap;
}

/* Action Button Styles */
.action-buttons {
    display: flex;
    gap: 12px;
    align-items: center;
}

.btn-primary {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 20px;
    background: white;
    color: var(--primary-color);
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    font-family: var(--font-family);
    text-decoration: none;
}

.btn-primary:hover {
    background: rgba(255, 255, 255, 0.9);
    color: var(--primary-color);
    text-decoration: none;
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

.btn-secondary {
    background: var(--secondary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    padding: 12px 20px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    transition: var(--transition-base);
}

.btn-secondary:hover {
    background: #5a6268;
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

.btn-success {
    background: var(--success-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    padding: 12px 20px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    display: inline-flex;
    align-items: center;
    gap: 8px;
    transition: var(--transition-base);
}

.btn-success:hover {
    background: #218838;
    transform: translateY(-1px);
    box-shadow: var(--shadow-md);
}

/* Search and Action Styles */
.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-input-wrapper {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 16px;
    z-index: 2;
    color: #64748b;
    width: 16px;
    height: 16px;
}

.search-input {
    padding: 12px 16px 12px 44px;
    border: 1px solid rgba(255, 255, 255, 0.3);
    border-radius: var(--radius-md);
    height: 44px;
    width: 300px;
    font-size: 14px;
    color: #374151;
    background: rgba(255, 255, 255, 0.9);
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.search-input:focus {
    outline: none;
    border-color: rgba(255, 255, 255, 0.8);
    background: white;
    box-shadow: 0 0 0 3px rgba(255, 255, 255, 0.2);
}

/* Table Section */
.table-section {
    background: var(--bg-primary);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
    margin: 0 20px 30px 20px;
    padding: 20px;
}

.table-container {
    overflow-x: auto;
}

.fuel-logger-table {
    width: 100%;
    border-collapse: collapse;
    font-size: 14px;
}

.fuel-logger-table thead {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.fuel-logger-table th {
    padding: 16px 12px;
    text-align: left;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e2e8f0;
    font-size: 13px;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    white-space: nowrap;
}

.fuel-logger-table td {
    padding: 16px 12px;
    border-bottom: 1px solid #f1f5f9;
    vertical-align: middle;
    white-space: nowrap;
}

/* No Data Styling */
.no-data-cell {
    padding: 64px 24px !important;
    text-align: center;
}

.no-data-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.no-data-content svg {
    color: var(--text-light);
}

.no-data-content h3 {
    color: var(--text-secondary);
    font-weight: 600;
    font-size: 18px;
    margin: 0;
}

.no-data-content p {
    color: var(--text-muted);
    font-weight: 400;
    font-size: 14px;
    margin: 0;
    max-width: 400px;
}

/* Action Button */
.action-btn {
    padding: 8px;
    border: none;
    background: var(--primary-color);
    color: white;
    border-radius: var(--radius-md);
    cursor: pointer;
    transition: var(--transition-base);
    display: inline-flex;
    align-items: center;
    justify-content: center;
}

.action-btn:hover {
    background: var(--primary-hover);
    transform: translateY(-1px);
}

.receipt-link {
    color: var(--primary-color);
    text-decoration: none;
    font-weight: 500;
}

.receipt-link:hover {
    text-decoration: underline;
}

/* Pagination Styles */
.pagination-container {
    display: flex;
    justify-content: space-between;
    align-items: center;
    padding: 20px;
    border-top: 1px solid var(--border-light);
    flex-wrap: wrap;
    gap: 15px;
}

.pagination-info {
    color: var(--text-secondary);
    font-size: 14px;
}

.pagination-info span {
    font-weight: 600;
    color: var(--text-primary);
}

.pagination-controls {
    display: flex;
    gap: 5px;
    align-items: center;
}

.pagination-btn {
    padding: 8px 12px;
    border: 1px solid var(--border-light);
    background: var(--bg-primary);
    color: var(--text-primary);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
    font-family: var(--font-family);
    min-width: 36px;
    text-align: center;
}

.pagination-btn:hover:not(:disabled) {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.pagination-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.pagination-ellipsis {
    padding: 8px 12px;
    color: var(--text-secondary);
    font-size: 14px;
}

/* Responsive Design */
@media (max-width: 768px) {
    .header-content {
        flex-direction: column;
        align-items: stretch;
        gap: 20px;
        padding: 0 15px;
    }
    
    .header-left {
        text-align: center;
    }
    
    .header-right {
        flex-direction: column;
        gap: 15px;
        align-items: stretch;
    }
    
    .search-container {
        order: 1;
    }
    
    .action-buttons {
        order: 2;
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .table-section {
        margin: 0 15px 30px 15px;
        padding: 15px;
    }
    
    .search-input {
        width: 100%;
    }
    
    .pagination-container {
        flex-direction: column;
        align-items: center;
        text-align: center;
    }
    
    .pagination-controls {
        flex-wrap: wrap;
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .header-title {
        font-size: 24px;
    }
    
    .header-subtitle {
        font-size: 14px;
    }
    
    .table-section {
        margin: 0 10px 20px 10px;
        padding: 10px;
    }
    
    .fuel-logger-table th,
    .fuel-logger-table td {
        padding: 8px 6px;
        font-size: 12px;
    }
    
    .btn-success,
    .btn-secondary {
        width: 100%;
        justify-content: center;
    }
}

/* ===== MODAL STYLES ===== */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    z-index: var(--z-index-modal);
    display: none;
    justify-content: center;
    align-items: center;
}

.modal.show {
    display: flex !important;
}

.modal.fade .modal-dialog {
    transition: transform 0.3s ease-out;
    transform: translate(0, -50px);
}

.modal.show .modal-dialog {
    transform: none;
}

.modal-dialog {
    width: 90%;
    max-width: 500px;
    margin: 0 auto;
}

.modal-dialog.modal-lg {
    max-width: 800px;
}

.modal-content {
    background: #ffffff;
    border: none;
    border-radius: 0;
    box-shadow: 0 20px 25px -5px rgba(0, 0, 0, 0.1), 0 10px 10px -5px rgba(0, 0, 0, 0.04);
}

.modal-header {
    padding: 20px 24px;
    border-bottom: 1px solid var(--border-color);
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: var(--primary-color);
    color: white;
}

.modal-title {
    margin: 0;
    color: white;
    font-size: 20px;
    font-weight: 600;
}

.modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: white;
    padding: 8px;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 0;
    transition: background-color 0.2s;
}

.modal-close:hover {
    background-color: rgba(255, 255, 255, 0.2);
}

.modal-body {
    padding: 24px;
}

/* ===== FORM STYLES ===== */
.fuel-entry-form,
.sorting-form,
.upload-form {
    display: flex;
    flex-direction: column;
    gap: 1.5rem;
}

.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.5rem;
}

.form-section {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.section-title {
    font-size: 1rem;
    font-weight: 600;
    color: #374151;
    margin: 0;
    font-family: 'Inter', sans-serif;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 0.5rem;
}

.form-group label {
    font-size: 0.875rem;
    font-weight: 600;
    color: #374151;
    font-family: 'Inter', sans-serif;
}

.form-control {
    border: 1px solid #d1d5db;
    border-radius: 0;
    padding: 12px 16px;
    font-size: 14px;
    transition: var(--transition-base);
    background: var(--bg-primary);
}

.form-control:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.form-control::placeholder {
    color: #94a3b8;
    font-weight: 400;
}

.input-with-button {
    display: flex;
    gap: 0.5rem;
}

.input-with-button .form-control {
    flex: 1;
}

.input-with-select {
    display: flex;
    gap: 0.5rem;
}

.input-with-select .form-control {
    flex: 1;
}

.btn-select {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    padding: 0.75rem;
    background: #f8fafc;
    border: 2px solid #e2e8f0;
    border-radius: 0;
    color: #64748b;
    cursor: pointer;
    transition: all 0.2s ease;
    min-width: 3rem;
}

.btn-select:hover {
    background: #e2e8f0;
    border-color: #cbd5e1;
    color: #475569;
}

.btn-upload {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
    color: #ffffff;
    border: none;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(139, 92, 246, 0.2);
}

.btn-upload:hover {
    background: linear-gradient(135deg, #7c3aed 0%, #6d28d9 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(139, 92, 246, 0.3);
}

/* ===== RADIO STYLES ===== */
.radio-group {
    display: flex;
    gap: 1rem;
    flex-wrap: wrap;
}

.radio-label {
    display: flex;
    align-items: center;
    gap: 0.5rem;
    cursor: pointer;
    font-size: 0.875rem;
    font-weight: 500;
    color: #374151;
    font-family: 'Inter', sans-serif;
}

.radio-label input[type="radio"] {
    display: none;
}

.radio-custom {
    width: 1rem;
    height: 1rem;
    border: 2px solid #d1d5db;
    border-radius: 50%;
    position: relative;
    transition: all 0.2s ease;
}

.radio-label input[type="radio"]:checked + .radio-custom {
    border-color: #3b82f6;
    background: #3b82f6;
}

.radio-label input[type="radio"]:checked + .radio-custom::after {
    content: '';
    position: absolute;
    top: 50%;
    left: 50%;
    transform: translate(-50%, -50%);
    width: 0.375rem;
    height: 0.375rem;
    background: #ffffff;
    border-radius: 50%;
}

/* ===== MODAL ACTIONS ===== */
.modal-actions {
    display: flex;
    gap: 1rem;
    justify-content: flex-end;
    margin-top: 2rem;
}

.btn-save {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: var(--success-color);
    color: white;
    border: 1px solid var(--success-color);
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-save:hover {
    background: var(--success-hover);
    border-color: var(--success-hover);
    transform: translateY(-1px);
}

.btn-reset {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: var(--warning-color);
    color: #000;
    border: 1px solid var(--warning-color);
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-reset:hover {
    background: var(--warning-hover);
    border-color: var(--warning-hover);
    transform: translateY(-1px);
}

.btn-cancel {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: var(--secondary-color);
    color: white;
    border: 1px solid var(--secondary-color);
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-cancel:hover {
    background: var(--secondary-hover);
    border-color: var(--secondary-hover);
    transform: translateY(-1px);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .fuel-logger-container {
        padding: 20px;
    }
    
    .fuel-logger-header {
        padding: 28px;
        margin-bottom: 20px;
    }
    
    .page-title h1 {
        font-size: 28px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-content {
        gap: 16px;
    }
    
    .action-buttons {
        gap: 10px;
    }
    
    .btn-create,
    .btn-sort,
    .btn-export {
        padding: 9px 14px;
        font-size: 13px;
    }
}

@media (max-width: 992px) {
    .fuel-logger-container {
        padding: 16px;
    }
    
    .fuel-logger-header {
        padding: 24px;
        margin-bottom: 18px;
    }
    
    .page-title h1 {
        font-size: 24px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-bar {
        padding: 16px 20px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .search-section {
        width: 100%;
        max-width: none;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: flex-start;
        flex-wrap: wrap;
        gap: 8px;
    }
    
    .btn-create,
    .btn-sort,
    .btn-export {
        padding: 10px 16px;
        font-size: 14px;
        min-height: 44px;
        flex: 1;
        min-width: 120px;
        justify-content: center;
    }
    
    .search-input {
        padding: 10px 12px 10px 40px;
        font-size: 16px;
        min-height: 44px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .fuel-logger-table {
        min-width: 800px;
    }
    
    .table-header th {
        padding: 12px 8px;
        font-size: 13px;
    }
    
    .table-body td {
        padding: 12px 8px;
        font-size: 13px;
    }
}

@media (max-width: 768px) {
    .fuel-logger-container {
        padding: 0;
    }
    
    .fuel-logger-header {
        padding: 16px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 20px;
        margin-bottom: 4px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .fuel-logger-content {
        margin-bottom: 16px;
    }
    
    .action-bar {
        padding: 12px 16px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 12px;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr 1fr;
        gap: 8px;
        width: 100%;
    }
    
    .btn-create {
        grid-column: 1 / -1;
        width: 100%;
        padding: 12px 16px;
        font-size: 14px;
        font-weight: 600;
        min-height: 48px;
    }
    
    .btn-sort,
    .btn-export {
        padding: 10px 8px;
        font-size: 12px;
        min-height: 44px;
        display: flex;
        flex-direction: column;
        align-items: center;
        gap: 4px;
    }
    
    .search-section {
        width: 100%;
        margin-top: 8px;
    }
    
    .search-container {
        width: 100%;
    }
    
    .search-input {
        width: 100%;
        padding: 12px 16px 12px 44px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
        border: 1px solid #e2e8f0;
    }
    
    .fuel-logger-table {
        min-width: 700px;
        font-size: 12px;
    }
    
    .table-header th {
        padding: 10px 6px;
        font-size: 11px;
        background-color: #f8fafc;
        font-weight: 600;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .table-body td {
        padding: 8px 6px;
        font-size: 11px;
        white-space: nowrap;
    }
    
    .empty-state {
        padding: 40px 16px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
    }
    
    .modal-dialog {
        width: 95%;
        margin: 1rem;
    }
    
    .modal-body {
        padding: 1.5rem;
    }
    
    .modal-actions {
        flex-direction: column;
    }
    
    .btn-save,
    .btn-reset,
    .btn-cancel {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 576px) {
    .fuel-logger-container {
        padding: 0;
    }
    
    .fuel-logger-header {
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .page-title h1 {
        font-size: 18px;
        margin-bottom: 3px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .fuel-logger-content {
        margin-bottom: 12px;
    }
    
    .action-bar {
        padding: 10px 12px;
    }
    
    .action-content {
        gap: 10px;
    }
    
    .action-buttons {
        display: grid;
        grid-template-columns: 1fr;
        gap: 6px;
        width: 100%;
    }
    
    .btn-create,
    .btn-export {
        width: 100%;
        padding: 14px 16px;
        font-size: 15px;
        font-weight: 700;
        min-height: 52px;
    }
    
    .btn-sort {
        width: 100%;
        padding: 12px 8px;
        font-size: 11px;
        min-height: 48px;
        flex-direction: row;
        justify-content: center;
        gap: 6px;
    }
    
    .search-input {
        padding: 14px 16px 14px 44px;
        font-size: 15px;
        min-height: 52px;
    }
    
    .table-container {
        border: 1px solid #d1d5db;
    }
    
    .fuel-logger-table {
        min-width: 600px;
        font-size: 11px;
    }
    
    .table-header th {
        padding: 8px 4px;
        font-size: 10px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 6px 4px;
        font-size: 10px;
    }
    
    .empty-state {
        padding: 30px 12px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
    
    .modal-dialog {
        width: 95%;
        margin: 0.5rem;
    }
    
    .modal-body {
        padding: 1rem;
    }
    
    .form-group {
        margin-bottom: 1rem;
    }
    
    .form-control {
        padding: 10px;
        font-size: 15px;
        min-height: 44px;
    }
    
    .btn-save,
    .btn-reset,
    .btn-cancel {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 44px;
    }
}

@media (max-width: 480px) {
    .fuel-logger-header {
        padding: 8px;
        margin-bottom: 8px;
    }
    
    .page-title h1 {
        font-size: 16px;
        margin-bottom: 2px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .fuel-logger-content {
        margin-bottom: 8px;
    }
    
    .action-bar {
        padding: 8px;
    }
    
    .action-content {
        gap: 8px;
    }
    
    .action-buttons {
        gap: 4px;
    }
    
    .btn-create,
    .btn-export {
        padding: 16px 12px;
        font-size: 16px;
        font-weight: 700;
        min-height: 56px;
    }
    
    .btn-sort {
        padding: 14px 6px;
        font-size: 10px;
        min-height: 52px;
        gap: 4px;
    }
    
    .search-input {
        padding: 16px 12px 16px 44px;
        font-size: 16px;
        min-height: 56px;
    }
    
    .fuel-logger-table {
        min-width: 500px;
        font-size: 10px;
    }
    
    .table-header th {
        padding: 6px 3px;
        font-size: 9px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 4px 3px;
        font-size: 9px;
    }
    
    .empty-state {
        padding: 20px 8px;
    }
    
    .empty-content h3 {
        font-size: 13px;
    }
    
    .empty-content p {
        font-size: 11px;
    }
    
    .modal-dialog {
        width: 95%;
        margin: 0.25rem;
    }
    
    .modal-body {
        padding: 0.75rem;
    }
    
    .form-group {
        margin-bottom: 0.75rem;
    }
    
    .form-control {
        padding: 8px;
        font-size: 14px;
        min-height: 40px;
    }
    
    .btn-save,
    .btn-reset,
    .btn-cancel {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 40px;
    }
}

@media (max-width: 360px) {
    .fuel-logger-header {
        padding: 6px;
        margin-bottom: 6px;
    }
    
    .page-title h1 {
        font-size: 14px;
        margin-bottom: 1px;
    }
    
    .page-title p {
        font-size: 10px;
    }
    
    .fuel-logger-content {
        margin-bottom: 6px;
    }
    
    .action-bar {
        padding: 6px;
    }
    
    .action-content {
        gap: 6px;
    }
    
    .action-buttons {
        gap: 3px;
    }
    
    .btn-create,
    .btn-export {
        padding: 18px 8px;
        font-size: 14px;
        font-weight: 700;
        min-height: 60px;
    }
    
    .btn-sort {
        padding: 16px 4px;
        font-size: 9px;
        min-height: 56px;
        gap: 2px;
    }
    
    .search-input {
        padding: 18px 8px 18px 44px;
        font-size: 14px;
        min-height: 60px;
    }
    
    .fuel-logger-table {
        min-width: 450px;
        font-size: 9px;
    }
    
    .table-header th {
        padding: 4px 2px;
        font-size: 8px;
        background-color: #f3f4f6;
    }
    
    .table-body td {
        padding: 3px 2px;
        font-size: 8px;
    }
    
    .empty-state {
        padding: 15px 6px;
    }
    
    .empty-content h3 {
        font-size: 12px;
    }
    
    .empty-content p {
        font-size: 10px;
    }
    
    .modal-dialog {
        width: 95%;
        margin: 0.125rem;
    }
    
    .modal-body {
        padding: 0.5rem;
    }
    
    .form-group {
        margin-bottom: 0.5rem;
    }
    
    .form-control {
        padding: 6px;
        font-size: 13px;
        min-height: 36px;
    }
    
    .btn-save,
    .btn-reset,
    .btn-cancel {
        padding: 6px 10px;
        font-size: 11px;
        min-height: 36px;
    }
}

/* ===== MODAL RESPONSIVE DESIGN ===== */
@media (max-width: 768px) {
    .modal-dialog {
        margin: 1rem;
        max-width: calc(100% - 2rem);
    }
    
    .modal-dialog.modal-lg {
        max-width: calc(100% - 2rem);
    }
    
    .modal-content {
        border-radius: 8px;
    }
    
    .modal-header {
        padding: 1rem;
    }
    
    .modal-header h4 {
        font-size: 1.1rem;
    }
    
    .modal-body {
        padding: 1rem;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .form-group {
        margin-bottom: 1rem;
    }
    
    .form-group label {
        font-size: 0.9rem;
        font-weight: 600;
    }
    
    .form-control {
        padding: 12px;
        font-size: 16px;
        min-height: 48px;
    }
    
    .input-with-button,
    .input-with-select {
        flex-direction: column;
        gap: 0.5rem;
    }
    
    .btn-select {
        padding: 12px;
        min-height: 48px;
    }
    
    .radio-group {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .btn-upload {
        padding: 12px 16px;
        font-size: 14px;
        min-height: 48px;
    }
    
    .btn-save,
    .btn-reset,
    .btn-cancel {
        padding: 12px 16px;
        font-size: 14px;
        min-height: 48px;
    }
}

@media (max-width: 576px) {
    .modal-dialog {
        margin: 0.5rem;
        max-width: calc(100% - 1rem);
    }
    
    .modal-header {
        padding: 0.75rem;
    }
    
    .modal-header h4 {
        font-size: 1rem;
    }
    
    .modal-body {
        padding: 0.75rem;
    }
    
    .form-group {
        margin-bottom: 0.75rem;
    }
    
    .form-group label {
        font-size: 0.85rem;
    }
    
    .form-control {
        padding: 10px;
        font-size: 15px;
        min-height: 44px;
    }
    
    .btn-select {
        padding: 10px;
        min-height: 44px;
    }
    
    .btn-upload {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 44px;
    }
    
    .btn-save,
    .btn-reset,
    .btn-cancel {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 44px;
    }
}

@media (max-width: 480px) {
    .modal-dialog {
        margin: 0.25rem;
        max-width: calc(100% - 0.5rem);
    }
    
    .modal-header {
        padding: 0.5rem;
    }
    
    .modal-header h4 {
        font-size: 0.9rem;
    }
    
    .modal-body {
        padding: 0.5rem;
    }
    
    .form-group {
        margin-bottom: 0.5rem;
    }
    
    .form-group label {
        font-size: 0.8rem;
    }
    
    .form-control {
        padding: 8px;
        font-size: 14px;
        min-height: 40px;
    }
    
    .btn-select {
        padding: 8px;
        min-height: 40px;
    }
    
    .btn-upload {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 40px;
    }
    
    .btn-save,
    .btn-reset,
    .btn-cancel {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 40px;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid rgba(0, 85, 165, 0.5);
    outline-offset: 2px;
}

/* ===== ADDITIONAL REQUEST_MAINTENANCE-STYLE IMPROVEMENTS ===== */
.fuel-logger-container {
    background: #f8fafc;
}

.fuel-logger-content {
    border-radius: 0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
}

.table-header th {
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
}

.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.action-buttons button {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: black !important;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-buttons button:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    transform: translateY(-1px);
}

/* Force sharp corners for all buttons */
.action-buttons button,
.action-buttons .btn-create,
.action-buttons .btn-sort,
.action-buttons .btn-export {
    border-radius: 0 !important;
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid var(--primary-color);
    outline-offset: 2px;
}

/* ===== PRINT STYLES ===== */
@media print {
    .action-bar,
    .modal {
        display: none !important;
    }
    
    .fuel-logger-container {
        background: white;
    }
    
    .fuel-logger-header {
        background: white;
        color: black;
    }
}
</style>
@endsection