@extends('welcome')

@section('title', 'Equipment Status KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="equipment-status-container">
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Equipment Data...</div>
        </div>
    </div>
    
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <button type="button" class="back-button" id="back-btn" onclick="handleBackButton()">
                    <i class="fas fa-arrow-left"></i>
                </button>
                <h1 class="header-title" id="page-title">Equipments Used</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn">
                    <i class="fas fa-file-excel"></i>
                    Export
                </button>
                <button type="button" class="create-btn" id="refresh-btn">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="equipment-status-content">
        <div class="content-box">
            <!-- Equipment List View -->
            <div id="equipment-list-view">
                <!-- Date Range Picker -->
                <div class="date-picker-container">
                    <div class="date-inputs">
                        <div class="date-input-group">
                            <label>From Date:</label>
                            <input type="date" id="from-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label>To Date:</label>
                            <input type="date" id="to-date" class="date-input">
                        </div>
                        <button class="btn-clear-dates" id="clear-dates">Clear</button>
                    </div>
                </div>
                
                <!-- Summary Stats -->
                <div class="summary-header">
                    <div class="summary-stat">
                        <i class="fas fa-cogs"></i>
                        <div>
                            <div class="stat-label">Total Equipment Used</div>
                            <div class="stat-value" id="total-equipment">0</div>
                        </div>
                    </div>
                    <div class="summary-stat">
                        <i class="fas fa-clipboard-check"></i>
                        <div>
                            <div class="stat-label">Total Inspections</div>
                            <div class="stat-value" id="total-inspections">0</div>
                        </div>
                    </div>
                    <div class="summary-stat">
                        <i class="fas fa-briefcase"></i>
                        <div>
                            <div class="stat-label">Total Job Cards</div>
                            <div class="stat-value" id="total-jobcards">0</div>
                        </div>
                    </div>
                </div>
                
                <!-- Equipment List -->
                <div id="equipment-list" class="equipment-list">
                    <div class="loading-state">
                        <i class="fas fa-spinner fa-spin"></i>
                        <span>Loading equipment...</span>
                    </div>
                </div>
            </div>
            
            <!-- Equipment Details View -->
            <div id="equipment-details-view" style="display: none;">
                <!-- Equipment Info Card -->
                <div class="equipment-info-card" id="equipment-info">
                    <div class="info-header">
                        <h2 id="equipment-name">Equipment Name</h2>
                    </div>
                    <div class="info-grid">
                        <div class="info-item">
                            <strong>Number:</strong> <span id="eq-number">-</span>
                        </div>
                        <div class="info-item">
                            <strong>Category:</strong> <span id="eq-category">-</span>
                        </div>
                        <div class="info-item">
                            <strong>Model:</strong> <span id="eq-model">-</span>
                        </div>
                        <div class="info-item">
                            <strong>Meter Reading:</strong> <span id="eq-meter">-</span>
                        </div>
                        <div class="info-item">
                            <strong>Status:</strong> <span id="eq-status">-</span>
                        </div>
                    </div>
                </div>
                
                <!-- Date Range for Details -->
                <div class="date-picker-container">
                    <div class="date-inputs">
                        <div class="date-input-group">
                            <label>From Date:</label>
                            <input type="date" id="details-from-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label>To Date:</label>
                            <input type="date" id="details-to-date" class="date-input">
                        </div>
                        <button class="btn-clear-dates" id="clear-details-dates">Clear</button>
                    </div>
                </div>
                
                <!-- Module Tabs -->
                <div class="module-tabs">
                    <button class="module-tab-btn active" data-module="inspection">
                        <i class="fas fa-clipboard-check"></i>
                        <span>Inspection</span>
                    </button>
                    <button class="module-tab-btn" data-module="jobCard">
                        <i class="fas fa-briefcase"></i>
                        <span>Job Card</span>
                    </button>
                    <button class="module-tab-btn" data-module="maintenance">
                        <i class="fas fa-wrench"></i>
                        <span>Maintenance</span>
                    </button>
                    <button class="module-tab-btn" data-module="serviceOrder">
                        <i class="fas fa-calendar-alt"></i>
                        <span>Service Order</span>
                    </button>
                </div>
                
                <!-- Tab Content -->
                <div id="tab-content-area" class="tab-content-area">
                    <div class="loading-state">
                        <i class="fas fa-spinner fa-spin"></i>
                        <span>Loading data...</span>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>

<script>
const firebaseConfig = { databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com" };
if (!firebase.apps.length) firebase.initializeApp(firebaseConfig);
const database = firebase.database();

let allInspections = [];
let uniqueEquipments = [];
let fromDate = null;
let toDate = null;
let currentView = 'list'; // 'list' or 'details'
let selectedEquipment = null;
let detailsFromDate = null;
let detailsToDate = null;
let activeModule = 'inspection';
let moduleData = [];

function showLoadingState(isLoading) {
    document.getElementById('loading-overlay').style.display = isLoading ? 'flex' : 'none';
}

async function fetchInspections() {
    showLoadingState(true);
    try {
        // 1. Fetch all inspections globally (recursive)
        const snap = await database.ref('/GlobalInspectionReport').once('value');
        const rootData = snap.val() || {};
        
        // 2. Fetch all job cards globally (recursive)
        const jobSnap = await database.ref('/GlobalWorkOrders').once('value');
        const jobRootData = jobSnap.val() || {};
        
        // 3. Build inspection to job card map
        const inspectionToJobCard = {};
        
        function extractJobCards(obj, path = []) {
            if (!obj || typeof obj !== 'object') return;
            
            Object.entries(obj).forEach(([key, item]) => {
                if (key === 'lastRequestNumber' || typeof item !== 'object' || item === null) return;
                
                const hasJobCardFields = item.jobCardNumber || item.requestNumber || item.status || item.equipment;
                if (hasJobCardFields) {
                    const linkId = item.inspectionId || item.linkedInspectionId || item.inspectionRef || item.inspection_id || (item.linkInspection && item.linkInspection.id);
                    if (linkId) {
                        inspectionToJobCard[linkId] = {
                            jobCardNumber: item.jobCardNumber || item.requestNumber || '-',
                            jobCardId: key,
                            jobCardStatus: item.status || '-',
                        };
                    }
                } else {
                    extractJobCards(item, [...path, key]);
                }
            });
        }
        
        extractJobCards(jobRootData);
        
        // 4. Extract all inspections recursively
        const inspectionsList = [];
        
        function extractInspections(obj, path = []) {
            if (!obj || typeof obj !== 'object') return;
            
            Object.entries(obj).forEach(([key, item]) => {
                if (typeof item !== 'object' || item === null) return;
                
                const hasInspectionFields = item.step1 || item.step2 || item.step3 || item.inspectionNumber;
                if (hasInspectionFields) {
                    const jobCard = inspectionToJobCard[key];
                    const createdAt = item?.step3?.createdAt || item?.step1?.createdAt || item?.createdAt || '';
                    
                    inspectionsList.push({
                        id: key,
                        date: createdAt,
                        equipment: item?.step1?.selectedEquipment || {},
                        jobCardStatus: jobCard ? jobCard.jobCardStatus : '-',
                        details: item,
                    });
                } else {
                    extractInspections(item, [...path, key]);
                }
            });
        }
        
        extractInspections(rootData);
        
        allInspections = inspectionsList;
        
        console.log('Fetched inspections:', allInspections.length);
        
        renderEquipmentList();
        updateSummary();
        
        showLoadingState(false);
        showNotification(`Loaded ${allInspections.length} inspections`, 'success');
    } catch (error) {
        console.error('Error:', error);
        showNotification('Error loading data', 'error');
        showLoadingState(false);
    }
}

function renderEquipmentList() {
    const container = document.getElementById('equipment-list');
    
    // Filter inspections by date and status
    const filteredInspections = allInspections.filter(item => {
        if (item.date) {
            const date = new Date(item.date);
            
            if (fromDate) {
                const from = new Date(fromDate);
                from.setHours(0, 0, 0, 0);
                const itemDate = new Date(date);
                itemDate.setHours(0, 0, 0, 0);
                if (itemDate < from) return false;
            }
            
            if (toDate) {
                const to = new Date(toDate);
                to.setHours(23, 59, 59, 999);
                if (date > to) return false;
            }
        }
        
        // Exclude completed/success/rejected
        const status = (item.jobCardStatus || '').toLowerCase();
        if (status === 'complete' || status === 'completed' || status === 'success' || status === 'rejected') return false;
        
        return true;
    });
    
    // Collect unique equipment
    const equipmentMap = {};
    filteredInspections.forEach(item => {
        const eq = item.equipment;
        if (eq && eq.equipmentNumber) {
            equipmentMap[eq.equipmentNumber] = eq;
        }
    });
    
    uniqueEquipments = Object.values(equipmentMap);
    
    if (uniqueEquipments.length === 0) {
        container.innerHTML = '<div class="no-data">No equipment found for selected criteria</div>';
        return;
    }
    
    const html = uniqueEquipments.map((eq, idx) => `
        <div class="equipment-card" onclick='showEquipmentDetails(${JSON.stringify(eq).replace(/'/g, "&apos;")})'>
            <div class="equipment-header">
                <i class="fas fa-cog"></i>
                <h3>${eq.equipmentName || '-'}</h3>
            </div>
            <div class="equipment-info-grid">
                <div class="equipment-info-item">
                    <strong>Number:</strong> ${eq.equipmentNumber || '-'}
                </div>
                <div class="equipment-info-item">
                    <strong>Category:</strong> ${eq.mainCategory || '-'}
                </div>
                <div class="equipment-info-item">
                    <strong>Model:</strong> ${eq.model || '-'}
                </div>
                <div class="equipment-info-item">
                    <strong>Meter:</strong> ${eq.meterReading || '-'} ${eq.meterUnit || ''}
                </div>
                <div class="equipment-info-item">
                    <strong>Status:</strong> <span class="eq-status">${eq.status || '-'}</span>
                </div>
            </div>
            <div class="view-details-btn">
                <i class="fas fa-eye"></i> View Details
            </div>
        </div>
    `).join('');
    
    container.innerHTML = html;
}

function updateSummary() {
    const filteredInspections = allInspections.filter(item => {
        if (item.date) {
            const date = new Date(item.date);
            if (fromDate) {
                const from = new Date(fromDate);
                from.setHours(0, 0, 0, 0);
                const itemDate = new Date(date);
                itemDate.setHours(0, 0, 0, 0);
                if (itemDate < from) return false;
            }
            if (toDate) {
                const to = new Date(toDate);
                to.setHours(23, 59, 59, 999);
                if (date > to) return false;
            }
        }
        const status = (item.jobCardStatus || '').toLowerCase();
        if (status === 'complete' || status === 'completed' || status === 'success' || status === 'rejected') return false;
        return true;
    });
    
    // Count job cards
    let jobCardCount = 0;
    filteredInspections.forEach(item => {
        if (item.details?.step1?.selectedEquipment) jobCardCount++;
    });
    
    document.getElementById('total-equipment').textContent = uniqueEquipments.length;
    document.getElementById('total-inspections').textContent = filteredInspections.length;
    document.getElementById('total-jobcards').textContent = jobCardCount;
}

function handleBackButton() {
    if (currentView === 'details') {
        showEquipmentList();
    } else {
        window.location.href = "{{ route('dashboard.team_kpi.home') }}";
    }
}

function showEquipmentDetails(equipment) {
    selectedEquipment = equipment;
    currentView = 'details';
    
    // Update UI
    document.getElementById('equipment-list-view').style.display = 'none';
    document.getElementById('equipment-details-view').style.display = 'block';
    document.getElementById('page-title').textContent = 'Equipment Details';
    
    // Update equipment info
    document.getElementById('equipment-name').textContent = equipment.equipmentName || '-';
    document.getElementById('eq-number').textContent = equipment.equipmentNumber || '-';
    document.getElementById('eq-category').textContent = equipment.mainCategory || '-';
    document.getElementById('eq-model').textContent = equipment.model || '-';
    document.getElementById('eq-meter').textContent = `${equipment.meterReading || '-'} ${equipment.meterUnit || ''}`;
    document.getElementById('eq-status').textContent = equipment.status || '-';
    
    // Fetch module data
    fetchModuleData();
}

function showEquipmentList() {
    currentView = 'list';
    selectedEquipment = null;
    detailsFromDate = null;
    detailsToDate = null;
    
    document.getElementById('equipment-list-view').style.display = 'block';
    document.getElementById('equipment-details-view').style.display = 'none';
    document.getElementById('page-title').textContent = 'Equipments Used';
    
    // Clear details date inputs
    document.getElementById('details-from-date').value = '';
    document.getElementById('details-to-date').value = '';
}

async function fetchModuleData() {
    if (!selectedEquipment || !selectedEquipment.equipmentNumber) return;
    
    showLoadingState(true);
    const eqNumber = selectedEquipment.equipmentNumber;
    let results = [];
    
    try {
        if (activeModule === 'inspection') {
            const snap = await database.ref('/GlobalInspectionReport').once('value');
            const data = snap.val() || {};
            
            function extractInspections(obj) {
                if (!obj || typeof obj !== 'object') return;
                Object.values(obj).forEach(item => {
                    if (typeof item !== 'object' || item === null) return;
                    if (item.step1 || item.step2 || item.step3) {
                        if (item?.step1?.selectedEquipment?.equipmentNumber === eqNumber) {
                            const createdAt = item?.step3?.createdAt || item?.step1?.createdAt;
                            if (filterByDate(createdAt)) {
                                results.push(item);
                            }
                        }
                    } else {
                        extractInspections(item);
                    }
                });
            }
            extractInspections(data);
            
        } else if (activeModule === 'jobCard') {
            const snap = await database.ref('/GlobalWorkOrders').once('value');
            const data = snap.val() || {};
            
            function extractJobCards(obj) {
                if (!obj || typeof obj !== 'object') return;
                Object.entries(obj).forEach(([key, item]) => {
                    if (key === 'lastRequestNumber' || typeof item !== 'object' || item === null) return;
                    if (item.equipment || item.jobCardNumber || item.status) {
                        if (item?.equipment?.equipmentNumber === eqNumber) {
                            if (filterByDate(item.createdAt)) {
                                results.push(item);
                            }
                        }
                    } else {
                        extractJobCards(item);
                    }
                });
            }
            extractJobCards(data);
            
        } else if (activeModule === 'maintenance') {
            const snap = await database.ref('/GlobalRequestMaintenancereport').once('value');
            const data = snap.val() || {};
            
            function extractMaintenance(obj) {
                if (!obj || typeof obj !== 'object') return;
                Object.values(obj).forEach(item => {
                    if (typeof item !== 'object' || item === null) return;
                    if (item.equipment || item.scheduleId || item.title) {
                        if (item?.equipment?.equipmentNumber === eqNumber) {
                            if (filterByDate(item.createdAt)) {
                                results.push(item);
                            }
                        }
                    } else {
                        extractMaintenance(item);
                    }
                });
            }
            extractMaintenance(data);
            
        } else if (activeModule === 'serviceOrder') {
            const snap = await database.ref('/GlobalServiceScheduled').once('value');
            const data = snap.val() || {};
            
            function extractServiceOrders(obj) {
                if (!obj || typeof obj !== 'object') return;
                Object.values(obj).forEach(item => {
                    if (typeof item !== 'object' || item === null) return;
                    if (item.equipment || item.scheduleId || item.title) {
                        const itemEqNum = item?.equipment?.equipmentNumber || item?.equipmentNumber;
                        if (itemEqNum === eqNumber) {
                            if (filterByDate(item.createdAt)) {
                                results.push(item);
                            }
                        }
                    } else {
                        extractServiceOrders(item);
                    }
                });
            }
            extractServiceOrders(data);
        }
        
        moduleData = results.sort((a, b) => {
            const dateA = a.createdAt || a?.step3?.createdAt || '';
            const dateB = b.createdAt || b?.step3?.createdAt || '';
            return new Date(dateB) - new Date(dateA);
        });
        
        renderModuleData();
        showLoadingState(false);
    } catch (error) {
        console.error('Error:', error);
        showLoadingState(false);
        document.getElementById('tab-content-area').innerHTML = '<div class="no-data">Error loading data</div>';
    }
}

function filterByDate(dateStr) {
    if (!dateStr) return false;
    const date = new Date(dateStr);
    
    if (detailsFromDate) {
        const from = new Date(detailsFromDate);
        from.setHours(0, 0, 0, 0);
        const itemDate = new Date(date);
        itemDate.setHours(0, 0, 0, 0);
        if (itemDate < from) return false;
    }
    
    if (detailsToDate) {
        const to = new Date(detailsToDate);
        to.setHours(23, 59, 59, 999);
        if (date > to) return false;
    }
    
    return true;
}

function renderModuleData() {
    const container = document.getElementById('tab-content-area');
    
    if (moduleData.length === 0) {
        container.innerHTML = '<div class="no-data">No data found for this equipment</div>';
        return;
    }
    
    let html = '';
    
    if (activeModule === 'inspection') {
        html = moduleData.map((item, idx) => `
            <div class="data-card">
                <h4>Inspection #${item.step1?.inspectionNumber || item.step1?.requestNumber || '-'}</h4>
                <div class="data-field"><strong>Date:</strong> ${item.step3?.createdAt ? new Date(item.step3.createdAt).toLocaleString('en-GB') : '-'}</div>
                <div class="data-field"><strong>Inspector:</strong> ${item.step1?.inspector || '-'}</div>
                <div class="data-field"><strong>Status:</strong> ${item.step3?.status || '-'}</div>
                <div class="data-field"><strong>Project:</strong> ${item.step1?.project || '-'}</div>
                <div class="data-field"><strong>Country:</strong> ${item.step1?.country || '-'}</div>
            </div>
        `).join('');
    } else if (activeModule === 'jobCard') {
        html = moduleData.map((item, idx) => `
            <div class="data-card">
                <h4>Job Card #${item.jobCardNumber || item.requestNumber || '-'}</h4>
                <div class="data-field"><strong>Date:</strong> ${item.createdAt ? new Date(item.createdAt).toLocaleString('en-GB') : '-'}</div>
                <div class="data-field"><strong>Status:</strong> <span class="status-badge" style="background: ${getStatusColor(item.status)}">${item.status || '-'}</span></div>
                <div class="data-field"><strong>Priority:</strong> ${item.priority || '-'}</div>
                <div class="data-field"><strong>Project:</strong> ${item.project || '-'}</div>
                <div class="data-field"><strong>Country:</strong> ${item.country || '-'}</div>
            </div>
        `).join('');
    } else if (activeModule === 'maintenance') {
        html = moduleData.map((item, idx) => `
            <div class="data-card">
                <h4>Maintenance #${item.scheduleId || item.requestNumber || '-'}</h4>
                <div class="data-field"><strong>Date:</strong> ${item.createdAt ? new Date(item.createdAt).toLocaleString('en-GB') : '-'}</div>
                <div class="data-field"><strong>Title:</strong> ${item.title || '-'}</div>
                <div class="data-field"><strong>Status:</strong> <span class="status-badge" style="background: ${getStatusColor(item.status)}">${item.status || '-'}</span></div>
                <div class="data-field"><strong>Priority:</strong> ${item.priority || '-'}</div>
                <div class="data-field"><strong>Project:</strong> ${item.project || '-'}</div>
            </div>
        `).join('');
    } else if (activeModule === 'serviceOrder') {
        html = moduleData.map((item, idx) => `
            <div class="data-card">
                <h4>Service Order #${item.scheduleId || '-'}</h4>
                <div class="data-field"><strong>Date:</strong> ${item.createdAt ? new Date(item.createdAt).toLocaleString('en-GB') : '-'}</div>
                <div class="data-field"><strong>Title:</strong> ${item.title || '-'}</div>
                <div class="data-field"><strong>Status:</strong> <span class="status-badge" style="background: ${getStatusColor(item.status)}">${item.status || '-'}</span></div>
                <div class="data-field"><strong>Project:</strong> ${item.project || '-'}</div>
                <div class="data-field"><strong>Last Service:</strong> ${item.lastServiceDate || '-'}</div>
            </div>
        `).join('');
    }
    
    container.innerHTML = html;
}

function getStatusColor(status) {
    if (typeof status !== 'string') return '#888';
    const s = status.toLowerCase();
    if (s === 'completed' || s === 'complete' || s === 'success') return '#2ecc40';
    if (s === 'rejected') return '#d81b60';
    if (s === 'processing' || s === 'in progress') return '#ff9800';
    if (s === 'initialized' || s === 'on hold') return '#fbc02d';
    return '#888';
}

function exportToExcel() {
    if (currentView === 'list') {
        exportEquipmentList();
    } else {
        exportEquipmentDetails();
    }
}

function exportEquipmentList() {
    if (uniqueEquipments.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    const excelData = uniqueEquipments.map((eq, index) => ({
        'S.No': index + 1,
        'Equipment Name': eq.equipmentName || '-',
        'Equipment Number': eq.equipmentNumber || '-',
        'Category': eq.mainCategory || '-',
        'Model': eq.model || '-',
        'Meter Reading': eq.meterReading || '-',
        'Meter Unit': eq.meterUnit || '-',
        'Status': eq.status || '-',
    }));
    
    const ws = XLSX.utils.json_to_sheet(excelData);
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Equipment List');
    
    const summary = [
        { 'Metric': 'Total Equipment', 'Value': uniqueEquipments.length },
        { 'Metric': 'Date Range From', 'Value': fromDate ? fromDate.toLocaleDateString('en-GB') : 'Not Set' },
        { 'Metric': 'Date Range To', 'Value': toDate ? toDate.toLocaleDateString('en-GB') : 'Not Set' },
        { 'Metric': 'Export Date', 'Value': new Date().toLocaleString('en-GB') }
    ];
    const wsSummary = XLSX.utils.json_to_sheet(summary);
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    XLSX.writeFile(wb, `Equipment_List_${new Date().toISOString().split('T')[0]}.xlsx`);
    showNotification('Equipment list exported successfully', 'success');
}

function exportEquipmentDetails() {
    if (moduleData.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    const moduleTitles = {
        inspection: 'Inspections',
        jobCard: 'Job Cards',
        maintenance: 'Maintenance',
        serviceOrder: 'Service Orders'
    };
    
    const wb = XLSX.utils.book_new();
    
    // Create equipment header info
    const equipmentHeader = [
        ['EQUIPMENT DETAILS'],
        ['Equipment Name:', selectedEquipment.equipmentName || '-'],
        ['Equipment Number:', selectedEquipment.equipmentNumber || '-'],
        ['Category:', selectedEquipment.mainCategory || '-'],
        ['Model:', selectedEquipment.model || '-'],
        ['Meter Reading:', `${selectedEquipment.meterReading || '-'} ${selectedEquipment.meterUnit || ''}`],
        ['Status:', selectedEquipment.status || '-'],
        [''],
        ['MODULE: ' + moduleTitles[activeModule]],
        ['']
    ];
    
    let excelData = [];
    
    if (activeModule === 'inspection') {
        excelData = moduleData.map((item, idx) => ({
            'S.No': idx + 1,
            'Inspection Number': item.step1?.inspectionNumber || item.step1?.requestNumber || '-',
            'Inspection Name': item.step1?.inspectionName || item.step1?.title || '-',
            'Date': item.step3?.createdAt ? new Date(item.step3.createdAt).toLocaleString('en-GB') : '-',
            'Inspector': item.step1?.inspector || '-',
            'Status': item.step3?.status || '-',
            'Project': item.step1?.project || '-',
            'Country': item.step1?.country || '-'
        }));
    } else if (activeModule === 'jobCard') {
        excelData = moduleData.map((item, idx) => ({
            'S.No': idx + 1,
            'Job Card Number': item.jobCardNumber || item.requestNumber || '-',
            'Equipment Name': selectedEquipment.equipmentName || '-',
            'Equipment Number': selectedEquipment.equipmentNumber || '-',
            'Date': item.createdAt ? new Date(item.createdAt).toLocaleString('en-GB') : '-',
            'Status': item.status || '-',
            'Priority': item.priority || '-',
            'Project': item.project || '-',
            'Country': item.country || '-'
        }));
    } else if (activeModule === 'maintenance') {
        excelData = moduleData.map((item, idx) => ({
            'S.No': idx + 1,
            'Schedule ID': item.scheduleId || '-',
            'Title': item.title || '-',
            'Equipment Name': selectedEquipment.equipmentName || '-',
            'Equipment Number': selectedEquipment.equipmentNumber || '-',
            'Date': item.createdAt ? new Date(item.createdAt).toLocaleString('en-GB') : '-',
            'Status': item.status || '-',
            'Priority': item.priority || '-',
            'Project': item.project || '-'
        }));
    } else if (activeModule === 'serviceOrder') {
        excelData = moduleData.map((item, idx) => ({
            'S.No': idx + 1,
            'Schedule ID': item.scheduleId || '-',
            'Title': item.title || '-',
            'Equipment Name': selectedEquipment.equipmentName || '-',
            'Equipment Number': selectedEquipment.equipmentNumber || '-',
            'Date': item.createdAt ? new Date(item.createdAt).toLocaleString('en-GB') : '-',
            'Status': item.status || '-',
            'Project': item.project || '-',
            'Last Service Date': item.lastServiceDate || '-'
        }));
    }
    
    // Combine header and data
    const ws = XLSX.utils.aoa_to_sheet(equipmentHeader);
    XLSX.utils.sheet_add_json(ws, excelData, { origin: -1, skipHeader: false });
    
    // Style the header (merge cells for equipment name)
    ws['!merges'] = [{ s: { r: 0, c: 0 }, e: { r: 0, c: 5 } }];
    
    // Set column widths
    ws['!cols'] = [
        { wch: 8 }, { wch: 20 }, { wch: 25 }, { wch: 25 }, { wch: 25 },
        { wch: 20 }, { wch: 18 }, { wch: 18 }, { wch: 20 }
    ];
    
    XLSX.utils.book_append_sheet(wb, ws, moduleTitles[activeModule]);
    
    const summary = [
        { 'Metric': 'Equipment Name', 'Value': selectedEquipment.equipmentName || '-' },
        { 'Metric': 'Equipment Number', 'Value': selectedEquipment.equipmentNumber || '-' },
        { 'Metric': 'Category', 'Value': selectedEquipment.mainCategory || '-' },
        { 'Metric': 'Model', 'Value': selectedEquipment.model || '-' },
        { 'Metric': 'Module', 'Value': moduleTitles[activeModule] },
        { 'Metric': 'Total Records', 'Value': moduleData.length },
        { 'Metric': 'Date Range From', 'Value': detailsFromDate ? detailsFromDate.toLocaleDateString('en-GB') : 'All Dates' },
        { 'Metric': 'Date Range To', 'Value': detailsToDate ? detailsToDate.toLocaleDateString('en-GB') : 'All Dates' },
        { 'Metric': 'Export Date', 'Value': new Date().toLocaleString('en-GB') }
    ];
    const wsSummary = XLSX.utils.json_to_sheet(summary);
    wsSummary['!cols'] = [{ wch: 25 }, { wch: 35 }];
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    const eqName = (selectedEquipment.equipmentName || 'Equipment').replace(/[^a-zA-Z0-9]/g, '_');
    XLSX.writeFile(wb, `${eqName}_${moduleTitles[activeModule]}_${new Date().toISOString().split('T')[0]}.xlsx`);
    showNotification('Data exported successfully', 'success');
}

function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `<div class="notification-content"><span>${message}</span><button class="notification-close">&times;</button></div>`;
    document.body.appendChild(notification);
    setTimeout(() => notification.remove(), 5000);
    notification.querySelector('.notification-close').addEventListener('click', () => notification.remove());
}

document.addEventListener('DOMContentLoaded', () => {
    // Set default date range (last 7 days)
    const today = new Date();
    today.setHours(23, 59, 59, 999);
    const oneWeekAgo = new Date(today);
    oneWeekAgo.setDate(today.getDate() - 7);
    oneWeekAgo.setHours(0, 0, 0, 0);
    
    fromDate = oneWeekAgo;
    toDate = today;
    
    document.getElementById('from-date').value = oneWeekAgo.toISOString().split('T')[0];
    document.getElementById('to-date').value = new Date().toISOString().split('T')[0];
    
    fetchInspections();
    
    // Main date range
    document.getElementById('from-date').addEventListener('change', function() {
        if (this.value) {
            fromDate = new Date(this.value);
            fromDate.setHours(0, 0, 0, 0);
            document.getElementById('to-date').min = this.value;
        } else {
            fromDate = null;
        }
        renderEquipmentList();
        updateSummary();
    });
    
    document.getElementById('to-date').addEventListener('change', function() {
        if (this.value) {
            toDate = new Date(this.value);
            toDate.setHours(23, 59, 59, 999);
        } else {
            toDate = null;
        }
        renderEquipmentList();
        updateSummary();
    });
    
    document.getElementById('clear-dates').addEventListener('click', () => {
        document.getElementById('from-date').value = '';
        document.getElementById('to-date').value = '';
        fromDate = null;
        toDate = null;
        renderEquipmentList();
        updateSummary();
        showNotification('Date range cleared', 'info');
    });
    
    // Details date range
    document.getElementById('details-from-date').addEventListener('change', function() {
        if (this.value) {
            detailsFromDate = new Date(this.value);
            detailsFromDate.setHours(0, 0, 0, 0);
            document.getElementById('details-to-date').min = this.value;
        } else {
            detailsFromDate = null;
        }
        fetchModuleData();
    });
    
    document.getElementById('details-to-date').addEventListener('change', function() {
        if (this.value) {
            detailsToDate = new Date(this.value);
            detailsToDate.setHours(23, 59, 59, 999);
        } else {
            detailsToDate = null;
        }
        fetchModuleData();
    });
    
    document.getElementById('clear-details-dates').addEventListener('click', () => {
        document.getElementById('details-from-date').value = '';
        document.getElementById('details-to-date').value = '';
        detailsFromDate = null;
        detailsToDate = null;
        fetchModuleData();
        showNotification('Date range cleared', 'info');
    });
    
    // Module tabs
    document.querySelectorAll('.module-tab-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const module = this.getAttribute('data-module');
            activeModule = module;
            
            document.querySelectorAll('.module-tab-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            
            fetchModuleData();
        });
    });
    
    // Refresh
    document.getElementById('refresh-btn').addEventListener('click', () => {
        showNotification('Refreshing data...', 'info');
        if (currentView === 'list') {
            fetchInspections();
        } else {
            fetchModuleData();
        }
    });
    
    // Export
    document.getElementById('export-excel-btn').addEventListener('click', exportToExcel);
});
</script>

<style>
@import url('https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap');
:root{--primary-color:#0055a5;--primary-hover:#003f7d;--text-primary:#212529;--text-secondary:#6c757d;--bg-light:#f8fafc;--bg-white:#ffffff;--border-light:#e2e8f0;--shadow-sm:0 .125rem .25rem rgba(0,0,0,.075);--shadow-md:0 .5rem 1rem rgba(0,0,0,.15);--shadow-lg:0 1rem 3rem rgba(0,0,0,.175);--radius-md:.375rem;--radius-lg:.5rem;--transition-base:all .15s ease-in-out;--font-family:'Jost',-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif}
.equipment-status-container{min-height:100vh;background:var(--bg-light);font-family:var(--font-family)}
.header-gradient{background:#0055a5;padding:20px 0;box-shadow:0 4px 12px rgba(0,85,165,.3)}
.header-content{width:100%;padding:0 20px;display:flex;align-items:center;justify-content:space-between}
.header-left{display:flex;align-items:center;gap:15px}
.back-button{color:white;font-size:20px;background:transparent;border:none;padding:8px;border-radius:50%;transition:var(--transition-base);cursor:pointer;display:flex;align-items:center;justify-content:center}
.back-button:hover{background-color:rgba(255,255,255,.1)}
.header-title{color:white;font-size:28px;font-weight:700;margin:0}
.header-right{display:flex;gap:15px}
.create-btn{background:white;color:var(--primary-color);border:none;padding:12px 24px;border-radius:var(--radius-md);font-size:16px;font-weight:600;cursor:pointer;transition:var(--transition-base);box-shadow:var(--shadow-sm);display:inline-flex;align-items:center;gap:8px}
.create-btn:hover{background:rgba(255,255,255,.9);transform:translateY(-2px);box-shadow:var(--shadow-md)}
.equipment-status-content{padding:30px 20px}
.content-box{background:white;padding:30px;border-radius:var(--radius-lg);box-shadow:var(--shadow-md);border:1px solid var(--border-light)}
.date-picker-container{margin-bottom:20px;display:flex;justify-content:center}
.date-inputs{display:flex;gap:15px;align-items:end;flex-wrap:wrap;justify-content:center}
.date-input-group{display:flex;flex-direction:column;gap:8px}
.date-input-group label{font-size:14px;font-weight:500;color:var(--text-primary)}
.date-input{padding:12px 16px;border:1px solid var(--border-light);border-radius:var(--radius-md);font-size:14px;font-family:var(--font-family);min-width:180px}
.date-input:focus{outline:none;border-color:var(--primary-color);box-shadow:0 0 0 3px rgba(0,85,165,.1)}
.btn-clear-dates{padding:12px 24px;background:white;color:var(--text-primary);border:1px solid var(--border-light);border-radius:var(--radius-md);font-size:14px;font-weight:600;cursor:pointer;transition:var(--transition-base)}
.btn-clear-dates:hover{background:var(--bg-light);border-color:var(--primary-color);color:var(--primary-color)}
.summary-header{display:grid;grid-template-columns:repeat(auto-fit,minmax(250px,1fr));gap:20px;margin-bottom:30px}
.summary-stat{display:flex;align-items:center;gap:15px;background:linear-gradient(135deg,#f8fafc 0%,#e3f2fd 100%);padding:20px;border-radius:var(--radius-lg);border:1px solid var(--border-light)}
.summary-stat i{font-size:32px;color:var(--primary-color)}
.stat-label{font-size:12px;color:var(--text-secondary);font-weight:500;text-transform:uppercase;margin-bottom:4px}
.stat-value{font-size:28px;color:var(--primary-color);font-weight:700}
.equipment-list{display:grid;grid-template-columns:repeat(auto-fit,minmax(350px,1fr));gap:20px}
.equipment-card{background:#F5F7FA;border-radius:10px;padding:20px;border:1px solid #E0E0E0;box-shadow:var(--shadow-sm);transition:var(--transition-base);cursor:pointer}
.equipment-card:hover{box-shadow:var(--shadow-md);transform:translateY(-2px);border-color:#0055a5}
.equipment-header{display:flex;align-items:center;gap:12px;margin-bottom:15px;padding-bottom:12px;border-bottom:2px solid #E0E0E0}
.equipment-header i{font-size:28px;color:#0055a5}
.equipment-header h3{font-size:18px;font-weight:700;color:#015185;margin:0}
.equipment-info-grid{display:grid;grid-template-columns:1fr;gap:8px;margin-bottom:12px}
.equipment-info-item{font-size:14px;color:#2267A2}
.equipment-info-item strong{color:#025383;margin-right:6px}
.eq-status{color:#2ecc40;font-weight:600}
.view-details-btn{margin-top:12px;padding:8px;background:#015185;color:white;text-align:center;border-radius:6px;font-weight:700;font-size:14px;display:flex;align-items:center;justify-content:center;gap:8px}
.equipment-info-card{background:linear-gradient(135deg,#e3f2fd 0%,#f8fafc 100%);border-radius:10px;padding:25px;margin-bottom:25px;border:2px solid #0055a5;box-shadow:var(--shadow-md)}
.info-header h2{color:#015185;font-size:24px;font-weight:700;margin:0 0 15px 0}
.info-grid{display:grid;grid-template-columns:repeat(auto-fit,minmax(200px,1fr));gap:15px}
.info-item{font-size:15px;color:var(--text-primary)}
.info-item strong{color:#0055a5;margin-right:8px}
.module-tabs{display:flex;gap:10px;margin-bottom:20px;flex-wrap:wrap;justify-content:center}
.module-tab-btn{padding:10px 20px;background:#E0E0E0;border:none;border-radius:18px;color:#015185;font-size:14px;font-weight:600;cursor:pointer;transition:var(--transition-base);display:flex;align-items:center;gap:8px}
.module-tab-btn.active{background:#015185;color:white;box-shadow:0 2px 4px rgba(0,85,165,.3)}
.module-tab-btn:hover:not(.active){background:#d0d0d0}
.tab-content-area{min-height:200px;background:#F5F7FA;border-radius:8px;padding:20px;border:1px solid #E0E0E0}
.data-card{background:white;border-radius:8px;padding:16px;margin-bottom:12px;border:1px solid #E0E0E0;box-shadow:var(--shadow-sm)}
.data-card:last-child{margin-bottom:0}
.data-card h4{color:#015185;font-weight:700;font-size:16px;margin:0 0 10px 0}
.data-field{margin-bottom:6px;font-size:14px;color:#2267A2}
.data-field strong{color:#025383;margin-right:6px}
.status-badge{color:white;padding:4px 10px;border-radius:12px;font-weight:600;font-size:13px}
.loading-state,.no-data{text-align:center;padding:60px;color:var(--text-secondary);font-size:18px}
.loading-state i{margin-right:10px;font-size:24px}
@keyframes spin{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}
.sync-modal{position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,.5);display:flex;align-items:center;justify-content:center;z-index:10001}
.sync-content{background:white;border-radius:var(--radius-lg);padding:40px;text-align:center;max-width:300px;box-shadow:var(--shadow-lg)}
.sync-icon{width:40px;height:40px;margin:0 auto 20px;font-size:40px;animation:spin 1s linear infinite}
.sync-text{font-size:18px;font-weight:600;margin-bottom:10px}
.sync-subtext{font-size:14px;color:var(--text-secondary)}
.notification{position:fixed;top:20px;right:20px;z-index:10000;padding:16px;border-radius:8px;box-shadow:0 4px 12px rgba(0,0,0,.15);font-family:var(--font-family);max-width:400px;background:white;animation:slideIn .3s}
.notification-success{border-left:4px solid #28a745}
.notification-error{border-left:4px solid #dc3545}
.notification-info{border-left:4px solid #17a2b8}
.notification-content{display:flex;justify-content:space-between;align-items:center}
.notification-close{background:none;border:none;font-size:18px;color:#999;cursor:pointer}
@keyframes slideIn{from{transform:translateX(100%);opacity:0}to{transform:translateX(0);opacity:1}}
@media (max-width:768px){.header-title{font-size:20px}.summary-header{grid-template-columns:1fr}.equipment-list{grid-template-columns:1fr}.header-right{flex-direction:column;width:100%}.create-btn{width:100%;justify-content:center}.date-inputs{flex-direction:column;align-items:stretch}.btn-clear-dates,.date-input{width:100%}.module-tabs{flex-direction:column}.info-grid{grid-template-columns:1fr}}
@media (max-width:480px){.header-content{flex-direction:column;gap:15px;align-items:flex-start}.header-right{width:100%}.create-btn{width:100%}}
</style>
@endsection
