import React, { useState, useEffect } from 'react';
import { View, Text, Button, TextInput, Alert, StyleSheet } from 'react-native';
import { launchImageLibrary } from 'react-native-image-picker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import axios from 'axios';

const DocumentuploadS = () => {
  const [file, setFile] = useState(null);
  const [uploadedBy, setUploadedBy] = useState('');
  const [category, setCategory] = useState('');
  const [description, setDescription] = useState('');
  const [uid, setUid] = useState('');
  const [documentType, setDocumentType] = useState('images');
  const [subfolderName, setSubfolderName] = useState('uploads');

  useEffect(() => {
    // Load user ID from AsyncStorage
    const loadUserData = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (userUid) {
          setUid(userUid);
        }
      } catch (error) {
        console.error('Error loading user data:', error);
      }
    };
    
    loadUserData();
  }, []);

  const pickDocument = async () => {
    const options = {
      mediaType: 'photo',
      quality: 1,
      includeBase64: false,
    };

    launchImageLibrary(options, (response) => {
      if (response.didCancel) {
        console.log('User cancelled image picker');
      } else if (response.errorCode) {
        Alert.alert('Error', response.errorMessage || 'Failed to pick image');
      } else if (response.assets && response.assets.length > 0) {
        const selectedImage = response.assets[0];
        setFile({
          uri: selectedImage.uri,
          name: selectedImage.fileName || selectedImage.uri.split('/').pop(),
          mimeType: selectedImage.type || 'image/jpeg',
        });
      }
    });
  };

  const uploadDocument = async () => {
    if (!file) {
      Alert.alert('Error', 'Please select a file first');
      return;
    }

    if (!uid) {
      Alert.alert('Error', 'User ID not found. Please try again.');
      return;
    }

    const formData = new FormData();
    formData.append('file', {
      uri: file.uri,
      name: file.name,
      type: file.mimeType || 'application/octet-stream',
    });
    formData.append('description', description || 'Image upload');
    formData.append('uid', uid);
    formData.append('documentType', documentType);
    formData.append('subfolderName', subfolderName);
    formData.append('type', 'image');
    formData.append('equipment', JSON.stringify({ 
      entryNumber: uploadedBy || 'N/A', 
      equipmentName: category || 'Image' 
    }));

    try {
      const response = await axios.post(
        'http://168.231.113.219:5000/Documents/upload',
        formData,
        {
          headers: {
            'Content-Type': 'multipart/form-data',
          },
        }
      );
      Alert.alert('Success', response.data.message || 'File uploaded successfully');
      setFile(null);
      setUploadedBy('');
      setCategory('');
      setDescription('');
    } catch (error) {
      const errorMessage = error.response?.data?.error || error.message || 'Failed to upload file';
      console.error('Upload error:', error.response ? error.response.data : error.message);
      Alert.alert('Error', errorMessage);
    }
  };

  return (
    <View style={styles.container}>
      <Text style={styles.title}>Upload Document/Image</Text>
      <TextInput
        style={styles.input}
        placeholder="Uploaded By"
        value={uploadedBy}
        onChangeText={setUploadedBy}
      />
      <TextInput
        style={styles.input}
        placeholder="Category"
        value={category}
        onChangeText={setCategory}
      />
      <TextInput
        style={styles.input}
        placeholder="Description"
        value={description}
        onChangeText={setDescription}
      />
      <Button title="Pick Image" onPress={pickDocument} />
      {file && <Text style={styles.fileInfo}>Selected: {file.name}</Text>}
      <Button title="Upload" onPress={uploadDocument} disabled={!file} />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    padding: 20,
    backgroundColor: '#fff',
  },
  title: {
    fontSize: 24,
    fontWeight: 'bold',
    marginBottom: 20,
  },
  input: {
    borderWidth: 1,
    borderColor: '#ccc',
    padding: 10,
    marginBottom: 10,
    borderRadius: 5,
  },
  fileInfo: {
    marginVertical: 10,
    color: '#333',
  },
});

export default DocumentuploadS;