
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Alert,
  Dimensions,
  Platform,
  ScrollView,
  Modal,
  ActivityIndicator,
  Share,
  Linking,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import { Buffer } from 'buffer';

const { width } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Loading' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const Hsemodule43step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [safetyOfficer, setSafetyOfficer] = useState('');
  const [siteSupervisor, setSiteSupervisor] = useState('');
  const [inspectionDate, setInspectionDate] = useState(new Date());
  const [inspectionTime, setInspectionTime] = useState(new Date());
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [showTimePicker, setShowTimePicker] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);
  const [dynamicFields, setDynamicFields] = useState([
    { id: 1, actionToBeTaken: '', date: new Date(), recommendation: '' },
  ]);
  const [showDatePickerFor, setShowDatePickerFor] = useState(null);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [reportId, setReportId] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  const isMounted = useRef(true);

  useEffect(() => {
    isMounted.current = true;
    const loadSavedData = async () => {
      try {
        const savedStep3 = await AsyncStorage.getItem('hseStep3');
        if (savedStep3) {
          const step3Data = JSON.parse(savedStep3);
          setSafetyOfficer(step3Data.safetyOfficer || '');
          setSiteSupervisor(step3Data.siteSupervisor || '');
          setInspectionDate(step3Data.inspectionDate ? new Date(step3Data.inspectionDate) : new Date());
          setInspectionTime(step3Data.inspectionTime ? new Date(step3Data.inspectionTime) : new Date());
          setDynamicFields(
            (step3Data.dynamicFields || [{ id: 1, actionToBeTaken: '', date: new Date(), recommendation: '' }]).map((field) => ({
              ...field,
              date: field.date ? new Date(field.date) : new Date(),
            }))
          );
        }
      } catch (error) {
        console.error('Error loading Step 3 data:', error);
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load saved data.', position: 'bottom' });
      }
    };
    loadSavedData();
    return () => {
      isMounted.current = false;
    };
  }, []);

  const getAdminUserId = async () => {
    const userEmail = await AsyncStorage.getItem('userEmail');
    if (!userEmail) throw new Error('User email not found. Please log in again.');
    const userSnapshot = await database()
      .ref('/Globalusers/admins')
      .orderByChild('email')
      .equalTo(userEmail)
      .once('value');
    const userData = userSnapshot.val();
    if (!userData) throw new Error('User not found in admin database.');
    return Object.keys(userData)[0];
  };

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener(async (state) => {
      if (state.isConnected) {
        try {
          const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
          for (const item of syncQueue) {
            if (item.type === 'hseReport' && item.data && item.data.userId) {
              try {
                await database()
                  .ref(`/GlobalHSEInspections/admins/${item.data.userId}/${item.data.id}`)
                  .set(item.data);
                const newQueue = syncQueue.filter((q) => !(q.type === 'hseReport' && q.data.id === item.data.id));
                await AsyncStorage.setItem('syncQueue', JSON.stringify(newQueue));
              } catch (error) {
                console.error(`Failed to sync report ${item.data.id}:`, error);
              }
            }
          }
        } catch (error) {
          console.error('Error syncing offline queue:', error);
        }
      }
    });
    return () => unsubscribe();
  }, []);

  const deepCleanObject = (obj) => {
    if (typeof obj !== 'object' || obj === null) return obj;
    if (Array.isArray(obj)) return obj.map(deepCleanObject);
    const cleaned = {};
    Object.keys(obj).forEach((key) => {
      const value = obj[key];
      cleaned[key] = typeof value === 'object' && value !== null ? deepCleanObject(value) : String(value || '');
    });
    return cleaned;
  };

  const saveStep3Data = async () => {
    try {
      const step3Data = {
        safetyOfficer,
        siteSupervisor,
        inspectionDate: inspectionDate.toISOString(),
        inspectionTime: inspectionTime.toISOString(),
        dynamicFields: dynamicFields.map((field) => ({
          ...field,
          date: field.date.toISOString(),
        })),
      };
      await AsyncStorage.setItem('hseStep3', JSON.stringify(step3Data));
    } catch (error) {
      console.error('Error saving Step 3 data:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to save Step 3 data locally.', position: 'bottom' });
    }
  };

  const handleBackPress = () => {
    Alert.alert(
      'Go Back',
      'Do you want to return to the previous step? Unsaved changes will be saved locally.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'OK',
          onPress: async () => {
            await saveStep3Data();
            setActiveStep(2);
          },
        },
      ],
      { cancelable: false }
    );
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(Platform.OS === 'ios' ? true : false);
    if (selectedDate && isMounted.current) {
      setInspectionDate(selectedDate);
      saveStep3Data();
    }
  };

  const handleTimeChange = (event, selectedTime) => {
    setShowTimePicker(Platform.OS === 'ios' ? true : false);
    if (selectedTime && isMounted.current) {
      setInspectionTime(selectedTime);
      saveStep3Data();
    }
  };

  const handleDynamicDateChange = (event, selectedDate, fieldId) => {
    setShowDatePickerFor(null);
    if (selectedDate && isMounted.current) {
      setDynamicFields((prev) =>
        prev.map((field) => (field.id === fieldId ? { ...field, date: selectedDate } : field))
      );
      saveStep3Data();
    }
  };

  const addMoreFields = () => {
    const firstEntry = dynamicFields[0];
    if (!firstEntry.actionToBeTaken.trim() || !firstEntry.recommendation.trim()) {
      Toast.show({
        type: 'error',
        text1: 'Required Fields',
        text2: 'Please fill the first entry completely before adding more.',
        position: 'bottom',
      });
      return;
    }
    if (dynamicFields.length < 15) {
      const newId = Math.max(...dynamicFields.map((f) => f.id)) + 1;
      setDynamicFields((prev) => [...prev, { id: newId, actionToBeTaken: '', date: new Date(), recommendation: '' }]);
      saveStep3Data();
    } else {
      Toast.show({ type: 'info', text1: 'Maximum Limit', text2: 'You can add up to 15 entries only.', position: 'bottom' });
    }
  };

  const removeField = (fieldId) => {
    if (dynamicFields.length > 1) {
      setDynamicFields((prev) => prev.filter((field) => field.id !== fieldId));
      saveStep3Data();
    }
  };

  const updateDynamicField = (fieldId, fieldName, value) => {
    setDynamicFields((prev) =>
      prev.map((field) => (field.id === fieldId ? { ...field, [fieldName]: value } : field))
    );
    saveStep3Data();
  };

  const formatDate = (date) => {
    if (!date) return 'Invalid Date';
    let d = date;
    if (typeof d === 'string') d = new Date(d);
    if (d instanceof Date && !isNaN(d)) {
      return d.toLocaleDateString('en-GB', { day: '2-digit', month: '2-digit', year: 'numeric' });
    }
    return 'Invalid Date';
  };

  const formatTime = (date) => {
    if (!date) return 'Invalid Time';
    let d = date;
    if (typeof d === 'string') d = new Date(d);
    if (d instanceof Date && !isNaN(d)) {
      return d.toLocaleTimeString('en-GB', { hour: '2-digit', minute: '2-digit', hour12: false });
    }
    return 'Invalid Time';
  };

  const uploadImageToStorage = async (uri, reportId, itemKey) => {
    try {
      if (!uri || typeof uri !== 'string' || (!uri.startsWith('file://') && !uri.startsWith('/'))) {
        return '';
      }
      const reference = storage().ref(`hse_inspections/${reportId}/${itemKey}`);
      const filePath = uri.startsWith('file://') ? uri.replace('file://', '') : uri;
      await reference.putFile(filePath);
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading image:', error);
      return '';
    }
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hseStep2');
      await AsyncStorage.removeItem('hseStep3');
      await AsyncStorage.removeItem(`hseNotes_${hseId}`);
      await AsyncStorage.removeItem(`hseResults_${hseId}`);
      const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
      const updatedQueue = syncQueue.filter((item) => item.type !== 'hseStep1' || item.data.id !== hseId);
      await AsyncStorage.setItem('syncQueue', JSON.stringify(updatedQueue));
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const queueOfflineReport = async (reportData) => {
    try {
      const syncQueue = JSON.parse(await AsyncStorage.getItem('syncQueue') || '[]');
      syncQueue.push({ type: 'hseReport', data: reportData });
      await AsyncStorage.setItem('syncQueue', JSON.stringify(syncQueue));
    } catch (error) {
      console.error('Error queuing offline report:', error);
      throw new Error('Failed to queue report for offline sync');
    }
  };

  const chunkArray = (array, size) => {
    const result = [];
    for (let i = 0; i < array.length; i += size) {
      result.push(array.slice(i, i + size));
    }
    return result;
  };

  const uploadChunkToFirebase = async (ref, chunk, processedItems, totalItems, setProgress, label = '') => {
    try {
      await Promise.all(
        chunk.map(async (item, index) => {
          const key = `item_${processedItems.current + index}`;
          await ref.child(key).set(item);
        })
      );
      processedItems.current += chunk.length;
      setProgress(Math.round((processedItems.current / totalItems) * 100));
    } catch (error) {
      console.error(`Error uploading chunk for ${label}:`, error);
      Toast.show({ type: 'error', text1: 'Upload Error', text2: `Failed to upload chunk for ${label}.`, position: 'bottom' });
      throw error;
    }
  };

  const flattenChecklist = (checklist) => {
    const flattened = {};
    if (!checklist || typeof checklist !== 'object') {
      return flattened;
    }
    
    Object.entries(checklist).forEach(([section, items]) => {
      if (Array.isArray(items)) {
        items.forEach((item, index) => {
          if (item && typeof item === 'object') {
            Object.entries(item).forEach(([itemName, options]) => {
              flattened[`${section}_${index}_${itemName}`] = options;
            });
          }
        });
      }
    });
    return flattened;
  };

  const getCurrentReportNumber = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      return currentNumber.toString().padStart(3, '0');
    } catch (error) {
      console.error('Error getting current report number:', error);
      return '001';
    }
  };

  const incrementReportCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      let currentNumber = snapshot.val();
      
      // Initialize counter if it doesn't exist
      if (currentNumber === null || currentNumber === undefined) {
        currentNumber = 0;
      }
      
      currentNumber++;
      
      // Update the counter in Firebase
      await counterRef.set(currentNumber);
      
      return currentNumber.toString().padStart(3, '0');
    } catch (error) {
      console.error('Error incrementing report counter:', error);
      return '001';
    }
  };

  const handleGenerateReport = async () => {
    setSyncModalVisible(true);
    setUploadProgress(0);
    try {
      if (!safetyOfficer.trim()) {
        Toast.show({ type: 'error', text1: 'Validation', text2: 'Please enter safety officer name.', position: 'bottom' });
        Alert.alert('Validation', 'Please enter safety officer name.');
        setSyncModalVisible(false);
        return;
      }
      if (!siteSupervisor.trim()) {
        Toast.show({ type: 'error', text1: 'Validation', text2: 'Please enter site supervisor name.', position: 'bottom' });
        Alert.alert('Validation', 'Please enter site supervisor name.');
        setSyncModalVisible(false);
        return;
      }
      if (dynamicFields.some((field) => !field.actionToBeTaken.trim() || !field.recommendation.trim())) {
        Toast.show({ type: 'error', text1: 'Validation', text2: 'Please fill all action and recommendation fields.', position: 'bottom' });
        Alert.alert('Validation', 'Please fill all action and recommendation fields.');
        setSyncModalVisible(false);
        return;
      }

      const userId = await getAdminUserId();
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      if (!savedStep1 || !savedStep2) {
        throw new Error('Previous step data (hseStep1 or hseStep2) is missing');
      }
      const step1 = JSON.parse(savedStep1);
      const step2 = JSON.parse(savedStep2);

      // Use the request number from step 1 data
      const newReportId = step1.requestNumber || `${step1.countryCode || 'ZMB'}-${step1.projectCode || 'PRJ'}-HSE-001`;
      setReportId(newReportId);

      const chunkSize = 10;
      const step1Data = deepCleanObject(step1);
      const step2Data = deepCleanObject(step2);
      const step3Data = {
        safetyOfficer: safetyOfficer.trim(),
        siteSupervisor: siteSupervisor.trim(),
        inspectionDate: inspectionDate.toISOString(),
        inspectionTime: inspectionTime.toISOString(),
        dynamicFields: dynamicFields.map((field) => ({
          ...field,
          date: field.date.toISOString(),
        })),
      };

      // Flatten the checklist to reduce nesting
      if (step1.selectedChecklist && step1.selectedChecklist.checklist) {
        step1Data.selectedChecklist.checklist = flattenedChecklist;
      }

      const reportData = {
        id: newReportId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
      };

      const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${newReportId}`);
      const processedItems = { current: 0 };
      const flattenedChecklist = step1.selectedChecklist && step1.selectedChecklist.checklist ? flattenChecklist(step1.selectedChecklist.checklist) : {};
      const totalItems = 4 + (Object.keys(flattenedChecklist).length || 0) + (step2.sections.length || 0) + dynamicFields.length; // Metadata + flattened checklist + sections + dynamic fields

      // Upload base report data
      await dbRef.update({
        ...reportData,
        'step1': step1Data,
        'step2': { ...step2Data, sections: null }, // Sections will be uploaded separately
        'step3': step3Data,
      });
      processedItems.current += 4;
      setUploadProgress(Math.round((processedItems.current / totalItems) * 100));

      // Upload step2.sections in chunks
      if (Array.isArray(step2.sections) && step2.sections.length > 0) {
        const sectionsChunks = chunkArray(step2.sections, chunkSize);
        for (const chunk of sectionsChunks) {
          await uploadChunkToFirebase(dbRef.child('step2/sections'), chunk, processedItems, totalItems, setUploadProgress, 'step2.sections');
          await new Promise((resolve) => setTimeout(resolve, 100));
        }
      }

      // Upload step3.dynamicFields in chunks
      if (Array.isArray(dynamicFields) && dynamicFields.length > 0) {
        const dynamicFieldsChunks = chunkArray(dynamicFields, chunkSize);
        for (const chunk of dynamicFieldsChunks) {
          await uploadChunkToFirebase(
            dbRef.child('step3/dynamicFields'),
            chunk.map((field) => ({ ...field, date: field.date.toISOString() })),
            processedItems,
            totalItems,
            setUploadProgress,
            'step3.dynamicFields'
          );
          await new Promise((resolve) => setTimeout(resolve, 100));
        }
      }

      // Generate PDF
      const report = {
        id: newReportId,
        step1: step1Data,
        step2: step2Data,
        step3: step3Data,
      };

      const pdfPath = await generatePDFWithHtml(report);
      const storagePath = `GlobalHSEInspections/admins/${userId}/reports/${newReportId}.pdf`;
      const pdfDownloadUrl = await uploadFileToFirebaseStorage(pdfPath, storagePath);

      // Update report with PDF URL
      await dbRef.update({
        pdfDownloadUrl,
        status: 'completed',
      });

      await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));

      // Increment the global report counter only after successful report generation
      await incrementReportCounter();

      await flushStepData(newReportId);

      if (isMounted.current) {
        setSyncModalVisible(false);
        setShowSuccessModal(true);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Report saved successfully.',
          position: 'bottom',
        });
      }
    } catch (error) {
      console.error('Error in handleGenerateReport:', error);
      if (isMounted.current) {
        setSyncModalVisible(false);
        setUploadProgress(0);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to save report.',
          position: 'bottom',
        });
        Alert.alert('Save Error', `Failed to save report.\n\nError: ${error.message}`, [{ text: 'OK' }], { cancelable: false });
      }
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
      }
    }
  };

  const handleCloseSuccessModal = () => {
    if (isMounted.current) {
      setShowSuccessModal(false);
      navigation.navigate('HSEInspectionHome');
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};

      // Get checklist data
      const checklistData = step1.selectedChecklist || {};
      const sections = step2.sections || [];
      const selections = step2.selections || {};
      const notes = step2.notes || {};
      const countInputs = step2.countInputs || {};

      // Helper functions to calculate totals
      const calculateTotalSafeItems = () => {
        let total = 0;
        sections.forEach((section) => {
          section.subItems?.forEach((item) => {
            if (item.inputType === 'Safe/Unsafe/Not Applicable') {
              const countInput = countInputs[item.key] || {};
              if (countInput.safe && countInput.safe !== '') {
                total += parseInt(countInput.safe) || 0;
              }
            } else {
              const selection = selections[item.key];
              if (selection === 'Safe' || selection === 'OK' || selection === 'Okay') {
                total += 1;
              }
            }
          });
        });
        return total;
      };

      const calculateTotalUnsafeItems = () => {
        let total = 0;
        sections.forEach((section) => {
          section.subItems?.forEach((item) => {
            if (item.inputType === 'Safe/Unsafe/Not Applicable') {
              const countInput = countInputs[item.key] || {};
              if (countInput.unsafe && countInput.unsafe !== '') {
                total += parseInt(countInput.unsafe) || 0;
              }
            } else {
              const selection = selections[item.key];
              if (selection === 'Unsafe' || selection === 'Faulty' || selection === 'Not Ok') {
                total += 1;
              }
            }
          });
        });
        return total;
      };

      const calculateTotalNAItems = () => {
        let total = 0;
        sections.forEach((section) => {
          section.subItems?.forEach((item) => {
            if (item.inputType === 'Safe/Unsafe/Not Applicable') {
              const countInput = countInputs[item.key] || {};
              if (countInput.notApplicable === 'checked') {
                total += 1;
              }
            } else {
              const selection = selections[item.key];
              if (selection === 'N/A' || selection === 'Not Applicable') {
                total += 1;
              }
            }
          });
        });
        return total;
      };

      // Generate checklist rows
      let checklistRows = '';
      if (sections.length > 0) {
        sections.forEach((section) => {
          if (section.subItems && section.subItems.length > 0) {
            // Add section header
            checklistRows += `
              <tr>
                <td colspan="5" class="section-header">${section.sectionName || 'SECTION'}</td>
              </tr>
            `;
            
            section.subItems.forEach((item) => {
              const selection = selections[item.key] || '';
              const comment = notes[item.key] || '';
              const countInput = step2.countInputs?.[item.key] || {};
              
              // Check if it's Safe/Unsafe/Not Applicable type
              if (item.inputType === 'Safe/Unsafe/Not Applicable') {
                const safeCount = countInput.safe || '';
                const unsafeCount = countInput.unsafe || '';
                const isNotApplicable = countInput.notApplicable === 'checked';
                
                checklistRows += `
                  <tr>
                    <td>${item.itemName || 'Unknown Item'}</td>
                    <td style="text-align: center;">${isNotApplicable ? '✓' : ''}</td>
                    <td style="text-align: center;">${safeCount || ''}</td>
                    <td style="text-align: center;">${unsafeCount || ''}</td>
                    <td>${comment}</td>
                  </tr>
                `;
              } else {
                // Handle other input types (radio buttons)
                const isSafe = selection === 'Safe' || selection === 'OK' || selection === 'Okay';
                const isUnsafe = selection === 'Unsafe' || selection === 'Faulty' || selection === 'Not Ok';
                const isNA = selection === 'N/A' || selection === 'Not Applicable';
                
                checklistRows += `
                  <tr>
                    <td>${item.itemName || 'Unknown Item'}</td>
                    <td style="text-align: center;">${isNA ? '✓' : ''}</td>
                    <td style="text-align: center;">${isSafe ? '✓' : ''}</td>
                    <td style="text-align: center;">${isUnsafe ? '✓' : ''}</td>
                    <td>${comment}</td>
                  </tr>
                `;
              }
            });
          }
        });
      }

      // Generate additional information rows
      let additionalInfoRows = '';
      if (step3.dynamicFields && step3.dynamicFields.length > 0) {
        step3.dynamicFields.forEach((field, index) => {
          additionalInfoRows += `
            <tr>
              <td>Action ${index + 1}</td>
              <td>${field.actionToBeTaken || ''}</td>
              <td>${field.date ? new Date(field.date).toLocaleDateString() : ''}</td>
              <td>${field.recommendation || ''}</td>
            </tr>
          `;
        });
      }

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>HSE Inspection Report</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 0;
              padding: 0;
              box-sizing: border-box;
            }
            .container {
              width: 100%;
              max-width: 800px;
              margin: 20px auto;
              border: 1px solid #000;
              box-sizing: border-box;
            }
            .header {
              display: flex;
              justify-content: space-between;
              align-items: center;
              background-color: #f0f0f0;
              padding: 10px;
              border-bottom: 1px solid #000;
            }
            .header .logo {
              width: 100px;
              height: 100px;
            }
     
            .header .details table {
              border-collapse: collapse;
              width: 100%;
            }
            .header .details td {
              padding: 2px 0;
              text-align: left;
            }
            .inspection-details {
              display: grid;
              grid-template-columns: 1fr 1fr;
              padding: 5px;
              border-bottom: 1px solid #000;
              gap: 2px;
            }
            .inspection-details div {
              padding: 2px;
            }
            .table {
              width: 100%;
              border-collapse: collapse;
            }
            .table th,
            .table td {
              padding: 8px;
              border: 1px solid #000;
              text-align: left;
            }
            .table th {
              background-color: #d3d3d3;
            }
            .table .section-header {
              background-color: #a9a9a9;
              font-weight: bold;
            }
            .additional-info {
              margin-top: 20px;
            }
            .additional-info h3 {
              background-color: #f0f0f0;
              padding: 10px;
              margin: 0;
              border-bottom: 1px solid #000;
            }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <div class="Logo"><img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo" width="200"></div>
              <div class="details">
                <table border="1">
                  <tr><td>Document name : ${checklistData.document_name || 'HSE Inspection Report'}</td></tr>
                  <tr><td>Document number : ${checklistData.document_id || 'HSE-043'}</td></tr>
                  <tr><td>version : ${checklistData.version || '1'}</td></tr>
                  <tr><td>Revision date : ${checklistData.version_date || new Date().toLocaleDateString()}</td></tr>
                  <tr><td>Approved by : ${checklistData.approved_by || 'HSM'}</td></tr>
                </table>
              </div>
            </div>
            <div class="inspection-details">
              <div>DATE OF INSPECTION</div><div>${step3.inspectionDate ? new Date(step3.inspectionDate).toLocaleDateString() : new Date().toLocaleDateString()}</div>
              <div>SAFETY OFFICER</div><div>${step3.safetyOfficer || 'N/A'}</div>
              <div>TIME OF INSPECTION</div><div>${step3.inspectionTime ? new Date(step3.inspectionTime).toLocaleTimeString() : new Date().toLocaleTimeString()}</div>
              <div>SITE SUPERVISOR</div><div>${step3.siteSupervisor || 'N/A'}</div>
            </div>
            <table class="table">
              <tr>
                <th>CAMP INSPECTION CHECK LIST</th>
                <th>Not Applicable</th>
                <th>Safe</th>
                <th>Unsafe</th>
                <th>Comments</th>
              </tr>
              ${checklistRows}
              <tr>
                <th colspan="5">Total Observations</th>
              </tr>
              <tr>
                <td>Total Safe Items</td>
                <td colspan="4">${calculateTotalSafeItems()}</td>
              </tr>
              <tr>
                <td>Total Unsafe Items</td>
                <td colspan="4">${calculateTotalUnsafeItems()}</td>
              </tr>
              <tr>
                <td>Total N/A Items</td>
                <td colspan="4">${calculateTotalNAItems()}</td>
              </tr>
            </table>
            
            <div class="additional-info">
              <h3>Additional Information</h3>
              <table class="table">
                <tr>
                  <th>Action</th>
                  <th>Action to be Taken</th>
                  <th>Date</th>
                  <th>Recommendation</th>
                </tr>
                ${additionalInfoRows}
              </table>
            </div>
          </div>
        </body>
        </html>
      `;

      const options = {
        html,
        fileName: report.id,
        directory: 'Documents',
      };

      const file = await RNHTMLtoPDF.convert(options);
      return file.filePath;
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    }
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath) => {
    try {
      const reference = storage().ref(storagePath);
      await reference.putFile(filePath);
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading file to Firebase Storage:', error);
      throw error;
    }
  };

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Alert.alert('Error', 'User not found in admin database.');
        navigation.replace('AdminLoginScreen');
        return;
      }
      const userId = Object.keys(userData)[0];

      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      // Navigate to HSEViewPDF screen instead of opening in browser
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => {
          setShowSuccessModal(false);
        }
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };



  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        <View style={styles.headingContainer}>
          <MaterialIcons name="assignment" size={32} color="#015185" />
          <Text style={styles.headingText}>HSE Inspection Report</Text>
          <Text style={styles.subheadingText}>Enter inspection details to generate report</Text>
        </View>
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Safety Officer *</Text>
          <TextInput
            style={styles.textInput}
            value={safetyOfficer}
            onChangeText={(text) => {
              setSafetyOfficer(text);
              saveStep3Data();
            }}
            placeholder="Enter safety officer name"
            placeholderTextColor="#888"
          />
        </View>
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Site Supervisor *</Text>
          <TextInput
            style={styles.textInput}
            value={siteSupervisor}
            onChangeText={(text) => {
              setSiteSupervisor(text);
              saveStep3Data();
            }}
            placeholder="Enter site supervisor name"
            placeholderTextColor="#888"
          />
        </View>
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Date of Inspection *</Text>
          <TouchableOpacity style={styles.pickerButton} onPress={() => setShowDatePicker(true)}>
            <Text style={styles.pickerButtonText}>{formatDate(inspectionDate)}</Text>
            <MaterialIcons name="calendar-today" size={20} color="#015185" />
          </TouchableOpacity>
        </View>
        <View style={styles.inputContainer}>
          <Text style={styles.inputLabel}>Time of Inspection *</Text>
          <TouchableOpacity style={styles.pickerButton} onPress={() => setShowTimePicker(true)}>
            <Text style={styles.pickerButtonText}>{formatTime(inspectionTime)}</Text>
            <MaterialIcons name="access-time" size={20} color="#015185" />
          </TouchableOpacity>
        </View>
        <View style={styles.dynamicSection}>
          <Text style={styles.sectionTitle}>Additional Information</Text>
          {dynamicFields.map((field, index) => (
            <View key={field.id} style={styles.dynamicFieldContainer}>
              <View style={styles.fieldHeader}>
                <Text style={styles.fieldNumber}>Entry {index + 1}</Text>
                {dynamicFields.length > 1 && (
                  <TouchableOpacity style={styles.removeButton} onPress={() => removeField(field.id)}>
                    <MaterialIcons name="delete" size={20} color="#f44336" />
                  </TouchableOpacity>
                )}
              </View>
              <View style={styles.inputContainer}>
                <Text style={styles.inputLabel}>Action to be taken</Text>
                <TextInput
                  style={styles.textInput}
                  value={field.actionToBeTaken}
                  onChangeText={(value) => updateDynamicField(field.id, 'actionToBeTaken', value)}
                  placeholder="Enter action to be taken"
                  placeholderTextColor="#888"
                />
              </View>
              <View style={styles.inputContainer}>
                <Text style={styles.inputLabel}>Date</Text>
                <TouchableOpacity style={styles.pickerButton} onPress={() => setShowDatePickerFor(field.id)}>
                  <Text style={styles.pickerButtonText}>{formatDate(field.date)}</Text>
                  <MaterialIcons name="calendar-today" size={20} color="#015185" />
                </TouchableOpacity>
              </View>
              <View style={styles.inputContainer}>
                <Text style={styles.inputLabel}>Recommendation</Text>
                <TextInput
                  style={styles.textInput}
                  value={field.recommendation}
                  onChangeText={(value) => updateDynamicField(field.id, 'recommendation', value)}
                  placeholder="Enter recommendation"
                  placeholderTextColor="#888"
                />
              </View>
            </View>
          ))}
          <TouchableOpacity style={styles.addMoreButton} onPress={addMoreFields}>
            <MaterialIcons name="add" size={24} color="#fff" />
            <Text style={styles.addMoreText}>Add More</Text>
          </TouchableOpacity>
        </View>
        <View style={styles.infoSection}>
          <MaterialIcons name="info" size={24} color="#015185" />
          <Text style={styles.infoText}>
            All assessment data has been collected. Fill in the inspection details to generate the final report.
          </Text>
        </View>
      </ScrollView>
      {showDatePicker && (
        <DateTimePicker
          value={inspectionDate}
          mode="date"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleDateChange}
        />
      )}
      {showTimePicker && (
        <DateTimePicker
          value={inspectionTime}
          mode="time"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleTimeChange}
        />
      )}
      {showDatePickerFor && (
        <DateTimePicker
          value={dynamicFields.find((field) => field.id === showDatePickerFor)?.date || new Date()}
          mode="date"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={(event, selectedDate) => handleDynamicDateChange(event, selectedDate, showDatePickerFor)}
        />
      )}
      <Modal visible={showSuccessModal} transparent animationType="fade">
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalTitle}>Report Saved Successfully</Text>
            <Text style={styles.successModalText}>
              Your HSE inspection report (ID: {reportId}) has been saved.
            </Text>
            <TouchableOpacity style={styles.modalButton} onPress={handleCloseSuccessModal}>
              <Text style={styles.modalButtonText}>OK</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>


      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              HSE Inspection Report Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.nextButton, syncModalVisible && styles.disabledButton]}
          onPress={async () => {
            try {
              await handleGenerateReport();
            } catch (error) {
              console.error('Generate Report button error:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: 'Unexpected error occurred.',
                position: 'bottom',
              });
            }
          }}
          disabled={syncModalVisible}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <SyncModal visible={syncModalVisible} onClose={() => setSyncModalVisible(false)} message="Processing" />
      <Toast />

    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  headingContainer: {
    alignItems: 'center',
    paddingVertical: 15,
  },
  headingText: {
    fontSize: 20,
    fontWeight: '600',
    color: '#015185',
    marginTop: 5,
  },
  subheadingText: {
    fontSize: 14,
    color: '#666',
    textAlign: 'center',
    marginTop: 5,
  },
  inputContainer: {
    marginBottom: width * 0.04,
  },
  inputLabel: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  pickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    backgroundColor: '#fff',
  },
  pickerButtonText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  dynamicSection: {
    marginTop: 20,
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
  },
  dynamicFieldContainer: {
    backgroundColor: '#f8f9fa',
    padding: 15,
    borderRadius: 8,
    marginBottom: 15,
    borderWidth: 1,
    borderColor: '#e0e0e0',
  },
  fieldHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 10,
  },
  fieldNumber: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  removeButton: {
    padding: 5,
  },
  addMoreButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#015185',
    paddingVertical: 12,
    borderRadius: 5,
    marginTop: 10,
  },
  addMoreText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 5,
  },
  infoSection: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e8f4fd',
    padding: 15,
    borderRadius: 8,
    marginTop: 20,
    marginBottom: 20,
    borderWidth: 1,
    borderColor: '#015185',
  },
  infoText: {
    flex: 1,
    fontSize: 14,
    color: '#333',
    marginLeft: 10,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  disabledButton: {
    opacity: 0.5,
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  modalButton: {
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  modalButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },


  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
});

export default Hsemodule43step3;