import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  FlatList,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import HseChecklists from './HSE43_44_45_126.json';
import HseChecklists41_42 from './HSE41_42.json';
import HseChecklists101_024 from './HSE101_024/HSE101_024Step3.json';
import HseChecklists020 from './HSE_020/HSE_020Json.json';
import HseChecklists071 from './HSE_071/HSE_071Json.json';
import HseChecklists077_078 from './HSE_077_078/HSE_077_078Json.json';
import { HSE_156Json } from './HSE_156/HSE_156Json.js';
import HseChecklists129 from './HSE_129/HSE_129_Json.json';
import HseChecklists072 from './HSE_072/HSE_072Json.json';

const { width } = Dimensions.get('window');

const HseSelectChecklsit = () => {
  const navigation = useNavigation();
  const [checklists, setChecklists] = useState([]);

  useEffect(() => {
    // Load checklists from all JSON files
    const allChecklists = [...HseChecklists, ...HseChecklists41_42];
    
    // Process HSE101_024 checklists to match the expected format
    const processedHse101_024 = HseChecklists101_024.map((item, index) => {
      // Handle both possible structures in the JSON
      const checklistData = item.Checklist_024_101 || item.document;
      if (checklistData) {
        return {
          document_name: checklistData.document_name || checklistData.title,
          document_id: checklistData.document_id,
          version: checklistData.version,
          version_date: checklistData.date,
          approved_by: checklistData.approved_by,
          title: checklistData.title,
          // Add sections based on the structure
          checklist: {
            'SAFETY_FEATURES': item.safety_features || [],
            'PPE_REQUIREMENTS': item.personal_protective_equipment || {},
            'NO_GO_AREAS': item.no_go_areas || {},
            'INDUCTION_CATEGORIES': item.induction_categories || {},
            'VISITOR_SIGN_OFF': item.visitor_sign_off || {},
            'WORKSHOP_RULES': item.workshop_area_rules || {}
          }
        };
      }
      return null;
    }).filter(Boolean); // Remove null items
    
    // Process HSE-020 checklist to match the expected format
    const processedHse020 = {
      document_name: HseChecklists020.document.document_name,
      document_id: HseChecklists020.document.document_id,
      version: HseChecklists020.document.version,
      version_date: HseChecklists020.document.version_date,
      approved_by: HseChecklists020.document.approved_by,
      title: HseChecklists020.document.title,
      checklist: {
        'OBSERVATION_DETAILS': HseChecklists020.observation_details || {},
        'REASON_FOR_PTO_VFL': HseChecklists020.reason_for_pto_vfl || {},
        'EVALUATION_CRITERIA': HseChecklists020.evaluation_criteria || {},
        'COMMENDATION_SECTION': HseChecklists020.commendation_section || {},
        'CORRECTION_SECTION': HseChecklists020.correction_section || {},
        'ACTION_ITEMS': HseChecklists020.action_items || {},
        'SIGNATURES': HseChecklists020.signatures || {}
      }
    };

    // Process HSE-071 checklist to match the expected format
    const processedHse071 = {
      document_name: HseChecklists071.document.name,
      document_id: HseChecklists071.document.id,
      version: HseChecklists071.document.version,
      version_date: HseChecklists071.document.versionDate,
      approved_by: HseChecklists071.document.approvedBy,
      title: HseChecklists071.report.title,
      checklist: {
        'FIELDS': HseChecklists071.report.fields || {},
        'INCOMING_SHIFT_REQUIREMENTS': HseChecklists071.report.incomingShiftRequirements || [],
        'SHIFT_CONCERN': HseChecklists071.report.shiftConcern || [],
        'EQUIPMENT_CHECKLIST': HseChecklists071.report.equipmentChecklist || [],
        'STANDARD_PROCEDURES': HseChecklists071.report.standardProcedures || []
      }
    };

    // Process HSE-156 checklist to match the expected format
    const processedHse156 = {
      document_name: HSE_156Json.safetyHarnessChecklist.metadata.documentName,
      document_id: HSE_156Json.safetyHarnessChecklist.metadata.documentId,
      version: HSE_156Json.safetyHarnessChecklist.metadata.version,
      version_date: HSE_156Json.safetyHarnessChecklist.metadata.versionDate,
      approved_by: HSE_156Json.safetyHarnessChecklist.metadata.approvedBy,
      title: HSE_156Json.safetyHarnessChecklist.title,
      checklist: {
        'INSPECTION_REGISTER': HSE_156Json.safetyHarnessChecklist.inspectionRegister || [],
        'DEVIATIONS': HSE_156Json.safetyHarnessChecklist.deviations || [],
        'CHECK_ITEMS': HSE_156Json.safetyHarnessChecklist.inspectionRegister?.[0]?.checkItems || {}
      }
    };

    // Process HSE-077_078 checklist to match the expected format
    const processedHse077_078 = {
      document_name: HseChecklists077_078.document.document_name,
      document_id: HseChecklists077_078.document.document_id,
      version: HseChecklists077_078.document.version,
      version_date: HseChecklists077_078.document.version_date,
      approved_by: HseChecklists077_078.document.approved_by,
      title: HseChecklists077_078.form.form_type,
      checklist: {
        'FORM_FIELDS': HseChecklists077_078.form || {},
        'INSTRUCTIONS': HseChecklists077_078.instructions || {}
      }
    };

    // Process HSE-129 checklist to match the expected format
    const processedHse129 = {
      document_name: HseChecklists129.document.document_name,
      document_id: HseChecklists129.document.document_id,
      version: HseChecklists129.document.version,
      version_date: HseChecklists129.document.version_date,
      approved_by: HseChecklists129.document.approved_by,
      title: HseChecklists129.form.form_type,
      checklist: {
        'FORM_FIELDS': HseChecklists129.form || {},
        'DRILL_INSTRUCTION': HseChecklists129.form || {}
      }
    };

    // Process HSE-072 checklist to match the expected format
    const processedHse072 = {
      document_name: HseChecklists072.document.name,
      document_id: HseChecklists072.document.id,
      version: HseChecklists072.document.version,
      version_date: HseChecklists072.document.versionDate,
      approved_by: HseChecklists072.document.approvedBy,
      title: HseChecklists072.report.title,
      checklist: {
        'FIELDS': HseChecklists072.report.fields || {},
        'INCOMING_SHIFT_REQUIREMENTS': HseChecklists072.report.incomingShiftRequirements || [],
        'SHIFT_CONCERN': HseChecklists072.report.shiftConcern || [],
        'EQUIPMENT_CHECKLIST': HseChecklists072.report.equipmentChecklist || [],
        'STANDARD_PROCEDURES': HseChecklists072.report.standardProcedures || []
      }
    };
    
    setChecklists([...allChecklists, ...processedHse101_024, processedHse020, processedHse071, processedHse156, processedHse077_078, processedHse129, processedHse072]);
  }, []);

  const handleChecklistSelect = async (checklist) => {
    try {
      // Save the selected checklist to AsyncStorage
      const savedHse = await AsyncStorage.getItem('hseStep1');
      let hseData = savedHse ? JSON.parse(savedHse) : {};
      hseData.selectedChecklist = checklist;
      await AsyncStorage.setItem('hseStep1', JSON.stringify(hseData));
      
      // Go back to previous screen (step 1)
      navigation.goBack();
    } catch (error) {
      console.error('Error saving selected checklist:', error);
      // Still go back even if saving fails
      navigation.goBack();
    }
  };

  const handleBackPress = () => {
    navigation.goBack();
  };

  const renderChecklistItem = ({ item }) => (
    <TouchableOpacity 
      style={styles.checklistItem} 
      onPress={() => handleChecklistSelect(item)}
      activeOpacity={0.7}
    >
      <View style={styles.checklistHeader}>
        <View style={styles.checklistTitleContainer}>
          <Text style={styles.checklistTitle}>{item.document_name}</Text>
          <View style={styles.checklistBadge}>
            <Text style={styles.checklistBadgeText}>{item.document_id || 'N/A'}</Text>
          </View>
        </View>
        <MaterialIcons name="chevron-right" size={24} color="#0078D4" />
      </View>
      
      <View style={styles.checklistDetails}>
        <View style={styles.detailRow}>
          <Text style={styles.detailLabel}>Document ID:</Text>
          <Text style={styles.detailValue}>{item.document_id || 'N/A'}</Text>
        </View>
        <View style={styles.detailRow}>
          <Text style={styles.detailLabel}>Version:</Text>
          <Text style={styles.detailValue}>{item.version || 'N/A'}</Text>
        </View>
        <View style={styles.detailRow}>
          <Text style={styles.detailLabel}>Date:</Text>
          <Text style={styles.detailValue}>{item.version_date || 'N/A'}</Text>
        </View>
        <View style={styles.detailRow}>
          <Text style={styles.detailLabel}>Approved by:</Text>
          <Text style={styles.detailValue}>{item.approved_by || 'N/A'}</Text>
        </View>
      </View>

      <View style={styles.checklistSections}>
        <Text style={styles.sectionsTitle}>Sections:</Text>
        <View style={styles.sectionsContainer}>
          {item.checklist && Object.keys(item.checklist).map((sectionKey, index) => {
            if (sectionKey !== 'TOTAL_OBSERVATIONS' && sectionKey !== 'BEHAVIOURAL' && sectionKey !== 'OPERATIONAL' && sectionKey !== 'ACTION_TO_BE_TAKEN') {
              return (
                <View key={index} style={styles.sectionChip}>
                  <Text style={styles.sectionChipText}>{sectionKey}</Text>
                </View>
              );
            }
            return null;
          })}
        </View>
      </View>
    </TouchableOpacity>
  );

  return (
    <View style={styles.container}>
      {/* Header */}
      <View style={styles.header}>
        <TouchableOpacity onPress={handleBackPress} style={styles.backButton}>
          <MaterialIcons name="arrow-back" size={24} color="#015185" />
        </TouchableOpacity>
        <Text style={styles.headerTitle}>Select HSE Checklist</Text>
        <View style={styles.placeholder} />
      </View>

      {/* Content */}
      <View style={styles.content}>
        <View style={styles.infoSection}>
          <MaterialIcons name="info" size={20} color="#015185" />
          <Text style={styles.infoText}>
            Select a checklist to proceed with your HSE inspection
          </Text>
        </View>

        <View style={styles.statsSection}>
          <Text style={styles.statsText}>
            Available Checklists: {checklists.length}
          </Text>
        </View>

        <FlatList
          data={checklists}
          renderItem={renderChecklistItem}
          keyExtractor={(item, index) => index.toString()}
          showsVerticalScrollIndicator={false}
          contentContainerStyle={styles.listContainer}
          ItemSeparatorComponent={() => <View style={styles.separator} />}
        />
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f8f9fa',
  },
  header: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: 20,
    paddingVertical: 15,
    backgroundColor: '#fff',
    borderBottomWidth: 1,
    borderBottomColor: '#e0e0e0',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  backButton: {
    padding: 5,
  },
  headerTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
  },
  placeholder: {
    width: 34,
  },
  content: {
    flex: 1,
    paddingHorizontal: 20,
    paddingTop: 20,
  },
  infoSection: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#e3f2fd',
    padding: 15,
    borderRadius: 8,
    marginBottom: 15,
  },
  infoText: {
    marginLeft: 10,
    fontSize: 14,
    color: '#015185',
    flex: 1,
  },
  statsSection: {
    backgroundColor: '#fff',
    padding: 12,
    borderRadius: 8,
    marginBottom: 20,
    borderLeftWidth: 4,
    borderLeftColor: '#0078D4',
  },
  statsText: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
  },
  listContainer: {
    paddingBottom: 20,
  },
  checklistItem: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    marginBottom: 15,
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  checklistHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'flex-start',
    marginBottom: 15,
  },
  checklistTitleContainer: {
    flex: 1,
    marginRight: 10,
  },
  checklistTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginBottom: 8,
    lineHeight: 22,
  },
  checklistBadge: {
    backgroundColor: '#0078D4',
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
    alignSelf: 'flex-start',
  },
  checklistBadgeText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
  },
  checklistDetails: {
    marginBottom: 15,
  },
  detailRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 5,
  },
  detailLabel: {
    fontSize: 14,
    color: '#666',
    fontWeight: '500',
  },
  detailValue: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
  },
  checklistSections: {
    borderTopWidth: 1,
    borderTopColor: '#f0f0f0',
    paddingTop: 15,
  },
  sectionsTitle: {
    fontSize: 14,
    fontWeight: '600',
    color: '#333',
    marginBottom: 10,
  },
  sectionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  sectionChip: {
    backgroundColor: '#f0f0f0',
    paddingHorizontal: 10,
    paddingVertical: 5,
    borderRadius: 15,
    marginRight: 8,
    marginBottom: 8,
  },
  sectionChipText: {
    fontSize: 12,
    color: '#666',
    fontWeight: '500',
  },
  separator: {
    height: 1,
    backgroundColor: '#f0f0f0',
    marginVertical: 10,
  },
});

export default HseSelectChecklsit;