
import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';

const { width, height } = Dimensions.get('window');

const HSE_129_Step2 = ({ setActiveStep }) => {
  const [formData, setFormData] = useState({
    date: '',
    client: '',
    project: '',
    clientRepresentative: '',
    clientContactNo: '',
    titanRepresentative: '',
    titanContactNo: '',
    holeId: '',
    dip: '',
    azimuth: '',
    plannedDepth: '',
    coreOrientationRequired: '',
    surveyIntervals: '',
    eohMultiShotSurveyRequired: '',
  });

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse129Step2');
      if (savedData) {
        setFormData(JSON.parse(savedData));
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async (data) => {
    try {
      await AsyncStorage.setItem('hse129Step2', JSON.stringify(data));
    } catch (error) {
      console.error('Error saving data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data',
        position: 'top',
      });
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    saveData(updatedData);
  };

  const handleNext = () => {
    // Validate required fields
    const requiredFields = ['date', 'client', 'project', 'holeId'];
    const missingFields = requiredFields.filter(field => !formData[field]);
    
    if (missingFields.length > 0) {
      Toast.show({
        type: 'error',
        text1: 'Required Fields Missing',
        text2: `Please fill: ${missingFields.join(', ')}`,
        position: 'top',
      });
      return;
    }
    
    setActiveStep(3);
  };

  const handleBack = () => {
    setActiveStep(1);
  };

  const renderInputField = (label, field, placeholder, required = false, keyboardType = 'default') => (
    <View style={styles.inputContainer}>
      <Text style={styles.label}>
        {label}
        {required && <Text style={styles.required}> *</Text>}
      </Text>
      <TextInput
        style={styles.input}
        value={formData[field]}
        onChangeText={(value) => handleInputChange(field, value)}
        placeholder={placeholder}
        keyboardType={keyboardType}
        placeholderTextColor="#999"
      />
    </View>
  );

  const renderRadioGroup = (label, field, options) => (
    <View style={styles.inputContainer}>
      <Text style={styles.label}>{label}</Text>
      <View style={styles.radioGroup}>
        {options.map((option) => (
          <TouchableOpacity
            key={option}
            style={styles.radioOption}
            onPress={() => handleInputChange(field, option)}
          >
            <View style={[
              styles.radioButton,
              formData[field] === option && styles.radioButtonSelected
            ]}>
              {formData[field] === option && (
                <View style={styles.radioButtonInner} />
              )}
            </View>
            <Text style={styles.radioLabel}>{option}</Text>
          </TouchableOpacity>
        ))}
      </View>
    </View>
  );

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.scrollView} contentContainerStyle={styles.scrollContent}>
        {/* Header */}
        <View style={styles.header}>
          <Text style={styles.headerTitle}>DRILL INSTRUCTION SHEET</Text>
          
        </View>

        {/* Form Content */}
        <View style={styles.formContainer}>
          {/* Single Field Rows */}
          {renderInputField('Date', 'date', 'Enter date', true)}
          {renderInputField('Client', 'client', 'Enter client name', true)}
          {renderInputField('Project', 'project', 'Enter project name', true)}
          {renderInputField('Client Representative', 'clientRepresentative', 'Enter client representative')}
          {renderInputField('Titan Representative', 'titanRepresentative', 'Enter Titan representative')}
          {renderInputField('Hole ID', 'holeId', 'Enter hole ID', true)}
          {renderInputField('Client Contact', 'clientContactNo', 'Enter client contact', false, 'phone-pad')}
          {renderInputField('Titan Contact', 'titanContactNo', 'Enter Titan contact', false, 'phone-pad')}
          {renderInputField('DIP', 'dip', 'Enter DIP value', false, 'numeric')}
          {renderInputField('Azimuth', 'azimuth', 'Enter azimuth', false, 'numeric')}
          {renderInputField('Planned Depth', 'plannedDepth', 'Enter planned depth', false, 'numeric')}
          {renderInputField('Survey Intervals', 'surveyIntervals', 'Enter survey intervals', false, 'numeric')}

          {/* Radio Groups */}
          <View style={styles.radioContainer}>
            {renderRadioGroup('Core Orientation Required', 'coreOrientationRequired', ['Yes', 'No'])}
            {renderRadioGroup('EOH Multi-Shot Survey Required', 'eohMultiShotSurveyRequired', ['Yes', 'No'])}
          </View>
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNext}
        >
          <Text style={styles.nextButtonText}>Next</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  header: {
    backgroundColor: '#f8f9fa',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 15,
  },
  headerInfo: {
    alignSelf: 'flex-end',
    backgroundColor: '#fff',
    padding: 10,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#dee2e6',
  },
  headerRow: {
    flexDirection: 'row',
    marginBottom: 5,
  },
  headerLabel: {
    fontSize: 12,
    color: '#666',
    marginRight: 10,
    fontWeight: '500',
  },
  headerValue: {
    fontSize: 12,
    color: '#333',
    fontWeight: '600',
  },
  formContainer: {
    padding: 20,
  },
  radioContainer: {
    marginTop: 10,
  },
  inputContainer: {
    marginBottom: 20,
  },
  label: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  required: {
    color: '#dc3545',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
  },

  radioGroup: {
    flexDirection: 'row',
    marginTop: 8,
  },
  radioOption: {
    flexDirection: 'row',
    alignItems: 'center',
    marginRight: 20,
  },
  radioButton: {
    width: 20,
    height: 20,
    borderRadius: 10,
    borderWidth: 2,
    borderColor: '#015185',
    marginRight: 8,
    justifyContent: 'center',
    alignItems: 'center',
  },
  radioButtonSelected: {
    backgroundColor: '#015185',
  },
  radioButtonInner: {
    width: 8,
    height: 8,
    borderRadius: 4,
    backgroundColor: '#fff',
  },
  radioLabel: {
    fontSize: 16,
    color: '#333',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
});

export default HSE_129_Step2;