import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  TextInput,
  Modal,
  Dimensions,
  Alert,
  Image,
  Platform,
  PermissionsAndroid,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import HSE_155_JSON from './HSE_155_json.js';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';

const { width } = Dimensions.get('window');
const { height } = Dimensions.get('window');

const HSE_155_Step2 = ({ setActiveStep }) => {
  const [sections, setSections] = useState([]);
  const [expandedSections, setExpandedSections] = useState({});
  const [selections, setSelections] = useState({});
  const [notes, setNotes] = useState({});
  const [noteModal, setNoteModal] = useState({
    visible: false,
    itemKey: '',
    currentNote: '',
  });
  const [images, setImages] = useState({});
  const [cameraModal, setCameraModal] = useState({
    visible: false,
    itemKey: '',
  });

  useEffect(() => {
    loadSavedData();
    setupChecklistData();
  }, []);

  const setupChecklistData = () => {
    // Use the imported JSON data
    const checklistData = HSE_155_JSON.sections || [];
    setSections(checklistData);
    
    // Initialize expanded sections
    const initialExpanded = {};
    checklistData.forEach((section) => {
      initialExpanded[section.key] = false;
    });
    setExpandedSections(initialExpanded);
  };

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse155Step2');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setSelections(parsedData.selections || {});
        setNotes(parsedData.notes || {});
        setImages(parsedData.images || {});
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const requestCameraPermission = async () => {
    if (Platform.OS === 'android') {
      try {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.CAMERA,
          {
            title: 'Camera Permission',
            message: 'This app needs access to your camera to take photos.',
            buttonNeutral: 'Ask Me Later',
            buttonNegative: 'Cancel',
            buttonPositive: 'OK',
          },
        );
        return granted === PermissionsAndroid.RESULTS.GRANTED;
      } catch (err) {
        console.warn(err);
        return false;
      }
    }
    return true;
  };

  const handleCameraAction = async (action) => {
    const { itemKey } = cameraModal;
    
    if (action === 'capture') {
      const hasPermission = await requestCameraPermission();
      if (!hasPermission) {
        Alert.alert('Permission Denied', 'Camera permission is required to take photos.');
        return;
      }
      
      try {
        const result = await launchCamera({
          mediaType: 'photo',
          quality: 0.8,
          includeBase64: false,
        });
        
        if (result.assets && result.assets[0]) {
          const imageUri = result.assets[0].uri;
          setImages(prev => ({
            ...prev,
            [itemKey]: imageUri
          }));
          saveData();
        }
      } catch (error) {
        console.error('Camera error:', error);
        Alert.alert('Error', 'Failed to capture image.');
      }
    } else if (action === 'upload') {
      try {
        const result = await launchImageLibrary({
          mediaType: 'photo',
          quality: 0.8,
          includeBase64: false,
        });
        
        if (result.assets && result.assets[0]) {
          const imageUri = result.assets[0].uri;
          setImages(prev => ({
            ...prev,
            [itemKey]: imageUri
          }));
          saveData();
        }
      } catch (error) {
        console.error('Image picker error:', error);
        Alert.alert('Error', 'Failed to select image.');
      }
    }
    
    setCameraModal({ visible: false, itemKey: '' });
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        selections,
        notes,
        images,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse155Step2', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const getOptionsForInputType = (inputType) => {
    switch (inputType) {
      case 'Yes/No':
        return ['YES/OK', 'NO/DEF'];
      default:
        return ['YES/OK', 'NO/DEF'];
    }
  };

  const handleSelectOption = async (itemKey, option) => {
    setSelections((prev) => {
      const updatedSelections = {
        ...prev,
        [itemKey]: option,
      };
      saveData();
      return updatedSelections;
    });
  };

  const openNoteModal = (itemKey) => {
    setNoteModal({
      visible: true,
      itemKey,
      currentNote: notes[itemKey] || '',
    });
  };

  const saveNote = () => {
    const { itemKey, currentNote } = noteModal;
    if (currentNote.trim()) {
      setNotes(prev => ({
        ...prev,
        [itemKey]: currentNote.trim()
      }));
      saveData();
    }
    setNoteModal({ visible: false, itemKey: '', currentNote: '' });
  };

  const toggleSection = (sectionKey) => {
    setExpandedSections((prev) => ({
      ...prev,
      [sectionKey]: !prev[sectionKey],
    }));
  };

  const renderItem = (item, index) => {
    const selectedOption = selections[item.key];
    const itemNote = notes[item.key] || '';
    const options = getOptionsForInputType(item.inputType);

    return (
      <View key={item.key} style={styles.centeredItem}>
        <View style={styles.itemContainer}>
          <Text style={styles.itemText}>
            {index + 1}. {item.itemName}
          </Text>
          <View style={styles.itemRow}>
            <View style={styles.optionsContainer}>
              {options.map((option) => {
                const isSelected = selectedOption === option;
                return (
                  <TouchableOpacity
                    key={option}
                    style={[
                      styles.optionButton,
                      isSelected && styles.selectedOptionButton,
                      isSelected && option === 'YES/OK' && styles.okButton,
                      isSelected && option === 'NO/DEF' && styles.rejectButton,
                    ]}
                    onPress={() => handleSelectOption(item.key, option)}
                  >
                    <Text
                      style={[
                        styles.optionText,
                        isSelected && styles.optionTextSelected,
                      ]}
                    >
                      {option}
                    </Text>
                  </TouchableOpacity>
                );
              })}
            </View>
            <View style={styles.iconsContainer}>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => openNoteModal(item.key)}
              >
                <MaterialIcons name="note-add" size={24} color="#333" />
              </TouchableOpacity>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => setCameraModal({ visible: true, itemKey: item.key })}
              >
                <MaterialIcons name="camera-alt" size={24} color="#333" />
              </TouchableOpacity>
            </View>
          </View>

          {itemNote && (
            <Text style={styles.noteText}>
              Note: {itemNote}
            </Text>
          )}
          
          {/* Image Display */}
          {images[item.key] && (
            <View style={styles.imageContainer}>
              <Image source={{ uri: images[item.key] }} style={styles.itemImage} />
              <TouchableOpacity
                style={styles.removeImageButton}
                onPress={() => {
                  setImages(prev => {
                    const newImages = { ...prev };
                    delete newImages[item.key];
                    return newImages;
                  });
                  saveData();
                }}
              >
                <MaterialIcons name="close" size={16} color="#fff" />
              </TouchableOpacity>
            </View>
          )}
        </View>
        <View style={styles.itemBorder} />
      </View>
    );
  };

  const validateSelections = () => {
    const allItems = sections.flatMap((section) => section.subItems);
    if (allItems.length === 0) {
      Alert.alert('No Items', 'The selected checklist has no valid items.');
      return false;
    }

    // Check if at least one item has been selected (minimum requirement)
    const hasAtLeastOneSelection = allItems.some((item) => selections[item.key] !== null && selections[item.key] !== '');
    if (!hasAtLeastOneSelection) {
      Alert.alert('Minimum Requirement', 'Please make at least one selection before proceeding.');
      return false;
    }

    return true;
  };

  const handleNext = () => {
    if (validateSelections()) {
      setActiveStep(3);
    }
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.checklistTitle}>Workshop Vehicle Pre-Release Checklist</Text>

        {sections.length > 0 ? (
          sections.map((section) => {
            // Calculate selected items count for this section
            const sectionItems = section.subItems || [];
            const selectedItemsCount = sectionItems.filter(item => selections[item.key]).length;
            const totalItemsCount = sectionItems.length;
            
            return (
              <View key={section.key}>
                <TouchableOpacity
                  style={styles.sectionHeader}
                  onPress={() => toggleSection(section.key)}
                >
                  <View style={styles.sectionHeaderContent}>
                    <MaterialIcons
                      name={expandedSections[section.key] ? 'expand-less' : 'expand-more'}
                      size={20}
                      color={expandedSections[section.key] ? '#015185' : '#333'}
                    />
                    <View style={styles.sectionHeaderTextContainer}>
                      <Text style={styles.sectionHeaderText}>{section.name}</Text>
                      <Text style={styles.sectionItemCount}>
                        ({selectedItemsCount}/{totalItemsCount} selected)
                      </Text>
                    </View>
                  </View>
                </TouchableOpacity>
                {expandedSections[section.key] && (
                  <View style={styles.sectionContent}>
                    {section.subItems.length > 0 ? (
                      section.subItems.map((item, index) => renderItem(item, index))
                    ) : (
                      <Text style={styles.emptySectionText}>No inspection items</Text>
                    )}
                  </View>
                )}
              </View>
            );
          })
        ) : (
          <Text style={styles.emptySectionText}>No sections available</Text>
        )}
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(1)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Note Modal */}
      <Modal
        visible={noteModal.visible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setNoteModal({ visible: false, itemKey: '', currentNote: '' })}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Note</Text>
            <TextInput
              style={styles.modalTextInput}
              value={noteModal.currentNote}
              onChangeText={(value) => setNoteModal(prev => ({ ...prev, currentNote: value }))}
              placeholder="Enter your note here..."
              multiline
              numberOfLines={4}
            />
            <View style={styles.modalButtons}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={() => setNoteModal({ visible: false, itemKey: '', currentNote: '' })}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.modalButtonPrimary]} onPress={saveNote}>
                <Text style={[styles.modalButtonText, styles.modalButtonTextPrimary]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Camera Modal */}
      <Modal
        visible={cameraModal.visible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setCameraModal({ visible: false, itemKey: '' })}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Image Source</Text>
            <View style={styles.cameraOptionsContainer}>
              <TouchableOpacity
                style={[styles.cameraOptionButton, styles.cameraOptionButtonPrimary]}
                onPress={() => handleCameraAction('capture')}
              >
                <MaterialIcons name="camera-alt" size={32} color="#fff" />
                <Text style={styles.cameraOptionText}>Capture Photo</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.cameraOptionButton, styles.cameraOptionButtonSecondary]}
                onPress={() => handleCameraAction('upload')}
              >
                <MaterialIcons name="photo-library" size={32} color="#fff" />
                <Text style={styles.cameraOptionText}>Upload Image</Text>
              </TouchableOpacity>
            </View>
            <TouchableOpacity
              style={styles.cancelCameraButton}
              onPress={() => setCameraModal({ visible: false, itemKey: '' })}
            >
              <Text style={styles.cancelCameraButtonText}>Cancel</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  checklistTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionHeaderTextContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  sectionItemCount: {
    fontSize: 14,
    color: '#666',
    marginLeft: 5,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  rejectButton: {
    backgroundColor: '#FF4B2B',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  iconsContainer: {
    flexDirection: 'row',
  },
  iconButton: {
    padding: 10,
  },
  noteText: {
    fontSize: 14,
    color: '#333',
    marginTop: 8,
    marginBottom: 8,
    fontStyle: 'italic',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  emptySectionText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    padding: 20,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 20,
    width: width * 0.9,
    maxHeight: height * 0.6,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 100,
    textAlignVertical: 'top',
    marginBottom: 20,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    marginHorizontal: 5,
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#ddd',
    backgroundColor: '#fff',
  },
  modalButtonPrimary: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  modalButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  modalButtonTextPrimary: {
    color: '#fff',
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  imageContainer: {
    marginTop: 10,
    position: 'relative',
    alignItems: 'center',
  },
  itemImage: {
    width: 120,
    height: 120,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  removeImageButton: {
    position: 'absolute',
    top: -8,
    right: -8,
    backgroundColor: '#FF4B2B',
    borderRadius: 12,
    width: 24,
    height: 24,
    justifyContent: 'center',
    alignItems: 'center',
  },
  cameraOptionsContainer: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    marginVertical: 20,
  },
  cameraOptionButton: {
    alignItems: 'center',
    padding: 20,
    borderRadius: 12,
    minWidth: 120,
  },
  cameraOptionButtonPrimary: {
    backgroundColor: '#015185',
  },
  cameraOptionButtonSecondary: {
    backgroundColor: '#4CAF50',
  },
  cameraOptionText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
    marginTop: 8,
  },
  cancelCameraButton: {
    backgroundColor: '#f2f2f2',
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    alignItems: 'center',
    marginTop: 10,
  },
  cancelCameraButtonText: {
    color: '#333',
    fontSize: 16,
    fontWeight: '600',
  },
});

export default HSE_155_Step2;