import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
  Alert,
  Modal,
  Image,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import SignatureCanvas from 'react-native-signature-canvas';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import RNFS from 'react-native-fs';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import { useNavigation } from '@react-navigation/native';

const { width, height } = Dimensions.get('window');

// SyncModal Component for Animated Loading
const SyncModal = ({ visible, message = 'Processing' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_129_Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    // Left column assessments
    drillPadAccessSafe: '',
    drillPadLevel: '',
    adequateSpaceAuxiliary: '',
    // Right column assessments
    drillPadSize: '',
    anyHazardsRisks: '',
    adequateSpaceVisitors: '',
    // Comments
    specialInstructions: '',
    // Client Representative
    clientRepresentativeName: '',
    clientSignature: '',
    // Titan Drilling Representative
    titanRepresentativeName: '',
    titanSignature: '',
  });

  const [isLoading, setIsLoading] = useState(false);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [reportId, setReportId] = useState(null);
  const [signatureModalVisible, setSignatureModalVisible] = useState(false);
  const [currentSignatureType, setCurrentSignatureType] = useState(null); // 'client' or 'titan'
  const [step1Data, setStep1Data] = useState(null);
  const [step2Data, setStep2Data] = useState(null);
  const isMounted = useRef(true);
  const signatureRef = useRef(null);

  useEffect(() => {
    loadData();
    // Check current counter value when component loads
    checkCurrentCounter();
    isMounted.current = true;
    return () => {
      isMounted.current = false;
    };
  }, []);

  const loadData = async () => {
    try {
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hse129Step2');
      const savedStep3 = await AsyncStorage.getItem('hse129Step3');
      
      if (savedStep1) {
        setStep1Data(JSON.parse(savedStep1));
      }
      if (savedStep2) {
        setStep2Data(JSON.parse(savedStep2));
      }
      if (savedStep3) {
        setFormData(JSON.parse(savedStep3));
      }
    } catch (error) {
      console.error('Error loading data:', error);
    }
  };

  const saveData = async (data) => {
    try {
      await AsyncStorage.setItem('hse129Step3', JSON.stringify(data));
    } catch (error) {
      console.error('Error saving data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data',
        position: 'top',
      });
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    saveData(updatedData);
  };

  const openSignatureModal = (type) => {
    setCurrentSignatureType(type);
    setSignatureModalVisible(true);
  };

  const closeSignatureModal = () => {
    setSignatureModalVisible(false);
    setCurrentSignatureType(null);
  };

  const handleSignatureComplete = (signature) => {
    if (signature && currentSignatureType) {
      const fieldName = currentSignatureType === 'client' ? 'clientSignature' : 'titanSignature';
      setFormData(prev => ({
        ...prev,
        [fieldName]: signature
      }));
      saveData({
        ...formData,
        [fieldName]: signature
      });
    }
    setSignatureModalVisible(false);
    setCurrentSignatureType(null);
  };

  const handleSignatureClear = () => {
    if (signatureRef.current) {
      signatureRef.current.clearSignature();
    }
  };

  const handleSignatureConfirm = () => {
    if (signatureRef.current) {
      signatureRef.current.readSignature();
    }
  };

  const handleSignatureData = (signature) => {
    console.log('Signature captured:', signature ? signature.substring(0, 50) + '...' : signature);
    handleSignatureComplete(signature);
  };

  // Firebase helper functions
  const getAdminUserId = async () => {
    const userEmail = await AsyncStorage.getItem('userEmail');
    if (!userEmail) throw new Error('User email not found. Please log in again.');
    const userSnapshot = await database()
      .ref('/Globalusers/admins')
      .orderByChild('email')
      .equalTo(userEmail)
      .once('value');
    const userData = userSnapshot.val();
    if (!userData) throw new Error('User not found in admin database.');
    return Object.keys(userData)[0];
  };

  const getCurrentReportNumber = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      return (currentNumber + 1).toString().padStart(3, '0');
    } catch (error) {
      console.error('Error getting current report number:', error);
      return '001';
    }
  };

  const incrementReportCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      let currentNumber = snapshot.val();
      
      // Initialize counter if it doesn't exist
      if (currentNumber === null || currentNumber === undefined) {
        currentNumber = 0;
      }
      
      currentNumber++;
      
      // Update the counter in Firebase
      await counterRef.set(currentNumber);
      
      return currentNumber.toString().padStart(3, '0');
    } catch (error) {
      console.error('Error incrementing report counter:', error);
      return '001';
    }
  };

  const checkCurrentCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      console.log('Current counter value in Firebase:', currentNumber);
      Toast.show({
        type: 'info',
        text1: 'Counter Info',
        text2: `Current counter value: ${currentNumber}`,
        position: 'top',
      });
      return currentNumber;
    } catch (error) {
      console.error('Error checking current counter:', error);
      return 0;
    }
  };

  const uploadFileToFirebaseStorage = async (filePath, storagePath) => {
    try {
      const reference = storage().ref(storagePath);
      await reference.putFile(filePath);
      const downloadUrl = await reference.getDownloadURL();
      return downloadUrl;
    } catch (error) {
      console.error('Error uploading file to Firebase Storage:', error);
      throw error;
    }
  };

  const flushStepData = async () => {
    try {
      await AsyncStorage.removeItem('hse129Step1');
      await AsyncStorage.removeItem('hse129Step2');
      await AsyncStorage.removeItem('hse129Step3');
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const validateForm = () => {
    if (!formData.drillPadAccessSafe) {
      Alert.alert('Missing Information', 'Please complete all required assessments.');
      return false;
    }
    
    if (!formData.drillPadLevel) {
      Alert.alert('Missing Information', 'Please complete all required assessments.');
      return false;
    }
    
    if (!formData.drillPadSize) {
      Alert.alert('Missing Information', 'Please complete all required assessments.');
      return false;
    }
    
    if (!formData.clientRepresentativeName.trim()) {
      Alert.alert('Missing Information', 'Please enter the client representative name.');
      return false;
    }
    
    if (!formData.clientSignature) {
      Alert.alert('Missing Information', 'Please provide the client representative signature.');
      return false;
    }
    
    if (!formData.titanRepresentativeName.trim()) {
      Alert.alert('Missing Information', 'Please enter the Titan drilling representative name.');
      return false;
    }
    
    if (!formData.titanSignature) {
      Alert.alert('Missing Information', 'Please provide the Titan drilling representative signature.');
      return false;
    }
    
    return true;
  };

  const handleSubmit = async () => {
    if (!validateForm()) {
      return;
    }

    setIsLoading(true);

    try {
      const userId = await getAdminUserId();
      const savedStep1 = await AsyncStorage.getItem('hseStep1');
      const savedStep2 = await AsyncStorage.getItem('hse129Step2');
      
      if (!savedStep2) {
        throw new Error('Step 2 data is missing. Please complete Step 2 first.');
      }

      const step1Data = savedStep1 ? JSON.parse(savedStep1) : {};
      const step2Data = JSON.parse(savedStep2);

      // Use the request number from step 1 data
      const newReportId = step1Data.requestNumber || `HSE-129-001`;
      setReportId(newReportId);

      const reportData = {
        id: newReportId,
        userId,
        createdAt: new Date().toISOString(),
        status: 'completed',
        checklist: step1Data.selectedChecklist || {},
        step1: step1Data,
        step2: step2Data,
        step3: formData,
        timestamp: new Date().toISOString(),
      };

      // Save to Firebase using the same path as HSE_077_078
      const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${newReportId}`);
      await dbRef.set(reportData);

      // Generate PDF
      const pdfPath = await generatePDFWithHtml(reportData);
      const storagePath = `GlobalHSEInspections/admins/${userId}/reports/${newReportId}.pdf`;
      const pdfDownloadUrl = await uploadFileToFirebaseStorage(pdfPath, storagePath);

      // Update report with PDF URL
      await dbRef.update({
        pdfDownloadUrl,
        status: 'completed',
      });

      await RNFS.unlink(pdfPath).catch((err) => console.warn('Error deleting temp PDF:', err));

      // Increment the global report counter only after successful report generation
      await incrementReportCounter();

      await flushStepData();

      if (isMounted.current) {
        setShowSuccessModal(true);
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: 'Site Assessment saved successfully.',
          position: 'top',
        });
      }
    } catch (error) {
      console.error('Error in handleSubmit:', error);
      if (isMounted.current) {
        setIsLoading(false);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: error.message || 'Failed to save report.',
          position: 'top',
        });
        Alert.alert('Save Error', `Failed to save report.\n\nError: ${error.message}`, [{ text: 'OK' }], { cancelable: false });
      }
    } finally {
      if (isMounted.current) {
        setIsLoading(false);
      }
    }
  };

  const generatePDFWithHtml = async (report) => {
    try {
      const step1 = report.step1 || {};
      const step2 = report.step2 || {};
      const step3 = report.step3 || {};
      const checklist = report.checklist || {};

      const html = `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Drill Instruction Sheet</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 20px;
              padding: 0;
              font-size: 12px;
              line-height: 1.4;
            }
            .header-row {
              display: flex;
              justify-content: space-between;
              align-items: flex-start;
              margin-bottom: 20px;
            }
            .logo-left {
              display: flex;
              align-items: center;
            }
            .logo-left img {
              width: 140px;
              height: 50px;
              object-fit: contain;
            }
            .company-name {
              font-size: 18px;
              font-weight: bold;
              color: #015185;
              margin-left: 10px;
            }
            .doc-info-box {
              width: 340px;
            }
            .doc-info-box table {
              width: 100%;
              border: 1px solid #222;
              border-collapse: collapse;
              font-size: 11px;
            }
            .doc-info-box td {
              border: 1px solid #222;
              padding: 4px 6px;
              text-align: left;
            }
            .title-section {
              text-align: center;
              margin: 20px 0;
            }
            .title-section h2 {
              margin: 0;
              font-size: 18px;
              font-weight: bold;
              text-decoration: underline;
              text-transform: uppercase;
            }
            .form-section {
              margin: 15px 0;
            }
            .form-row {
              display: flex;
              margin-bottom: 10px;
              align-items: flex-start;
            }
            .form-label {
              font-weight: bold;
              min-width: 200px;
              margin-right: 10px;
            }
            .form-value {
              flex: 1;
              border-bottom: 1px solid #000;
              min-height: 20px;
              padding: 2px 5px;
            }
            .form-textarea {
              flex: 1;
              border: 1px solid #000;
              min-height: 80px;
              padding: 5px;
              margin-top: 5px;
            }
            .assessment-table {
              width: 100%;
              border: 1px solid #000;
              border-collapse: collapse;
              margin: 15px 0;
            }
            .assessment-table th {
              background-color: #015185;
              color: white;
              padding: 8px;
              text-align: center;
              font-weight: bold;
            }
            .assessment-table td {
              border: 1px solid #000;
              padding: 8px;
              text-align: left;
            }
            .assessment-table .yes-no {
              text-align: center;
              font-weight: bold;
            }
            .signature-section {
              margin-top: 20px;
            }
            .signature-container {
              display: flex;
              justify-content: space-between;
              margin-top: 10px;
            }
            .signature-box {
              width: 48%;
              border: 1px solid #000;
              padding: 10px;
              min-height: 60px;
            }
            .signature-label {
              font-weight: bold;
              margin-bottom: 5px;
            }
            .signature-image {
              text-align: center;
              margin-top: 10px;
            }
            .signature-image img {
              max-width: 150px;
              max-height: 60px;
              border: 1px solid #ccc;
            }
          </style>
        </head>
        <body>
          <div class="header-row">
            <div class="logo-left">
              <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo"/>
              <div class="company-name">TITAN DRILLING</div>
            </div>
            <div class="doc-info-box">
              <table>
                <tr><td>Document Name</td><td>Drill Instruction Sheet</td></tr>
                <tr><td>Document ID</td><td>HSE-129</td></tr>
                <tr><td>Version Date</td><td>03/08/2021</td></tr>
                <tr><td>Version</td><td>1</td></tr>
                <tr><td>Approved</td><td>Ruan Du Toit</td></tr>
                <tr><td>Request #</td><td>${report.id || 'N/A'}</td></tr>
              </table>
            </div>
          </div>

          <div class="title-section">
            <h2>DRILL INSTRUCTION SHEET</h2>
          </div>

          <div class="form-section">
            <div class="form-row">
              <div class="form-label">Date:</div>
              <div class="form-value">${step2.date || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Request ID:</div>
              <div class="form-value">${report.id || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Country:</div>
              <div class="form-value">${step1.country || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Project:</div>
              <div class="form-value">${step1.project || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Client:</div>
              <div class="form-value">${step2.client || ''}</div>
            </div>
            
            <div class="form-row">
              <div class="form-label">Hole ID:</div>
              <div class="form-value">${step2.holeId || ''}</div>
            </div>
          </div>

          <div class="assessment-section">
            <h3>SITE ASSESSMENT</h3>
            <table class="assessment-table">
              <thead>
                <tr>
                  <th>DESCRIPTION</th>
                  <th>YES / NO</th>
                </tr>
              </thead>
              <tbody>
                <tr>
                  <td>Drill pad access safe</td>
                  <td class="yes-no">${step3.drillPadAccessSafe || ''}</td>
                </tr>
                <tr>
                  <td>Drill pad level</td>
                  <td class="yes-no">${step3.drillPadLevel || ''}</td>
                </tr>
                <tr>
                  <td>Adequate space for auxiliary equipment parking</td>
                  <td class="yes-no">${step3.adequateSpaceAuxiliary || ''}</td>
                </tr>
                <tr>
                  <td>Drill pad size 16m x 16m</td>
                  <td class="yes-no">${step3.drillPadSize || ''}</td>
                </tr>
                <tr>
                  <td>Any Hazards or Risks</td>
                  <td class="yes-no">${step3.anyHazardsRisks || ''}</td>
                </tr>
                <tr>
                  <td>Adequate space for visitor's vehicle parking</td>
                  <td class="yes-no">${step3.adequateSpaceVisitors || ''}</td>
                </tr>
              </tbody>
            </table>
          </div>

          <div class="form-section">
            <div class="form-row">
              <div class="form-label">Special Instructions / Comments:</div>
              <div class="form-textarea">${step3.specialInstructions || ''}</div>
            </div>
          </div>

          <div class="signature-section">
            <div class="signature-container">
              <div class="signature-box">
                <div class="signature-label">CLIENT REPRESENTATIVE SIGNATURE:</div>
                <div class="signature-label">Name: ${step3.clientRepresentativeName || ''}</div>
                <div class="signature-image">
                  ${step3.clientSignature ? `<img src="${step3.clientSignature}" alt="Client Signature"/>` : ''}
                </div>
              </div>
              <div class="signature-box">
                <div class="signature-label">TITAN DRILLING REPRESENTATIVE SIGNATURE:</div>
                <div class="signature-label">Name: ${step3.titanRepresentativeName || ''}</div>
                <div class="signature-image">
                  ${step3.titanSignature ? `<img src="${step3.titanSignature}" alt="Titan Signature"/>` : ''}
                </div>
              </div>
            </div>
          </div>
        </body>
        </html>
      `

      const options = {
        html,
        fileName: report.id,
        directory: 'Documents',
      }

      const file = await RNHTMLtoPDF.convert(options)
      return file.filePath
    } catch (error) {
      console.error('Error generating PDF:', error)
      throw error
    }
  }

  const handleViewPdf = async () => {
    if (!reportId) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userEmail = await AsyncStorage.getItem('userEmail');
      if (!userEmail) {
        Alert.alert('Error', 'User not authenticated. Please log in again.');
        navigation.replace('AdminLoginScreen');
        return;
      }

      const userSnapshot = await database()
        .ref('/Globalusers/admins')
        .orderByChild('email')
        .equalTo(userEmail)
        .once('value');
      const userData = userSnapshot.val();
      if (!userData) {
        Alert.alert('Error', 'User not found in admin database.');
        navigation.replace('AdminLoginScreen');
        return;
      }
      const userId = Object.keys(userData)[0];

      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      navigation.navigate('HSEViewPDF', { 
        pdfUrl: pdfUrl, 
        onGoBack: () => { 
          setShowSuccessModal(false) 
        } 
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    } finally {
      setShowSuccessModal(false);
    }
  };

  const handleBack = () => {
    setActiveStep(2);
  };

  const canvasStyle = `
    .m-signature-pad { height: ${height * 0.4}px; width: 100%; margin: 0; padding: 0; }
    .m-signature-pad--body { border: 1px solid #ddd; height: 100%; width: 100%; }
    canvas { height: 100% !important; width: 100% !important; }
  `

  const renderAssessmentRow = (label, field, required = false) => (
    <View style={styles.assessmentRow}>
      <View style={styles.assessmentTextContainer}>
        <Text style={styles.assessmentLabel}>
          {label}
          {required && <Text style={styles.required}> *</Text>}
        </Text>
      </View>
      <View style={styles.assessmentRadioContainer}>
        <TouchableOpacity
          style={styles.radioOption}
          onPress={() => handleInputChange(field, 'Yes')}
        >
          <View style={[
            styles.radioButton,
            formData[field] === 'Yes' && styles.radioButtonSelected
          ]}>
            {formData[field] === 'Yes' && (
              <View style={styles.radioButtonInner} />
            )}
          </View>
          <Text style={styles.radioLabel}>YES</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={styles.radioOption}
          onPress={() => handleInputChange(field, 'No')}
        >
          <View style={[
            styles.radioButton,
            formData[field] === 'No' && styles.radioButtonSelected
          ]}>
            {formData[field] === 'No' && (
              <View style={styles.radioButtonInner} />
            )}
          </View>
          <Text style={styles.radioLabel}>NO</Text>
        </TouchableOpacity>
      </View>
    </View>
  );

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent} showsVerticalScrollIndicator={false}>
        
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Site Assessment</Text>
          
          {/* Assessment Table */}
          <View style={styles.tableContainer}>
            <View style={styles.tableHeader}>
              <Text style={styles.tableHeaderText}>DESCRIPTION</Text>
              <Text style={styles.tableHeaderText}>YES / NO</Text>
            </View>

            <View style={styles.tableContent}>
              {renderAssessmentRow('Drill pad access safe', 'drillPadAccessSafe', true)}
              {renderAssessmentRow('Drill pad level', 'drillPadLevel', true)}
              {renderAssessmentRow('Adequate space for auxiliary equipment parking', 'adequateSpaceAuxiliary')}
              {renderAssessmentRow('Drill pad size 16m x 16m', 'drillPadSize', true)}
              {renderAssessmentRow('Any Hazards or Risks', 'anyHazardsRisks')}
              {renderAssessmentRow('Adequate space for visitor\'s vehicle parking', 'adequateSpaceVisitors')}
            </View>
          </View>
        </View>

        {/* Special Instructions/Comments */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Special Instructions / Comments</Text>
          <TextInput
            style={[styles.textInput, styles.textArea]}
            value={formData.specialInstructions}
            onChangeText={(value) => handleInputChange('specialInstructions', value)}
            placeholder="Enter special instructions or comments..."
            multiline
            numberOfLines={4}
            textAlignVertical="top"
            placeholderTextColor="#888"
          />
        </View>

        {/* Signature Section */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Signatures</Text>
          
          {/* Client Representative */}
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Name of Client Representative:</Text>
            <TextInput
              style={styles.textInput}
              value={formData.clientRepresentativeName}
              onChangeText={(text) => handleInputChange('clientRepresentativeName', text)}
              placeholder="Enter client representative name"
              placeholderTextColor="#888"
            />
          </View>

          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Signature Client Representative:</Text>
            {formData.clientSignature ? (
              <View style={styles.signatureContainer}>
                <Text style={styles.signatureStatus}>Signature Completed</Text>
                <Image
                  source={{ uri: formData.clientSignature }}
                  style={styles.signatureImage}
                  resizeMode="contain"
                />
                <TouchableOpacity
                  style={styles.resignButton}
                  onPress={() => openSignatureModal('client')}
                >
                  <Text style={styles.resignButtonText}>Re-sign</Text>
                </TouchableOpacity>
              </View>
            ) : (
              <TouchableOpacity 
                style={styles.signatureBox} 
                onPress={() => openSignatureModal('client')}
              >
                <MaterialIcons name="edit" size={40} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Tap to sign</Text>
              </TouchableOpacity>
            )}
          </View>

          {/* Titan Drilling Representative */}
          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Name of Titan Drilling Representative:</Text>
            <TextInput
              style={styles.textInput}
              value={formData.titanRepresentativeName}
              onChangeText={(text) => handleInputChange('titanRepresentativeName', text)}
              placeholder="Enter Titan drilling representative name"
              placeholderTextColor="#888"
            />
          </View>

          <View style={styles.inputContainer}>
            <Text style={styles.inputLabel}>Signature Titan Drilling Representative:</Text>
            {formData.titanSignature ? (
              <View style={styles.signatureContainer}>
                <Text style={styles.signatureStatus}>Signature Completed</Text>
                <Image
                  source={{ uri: formData.titanSignature }}
                  style={styles.signatureImage}
                  resizeMode="contain"
                />
                <TouchableOpacity
                  style={styles.resignButton}
                  onPress={() => openSignatureModal('titan')}
                >
                  <Text style={styles.resignButtonText}>Re-sign</Text>
                </TouchableOpacity>
              </View>
            ) : (
              <TouchableOpacity 
                style={styles.signatureBox} 
                onPress={() => openSignatureModal('titan')}
              >
                <MaterialIcons name="edit" size={40} color="#015185" />
                <Text style={styles.signaturePlaceholder}>Tap to sign</Text>
              </TouchableOpacity>
            )}
          </View>
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity
          style={[styles.nextButton, isLoading && styles.disabledButton]}
          onPress={handleSubmit}
          disabled={isLoading}
        >
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* SyncModal for Loading */}
      <SyncModal visible={isLoading} message="Generating Report" />

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <Text style={styles.successModalTitle}>Success</Text>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalText}>
              Site Assessment Generated and Saved
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false)
                  navigation.navigate('HSEInspectionHome')
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Signature Modal */}
      <Modal
        visible={signatureModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={closeSignatureModal}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <View style={styles.modalHeader}>
              <Text style={styles.modalTitle}>
                {currentSignatureType === 'client' ? 'Client Representative' : 'Titan Drilling Representative'} Signature
              </Text>
              <TouchableOpacity onPress={closeSignatureModal}>
                <MaterialIcons name="close" size={24} color="#333" />
              </TouchableOpacity>
            </View>
            
            <View style={styles.signatureCanvasContainer}>
              <SignatureCanvas
                ref={signatureRef}
                onOK={handleSignatureData}
                onEmpty={() => {
                  console.log('Signature canvas is empty')
                }}
                onBegin={() => console.log('User started drawing')}
                onEnd={() => console.log('User finished drawing')}
                descriptionText="Sign here"
                clearText="Clear"
                confirmText="Confirm"
                webStyle={canvasStyle}
                penColor="#000000"
                backgroundColor="#ffffff"
                minStrokeWidth={2}
                maxStrokeWidth={4}
                onError={(error) => {
                  console.error('SignatureCanvas WebView error:', error)
                  Alert.alert('Error', 'Failed to load signature canvas.')
                }}
              />
            </View>
            
            <View style={styles.modalButtons}>
              <TouchableOpacity 
                style={styles.modalCancelButton}
                onPress={closeSignatureModal}
              >
                <Text style={styles.modalCancelButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalClearButton}
                onPress={handleSignatureClear}
              >
                <Text style={styles.modalClearButtonText}>Clear</Text>
              </TouchableOpacity>
              <TouchableOpacity 
                style={styles.modalConfirmButton}
                onPress={handleSignatureConfirm}
              >
                <Text style={styles.modalConfirmButtonText}>Confirm</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
      
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  section: {
    marginBottom: 25,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
  },
  inputContainer: {
    marginBottom: 15,
  },
  inputLabel: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 5,
    fontWeight: '500',
  },
  textInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 12,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  textArea: {
    height: 100,
    textAlignVertical: 'top',
  },
  tableContainer: {
    borderWidth: 1,
    borderColor: '#dee2e6',
    borderRadius: 8,
    backgroundColor: '#fff',
    marginBottom: 15,
  },
  tableHeader: {
    flexDirection: 'row',
    backgroundColor: '#015185',
    paddingVertical: 12,
    paddingHorizontal: 15,
    borderTopLeftRadius: 7,
    borderTopRightRadius: 7,
  },
  tableHeaderText: {
    flex: 1,
    fontSize: 16,
    fontWeight: 'bold',
    color: '#fff',
    textAlign: 'center',
  },
  tableContent: {
    padding: 15,
  },
  assessmentRow: {
    flexDirection: 'row',
    alignItems: 'flex-start',
    marginBottom: 15,
    paddingBottom: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    minHeight: 40,
  },
  assessmentTextContainer: {
    flex: 1,
    marginRight: 15,
    justifyContent: 'center',
  },
  assessmentRadioContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    minWidth: 120,
  },
  assessmentLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
    marginBottom: 8,
  },
  required: {
    color: '#dc3545',
    fontWeight: 'bold',
  },
  radioOption: {
    flexDirection: 'row',
    alignItems: 'center',
    marginHorizontal: 8,
  },
  radioButton: {
    width: 18,
    height: 18,
    borderRadius: 9,
    borderWidth: 2,
    borderColor: '#015185',
    marginRight: 6,
    justifyContent: 'center',
    alignItems: 'center',
  },
  radioButtonSelected: {
    backgroundColor: '#015185',
  },
  radioButtonInner: {
    width: 6,
    height: 6,
    borderRadius: 3,
    backgroundColor: '#fff',
  },
  radioLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  signatureContainer: {
    alignItems: 'center',
  },
  signatureStatus: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
  },
  signatureImage: {
    width: width * 0.5,
    height: height * 0.12,
    borderRadius: 10,
    backgroundColor: '#f5f5f5',
  },
  resignButton: {
    backgroundColor: '#015185',
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    marginTop: 10,
  },
  resignButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: '600',
  },
  signatureBox: {
    height: height * 0.15,
    backgroundColor: '#f5f5f5',
    borderRadius: 10,
    justifyContent: 'center',
    alignItems: 'center',
    borderWidth: 2,
    borderColor: '#015185',
    borderStyle: 'dashed',
  },
  signaturePlaceholder: {
    fontSize: 16,
    color: '#666',
    marginTop: 10,
  },
  disabledButton: {
    opacity: 0.5,
  },
  // Modal Styles
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '95%',
    maxHeight: '90%',
  },
  modalHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 20,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
  },
  signatureCanvasContainer: {
    alignItems: 'center',
    marginBottom: 20,
    height: height * 0.4,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 10,
    overflow: 'hidden',
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalCancelButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#015185',
    marginRight: 5,
  },
  modalCancelButtonText: {
    color: '#015185',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalClearButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#FF6B6B',
    marginHorizontal: 5,
  },
  modalClearButtonText: {
    color: '#FF6B6B',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  modalConfirmButton: {
    flex: 1,
    paddingVertical: 8,
    paddingHorizontal: 15,
    borderRadius: 5,
    backgroundColor: '#015185',
    marginLeft: 5,
  },
  modalConfirmButtonText: {
    color: '#fff',
    fontSize: 12,
    fontWeight: '600',
    textAlign: 'center',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '60%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: 16,
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 18,
    borderRadius: 5,
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginRight: 5,
  },
  bottomicon1: {
    marginLeft: 5,
  },
});

export default HSE_129_Step3;