@extends('welcome')

@section('title', 'Labor Codes')

@section('content')
<div class="labor-codes-container">
    <div class="labor-codes-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Labor Codes</h1>
                <p>Manage and track labor codes and hourly rates efficiently</p>
            </div>
        </div>
    </div>
    
    <div class="labor-codes-content">
        <div class="action-bar">
            <div class="action-content">
                <div class="search-section">
                    <div class="search-container">
                        <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="11" cy="11" r="8"/>
                            <path d="m21 21-4.35-4.35"/>
                        </svg>
                        <input type="text" class="search-input" placeholder="Search labor codes..." id="laborCodesSearch">
                    </div>
                </div>
                
                <div class="action-buttons">
                    <button class="btn-add" data-bs-toggle="modal" data-bs-target="#add-checklist">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M12 5v14m-7-7h14"/>
                        </svg>
                        <span>Add Labor Code</span>
                    </button>
                </div>
            </div>
        </div>
        
        <div class="table-section">
            <div class="table-container">
                <table class="labor-codes-table">
                    <thead class="table-header">
                        <tr>
                            <th>Labor Code</th>
                            <th>Hourly Rate</th>
                            <th>Description</th>
                        </tr>
                    </thead>
                    <tbody class="table-body" id="laborCodesTableBody">
                        <tr class="empty-row">
                            <td colspan="3" class="empty-state">
                                <div class="empty-content">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M12 2L2 7l10 5 10-5-10-5z"/>
                                        <path d="M2 17l10 5 10-5"/>
                                        <path d="M2 12l10 5 10-5"/>
                                    </svg>
                                    <h3>No Labor Codes</h3>
                                    <p>There are currently no labor codes to display.</p>
                                </div>
                            </td>
                        </tr>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>
<script>
(function() {
    'use strict';

    // Configuration object for centralized settings
    const CONFIG = {
        selectors: {
            laborCodesSearch: '#laborCodesSearch',
            laborCodesTableBody: '#laborCodesTableBody',
            sidebar: '.sidebar'
        },
        messages: {
            loading: 'Loading...',
            error: 'An error occurred. Please try again.',
            success: 'Operation completed successfully!'
        },
        timing: {
            debounceDelay: 300
        }
    };

    // DOM element caching
    let elements = {};

    // Initialize DOM elements
    function initElements() {
        elements = {
            laborCodesSearch: document.querySelector(CONFIG.selectors.laborCodesSearch),
            laborCodesTableBody: document.querySelector(CONFIG.selectors.laborCodesTableBody),
            sidebar: document.querySelector(CONFIG.selectors.sidebar)
        };
    }

    // Initialize the application
    function init() {
        initElements();
        initEventListeners();
    }

    // Initialize event listeners
    function initEventListeners() {
        // Search functionality
        if (elements.laborCodesSearch) {
            elements.laborCodesSearch.addEventListener('input', function() {
                filterTable(CONFIG.selectors.laborCodesTableBody, this.value);
            });
        }
    }

    // Toggle sidebar
    function toggleSidebar() {
        if (elements.sidebar) {
            elements.sidebar.classList.toggle('collapsed');
        }
    }

    // Filter table function
    function filterTable(tableBodyId, searchTerm) {
        const tableBody = document.querySelector(tableBodyId);
        if (!tableBody) return;
        
        const rows = tableBody.getElementsByTagName('tr');
        
        for (let row of rows) {
            const cells = row.getElementsByTagName('td');
            let found = false;
            
            for (let cell of cells) {
                if (cell.textContent.toLowerCase().includes(searchTerm.toLowerCase())) {
                    found = true;
                    break;
                }
            }
            
            row.style.display = found ? '' : 'none';
        }
    }

    // Make functions globally accessible
    window.toggleSidebar = toggleSidebar;

    // Initialize the application
    init();
})();

    // Form submission handling
    document.addEventListener('DOMContentLoaded', function() {
        const laborCodeForm = document.getElementById('laborCodeForm');
        const modal = document.getElementById('add-checklist');
        
        if (laborCodeForm) {
            laborCodeForm.addEventListener('submit', function(e) {
                e.preventDefault();
                
                const formData = new FormData(this);
                const laborCode = formData.get('laborCode');
                const hourlyRate = formData.get('hourlyRate');
                const description = formData.get('description');
                
                // Validate required fields
                if (!laborCode || !hourlyRate) {
                    showNotification('Please fill in all required fields', 'error');
                    return;
                }
                
                // Simulate form submission
                showNotification('Labor code added successfully!', 'success');
                
                // Close modal properly
                closeModal();
                
                // Reset form
                this.reset();
            });
        }
        
        // Handle modal close events
        if (modal) {
            // Close modal when clicking cancel button or close button
            modal.addEventListener('click', function(e) {
                if (e.target.classList.contains('btn-cancel') || 
                    e.target.classList.contains('modal-close') ||
                    e.target.closest('.btn-cancel') ||
                    e.target.closest('.modal-close')) {
                    closeModal();
                }
            });
            
            // Close modal when clicking outside
            modal.addEventListener('click', function(e) {
                if (e.target === modal) {
                    closeModal();
                }
            });
            
            // Close modal with Escape key
            document.addEventListener('keydown', function(e) {
                if (e.key === 'Escape' && modal.classList.contains('show')) {
                    closeModal();
                }
            });
        }
    });
    
    // Function to properly close modal
    function closeModal() {
        const modal = document.getElementById('add-checklist');
        if (modal) {
            modal.classList.remove('show');
            modal.style.display = 'none';
            
            // Remove backdrop if it exists
            const backdrop = document.querySelector('.modal-backdrop');
            if (backdrop) {
                backdrop.remove();
            }
            
            // Remove modal-open class from body
            document.body.classList.remove('modal-open');
            document.body.style.overflow = '';
            document.body.style.paddingRight = '';
        }
    }

    // Notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `labor-notification labor-notification-${type}`;
        notification.textContent = message;
        
        notification.style.cssText = `
            position: fixed;
            top: 80px;
            right: 20px;
            padding: 12px 20px;
            border-radius: 0;
            color: white;
            font-weight: 500;
            font-size: 14px;
            z-index: 10001;
            opacity: 0;
            transform: translateX(100%);
            transition: all 0.3s ease;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
            font-family: 'Inter', sans-serif;
        `;
        
        if (type === 'success') {
            notification.style.background = 'linear-gradient(135deg, #10b981 0%, #059669 100%)';
        } else if (type === 'error') {
            notification.style.background = 'linear-gradient(135deg, #ef4444 0%, #dc2626 100%)';
        } else if (type === 'warning') {
            notification.style.background = 'linear-gradient(135deg, #f59e0b 0%, #d97706 100%)';
        } else {
            notification.style.background = 'linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)';
        }
        
        document.body.appendChild(notification);
        
        setTimeout(() => {
            notification.style.opacity = '1';
            notification.style.transform = 'translateX(0)';
        }, 100);
        
        setTimeout(() => {
            notification.style.opacity = '0';
            notification.style.transform = 'translateX(100%)';
            setTimeout(() => {
                if (notification.parentNode) {
                    document.body.removeChild(notification);
                }
            }, 300);
        }, 3000);
    }
</script>
</div>
<!-- Add Labor Code Modal -->
<div class="modal fade" id="add-checklist" tabindex="-1" aria-labelledby="addLaborCodeModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h4 class="modal-title" id="addLaborCodeModalLabel">Add Labor Code</h4>
                <button type="button" class="modal-close" data-bs-dismiss="modal" aria-label="Close">&times;</button>
            </div>
            <div class="modal-body">
                <form class="labor-form" id="laborCodeForm">
                    <div class="form-group">
                        <label for="laborCode">Labor Code * :</label>
                        <input type="text" class="form-control" placeholder="Enter labor code" id="laborCode" name="laborCode" required>
                    </div>
                    <div class="form-group">
                        <label for="hourlyRate">Hourly Rate * :</label>
                        <input type="number" class="form-control" placeholder="Enter hourly rate" id="hourlyRate" name="hourlyRate" step="0.01" required>
                    </div>
                    <div class="form-group">
                        <label for="description">Description :</label>
                        <textarea class="form-control" placeholder="Enter description" id="description" name="description" rows="3"></textarea>
                    </div>
                    <div class="modal-actions">
                        <button type="submit" class="btn-save">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M12 5v14m-7-7h14"/>
                            </svg>
                            Add Labor Code
                        </button>
                        <button type="button" class="btn-cancel" data-bs-dismiss="modal">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M18 6L6 18M6 6l12 12"/>
                            </svg>
                            Cancel
                        </button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>

<style>
/* CSS Variables for consistent theming */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --secondary-hover: #5a6268;
    --success-color: #28a745;
    --success-hover: #218838;
    --warning-color: #ffc107;
    --warning-hover: #e0a800;
    --danger-color: #dc3545;
    --danger-hover: #c82333;
    --info-color: #17a2b8;
    --info-hover: #138496;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #6c757d;
    --text-light: #ffffff;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-hover: #e9ecef;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --transition-slow: all 0.3s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-modal: 1050;
    --z-index-dropdown: 1000;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== LABOR CODES CONTAINER ===== */
.labor-codes-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 24px;
}

/* ===== HEADER SECTION ===== */
.labor-codes-header {
    background: var(--primary-color);
    border-radius: 0;
    padding: 32px;
    margin-bottom: 24px;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
    border: 1px solid rgba(255, 255, 255, 0.1);
}

.header-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.page-title h1 {
    color: var(--text-light);
    font-size: 32px;
    font-weight: 700;
    margin: 0 0 8px 0;
    text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
}

.page-title p {
    color: rgba(255, 255, 255, 0.9);
    font-size: 16px;
    font-weight: 400;
    margin: 0;
}

/* ===== CONTENT SECTION ===== */
.labor-codes-content {
    background: var(--bg-primary);
    border-radius: 0;
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    overflow: hidden;
}

/* ===== ACTION BAR ===== */
.action-bar {
    background: var(--bg-light);
    border-bottom: 1px solid var(--border-light);
    padding: 20px 24px;
}

.action-content {
    display: flex;
    align-items: center;
    justify-content: space-between;
    gap: 20px;
}

.search-section {
    flex: 1;
    max-width: 400px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: var(--text-muted);
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid var(--border-color);
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: var(--bg-primary);
    transition: var(--transition-base);
}

.search-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.action-buttons {
    display: flex;
    align-items: center;
    gap: 12px;
}

.btn-add {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid var(--border-color);
    border-radius: 0;
    background: var(--success-color);
    color: var(--text-light);
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: var(--transition-base);
}

.btn-add:hover {
    background: var(--success-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
    text-decoration: none;
}

/* ===== TABLE SECTION ===== */
.table-section {
    padding: 0;
}

.table-container {
    overflow-x: auto;
}

.labor-codes-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
}

.table-header {
    background: linear-gradient(135deg, #f8fafc 0%, #f1f5f9 100%);
}

.table-header th {
    padding: 16px 12px;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    color: #374151;
    border-bottom: 1px solid #e5e7eb;
    border-radius: 0;
}

.table-body {
    background: white;
}

.table-row {
    border-bottom: 1px solid #f3f4f6;
    transition: all 0.2s ease;
}

.table-row:hover {
    background: #f8fafc;
    transform: translateY(-1px);
    box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
}

.table-row td {
    padding: 16px 12px;
    font-size: 14px;
    color: #374151;
    vertical-align: middle;
}

/* ===== EMPTY STATE ===== */
.empty-row {
    background: white;
}

.empty-state {
    text-align: center;
    padding: 48px 24px;
    background: white;
}

.empty-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    gap: 16px;
}

.empty-content svg {
    color: #9ca3af;
}

.empty-content h3 {
    font-size: 18px;
    font-weight: 600;
    color: #374151;
    margin: 0;
}

.empty-content p {
    font-size: 14px;
    color: #6b7280;
    margin: 0;
}

/* ===== MODAL STYLES ===== */
.modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    z-index: 1055;
    display: none;
    justify-content: center;
    align-items: center;
}

.modal.show {
    display: flex !important;
}

.modal.fade .modal-dialog {
    transition: transform 0.3s ease-out;
    transform: translate(0, -50px);
}

.modal.show .modal-dialog {
    transform: none;
}

/* Ensure modal backdrop doesn't interfere */
.modal-backdrop {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    z-index: 1050;
}

/* Remove body scroll lock when modal is closed */
body.modal-open {
    overflow: hidden;
    padding-right: 0 !important;
}

.modal-dialog {
    width: 90%;
    max-width: 500px;
    margin: 0 auto;
}

.modal-content {
    background: white;
    border-radius: 0;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
    overflow: hidden;
}

.modal-header {
    padding: 20px 24px;
    border-bottom: 1px solid #e2e8f0;
    display: flex;
    justify-content: space-between;
    align-items: center;
    background: #0055a5;
    color: white;
}

.modal-title {
    margin: 0;
    color: white;
    font-size: 20px;
    font-weight: 600;
}

.modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: white;
    padding: 8px;
    width: 40px;
    height: 40px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 0;
    transition: background-color 0.2s;
}

.modal-close:hover {
    background-color: rgba(255, 255, 255, 0.2);
}

.modal-body {
    padding: 24px;
}

.labor-form {
    display: flex;
    flex-direction: column;
    gap: 20px;
}

.form-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.form-group label {
    font-weight: 500;
    color: #374151;
    font-size: 14px;
}

.form-control {
    border: 1px solid #d1d5db;
    border-radius: 0;
    padding: 12px 16px;
    font-size: 14px;
    transition: all 0.2s ease;
    background: white;
}

.form-control:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

textarea.form-control {
    min-height: 80px;
    resize: vertical;
}

.modal-actions {
    display: flex;
    justify-content: flex-end;
    gap: 12px;
    margin-top: 20px;
}

.btn-save {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: #00B894;
    color: white;
    border: 1px solid #00B894;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-save:hover {
    background: #00a085;
    border-color: #00a085;
    transform: translateY(-1px);
}

.btn-cancel {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: #545d6b;
    color: white;
    border: 1px solid #545d6b;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel:hover {
    background: #3d4449;
    border-color: #3d4449;
    transform: translateY(-1px);
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .labor-codes-container {
        padding: 20px;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 16px;
    }
    
    .search-section {
        max-width: none;
    }
    
    .action-buttons {
        justify-content: center;
    }
}

@media (max-width: 992px) {
    .labor-codes-container {
        padding: 15px;
    }
    
    .labor-codes-header {
        padding: 20px;
        margin-bottom: 20px;
    }
    
    .page-title h1 {
        font-size: 26px;
    }
    
    .page-title p {
        font-size: 15px;
    }
    
    .action-bar {
        padding: 16px 20px;
    }
    
    .btn-add {
        padding: 10px 16px;
        font-size: 13px;
    }
    
    .modal-dialog {
        width: 95%;
        max-width: 450px;
    }
    
    .table-header th,
    .table-row td {
        padding: 14px 10px;
        font-size: 13px;
    }
}

@media (max-width: 768px) {
    .labor-codes-container {
        padding: 12px;
    }
    
    .labor-codes-header {
        padding: 16px;
        margin-bottom: 16px;
    }
    
    .page-title h1 {
        font-size: 22px;
    }
    
    .page-title p {
        font-size: 14px;
    }
    
    .action-bar {
        padding: 12px 16px;
    }
    
    .action-content {
        gap: 12px;
    }
    
    .search-section {
        order: 1;
    }
    
    .action-buttons {
        order: 2;
        flex-direction: column;
        gap: 8px;
    }
    
    .btn-add {
        width: 100%;
        justify-content: center;
        padding: 12px 16px;
        font-size: 14px;
    }
    
    .search-input {
        padding: 12px 12px 12px 40px;
        font-size: 14px;
    }
    
    .search-icon {
        left: 12px;
        width: 18px;
        height: 18px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .labor-codes-table {
        min-width: 600px;
    }
    
    .table-header th,
    .table-row td {
        padding: 12px 8px;
        font-size: 13px;
        white-space: nowrap;
    }
    
    .table-header th:first-child,
    .table-row td:first-child {
        position: sticky;
        left: 0;
        background: inherit;
        z-index: 1;
    }
    
    .modal-dialog {
        width: 100%;
        height: 100%;
        margin: 0;
        max-width: none;
    }
    
    .modal-content {
        height: 100%;
        border-radius: 0;
        display: flex;
        flex-direction: column;
    }
    
    .modal-header {
        padding: 16px 20px;
        flex-shrink: 0;
    }
    
    .modal-body {
        padding: 20px;
        flex: 1;
        overflow-y: auto;
    }
    
    .modal-actions {
        flex-direction: column;
        gap: 8px;
        margin-top: 16px;
    }
    
    .btn-save,
    .btn-cancel {
        width: 100%;
        justify-content: center;
        padding: 12px 16px;
    }
    
    .empty-state {
        padding: 40px 20px;
    }
    
    .empty-content h3 {
        font-size: 16px;
    }
    
    .empty-content p {
        font-size: 14px;
    }
}

@media (max-width: 576px) {
    .labor-codes-container {
        padding: 8px;
    }
    
    .labor-codes-header {
        padding: 12px;
        margin-bottom: 12px;
    }
    
    .page-title h1 {
        font-size: 20px;
    }
    
    .page-title p {
        font-size: 13px;
    }
    
    .action-bar {
        padding: 8px 12px;
    }
    
    .action-content {
        gap: 8px;
    }
    
    .search-input {
        padding: 10px 10px 10px 36px;
        font-size: 13px;
    }
    
    .search-icon {
        left: 10px;
        width: 16px;
        height: 16px;
    }
    
    .btn-add {
        padding: 10px 14px;
        font-size: 13px;
    }
    
    .labor-codes-table {
        min-width: 500px;
    }
    
    .table-header th,
    .table-row td {
        padding: 8px 6px;
        font-size: 12px;
    }
    
    .empty-state {
        padding: 32px 16px;
    }
    
    .empty-content h3 {
        font-size: 15px;
    }
    
    .empty-content p {
        font-size: 13px;
    }
    
    .modal-header {
        padding: 12px 16px;
    }
    
    .modal-title {
        font-size: 18px;
    }
    
    .modal-body {
        padding: 16px;
    }
    
    .form-group {
        gap: 6px;
    }
    
    .form-group label {
        font-size: 13px;
    }
    
    .form-control {
        padding: 10px 12px;
        font-size: 13px;
    }
    
    .btn-save,
    .btn-cancel {
        padding: 10px 14px;
        font-size: 13px;
    }
}

@media (max-width: 480px) {
    .labor-codes-container {
        padding: 4px;
    }
    
    .labor-codes-header {
        padding: 10px;
        margin-bottom: 10px;
    }
    
    .page-title h1 {
        font-size: 18px;
    }
    
    .page-title p {
        font-size: 12px;
    }
    
    .action-bar {
        padding: 6px 8px;
    }
    
    .search-input {
        padding: 8px 8px 8px 32px;
        font-size: 12px;
    }
    
    .search-icon {
        left: 8px;
        width: 14px;
        height: 14px;
    }
    
    .btn-add {
        padding: 8px 12px;
        font-size: 12px;
    }
    
    .table-header th,
    .table-row td {
        padding: 6px 4px;
        font-size: 11px;
    }
    
    .empty-state {
        padding: 24px 12px;
    }
    
    .empty-content svg {
        width: 40px;
        height: 40px;
    }
    
    .empty-content h3 {
        font-size: 14px;
    }
    
    .empty-content p {
        font-size: 12px;
    }
    
    .modal-header {
        padding: 10px 12px;
    }
    
    .modal-title {
        font-size: 16px;
    }
    
    .modal-body {
        padding: 12px;
    }
    
    .form-group label {
        font-size: 12px;
    }
    
    .form-control {
        padding: 8px 10px;
        font-size: 12px;
    }
    
    .btn-save,
    .btn-cancel {
        padding: 8px 12px;
        font-size: 12px;
    }
}

/* Additional responsive utilities */
@media (max-width: 768px) {
    .table-container::-webkit-scrollbar {
        height: 6px;
    }
    
    .table-container::-webkit-scrollbar-track {
        background: #f1f1f1;
    }
    
    .table-container::-webkit-scrollbar-thumb {
        background: #c1c1c1;
        border-radius: 3px;
    }
    
    .table-container::-webkit-scrollbar-thumb:hover {
        background: #a8a8a8;
    }
}

/* Ensure proper spacing on all screen sizes */
@media (max-width: 768px) {
    .labor-codes-content {
        margin: 0;
    }
    
    .action-bar {
        margin: 0;
    }
    
    .table-section {
        margin: 0;
    }
}

/* Fix for very small screens */
@media (max-width: 360px) {
    .labor-codes-container {
        padding: 2px;
    }
    
    .labor-codes-header {
        padding: 8px;
    }
    
    .page-title h1 {
        font-size: 16px;
    }
    
    .page-title p {
        font-size: 11px;
    }
    
    .action-bar {
        padding: 4px 6px;
    }
    
    .search-input {
        padding: 6px 6px 6px 28px;
        font-size: 11px;
    }
    
    .search-icon {
        left: 6px;
        width: 12px;
        height: 12px;
    }
    
    .btn-add {
        padding: 6px 10px;
        font-size: 11px;
    }
    
    .table-header th,
    .table-row td {
        padding: 4px 2px;
        font-size: 10px;
    }
}

/* ===== FOCUS STATES ===== */
.search-input:focus,
.btn-add:focus,
.form-control:focus,
.btn-save:focus,
.btn-cancel:focus {
    outline: 2px solid rgba(0, 85, 165, 0.5);
    outline-offset: 2px;
}

/* ===== SMOOTH ANIMATIONS ===== */
* {
    transition: all 0.2s ease;
}

/* ===== ADDITIONAL REQUEST_MAINTENANCE-STYLE IMPROVEMENTS ===== */
.labor-codes-container {
    background: #f8fafc;
}

.labor-codes-content {
    border-radius: 0;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
    border: 1px solid #e2e8f0;
}

.table-header th {
    font-weight: 600;
    color: #374151;
    border-bottom: 2px solid #e5e7eb;
}

.action-bar {
    background: #f8fafc;
    border-bottom: 1px solid #e2e8f0;
    padding: 20px 24px;
}

.search-container {
    position: relative;
    display: flex;
    align-items: center;
}

.search-icon {
    position: absolute;
    left: 12px;
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 12px 12px 12px 44px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 14px;
    font-weight: 400;
    background: white;
    transition: all 0.2s ease;
}

.action-buttons button {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 0;
    background: white;
    color: #374151;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s ease;
}

.action-buttons button:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    transform: translateY(-1px);
}

/* ===== HOVER EFFECTS ===== */
.btn-add:active,
.btn-save:active,
.btn-cancel:active {
    transform: translateY(0);
}

/* ===== LOADING STATES ===== */
.btn-add:disabled,
.btn-save:disabled {
    opacity: 0.6;
    cursor: not-allowed;
    transform: none;
}

.btn-add:disabled:hover,
.btn-save:disabled:hover {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    border-color: #059669;
    transform: none;
}
</style>
@endsection
@endsection