@extends('welcome')

@section('title', 'Workorder KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="workorder-kpi-container">
    <!-- Loading Overlay - Matching Workorder Design -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Workorder Data...</div>
        </div>
    </div>
    
    <!-- Header - Matching Workorder Design -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url()->previous() }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Workorder KPI Dashboard</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn-header">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                </button>
                <button type="button" class="create-btn" id="refresh-data-btn-header">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="workorder-kpi-content">
        <div class="content-box">
            <div class="time-period-selector">
                <div class="period-buttons">
                    <button class="period-btn" data-period="today">Today</button>
                    <button class="period-btn active" data-period="7days">Last 7 Days</button>
                    <button class="period-btn" data-period="30days">Last 30 Days</button>
                    <button class="period-btn" data-period="90days">Last 90 Days</button>
                    <button class="period-btn" data-period="custom">Custom Date</button>
                </div>
                
                <!-- Custom Date Range Picker -->
                <div class="custom-date-range" id="custom-date-range" style="display: none;">
                    <div class="date-range-inputs">
                        <div class="date-input-group">
                            <label for="start-date">Start Date:</label>
                            <input type="date" id="start-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label for="end-date">End Date:</label>
                            <input type="date" id="end-date" class="date-input">
                        </div>
                        <button class="btn-apply-dates" id="apply-custom-dates">Apply</button>
                        <button class="btn-clear-dates" id="clear-custom-dates">Clear</button>
                    </div>
                </div>
            </div>
            
            <div class="modern-tabs">
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="overview">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                        Overview
                    </button>
                    <button class="tab-btn" data-tab="datewise">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                            <line x1="16" y1="2" x2="16" y2="6"/>
                            <line x1="8" y1="2" x2="8" y2="6"/>
                            <line x1="3" y1="10" x2="21" y2="10"/>
                        </svg>
                        Datewise
                    </button>
                    <button class="tab-btn" data-tab="status-priority">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M9 12l2 2 4-4"/>
                            <circle cx="12" cy="12" r="10"/>
                        </svg>
                        Status & Priority
                    </button>
                    <button class="tab-btn" data-tab="workorder-type">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                            <polyline points="14,2 14,8 20,8"/>
                            <line x1="16" y1="13" x2="8" y2="13"/>
                        </svg>
                        Workorder Type
                    </button>
                    <button class="tab-btn" data-tab="by-equipment">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                        </svg>
                        By Equipment
                    </button>
                    <button class="tab-btn" data-tab="cost-analysis">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <line x1="12" y1="1" x2="12" y2="23"/>
                            <path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                        </svg>
                        Cost Analysis
                    </button>
                </div>
                <div class="tab-content">
                    <div id="overview" class="tab-pane active">
                        <div class="kpi-cards-grid">
                            <!-- KPI Card Template -->
                            <div class="kpi-card" data-kpi="total-workorder">
                                <div class="kpi-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                        <polyline points="14,2 14,8 20,8"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Workorders</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="initiated-workorder">
                                <div class="kpi-icon info">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"/>
                                        <polyline points="12 6 12 12 16 14"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Initiated</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="inprogress-workorder">
                                <div class="kpi-icon warning">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"/>
                                        <polyline points="8 12 12 16 16 12"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>In Progress</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="completed-workorder">
                                <div class="kpi-icon success">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M20 6L9 17l-5-5"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Completed</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                        </div>
                        <div class="overview-bottom-section">
                            <div class="additional-kpi-cards">
                                <div class="kpi-card">
                                    <div class="kpi-icon secondary">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <line x1="12" y1="1" x2="12" y2="23"/>
                                            <path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                                        </svg>
                                    </div>
                                    <div class="kpi-content">
                                        <h3>Total Cost</h3>
                                        <div class="kpi-value">$0</div>
                                        <div class="kpi-label">Amount</div>
                                    </div>
                                </div>
                                
                                <div class="kpi-card">
                                    <div class="kpi-icon danger">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <circle cx="12" cy="12" r="10"/>
                                            <line x1="15" y1="9" x2="9" y2="15"/>
                                            <line x1="9" y1="9" x2="15" y2="15"/>
                                        </svg>
                                    </div>
                                    <div class="kpi-content">
                                        <h3>Overdue</h3>
                                        <div class="kpi-value">00</div>
                                        <div class="kpi-label">Count</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="chart-section">
                                <div class="chart-card">
                                    <div class="chart-header">
                                        <h3>Workorder Status Distribution</h3>
                                    </div>
                                    <div class="chart-container">
                                        <canvas id="status-distribution"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    <div id="datewise" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Date wise Workorder Count</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="datewise-count"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Date wise Workorder Status</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="datewise-status"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="status-priority" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Workorder by Status</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="by-status"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Workorder by Priority</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="by-priority"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Priority Distribution by Status</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="priority-status"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="workorder-type" class="tab-pane">
                        <div class="chart-section">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Workorder Count by Type</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="by-type"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="by-equipment" class="tab-pane">
                        <div class="filter-section">
                            <div class="filter-form">
                                <div class="form-group">
                                    <label for="equipment-filter">Select Particular Equipment:</label>
                                    <select class="form-select" id="equipment-filter" name="equipment">
                                        <option value="">All Equipment</option>
                                    </select>
                                </div>
                                <button class="btn-go">Go</button>
                            </div>
                        </div>
                        
                        <div class="chart-section">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Workorder Count by Equipment</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="by-equipment-chart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="cost-analysis" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Total Cost by Status</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="cost-by-status"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Cost Breakdown</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="cost-breakdown"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Date wise Cost Trend</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="cost-trend"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<script>
    function toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        sidebar.classList.toggle('collapsed');
    }
</script>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>
<!-- Chart.js Library -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/2.9.4/Chart.js"></script>
    
<!-- SheetJS Library for Excel Export -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
<script>
// MongoDB API Configuration
const MONGODB_JOBCARD_API = 'http://168.231.113.219:5005';

// Global variables
let allWorkorderData = [];
let chartInstances = {};
let currentDateRange = 7; // Default to last 7 days
let customStartDate = null;
let customEndDate = null;

// Fetch workorder/job card data from MongoDB
async function fetchWorkorderData() {
    try {
        console.log('🔄 Fetching workorders/job cards from MongoDB...');
        console.log('🔗 API URL:', `${MONGODB_JOBCARD_API}/jobcards/all`);
        
        // Show loading state
        showLoadingState(true);
        
        const response = await fetch(`${MONGODB_JOBCARD_API}/jobcards/all?page=1&limit=10000`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'Accept': 'application/json'
            }
        });
        
        console.log('📡 Response status:', response.status);
        
        if (!response.ok) {
            const errorText = await response.text();
            console.error('❌ Response error:', errorText);
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        console.log('📥 Workorder/Job Card MongoDB response:', result);
        
        allWorkorderData = [];
        
        if (result.success && result.data && Array.isArray(result.data)) {
            console.log('📦 Processing', result.data.length, 'workorder/job card records...');
            
            // Map MongoDB data structure to expected format
            result.data.forEach((jobCard, index) => {
                const processedWorkorder = {
                    id: jobCard._id || jobCard.id,
                    _id: jobCard._id || jobCard.id,
                    ...jobCard,
                    // Normalize field names for consistency
                    title: jobCard.title || jobCard.workOrderTitle || 'N/A',
                    status: jobCard.status || 'Hold',
                    priority: jobCard.priority || 'Medium',
                    workOrderType: jobCard.workOrderType || jobCard.serviceType || 'General',
                    requestNumber: jobCard.requestNumber || 'N/A',
                    country: jobCard.location?.country || jobCard.country || 'N/A',
                    project: jobCard.location?.project || jobCard.project || 'N/A',
                    equipmentName: jobCard.equipment?.equipmentName || 'N/A',
                    createdAt: jobCard.createdAt || jobCard.updatedAt,
                    // Cost fields
                    totalWoCost: jobCard.totalWoCost || 0,
                    estimatedCost: jobCard.estimatedCost || 0,
                    totalLaborCost: jobCard.totalLaborCost || 0,
                    partMaterialCost: jobCard.partMaterialCost || 0,
                    additionalCost: jobCard.additionalCost || 0,
                    taxAmount: jobCard.taxAmount || 0,
                    // Dates
                    startDate: jobCard.startDate,
                    dueDate: jobCard.dueDate,
                    completedDate: jobCard.completedDate
                };
                
                allWorkorderData.push(processedWorkorder);
                
                // Log first 3 samples for debugging
                if (index < 3) {
                    console.log(`📄 Sample workorder ${index + 1}:`, {
                        id: processedWorkorder.id,
                        requestNumber: processedWorkorder.requestNumber,
                        title: processedWorkorder.title,
                        status: processedWorkorder.status,
                        priority: processedWorkorder.priority,
                        country: processedWorkorder.country,
                        project: processedWorkorder.project,
                        createdAt: processedWorkorder.createdAt
                    });
                }
            });
        } else {
            console.warn('⚠️ No data or invalid format in response:', result);
        }
        
        console.log('✅ Fetched workorders/job cards:', allWorkorderData.length);
        
        if (allWorkorderData.length === 0) {
            showNotification('No workorder data found', 'info');
        } else {
            showNotification(`Loaded ${allWorkorderData.length} workorder(s) from MongoDB`, 'success');
        }
        
        updateDashboard();
        showLoadingState(false);
    } catch (error) {
        console.error('❌ Error fetching workorders from MongoDB:', error);
        allWorkorderData = [];
        showNotification('Error fetching workorder data: ' + error.message, 'error');
        showLoadingState(false);
    }
}

// Show/hide loading state
function showLoadingState(isLoading) {
    const loadingOverlay = document.getElementById('loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.style.display = isLoading ? 'flex' : 'none';
    }
}

// Filter data by date range
function filterDataByDateRange(days) {
    console.log(`🔍 Filtering ${allWorkorderData.length} workorders by date range: ${days}`);
    
    // Handle custom date range
    if (days === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate);
            start.setHours(0, 0, 0, 0);
            const end = new Date(customEndDate);
            end.setHours(23, 59, 59, 999);
            
            const filtered = allWorkorderData.filter(workorder => {
                const createdAt = workorder.createdAt || workorder.updatedAt;
                if (!createdAt) {
                    console.log('⚠️ Workorder without date, including it:', workorder.requestNumber || workorder.id);
                    return true; // Include items without dates
                }
                
                const workorderDate = new Date(createdAt);
                if (isNaN(workorderDate.getTime())) {
                    console.log('⚠️ Invalid date format:', createdAt);
                    return true; // Include items with invalid dates
                }
                return workorderDate >= start && workorderDate <= end;
            });
            
            console.log(`✅ Filtered (custom): ${filtered.length} workorders`);
            return filtered;
        }
        // If no custom dates selected, return all data
        return allWorkorderData;
    }
    
    const now = new Date();
    const startDate = new Date(now);
    
    if (days === 'today') {
        startDate.setHours(0, 0, 0, 0);
    } else {
        startDate.setDate(now.getDate() - parseInt(days));
        startDate.setHours(0, 0, 0, 0);
    }
    
    const filtered = allWorkorderData.filter(workorder => {
        const createdAt = workorder.createdAt || workorder.updatedAt;
        if (!createdAt) {
            console.log('⚠️ Workorder without date, including it:', workorder.requestNumber || workorder.id);
            return true; // Include items without dates to show all data
        }
        
        const workorderDate = new Date(createdAt);
        if (isNaN(workorderDate.getTime())) {
            console.log('⚠️ Invalid date format:', createdAt, 'for workorder:', workorder.requestNumber || workorder.id);
            return true; // Include items with invalid dates
        }
        
        return workorderDate >= startDate && workorderDate <= now;
    });
    
    console.log(`✅ Filtered (${days} days): ${filtered.length} of ${allWorkorderData.length} workorders`);
    return filtered;
}

// Update KPI cards
function updateKPICards(filteredData) {
    console.log('📊 Updating KPI cards with', filteredData.length, 'workorders');
    
    const totalWorkorders = filteredData.length;
    
    // Count by status (case-insensitive for MongoDB data)
    let initiated = 0;
    let inProgress = 0;
    let completed = 0;
    let overdue = 0;
    let totalCost = 0;
    const now = new Date();
    
    filteredData.forEach(wo => {
        const status = (wo.status || 'Hold').toLowerCase();
        
        // Count by status
        if (status === 'hold' || status === 'initiated' || status === 'assigned') {
            initiated++;
        } else if (status === 'in_progress' || status === 'in progress') {
            inProgress++;
        } else if (status === 'complete' || status === 'completed' || status === 'closed') {
            completed++;
        }
        
        // Calculate total cost
        const cost = parseFloat(wo.totalWoCost || wo.estimatedCost || 0);
        totalCost += cost;
        
        // Check if overdue
        if (wo.dueDate) {
            const dueDate = new Date(wo.dueDate);
            if (!isNaN(dueDate.getTime()) && dueDate < now && status !== 'complete' && status !== 'completed' && status !== 'closed') {
                overdue++;
            }
        }
    });
    
    console.log('📊 KPI Values:', {
        total: totalWorkorders,
        initiated,
        inProgress,
        completed,
        overdue,
        totalCost: totalCost.toFixed(2)
    });
    
    // Update card values with null checks
    const totalEl = document.querySelector('[data-kpi="total-workorder"] .kpi-value');
    const initiatedEl = document.querySelector('[data-kpi="initiated-workorder"] .kpi-value');
    const inProgressEl = document.querySelector('[data-kpi="inprogress-workorder"] .kpi-value');
    const completedEl = document.querySelector('[data-kpi="completed-workorder"] .kpi-value');
    
    if (totalEl) totalEl.textContent = String(totalWorkorders).padStart(2, '0');
    if (initiatedEl) initiatedEl.textContent = String(initiated).padStart(2, '0');
    if (inProgressEl) inProgressEl.textContent = String(inProgress).padStart(2, '0');
    if (completedEl) completedEl.textContent = String(completed).padStart(2, '0');
    
    const additionalCards = document.querySelectorAll('.additional-kpi-cards .kpi-card');
    if (additionalCards[0]) {
        const costEl = additionalCards[0].querySelector('.kpi-value');
        if (costEl) costEl.textContent = '$' + totalCost.toFixed(2);
    }
    if (additionalCards[1]) {
        const overdueEl = additionalCards[1].querySelector('.kpi-value');
        if (overdueEl) overdueEl.textContent = String(overdue).padStart(2, '0');
    }
}

// Update all dashboard components
function updateDashboard() {
    console.log('📊 Updating dashboard...');
    console.log('📊 Total workorders:', allWorkorderData.length);
    
    const filteredData = filterDataByDateRange(currentDateRange);
    
    console.log('📊 Filtered workorders:', filteredData.length);
    
    updateKPICards(filteredData);
    updateStatusDistributionChart(filteredData);
    updateDatewiseCharts(filteredData);
    updateStatusPriorityCharts(filteredData);
    updateWorkorderTypeChart(filteredData);
    updateEquipmentChart(filteredData);
    updateCostAnalysisCharts(filteredData);
    
    // Populate equipment dropdown with all data
    populateEquipmentDropdown(allWorkorderData);
    
    console.log('✅ Dashboard updated successfully');
}

// Chart update functions
function updateStatusDistributionChart(data) {
    const statusCounts = {};
    
    console.log('📊 Updating status distribution chart with', data.length, 'workorders');
    
    data.forEach(wo => {
        const status = wo.status || 'Hold';
        statusCounts[status] = (statusCounts[status] || 0) + 1;
    });
    
    console.log('📊 Status counts:', statusCounts);
    
    const labels = Object.keys(statusCounts);
    const values = Object.values(statusCounts);
    
    if (labels.length === 0) {
        updateChart('status-distribution', ['No Data'], [0], ['#e0e0e0']);
    } else {
        updateChart('status-distribution', labels, values,
                    ['#0055A5', '#FFA500', '#10b981', '#DC0D0E', '#6c757d']);
    }
}

function updateDatewiseCharts(data) {
    const dateRange = currentDateRange === 'today' ? 1 : (currentDateRange === 'custom' ? 7 : parseInt(currentDateRange)) || 7;
    const dates = [];
    const counts = {};
    const statusCounts = {};
    
    // Generate date labels
    for (let i = dateRange - 1; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        const dateStr = date.toLocaleDateString('en-GB');
        dates.push(dateStr);
        counts[dateStr] = 0;
        statusCounts[dateStr] = { hold: 0, assigned: 0, inProgress: 0, complete: 0 };
    }
    
    // Count workorders by date
    data.forEach(workorder => {
        const createdAt = workorder.createdAt || workorder.updatedAt;
        if (createdAt) {
            const date = new Date(createdAt);
            if (!isNaN(date.getTime())) {
                const dateStr = date.toLocaleDateString('en-GB');
                if (counts.hasOwnProperty(dateStr)) {
                    counts[dateStr]++;
                    const status = (workorder.status || 'Hold').toLowerCase();
                    if (status === 'hold' || status === 'initiated') {
                        statusCounts[dateStr].hold++;
                    } else if (status === 'assigned') {
                        statusCounts[dateStr].assigned++;
                    } else if (status === 'in_progress' || status === 'in progress') {
                        statusCounts[dateStr].inProgress++;
                    } else if (status === 'complete' || status === 'completed') {
                        statusCounts[dateStr].complete++;
                    }
                }
            }
        }
    });
    
    console.log('📊 Date-wise counts:', {
        dates: dates.length,
        counts,
        statusCounts
    });
    
    // Update count chart
    updateChart('datewise-count', dates, dates.map(d => counts[d]),
                ['#0055A5', '#5899DA', '#DC0D0E', '#945ECF']);
    
    // Update status chart (stacked)
    const ctx = document.getElementById('datewise-status');
    if (!ctx) return;
    
    if (chartInstances['datewise-status']) {
        chartInstances['datewise-status'].destroy();
    }
    
    const hasData = data.length > 0;
    
    if (!hasData) {
        chartInstances['datewise-status'] = new Chart('datewise-status', {
            type: 'bar',
            data: {
                labels: ['No Data'],
                datasets: [{
                    label: 'No Data',
                    backgroundColor: '#e0e0e0',
                    data: [0]
                }]
            },
            options: {
                legend: { display: false },
                scales: {
                    yAxes: [{ ticks: { beginAtZero: true } }]
                }
            }
        });
    } else {
        chartInstances['datewise-status'] = new Chart('datewise-status', {
            type: 'bar',
            data: {
                labels: dates,
                datasets: [
                    {
                        label: 'Hold/Initiated',
                        backgroundColor: '#0055A5',
                        data: dates.map(d => statusCounts[d].hold)
                    },
                    {
                        label: 'Assigned',
                        backgroundColor: '#06b6d4',
                        data: dates.map(d => statusCounts[d].assigned)
                    },
                    {
                        label: 'In Progress',
                        backgroundColor: '#FFA500',
                        data: dates.map(d => statusCounts[d].inProgress)
                    },
                    {
                        label: 'Completed',
                        backgroundColor: '#10b981',
                        data: dates.map(d => statusCounts[d].complete)
                    }
                ]
            },
            options: {
                legend: { display: true, position: 'top' },
                scales: {
                    xAxes: [{ stacked: true }],
                    yAxes: [{ stacked: true, ticks: { beginAtZero: true, precision: 0 } }]
                },
                responsive: true,
                maintainAspectRatio: false
            }
        });
    }
}

function updateStatusPriorityCharts(data) {
    // By Status
    const statuses = {};
    data.forEach(wo => {
        const status = wo.status || 'Hold';
        statuses[status] = (statuses[status] || 0) + 1;
    });
    
    if (Object.keys(statuses).length === 0) {
        updateChart('by-status', ['No Data'], [0], ['#e0e0e0']);
    } else {
        updateChart('by-status', Object.keys(statuses), Object.values(statuses),
                    ['#0055A5', '#06b6d4', '#FFA500', '#10b981', '#DC0D0E']);
    }
    
    // By Priority
    const priorities = {};
    data.forEach(wo => {
        const priority = wo.priority || 'Medium';
        priorities[priority] = (priorities[priority] || 0) + 1;
    });
    
    if (Object.keys(priorities).length === 0) {
        updateChart('by-priority', ['No Data'], [0], ['#e0e0e0']);
    } else {
        updateChart('by-priority', Object.keys(priorities), Object.values(priorities),
                    ['#DC0D0E', '#FFA500', '#10b981', '#0055A5']);
    }
    
    // Priority Distribution by Status
    const ctx = document.getElementById('priority-status');
    if (!ctx) return;
    
    if (chartInstances['priority-status']) {
        chartInstances['priority-status'].destroy();
    }
    
    if (data.length === 0) {
        chartInstances['priority-status'] = new Chart('priority-status', {
            type: 'bar',
            data: {
                labels: ['No Data'],
                datasets: [{
                    label: 'No Data',
                    backgroundColor: '#e0e0e0',
                    data: [0]
                }]
            },
            options: {
                legend: { display: false },
                scales: {
                    yAxes: [{ ticks: { beginAtZero: true } }]
                }
            }
        });
        return;
    }
    
    const priorityStatusData = {
        'High': { hold: 0, assigned: 0, inProgress: 0, complete: 0 },
        'Medium': { hold: 0, assigned: 0, inProgress: 0, complete: 0 },
        'Low': { hold: 0, assigned: 0, inProgress: 0, complete: 0 },
        'Emergency': { hold: 0, assigned: 0, inProgress: 0, complete: 0 }
    };
    
    data.forEach(wo => {
        const priority = wo.priority || 'Medium';
        const status = (wo.status || 'Hold').toLowerCase();
        
        if (priorityStatusData[priority]) {
            if (status === 'hold' || status === 'initiated') {
                priorityStatusData[priority].hold++;
            } else if (status === 'assigned') {
                priorityStatusData[priority].assigned++;
            } else if (status === 'in_progress' || status === 'in progress') {
                priorityStatusData[priority].inProgress++;
            } else if (status === 'complete' || status === 'completed') {
                priorityStatusData[priority].complete++;
            }
        }
    });
    
    chartInstances['priority-status'] = new Chart('priority-status', {
        type: 'bar',
        data: {
            labels: ['Emergency', 'High', 'Medium', 'Low'],
            datasets: [
                {
                    label: 'Hold',
                    backgroundColor: '#0055A5',
                    data: ['Emergency', 'High', 'Medium', 'Low'].map(p => priorityStatusData[p].hold)
                },
                {
                    label: 'Assigned',
                    backgroundColor: '#06b6d4',
                    data: ['Emergency', 'High', 'Medium', 'Low'].map(p => priorityStatusData[p].assigned)
                },
                {
                    label: 'In Progress',
                    backgroundColor: '#FFA500',
                    data: ['Emergency', 'High', 'Medium', 'Low'].map(p => priorityStatusData[p].inProgress)
                },
                {
                    label: 'Completed',
                    backgroundColor: '#10b981',
                    data: ['Emergency', 'High', 'Medium', 'Low'].map(p => priorityStatusData[p].complete)
                }
            ]
        },
        options: {
            legend: { display: true, position: 'top' },
            scales: {
                xAxes: [{ stacked: true }],
                yAxes: [{ stacked: true, ticks: { beginAtZero: true, precision: 0 } }]
            },
            responsive: true,
            maintainAspectRatio: false
        }
    });
}

function updateWorkorderTypeChart(data) {
    const types = {};
    data.forEach(wo => {
        const type = wo.workOrderType || 'General';
        types[type] = (types[type] || 0) + 1;
    });
    
    console.log('📊 Workorder types:', types);
    
    if (Object.keys(types).length === 0) {
        updateChart('by-type', ['No Data'], [0], ['#e0e0e0']);
    } else {
        updateChart('by-type', Object.keys(types), Object.values(types),
                    ['#0055A5', '#945ECF', '#E8743B', '#10b981', '#FFA500', '#DC0D0E']);
    }
}

function updateEquipmentChart(data, selectedEquipment = null) {
    let filteredData = data;
    
    console.log('📊 Updating equipment chart with', data.length, 'workorders');
    
    if (selectedEquipment) {
        filteredData = data.filter(wo => {
            const equipmentName = wo.equipment?.equipmentName || wo.equipmentName || '';
            return equipmentName === selectedEquipment;
        });
        console.log('📊 Filtered to equipment:', selectedEquipment, '=', filteredData.length, 'workorders');
    }
    
    const equipment = {};
    filteredData.forEach(wo => {
        const equipmentName = wo.equipment?.equipmentName || wo.equipmentName || 'Unknown';
        if (equipmentName !== 'Unknown' && equipmentName !== 'N/A') {
            equipment[equipmentName] = (equipment[equipmentName] || 0) + 1;
        }
    });
    
    console.log('📊 Equipment counts:', equipment);
    
    if (Object.keys(equipment).length === 0) {
        updateChart('by-equipment-chart', ['No Data'], [0], ['#e0e0e0']);
    } else {
        updateChart('by-equipment-chart', Object.keys(equipment), Object.values(equipment),
                    ['#0055A5', '#945ECF', '#E8743B', '#10b981']);
    }
}

function updateCostAnalysisCharts(data) {
    console.log('📊 Updating cost analysis charts with', data.length, 'workorders');
    
    // Cost by Status
    const statusCosts = {};
    data.forEach(wo => {
        const status = wo.status || 'Hold';
        const cost = parseFloat(wo.totalWoCost || wo.estimatedCost || 0);
        statusCosts[status] = (statusCosts[status] || 0) + cost;
    });
    
    if (Object.keys(statusCosts).length === 0) {
        updateChart('cost-by-status', ['No Data'], [0], ['#e0e0e0']);
    } else {
        updateChart('cost-by-status', Object.keys(statusCosts), Object.values(statusCosts),
                    ['#0055A5', '#06b6d4', '#FFA500', '#10b981', '#DC0D0E']);
    }
    
    // Cost Breakdown
    const costBreakdown = {
        'Labor': 0,
        'Parts': 0,
        'Additional': 0,
        'Tax': 0
    };
    data.forEach(wo => {
        costBreakdown['Labor'] += parseFloat(wo.totalLaborCost || 0);
        costBreakdown['Parts'] += parseFloat(wo.partMaterialCost || 0);
        costBreakdown['Additional'] += parseFloat(wo.additionalCost || 0);
        costBreakdown['Tax'] += parseFloat(wo.taxAmount || 0);
    });
    
    console.log('📊 Cost breakdown:', costBreakdown);
    
    const hasCostData = Object.values(costBreakdown).some(v => v > 0);
    
    if (!hasCostData) {
        updateChart('cost-breakdown', ['No Cost Data'], [0], ['#e0e0e0']);
    } else {
        updateChart('cost-breakdown', Object.keys(costBreakdown), Object.values(costBreakdown),
                    ['#0055A5', '#945ECF', '#E8743B', '#FFA500']);
    }
    
    // Date wise Cost Trend
    const dateRange = currentDateRange === 'today' ? 1 : (currentDateRange === 'custom' ? 7 : parseInt(currentDateRange)) || 7;
    const dates = [];
    const dateCosts = {};
    
    for (let i = dateRange - 1; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        const dateStr = date.toLocaleDateString('en-GB');
        dates.push(dateStr);
        dateCosts[dateStr] = 0;
    }
    
    data.forEach(wo => {
        const createdAt = wo.createdAt || wo.updatedAt;
        if (createdAt) {
            const date = new Date(createdAt);
            if (!isNaN(date.getTime())) {
                const dateStr = date.toLocaleDateString('en-GB');
                if (dateCosts.hasOwnProperty(dateStr)) {
                    const cost = parseFloat(wo.totalWoCost || wo.estimatedCost || 0);
                    dateCosts[dateStr] += cost;
                }
            }
        }
    });
    
    console.log('📊 Date-wise cost trend:', dateCosts);
    
    updateChart('cost-trend', dates, dates.map(d => dateCosts[d]),
                ['#0055A5', '#945ECF', '#E8743B']);
}

// Generic chart update function
function updateChart(chartId, labels, data, colors) {
    const ctx = document.getElementById(chartId);
    if (!ctx) {
        console.warn('⚠️ Chart canvas not found:', chartId);
        return;
    }
    
    // Destroy existing chart
    if (chartInstances[chartId]) {
        chartInstances[chartId].destroy();
    }
    
    // Handle empty data
    if (!labels || labels.length === 0 || !data || data.length === 0) {
        labels = ['No Data'];
        data = [0];
        colors = ['#e0e0e0'];
    }
    
    // Create new chart
    chartInstances[chartId] = new Chart(chartId, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                backgroundColor: colors,
                data: data
            }]
        },
        options: {
            legend: { display: false },
            scales: {
                yAxes: [{ 
                    ticks: { 
                        beginAtZero: true,
                        precision: 0,
                        stepSize: 1
                    } 
                }]
            },
            responsive: true,
            maintainAspectRatio: false,
            tooltips: {
                enabled: labels[0] !== 'No Data',
                mode: 'index',
                intersect: false
            }
        }
    });
    
    console.log(`✅ Chart '${chartId}' updated with ${labels.length} items`);
}

// Populate equipment dropdown
function populateEquipmentDropdown(data) {
    const equipment = new Set();
    data.forEach(wo => {
        const equipmentName = wo.equipment?.equipmentName || wo.equipmentName;
        if (equipmentName && equipmentName !== 'Unknown' && equipmentName !== 'N/A') {
            equipment.add(equipmentName);
        }
    });
    
    const dropdown = document.getElementById('equipment-filter');
    if (dropdown) {
        dropdown.innerHTML = '<option value="">All Equipment</option>';
        Array.from(equipment).sort().forEach(eq => {
            const option = document.createElement('option');
            option.value = eq;
            option.textContent = eq;
            dropdown.appendChild(option);
        });
    }
}

// Export to Excel functionality
async function exportToExcel() {
    try {
        console.log('📊 Exporting workorders to Excel...');
        
        const filteredData = filterDataByDateRange(currentDateRange);
        
        console.log('📊 Export data count:', filteredData.length);
        
        if (filteredData.length === 0) {
            showNotification('No data to export', 'error');
            return;
        }
        
        showNotification('Preparing Excel export...', 'info');
        
        // Prepare data for Excel
        const excelData = filteredData.map((workorder, index) => {
            const equipment = workorder.equipment || {};
            const location = workorder.location || {};
            
            // Format dates properly
            const createdAt = workorder.createdAt || workorder.updatedAt;
            const formattedCreatedAt = createdAt ? new Date(createdAt).toLocaleString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            }) : 'N/A';
            
            const formattedStartDate = workorder.startDate ? new Date(workorder.startDate).toLocaleDateString('en-GB') : 'N/A';
            const formattedDueDate = workorder.dueDate ? new Date(workorder.dueDate).toLocaleDateString('en-GB') : 'N/A';
            const formattedCompletedDate = workorder.completedDate ? new Date(workorder.completedDate).toLocaleDateString('en-GB') : 'N/A';
            
            // Format assignees
            const assignees = Array.isArray(workorder.assignTo) 
                ? workorder.assignTo.map(a => a.name || a.fullName || '').join(', ') 
                : 'N/A';
            
            return {
                'S.No': index + 1,
                'Workorder ID': workorder.id || workorder._id || 'N/A',
                'Request Number': workorder.requestNumber || 'N/A',
                'Title': workorder.title || 'N/A',
                'Status': workorder.status || 'N/A',
                'Priority': workorder.priority || 'N/A',
                'Type': workorder.workOrderType || 'N/A',
                'Country': location.country || workorder.country || 'N/A',
                'Project': location.project || workorder.project || 'N/A',
                'Equipment Name': equipment.equipmentName || workorder.equipmentName || 'N/A',
                'Equipment Number': equipment.vehicleNumber || equipment.equipmentNumber || 'N/A',
                'Equipment Category': equipment.mainCategory || 'N/A',
                'Model': equipment.model || 'N/A',
                'Meter Reading': equipment.meterReading || 'N/A',
                'Description': workorder.description || 'N/A',
                'Notes': workorder.notes || 'N/A',
                'Assigned To': assignees,
                'Service Schedule ID': workorder.serviceScheduleId || 'N/A',
                'Service Type': workorder.serviceType || 'N/A',
                'Estimated Cost': workorder.estimatedCost || 0,
                'Total Cost': workorder.totalWoCost || 0,
                'Labor Cost': workorder.totalLaborCost || 0,
                'Parts Cost': workorder.partMaterialCost || 0,
                'Additional Cost': workorder.additionalCost || 0,
                'Tax Amount': workorder.taxAmount || 0,
                'Start Date': formattedStartDate,
                'Due Date': formattedDueDate,
                'Completed Date': formattedCompletedDate,
                'Created At': formattedCreatedAt,
                'GPS Address': location.address || 'N/A',
                'Coordinates': location.coordinates || 'N/A'
            };
        });
    
        // Create workbook and worksheet
        const wb = XLSX.utils.book_new();
        const ws = XLSX.utils.json_to_sheet(excelData);
        
        // Set column widths
        const colWidths = [
            { wch: 6 },  { wch: 15 }, { wch: 20 }, { wch: 30 }, { wch: 12 },
            { wch: 12 }, { wch: 20 }, { wch: 12 }, { wch: 15 }, { wch: 20 },
            { wch: 18 }, { wch: 18 }, { wch: 15 }, { wch: 12 }, { wch: 15 },
            { wch: 30 }, { wch: 30 }, { wch: 30 }, { wch: 20 }, { wch: 15 },
            { wch: 15 }, { wch: 15 }, { wch: 15 }, { wch: 15 }, { wch: 15 },
            { wch: 15 }, { wch: 15 }, { wch: 15 }, { wch: 20 }, { wch: 30 },
            { wch: 25 }
        ];
        ws['!cols'] = colWidths;
        
        XLSX.utils.book_append_sheet(wb, ws, 'Workorder Data');
        
        // Create summary sheet
        const totalWorkorders = filteredData.length;
        const totalCost = filteredData.reduce((sum, wo) => sum + parseFloat(wo.totalWoCost || wo.estimatedCost || 0), 0);
        
        const statusCounts = {};
        filteredData.forEach(wo => {
            const status = wo.status || 'Hold';
            statusCounts[status] = (statusCounts[status] || 0) + 1;
        });
        
        const summaryData = [
            { 'Metric': 'Total Workorders', 'Value': totalWorkorders },
            { 'Metric': 'Hold', 'Value': statusCounts['Hold'] || 0 },
            { 'Metric': 'Assigned', 'Value': statusCounts['assigned'] || 0 },
            { 'Metric': 'In Progress', 'Value': statusCounts['in_progress'] || 0 },
            { 'Metric': 'Completed', 'Value': statusCounts['complete'] || 0 },
            { 'Metric': 'Total Cost', 'Value': `$${totalCost.toFixed(2)}` },
            { 'Metric': 'Date Range', 'Value': getCurrentDateRangeText() },
            { 'Metric': 'Export Date', 'Value': new Date().toLocaleString('en-GB', {
                day: '2-digit',
                month: '2-digit',
                year: 'numeric',
                hour: '2-digit',
                minute: '2-digit'
            }) }
        ];
        
        const wsSummary = XLSX.utils.json_to_sheet(summaryData);
        wsSummary['!cols'] = [{ wch: 25 }, { wch: 30 }];
        XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
        
        // Generate filename
        const dateRangeText = getCurrentDateRangeText().replace(/\s+/g, '_');
        const timestamp = new Date().toISOString().slice(0, 10);
        const filename = `Workorder_KPI_${dateRangeText}_${timestamp}.xlsx`;
        
        // Export file
        XLSX.writeFile(wb, filename);
        
        showNotification(`✅ Exported ${totalWorkorders} workorder(s) to Excel successfully!`, 'success');
        console.log('✅ Excel export completed:', filename);
    } catch (error) {
        console.error('❌ Error exporting to Excel:', error);
        showNotification('Error exporting to Excel: ' + error.message, 'error');
    }
}

// Get current date range as text
function getCurrentDateRangeText() {
    if (currentDateRange === 'today') {
        return 'Today';
    } else if (currentDateRange === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            const end = new Date(customEndDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            return `Custom_${start}_to_${end}`;
        }
        return 'All_Data';
    } else {
        return `Last_${currentDateRange}_Days`;
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', () => {
    fetchWorkorderData();
    
    // Export button
    const exportBtnHeader = document.getElementById('export-excel-btn-header');
    if (exportBtnHeader) {
        exportBtnHeader.addEventListener('click', () => {
            exportToExcel();
        });
    }
    
    // Refresh button
    const refreshBtnHeader = document.getElementById('refresh-data-btn-header');
    if (refreshBtnHeader) {
        refreshBtnHeader.addEventListener('click', () => {
            showNotification('Refreshing data...', 'info');
            fetchWorkorderData();
        });
    }
    
    // Custom date range functionality
    const customDateRange = document.getElementById('custom-date-range');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');
    const applyDatesBtn = document.getElementById('apply-custom-dates');
    const clearDatesBtn = document.getElementById('clear-custom-dates');
    
    // Apply custom dates
    if (applyDatesBtn) {
        applyDatesBtn.addEventListener('click', () => {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                showNotification('Please select both start and end dates', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                showNotification('Start date cannot be after end date', 'error');
                return;
            }
            
            customStartDate = startDate;
            customEndDate = endDate;
            currentDateRange = 'custom';
            
            // Update active button
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === 'custom') {
                    btn.classList.add('active');
                }
            });
            
            updateDashboard();
            showNotification(`Showing data from ${new Date(startDate).toLocaleDateString('en-GB')} to ${new Date(endDate).toLocaleDateString('en-GB')}`, 'success');
        });
    }
    
    // Clear custom dates
    if (clearDatesBtn) {
        clearDatesBtn.addEventListener('click', () => {
            startDateInput.value = '';
            endDateInput.value = '';
            customStartDate = null;
            customEndDate = null;
            
            // Switch back to default (7 days)
            currentDateRange = 7;
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === '7days') {
                    btn.classList.add('active');
                }
            });
            
            // Hide custom date range picker
            if (customDateRange) {
                customDateRange.style.display = 'none';
            }
            
            updateDashboard();
            showNotification('Custom date range cleared', 'info');
        });
    }
    
    // Set max date to today
    const today = new Date().toISOString().split('T')[0];
    if (startDateInput) startDateInput.max = today;
    if (endDateInput) endDateInput.max = today;
    
    // When start date changes, update end date minimum
    if (startDateInput && endDateInput) {
        startDateInput.addEventListener('change', () => {
            if (startDateInput.value) {
                endDateInput.min = startDateInput.value;
                if (endDateInput.value && new Date(endDateInput.value) < new Date(startDateInput.value)) {
                    endDateInput.value = '';
                }
            }
        });
    }
    
    // Equipment filter
    const equipmentFilter = document.getElementById('equipment-filter');
    const equipmentGoBtn = document.querySelector('#by-equipment .btn-go');
    if (equipmentGoBtn && equipmentFilter) {
        equipmentGoBtn.addEventListener('click', () => {
            const selectedEquipment = equipmentFilter.value;
            const filteredData = filterDataByDateRange(currentDateRange);
            updateEquipmentChart(filteredData, selectedEquipment || null);
            if (selectedEquipment) {
                showNotification(`Equipment chart filtered: ${selectedEquipment}`, 'success');
            } else {
                showNotification('Showing all equipment', 'info');
            }
        });
    }
});
</script>

<style>
/* CSS Variables - Matching Inspection KPI Design */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Jost', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
    --z-index-modal: 1055;
}

/* MAIN CONTAINER */
.workorder-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
    color: var(--text-primary);
}

/* HEADER STYLES */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.create-btn i {
    font-size: 18px;
}

/* MAIN CONTENT */
.workorder-kpi-content {
    padding: 30px 20px;
    background: var(--bg-light);
    min-height: auto;
    overflow: visible;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    font-family: var(--font-family);
    margin-bottom: 1rem;
    margin: 0;
    overflow: hidden;
}

/* TIME PERIOD SELECTOR */
.time-period-selector {
    margin-bottom: 2rem;
}

.period-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.period-btn {
    padding: 12px 24px;
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.period-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.period-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

/* Custom Date Range Picker */
.custom-date-range {
    margin-top: 1rem;
    padding: 1.5rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.date-range-inputs {
    display: flex;
    align-items: end;
    gap: 1rem;
    flex-wrap: wrap;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.date-input-group label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.date-input {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
    transition: var(--transition-base);
    min-width: 180px;
    cursor: pointer;
}

.date-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-apply-dates,
.btn-clear-dates {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.btn-apply-dates {
    background: var(--primary-color);
    color: white;
}

.btn-apply-dates:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-clear-dates {
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-clear-dates:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

/* MODERN TABS */
.modern-tabs {
    margin-bottom: 2rem;
}

.tab-navigation {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 0;
    margin-bottom: 2rem;
    overflow-x: auto;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: transparent;
    color: var(--text-secondary);
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    position: relative;
    border-bottom: 2px solid transparent;
}

.tab-btn:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-btn.active {
    color: var(--primary-color);
    background: var(--bg-light);
    border-bottom-color: var(--primary-color);
}

/* TAB CONTENT */
.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

/* KPI CARDS */
.kpi-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.kpi-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 1rem;
}

.kpi-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.kpi-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--bg-white);
    flex-shrink: 0;
}

.kpi-icon.success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}

.kpi-icon.warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
}

.kpi-icon.info {
    background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%);
}

.kpi-icon.secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
}

.kpi-icon.danger {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
}

.kpi-content {
    flex: 1;
}

.kpi-content h3 {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0 0 8px 0;
}

.kpi-value {
    font-size: 32px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0 0 4px 0;
    line-height: 1;
}

.kpi-label {
    font-size: 12px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* OVERVIEW BOTTOM SECTION */
.overview-bottom-section {
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 1.5rem;
    margin-top: 1.5rem;
}

.additional-kpi-cards {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.chart-section {
    display: flex;
    flex-direction: column;
}

/* CHARTS SECTION */
.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.chart-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
    border-color: var(--primary-color);
}

.chart-header {
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 1px solid var(--border-light);
}

.chart-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.chart-container {
    position: relative;
    height: 250px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chart-container canvas {
    max-width: 100%;
    max-height: 100%;
}

/* FILTER SECTION */
.filter-section {
    margin-bottom: 2rem;
}

.filter-form {
    display: flex;
    align-items: end;
    gap: 1rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
}

.form-group {
    flex: 1;
    display: flex;
    flex-direction: column;
    margin-bottom: 0;
}

.form-group label {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
    font-size: 14px;
}

.form-select {
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    padding: 12px 16px;
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.form-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-go {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    white-space: nowrap;
    height: fit-content;
    min-height: 44px;
}

.btn-go:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

/* LOADING SPINNER */
@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Loading/Sync Modal */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    font-size: 40px;
    animation: spin 1s linear infinite;
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

/* RESPONSIVE DESIGN */
@media (max-width: 992px) {
    .overview-bottom-section {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .header-right {
        gap: 10px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .workorder-kpi-content {
        padding: 20px 15px;
    }
    
    .content-box {
        padding: 20px;
    }
    
    .period-buttons {
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
    }
    
    .kpi-cards-grid {
        grid-template-columns: 1fr;
    }
    
    .charts-grid {
        grid-template-columns: 1fr;
    }
    
    .tab-navigation {
        flex-direction: column;
        gap: 0;
    }
    
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }
    
    .btn-go {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .header-right {
        width: 100%;
        flex-direction: column;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
    
    .content-box {
        padding: 14px;
    }
}

/* Notification Styles */
.notification {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: var(--z-index-notification);
    padding: 16px;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    font-family: var(--font-family);
    font-weight: 500;
    max-width: 400px;
    min-width: 300px;
    animation: slideIn 0.3s ease;
    background: white;
}

.notification-success {
    border-left: 4px solid #28a745;
}

.notification-error {
    border-left: 4px solid #dc3545;
}

.notification-info {
    border-left: 4px solid #17a2b8;
}

.notification-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 1rem;
}

.notification-content span {
    color: #333;
    font-size: 14px;
}

.notification-close {
    background: none;
    border: none;
    font-size: 18px;
    color: #999;
    cursor: pointer;
    padding: 0;
    line-height: 1;
    margin-left: 12px;
}

.notification-close:hover {
    color: #666;
}

@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

/* ACCESSIBILITY */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* FOCUS STYLES */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}
</style>

<script>
// Optimized Workorder KPI Functionality
(function() {
    'use strict';
    
    // Configuration
    const CONFIG = {
        notificationTimeout: 5000,
        selectors: {
            tabButtons: '.tab-btn',
            tabPanes: '.tab-pane',
            periodButtons: '.period-btn',
            kpiCards: '.kpi-card'
        }
    };

    // Cached DOM elements
    let elements = {};
    
    // Initialize DOM elements
    function initElements() {
        elements = {
            tabButtons: document.querySelectorAll(CONFIG.selectors.tabButtons),
            tabPanes: document.querySelectorAll(CONFIG.selectors.tabPanes),
            periodButtons: document.querySelectorAll(CONFIG.selectors.periodButtons),
            kpiCards: document.querySelectorAll(CONFIG.selectors.kpiCards)
        };
    }

    // Tab switching functionality
    function initTabSwitching() {
        elements.tabButtons.forEach(button => {
            button.addEventListener('click', function() {
                const targetTab = this.getAttribute('data-tab');

                // Remove active class from all tabs and panes
                elements.tabButtons.forEach(btn => btn.classList.remove('active'));
                elements.tabPanes.forEach(pane => pane.classList.remove('active'));

                // Add active class to clicked tab and corresponding pane
                this.classList.add('active');
                const targetPane = document.getElementById(targetTab);
                if (targetPane) {
                    targetPane.classList.add('active');
                }
            });
        });
    }

    // Time period selector functionality
    function initPeriodSelector() {
        const customDateRange = document.getElementById('custom-date-range');
        
        elements.periodButtons.forEach(button => {
            button.addEventListener('click', function() {
                // Remove active class from all period buttons
                elements.periodButtons.forEach(btn => btn.classList.remove('active'));
                
                // Add active class to clicked button
                this.classList.add('active');
                
                const period = this.getAttribute('data-period');
                console.log('Period changed to:', period);
                
                // Show/hide custom date range picker
                if (period === 'custom') {
                    if (customDateRange) {
                        customDateRange.style.display = 'block';
                    }
                    currentDateRange = 'custom';
                    return;
                } else {
                    if (customDateRange) {
                        customDateRange.style.display = 'none';
                    }
                    
                    // Update current date range
                    if (period === 'today') {
                        currentDateRange = 'today';
                    } else if (period === '7days') {
                        currentDateRange = 7;
                    } else if (period === '30days') {
                        currentDateRange = 30;
                    } else if (period === '90days') {
                        currentDateRange = 90;
                    }
                    
                    updateDashboard();
                }
            });
        });
    }

    // Notification system
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span>${message}</span>
                <button class="notification-close" aria-label="Close notification">&times;</button>
            </div>
        `;
        
        document.body.appendChild(notification);
        
        // Auto remove after timeout
        setTimeout(() => removeNotification(notification), CONFIG.notificationTimeout);
        
        // Close button functionality
        notification.querySelector('.notification-close').addEventListener('click', () => {
            removeNotification(notification);
        });
    }
    
    function removeNotification(notification) {
        if (notification && notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }
    
    // Make showNotification available globally
    window.showNotification = showNotification;

    // Initialize everything
    function init() {
        initElements();
        initTabSwitching();
        initPeriodSelector();
    }

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
</script>
@endsection

