@extends('welcome')

@section('title', 'Team KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="team-kpi-container">
    <!-- Loading Overlay - Matching Inspection Design -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Team Data...</div>
        </div>
    </div>
    
    <!-- Header - Matching Inspection Design -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url()->previous() }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Team & Employee KPI Dashboard</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="exportExcelBtn">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                </button>
                <button type="button" class="create-btn" id="refreshDataBtn">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="team-kpi-content">
        <div class="content-box">
            <!-- Time Period Selector -->
            <div class="time-period-selector">
                <div class="period-buttons">
                    <button class="period-btn" data-period="today">Today</button>
                    <button class="period-btn active" data-period="7days">Last 7 Days</button>
                    <button class="period-btn" data-period="30days">Last 30 Days</button>
                    <button class="period-btn" data-period="90days">Last 90 Days</button>
                    <button class="period-btn" data-period="custom">Custom Date</button>
                </div>
                
                <!-- Custom Date Range Picker -->
                <div class="custom-date-range" id="custom-date-range" style="display: none;">
                    <div class="date-range-inputs">
                        <div class="date-input-group">
                            <label for="start-date">Start Date:</label>
                            <input type="date" id="start-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label for="end-date">End Date:</label>
                            <input type="date" id="end-date" class="date-input">
                        </div>
                        <button class="btn-apply-dates" id="apply-custom-dates">Apply</button>
                        <button class="btn-clear-dates" id="clear-custom-dates">Clear</button>
                    </div>
                </div>
            </div>
            
            <!-- Modern Tabs -->
            <div class="modern-tabs">
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="overview">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                        Overview
                    </button>
                    <button class="tab-btn" data-tab="by-role">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                            <circle cx="9" cy="7" r="4"/>
                            <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                            <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                        </svg>
                        By Role
                    </button>
                    <button class="tab-btn" data-tab="by-country">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <circle cx="12" cy="12" r="10"/>
                            <line x1="2" y1="12" x2="22" y2="12"/>
                            <path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"/>
                        </svg>
                        By Country
                    </button>
                    <button class="tab-btn" data-tab="by-project">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M3 3h18v18H3zM9 9h6v6H9z"/>
                        </svg>
                        By Project
                    </button>
                    <button class="tab-btn" data-tab="details">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <line x1="8" y1="6" x2="21" y2="6"/>
                            <line x1="8" y1="12" x2="21" y2="12"/>
                            <line x1="8" y1="18" x2="21" y2="18"/>
                            <line x1="3" y1="6" x2="3.01" y2="6"/>
                            <line x1="3" y1="12" x2="3.01" y2="12"/>
                            <line x1="3" y1="18" x2="3.01" y2="18"/>
                        </svg>
                        Employee Details
                    </button>
                </div>

                <!-- Tab Content -->
                <div class="tab-content">
                    <!-- Overview Tab -->
                    <div id="overview" class="tab-pane active">
                        <div class="kpi-cards-grid">
                            <div class="kpi-card" data-kpi="total-users">
                                <div class="kpi-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M17 21v-2a4 4 0 0 0-4-4H5a4 4 0 0 0-4 4v2"/>
                                        <circle cx="9" cy="7" r="4"/>
                                        <path d="M23 21v-2a4 4 0 0 0-3-3.87"/>
                                        <path d="M16 3.13a4 4 0 0 1 0 7.75"/>
                                    </svg>
                                </div>
                                <div class="kpi-details">
                                    <h3 class="kpi-title">Total Team Members</h3>
                                    <p class="kpi-value" id="total-users-count">0</p>
                                    <p class="kpi-subtitle">All registered users</p>
                                </div>
                            </div>

                            <div class="kpi-card" data-kpi="active-users">
                                <div class="kpi-icon" style="background: linear-gradient(135deg, #10b981 0%, #059669 100%);">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"/>
                                        <polyline points="22 4 12 14.01 9 11.01"/>
                                    </svg>
                                </div>
                                <div class="kpi-details">
                                    <h3 class="kpi-title">Online/Active</h3>
                                    <p class="kpi-value" id="active-users-count">0</p>
                                    <p class="kpi-subtitle">Logged in users</p>
                                </div>
                            </div>

                            <div class="kpi-card" data-kpi="total-countries">
                                <div class="kpi-icon" style="background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"/>
                                        <line x1="2" y1="12" x2="22" y2="12"/>
                                        <path d="M12 2a15.3 15.3 0 0 1 4 10 15.3 15.3 0 0 1-4 10 15.3 15.3 0 0 1-4-10 15.3 15.3 0 0 1 4-10z"/>
                                    </svg>
                                </div>
                                <div class="kpi-details">
                                    <h3 class="kpi-title">Countries</h3>
                                    <p class="kpi-value" id="countries-count">0</p>
                                    <p class="kpi-subtitle">Operating locations</p>
                                </div>
                            </div>

                            <div class="kpi-card" data-kpi="total-projects">
                                <div class="kpi-icon" style="background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M3 3h18v18H3zM9 9h6v6H9z"/>
                                    </svg>
                                </div>
                                <div class="kpi-details">
                                    <h3 class="kpi-title">Projects</h3>
                                    <p class="kpi-value" id="projects-count">0</p>
                                    <p class="kpi-subtitle">Active projects</p>
                                </div>
                            </div>
                        </div>

                        <!-- Charts Section -->
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Team Members by Role</h3>
                                </div>
                                <div class="chart-body">
                                    <canvas id="roleChart"></canvas>
                                </div>
                            </div>

                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Team Members by Country</h3>
                                </div>
                                <div class="chart-body">
                                    <canvas id="countryChart"></canvas>
                                </div>
                            </div>
                        </div>

                        <!-- Activity Card -->
                        <div class="activity-card">
                            <div class="activity-header">
                                <h3>Recent Activity</h3>
                                <span class="activity-count" id="recent-activity-count">0 users active recently</span>
                            </div>
                            <div class="activity-info">
                                <span>Users created/updated in selected period</span>
                            </div>
                        </div>
                    </div>

                    <!-- By Role Tab -->
                    <div id="by-role" class="tab-pane">
                        <div class="role-stats-grid" id="role-stats-grid">
                            <!-- Will be populated dynamically -->
                        </div>
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Role Distribution</h3>
                            </div>
                            <div class="chart-body">
                                <canvas id="roleDistributionChart"></canvas>
                            </div>
                        </div>
                        <div class="data-table-container">
                            <h3>Team Members by Role</h3>
                            <table class="data-table" id="role-table">
                                <thead>
                                    <tr>
                                        <th>Role</th>
                                        <th>Count</th>
                                        <th>Percentage</th>
                                    </tr>
                                </thead>
                                <tbody id="role-table-body">
                                    <!-- Will be populated dynamically -->
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- By Country Tab -->
                    <div id="by-country" class="tab-pane">
                        <div class="country-stats-grid" id="country-stats-grid">
                            <!-- Will be populated dynamically -->
                        </div>
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Country Distribution</h3>
                            </div>
                            <div class="chart-body">
                                <canvas id="countryDistributionChart"></canvas>
                            </div>
                        </div>
                        <div class="data-table-container">
                            <h3>Team Members by Country</h3>
                            <table class="data-table" id="country-table">
                                <thead>
                                    <tr>
                                        <th>Country</th>
                                        <th>Count</th>
                                        <th>Percentage</th>
                                    </tr>
                                </thead>
                                <tbody id="country-table-body">
                                    <!-- Will be populated dynamically -->
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- By Project Tab -->
                    <div id="by-project" class="tab-pane">
                        <div class="project-stats-grid" id="project-stats-grid">
                            <!-- Will be populated dynamically -->
                        </div>
                        <div class="chart-card">
                            <div class="chart-header">
                                <h3>Project Distribution</h3>
                            </div>
                            <div class="chart-body">
                                <canvas id="projectDistributionChart"></canvas>
                            </div>
                        </div>
                        <div class="data-table-container">
                            <h3>Team Members by Project</h3>
                            <table class="data-table" id="project-table">
                                <thead>
                                    <tr>
                                        <th>Project</th>
                                        <th>Count</th>
                                        <th>Percentage</th>
                                    </tr>
                                </thead>
                                <tbody id="project-table-body">
                                    <!-- Will be populated dynamically -->
                                </tbody>
                            </table>
                        </div>
                    </div>

                    <!-- Employee Details Tab -->
                    <div id="details" class="tab-pane">
                        <div class="table-controls">
                            <div class="search-box">
                                <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" id="user-search" placeholder="Search users...">
                            </div>
                            <div class="filter-controls">
                                <select id="role-filter" class="filter-select">
                                    <option value="">All Roles</option>
                                </select>
                                <select id="country-filter" class="filter-select">
                                    <option value="">All Countries</option>
                                </select>
                                <select id="project-filter" class="filter-select">
                                    <option value="">All Projects</option>
                                </select>
                            </div>
                        </div>
                        <div class="data-table-container">
                            <table class="data-table" id="user-details-table">
                                <thead>
                                    <tr>
                                        <th>Name</th>
                                        <th>Email</th>
                                        <th>Role</th>
                                        <th>Country</th>
                                        <th>Project</th>
                                        <th>Phone</th>
                                        <th>Created At</th>
                                    </tr>
                                </thead>
                                <tbody id="user-details-body">
                                    <!-- Will be populated dynamically -->
                                </tbody>
                            </table>
                        </div>
                        <div class="table-pagination">
                            <button class="pagination-btn" id="prev-page" disabled>Previous</button>
                            <span class="pagination-info">
                                Page <span id="current-page">1</span> of <span id="total-pages">1</span>
                            </span>
                            <button class="pagination-btn" id="next-page">Next</button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<style>
/* CSS Variables for consistent theming - Matching Inspection KPI */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
    --z-index-modal: 1055;
}

/* ===== MAIN CONTAINER ===== */
.team-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
    color: var(--text-primary);
}

/* ===== HEADER STYLES - Matching Inspection Design ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.create-btn i {
    font-size: 18px;
}

/* ===== MAIN CONTENT ===== */
.team-kpi-content {
    padding: 30px 20px;
    background: var(--bg-light);
    min-height: auto;
    overflow: visible;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    font-family: var(--font-family);
    margin-bottom: 1rem;
    margin: 0;
    overflow: hidden;
}

/* Loading/Sync Modal - Matching Inspection Design */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    font-size: 40px;
    animation: spin 1s linear infinite;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

/* ===== TIME PERIOD SELECTOR ===== */
.time-period-selector {
    margin-bottom: 2rem;
}

.period-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.period-btn {
    padding: 12px 24px;
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.period-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.period-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

/* Custom Date Range Picker */
.custom-date-range {
    margin-top: 1rem;
    padding: 1.5rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.date-range-inputs {
    display: flex;
    align-items: end;
    gap: 1rem;
    flex-wrap: wrap;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.date-input-group label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.date-input {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
    transition: var(--transition-base);
    min-width: 180px;
    cursor: pointer;
}

.date-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-apply-dates,
.btn-clear-dates {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.btn-apply-dates {
    background: var(--primary-color);
    color: white;
}

.btn-apply-dates:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-clear-dates {
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-clear-dates:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

/* ===== MODERN TABS ===== */
.modern-tabs {
    margin-bottom: 2rem;
}

.tab-navigation {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 0;
    margin-bottom: 2rem;
    overflow-x: auto;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: transparent;
    color: var(--text-secondary);
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    position: relative;
    border-bottom: 2px solid transparent;
}

.tab-btn:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-btn.active {
    color: var(--primary-color);
    background: var(--bg-light);
    border-bottom-color: var(--primary-color);
}

.tab-btn.active::after {
    content: none;
}

/* ===== TAB CONTENT ===== */
.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

/* ===== KPI CARDS ===== */
.kpi-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.kpi-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 1rem;
}

.kpi-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.kpi-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--bg-white);
    flex-shrink: 0;
}

.kpi-icon svg {
    stroke: white;
}

.kpi-details {
    flex: 1;
}

.kpi-title {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0 0 8px 0;
}

.kpi-value {
    font-size: 32px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0 0 4px 0;
    line-height: 1;
}

.kpi-subtitle {
    font-size: 12px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* ===== CHARTS SECTION ===== */
.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.chart-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
    border-color: var(--primary-color);
}

.chart-header {
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 1px solid var(--border-light);
}

.chart-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.chart-body {
    position: relative;
    height: 250px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chart-body canvas {
    max-width: 100%;
    max-height: 100%;
}

/* ===== ACTIVITY CARD ===== */
.activity-card {
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-lg);
    padding: 2rem;
    color: white;
    margin-bottom: 2rem;
    box-shadow: var(--shadow-md);
}

.activity-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1rem;
}

.activity-header h3 {
    font-size: 20px;
    font-weight: 600;
    margin: 0;
}

.activity-count {
    font-size: 24px;
    font-weight: 700;
}

.activity-info {
    font-size: 14px;
    opacity: 0.9;
}

/* ===== STATS GRID ===== */
.role-stats-grid,
.country-stats-grid,
.project-stats-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.stat-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    text-align: center;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.stat-card:hover {
    border-color: var(--primary-color);
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
}

.stat-card h4 {
    font-size: 14px;
    color: var(--text-secondary);
    margin: 0 0 10px 0;
    font-weight: 500;
}

.stat-card .stat-value {
    font-size: 36px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0;
}

/* ===== DATA TABLES ===== */
.data-table-container {
    margin-bottom: 2rem;
    overflow-x: auto;
    border-radius: var(--radius-lg);
    border: 1px solid var(--border-light);
}

.data-table-container h3 {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0 0 1rem 0;
}

.data-table {
    width: 100%;
    border-collapse: collapse;
    background: white;
    border-radius: var(--radius-lg);
    overflow: hidden;
}

.data-table thead {
    background: var(--primary-color);
    color: white;
}

.data-table th {
    padding: 1rem;
    text-align: left;
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
}

.data-table tbody tr {
    border-bottom: 1px solid var(--border-light);
    transition: var(--transition-base);
}

.data-table tbody tr:hover {
    background: var(--bg-light);
}

.data-table tbody tr:last-child {
    border-bottom: none;
}

.data-table td {
    padding: 1rem;
    font-size: 14px;
    color: var(--text-secondary);
}

/* ===== TABLE CONTROLS ===== */
.table-controls {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
    gap: 1rem;
    flex-wrap: wrap;
}

.search-box {
    display: flex;
    align-items: center;
    gap: 10px;
    padding: 12px 16px;
    background: white;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    flex: 1;
    min-width: 250px;
    transition: var(--transition-base);
}

.search-box:focus-within {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.search-box svg {
    stroke: var(--text-muted);
}

.search-box input {
    flex: 1;
    border: none;
    outline: none;
    font-size: 14px;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.filter-controls {
    display: flex;
    gap: 12px;
    flex-wrap: wrap;
}

.filter-select {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    color: var(--text-secondary);
    background: white;
    cursor: pointer;
    outline: none;
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.filter-select:hover,
.filter-select:focus {
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

/* ===== PAGINATION ===== */
.table-pagination {
    display: flex;
    justify-content: center;
    align-items: center;
    gap: 1rem;
    padding: 1.5rem 0;
}

.pagination-btn {
    padding: 10px 20px;
    border: 1px solid var(--border-color);
    background: white;
    color: var(--text-secondary);
    border-radius: var(--radius-md);
    cursor: pointer;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
}

.pagination-btn:hover:not(:disabled) {
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.pagination-btn:disabled {
    opacity: 0.5;
    cursor: not-allowed;
}

.pagination-info {
    font-size: 14px;
    color: var(--text-secondary);
    font-weight: 500;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 992px) {
    .kpi-cards-grid {
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 16px;
    }
    
    .charts-grid {
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 16px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .header-right {
        gap: 10px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .team-kpi-content {
        padding: 20px 15px;
    }
    
    .content-box {
        padding: 20px;
    }
    
    .time-period-selector {
        margin-bottom: 1.5rem;
    }
    
    .period-buttons {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 8px;
    }
    
    .period-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
        text-align: center;
    }
    
    .custom-date-range {
        padding: 1rem;
    }
    
    .date-range-inputs {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .date-input-group {
        width: 100%;
    }
    
    .date-input {
        width: 100%;
        min-width: auto;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        width: 100%;
        justify-content: center;
    }
    
    .tab-navigation {
        flex-direction: column;
        gap: 0;
        border-bottom: 1px solid var(--border-color);
    }
    
    .tab-btn {
        justify-content: flex-start;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 44px;
        border-bottom: 1px solid var(--border-color);
    }
    
    .kpi-cards-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .kpi-card {
        padding: 16px;
    }
    
    .charts-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .chart-card {
        padding: 16px;
    }
    
    .chart-body {
        height: 200px;
    }
    
    .table-controls {
        flex-direction: column;
        align-items: stretch;
    }

    .search-box {
        width: 100%;
    }

    .filter-controls {
        width: 100%;
        flex-direction: column;
    }

    .filter-select {
        width: 100%;
    }
}

@media (max-width: 576px) {
    .header-title {
        font-size: 20px;
    }
    
    .header-right {
        flex-direction: column;
        gap: 8px;
        align-items: stretch;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
        padding: 10px 12px;
        font-size: 13px;
    }
    
    .content-box {
        padding: 14px;
    }
    
    .period-buttons {
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
        gap: 6px;
    }
    
    .period-btn {
        padding: 8px 10px;
        font-size: 12px;
        min-height: 36px;
    }
    
    .custom-date-range {
        padding: 0.875rem;
    }
    
    .date-range-inputs {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .date-input-group {
        width: 100%;
    }
    
    .date-input {
        width: 100%;
        min-width: auto;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .header-title {
        font-size: 18px;
    }
    
    .back-button {
        font-size: 18px;
    }
    
    .header-right {
        width: 100%;
    }
    
    .create-btn {
        font-size: 12px;
        padding: 9px 12px;
    }
    
    .content-box {
        padding: 12px;
    }
    
    .period-buttons {
        grid-template-columns: 1fr;
        gap: 6px;
    }
    
    .custom-date-range {
        padding: 0.75rem;
    }
    
    .date-input {
        padding: 9px;
        font-size: 13px;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        padding: 9px 12px;
        font-size: 12px;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* ===== PRINT STYLES ===== */
@media print {
    .header-gradient,
    .period-buttons,
    .tab-navigation {
        display: none !important;
    }
    
    .team-kpi-container {
        background: white;
    }
    
    .kpi-card,
    .chart-card {
        border: 1px solid #000;
        break-inside: avoid;
    }
}
</style>

<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/xlsx@0.18.5/dist/xlsx.full.min.js"></script>
<script>
let userData = [];
let allUserData = []; // Store all data without time filter
let filteredData = [];
let currentPage = 1;
const itemsPerPage = 10;
let currentTimePeriod = '7days'; // Default time period
let customStartDate = null;
let customEndDate = null;

// MongoDB Backend API Configuration
const API_BASE_URL = 'http://168.231.113.219:5003';

// Date formatting helper - Day/Month/Year with time
function formatDate(dateValue) {
    if (!dateValue) return 'N/A';
    
    try {
        const date = dateValue instanceof Date ? dateValue : new Date(dateValue);
        if (isNaN(date.getTime())) return 'N/A';
        
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        const hours = String(date.getHours()).padStart(2, '0');
        const minutes = String(date.getMinutes()).padStart(2, '0');
        const seconds = String(date.getSeconds()).padStart(2, '0');
        
        return `${day}/${month}/${year} ${hours}:${minutes}:${seconds}`;
    } catch (e) {
        return 'N/A';
    }
}

// Date formatting helper - Day/Month/Year only (no time)
function formatDateOnly(dateValue) {
    if (!dateValue) return 'N/A';
    
    try {
        const date = dateValue instanceof Date ? dateValue : new Date(dateValue);
        if (isNaN(date.getTime())) return 'N/A';
        
        const day = String(date.getDate()).padStart(2, '0');
        const month = String(date.getMonth() + 1).padStart(2, '0');
        const year = date.getFullYear();
        
        return `${day}/${month}/${year}`;
    } catch (e) {
        return 'N/A';
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', function() {
    initializeDashboard();
});

async function initializeDashboard() {
    showLoading();
    await fetchUserData();
    hideLoading();
    updateDashboard();
    setupEventListeners();
}

function showLoading() {
    document.getElementById('loading-overlay').style.display = 'flex';
}

function hideLoading() {
    document.getElementById('loading-overlay').style.display = 'none';
}

// Fetch user data from MongoDB Backend API
async function fetchUserData() {
    try {
        const response = await fetch(`${API_BASE_URL}/users/all`);
        
        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }
        
        const result = await response.json();
        
        if (!result || !result.success || !result.data) {
            console.warn('No user data found');
            allUserData = [];
            userData = [];
            return;
        }
        
        const data = result.data;
        
        // Map MongoDB data structure to KPI format
        allUserData = data.map((value) => {
            // Handle date fields - convert to Date objects or ISO strings
            const parseDate = (dateValue) => {
                if (!dateValue) return null;
                if (dateValue instanceof Date) return dateValue;
                if (typeof dateValue === 'string') return new Date(dateValue);
                if (dateValue.$date) return new Date(dateValue.$date);
                return null;
            };
            
            return {
                _id: value._id || value.uid || 'unknown',
                uid: value.uid || value._id || 'unknown',
                employeeNumber: value.employeeNumber || value._id || 'N/A',
                name: value.name || value.email?.split('@')[0] || 'Unknown',
                username: value.username || value.name || value.email?.split('@')[0] || 'Unknown',
                email: value.email || 'N/A',
                phoneNumber: value.mobile || value.phoneNumber || value.phone || 'N/A',
                mobile: value.mobile || value.phoneNumber || value.phone || 'N/A',
                role: value.role || value.department || 'TeamMember',
                department: value.department || value.role || 'N/A',
                country: value.country || (Array.isArray(value.countries) && value.countries.length > 0 ? value.countries[0] : 'N/A'),
                project: value.project || (Array.isArray(value.projects) && value.projects.length > 0 ? value.projects[0] : 'N/A'),
                countries: Array.isArray(value.countries) ? value.countries : (value.country ? [value.country] : []),
                projects: Array.isArray(value.projects) ? value.projects : (value.project ? [value.project] : []),
                loginStatus: value.loginStatus !== undefined ? Boolean(value.loginStatus) : false,
                isActive: value.isActive !== undefined ? Boolean(value.isActive) : true,
                engagementDate: parseDate(value.engagementDate),
                dateExamined: parseDate(value.dateExamined),
                expiryMedicals: parseDate(value.expiryMedicals),
                driverLicenseNumber: value.driverLicenseNumber || 'N/A',
                driverLicenseCode: value.driverLicenseCode || 'N/A',
                nrcIdNumber: value.nrcIdNumber || 'N/A',
                levelFirstAid: value.levelFirstAid || 'N/A',
                clientInduction: value.clientInduction || 'N/A',
                modules: Array.isArray(value.modules) ? value.modules : [],
                notifications: value.notifications || {},
                createdAt: parseDate(value.createdAt) || parseDate(value.uploadTimestamp) || parseDate(value.engagementDate),
                updatedAt: parseDate(value.updatedAt) || parseDate(value.uploadTimestamp),
                lastLogin: parseDate(value.lastLogin),
                profileImage: value.profileImage || value.profilePic || null,
                uploadTimestamp: parseDate(value.uploadTimestamp),
                isBlocked: value.isBlocked || false
            };
        });
        
        // Sort by uploadTimestamp or createdAt (newest first)
        allUserData.sort((a, b) => {
            const dateA = a.uploadTimestamp || a.createdAt || new Date(0);
            const dateB = b.uploadTimestamp || b.createdAt || new Date(0);
            return dateB - dateA;
        });
        
        // Apply initial time filter
        applyTimeFilter(currentTimePeriod);
        
        console.log('User data loaded:', allUserData.length, 'total users');
        console.log('Filtered data:', userData.length, 'users for', currentTimePeriod);
        
        if (allUserData.length === 0) {
            showNotification('No user data found', 'info');
        } else {
            showNotification(`Loaded ${allUserData.length} team member(s)`, 'success');
        }
        
    } catch (error) {
        console.error('Error fetching user data:', error);
        showNotification('Failed to load user data. Please check if the backend server is running.', 'error');
        allUserData = [];
        userData = [];
        filteredData = [];
    }
}

// Apply time period filter
function applyTimeFilter(period) {
    const now = new Date();
    let startDate = null;
    
    // Helper to get user date for filtering
    const getUserDate = (user) => {
        // Try uploadTimestamp first (most reliable), then createdAt, then updatedAt, then lastLogin
        const dateValue = user.uploadTimestamp || user.createdAt || user.updatedAt || user.lastLogin || user.engagementDate;
        if (!dateValue) return null;
        
        if (dateValue instanceof Date) {
            return dateValue;
        }
        
        const date = new Date(dateValue);
        return isNaN(date.getTime()) ? null : date;
    };
    
    // Handle custom date range
    if (period === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate);
            start.setHours(0, 0, 0, 0);
            const end = new Date(customEndDate);
            end.setHours(23, 59, 59, 999);
            
            userData = allUserData.filter(user => {
                const userDate = getUserDate(user);
                if (!userDate) return false;
                return userDate >= start && userDate <= end;
            });
            
            filteredData = [...userData];
            console.log('Custom date filter applied. Items:', userData.length);
            return;
        }
        // If no custom dates selected, return all data
        userData = [...allUserData];
        filteredData = [...userData];
        return;
    }
    
    switch(period) {
        case 'today':
            startDate = new Date(now.getFullYear(), now.getMonth(), now.getDate());
            break;
        case '7days':
            startDate = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
            break;
        case '30days':
            startDate = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
            break;
        case '90days':
            startDate = new Date(now.getTime() - 90 * 24 * 60 * 60 * 1000);
            break;
    }
    
    // Filter users based on uploadTimestamp, createdAt, updatedAt, or lastLogin
    if (startDate) {
        userData = allUserData.filter(user => {
            const userDate = getUserDate(user);
            if (!userDate) return false;
            return userDate >= startDate && userDate <= now;
        });
    } else {
        userData = [...allUserData];
    }
    
    filteredData = [...userData];
    console.log('Time filter applied:', period, 'Users:', userData.length);
}

// Update entire dashboard
function updateDashboard() {
    updateOverviewTab();
    updateRoleTab();
    updateCountryTab();
    updateProjectTab();
    updateDetailsTab();
}

// Update Overview Tab
function updateOverviewTab() {
    const total = userData.length;
    const totalAll = allUserData.length; // Show total from all data
    const active = userData.filter(u => u.loginStatus === true || u.isActive === true).length;
    
    // Count unique countries (from both country field and countries array)
    const uniqueCountries = new Set();
    userData.forEach(u => {
        if (u.country && u.country !== 'N/A') {
            uniqueCountries.add(u.country);
        }
        if (Array.isArray(u.countries) && u.countries.length > 0) {
            u.countries.forEach(c => {
                if (c && c !== 'N/A') uniqueCountries.add(c);
            });
        }
    });
    
    // Count unique projects (from both project field and projects array)
    const uniqueProjects = new Set();
    userData.forEach(u => {
        if (u.project && u.project !== 'N/A') {
            uniqueProjects.add(u.project);
        }
        if (Array.isArray(u.projects) && u.projects.length > 0) {
            u.projects.forEach(p => {
                if (p && p !== 'N/A') uniqueProjects.add(p);
            });
        }
    });
    
    document.getElementById('total-users-count').textContent = total;
    document.getElementById('active-users-count').textContent = active;
    document.getElementById('countries-count').textContent = uniqueCountries.size;
    document.getElementById('projects-count').textContent = uniqueProjects.size;
    document.getElementById('recent-activity-count').textContent = `${total} of ${totalAll} users in selected period`;
    
    // Update charts
    updateRoleChart();
    updateCountryChart();
}

// Update Role Chart
function updateRoleChart() {
    const roleCount = {};
    userData.forEach(u => {
        const role = u.role || 'User';
        roleCount[role] = (roleCount[role] || 0) + 1;
    });
    
    const ctx = document.getElementById('roleChart');
    
    if (window.roleChartInstance) {
        window.roleChartInstance.destroy();
    }
    
    window.roleChartInstance = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: Object.keys(roleCount),
            datasets: [{
                label: 'User Count',
                data: Object.values(roleCount),
                backgroundColor: 'rgba(59, 130, 246, 0.8)',
                borderColor: 'rgba(59, 130, 246, 1)',
                borderWidth: 2,
                borderRadius: 8
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });
}

// Update Country Chart
function updateCountryChart() {
    const countryCount = {};
    
    // Count users by their primary country or first country in countries array
    userData.forEach(u => {
        let userCountry = u.country;
        
        // If country is N/A or empty, try to get from countries array
        if (!userCountry || userCountry === 'N/A') {
            if (Array.isArray(u.countries) && u.countries.length > 0) {
                userCountry = u.countries[0];
            }
        }
        
        if (userCountry && userCountry !== 'N/A') {
            countryCount[userCountry] = (countryCount[userCountry] || 0) + 1;
        }
    });
    
    const ctx = document.getElementById('countryChart');
    
    if (window.countryChartInstance) {
        window.countryChartInstance.destroy();
    }
    
    const colors = ['#3b82f6', '#10b981', '#f59e0b', '#ef4444', '#8b5cf6', '#ec4899', '#06b6d4', '#84cc16'];
    
    window.countryChartInstance = new Chart(ctx, {
        type: 'doughnut',
        data: {
            labels: Object.keys(countryCount),
            datasets: [{
                data: Object.values(countryCount),
                backgroundColor: colors.slice(0, Object.keys(countryCount).length),
                borderWidth: 2,
                borderColor: '#ffffff'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
}

// Update Role Tab
function updateRoleTab() {
    const roleStats = {};
    
    userData.forEach(u => {
        const role = u.role || 'User';
        roleStats[role] = (roleStats[role] || 0) + 1;
    });
    
    // Update stats grid
    const statsGrid = document.getElementById('role-stats-grid');
    statsGrid.innerHTML = '';
    
    Object.entries(roleStats).forEach(([role, count]) => {
        const card = document.createElement('div');
        card.className = 'stat-card';
        card.innerHTML = `
            <h4>${role}</h4>
            <p class="stat-value">${count}</p>
        `;
        statsGrid.appendChild(card);
    });
    
    // Update table
    const tableBody = document.getElementById('role-table-body');
    tableBody.innerHTML = '';
    
    const total = userData.length;
    
    Object.entries(roleStats).forEach(([role, count]) => {
        const percentage = total > 0 ? ((count / total) * 100).toFixed(1) : 0;
        const row = document.createElement('tr');
        row.innerHTML = `
            <td><strong>${role}</strong></td>
            <td>${count}</td>
            <td>${percentage}%</td>
        `;
        tableBody.appendChild(row);
    });
    
    // Update role distribution chart
    updateRoleDistributionChart(roleStats);
    
    // Update role filter dropdown
    const roleFilter = document.getElementById('role-filter');
    roleFilter.innerHTML = '<option value="">All Roles</option>';
    Object.keys(roleStats).forEach(role => {
        const option = document.createElement('option');
        option.value = role;
        option.textContent = role;
        roleFilter.appendChild(option);
    });
}

// Update Role Distribution Chart
function updateRoleDistributionChart(roleStats) {
    const ctx = document.getElementById('roleDistributionChart');
    
    if (window.roleDistributionChartInstance) {
        window.roleDistributionChartInstance.destroy();
    }
    
    window.roleDistributionChartInstance = new Chart(ctx, {
        type: 'pie',
        data: {
            labels: Object.keys(roleStats),
            datasets: [{
                data: Object.values(roleStats),
                backgroundColor: [
                    '#3b82f6', '#10b981', '#f59e0b', '#ef4444', 
                    '#8b5cf6', '#ec4899', '#06b6d4', '#84cc16'
                ],
                borderWidth: 2,
                borderColor: '#ffffff'
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    position: 'bottom'
                }
            }
        }
    });
}

// Update Country Tab
function updateCountryTab() {
    const countryStats = {};
    
    userData.forEach(u => {
        let userCountry = u.country;
        
        // If country is N/A or empty, try to get from countries array
        if (!userCountry || userCountry === 'N/A') {
            if (Array.isArray(u.countries) && u.countries.length > 0) {
                userCountry = u.countries[0];
            }
        }
        
        if (userCountry && userCountry !== 'N/A') {
            countryStats[userCountry] = (countryStats[userCountry] || 0) + 1;
        }
    });
    
    // Update stats grid
    const statsGrid = document.getElementById('country-stats-grid');
    statsGrid.innerHTML = '';
    
    Object.entries(countryStats).forEach(([country, count]) => {
        const card = document.createElement('div');
        card.className = 'stat-card';
        card.innerHTML = `
            <h4>${country}</h4>
            <p class="stat-value">${count}</p>
        `;
        statsGrid.appendChild(card);
    });
    
    // Update table
    const tableBody = document.getElementById('country-table-body');
    tableBody.innerHTML = '';
    
    const total = userData.length;
    
    Object.entries(countryStats).forEach(([country, count]) => {
        const percentage = total > 0 ? ((count / total) * 100).toFixed(1) : 0;
        const row = document.createElement('tr');
        row.innerHTML = `
            <td><strong>${country}</strong></td>
            <td>${count}</td>
            <td>${percentage}%</td>
        `;
        tableBody.appendChild(row);
    });
    
    // Update country distribution chart
    updateCountryDistributionChart(countryStats);
    
    // Update country filter dropdown
    const countryFilter = document.getElementById('country-filter');
    countryFilter.innerHTML = '<option value="">All Countries</option>';
    Object.keys(countryStats).forEach(country => {
        const option = document.createElement('option');
        option.value = country;
        option.textContent = country;
        countryFilter.appendChild(option);
    });
}

// Update Country Distribution Chart
function updateCountryDistributionChart(countryStats) {
    const ctx = document.getElementById('countryDistributionChart');
    
    if (window.countryDistributionChartInstance) {
        window.countryDistributionChartInstance.destroy();
    }
    
    window.countryDistributionChartInstance = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: Object.keys(countryStats),
            datasets: [{
                label: 'User Count',
                data: Object.values(countryStats),
                backgroundColor: 'rgba(16, 185, 129, 0.8)',
                borderColor: 'rgba(16, 185, 129, 1)',
                borderWidth: 2,
                borderRadius: 8
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });
}

// Update Project Tab
function updateProjectTab() {
    const projectStats = {};
    
    userData.forEach(u => {
        let userProject = u.project;
        
        // If project is N/A or empty, try to get from projects array
        if (!userProject || userProject === 'N/A') {
            if (Array.isArray(u.projects) && u.projects.length > 0) {
                userProject = u.projects[0];
            }
        }
        
        if (userProject && userProject !== 'N/A') {
            projectStats[userProject] = (projectStats[userProject] || 0) + 1;
        } else {
            // Count unassigned users
            projectStats['Unassigned'] = (projectStats['Unassigned'] || 0) + 1;
        }
    });
    
    // Update stats grid
    const statsGrid = document.getElementById('project-stats-grid');
    statsGrid.innerHTML = '';
    
    Object.entries(projectStats).forEach(([project, count]) => {
        const card = document.createElement('div');
        card.className = 'stat-card';
        card.innerHTML = `
            <h4>${project}</h4>
            <p class="stat-value">${count}</p>
        `;
        statsGrid.appendChild(card);
    });
    
    // Update table
    const tableBody = document.getElementById('project-table-body');
    tableBody.innerHTML = '';
    
    const total = userData.length;
    
    Object.entries(projectStats).forEach(([project, count]) => {
        const percentage = total > 0 ? ((count / total) * 100).toFixed(1) : 0;
        const row = document.createElement('tr');
        row.innerHTML = `
            <td><strong>${project}</strong></td>
            <td>${count}</td>
            <td>${percentage}%</td>
        `;
        tableBody.appendChild(row);
    });
    
    // Update project distribution chart
    updateProjectDistributionChart(projectStats);
    
    // Update project filter dropdown
    const projectFilter = document.getElementById('project-filter');
    projectFilter.innerHTML = '<option value="">All Projects</option>';
    Object.keys(projectStats).forEach(project => {
        const option = document.createElement('option');
        option.value = project;
        option.textContent = project;
        projectFilter.appendChild(option);
    });
}

// Update Project Distribution Chart
function updateProjectDistributionChart(projectStats) {
    const ctx = document.getElementById('projectDistributionChart');
    
    if (window.projectDistributionChartInstance) {
        window.projectDistributionChartInstance.destroy();
    }
    
    window.projectDistributionChartInstance = new Chart(ctx, {
        type: 'bar',
        data: {
            labels: Object.keys(projectStats),
            datasets: [{
                label: 'User Count',
                data: Object.values(projectStats),
                backgroundColor: 'rgba(139, 92, 246, 0.8)',
                borderColor: 'rgba(139, 92, 246, 1)',
                borderWidth: 2,
                borderRadius: 8
            }]
        },
        options: {
            responsive: true,
            maintainAspectRatio: false,
            plugins: {
                legend: {
                    display: false
                }
            },
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        stepSize: 1
                    }
                }
            }
        }
    });
}

// Update Details Tab
function updateDetailsTab() {
    const tbody = document.getElementById('user-details-body');
    tbody.innerHTML = '';
    
    const start = (currentPage - 1) * itemsPerPage;
    const end = start + itemsPerPage;
    const paginatedData = filteredData.slice(start, end);
    
    paginatedData.forEach(user => {
        const row = document.createElement('tr');
        
        const createdDate = formatDate(user.createdAt || user.uploadTimestamp || user.engagementDate);
        
        // Get display country (primary or first from array)
        let displayCountry = user.country;
        if (!displayCountry || displayCountry === 'N/A') {
            if (Array.isArray(user.countries) && user.countries.length > 0) {
                displayCountry = user.countries.join(', ');
            } else {
                displayCountry = 'N/A';
            }
        }
        
        // Get display project (primary or first from array)
        let displayProject = user.project;
        if (!displayProject || displayProject === 'N/A') {
            if (Array.isArray(user.projects) && user.projects.length > 0) {
                displayProject = user.projects.join(', ');
            } else {
                displayProject = 'N/A';
            }
        }
        
        row.innerHTML = `
            <td><strong>${user.name}</strong></td>
            <td>${user.email}</td>
            <td>${user.role}</td>
            <td>${displayCountry}</td>
            <td>${displayProject}</td>
            <td>${user.mobile}</td>
            <td>${createdDate}</td>
        `;
        
        tbody.appendChild(row);
    });
    
    updatePagination();
}

// Update Pagination
function updatePagination() {
    const totalPages = Math.ceil(filteredData.length / itemsPerPage);
    
    document.getElementById('current-page').textContent = currentPage;
    document.getElementById('total-pages').textContent = totalPages;
    
    document.getElementById('prev-page').disabled = currentPage === 1;
    document.getElementById('next-page').disabled = currentPage === totalPages || totalPages === 0;
}

// Setup Event Listeners
function setupEventListeners() {
    const customDateRange = document.getElementById('custom-date-range');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');
    const applyDatesBtn = document.getElementById('apply-custom-dates');
    const clearDatesBtn = document.getElementById('clear-custom-dates');
    
    // Set max date to today for date inputs
    const today = new Date().toISOString().split('T')[0];
    if (startDateInput) startDateInput.max = today;
    if (endDateInput) endDateInput.max = today;
    
    // When start date changes, update end date minimum
    if (startDateInput && endDateInput) {
        startDateInput.addEventListener('change', () => {
            if (startDateInput.value) {
                endDateInput.min = startDateInput.value;
                if (endDateInput.value && new Date(endDateInput.value) < new Date(startDateInput.value)) {
                    endDateInput.value = '';
                }
            }
        });
    }
    
    // Apply custom dates
    if (applyDatesBtn) {
        applyDatesBtn.addEventListener('click', () => {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                showNotification('Please select both start and end dates', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                showNotification('Start date cannot be after end date', 'error');
                return;
            }
            
            customStartDate = startDate;
            customEndDate = endDate;
            currentTimePeriod = 'custom';
            
            applyTimeFilter('custom');
            currentPage = 1;
            updateDashboard();
            
            showNotification(`Showing data from ${formatDateOnly(startDate)} to ${formatDateOnly(endDate)}`, 'success');
        });
    }
    
    // Clear custom dates
    if (clearDatesBtn) {
        clearDatesBtn.addEventListener('click', () => {
            startDateInput.value = '';
            endDateInput.value = '';
            customStartDate = null;
            customEndDate = null;
            
            currentTimePeriod = '7days';
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === '7days') {
                    btn.classList.add('active');
                }
            });
            
            if (customDateRange) {
                customDateRange.style.display = 'none';
            }
            
            applyTimeFilter('7days');
            currentPage = 1;
            updateDashboard();
            showNotification('Custom date range cleared', 'info');
        });
    }
    
    // Time Period Buttons
    document.querySelectorAll('.period-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const period = this.getAttribute('data-period');
            
            document.querySelectorAll('.period-btn').forEach(b => b.classList.remove('active'));
            this.classList.add('active');
            
            if (period === 'custom') {
                if (customDateRange) {
                    customDateRange.style.display = 'block';
                }
                
                if (!startDateInput.value || !endDateInput.value) {
                    const end = new Date();
                    const start = new Date();
                    start.setDate(end.getDate() - 7);
                    
                    startDateInput.value = start.toISOString().split('T')[0];
                    endDateInput.value = end.toISOString().split('T')[0];
                }
                
                currentTimePeriod = 'custom';
                return;
            } else {
                if (customDateRange) {
                    customDateRange.style.display = 'none';
                }
                
                currentTimePeriod = period;
                applyTimeFilter(period);
                currentPage = 1;
                updateDashboard();
                
                const periodText = period === 'today' ? 'Today' : 
                                  `Last ${period.replace('days', '')} Days`;
                showNotification(`Showing team data for: ${periodText}`, 'info');
            }
        });
    });
    
    // Tab Navigation
    document.querySelectorAll('.tab-btn').forEach(btn => {
        btn.addEventListener('click', function() {
            const targetTab = this.getAttribute('data-tab');
            
            document.querySelectorAll('.tab-btn').forEach(b => b.classList.remove('active'));
            document.querySelectorAll('.tab-pane').forEach(p => p.classList.remove('active'));
            
            this.classList.add('active');
            document.getElementById(targetTab).classList.add('active');
        });
    });
    
    // Search
    document.getElementById('user-search').addEventListener('input', function(e) {
        filterData();
    });
    
    // Filters
    document.getElementById('role-filter').addEventListener('change', filterData);
    document.getElementById('country-filter').addEventListener('change', filterData);
    document.getElementById('project-filter').addEventListener('change', filterData);
    
    // Pagination
    document.getElementById('prev-page').addEventListener('click', () => {
        if (currentPage > 1) {
            currentPage--;
            updateDetailsTab();
        }
    });
    
    document.getElementById('next-page').addEventListener('click', () => {
        const totalPages = Math.ceil(filteredData.length / itemsPerPage);
        if (currentPage < totalPages) {
            currentPage++;
            updateDetailsTab();
        }
    });
    
    // Refresh Button
    document.getElementById('refreshDataBtn').addEventListener('click', async () => {
        showNotification('Refreshing team data...', 'info');
        showLoading();
        await fetchUserData();
        hideLoading();
        updateDashboard();
    });
    
    // Export Button
    document.getElementById('exportExcelBtn').addEventListener('click', exportToExcel);
}

// Filter Data
function filterData() {
    const searchTerm = document.getElementById('user-search').value.toLowerCase();
    const roleFilter = document.getElementById('role-filter').value;
    const countryFilter = document.getElementById('country-filter').value;
    const projectFilter = document.getElementById('project-filter').value;
    
    filteredData = userData.filter(user => {
        const matchesSearch = 
            user.name.toLowerCase().includes(searchTerm) ||
            user.email.toLowerCase().includes(searchTerm) ||
            user.username.toLowerCase().includes(searchTerm) ||
            user.employeeNumber.toLowerCase().includes(searchTerm) ||
            user.mobile.toLowerCase().includes(searchTerm);
        
        const matchesRole = !roleFilter || user.role === roleFilter || user.department === roleFilter;
        
        // Check if user matches country filter (check both country field and countries array)
        const matchesCountry = !countryFilter || 
                               user.country === countryFilter || 
                               (Array.isArray(user.countries) && user.countries.includes(countryFilter));
        
        // Check if user matches project filter (check both project field and projects array)
        const matchesProject = !projectFilter || 
                               user.project === projectFilter || 
                               (Array.isArray(user.projects) && user.projects.includes(projectFilter));
        
        return matchesSearch && matchesRole && matchesCountry && matchesProject;
    });
    
    currentPage = 1;
    updateDetailsTab();
}

// Export to Excel
function exportToExcel() {
    if (userData.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    // Prepare user data
    const excelData = userData.map((user, index) => {
        const createdDate = formatDate(user.createdAt || user.uploadTimestamp || user.engagementDate);
        const lastLoginDate = formatDate(user.lastLogin);
        const engagementDate = formatDateOnly(user.engagementDate);
        const dateExamined = formatDateOnly(user.dateExamined);
        const expiryMedicals = formatDateOnly(user.expiryMedicals);
        
        // Get all countries
        const allCountries = user.country && user.country !== 'N/A' ? user.country : '';
        const countriesArray = Array.isArray(user.countries) && user.countries.length > 0 ? user.countries.join(', ') : '';
        const countriesDisplay = [allCountries, countriesArray].filter(Boolean).join(', ') || 'N/A';
        
        // Get all projects
        const allProjects = user.project && user.project !== 'N/A' ? user.project : '';
        const projectsArray = Array.isArray(user.projects) && user.projects.length > 0 ? user.projects.join(', ') : '';
        const projectsDisplay = [allProjects, projectsArray].filter(Boolean).join(', ') || 'N/A';
        
        return {
            'S.No': index + 1,
            'Employee Number': user.employeeNumber,
            'Name': user.name,
            'Email': user.email,
            'Mobile': user.mobile,
            'Role': user.role,
            'Department': user.department,
            'Countries': countriesDisplay,
            'Projects': projectsDisplay,
            'Login Status': user.loginStatus ? 'Online' : 'Offline',
            'Driver License Number': user.driverLicenseNumber,
            'Driver License Code': user.driverLicenseCode,
            'NRC/ID Number': user.nrcIdNumber,
            'First Aid Level': user.levelFirstAid,
            'Client Induction': user.clientInduction,
            'Engagement Date': engagementDate,
            'Date Examined': dateExamined,
            'Medicals Expiry': expiryMedicals,
            'Last Login': lastLoginDate,
            'Created At': createdDate
        };
    });
    
    // Create workbook
    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.json_to_sheet(excelData);
    
    // Set column widths
    const colWidths = [
        { wch: 6 },   // S.No
        { wch: 15 },  // Employee Number
        { wch: 25 },  // Name
        { wch: 30 },  // Email
        { wch: 15 },  // Mobile
        { wch: 18 },  // Role
        { wch: 18 },  // Department
        { wch: 20 },  // Countries
        { wch: 25 },  // Projects
        { wch: 12 },  // Login Status
        { wch: 20 },  // Driver License Number
        { wch: 18 },  // Driver License Code
        { wch: 18 },  // NRC/ID Number
        { wch: 15 },  // First Aid Level
        { wch: 20 },  // Client Induction
        { wch: 15 },  // Engagement Date
        { wch: 15 },  // Date Examined
        { wch: 15 },  // Medicals Expiry
        { wch: 20 },  // Last Login
        { wch: 20 }   // Created At
    ];
    ws['!cols'] = colWidths;
    
    XLSX.utils.book_append_sheet(wb, ws, 'Team Data');
    
    // Create summary sheet
    const total = userData.length;
    const totalAll = allUserData.length;
    const active = userData.filter(u => u.loginStatus === true || u.isActive === true).length;
    const online = userData.filter(u => u.loginStatus === true).length;
    
    const roleCount = {};
    userData.forEach(u => {
        const role = u.role || u.department || 'User';
        roleCount[role] = (roleCount[role] || 0) + 1;
    });
    
    const uniqueCountries = new Set();
    userData.forEach(u => {
        if (u.country && u.country !== 'N/A') uniqueCountries.add(u.country);
        if (Array.isArray(u.countries)) {
            u.countries.forEach(c => { if (c && c !== 'N/A') uniqueCountries.add(c); });
        }
    });
    
    const uniqueProjects = new Set();
    userData.forEach(u => {
        if (u.project && u.project !== 'N/A') uniqueProjects.add(u.project);
        if (Array.isArray(u.projects)) {
            u.projects.forEach(p => { if (p && p !== 'N/A') uniqueProjects.add(p); });
        }
    });
    
    const summaryData = [
        { 'Metric': 'Total Team Members (All Time)', 'Value': totalAll },
        { 'Metric': 'Team Members in Selected Period', 'Value': total },
        { 'Metric': 'Currently Online', 'Value': online },
        { 'Metric': 'Active Users', 'Value': active },
        { 'Metric': 'Operating Countries', 'Value': uniqueCountries.size },
        { 'Metric': 'Active Projects', 'Value': uniqueProjects.size },
        { 'Metric': 'Roles/Departments Breakdown', 'Value': Object.entries(roleCount).map(([r, c]) => `${r}: ${c}`).join(', ') },
        { 'Metric': 'Countries List', 'Value': Array.from(uniqueCountries).join(', ') },
        { 'Metric': 'Projects List', 'Value': Array.from(uniqueProjects).join(', ') },
        { 'Metric': 'Date Range', 'Value': getCurrentDateRangeText() },
        { 'Metric': 'Export Date', 'Value': formatDate(new Date()) }
    ];
    
    const wsSummary = XLSX.utils.json_to_sheet(summaryData);
    wsSummary['!cols'] = [{ wch: 25 }, { wch: 50 }];
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    // Generate filename
    const dateRangeText = getCurrentDateRangeText().replace(/\s+/g, '_');
    const timestamp = new Date().toISOString().slice(0, 10);
    const filename = `Team_KPI_${dateRangeText}_${timestamp}.xlsx`;
    
    XLSX.writeFile(wb, filename);
    
    showNotification(`Exported ${userData.length} team member(s) to Excel successfully`, 'success');
}

// Get current date range as text
function getCurrentDateRangeText() {
    if (currentTimePeriod === 'today') {
        return 'Today';
    } else if (currentTimePeriod === 'custom') {
        if (customStartDate && customEndDate) {
            const start = formatDateOnly(customStartDate).replace(/\//g, '-');
            const end = formatDateOnly(customEndDate).replace(/\//g, '-');
            return `Custom_${start}_to_${end}`;
        }
        return 'All_Data';
    } else {
        return `Last_${currentTimePeriod.replace('days', '')}_Days`;
    }
}

// Notification system
function showNotification(message, type = 'info') {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `
        <div class="notification-content">
            <span>${message}</span>
            <button class="notification-close" aria-label="Close notification">&times;</button>
        </div>
    `;
    
    document.body.appendChild(notification);
    
    setTimeout(() => removeNotification(notification), 5000);
    
    notification.querySelector('.notification-close').addEventListener('click', () => {
        removeNotification(notification);
    });
}

function removeNotification(notification) {
    if (notification && notification.parentNode) {
        notification.parentNode.removeChild(notification);
    }
}

// Add notification styles
const notificationStyles = document.createElement('style');
notificationStyles.textContent = `
    .notification {
        position: fixed;
        top: 20px;
        right: 20px;
        z-index: var(--z-index-notification);
        padding: 16px;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        font-family: var(--font-family);
        font-weight: 500;
        max-width: 400px;
        min-width: 300px;
        animation: slideIn 0.3s ease;
        background: white;
    }
    
    .notification-success {
        border-left: 4px solid #28a745;
    }
    
    .notification-error {
        border-left: 4px solid #dc3545;
    }
    
    .notification-info {
        border-left: 4px solid #17a2b8;
    }
    
    .notification-content {
        display: flex;
        justify-content: space-between;
        align-items: center;
        gap: 1rem;
    }
    
    .notification-content span {
        color: #333;
        font-size: 14px;
    }
    
    .notification-close {
        background: none;
        border: none;
        font-size: 18px;
        color: #999;
        cursor: pointer;
        padding: 0;
        line-height: 1;
        margin-left: 12px;
    }
    
    .notification-close:hover {
        color: #666;
    }
    
    @keyframes slideIn {
        from {
            transform: translateX(100%);
            opacity: 0;
        }
        to {
            transform: translateX(0);
            opacity: 1;
        }
    }
`;
document.head.appendChild(notificationStyles);
</script>
@endsection

