<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use App\Services\FirebaseStorageService;

class NoticeController extends Controller
{
    protected $firebaseStorage;

    public function __construct(FirebaseStorageService $firebaseStorage)
    {
        $this->firebaseStorage = $firebaseStorage;
    }
    public function index()
    {
        try {
            // Fetch notices from MongoDB
            $notices = $this->fetchNoticesFromMongoDB();
            
            // Add serial numbers and format data
            foreach ($notices as $index => &$notice) {
                $notice['serial_number'] = $index + 1;
                $notice['time_ago'] = $this->timeAgo($notice['createdAt']);
                $notice['formatted_date'] = $this->formatDate($notice['createdAt']);
                
                // Ensure imageUrl key exists (set to null if not present)
                if (!isset($notice['imageUrl'])) {
                    $notice['imageUrl'] = null;
                }
                
                // Ensure other required keys exist
                if (!isset($notice['title'])) {
                    $notice['title'] = 'Untitled Notice';
                }
                if (!isset($notice['description'])) {
                    $notice['description'] = 'No description available';
                }
                if (!isset($notice['createdAt'])) {
                    $notice['createdAt'] = now()->toISOString();
                }
            }
            
            return view('notice.notice_board', compact('notices'));
        } catch (\Exception $e) {
            Log::error('Error fetching notices: ' . $e->getMessage());
            return view('notice.notice_board', ['notices' => []]);
        }
    }

    public function create()
    {
        return view('notice.add_notice');
    }

    public function store(Request $request)
    {
        $request->validate([
            'title' => 'required|string|max:255',
            'description' => 'required|string',
            'image' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:5120', // 5MB max
        ]);

        try {
            $noticeId = time() . '_' . uniqid();
            // Get current user ID dynamically from auth or fallback
            $userId = auth()->check() ? auth()->id() : 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
            
            // Handle image upload to MongoDB if present
            $imageUrl = null;
            if ($request->hasFile('image')) {
                // Get MongoDB Notice API URL from environment (dynamic)
                $baseUrl = env('MONGODB_NOTICE_API_BASE_URL', 'http://168.231.113.219:5015');
                $uploadUrl = "{$baseUrl}/notices/upload-image/{$userId}";
                
                // Upload image to MongoDB Notice API
                $imageResponse = Http::timeout(30)->attach(
                    'image',
                    file_get_contents($request->file('image')->getRealPath()),
                    $request->file('image')->getClientOriginalName()
                )->post($uploadUrl);
                
                if ($imageResponse->successful()) {
                    $imageResult = $imageResponse->json();
                    $imageUrl = $imageResult['imageUrl'] ?? null;
                }
            }

            // Prepare notice data
            $noticeData = [
                'noticeId' => $noticeId,
                'title' => $request->title,
                'description' => $request->description,
                'imageUrl' => $imageUrl,
                'createdAt' => now()->toISOString(),
                'createdBy' => $userId,
            ];

            // Save to MongoDB
            $this->saveNoticeToMongoDB($noticeData, $userId);

            // Send push notification to all users (fire-and-forget)
            $this->sendNoticeToAllUsers($request->title, $request->description, $noticeId);

            return redirect()->route('notice.index')->with('success', 'Notice created successfully!');
        } catch (\Exception $e) {
            Log::error('Error creating notice: ' . $e->getMessage());
            return back()->with('error', 'Failed to create notice: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        try {
            $notice = $this->fetchNoticeById($id);
            if (!$notice) {
                return redirect()->route('notice.index')->with('error', 'Notice not found');
            }
            
            // Ensure all required keys exist
            if (!isset($notice['imageUrl'])) {
                $notice['imageUrl'] = null;
            }
            if (!isset($notice['title'])) {
                $notice['title'] = 'Untitled Notice';
            }
            if (!isset($notice['description'])) {
                $notice['description'] = 'No description available';
            }
            if (!isset($notice['createdAt'])) {
                $notice['createdAt'] = now()->toISOString();
            }
            
            return view('notice.view_notice', compact('notice'));
        } catch (\Exception $e) {
            Log::error('Error fetching notice: ' . $e->getMessage());
            return redirect()->route('notice.index')->with('error', 'Failed to fetch notice');
        }
    }

    public function destroy($id)
    {
        try {
            $this->deleteNoticeFromMongoDB($id);
            return redirect()->route('notice.index')->with('success', 'Notice deleted successfully!');
        } catch (\Exception $e) {
            Log::error('Error deleting notice: ' . $e->getMessage());
            return back()->with('error', 'Failed to delete notice: ' . $e->getMessage());
        }
    }

    private function fetchNoticesFromMongoDB()
    {
        try {
            // Get current user ID (admin) - dynamic from auth or fallback
            $userId = auth()->check() ? auth()->id() : 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
            
            // Get MongoDB Notice API URL from environment (dynamic)
            $baseUrl = env('MONGODB_NOTICE_API_BASE_URL', 'http://168.231.113.219:5015');
            $apiUrl = "{$baseUrl}/notices/all/{$userId}";
            
            $response = Http::timeout(10)->get($apiUrl);
            
            if (!$response->successful()) {
                throw new \Exception('Failed to fetch notices from MongoDB');
            }

            $result = $response->json();
            
            if (!isset($result['success']) || !$result['success']) {
                throw new \Exception('MongoDB API returned error');
            }

            $allNotices = $result['data'] ?? [];

            // Format notices for display
            foreach ($allNotices as &$notice) {
                // Convert MongoDB _id to noticeId if needed
                if (!isset($notice['noticeId']) && isset($notice['_id'])) {
                    $notice['noticeId'] = $notice['_id'];
                }
                
                // Ensure createdAt is in ISO format
                if (isset($notice['createdAt']) && is_array($notice['createdAt'])) {
                    // MongoDB date object
                    $notice['createdAt'] = date('c', strtotime($notice['createdAt']['date'] ?? 'now'));
                }
            }

            return $allNotices;
        } catch (\Exception $e) {
            Log::error('Error fetching notices from MongoDB: ' . $e->getMessage());
            return [];
        }
    }

    private function fetchNoticeById($id)
    {
        try {
            // Get current user ID (admin) - dynamic from auth or fallback
            $userId = auth()->check() ? auth()->id() : 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
            
            // Get MongoDB Notice API URL from environment (dynamic)
            $baseUrl = env('MONGODB_NOTICE_API_BASE_URL', 'http://168.231.113.219:5015');
            $apiUrl = "{$baseUrl}/notices/{$id}?userId={$userId}";
            
            $response = Http::timeout(10)->get($apiUrl);
            
            if (!$response->successful()) {
                return null;
            }

            $result = $response->json();
            
            if (!isset($result['success']) || !$result['success']) {
                return null;
            }

            $notice = $result['data'] ?? null;
            
            // Format notice data
            if ($notice) {
                if (!isset($notice['noticeId']) && isset($notice['_id'])) {
                    $notice['noticeId'] = $notice['_id'];
                }
                
                // Ensure createdAt is in ISO format
                if (isset($notice['createdAt']) && is_array($notice['createdAt'])) {
                    $notice['createdAt'] = date('c', strtotime($notice['createdAt']['date'] ?? 'now'));
                }
            }

            return $notice;
        } catch (\Exception $e) {
            Log::error('Error fetching notice by ID from MongoDB: ' . $e->getMessage());
            return null;
        }
    }

    private function saveNoticeToMongoDB($noticeData, $userId)
    {
        try {
            // Get MongoDB Notice API URL from environment (dynamic)
            $baseUrl = env('MONGODB_NOTICE_API_BASE_URL', 'http://168.231.113.219:5015');
            $apiUrl = "{$baseUrl}/notices";
            
            $response = Http::timeout(10)->post($apiUrl, $noticeData);

            if (!$response->successful()) {
                throw new \Exception('Failed to save notice to MongoDB');
            }

            $result = $response->json();
            
            if (!isset($result['success']) || !$result['success']) {
                throw new \Exception($result['error'] ?? 'Failed to save notice');
            }

            return true;
        } catch (\Exception $e) {
            Log::error('Error saving notice to MongoDB: ' . $e->getMessage());
            throw $e;
        }
    }

    private function deleteNoticeFromMongoDB($noticeId)
    {
        try {
            // Get current user ID (admin) - dynamic from auth or fallback
            $userId = auth()->check() ? auth()->id() : 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
            
            // Get MongoDB Notice API URL from environment (dynamic)
            $baseUrl = env('MONGODB_NOTICE_API_BASE_URL', 'http://168.231.113.219:5015');
            $apiUrl = "{$baseUrl}/notices/{$noticeId}/{$userId}";
            
            $response = Http::timeout(10)->delete($apiUrl);
            
            if (!$response->successful()) {
                throw new \Exception('Failed to delete notice from MongoDB');
            }

            $result = $response->json();
            
            if (!isset($result['success']) || !$result['success']) {
                throw new \Exception($result['error'] ?? 'Failed to delete notice');
            }

            return true;
        } catch (\Exception $e) {
            Log::error('Error deleting notice from MongoDB: ' . $e->getMessage());
            throw $e;
        }
    }


    private function timeAgo($isoDate)
    {
        try {
            $now = now();
            $date = \Carbon\Carbon::parse($isoDate);
            
            // Check if date is in the future
            if ($date->isFuture()) {
                return 'Just now';
            }
            
            $diffInSeconds = $now->diffInSeconds($date);
            $diffInMinutes = $now->diffInMinutes($date);
            $diffInHours = $now->diffInHours($date);
            $diffInDays = $now->diffInDays($date);

            if ($diffInSeconds < 60) {
                return 'Just now';
            } elseif ($diffInMinutes < 60) {
                return $diffInMinutes . ' minute' . ($diffInMinutes > 1 ? 's' : '') . ' ago';
            } elseif ($diffInHours < 24) {
                return $diffInHours . ' hour' . ($diffInHours > 1 ? 's' : '') . ' ago';
            } elseif ($diffInDays === 0) {
                return 'Today';
            } elseif ($diffInDays === 1) {
                return 'Yesterday';
            } elseif ($diffInDays < 7) {
                return $diffInDays . ' days ago';
            } elseif ($diffInDays < 30) {
                $weeks = floor($diffInDays / 7);
                return $weeks . ' week' . ($weeks > 1 ? 's' : '') . ' ago';
            } elseif ($diffInDays < 365) {
                $months = floor($diffInDays / 30);
                return $months . ' month' . ($months > 1 ? 's' : '') . ' ago';
            } else {
                $years = floor($diffInDays / 365);
                return $years . ' year' . ($years > 1 ? 's' : '') . ' ago';
            }
        } catch (\Exception $e) {
            return 'Unknown';
        }
    }

    private function formatDate($isoDate)
    {
        $date = \Carbon\Carbon::parse($isoDate);
        return $date->format('M d, Y');
    }

    private function sendNoticeToAllUsers($title, $description, $noticeId)
    {
        try {
            // Get push notification URL from environment (dynamic)
            $pushNotificationUrl = env('PUSH_NOTIFICATION_SEND_TO_ALL_URL', 'http://168.231.113.219:5016/push-notifications/send-to-all');
            
            // Truncate description if too long (FCM limit is ~4KB for entire payload)
            $truncatedDescription = strlen($description) > 200 
                ? substr($description, 0, 197) . '...' 
                : $description;
            
            // Prepare push notification payload (dynamic)
            $notificationPayload = [
                'title' => '📢 New Notice: ' . (strlen($title) > 50 ? substr($title, 0, 47) . '...' : $title),
                'body' => $truncatedDescription,
                'data' => [
                    'type' => 'notice',
                    'noticeId' => (string)$noticeId,
                    'screen' => 'NoticeBoard',
                    'timestamp' => now()->toISOString()
                ]
            ];

            // Send to Push Notification API dynamically - fire-and-forget (non-blocking)
            $response = Http::timeout(5)
                ->retry(2, 100) // Retry 2 times with 100ms delay
                ->post($pushNotificationUrl, $notificationPayload);
            
            if ($response->successful()) {
                $result = $response->json();
                Log::info('✅ Push notification sent to all users for notice: ' . $noticeId, [
                    'successCount' => $result['successCount'] ?? 0,
                    'totalTokens' => $result['totalTokens'] ?? 0,
                    'url' => $pushNotificationUrl
                ]);
            } else {
                Log::warning('⚠️ Push notification API returned non-success status', [
                    'status' => $response->status(),
                    'noticeId' => $noticeId,
                    'url' => $pushNotificationUrl
                ]);
            }
        } catch (\Exception $e) {
            // Log error but don't fail the notice creation (non-critical)
            Log::warning('⚠️ Failed to send push notification (non-critical): ' . $e->getMessage(), [
                'noticeId' => $noticeId,
                'url' => env('PUSH_NOTIFICATION_SEND_TO_ALL_URL', 'http://168.231.113.219:5016/push-notifications/send-to-all'),
                'trace' => $e->getTraceAsString()
            ]);
        }
    }
}
