<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class HRApplicationController extends Controller
{
    /**
     * Get all applications by type from MongoDB
     */
    public function getApplicationsByType($applicationType)
    {
        try {
            $adminUid = auth()->check() ? auth()->id() : 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'http://168.231.113.219:5013');
            
            $response = Http::timeout(10)->get("{$baseUrl}/hr-applications/{$applicationType}/{$adminUid}");
            
            if (!$response->successful()) {
                Log::error("Failed to fetch {$applicationType} applications", [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'data' => [],
                    'count' => 0
                ], 500);
            }
            
            return response()->json($response->json());
        } catch (\Exception $e) {
            Log::error("Error fetching {$applicationType} applications: " . $e->getMessage());
            return response()->json([
                'success' => false,
                'data' => [],
                'count' => 0,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Create new HR application in MongoDB
     */
    public function store(Request $request)
    {
        try {
            $adminUid = auth()->check() ? auth()->id() : 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'http://168.231.113.219:5013');
            
            // Prepare application data
            $applicationData = $request->all();
            $applicationData['adminUid'] = $adminUid;
            $applicationData['createdAt'] = now()->toISOString();
            
            Log::info('Creating HR application', [
                'type' => $applicationData['applicationType'] ?? 'unknown',
                'adminUid' => $adminUid
            ]);
            
            $response = Http::timeout(15)->post("{$baseUrl}/hr-applications", $applicationData);
            
            if (!$response->successful()) {
                Log::error('Failed to create HR application', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'error' => 'Failed to create application'
                ], 500);
            }
            
            $result = $response->json();
            Log::info('HR application created successfully', [
                'applicationNumber' => $result['data']['applicationNumber'] ?? 'N/A'
            ]);
            
            return response()->json($result);
        } catch (\Exception $e) {
            Log::error('Error creating HR application: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Update application status (approve/reject)
     */
    public function updateStatus(Request $request, $id)
    {
        try {
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'http://168.231.113.219:5013');
            
            $statusData = [
                'status' => $request->status,
                'comments' => $request->comments,
                'approverUid' => auth()->check() ? auth()->id() : 'cahj3jJqe5Z16pQdLGSWrNgm84o1',
                'approverName' => $request->approverName,
                'approverRole' => $request->approverRole,
                'reason' => $request->reason,
                'timestamp' => now()->toISOString()
            ];
            
            Log::info('Updating HR application status', [
                'id' => $id,
                'status' => $request->status,
                'approverRole' => $request->approverRole
            ]);
            
            $response = Http::timeout(10)->put("{$baseUrl}/hr-applications/{$id}/status", $statusData);
            
            if (!$response->successful()) {
                Log::error('Failed to update application status', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'error' => 'Failed to update status'
                ], 500);
            }
            
            return response()->json($response->json());
        } catch (\Exception $e) {
            Log::error('Error updating application status: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Delete HR application
     */
    public function destroy($id)
    {
        try {
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'http://168.231.113.219:5013');
            
            Log::info('Deleting HR application', ['id' => $id]);
            
            $response = Http::timeout(10)->delete("{$baseUrl}/hr-applications/{$id}");
            
            if (!$response->successful()) {
                Log::error('Failed to delete application', [
                    'status' => $response->status(),
                    'body' => $response->body()
                ]);
                return response()->json([
                    'success' => false,
                    'error' => 'Failed to delete application'
                ], 500);
            }
            
            return response()->json($response->json());
        } catch (\Exception $e) {
            Log::error('Error deleting application: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => $e->getMessage()
            ], 500);
        }
    }
    
    /**
     * Get application counts by type
     */
    public function getCounts()
    {
        try {
            $adminUid = auth()->check() ? auth()->id() : 'cahj3jJqe5Z16pQdLGSWrNgm84o1';
            $baseUrl = env('MONGODB_HR_API_BASE_URL', 'http://168.231.113.219:5013');
            
            $response = Http::timeout(10)->get("{$baseUrl}/hr-applications/counts/{$adminUid}");
            
            if (!$response->successful()) {
                Log::error('Failed to fetch HR application counts', [
                    'status' => $response->status()
                ]);
                return response()->json([
                    'success' => false,
                    'data' => []
                ], 500);
            }
            
            return response()->json($response->json());
        } catch (\Exception $e) {
            Log::error('Error fetching counts: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'data' => [],
                'error' => $e->getMessage()
            ], 500);
        }
    }
}
