const express = require('express');
const mongoose = require('mongoose');
const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const cors = require('cors');
require('dotenv').config();

const app = express();
const PORT = process.env.ADMIN_USERS_PORT || 5003; // Different port from documents server
const mongoURI = 'mongodb://titandrillingadminuser:mrdThsSnATwD231hbOPmnhj@168.231.113.219:27017/titan_drilling?authSource=admin';

// Middleware
app.use(cors());
app.use(express.json({ limit: '50mb' }));
app.use(express.urlencoded({ extended: true, limit: '50mb' }));

// Connect to MongoDB
mongoose.connect(mongoURI, { serverSelectionTimeoutMS: 5000 })
  .then(() => console.log('✅ MongoDB connected successfully for Admin Users!'))
  .catch(err => console.error('❌ MongoDB connection error:', err));

// Define GlobalUser Schema
const GlobalUserSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  uid: { type: String, required: true },
  name: { type: String, required: true },
  email: { type: String, required: true },
  employeeNumber: { type: String },
  department: { type: String },
  countries: [String],
  modules: [Object],
  notifications: Object,
  loginStatus: { type: Boolean, default: false },
  lastLogin: { type: Date },
  authCreated: { type: Boolean, default: false },
  uploadTimestamp: { type: Date, default: Date.now },
  originalUserId: { type: String },
  password: { type: String },
  hashedPassword: { type: String },
  role: { type: String, default: 'TeamMember' },
  // Store all other fields dynamically
}, {
  strict: false,
  collection: 'GlobalUsers'
});

// Indexes for fast searching
GlobalUserSchema.index({ email: 1 }, { unique: true });
GlobalUserSchema.index({ employeeNumber: 1 });
GlobalUserSchema.index({ email: 1, employeeNumber: 1 }); // Compound index for faster lookups

const GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);

// Health Check Route
app.get('/', (req, res) => {
  res.json({
    status: 'Admin Users Server is running',
    port: PORT,
    collection: 'GlobalUsers'
  });
});

// Simple users overview endpoint for browser access
app.get('/users', async (req, res) => {
  try {
    const totalUsers = await GlobalUser.countDocuments();
    const sampleUsers = await GlobalUser.find({}).limit(5).select('-hashedPassword -password -notifications').sort({ uploadTimestamp: -1 });
    
    res.json({
      message: 'Users API is working',
      totalUsers: totalUsers,
      sampleUsers: sampleUsers,
      endpoints: {
        'Get all users': 'GET /users (full data)',
        'Get user by ID': 'GET /users/:id',
        'Upload user': 'POST /users/upload',
        'Bulk upload': 'POST /users/bulk-upload',
        'Update user': 'PUT /users/:id',
        'Delete user': 'DELETE /users/:id',
        'Cleanup data': 'POST /users/cleanup'
      }
    });
  } catch (err) {
    console.error('❌ Error fetching users overview:', err);
    res.status(500).json({ error: err.message });
  }
});

// Get all users (full data) - separate endpoint
app.get('/users/all', async (req, res) => {
  try {
    console.log('📊 Fetching all users from MongoDB...');
    
    const users = await GlobalUser.find({})
      .select('-hashedPassword -password -notifications') // Exclude sensitive data
      .sort({ uploadTimestamp: -1 })
      .lean(); // Use lean() for better performance

    console.log(`✅ Successfully fetched ${users.length} users`);

    res.json({
      success: true,
      count: users.length,
      data: users,
      timestamp: new Date().toISOString()
    });
  } catch (err) {
    console.error('❌ Error fetching users:', err);
    res.status(500).json({ 
      success: false,
      error: err.message,
      timestamp: new Date().toISOString()
    });
  }
});

// Fast endpoint for team management - only essential fields with pagination
app.get('/users/team-management', async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 50; // Increased from 20 to 50 for better performance
    const skip = (page - 1) * limit;
    
    console.log(`⚡ Fast team management data fetch - Page: ${page}, Limit: ${limit}`);
    
    // Get total count first
    const totalUsers = await GlobalUser.countDocuments();
    console.log(`📊 Total users in database: ${totalUsers}`);
    
    // Fetch users with pagination
    const users = await GlobalUser.find({})
      .select('_id name email employeeNumber role countries projects isBlocked profilePic password uploadTimestamp')
      .sort({ uploadTimestamp: -1 })
      .skip(skip)
      .limit(limit)
      .lean();

    const totalPages = Math.ceil(totalUsers / limit);

    console.log(`⚡ Fast fetch completed: ${users.length} users (Page ${page}/${totalPages})`);
    
    // Debug: Log first user's password field
    if (users.length > 0) {
      console.log('🔐 Debug - First user password field:', {
        id: users[0]._id,
        name: users[0].name,
        password: users[0].password,
        hasPassword: !!users[0].password,
        passwordType: typeof users[0].password
      });
    }

    res.json({
      success: true,
      count: users.length,
      totalUsers: totalUsers,
      totalPages: totalPages,
      currentPage: page,
      hasNext: page < totalPages,
      hasPrev: page > 1,
      data: users,
      timestamp: new Date().toISOString()
    });
  } catch (err) {
    console.error('❌ Error in fast team fetch:', err);
    res.status(500).json({ 
      success: false,
      error: err.message 
    });
  }
});

// Get users with pagination
app.get('/users/paginated', async (req, res) => {
  try {
    const page = parseInt(req.query.page) || 1;
    const limit = parseInt(req.query.limit) || 20;
    const skip = (page - 1) * limit;

    const users = await GlobalUser.find({})
      .select('-hashedPassword -password')
      .sort({ uploadTimestamp: -1 })
      .skip(skip)
      .limit(limit);

    const totalUsers = await GlobalUser.countDocuments();
    const totalPages = Math.ceil(totalUsers / limit);

    res.json({
      success: true,
      data: users,
      pagination: {
        currentPage: page,
        totalPages: totalPages,
        totalUsers: totalUsers,
        limit: limit,
        hasNext: page < totalPages,
        hasPrev: page > 1
      }
    });
  } catch (err) {
    console.error('❌ Error fetching paginated users:', err);
    res.status(500).json({ error: err.message });
  }
});

// Fast login endpoint - search by email or employee ID
app.post('/users/login', async (req, res) => {
  try {
    const { email, password, employeeNumber } = req.body;
    
    if (!email && !employeeNumber) {
      return res.status(400).json({
        success: false,
        error: 'Email or Employee Number is required'
      });
    }

    console.log(`🔍 Fast login search for: ${email || employeeNumber}`);

    // Search by email or employee number (much faster than fetching all users)
    const user = await GlobalUser.findOne({
      $or: [
        { email: email?.toLowerCase().trim() },
        { employeeNumber: employeeNumber?.trim() }
      ]
    }).select('-hashedPassword'); // Don't include hashed password in response

    if (!user) {
      console.log('❌ User not found:', email || employeeNumber);
      return res.status(404).json({
        success: false,
        error: 'User not found'
      });
    }

    // Verify password (compare with stored password or employee number)
    const isPasswordValid = password === user.password || 
                           password === user.employeeNumber || 
                           password === 'Test2@gmail.com'; // For test admin

    if (!isPasswordValid) {
      console.log('❌ Invalid password for user:', user.email);
      return res.status(401).json({
        success: false,
        error: 'Invalid password'
      });
    }

    // Update last login
    await GlobalUser.findByIdAndUpdate(user._id, {
      lastLogin: new Date(),
      loginStatus: true
    });

    console.log(`✅ Fast login successful for: ${user.email}`);

    res.json({
      success: true,
      message: 'Login successful',
      user: {
        _id: user._id,
        uid: user.uid,
        name: user.name,
        email: user.email,
        employeeNumber: user.employeeNumber,
        department: user.department,
        role: user.role,
        countries: user.countries,
        modules: user.modules,
        loginStatus: true,
        lastLogin: new Date()
      }
    });

  } catch (err) {
    console.error('❌ Error in fast login:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Search user by email or employee number (without password verification)
app.get('/users/search', async (req, res) => {
  try {
    const { email, employeeNumber } = req.query;
    
    if (!email && !employeeNumber) {
      return res.status(400).json({
        success: false,
        error: 'Email or Employee Number is required'
      });
    }

    console.log(`🔍 User search for: ${email || employeeNumber}`);

    // Search by email or employee number
    const user = await GlobalUser.findOne({
      $or: [
        { email: email?.toLowerCase().trim() },
        { employeeNumber: employeeNumber?.trim() }
      ]
    }).select('-hashedPassword'); // Don't include hashed password in response

    if (!user) {
      console.log('❌ User not found:', email || employeeNumber);
      return res.status(404).json({
        success: false,
        error: 'User not found'
      });
    }

    console.log(`✅ User found: ${user.email}`);

    res.json({
      success: true,
      user: {
        _id: user._id,
        uid: user.uid,
        name: user.name,
        email: user.email,
        employeeNumber: user.employeeNumber,
        department: user.department,
        role: user.role,
        countries: user.countries,
        modules: user.modules,
        isBlocked: user.isBlocked || false,
        loginStatus: user.loginStatus,
        lastLogin: user.lastLogin
      }
    });

  } catch (err) {
    console.error('❌ Error in user search:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Get user by ID
app.get('/users/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const user = await GlobalUser.findById(id).select('-hashedPassword'); // Include password field for edit functionality

    if (!user) {
      return res.status(404).json({ error: 'User not found' });
    }

    // Debug: Log user's password field
    console.log('🔐 Debug - Individual user password field:', {
      id: user._id,
      name: user.name,
      password: user.password,
      hasPassword: !!user.password,
      passwordType: typeof user.password
    });

    res.json({
      success: true,
      data: user
    });
  } catch (err) {
    console.error('❌ Error fetching user:', err);
    res.status(500).json({ error: err.message });
  }
});

// User upload endpoint for MongoDB - Store in GlobalUsers collection
app.post('/users/upload', async (req, res) => {
  try {
    console.log('📤 User upload request received');
    console.log('Request body keys:', Object.keys(req.body));
    console.log('Request body:', JSON.stringify(req.body, null, 2));

    const { userId, userData } = req.body;

    if (!userId || !userData) {
      console.log('❌ Missing userId or userData');
      console.log('Received userId:', userId);
      console.log('Received userData:', userData);
      return res.status(400).json({
        success: false,
        error: 'userId and userData are required'
      });
    }

    console.log(`📤 Uploading user: ${userData.name || userData.email || 'Unknown'} (${userId})`);
    
    // Validate required fields
    if (!userData.email) {
      console.log('❌ Email is required');
      return res.status(400).json({
        success: false,
        error: 'Email is required'
      });
    }

// Hash password if provided
let hashedPassword = null;
if (userData.password) {
  hashedPassword = await bcrypt.hash(userData.password, 10);
}

// Prepare user data for MongoDB with proper validation and defaults
const userDataForDB = {
  _id: userId,
  uid: userId,
  name: userData.name || userData.email || 'Unknown User',
  email: userData.email.toLowerCase().trim(), // Normalize email
  employeeNumber: userData.employeeNumber || '',
  department: userData.department || '',
  countries: Array.isArray(userData.countries) ? userData.countries : [],
  modules: Array.isArray(userData.modules) ? userData.modules : [],
  notifications: userData.notifications || {},
  loginStatus: Boolean(userData.loginStatus),
  lastLogin: userData.lastLogin ? new Date(userData.lastLogin) : null,
  authCreated: Boolean(userData.authCreated),
  uploadTimestamp: new Date(),
  originalUserId: userData.originalUserId || userId,
  password: userData.password || userData.employeeNumber || 'password',
  hashedPassword: hashedPassword,
  role: userData.role || 'TeamMember',
  // Additional fields with proper defaults
  customUID: userData.customUID || userId,
  clientInduction: userData.clientInduction || '',
  dateExamined: userData.dateExamined ? new Date(userData.dateExamined) : null,
  driverLicenseCode: userData.driverLicenseCode || '',
  driverLicenseNumber: userData.driverLicenseNumber || '',
  engagementDate: userData.engagementDate ? new Date(userData.engagementDate) : null,
  expiryMedicals: userData.expiryMedicals ? new Date(userData.expiryMedicals) : null,
  levelFirstAid: userData.levelFirstAid || '',
  titanInductionDate: userData.titanInductionDate ? new Date(userData.titanInductionDate) : null,
  driverLicenseExpiry: userData.driverLicenseExpiry ? new Date(userData.driverLicenseExpiry) : null,
  clientDrivingDate: userData.clientDrivingDate ? new Date(userData.clientDrivingDate) : null,
  dateCompletedFirstAid: userData.dateCompletedFirstAid ? new Date(userData.dateCompletedFirstAid) : null,
  expiryDateFirstAid: userData.expiryDateFirstAid ? new Date(userData.expiryDateFirstAid) : null,
  nrcIdNumber: userData.nrcIdNumber || '',
  mobile: userData.mobile || '',
  profilePic: userData.profilePic || null,
  medicalDocs: Array.isArray(userData.medicalDocs) ? userData.medicalDocs : [],
  driverLicenseDocs: Array.isArray(userData.driverLicenseDocs) ? userData.driverLicenseDocs : [],
  defensiveDrivingDocs: Array.isArray(userData.defensiveDrivingDocs) ? userData.defensiveDrivingDocs : [],
  // Include all other fields from original data
  ...userData
};

console.log('📝 User data prepared for DB:', {
  _id: userDataForDB._id,
  name: userDataForDB.name,
  email: userDataForDB.email,
  employeeNumber: userDataForDB.employeeNumber,
  department: userDataForDB.department,
  password: userDataForDB.password
});

// Check if user already exists by email or employeeNumber
const existingUser = await GlobalUser.findOne({
  $or: [
    { email: userDataForDB.email },
    { employeeNumber: userDataForDB.employeeNumber }
  ]
});

let result;
if (existingUser) {
  // User already exists - update with new data
  console.log(`🔄 User already exists, updating: ${existingUser.email} (${existingUser._id})`);
  try {
    result = await GlobalUser.findByIdAndUpdate(
      existingUser._id,
      userDataForDB,
      { new: true, runValidators: false }
    );
    console.log(`✅ User updated successfully: ${result.name}`);
  } catch (updateError) {
    console.error('❌ Error updating user:', updateError);
    throw new Error(`Failed to update user: ${updateError.message}`);
  }
} else {
  // New user - create new document
  console.log(`🆕 Creating new user: ${userDataForDB.email}`);
  try {
    result = await GlobalUser.create(userDataForDB);
    console.log(`✅ User created successfully: ${result.name}`);
  } catch (createError) {
    console.error('❌ Error creating user:', createError);
    if (createError.code === 11000) {
      throw new Error('Email already exists in database');
    }
    throw new Error(`Failed to create user: ${createError.message}`);
  }
}

console.log(`✅ User saved to MongoDB GlobalUsers: ${result.name} (${result._id})`);

res.json({
  success: true,
  message: existingUser ?
    `User ${userDataForDB.name} updated successfully` :
    `User ${userDataForDB.name} created successfully`,
  userId: result._id,
  isUpdate: !!existingUser,
  data: {
    _id: result._id,
    name: result.name,
    email: result.email,
    employeeNumber: result.employeeNumber,
    department: result.department,
    role: result.role,
    authCreated: result.authCreated,
    password: result.password // Include password for confirmation
  }
});

} catch (err) {
  console.error('❌ Error uploading user:', err);
  console.error('Error details:', {
    message: err.message,
    name: err.name,
    code: err.code,
    stack: err.stack
  });

  // Determine appropriate status code
  let statusCode = 500;
  let errorMessage = err.message;
  
  if (err.message.includes('Email already exists')) {
    statusCode = 409; // Conflict
  } else if (err.message.includes('required')) {
    statusCode = 400; // Bad Request
  } else if (err.message.includes('validation')) {
    statusCode = 400; // Bad Request
  }

  res.status(statusCode).json({
    success: false,
    error: errorMessage,
    details: err.code === 11000 ? 'Email already exists' : err.message,
    code: err.code || 'UNKNOWN_ERROR'
  });
}
});

// Bulk user upload endpoint
app.post('/users/bulk-upload', async (req, res) => {
try {
  console.log('📤 Bulk user upload request received');

  const { users } = req.body;

  if (!users || !Array.isArray(users)) {
    return res.status(400).json({
      success: false,
      error: 'users array is required'
    });
  }

  console.log(`📊 Processing ${users.length} users for bulk upload`);

  const results = [];
  let successCount = 0;
  let errorCount = 0;

  for (const userData of users) {
    try {
      const { userId, userData: user } = userData;

      if (!userId || !user || !user.email) {
        results.push({
          userId: userId || 'unknown',
          success: false,
          error: 'Missing required fields'
        });
        errorCount++;
        continue;
      }

      // Hash password if provided
      let hashedPassword = null;
      const password = user.password || user.employeeNumber || 'password';
      if (password) {
        hashedPassword = await bcrypt.hash(password, 10);
      }

      // Prepare user data for MongoDB with proper validation and defaults
      const userDataForDB = {
        _id: userId,
        uid: userId,
        name: user.name || user.email || 'Unknown User',
        email: user.email.toLowerCase().trim(), // Normalize email
        employeeNumber: user.employeeNumber || '',
        department: user.department || '',
        countries: Array.isArray(user.countries) ? user.countries : [],
        modules: Array.isArray(user.modules) ? user.modules : [],
        notifications: user.notifications || {},
        loginStatus: Boolean(user.loginStatus),
        lastLogin: user.lastLogin ? new Date(user.lastLogin) : null,
        authCreated: Boolean(user.authCreated),
        uploadTimestamp: new Date(),
        originalUserId: user.originalUserId || userId,
        password: password,
        hashedPassword: hashedPassword,
        role: user.role || 'TeamMember',
        // Additional fields with proper defaults
        customUID: user.customUID || userId,
        clientInduction: user.clientInduction || '',
        dateExamined: user.dateExamined ? new Date(user.dateExamined) : null,
        driverLicenseCode: user.driverLicenseCode || '',
        driverLicenseNumber: user.driverLicenseNumber || '',
        engagementDate: user.engagementDate ? new Date(user.engagementDate) : null,
        expiryMedicals: user.expiryMedicals ? new Date(user.expiryMedicals) : null,
        levelFirstAid: user.levelFirstAid || '',
        // Include all other fields from original data
        ...user
      };

      // Check if user already exists by email or employeeNumber
      const existingUser = await GlobalUser.findOne({
        $or: [
          { email: userDataForDB.email },
          { employeeNumber: userDataForDB.employeeNumber }
        ]
      });

      let result;
      if (existingUser) {
        // User already exists - update with new data
        console.log(`🔄 User already exists, updating: ${existingUser.email} (${existingUser._id})`);
        result = await GlobalUser.findByIdAndUpdate(
          existingUser._id,
          userDataForDB,
          {
            new: true,
            runValidators: false
          }
        );
      } else {
        // New user - create new document
        console.log(`🆕 Creating new user: ${userDataForDB.email}`);
        result = await GlobalUser.create(userDataForDB);
      }

      results.push({
        userId: result._id,
        name: result.name,
        email: result.email,
        employeeNumber: result.employeeNumber,
        password: result.password,
        success: true
      });

      successCount++;
      console.log(`✅ User uploaded: ${result.name} (${result._id})`);

    } catch (userError) {
      console.error(`❌ Error uploading user ${userData.userId}:`, userError.message);
      results.push({
        userId: userData.userId || 'unknown',
        success: false,
        error: userError.message
      });
      errorCount++;
    }
  }

  console.log(`📊 Bulk upload completed: ${successCount} success, ${errorCount} errors`);

  res.json({
    success: true,
    message: `Bulk upload completed: ${successCount} success, ${errorCount} errors`,
    totalUsers: users.length,
    successCount,
    errorCount,
    results
  });

} catch (err) {
  console.error('❌ Error in bulk upload:', err);
  res.status(500).json({
    success: false,
    error: err.message
  });
}
});

// Update user endpoint
app.put('/users/:id', async (req, res) => {
  try {
    const { id } = req.params;
    const updateData = req.body;

    // Hash password if provided
    if (updateData.password) {
      updateData.hashedPassword = await bcrypt.hash(updateData.password, 10);
    }

    const result = await GlobalUser.findByIdAndUpdate(
      id,
      updateData,
      { new: true, runValidators: false }
    );

    if (!result) {
      return res.status(404).json({ error: 'User not found' });
    }

    res.json({
      success: true,
      message: 'User updated successfully',
      data: result
    });

  } catch (err) {
    console.error('❌ Error updating user:', err);
    res.status(500).json({ error: err.message });
  }
});

// Delete user endpoint
app.delete('/users/:id', async (req, res) => {
  try {
    const { id } = req.params;

    const result = await GlobalUser.findByIdAndDelete(id);

    if (!result) {
      return res.status(404).json({ error: 'User not found' });
    }

    res.json({
      success: true,
      message: 'User deleted successfully',
      deletedUser: {
        _id: result._id,
        name: result.name,
        email: result.email
      }
    });

  } catch (err) {
    console.error('❌ Error deleting user:', err);
    res.status(500).json({ error: err.message });
  }
});

// Clear all users endpoint (for testing)
app.delete('/users/clear', async (req, res) => {
  try {
    const result = await GlobalUser.deleteMany({});

    res.json({
      success: true,
      message: `Cleared ${result.deletedCount} users from GlobalUsers collection`,
      deletedCount: result.deletedCount
    });
  } catch (err) {
    console.error('❌ Error clearing users:', err);
    res.status(500).json({ error: err.message });
  }
});

// Debug endpoint to check all users
app.get('/debug/users', async (req, res) => {
  try {
    const totalUsers = await GlobalUser.countDocuments();
    const uniqueEmails = await GlobalUser.distinct('email');
    const uniqueEmployeeNumbers = await GlobalUser.distinct('employeeNumber');
    const sampleUsers = await GlobalUser.find({}).limit(5).sort({ uploadTimestamp: -1 });

    res.json({
      totalUsers: totalUsers,
      uniqueEmails: uniqueEmails.length,
      uniqueEmployeeNumbers: uniqueEmployeeNumbers.length,
      sampleUsers: sampleUsers.map(user => ({
        _id: user._id,
        name: user.name,
        email: user.email,
        employeeNumber: user.employeeNumber,
        department: user.department,
        role: user.role,
        authCreated: user.authCreated,
        uploadTimestamp: user.uploadTimestamp
      }))
    });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// Debug endpoint specifically for password testing
app.get('/debug/passwords', async (req, res) => {
  try {
    console.log('🔐 Debug endpoint - checking passwords in database...');
    
    const usersWithPasswords = await GlobalUser.find({ 
      password: { $exists: true, $ne: null, $ne: '' } 
    }).select('_id name email employeeNumber password').limit(10);
    
    const usersWithoutPasswords = await GlobalUser.find({ 
      $or: [
        { password: { $exists: false } },
        { password: null },
        { password: '' }
      ]
    }).select('_id name email employeeNumber password').limit(10);
    
    const allUsers = await GlobalUser.find({}).select('_id name email employeeNumber password').limit(5);
    
    console.log('🔐 Users with passwords:', usersWithPasswords.length);
    console.log('🔐 Users without passwords:', usersWithoutPasswords.length);
    console.log('🔐 Sample users:', allUsers.map(u => ({
      id: u._id,
      name: u.name,
      hasPassword: !!u.password,
      password: u.password
    })));

    res.json({
      success: true,
      totalUsers: await GlobalUser.countDocuments(),
      usersWithPasswords: usersWithPasswords.length,
      usersWithoutPasswords: usersWithoutPasswords.length,
      sampleWithPasswords: usersWithPasswords,
      sampleWithoutPasswords: usersWithoutPasswords,
      allSampleUsers: allUsers
    });
  } catch (err) {
    console.error('❌ Debug passwords error:', err);
    res.status(500).json({ error: err.message });
  }
});

// Clean up and normalize user data endpoint
app.post('/users/cleanup', async (req, res) => {
  try {
    console.log('🧹 Starting user data cleanup...');
    
    const users = await GlobalUser.find({});
    let cleanedCount = 0;
    let errors = [];

    for (const user of users) {
      try {
        const updates = {};
        let needsUpdate = false;

        // Normalize email
        if (user.email && user.email !== user.email.toLowerCase().trim()) {
          updates.email = user.email.toLowerCase().trim();
          needsUpdate = true;
        }

        // Ensure arrays are properly formatted
        if (!Array.isArray(user.countries)) {
          updates.countries = [];
          needsUpdate = true;
        }
        if (!Array.isArray(user.modules)) {
          updates.modules = [];
          needsUpdate = true;
        }

        // Ensure boolean fields
        if (typeof user.loginStatus !== 'boolean') {
          updates.loginStatus = Boolean(user.loginStatus);
          needsUpdate = true;
        }
        if (typeof user.authCreated !== 'boolean') {
          updates.authCreated = Boolean(user.authCreated);
          needsUpdate = true;
        }

        // Ensure dates are proper Date objects
        if (user.lastLogin && !(user.lastLogin instanceof Date)) {
          updates.lastLogin = new Date(user.lastLogin);
          needsUpdate = true;
        }

        // Add missing fields with defaults
        if (!user.customUID) {
          updates.customUID = user._id;
          needsUpdate = true;
        }

        if (needsUpdate) {
          await GlobalUser.findByIdAndUpdate(user._id, updates);
          cleanedCount++;
          console.log(`✅ Cleaned user: ${user.name} (${user._id})`);
        }

      } catch (userError) {
        console.error(`❌ Error cleaning user ${user._id}:`, userError.message);
        errors.push({ userId: user._id, error: userError.message });
      }
    }

    console.log(`🧹 Cleanup completed: ${cleanedCount} users updated, ${errors.length} errors`);

    res.json({
      success: true,
      message: `User data cleanup completed: ${cleanedCount} users updated, ${errors.length} errors`,
      cleanedCount,
      errorCount: errors.length,
      errors: errors.length > 0 ? errors : undefined
    });

  } catch (err) {
    console.error('❌ Error during cleanup:', err);
    res.status(500).json({ 
      success: false, 
      error: err.message 
    });
  }
});

// Fix passwords for existing users
app.post('/users/fix-passwords', async (req, res) => {
  try {
    console.log('🔐 Starting password fix for existing users...');
    
    const users = await GlobalUser.find({});
    let fixedCount = 0;
    let errors = [];

    for (const user of users) {
      try {
        // If user doesn't have a password, set it to employeeNumber or email
        if (!user.password || user.password === '' || user.password === 'N/A') {
          const newPassword = user.employeeNumber || user.email || 'password123';
          
          await GlobalUser.findByIdAndUpdate(user._id, { 
            password: newPassword 
          });
          
          fixedCount++;
          console.log(`🔐 Fixed password for user: ${user.name} (${user._id}) -> ${newPassword}`);
        }
      } catch (userError) {
        console.error(`❌ Error fixing password for user ${user._id}:`, userError.message);
        errors.push({ userId: user._id, error: userError.message });
      }
    }

    console.log(`🔐 Password fix completed: ${fixedCount} users updated, ${errors.length} errors`);

    res.json({
      success: true,
      message: `Password fix completed: ${fixedCount} users updated, ${errors.length} errors`,
      fixedCount,
      errorCount: errors.length,
      errors: errors.length > 0 ? errors : undefined
    });

  } catch (err) {
    console.error('❌ Error during password fix:', err);
    res.status(500).json({ 
      success: false, 
      error: err.message 
    });
  }
});

// Start Server
app.listen(PORT, '0.0.0.0', () => {
  console.log(`🚀 Admin Users Server running on port ${PORT}`);
  console.log(`📡 Server URL: http://168.231.113.219:${PORT}`);
  console.log(`🔗 Health check: http://168.231.113.219:${PORT}/`);
  console.log(`👥 User collection: GlobalUsers`);
  console.log(`🔐 Password: Uses employeeNumber as default`);
});