const express = require('express');
const mongoose = require('mongoose');
const cors = require('cors');
const fetch = require('node-fetch');

const app = express();
const PORT = 5010;

// Middleware
app.use(cors());
app.use(express.json());
app.use(express.urlencoded({ extended: true }));

// MongoDB Connection
const mongoURI = 'mongodb://titandrillingadminuser:mrdThsSnATwD231hbOPmnhj@168.231.113.219:27017/titan_drilling?authSource=admin';

// Connect to MongoDB with better error handling
const connectDB = async () => {
  try {
    await mongoose.connect(mongoURI, { 
      serverSelectionTimeoutMS: 30000,
      maxPoolSize: 5,
      socketTimeoutMS: 45000,
      bufferCommands: false,
      connectTimeoutMS: 30000,
      heartbeatFrequencyMS: 10000,
      retryWrites: true,
      retryReads: true
    });
    console.log('✅ MongoDB connected successfully for Service Schedule Handler!');
  } catch (err) {
    console.error('❌ MongoDB connection error:', err);
    console.error('⚠️ Will retry connection in 10 seconds...');
    setTimeout(() => {
      connectDB();
    }, 10000);
  }
};

// MongoDB connection event handlers
mongoose.connection.on('disconnected', () => {
  console.log('⚠️ MongoDB disconnected');
  setTimeout(() => {
    connectDB();
  }, 5000);
});

mongoose.connection.on('error', (err) => {
  console.error('❌ MongoDB connection error:', err);
});

// Service Schedule Schema
const ServiceScheduleSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  id: { type: String, required: true },
  adminUid: { type: String, required: true },
  scheduleId: { type: String, required: true },
  country: { type: String, required: true },
  project: { type: String, required: true },
  title: { type: String, required: true },
  equipment: {
    entryNumber: String,
    equipmentName: String,
    equipmentNumber: String,
    meterReading: Number,
    meterUnit: String,
  },
  note: { type: String, required: true },
  lastServiceDate: { type: String, required: true },
  meterReading: { type: Number, required: true },
  meterUnit: { type: String, required: true },
  repeatType: { type: String, required: true },
  intervals: [{
    checked: Boolean,
    value: String,
    unit: String
  }],
  intervalUnit: { type: String, required: true },
  notifyBefore: {
    value: String,
    unit: String,
    time: String
  },
  tasks: [String],
  status: { type: String, default: 'upcoming' },
  createdAt: { type: String, required: true },
  dueHours: String,
  hoursToGo: String,
  daysToGo: String,
  isArchived: { type: Boolean, default: false },
  jobCardId: { type: String },
  jobCardRequestNumber: { type: String },
  jobCardTitle: { type: String },
  jobCardStatus: { type: String },
  jobCardCreatedAt: { type: String },
  updatedAt: { type: Date, default: Date.now }
}, { 
  timestamps: false,
  collection: 'GlobalServiceSchdule'
});

// Counter Schema for auto-incrementing schedule IDs
const CounterSchema = new mongoose.Schema({
  _id: { type: String, required: true },
  counter: { type: Number, default: 0 }
}, {
  collection: 'GlobalServiceSchdule'
});

// Check if models already exist before creating them
const ServiceSchedule = mongoose.models.GlobalServiceSchdule || mongoose.model('GlobalServiceSchdule', ServiceScheduleSchema);
const Counter = mongoose.models.ServiceScheduleCounter || mongoose.model('ServiceScheduleCounter', CounterSchema);

// Helper function to send service schedule notifications
const sendServiceScheduleNotifications = async (scheduleData, createdByName, creatorEmail) => {
  try {
    console.log('🔔 Sending service schedule notifications...');
    
    const notificationPayload = {
      scheduleData: {
        _id: scheduleData._id,
        id: scheduleData.id,
        scheduleId: scheduleData.scheduleId,
        title: scheduleData.title,
        equipmentName: scheduleData.equipment?.equipmentName || 'N/A',
        repeatType: scheduleData.repeatType || 'N/A',
        intervalUnit: scheduleData.intervalUnit || 'N/A',
        country: scheduleData.country,
        project: scheduleData.project,
        note: scheduleData.note || '',
        lastServiceDate: scheduleData.lastServiceDate || '',
        tasksCount: scheduleData.tasks?.length || 0,
      },
      createdByName,
      creatorEmail,
    };
    
    const response = await fetch('http://168.231.113.219:5024/api/service-schedule-notifications/new-schedule', {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(notificationPayload),
    });
    
    const result = await response.json();
    
    if (result.success) {
      console.log('✅ Service schedule notifications sent successfully!');
      console.log('📊 Recipients:', result.recipients);
    } else {
      console.error('❌ Failed to send notifications:', result.error);
    }
    
    return result;
  } catch (error) {
    console.error('❌ Error sending service schedule notifications:', error);
    return { success: false, error: error.message };
  }
};

// Routes

// Get all service schedules
app.get('/service-schedules', async (req, res) => {
  try {
    console.log('📋 Fetching all service schedules...');
    
    const schedules = await ServiceSchedule.find({ isArchived: false }).sort({ createdAt: -1 });
    
    console.log(`✅ Found ${schedules.length} service schedules`);
    
    res.json({
      success: true,
      count: schedules.length,
      data: schedules
    });
  } catch (error) {
    console.error('❌ Error fetching service schedules:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get service schedules by user ID
app.get('/service-schedules/user/:userId', async (req, res) => {
  try {
    const { userId } = req.params;
    console.log(`📋 Fetching service schedules for user: ${userId}`);
    
    const schedules = await ServiceSchedule.find({ 
      adminUid: userId,
      isArchived: false 
    }).sort({ createdAt: -1 });
    
    console.log(`✅ Found ${schedules.length} service schedules for user ${userId}`);
    
    res.json({
      success: true,
      count: schedules.length,
      data: schedules
    });
  } catch (error) {
    console.error('❌ Error fetching user service schedules:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get single service schedule by ID
app.get('/service-schedules/:id', async (req, res) => {
  try {
    const { id } = req.params;
    console.log(`📋 Fetching service schedule with ID: ${id}`);
    
    const schedule = await ServiceSchedule.findOne({ 
      $or: [
        { id: id },
        { _id: id },
        { scheduleId: id }
      ],
      isArchived: false 
    });
    
    if (!schedule) {
      console.log(`❌ Service schedule not found for ID: ${id}`);
      return res.status(404).json({
        success: false,
        error: 'Service schedule not found'
      });
    }
    
    console.log(`✅ Service schedule found: ${schedule.scheduleId}`);
    
    res.json({
      success: true,
      data: schedule
    });
  } catch (error) {
    console.error('❌ Error fetching service schedule:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Get or create counter
app.get('/service-schedules/counter/get', async (req, res) => {
  try {
    console.log('🔢 Fetching service schedule counter...');
    
    let counter = await Counter.findOne({ _id: 'service_schedule_counter' });
    
    if (!counter) {
      console.log('🔢 Counter not found, creating new one...');
      counter = new Counter({
        _id: 'service_schedule_counter',
        counter: 0
      });
      await counter.save();
    }
    
    console.log(`✅ Current counter value: ${counter.counter}`);
    
    res.json({
      success: true,
      counter: counter.counter
    });
  } catch (error) {
    console.error('❌ Error fetching counter:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Increment counter
app.post('/service-schedules/counter/increment', async (req, res) => {
  try {
    console.log('🔢 Incrementing service schedule counter...');
    
    let counter = await Counter.findOne({ _id: 'service_schedule_counter' });
    
    if (!counter) {
      counter = new Counter({
        _id: 'service_schedule_counter',
        counter: 1
      });
    } else {
      counter.counter += 1;
    }
    
    await counter.save();
    
    console.log(`✅ Counter incremented to: ${counter.counter}`);
    
    res.json({
      success: true,
      counter: counter.counter
    });
  } catch (error) {
    console.error('❌ Error incrementing counter:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Create new service schedule
app.post('/service-schedules', async (req, res) => {
  try {
    console.log('📝 Creating new service schedule...');
    console.log('📊 Request data includes:', {
      scheduleId: req.body.scheduleId,
      title: req.body.title,
      country: req.body.country,
      project: req.body.project,
      equipment: req.body.equipment?.equipmentName || 'N/A',
      tasksCount: req.body.tasks?.length || 0
    });
    
    const scheduleData = {
      _id: req.body.id || Date.now().toString(),
      id: req.body.id || Date.now().toString(),
      adminUid: req.body.adminUid,
      ...req.body,
      createdAt: req.body.createdAt || new Date().toISOString(),
      updatedAt: new Date()
    };
    
    const newSchedule = new ServiceSchedule(scheduleData);
    await newSchedule.save();
    
    console.log('✅ Service schedule created:', newSchedule.scheduleId);
    console.log('📊 Saved record includes:', {
      id: newSchedule.id,
      scheduleId: newSchedule.scheduleId,
      title: newSchedule.title,
      country: newSchedule.country,
      project: newSchedule.project,
      status: newSchedule.status
    });
    
    // Send notifications in background (fire-and-forget)
    setImmediate(async () => {
      try {
        // Get creator name and email from adminUid
        const GlobalUserSchema = new mongoose.Schema({}, { strict: false, collection: 'GlobalUsers' });
        let GlobalUser;
        try {
          GlobalUser = mongoose.model('GlobalUser');
        } catch (error) {
          GlobalUser = mongoose.model('GlobalUser', GlobalUserSchema);
        }
        
        const creator = await GlobalUser.findOne({ _id: scheduleData.adminUid }).select('name email');
        const createdByName = creator?.name || 'Unknown User';
        const creatorEmail = creator?.email || '';
        
        console.log('📧 Sending service schedule notifications for creator:', createdByName);
        await sendServiceScheduleNotifications(newSchedule.toObject(), createdByName, creatorEmail);
      } catch (notifError) {
        console.error('❌ Error sending service schedule notifications:', notifError);
      }
    });
    
    res.json({
      success: true,
      data: newSchedule
    });
  } catch (error) {
    console.error('❌ Error creating service schedule:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Update service schedule
app.put('/service-schedules/:id', async (req, res) => {
  try {
    const { id } = req.params;
    console.log(`📝 Updating service schedule: ${id}`);
    console.log(`📝 Update data received:`, req.body);
    
    const updateData = {
      ...req.body,
      updatedAt: new Date()
    };
    
    // Log if this is a job card update
    if (updateData.jobCardId || updateData.jobCardRequestNumber) {
      console.log('📋 Job card update detected:');
      console.log('  - Job Card ID:', updateData.jobCardId);
      console.log('  - Request Number:', updateData.jobCardRequestNumber);
      console.log('  - Job Card Title:', updateData.jobCardTitle);
      console.log('  - Status:', updateData.jobCardStatus);
      console.log('  - Created At:', updateData.jobCardCreatedAt);
    }
    
    // Remove _id and id from update data to prevent modification
    delete updateData._id;
    delete updateData.id;
    
    const updatedSchedule = await ServiceSchedule.findOneAndUpdate(
      { 
        $or: [
          { id: id },
          { _id: id },
          { scheduleId: id }
        ]
      },
      updateData,
      { new: true }
    );
    
    if (!updatedSchedule) {
      console.log(`❌ Service schedule not found for update: ${id}`);
      return res.status(404).json({
        success: false,
        error: 'Service schedule not found'
      });
    }
    
    console.log('✅ Service schedule updated:', updatedSchedule.scheduleId);
    
    // Log updated job card fields if present
    if (updatedSchedule.jobCardId || updatedSchedule.jobCardRequestNumber) {
      console.log('✅ Job card fields in updated schedule:');
      console.log('  - Job Card ID:', updatedSchedule.jobCardId);
      console.log('  - Request Number:', updatedSchedule.jobCardRequestNumber);
      console.log('  - Job Card Title:', updatedSchedule.jobCardTitle);
      console.log('  - Status:', updatedSchedule.jobCardStatus);
      console.log('  - Created At:', updatedSchedule.jobCardCreatedAt);
    }
    
    res.json({
      success: true,
      data: updatedSchedule
    });
  } catch (error) {
    console.error('❌ Error updating service schedule:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Delete service schedule (soft delete)
app.delete('/service-schedules/:id', async (req, res) => {
  try {
    const { id } = req.params;
    console.log(`🗑️ Deleting service schedule: ${id}`);
    
    const deletedSchedule = await ServiceSchedule.findOneAndUpdate(
      { 
        $or: [
          { id: id },
          { _id: id },
          { scheduleId: id }
        ]
      },
      { isArchived: true, updatedAt: new Date() },
      { new: true }
    );
    
    if (!deletedSchedule) {
      console.log(`❌ Service schedule not found for deletion: ${id}`);
      return res.status(404).json({
        success: false,
        error: 'Service schedule not found'
      });
    }
    
    console.log('✅ Service schedule archived:', deletedSchedule.scheduleId);
    
    res.json({
      success: true,
      message: 'Service schedule deleted successfully',
      data: deletedSchedule
    });
  } catch (error) {
    console.error('❌ Error deleting service schedule:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});

// Test endpoint
app.get('/', (req, res) => {
  res.json({
    status: 'Service Schedule Server is running',
    port: PORT,
    collection: 'GlobalServiceSchdule',
    endpoints: {
      schedules: '/service-schedules',
      userSchedules: '/service-schedules/user/:userId',
      singleSchedule: '/service-schedules/:id',
      counter: '/service-schedules/counter/get',
      incrementCounter: '/service-schedules/counter/increment'
    }
  });
});

// Error handling middleware
app.use((error, req, res, next) => {
  console.error('❌ Unhandled error:', error);
  res.status(500).json({
    success: false,
    error: 'Internal server error'
  });
});

// Start server
const startServer = async () => {
  try {
    // Wait for MongoDB connection first
    await connectDB();
    
    // Wait a bit for connection to stabilize
    await new Promise(resolve => setTimeout(resolve, 2000));
    
    app.listen(PORT, () => {
      console.log(`🚀 Service Schedule Handler server running on port ${PORT}`);
    });
  } catch (error) {
    console.error('❌ Failed to start server:', error);
    process.exit(1);
  }
};

startServer();

module.exports = app;

