const express = require('express');
const mongoose = require('mongoose');
const multer = require('multer');
const crypto = require('crypto');
const { GridFSBucket } = require('mongodb');
const http = require('http');
const socketIo = require('socket.io');
const path = require('path');
const fs = require('fs');

const app = express();
const server = http.createServer(app);
const io = socketIo(server);

const mongoURI = 'mongodb://titandrillingadminuser:mrdThsSnATwD231hbOPmnhj@168.231.113.219:27017/titan_drilling?authSource=admin';
const PORT = process.env.DOCUMENTS_PORT || 5002;
let gfs;

// File upload directory
// Use local path or absolute path from environment variable
const UPLOAD_DIR = process.env.UPLOAD_DIR || path.join(__dirname, 'uploads', 'documents');

// Ensure upload directory exists
if (!fs.existsSync(UPLOAD_DIR)) {
  fs.mkdirSync(UPLOAD_DIR, { recursive: true });
  console.log(`📁 Created upload directory: ${UPLOAD_DIR}`);
} else {
  console.log(`📁 Using upload directory: ${UPLOAD_DIR}`);
}

// Connect to MongoDB
mongoose.connect(mongoURI, { serverSelectionTimeoutMS: 5000 })
  .then(() => console.log('✅ MongoDB connected successfully!'))
  .catch(err => console.error('❌ MongoDB connection error:', err));

// Initialize GridFSBucket after MongoDB connection
mongoose.connection.once('open', () => {
  gfs = new GridFSBucket(mongoose.connection.db, {
    bucketName: 'documents'
  });
  console.log('📦 GridFS bucket initialized');
});

// Define Document Schema - Support both file path and Base64 storage
const DocumentSchema = new mongoose.Schema({
  id: { type: String, required: true, unique: true },
  uid: { type: String, required: true, index: true },
  documentType: { type: String, required: true, index: true },
  type: { type: String, required: true },
  description: { type: String, required: true },
  subfolderName: { type: String, required: true, index: true },
  equipment: { type: Object },
  filePath: { type: String }, // File path on server filesystem (for new uploads)
  fileData: { type: String }, // Base64 data (for legacy documents)
  fileType: { type: String, required: true },
  fileName: { type: String, required: true },
  fileSize: { type: Number },
  mimeType: { type: String, required: true },
  createdAt: { type: Date, default: Date.now, index: true },
  updatedAt: { type: Date, default: Date.now },
  databaseName: { type: String, default: 'titan_drilling' },
}, {
  timestamps: true,
  collection: 'documents'
});

DocumentSchema.index({ uid: 1, documentType: 1, subfolderName: 1 });
DocumentSchema.index({ createdAt: -1 });

const Document = mongoose.model('Document', DocumentSchema);

// Configure Multer to store files on disk with user-specific folders
const storage = multer.diskStorage({
  destination: function (req, file, cb) {
    // For now, store in a temporary directory first
    // We'll move the file to user-specific folder after processing
    const tempDir = path.join(UPLOAD_DIR, 'temp');
    
    // Ensure temp directory exists
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
      console.log(`📁 Created temp directory: ${tempDir}`);
    }
    
    console.log(`📁 Storing file temporarily in: ${tempDir}`);
    cb(null, tempDir);
  },
  filename: function (req, file, cb) {
    // Generate unique filename with timestamp and original extension
    const uniqueSuffix = Date.now() + '-' + Math.round(Math.random() * 1E9);
    const fileExtension = path.extname(file.originalname);
    cb(null, file.fieldname + '-' + uniqueSuffix + fileExtension);
  }
});
const upload = multer({ 
  storage: storage,
  limits: { fileSize: 50 * 1024 * 1024 } // 50MB limit for file uploads
});

// Middleware
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use((err, req, res, next) => {
  if (err instanceof multer.MulterError) {
    console.error('Multer error:', err);
    return res.status(400).json({ error: err.message });
  } else if (err) {
    console.error('Server error:', err);
    return res.status(500).json({ error: 'Server error during upload' });
  }
  next();
});

// Health Check Route
app.get('/', (req, res) => {
  res.json({ status: 'Server is running' });
});

// Main endpoint for all documents - GlobalDocumentData
app.get('/GlobalDocumentData', async (req, res) => {
  try {
    const documents = await Document.find({}).select('-fileData').sort({ createdAt: -1 });
    
    // Add file URLs to each document
    const documentsWithUrls = documents.map(doc => {
      const docObj = doc.toObject();
      return {
        ...docObj,
        file: `http://168.231.113.219:${PORT}/image/${docObj.id}`
      };
    });
    
    console.log(`📄 Retrieved ${documentsWithUrls.length} documents`);
    
    res.json(documentsWithUrls);
  } catch (err) {
    console.error('❌ Error fetching documents:', err);
    res.status(500).json({ error: err.message });
  }
});

// Simple documents list endpoint for browser access
app.get('/documents', async (req, res) => {
  try {
    const totalDocs = await Document.countDocuments();
    const sampleDocs = await Document.find({}).limit(10).select('-filePath -fileData').sort({ createdAt: -1 });
    
    res.json({
      message: 'Documents API is working',
      totalDocuments: totalDocs,
      sampleDocuments: sampleDocs,
      endpoints: {
        'Get all documents': 'GET /GlobalDocumentData',
        'Get user documents': 'GET /Documents/:uid',
        'Get by subfolder': 'GET /Documents/:uid/:documentType/:subfolderName',
        'Upload document': 'POST /Documents/upload',
        'Serve file': 'GET /image/:id',
        'Delete document': 'DELETE /Documents/:uid/:documentType/:subfolderName/:id'
      }
    });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// Debug endpoint to check all documents and UIDs
app.get('/debug/documents', async (req, res) => {
  try {
    const totalDocs = await Document.countDocuments();
    const uniqueUids = await Document.distinct('uid');
    const sampleDocs = await Document.find({}).limit(5).sort({ createdAt: -1 });
    
    res.json({
      totalDocuments: totalDocs,
      uniqueUids: uniqueUids,
      sampleDocuments: sampleDocs.map(doc => ({
        id: doc.id,
        uid: doc.uid,
        documentType: doc.documentType,
        subfolderName: doc.subfolderName,
        fileName: doc.fileName,
        fileSize: doc.fileSize,
        mimeType: doc.mimeType,
        hasFilePath: !!doc.filePath,
        hasFileData: !!doc.fileData,
        storageType: doc.filePath ? 'filesystem' : (doc.fileData ? 'base64' : 'unknown'),
        createdAt: doc.createdAt
      }))
    });
  } catch (err) {
    res.status(500).json({ error: err.message });
  }
});

// Migration endpoint to move existing files to user-specific folders
app.post('/migrate/files-to-user-folders', async (req, res) => {
  try {
    console.log('🔄 Starting file migration to user-specific folders...');
    
    // Find all documents with filePath that are not in user folders
    const documents = await Document.find({ 
      filePath: { $exists: true },
      filePath: { $not: { $regex: /\/[^\/]+\/file-/ } } // Not already in user folder
    });
    
    console.log(`📄 Found ${documents.length} documents to migrate`);
    
    let migratedCount = 0;
    let errorCount = 0;
    
    for (const doc of documents) {
      try {
        const oldPath = doc.filePath;
        
        // Check if old file exists
        if (!fs.existsSync(oldPath)) {
          console.log(`⚠️ File not found: ${oldPath}`);
          continue;
        }
        
        // Create new user-specific directory
        const userDir = path.join(UPLOAD_DIR, doc.uid);
        if (!fs.existsSync(userDir)) {
          fs.mkdirSync(userDir, { recursive: true });
          console.log(`📁 Created user directory: ${userDir}`);
        }
        
        // Generate new filename
        const fileExtension = path.extname(doc.fileName);
        const newFileName = `file-${Date.now()}-${Math.round(Math.random() * 1E9)}${fileExtension}`;
        const newPath = path.join(userDir, newFileName);
        
        // Move file to new location
        fs.renameSync(oldPath, newPath);
        
        // Update document with new path
        doc.filePath = newPath;
        await doc.save();
        
        console.log(`✅ Migrated: ${doc.fileName} -> ${newPath}`);
        migratedCount++;
        
      } catch (error) {
        console.error(`❌ Error migrating document ${doc.id}:`, error.message);
        errorCount++;
      }
    }
    
    res.json({
      success: true,
      message: `Migration completed: ${migratedCount} files migrated, ${errorCount} errors`,
      migrated: migratedCount,
      errors: errorCount,
      total: documents.length
    });
    
  } catch (err) {
    console.error('❌ Migration error:', err);
    res.status(500).json({ error: err.message });
  }
});

// Document Upload Route - Store files on server filesystem
app.post('/Documents/upload', upload.single('file'), async (req, res) => {
  try {
    console.log('📤 Upload request received');
    console.log('📤 Request body:', req.body);
    console.log('📤 Request file:', req.file);
    console.log('📤 Request headers:', req.headers);

    if (!req.file) {
      console.error('❌ No file uploaded');
      return res.status(400).json({ error: 'No file uploaded' });
    }

    // Validate UID is provided
    if (!req.body.uid) {
      console.error('❌ No UID provided');
      return res.status(400).json({ error: 'User UID is required' });
    }

    console.log('📤 Uploading file:', req.file.originalname, 'Size:', req.file.size, 'bytes');
    console.log('👤 User UID:', req.body.uid);
    console.log('📁 File saved to temp:', req.file.path);
    
    // Create user-specific directory
    const userDir = path.join(UPLOAD_DIR, req.body.uid);
    if (!fs.existsSync(userDir)) {
      fs.mkdirSync(userDir, { recursive: true });
      console.log(`📁 Created user directory: ${userDir}`);
    }
    
    // Move file from temp to user directory
    const finalFileName = path.basename(req.file.path);
    let finalPath = path.join(userDir, finalFileName);
    
    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to user directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to user directory:', moveError);
      // If move fails, use the temp path
      finalPath = req.file.path;
    }
    
    const documentData = {
      id: Date.now().toString(),
      uid: req.body.uid, // Now guaranteed to exist
      documentType: req.body.documentType || 'general',
      type: req.body.type || 'image',
      description: req.body.description || '',
      subfolderName: req.body.subfolderName || 'default',
      equipment: req.body.equipment ? JSON.parse(req.body.equipment) : null,
      filePath: finalPath, // Final file path in user directory
      fileType: path.extname(req.file.originalname).slice(1),
      fileName: req.file.originalname,
      fileSize: req.file.size,
      mimeType: req.file.mimetype,
      databaseName: 'titan_drilling',
    };
    
    console.log('📝 Saving document metadata:', {
      id: documentData.id,
      uid: documentData.uid,
      documentType: documentData.documentType,
      subfolderName: documentData.subfolderName,
      fileName: documentData.fileName,
      fileSize: documentData.fileSize,
      filePath: documentData.filePath
    });
    
    const newDocument = new Document(documentData);
    await newDocument.save();
    
    console.log('✅ File saved to server filesystem:', newDocument.id);
    console.log('📁 File location:', newDocument.filePath);
    
    io.emit('document:added', newDocument);
    
    res.json({
      success: true,
      message: '✅ File uploaded successfully to server filesystem!',
      data: {
        id: newDocument.id,
        uid: newDocument.uid,
        documentType: newDocument.documentType,
        type: newDocument.type,
        description: newDocument.description,
        subfolderName: newDocument.subfolderName,
        equipment: newDocument.equipment,
        fileName: newDocument.fileName,
        fileType: newDocument.fileType,
        fileSize: newDocument.fileSize,
        mimeType: newDocument.mimeType,
        createdAt: newDocument.createdAt,
        file: `http://168.231.113.219:${PORT}/image/${newDocument.id}`
      },
    });
  } catch (err) {
    console.error('❌ Error uploading document:', err);
    console.error('❌ Error stack:', err.stack);
    
    // Handle specific error types
    if (err.code === 'LIMIT_FILE_SIZE') {
      return res.status(400).json({ error: 'File too large. Maximum size is 50MB.' });
    }
    
    if (err.code === 'ENOENT') {
      return res.status(500).json({ error: 'File system error: Directory not found' });
    }
    
    if (err.name === 'ValidationError') {
      return res.status(400).json({ error: `Validation error: ${err.message}` });
    }
    
    res.status(500).json({ 
      error: 'Internal server error during upload',
      details: process.env.NODE_ENV === 'development' ? err.message : 'Upload failed'
    });
  }
});

// Get all documents for a user
app.get('/Documents/:uid', async (req, res) => {
  try {
    const { uid } = req.params;
    console.log(`🔍 Searching for documents with UID: ${uid}`);
    
    let documents = await Document.find({ uid }).select('-filePath').sort({ createdAt: -1 });
    console.log(`📄 Found ${documents.length} documents for user ${uid}`);
    
    // If no documents found for this UID, try to find documents for any UID (fallback)
    if (documents.length === 0) {
      console.log(`⚠️ No documents found for UID ${uid}, trying fallback...`);
      documents = await Document.find({}).select('-filePath').sort({ createdAt: -1 }).limit(10);
      console.log(`📄 Fallback: Found ${documents.length} total documents`);
    }
    
    // Add file URLs (no filePath in list view for performance)
    const documentsWithUrls = documents.map(doc => {
      const docObj = doc.toObject();
      return {
        ...docObj,
        file: `http://168.231.113.219:${PORT}/image/${docObj.id}`
      };
    });
    
    res.json({
      success: true,
      count: documentsWithUrls.length,
      data: documentsWithUrls,
      searchedUid: uid,
      actualUids: documents.length > 0 ? [...new Set(documents.map(d => d.uid))] : []
    });
  } catch (err) {
    console.error('❌ Error fetching documents:', err);
    res.status(500).json({ error: err.message });
  }
});

// Get documents by subfolder
app.get('/Documents/:uid/:documentType/:subfolderName', async (req, res) => {
  try {
    const { uid, documentType, subfolderName } = req.params;
    const documents = await Document.find({
      uid,
      documentType,
      subfolderName,
    }).select('-filePath').sort({ createdAt: -1 });
    
    console.log(`📂 Found ${documents.length} documents in ${documentType}/${subfolderName} for user ${uid}`);
    
    // Add file URLs
    const documentsWithUrls = documents.map(doc => {
      const docObj = doc.toObject();
      return {
        ...docObj,
        file: `http://168.231.113.219:${PORT}/image/${docObj.id}`
      };
    });
    
    res.json({
      success: true,
      count: documentsWithUrls.length,
      data: documentsWithUrls,
    });
  } catch (err) {
    console.error('❌ Error fetching documents:', err);
    res.status(500).json({ error: err.message });
  }
});

// Delete document
app.delete('/Documents/:uid/:documentType/:subfolderName/:id', async (req, res) => {
  try {
    const { uid, documentType, subfolderName, id } = req.params;
    
    const document = await Document.findOneAndDelete({
      id,
      uid,
      documentType,
      subfolderName,
    });
    
    if (!document) {
      return res.status(404).json({ error: 'Document not found' });
    }
    
    // Delete the actual file from filesystem
    if (document.filePath && fs.existsSync(document.filePath)) {
      fs.unlinkSync(document.filePath);
      console.log(`🗑️ File deleted from filesystem: ${document.filePath}`);
    }
    
    console.log(`🗑️ Document deleted: ${id} (File and metadata removed)`);
    io.emit('document:deleted', { id });
    
    res.json({
      success: true,
      message: '✅ Document and file deleted completely!',
    });
  } catch (err) {
    console.error('❌ Error deleting document:', err);
    res.status(500).json({ error: err.message });
  }
});

// Serve Files by Document ID - Support both filesystem and Base64 storage
app.get('/documents/:id', async (req, res) => {
  try {
    const documentId = req.params.id;
    const isDownload = req.query.download === 'true';
    console.log(`📷 Requesting document: ${documentId}, Download: ${isDownload}`);
    
    // Find document by ID (using the string id field, not MongoDB's _id)
    const document = await Document.findOne({ id: documentId });
    
    if (!document) {
      return res.status(404).json({ error: 'Document not found' });
    }

    console.log(`📤 Serving file: ${document.fileName}, Size: ${document.fileSize} bytes, Type: ${document.mimeType}`);
    
    // Set proper headers for file serving
    res.set('Content-Type', document.mimeType);
    res.set('Cache-Control', 'public, max-age=3600');
    res.set('Access-Control-Allow-Origin', '*');
    res.set('Content-Length', document.fileSize);
    
    // If this is a download request, set attachment header
    if (isDownload) {
      res.set('Content-Disposition', `attachment; filename="${document.fileName}"`);
      console.log(`⬇️ Forcing download: ${document.fileName}`);
    }
    
    // Check if document has filePath (new filesystem storage) or fileData (legacy Base64)
    if (document.filePath && fs.existsSync(document.filePath)) {
      // New filesystem storage
      console.log(`📁 Serving from filesystem: ${document.filePath}`);
      res.sendFile(document.filePath);
    } else if (document.fileData) {
      // Legacy Base64 storage
      console.log(`📦 Serving from Base64 data (legacy)`);
      const fileBuffer = Buffer.from(document.fileData, 'base64');
      res.send(fileBuffer);
    } else {
      console.error(`❌ No file data found for document: ${documentId}`);
      return res.status(404).json({ error: 'File data not found' });
    }
    
  } catch (err) {
    console.error('❌ Error serving document:', err);
    res.status(500).json({ error: err.message });
  }
});

// Alternative endpoint for image serving - Support both filesystem and Base64
app.get('/image/:id', async (req, res) => {
  try {
    const documentId = req.params.id;
    console.log(`🖼️ Requesting image: ${documentId}`);
    
    const document = await Document.findOne({ id: documentId });
    
    if (!document) {
      return res.status(404).json({ error: 'Image not found' });
    }

    console.log(`📤 Serving image: ${document.fileName}, Size: ${document.fileSize} bytes`);
    
    // Set image headers
    res.set('Content-Type', document.mimeType || 'image/jpeg');
    res.set('Cache-Control', 'public, max-age=3600');
    res.set('Access-Control-Allow-Origin', '*');
    
    // Check if document has filePath (new filesystem storage) or fileData (legacy Base64)
    if (document.filePath && fs.existsSync(document.filePath)) {
      // New filesystem storage
      console.log(`📁 Serving image from filesystem: ${document.filePath}`);
      res.sendFile(document.filePath);
    } else if (document.fileData) {
      // Legacy Base64 storage
      console.log(`📦 Serving image from Base64 data (legacy)`);
      const imageBuffer = Buffer.from(document.fileData, 'base64');
      res.send(imageBuffer);
    } else {
      console.error(`❌ No image data found for document: ${documentId}`);
      return res.status(404).json({ error: 'Image data not found' });
    }
    
  } catch (err) {
    console.error('❌ Error serving image:', err);
    res.status(500).json({ error: err.message });
  }
});

// User upload endpoint for MongoDB - Store in GlobalUsers collection
app.post('/users/upload', async (req, res) => {
  try {
    const { userId, userData } = req.body;
    
    if (!userId || !userData) {
      return res.status(400).json({ 
        success: false, 
        error: 'userId and userData are required' 
      });
    }

    console.log(`📤 Uploading user: ${userData.name} (${userId})`);

    // Create or update user in MongoDB - Store in GlobalUsers collection
    const User = mongoose.model('GlobalUser', new mongoose.Schema({
      _id: { type: String, required: true },
      uid: { type: String, required: true },
      name: { type: String, required: true },
      email: { type: String, required: true },
      employeeNumber: { type: String },
      department: { type: String },
      countries: [String],
      modules: [Object],
      notifications: Object,
      loginStatus: { type: Boolean, default: false },
      lastLogin: { type: Date },
      authCreated: { type: Boolean, default: false },
      uploadTimestamp: { type: Date, default: Date.now },
      originalUserId: { type: String },
      // Store all other fields dynamically
      [String]: mongoose.Schema.Types.Mixed
    }, { 
      strict: false,
      collection: 'GlobalUsers' // Store in GlobalUsers collection
    }));

    // Use upsert to create or update
    const result = await User.findByIdAndUpdate(
      userId,
      { ...userData, _id: userId },
      { upsert: true, new: true }
    );

    console.log(`✅ User saved to MongoDB GlobalUsers: ${result.name} (${result._id})`);

    res.json({
      success: true,
      message: `User ${userData.name} uploaded successfully`,
      userId: result._id,
      data: result
    });

  } catch (err) {
    console.error('❌ Error uploading user:', err);
    res.status(500).json({ 
      success: false, 
      error: err.message 
    });
  }
});

// Team Employee Documents Upload Endpoint
app.post('/team-employee/upload', upload.single('file'), async (req, res) => {
  try {
    console.log('📤 Team Employee document upload request received');
    console.log('Request body:', req.body);
    console.log('File info:', req.file ? {
      originalname: req.file.originalname,
      mimetype: req.file.mimetype,
      size: req.file.size,
      path: req.file.path
    } : 'No file');

    if (!req.file) {
      return res.status(400).json({
        success: false,
        error: 'No file uploaded'
      });
    }

    const { uid, field, userId } = req.body;

    if (!uid || !field) {
      return res.status(400).json({
        success: false,
        error: 'uid and field are required'
      });
    }

    // Create team employee documents directory structure
    const teamEmployeeDir = path.join(UPLOAD_DIR, 'TeamEmployeeDocx', uid);
    if (!fs.existsSync(teamEmployeeDir)) {
      fs.mkdirSync(teamEmployeeDir, { recursive: true });
      console.log(`📁 Created team employee directory: ${teamEmployeeDir}`);
    }

    // Move file from temp to team employee directory
    const finalFileName = path.basename(req.file.path);
    const finalPath = path.join(teamEmployeeDir, finalFileName);

    try {
      fs.renameSync(req.file.path, finalPath);
      console.log(`📁 File moved to team employee directory: ${finalPath}`);
    } catch (moveError) {
      console.error('❌ Error moving file to team employee directory:', moveError);
      return res.status(500).json({
        success: false,
        error: 'Failed to save file'
      });
    }

    // Create document record
    const documentData = {
      uid: uid,
      userId: userId || uid,
      fileName: req.file.originalname,
      filePath: finalPath,
      fileType: path.extname(req.file.originalname).toLowerCase(),
      mimeType: req.file.mimetype,
      field: field, // e.g., 'profilePic', 'medicalDocs', 'driverLicenseDocs'
      uploadTimestamp: new Date(),
      documentType: 'TeamEmployee',
      subfolderName: field
    };

    const newDocument = new Document(documentData);
    await newDocument.save();

    console.log(`✅ Team employee document saved: ${newDocument.fileName} (${newDocument.id})`);

    res.json({
      success: true,
      message: 'Team employee document uploaded successfully',
      document: {
        id: newDocument.id,
        fileName: newDocument.fileName,
        field: newDocument.field,
        fileType: newDocument.fileType,
        mimeType: newDocument.mimeType,
        file: `http://168.231.113.219:${PORT}/image/${newDocument.id}`,
        uploadTimestamp: newDocument.uploadTimestamp
      }
    });

  } catch (err) {
    console.error('❌ Error uploading team employee document:', err);
    res.status(500).json({
      success: false,
      error: 'Internal server error during team employee document upload',
      details: process.env.NODE_ENV === 'development' ? err.message : 'Upload failed'
    });
  }
});

// Get Team Employee Documents
app.get('/team-employee/:uid', async (req, res) => {
  try {
    const { uid } = req.params;
    console.log(`📁 Fetching team employee documents for UID: ${uid}`);

    const documents = await Document.find({
      uid: uid,
      documentType: 'TeamEmployee'
    }).sort({ uploadTimestamp: -1 });

    console.log(`📁 Found ${documents.length} team employee documents for UID: ${uid}`);

    const documentsWithUrls = documents.map(doc => ({
      id: doc.id,
      fileName: doc.fileName,
      field: doc.field,
      fileType: doc.fileType,
      mimeType: doc.mimeType,
      file: `http://168.231.113.219:${PORT}/image/${doc.id}`,
      uploadTimestamp: doc.uploadTimestamp
    }));

    res.json({
      success: true,
      count: documents.length,
      data: documentsWithUrls,
      timestamp: new Date().toISOString()
    });

  } catch (err) {
    console.error('❌ Error fetching team employee documents:', err);
    res.status(500).json({
      success: false,
      error: err.message
    });
  }
});

// Start Server
server.listen(PORT, '0.0.0.0', () => {
  console.log(`🚀 Server running on port ${PORT}`);
  console.log(`📡 Server URL: http://168.231.113.219:${PORT}`);
  console.log(`🔗 Health check: http://168.231.113.219:${PORT}/`);
  console.log('⚡ Socket.IO: Enabled for real-time updates');
  console.log(`👥 User collection: GlobalUsers`);
});