
import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  ScrollView,
  Dimensions,
  Alert,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AntDesign from 'react-native-vector-icons/AntDesign';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const HSE_131Step2 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [checklistData, setChecklistData] = useState({});
  const [expandedSections, setExpandedSections] = useState({
    webbing: false,
    metalFitting: false,
    lanyard: false,
  });

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedStep2 = await AsyncStorage.getItem('hseStep2');
      if (savedStep2) {
        const step2Data = JSON.parse(savedStep2);
        setChecklistData(step2Data.checklistData || {});
      }
    } catch (error) {
      console.error('Error loading Step 2 data:', error);
    }
  };

  const saveStep2Data = async () => {
    try {
      const step2Data = {
        checklistData,
        timestamp: new Date().toISOString(),
      };
      await AsyncStorage.setItem('hseStep2', JSON.stringify(step2Data));
    } catch (error) {
      console.error('Error saving Step 2 data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data.',
        position: 'bottom',
      });
    }
  };

  const handleSelection = (category, item, value) => {
    setChecklistData(prev => ({
      ...prev,
      [category]: {
        ...prev[category],
        [item]: value
      }
    }));
    saveStep2Data();
  };

  const toggleSection = (section) => {
    setExpandedSections(prev => ({
      ...prev,
      [section]: !prev[section]
    }));
  };

  const validateForm = () => {
    // Check if at least one item in each category has been answered
    const webbingItems = ['corrosionOrDamage', 'safetyPinInPlace', 'antiTamperSeal', 'perishedOrBlockedHose'];
    const metalFittingItems = ['leakingGlandsOrSeals', 'perishedOrBlockedHose', 'nozzleMissingOrBlocked'];

    const hasWebbingAnswer = webbingItems.some(item => checklistData.webbing?.[item]);
    const hasMetalFittingAnswer = metalFittingItems.some(item => checklistData.metalFitting?.[item]);
    return hasWebbingAnswer || hasMetalFittingAnswer;
  };

  const handleNext = async () => {
    if (!validateForm()) {
      Toast.show({
        type: 'error',
        text1: 'Validation Error',
        text2: 'Please answer at least one item in each category.',
        position: 'bottom',
      });
      return;
    }

    await saveStep2Data();
    setActiveStep(3);
  };

  const handleBackPress = () => {
    Alert.alert(
      'Go Back',
      'Do you want to return to the previous step? Unsaved changes will be saved locally.',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'OK',
          onPress: async () => {
            await saveStep2Data();
            setActiveStep(1);
          },
        },
      ],
      { cancelable: false }
    );
  };

  const renderChecklistItem = (category, itemKey, itemLabel, index) => {
    const currentValue = checklistData[category]?.[itemKey] || '';
    
    return (
      <View key={itemKey} style={styles.centeredItem}>
        <View style={styles.itemContainer}>
          <Text style={styles.itemText}>
            {index + 1}. {itemLabel}
          </Text>
          <View style={styles.itemRow}>
            <View style={styles.optionsContainer}>
              <TouchableOpacity
                style={[
                  styles.optionButton,
                  currentValue === 'Yes' && styles.selectedOptionButton,
                  currentValue === 'Yes' && styles.okButton,
                ]}
                onPress={() => handleSelection(category, itemKey, 'Yes')}
              >
                <Text style={[
                  styles.optionText,
                  currentValue === 'Yes' && styles.optionTextSelected,
                ]}>
                  Yes
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.optionButton,
                  currentValue === 'No' && styles.selectedOptionButton,
                  currentValue === 'No' && styles.rejectButton,
                ]}
                onPress={() => handleSelection(category, itemKey, 'No')}
              >
                <Text style={[
                  styles.optionText,
                  currentValue === 'No' && styles.optionTextSelected,
                ]}>
                  No
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.optionButton,
                  currentValue === 'N/A' && styles.selectedOptionButton,
                  currentValue === 'N/A' && styles.naButton,
                ]}
                onPress={() => handleSelection(category, itemKey, 'N/A')}
              >
                <Text style={[
                  styles.optionText,
                  currentValue === 'N/A' && styles.optionTextSelected,
                ]}>
                  N/A
                </Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
        <View style={styles.itemBorder} />
      </View>
    );
  };

  const renderSection = (sectionKey, sectionTitle, items) => {
    const isExpanded = expandedSections[sectionKey];
    
    return (
      <View key={sectionKey}>
        <TouchableOpacity
          style={styles.sectionHeader}
          onPress={() => toggleSection(sectionKey)}
        >
          <View style={styles.sectionHeaderContent}>
            <AntDesign
              name={isExpanded ? 'caretup' : 'caretdown'}
              size={20}
              color={isExpanded ? 'red' : 'black'}
            />
            <Text style={styles.sectionHeaderText}>{sectionTitle}</Text>
          </View>
        </TouchableOpacity>
        {isExpanded && (
          <View style={styles.sectionContent}>
            {items.map(([itemKey, itemLabel], index) => 
              renderChecklistItem(sectionKey, itemKey, itemLabel, index)
            )}
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.container}>
      <Text style={styles.checklistTitle}>Safety Harness Inspection</Text>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <View style={styles.headingContainer}>
          <Text style={styles.headingText}>
            Tap on section to expand
          </Text>
        </View>

        {renderSection('webbing', 'Fire Extiguishers', [
          ['corrosionOrDamage', 'CORROSION or DAMAGE'],
          ['safetyPinInPlace', 'SAFETY PIN IN PLACE'],
          ['antiTamperSeal', 'ANTI-TAMPER SEAL'],
          ['perishedOrBlockedHose', 'PERISHED/BLOCKED HOSE']
        ])}

        {renderSection('metalFitting', 'Fire Hosereels', [
          ['leakingGlandsOrSeals', 'LEAKING GLANDS or SEALS'],
          ['perishedOrBlockedHose', 'PERISHED/BLOCKED HOSE'],
          ['nozzleMissingOrBlocked', 'NOZZLE MISSING/BLOCKED']
        ])}

        <View style={styles.infoBox}>
          <Text style={styles.infoParagraph}>
            If all in good order, enter "ok" in the appropriate space. If defects are observed enter defect code instead. Defective items must be tagged "UNSAFE FOR USE" and must be repaired promptly.
          </Text>
          <Text style={styles.codesHeader}>List of Defect Codes</Text>
          <Text style={styles.codeLine}>CD = Cylinder Damaged/Corrossion</Text>
          <Text style={styles.codeLine}>HD = Hose Damaged</Text>
          <Text style={styles.codeLine}>SP = Safety Pin or Seal Missing</Text>
          <Text style={styles.codeLine}>DP = Depressuried</Text>
          <Text style={styles.codeLine}>LK = leaking Gland or seal</Text>
          <Text style={styles.codeLine}>NZ = Missing or blocked nozzle</Text>
          <Text style={styles.codeLine}>OVD = Service date overdue</Text>
          <Text style={styles.footerNote}>Fire CO-ORDINATOR TO INITIAL & DATE</Text>
        </View>
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBackPress}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Cancel</Text>
        </TouchableOpacity>
        
        <TouchableOpacity
          style={styles.nextButton}
          onPress={handleNext}
        >
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>
      
      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  headingContainer: {
    paddingVertical: 10,
    marginHorizontal: '5%',
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  rejectButton: {
    backgroundColor: '#FF4B2B',
  },
  naButton: {
    backgroundColor: '#FFA500',
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    marginTop: 10,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  checklistTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginTop: 18,
    marginBottom: 4,
  },
  infoBox: {
    marginHorizontal: '5%',
    marginTop: 10,
    padding: 12,
    backgroundColor: '#F9FBFD',
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#E2ECF5',
  },
  infoParagraph: {
    fontSize: 14,
    color: '#333',
    lineHeight: 20,
    marginBottom: 8,
  },
  codesHeader: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginTop: 4,
    marginBottom: 6,
  },
  codeLine: {
    fontSize: 14,
    color: '#333',
    lineHeight: 20,
  },
  footerNote: {
    fontSize: 14,
    color: '#333',
    marginTop: 10,
    fontWeight: '600',
  },
});

export default HSE_131Step2;