@extends('welcome')

@section('title', 'Vehicles Inventory')

<meta name="csrf-token" content="{{ csrf_token() }}">

@php
    function getStatusColor($status) {
        $status = strtolower($status);
        switch ($status) {
            case 'available':
            case 'active':
            case 'operational':
                return 'success';
            case 'in maintenance':
            case 'maintenance required':
            case 'servicing':
                return 'warning';
            case 'breakdown':
            case 'out of service':
            case 'repair':
                return 'danger';
            case 'deployed':
            case 'in use':
                return 'primary';
            case 'idle':
            case 'standby':
                return 'info';
            case 'discontinue':
            case 'retired':
                return 'secondary';
            default:
                return 'info';
        }
    }

    function getPriorityColor($priority) {
        $priority = strtolower($priority);
        switch ($priority) {
            case 'high':
            case 'urgent':
            case 'emergency':
                return 'danger';
            case 'medium':
                return 'warning';
            case 'low':
                return 'success';
            case 'normal':
                return 'info';
            default:
                return 'secondary';
        }
    }
@endphp

@section('content')
<div class="vehicles-inventory-container">
    <div class="vehicles-inventory-header">
        <div class="header-content">
            <div class="page-title">
                <h1>Vehicles Inventory</h1>
                <p>Manage your vehicle fleet, track status, and monitor equipment efficiently</p>
            </div>
        </div>
    </div>
    
    <div class="vehicles-inventory-content">
        <div class="content-box">
            <div class="action-bar">
                <div class="action-content">
                    <div class="page-title">
                        <h1>Vehicles Inventory</h1>
                    </div>
                    <div class="action-buttons">
                        <div class="search-section">
                            <div class="search-container">
                                <svg class="search-icon" xmlns="http://www.w3.org/2000/svg" width="20" height="20" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="11" cy="11" r="8"/>
                                    <path d="m21 21-4.35-4.35"/>
                                </svg>
                                <input type="text" class="search-input" placeholder="Search vehicles..." id="search-input">
                            </div>
                        </div>
                        <div class="filter-section">
                            <select class="status-filter" id="statusFilter">
                                <option value="all">All Status</option>
                                <option value="active">Active</option>
                                <option value="maintenance required">Maintenance Required</option>
                                <option value="in maintenance">In Maintenance</option>
                                <option value="breakdown">Breakdown</option>
                                <option value="idle">Idle</option>
                            </select>
                        </div>
                        <a href="{{ route('vehicles.add-vehicle') }}" class="btn-create">
                            <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                <path d="M12 5v14m-7-7h14"/>
                            </svg>
                            Add New Vehicle
                        </a>
                    </div>
                </div>
            </div>
            <div class="table-section">
                <div class="table-container">
                    <table class="vehicles-inventory-table">
                        <thead class="table-header">
                            <tr>
                                <th>S.No</th>
                                <th>Action</th>
                                <th>Vehicle Number</th>
                                <th>Vehicle Name</th>
                                <th>Category</th>
                                <th>Model</th>
                                <th>Status</th>
                                <th>Meter Reading</th>
                                <th>Site</th>
                                <th>Operator</th>
                            </tr>
                        </thead>
                        <tbody class="table-body" id="equipmentTableBody">
                            <tr class="loading-row">
                                <td colspan="10">
                                    <div class="loading-content">
                                        <div class="spinner"></div>
                                        <h3>Loading Equipment...</h3>
                                        <p>Please wait while we fetch equipment data from MongoDB.</p>
                                    </div>
                                </td>
                            </tr>
                        </tbody>
                    </table>
                </div>
            </div>
            
            <!-- Pagination Section (Dynamic) -->
            <div class="pagination-section" id="paginationSection" style="display: none;">
                <div class="pagination-content">
                    <div class="pagination-info">
                        <span class="pagination-text" id="paginationInfo">
                            Showing 0 to 0 of 0 entries
                        </span>
                    </div>
                    
                    <div class="pagination-controls" id="paginationControls">
                        <!-- Pagination controls will be dynamically generated by JavaScript -->
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>


<style>
.custom-modal {
    position: fixed;
    top: 0;
    left: 0;
    width: 100%;
    height: 100%;
    background-color: rgba(0, 0, 0, 0.5);
    z-index: 9999;
    display: flex;
    justify-content: center;
    align-items: flex-start;
    padding-top: 120px; /* Increased padding to clear navbar completely */
}

.custom-modal-content {
    background: white;
    border-radius: 8px;
    width: 95%;
    max-width: 1200px;
    max-height: 75vh;
    box-shadow: 0 4px 20px rgba(0, 0, 0, 0.3);
    display: flex;
    flex-direction: column;
    overflow: hidden;
    margin: 0 auto;
}

.custom-modal-header {
    padding: 20px;
    border-bottom: 1px solid #dee2e6;
    display: flex;
    justify-content: space-between;
    align-items: center;
    background-color: #007bff;
    color: white;
    border-radius: 8px 8px 0 0;
}

.custom-modal-title {
    margin: 0;
    color: white;
    font-size: 1.5rem;
}

.custom-modal-close {
    background: none;
    border: none;
    font-size: 24px;
    cursor: pointer;
    color: white;
    padding: 0;
    width: 30px;
    height: 30px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 50%;
    transition: background-color 0.2s;
}

.custom-modal-close:hover {
    background-color: rgba(255, 255, 255, 0.2);
}

.custom-modal-body {
    padding: 20px;
    overflow-y: auto;
    flex: 1;
}

.custom-modal-footer {
    padding: 20px;
    border-top: 1px solid #dee2e6;
    display: flex;
    justify-content: space-between;
    background-color: #f8f9fa;
    border-radius: 0 0 8px 8px;
}

.form-columns {
    display: flex;
    gap: 30px;
    margin-bottom: 20px;
}

.left-column, .right-column {
    flex: 1;
}

.form-group {
    margin-bottom: 15px;
    display: flex;
    flex-direction: column;
}

.form-group label {
    font-weight: 500;
    color: #333;
    margin-bottom: 8px;
    font-size: 14px;
    display: block;
}

.form-control, .form-select {
    border: 1px solid #ced4da;
    border-radius: 4px;
    padding: 10px 12px;
    font-size: 14px;
    transition: border-color 0.2s;
    width: 100%;
    box-sizing: border-box;
}

.form-control:focus, .form-select:focus {
    outline: none;
    border-color: #007bff;
    box-shadow: 0 0 0 0.2rem rgba(0, 123, 255, 0.25);
}

.meter-input-group {
    display: flex;
    gap: 10px;
}

.meter-input-group .form-control {
    flex: 1;
}

.meter-input-group .form-select {
    width: 120px;
}

.form-check-input {
    margin-top: 8px;
    margin-left: 0;
}

.form-check-input:checked {
    background-color: #007bff;
    border-color: #007bff;
}

textarea.form-control {
    min-height: 80px;
    resize: vertical;
}

.assign-inspection {
    margin-top: 30px;
    border-top: 1px solid #dee2e6;
    padding-top: 20px;
}

.inspection-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 20px;
}

.inspection-header h3 {
    margin: 0;
    font-size: 1.2rem;
    color: #333;
    font-weight: 600;
}

.checklist-btn {
    background-color: #007bff;
    color: white;
    border: none;
    padding: 8px 16px;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    display: flex;
    align-items: center;
    gap: 8px;
    transition: background-color 0.2s;
}

.checklist-btn:hover {
    background-color: #0056b3;
}

.inspection-content {
    background-color: #f8f9fa;
    padding: 20px;
    border-radius: 8px;
    text-align: center;
    min-height: 60px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.empty-data {
    color: #6c757d;
    font-style: italic;
    margin: 0;
}

.upload-btn {
    background-color: #007bff;
    color: white;
    border: none;
    padding: 8px 16px;
    border-radius: 4px;
    cursor: pointer;
    font-size: 14px;
    display: flex;
    align-items: center;
    gap: 8px;
    transition: background-color 0.2s;
}

.upload-btn:hover {
    background-color: #0056b3;
}

.add-vehicle-btn {
    background-color: #007bff;
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 4px;
    cursor: pointer;
    font-size: 16px;
    display: flex;
    align-items: center;
    gap: 8px;
    transition: background-color 0.2s;
}

.add-vehicle-btn:hover {
    background-color: #0056b3;
}

.cancel-btn {
    background-color: #6c757d;
    color: white;
    border: none;
    padding: 12px 24px;
    border-radius: 4px;
    cursor: pointer;
    font-size: 16px;
    display: flex;
    align-items: center;
    gap: 8px;
    transition: background-color 0.2s;
}

.cancel-btn:hover {
    background-color: #545b62;
}

.custom-modal-footer {
    padding: 20px;
    border-top: 1px solid #dee2e6;
    display: flex;
    justify-content: space-between;
    background-color: #f8f9fa;
    border-radius: 0 0 8px 8px;
}

/* Dropdown Styles */
.dropdown {
    position: relative;
}

.action-btn {
    display: inline-flex;
    align-items: center;
    gap: 6px;
    padding: 8px 16px;
    background: #0055a5;
    color: white;
    border: none;
    border-radius: 6px;
    font-size: 14px;
    font-weight: 500;
    cursor: pointer;
    transition: all 0.2s;
    font-family: 'Jost', sans-serif;
}

.action-btn:hover {
    background: #003d7a;
    transform: translateY(-1px);
}

.dropdown-menu {
    position: absolute;
    top: 100%;
    left: 0;
    background: white;
    border: 1px solid #d1d5db;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.15);
    border-radius: 8px;
    padding: 8px 0;
    min-width: 200px;
    z-index: 1000;
    list-style: none;
    margin: 5px 0 0 0;
    display: none;
    opacity: 0;
    transform: translateY(-10px);
    transition: all 0.2s ease;
}

.dropdown-menu.show {
    display: block;
    opacity: 1;
    transform: translateY(0);
}

.dropdown-item {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 10px 16px;
    color: #374151;
    text-decoration: none;
    font-size: 13px;
    font-weight: 400;
    transition: all 0.2s;
    cursor: pointer;
}

.dropdown-item:hover {
    background: #f3f4f6;
    color: #1a202c;
}

.dropdown-item.danger {
    color: #dc2626;
}

.dropdown-item.danger:hover {
    background: #fef2f2;
    color: #b91c1c;
}

.dropdown-divider {
    height: 1px;
    background: #e5e7eb;
    margin: 8px 0;
}

/* Status Filter */
.filter-section {
    margin-right: 10px;
}

.status-filter {
    padding: 10px 16px;
    border: 1px solid #d1d5db;
    border-radius: 8px;
    background: white;
    font-size: 14px;
    font-weight: 500;
    color: #374151;
    cursor: pointer;
    transition: all 0.2s;
    font-family: 'Jost', sans-serif;
    min-width: 180px;
}

.status-filter:hover {
    border-color: #0055a5;
}

/* Loading Spinner */
.loading-content {
    display: flex;
    flex-direction: column;
    align-items: center;
    justify-content: center;
    padding: 60px 20px;
}

.spinner {
    width: 50px;
    height: 50px;
    border: 4px solid #e2e8f0;
    border-top: 4px solid #0055a5;
    border-radius: 50%;
    animation: spin 1s linear infinite;
    margin-bottom: 20px;
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.loading-content h3 {
    color: #1a202c;
    margin: 0 0 10px 0;
}

.loading-content p {
    color: #6b7280;
    margin: 0;
}
</style>

<script>
    // MongoDB Equipment Loading with Pagination and Filter
    let currentPage = 1;
    let totalPages = 0;
    let totalRecords = 0;
    let perPage = 20;
    let currentStatus = 'all';
    let allEquipment = [];
    
    // Load equipment from MongoDB
    async function loadEquipment(page = 1, status = 'all') {
        try {
            console.log(`🔄 Loading equipment from MongoDB (Page ${page}, Status: ${status})...`);
            
            const response = await fetch(`/api/equipment/all?page=${page}&limit=${perPage}&status=${status}`);
            const result = await response.json();
            
            console.log('MongoDB Equipment API response:', result);
            
            if (result.success && result.data) {
                allEquipment = result.data;
                
                // Update pagination state
                if (result.pagination) {
                    currentPage = result.pagination.currentPage;
                    totalPages = result.pagination.totalPages;
                    totalRecords = result.pagination.totalRecords;
                }
                
                console.log(`✅ Loaded ${allEquipment.length} equipment items (Page ${currentPage} of ${totalPages})`);
                
                renderEquipmentTable();
                renderPagination();
            } else {
                showEmptyState();
                hidePagination();
            }
        } catch (error) {
            console.error('❌ Error loading equipment:', error);
            showEmptyState();
            hidePagination();
        }
    }
    
    // Render equipment table
    function renderEquipmentTable() {
        const tbody = document.getElementById('equipmentTableBody');
        if (!tbody) return;
        
        if (allEquipment.length === 0) {
            showEmptyState();
            return;
        }
        
        // Calculate serial number in reverse (newest = highest number)
        // Page 1: 585, 584, 583... (if total is 585)
        // Page 2: 565, 564, 563...
        
        tbody.innerHTML = allEquipment.map((eq, index) => {
            const serialNumber = totalRecords - ((currentPage - 1) * perPage) - index;
            const statusColor = getStatusBadgeClass(eq.status);
            const equipmentId = eq.id || eq._id;
            return `
                <tr>
                    <td style="text-align: center; font-weight: 600; color: #6b7280;">${serialNumber}</td>
                    <td>
                        <div class="dropdown">
                            <button class="action-btn" type="button" aria-expanded="false">
                                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                    <circle cx="12" cy="12" r="3"></circle>
                                    <path d="M12 1v6m0 6v6m11-7h-6m-6 0H1"></path>
                                </svg>
                                Actions
                            </button>
                            <ul class="dropdown-menu" role="menu">
                                <li role="none">
                                    <a class="dropdown-item" href="#" onclick="event.preventDefault(); viewEquipment('${equipmentId}');">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M1 12s4-8 11-8 11 8 11 8-4 8-11 8-11-8-11-8z"></path>
                                            <circle cx="12" cy="12" r="3"></circle>
                                        </svg>
                                        View Details
                                    </a>
                                </li>
                                <li role="none">
                                    <a class="dropdown-item" href="#" onclick="event.preventDefault(); updateEquipment('${equipmentId}');">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <path d="M11 4H4a2 2 0 0 0-2 2v14a2 2 0 0 0 2 2h14a2 2 0 0 0 2-2v-7"></path>
                                            <path d="M18.5 2.5a2.121 2.121 0 0 1 3 3L12 15l-4 1 1-4 9.5-9.5z"></path>
                                        </svg>
                                        Update Equipment
                                    </a>
                                </li>
                                <li class="dropdown-divider"></li>
                                <li role="none">
                                    <a class="dropdown-item danger" href="#" onclick="event.preventDefault(); deleteEquipment('${equipmentId}', '${eq.equipmentName || eq.vehicleName || 'this equipment'}');">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="14" height="14" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2">
                                            <polyline points="3,6 5,6 21,6"></polyline>
                                            <path d="M19,6v14a2,2,0,0,1-2,2H7a2,2,0,0,1-2-2V6M8,6V4a2,2,0,0,1,2-2h4a2,2,0,0,1,2,2V6"></path>
                                        </svg>
                                        Delete Equipment
                                    </a>
                                </li>
                            </ul>
                        </div>
                    </td>
                    <td>${eq.vehicleNumber || eq.registrationNumber || eq.id || 'N/A'}</td>
                    <td>${eq.vehicleName || eq.equipmentName || eq.customName || 'N/A'}</td>
                    <td>${eq.category || eq.mainCategory || 'N/A'}</td>
                    <td>${eq.model || 'N/A'}</td>
                    <td><span class="status-badge ${statusColor}">${eq.status || 'N/A'}</span></td>
                    <td>${eq.meterReading || 'N/A'} ${eq.meterUnit || ''}</td>
                    <td>${eq.site || eq.location || eq.country || 'N/A'}</td>
                    <td>${eq.operator || eq.operatorName || 'N/A'}</td>
                </tr>
            `;
        }).join('');
    }
    
    // Get status badge class
    function getStatusBadgeClass(status) {
        if (!status) return 'status-info';
        const s = status.toLowerCase();
        if (s.includes('active') || s.includes('available') || s.includes('operational')) return 'status-success';
        if (s.includes('maintenance') || s.includes('servicing')) return 'status-warning';
        if (s.includes('breakdown') || s.includes('repair') || s.includes('out of service')) return 'status-danger';
        if (s.includes('deployed') || s.includes('in use')) return 'status-primary';
        if (s.includes('idle') || s.includes('standby')) return 'status-info';
        return 'status-info';
    }
    
    // Show empty state
    function showEmptyState() {
        const tbody = document.getElementById('equipmentTableBody');
        if (!tbody) return;
        
        tbody.innerHTML = `
            <tr class="empty-row">
                <td colspan="10">
                    <div class="empty-state">
                        <div class="empty-content">
                            <svg xmlns="http://www.w3.org/2000/svg" width="64" height="64" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="1.5">
                                <path d="M19 17h2c.6 0 1-.4 1-1v-3c0-.9-.7-1.7-1.5-1.9L18.7 8.3c-.3-.8-1-1.3-1.9-1.3H7.2c-.9 0-1.6.5-1.9 1.3L3.5 11.1C2.7 11.3 2 12.1 2 13v3c0 .6.4 1 1 1h2m14 0H5m14 0v3c0 .6-.4 1-1 1H6c-.6 0-1-.4-1-1v-3"/>
                            </svg>
                            <h3>No Equipment Found</h3>
                            <p>No equipment matches your current filter. Try adjusting the status filter.</p>
                        </div>
                    </div>
                </td>
            </tr>
        `;
    }
    
    // Render pagination
    function renderPagination() {
        const section = document.getElementById('paginationSection');
        const info = document.getElementById('paginationInfo');
        const controls = document.getElementById('paginationControls');
        
        if (!section || !info || !controls) return;
        
        if (totalPages <= 1) {
            section.style.display = 'none';
            return;
        }
        
        section.style.display = 'flex';
        
        const start = ((currentPage - 1) * perPage) + 1;
        const end = Math.min(currentPage * perPage, totalRecords);
        info.textContent = `Showing ${start} to ${end} of ${totalRecords} entries`;
        
        let html = '';
        
        // Previous button
        if (currentPage > 1) {
            html += `<button class="pagination-btn prev-btn" onclick="goToPage(${currentPage - 1})">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="15,18 9,12 15,6"/></svg>
                <span>Previous</span>
            </button>`;
        } else {
            html += `<span class="pagination-btn prev-btn disabled">
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="15,18 9,12 15,6"/></svg>
                <span>Previous</span>
            </span>`;
        }
        
        // Page numbers
        html += '<div class="page-numbers">';
        const startPage = Math.max(1, currentPage - 2);
        const endPage = Math.min(totalPages, currentPage + 2);
        
        if (startPage > 1) {
            html += `<button class="page-number" onclick="goToPage(1)">1</button>`;
            if (startPage > 2) html += `<span class="page-ellipsis">...</span>`;
        }
        
        for (let i = startPage; i <= endPage; i++) {
            if (i === currentPage) {
                html += `<span class="page-number current">${i}</span>`;
            } else {
                html += `<button class="page-number" onclick="goToPage(${i})">${i}</button>`;
            }
        }
        
        if (endPage < totalPages) {
            if (endPage < totalPages - 1) html += `<span class="page-ellipsis">...</span>`;
            html += `<button class="page-number" onclick="goToPage(${totalPages})">${totalPages}</button>`;
        }
        
        html += '</div>';
        
        // Next button
        if (currentPage < totalPages) {
            html += `<button class="pagination-btn next-btn" onclick="goToPage(${currentPage + 1})">
                <span>Next</span>
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="9,18 15,12 9,6"/></svg>
            </button>`;
        } else {
            html += `<span class="pagination-btn next-btn disabled">
                <span>Next</span>
                <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2"><polyline points="9,18 15,12 9,6"/></svg>
            </span>`;
        }
        
        controls.innerHTML = html;
    }
    
    // Hide pagination
    function hidePagination() {
        const section = document.getElementById('paginationSection');
        if (section) section.style.display = 'none';
    }
    
    // Go to specific page
    function goToPage(page) {
        if (page < 1 || page > totalPages || page === currentPage) return;
        loadEquipment(page, currentStatus);
        window.scrollTo({ top: 0, behavior: 'smooth' });
    }
    
    // Change status filter
    function filterByStatus(status) {
        currentStatus = status;
        currentPage = 1;
        loadEquipment(1, status);
    }
    
    // View Equipment Details
    async function viewEquipment(equipmentId) {
        console.log('Viewing equipment:', equipmentId);
        alert(`View Equipment: ${equipmentId}\n\nThis feature will show detailed equipment information.`);
        // TODO: Implement view details modal
    }
    
    // Update Equipment
    function updateEquipment(equipmentId) {
        console.log('Updating equipment:', equipmentId);
        // Redirect to edit page
        window.location.href = `/vehicles/add-vehicle?edit=true&equipmentId=${equipmentId}`;
    }
    
    // Delete Equipment with Confirmation
    async function deleteEquipment(equipmentId, equipmentName) {
        try {
            console.log('Attempting to delete equipment:', equipmentId);
            
            // Show confirmation dialog
            const confirmed = confirm(`⚠️ Are you sure you want to delete this equipment?\n\n${equipmentName}\n\nThis action cannot be undone!`);
            
            if (!confirmed) {
                console.log('Delete cancelled');
                return;
            }
            
            // Show loading
            alert('Deleting equipment...');
            
            // Call API to delete (you'll need to implement this endpoint)
            const response = await fetch(`/api/equipment/delete/${equipmentId}`, {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').getAttribute('content')
                }
            });
            
            const result = await response.json();
            
            if (result.success) {
                alert(`✅ Equipment ${equipmentName} deleted successfully!`);
                // Reload current page
                loadEquipment(currentPage, currentStatus);
            } else {
                alert('❌ Failed to delete equipment: ' + (result.message || 'Unknown error'));
            }
            
        } catch (error) {
            console.error('Error deleting equipment:', error);
            alert('❌ Error deleting equipment: ' + error.message);
        }
    }
    
    // Initialize dropdown functionality
    function initializeDropdowns() {
        document.addEventListener('click', function(e) {
            // Toggle dropdown when action button is clicked
            if (e.target.closest('.action-btn')) {
                e.preventDefault();
                e.stopPropagation();
                
                const button = e.target.closest('.action-btn');
                const dropdown = button.parentElement.querySelector('.dropdown-menu');
                
                // Close all other dropdowns
                document.querySelectorAll('.dropdown-menu.show').forEach(menu => {
                    if (menu !== dropdown) {
                        menu.classList.remove('show');
                    }
                });
                
                // Toggle current dropdown
                if (dropdown) {
                    dropdown.classList.toggle('show');
                }
            } else {
                // Close all dropdowns when clicking outside
                if (!e.target.closest('.dropdown-menu')) {
                    document.querySelectorAll('.dropdown-menu.show').forEach(dropdown => {
                        dropdown.classList.remove('show');
                    });
                }
            }
        });
    }
    
    // Initialize
    document.addEventListener('DOMContentLoaded', function() {
        // Status filter change
        const statusFilter = document.getElementById('statusFilter');
        if (statusFilter) {
            statusFilter.addEventListener('change', function() {
                filterByStatus(this.value);
            });
        }
        
        // Initialize dropdown functionality
        initializeDropdowns();
        
        // Load equipment on page load
        loadEquipment(currentPage, currentStatus);
    });
    
    // Expose functions globally
    window.goToPage = goToPage;
    window.filterByStatus = filterByStatus;
    window.viewEquipment = viewEquipment;
    window.updateEquipment = updateEquipment;
    window.deleteEquipment = deleteEquipment;
    
    function toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        sidebar.classList.toggle('collapsed');
    }

    function viewVehicle(vehicleId) {
        console.log('Viewing vehicle:', vehicleId);
        // Implement view functionality
    }

    // Search functionality
    document.getElementById('search-input').addEventListener('input', function() {
        filterTable(this.value);
    });

    function filterTable(searchTerm) {
        const tableBody = document.getElementById('equipmentTableBody');
        const rows = tableBody.getElementsByTagName('tr');
        
        for (let row of rows) {
            const cells = row.getElementsByTagName('td');
            let found = false;
            
            for (let cell of cells) {
                if (cell.textContent.toLowerCase().includes(searchTerm.toLowerCase())) {
                    found = true;
                    break;
                }
            }
            
            row.style.display = found ? '' : 'none';
        }
    }

    // Change records per page
    function changePerPage(perPage) {
        const url = new URL(window.location);
        url.searchParams.set('per_page', perPage);
        url.searchParams.set('page', 1); // Reset to first page
        window.location.href = url.toString();
    }


</script>
@endsection

<style>
/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== MAIN CONTAINER ===== */
.vehicles-inventory-container {
    min-height: 100vh;
    background: #f8fafc;
    font-family: 'Inter', sans-serif;
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
}

/* ===== HEADER STYLES ===== */
.vehicles-inventory-header {
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    padding: 2rem 0;
    margin-bottom: 0;
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1);
    display: none;
}

.header-content {
    max-width: 1200px;
    margin: 0 auto;
    padding: 0 2rem;
}

.page-title h1 {
    font-size: 2.5rem;
    font-weight: 700;
    color: #ffffff;
    margin: 0 0 0.5rem 0;
    letter-spacing: -0.025em;
}

.page-title p {
    font-size: 1.125rem;
    color: #cbd5e1;
    margin: 0;
    font-weight: 400;
}

/* ===== MAIN CONTENT ===== */
.vehicles-inventory-content {
    min-height: 100vh;
}

.content-box {
    background: white;
    padding: 32px 32px 32px 40px;
    border-radius: 0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
    border: 1px solid #e2e8f0;
    margin-bottom: 2rem;
}

/* ===== ACTION BAR ===== */
.action-bar {
    background: transparent;
    border: none;
    border-radius: 0;
    padding: 0;
    margin-bottom: 32px;
    box-shadow: none;
}

.action-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 1rem;
    flex-wrap: wrap;
}

.page-title h1, .page-title h2 {
    color: #0f172a;
    font-weight: 700;
    font-size: 32px;
    line-height: 1.2;
    letter-spacing: -0.025em;
    margin: 0;
}

.page-title h2 {
    font-size: 24px;
}

/* ===== ACTION BUTTONS ===== */
.action-buttons {
    display: flex;
    gap: 0.75rem;
    align-items: center;
    flex-wrap: wrap;
}

.btn-create {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: #0055a5;
    color: #ffffff;
    border: none;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-create:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black !important;
}

/* ===== SEARCH SECTION ===== */
.search-section {
    flex: 1;
    min-width: 300px;
}

.search-container {
    position: relative;
    max-width: 400px;
}

.search-icon {
    position: absolute;
    left: 1rem;
    top: 50%;
    transform: translateY(-50%);
    color: #64748b;
    z-index: 1;
}

.search-input {
    width: 100%;
    padding: 0.75rem 1rem 0.75rem 3rem;
    border: 2px solid #e2e8f0;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 500;
    color: #1e293b;
    background: #ffffff;
    transition: all 0.2s ease;
    font-family: 'Inter', sans-serif;
}

.search-input:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.search-input::placeholder {
    color: #94a3b8;
    font-weight: 400;
}

/* ===== TABLE SECTION ===== */
.table-section {
    background: transparent;
    border: none;
    border-radius: 0;
    overflow: hidden;
    box-shadow: none;
}

.table-container {
    overflow-x: auto;
    overflow-y: visible;
    max-height: none;
}

.vehicles-inventory-table {
    width: 100%;
    border-collapse: collapse;
    font-family: 'Inter', sans-serif;
    background: #ffffff;
    border: 1px solid #e2e8f0;
    border-radius: 0;
    box-shadow: 0 1px 3px 0 rgba(0, 0, 0, 0.1), 0 1px 2px 0 rgba(0, 0, 0, 0.06);
}

.table-header {
    background: linear-gradient(135deg, #f1f5f9 0%, #e2e8f0 100%);
}

.table-header th {
    padding: 1rem 1.5rem;
    text-align: left;
    font-weight: 600;
    font-size: 0.875rem;
    color: #475569;
    text-transform: uppercase;
    letter-spacing: 0.05em;
    border-bottom: 2px solid #e2e8f0;
    white-space: nowrap;
}

.table-body tr {
    border-bottom: 1px solid #f1f5f9;
    transition: background-color 0.2s ease;
}

.table-body tr:hover {
    background-color: #f8fafc;
}

.table-body td {
    padding: 1rem 1.5rem;
    font-size: 0.875rem;
    color: #334155;
    font-weight: 500;
}

/* ===== BUTTON STYLES ===== */
.btn-view {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: #0055a5;
    color: #ffffff;
    border: none;
    border-radius: 0;
    font-size: 0.75rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
    /* box-shadow: 0 2px 4px rgba(59, 130, 246, 0.2); */
}

.btn-view:hover {
    background: rgba(255, 255, 255, 0.2);
    transform: translateY(-1px);
    box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.1), 0 2px 4px -1px rgba(0, 0, 0, 0.06);
    color: black !important;
}

/* ===== STATUS BADGES ===== */
.status-badge {
    display: inline-flex;
    align-items: center;
    padding: 0.25rem 0.75rem;
    border-radius: 0;
    font-size: 0.75rem;
    font-weight: 600;
    text-transform: uppercase;
    letter-spacing: 0.05em;
}

.status-success {
    /* background: linear-gradient(135deg, #10b981 0%, #059669 100%); */
    color:rgb(0, 0, 0);
}

.status-warning {
    /* background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%); */
    color: rgb(0, 0, 0);
}

.status-danger {
    /* background: linear-gradient(135deg, #ef4444 0%, #dc2626 100%); */
    color: rgb(0, 0, 0);
}

.status-primary {
    /* background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%); */
    color: rgb(0, 0, 0);
}

.status-info {
    /* background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%); */
    color: rgb(0, 0, 0);
}

.status-secondary {
    /* background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%); */
    color: rgb(0, 0, 0);
}

/* ===== EMPTY STATE ===== */
.empty-row td {
    padding: 0;
    border: none;
    height: auto;
    min-height: 200px;
}

.empty-state {
    padding: 4rem 2rem;
    text-align: center;
    background: #ffffff;
    min-height: 200px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.empty-content svg {
    color: #cbd5e1;
    margin-bottom: 1.5rem;
}

.empty-content h3 {
    font-size: 1.25rem;
    font-weight: 600;
    color: #475569;
    margin: 0 0 0.5rem 0;
}

.empty-content p {
    font-size: 0.875rem;
    color: #64748b;
    margin: 0;
}

/* ===== MODAL STYLES ===== */
.modal {
    font-family: 'Inter', sans-serif;
}

.modal.show {
    display: block;
}

.modal.fade .modal-dialog {
    transition: transform 0.3s ease-out;
    transform: translate(0, -50px);
}

.modal.show .modal-dialog {
    transform: none;
}

.modal-dialog {
    max-width: 1200px;
    margin: 1.75rem auto;
}

.modal-dialog.modal-xl {
    max-width: 1200px;
}

.modal-content {
    background: #ffffff;
    border: 1px solid #e2e8f0;
    border-radius: 0;
    box-shadow: 0 10px 25px rgba(0, 0, 0, 0.1);
}

.modal-header {
    background: linear-gradient(135deg, #1e293b 0%, #334155 100%);
    color: #ffffff;
    border-bottom: 1px solid #e2e8f0;
    padding: 1.5rem 2rem;
    border-radius: 0;
}

.modal-title {
    font-size: 1.5rem;
    font-weight: 700;
    margin: 0;
    color: #ffffff;
}

.modal-close {
    background: none;
    border: none;
    color: #ffffff;
    font-size: 1.5rem;
    cursor: pointer;
    padding: 0;
    width: 32px;
    height: 32px;
    display: flex;
    align-items: center;
    justify-content: center;
    border-radius: 0;
    transition: background-color 0.2s;
}

.modal-close:hover {
    background-color: rgba(255, 255, 255, 0.1);
}

.modal-body {
    padding: 2rem;
    max-height: 70vh;
    overflow-y: auto;
}

.modal-footer {
    background: #f8fafc;
    border-top: 1px solid #e2e8f0;
    padding: 1.5rem 2rem;
    display: flex;
    justify-content: flex-end;
    gap: 1rem;
    border-radius: 0;
}

/* ===== FORM STYLES ===== */
.vehicle-form {
    font-family: 'Inter', sans-serif;
}

.form-grid {
    display: grid;
    grid-template-columns: 1fr 1fr;
    gap: 1.5rem;
    margin-bottom: 1.5rem;
}

.form-group {
    display: flex;
    flex-direction: column;
}

.form-label {
    font-weight: 600;
    color: #374151;
    margin-bottom: 0.5rem;
    font-size: 0.875rem;
}

.form-control, .form-select {
    border: 2px solid #e2e8f0;
    border-radius: 0;
    padding: 0.75rem 1rem;
    font-size: 0.875rem;
    font-weight: 500;
    color: #1e293b;
    background: #ffffff;
    transition: all 0.2s ease;
    font-family: 'Inter', sans-serif;
}

.form-control:focus, .form-select:focus {
    outline: none;
    border-color: #3b82f6;
    box-shadow: 0 0 0 3px rgba(59, 130, 246, 0.1);
}

.form-control::placeholder {
    color: #94a3b8;
    font-weight: 400;
}

.input-with-select {
    display: flex;
    gap: 0.5rem;
}

.input-with-select .form-control {
    flex: 1;
}

.input-with-select .form-select {
    width: 120px;
}

/* ===== CHECKBOX STYLES ===== */
.checkbox-group {
    display: flex;
    flex-direction: column;
    gap: 0.75rem;
}

.checkbox-label {
    display: flex;
    align-items: center;
    gap: 0.75rem;
    cursor: pointer;
    font-size: 0.875rem;
    color: #374151;
}

.checkbox-label input[type="checkbox"] {
    display: none;
}

.checkbox-custom {
    width: 20px;
    height: 20px;
    border: 2px solid #d1d5db;
    border-radius: 0;
    background: #ffffff;
    position: relative;
    transition: all 0.2s ease;
}

.checkbox-label input[type="checkbox"]:checked + .checkbox-custom {
    background: #3b82f6;
    border-color: #3b82f6;
}

.checkbox-label input[type="checkbox"]:checked + .checkbox-custom::after {
    content: '';
    position: absolute;
    left: 6px;
    top: 2px;
    width: 6px;
    height: 10px;
    border: solid #ffffff;
    border-width: 0 2px 2px 0;
    transform: rotate(45deg);
}

/* ===== BUTTON STYLES ===== */
.btn-upload {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, #8b5cf6 0%, #7c3aed 100%);
    color: #ffffff;
    border: none;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(139, 92, 246, 0.2);
}

.btn-upload:hover {
    background: linear-gradient(135deg, #7c3aed 0%, #6d28d9 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(139, 92, 246, 0.3);
}

.btn-save {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
    color: #ffffff;
    border: none;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(16, 185, 129, 0.2);
}

.btn-save:hover {
    background: linear-gradient(135deg, #059669 0%, #047857 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(16, 185, 129, 0.3);
}

.btn-cancel {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: #ffffff;
    color: #64748b;
    border: 2px solid #e2e8f0;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
}

.btn-cancel:hover {
    background: #f8fafc;
    border-color: #cbd5e1;
    color: #475569;
}

/* ===== INSPECTION SECTION ===== */
.inspection-section {
    margin-top: 2rem;
    padding-top: 2rem;
    border-top: 1px solid #e2e8f0;
}

.section-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 1.5rem;
}

.section-header h3 {
    font-size: 1.25rem;
    font-weight: 700;
    color: #1e293b;
    margin: 0;
}

.btn-checklist {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.75rem 1.5rem;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    color: #ffffff;
    border: none;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 600;
    font-family: 'Inter', sans-serif;
    cursor: pointer;
    transition: all 0.2s ease;
    box-shadow: 0 2px 4px rgba(59, 130, 246, 0.2);
}

.btn-checklist:hover {
    background: linear-gradient(135deg, #1d4ed8 0%, #1e40af 100%);
    transform: translateY(-1px);
    box-shadow: 0 4px 8px rgba(59, 130, 246, 0.3);
}

.inspection-content {
    background: #f8fafc;
    border: 2px dashed #cbd5e1;
    border-radius: 0;
    padding: 2rem;
    text-align: center;
    min-height: 120px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.inspection-content .empty-state {
    padding: 0;
    margin: 0;
    min-height: auto;
}

.inspection-content .empty-state svg {
    color: #cbd5e1;
    margin-bottom: 1rem;
}

.inspection-content .empty-state p {
    color: #64748b;
    font-size: 0.875rem;
    margin: 0;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 1200px) {
    .vehicles-inventory-container {
        padding: 0;
    }
    
    .vehicles-inventory-content {
        padding: 20px 0 0 14px;
    }
    
    .content-box {
        padding: 28px 28px 28px 36px;
        margin-left: 0.5rem;
    }
    
    .action-content {
        flex-direction: column;
        align-items: stretch;
        gap: 16px;
    }
    
    .action-buttons {
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .search-section {
        min-width: 280px;
    }
    
    .form-grid {
        grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
        gap: 1.25rem;
    }
    
    .modal-dialog {
        margin: 1rem;
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 1.25rem 1.75rem;
    }
}

@media (max-width: 992px) {
    .vehicles-inventory-header {
        padding: 1.5rem 0;
    }
    
    .page-title h1 {
        font-size: 2.25rem;
    }
    
    .page-title h2 {
        font-size: 1.75rem;
    }
    
    .action-bar {
        padding: 0;
        margin-bottom: 24px;
    }
    
    .action-buttons {
        gap: 8px;
    }
    
    .action-buttons button {
        padding: 8px 12px;
        font-size: 13px;
    }
    
    .search-section {
        min-width: 250px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .table-container {
        overflow-x: auto;
    }
    
    .vehicles-inventory-table {
        min-width: 800px;
    }
    
    .table-header th:first-child,
    .table-body td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 1;
    }
    
    .modal-dialog {
        margin: 0.5rem;
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 1rem 1.5rem;
    }
    
    .section-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 1rem;
    }
    
    .vehicles-inventory-content {
        padding: 1rem 1rem 1rem 2rem;
    }
    
    .content-box {
        padding: 1.5rem 1.5rem 1.5rem 2rem;
        margin-left: 0.25rem;
    }
    
    .page-title h1 {
        font-size: 2rem;
    }
    
    .page-title h2 {
        font-size: 1.5rem;
    }
    
    .action-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 1.5rem;
    }
    
    .action-buttons {
        width: 100%;
        justify-content: flex-start;
        flex-wrap: wrap;
    }
    
    .search-section {
        width: 100%;
        min-width: auto;
        margin-bottom: 1rem;
    }
    
    .search-input {
        width: 100%;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .vehicles-inventory-table {
        min-width: 700px;
    }
    
    .table-header th:first-child,
    .table-body td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 1;
    }
    
    .table-header th,
    .table-body td {
        padding: 0.75rem 1rem;
        font-size: 0.8rem;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 1rem;
    }
    
    .modal-dialog {
        margin: 0.5rem;
        max-width: calc(100% - 1rem);
    }
    
    .modal-dialog.modal-xl {
        max-width: calc(100% - 1rem);
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 1rem 1.5rem;
    }
    
    .modal-footer {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .btn-save, .btn-cancel {
        width: 100%;
        justify-content: center;
    }
    
    .section-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 1rem;
    }
    
    .inspection-content {
        padding: 1.5rem;
    }
}

@media (max-width: 576px) {
    .vehicles-inventory-content {
        padding: 0.75rem 0.75rem 0.75rem 1.5rem;
    }
    
    .content-box {
        padding: 1rem 1rem 1rem 1.5rem;
        margin-left: 0.25rem;
    }
    
    .page-title h1 {
        font-size: 1.75rem;
    }
    
    .page-title h2 {
        font-size: 1.25rem;
    }
    
    .action-buttons {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .action-buttons button {
        width: 100%;
        justify-content: center;
        padding: 12px 16px;
        font-size: 14px;
    }
    
    .search-input {
        padding: 10px 12px 10px 2.5rem;
        font-size: 14px;
    }
    
    .form-grid {
        grid-template-columns: 1fr;
        gap: 0.75rem;
    }
    
    .form-group {
        margin-bottom: 1rem;
    }
    
    .form-control, .form-select {
        padding: 10px 12px;
        font-size: 14px;
    }
    
    .table-container {
        overflow-x: auto;
        -webkit-overflow-scrolling: touch;
    }
    
    .vehicles-inventory-table {
        min-width: 600px;
    }
    
    .table-header th,
    .table-body td {
        padding: 0.5rem 0.75rem;
        font-size: 0.75rem;
    }
    
    .table-header th:first-child,
    .table-body td:first-child {
        position: sticky;
        left: 0;
        background: #f8fafc;
        z-index: 1;
    }
    
    .modal-dialog {
        margin: 0.25rem;
        max-width: calc(100% - 0.5rem);
    }
    
    .modal-dialog.modal-xl {
        max-width: calc(100% - 0.5rem);
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 0.75rem 1rem;
    }
    
    .modal-title {
        font-size: 1.25rem;
    }
    
    .btn-save, .btn-cancel {
        width: 100%;
        justify-content: center;
        padding: 12px 16px;
        font-size: 14px;
    }
    
    .section-header {
        flex-direction: column;
        align-items: flex-start;
        gap: 0.75rem;
    }
    
    .inspection-content {
        padding: 1rem;
    }
    
    .empty-state {
        padding: 2rem 1rem;
        min-height: 150px;
    }
    
    .empty-content svg {
        width: 48px;
        height: 48px;
    }
    
    .empty-content h3 {
        font-size: 1.125rem;
    }
    
    .empty-content p {
        font-size: 0.8rem;
    }
}

@media (max-width: 480px) {
    .vehicles-inventory-content {
        padding: 0.5rem 0.5rem 0.5rem 1rem;
    }
    
    .content-box {
        padding: 0.75rem 0.75rem 0.75rem 1rem;
        margin-left: 0.25rem;
    }
    
    .page-title h1 {
        font-size: 1.5rem;
    }
    
    .page-title h2 {
        font-size: 1.125rem;
    }
    
    .action-buttons button {
        padding: 10px 14px;
        font-size: 13px;
    }
    
    .search-input {
        padding: 8px 10px 8px 2.25rem;
        font-size: 13px;
    }
    
    .form-control, .form-select {
        padding: 8px 10px;
        font-size: 13px;
    }
    
    .table-header th,
    .table-body td {
        padding: 0.5rem 0.5rem;
        font-size: 0.7rem;
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 0.5rem 0.75rem;
    }
    
    .modal-title {
        font-size: 1.125rem;
    }
    
    .btn-save, .btn-cancel {
        padding: 10px 14px;
        font-size: 13px;
    }
    
    .inspection-content {
        padding: 0.75rem;
    }
    
    .empty-state {
        padding: 1.5rem 0.75rem;
        min-height: 120px;
    }
    
    .empty-content svg {
        width: 32px;
        height: 32px;
    }
    
    .empty-content h3 {
        font-size: 1rem;
    }
    
    .empty-content p {
        font-size: 0.75rem;
    }
}

/* ===== LANDSCAPE ORIENTATION ===== */
@media (max-width: 768px) and (orientation: landscape) {
    .vehicles-inventory-content {
        padding: 0.5rem 0.5rem 0.5rem 1rem;
    }
    
    .content-box {
        padding: 1rem 1rem 1rem 1.5rem;
        margin-left: 0.25rem;
    }
    
    .modal-dialog {
        margin: 0.25rem;
        max-width: calc(100% - 0.5rem);
    }
    
    .modal-header, .modal-body, .modal-footer {
        padding: 0.75rem 1rem;
    }
}

/* ===== TOUCH DEVICES ===== */
@media (hover: none) and (pointer: coarse) {
    .action-buttons button,
    .btn-create,
    .btn-view,
    .btn-save,
    .btn-cancel,
    .btn-upload,
    .btn-checklist {
        min-height: 44px;
        padding: 12px 16px;
    }
    
    .form-control,
    .form-select {
        min-height: 44px;
        padding: 12px 16px;
    }
    
    .search-input {
        min-height: 44px;
        padding: 12px 16px 12px 3rem;
    }
    
    .modal-close {
        min-width: 44px;
        min-height: 44px;
    }
    
    .checkbox-label {
        min-height: 44px;
        padding: 8px 0;
    }
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* ===== PAGINATION STYLES ===== */
.pagination-section {
    background: #ffffff;
    border-top: 1px solid #e2e8f0;
    padding: 1.5rem 2rem;
    margin-top: 0;
}

.pagination-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 1rem;
    flex-wrap: wrap;
}

.pagination-info {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.pagination-text {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 500;
}

.pagination-controls {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.pagination-btn {
    display: inline-flex;
    align-items: center;
    gap: 0.5rem;
    padding: 0.5rem 1rem;
    background: #ffffff;
    color: #374151;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 500;
    text-decoration: none;
    transition: all 0.2s ease;
    cursor: pointer;
}

.pagination-btn:hover:not(.disabled) {
    background: #f3f4f6;
    border-color: #9ca3af;
    color: #111827;
}

.pagination-btn.disabled {
    background: #f9fafb;
    color: #9ca3af;
    border-color: #e5e7eb;
    cursor: not-allowed;
}

.page-numbers {
    display: flex;
    align-items: center;
    gap: 0.25rem;
    margin: 0 1rem;
}

.page-number {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 2.5rem;
    height: 2.5rem;
    background: #ffffff;
    color: #374151;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 500;
    text-decoration: none;
    transition: all 0.2s ease;
    cursor: pointer;
}

.page-number:hover {
    background: #f3f4f6;
    border-color: #9ca3af;
    color: #111827;
}

.page-number.current {
    background: #0055a5;
    color: #ffffff;
    border-color: #0055a5;
}

.page-number.current:hover {
    background: #004494;
    border-color: #004494;
    color: #ffffff;
}

.page-ellipsis {
    display: inline-flex;
    align-items: center;
    justify-content: center;
    width: 2.5rem;
    height: 2.5rem;
    color: #9ca3af;
    font-size: 0.875rem;
    font-weight: 500;
}

.records-per-page {
    display: flex;
    align-items: center;
    gap: 0.5rem;
}

.records-per-page label {
    font-size: 0.875rem;
    color: #374151;
    font-weight: 500;
}

.per-page-select {
    padding: 0.5rem 0.75rem;
    border: 1px solid #d1d5db;
    border-radius: 0;
    font-size: 0.875rem;
    font-weight: 500;
    color: #374151;
    background: #ffffff;
    cursor: pointer;
    transition: all 0.2s ease;
}

.per-page-select:focus {
    outline: none;
    border-color: #0055a5;
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.records-per-page span {
    font-size: 0.875rem;
    color: #64748b;
    font-weight: 500;
}

/* Responsive Pagination */
@media (max-width: 768px) {
    .pagination-content {
        flex-direction: column;
        gap: 1rem;
        align-items: stretch;
    }
    
    .pagination-controls {
        justify-content: center;
        flex-wrap: wrap;
    }
    
    .page-numbers {
        margin: 0 0.5rem;
    }
    
    .records-per-page {
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .pagination-section {
        padding: 1rem;
    }
    
    .pagination-btn {
        padding: 0.375rem 0.75rem;
        font-size: 0.8rem;
    }
    
    .page-number {
        width: 2rem;
        height: 2rem;
        font-size: 0.8rem;
    }
    
    .page-ellipsis {
        width: 2rem;
        height: 2rem;
        font-size: 0.8rem;
    }
}

/* ===== PRINT STYLES ===== */
@media print {
    .action-bar,
    .modal,
    .pagination-section {
        display: none !important;
    }
    
    .vehicles-inventory-container {
        background: white;
    }
    
    .vehicles-inventory-header {
        background: white;
        color: black;
    }
}
</style>
