@extends('welcome')

@section('title', 'Fuel Management KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="fuel-kpi-container">
    <!-- Loading Overlay -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Fuel Data...</div>
        </div>
    </div>
    
    <!-- Header - Matching Workorder Design -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url()->previous() }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Fuel Management KPI Dashboard</h1>
                    </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn-header">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                        </button>
                <button type="button" class="create-btn" id="refresh-data-btn-header">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                        </button>
                    </div>
                </div>
            </div>
    
    <div class="fuel-kpi-content">
            <div class="content-box">
            <div class="time-period-selector">
                <div class="period-buttons">
                    <button class="period-btn" data-period="today">Today</button>
                    <button class="period-btn active" data-period="7days">Last 7 Days</button>
                    <button class="period-btn" data-period="30days">Last 30 Days</button>
                    <button class="period-btn" data-period="90days">Last 90 Days</button>
                    <button class="period-btn" data-period="custom">Custom Date</button>
                </div>
                
                <!-- Custom Date Range Picker -->
                <div class="custom-date-range" id="custom-date-range" style="display: none;">
                    <div class="date-range-inputs">
                        <div class="date-input-group">
                            <label for="start-date">Start Date:</label>
                            <input type="date" id="start-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label for="end-date">End Date:</label>
                            <input type="date" id="end-date" class="date-input">
                        </div>
                        <button class="btn-apply-dates" id="apply-custom-dates">Apply</button>
                        <button class="btn-clear-dates" id="clear-custom-dates">Clear</button>
                    </div>
                </div>
            </div>
            
            <div class="modern-tabs">
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="overview">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                        Overview
                    </button>
                    <button class="tab-btn" data-tab="vehicle">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                            <path d="M14 11a5 5 0 0 0-10 0"/>
                        </svg>
                        Vehicle
                    </button>
                    <button class="tab-btn" data-tab="vehicle-category">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M12 2v20M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                        </svg>
                        Vehicle Category
                    </button>
                </div>
                    <div class="tab-content">
                    <div id="overview" class="tab-pane active">
                        <div class="kpi-cards-grid">
                            <!-- KPI Card Template -->
                            <div class="kpi-card" data-kpi="fuel-logs">
                                <div class="kpi-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                        <polyline points="14,2 14,8 20,8"/>
                                        <line x1="16" y1="13" x2="8" y2="13"/>
                                        <line x1="16" y1="17" x2="8" y2="17"/>
                                        <polyline points="10,9 9,9 8,9"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Fuel Logs</h3>
                                    <div class="kpi-value">0</div>
                                    <div class="kpi-label">Logs</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="fuel-quantity">
                                <div class="kpi-icon info">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M12 2v20M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Fuel Quantity</h3>
                                    <div class="kpi-value">0</div>
                                    <div class="kpi-label">Liters</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="outstation-quantity">
                                <div class="kpi-icon warning">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                                        <path d="M14 11a5 5 0 0 0-10 0"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Outstation Quantity</h3>
                                    <div class="kpi-value">0</div>
                                    <div class="kpi-label">Liters</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="average-quantity">
                                <div class="kpi-icon secondary">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M3 3v18h18"/>
                                        <path d="M18.7 8l-5.1 5.2-2.8-2.7L7 14.3"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Average Fuel Quantity</h3>
                                    <div class="kpi-value">0</div>
                                    <div class="kpi-label">Liters</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="fuel-cost">
                                <div class="kpi-icon success">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <line x1="12" y1="1" x2="12" y2="23"/>
                                        <path d="M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Fuel Cost</h3>
                                    <div class="kpi-value">$0.00</div>
                                    <div class="kpi-label">Cost</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="outstation-cost">
                                <div class="kpi-icon info">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                                        <path d="M14 11a5 5 0 0 0-10 0"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Outstation Cost</h3>
                                    <div class="kpi-value">$0.00</div>
                                    <div class="kpi-label">Cost</div>
                                </div>
                            </div>
                        </div>
                        
                        <div class="overview-bottom-section">
                            <div class="additional-kpi-cards">
                                <div class="kpi-card">
                                    <div class="kpi-icon warning">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M12 2v20M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                                        </svg>
                                    </div>
                                    <div class="kpi-content">
                                        <h3>Bunker wise Fuel Quantity</h3>
                                        <div class="kpi-value">No Data</div>
                                        <div class="kpi-label">Bunker</div>
                                    </div>
                                </div>
                                
                                <div class="kpi-card">
                                    <div class="kpi-icon secondary">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M12 2v20M17 5H9.5a3.5 3.5 0 0 0 0 7h5a3.5 3.5 0 0 1 0 7H6"/>
                                        </svg>
                                    </div>
                                    <div class="kpi-content">
                                        <h3>Fuel Type</h3>
                                        <div class="kpi-value">No Data</div>
                                        <div class="kpi-label">Type</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="chart-section">
                                <div class="chart-card">
                                    <div class="chart-header">
                                        <h3>Date wise Fuel Quantity</h3>
                                    </div>
                                    <div class="chart-container">
                                        <canvas id="running"></canvas>
                                    </div>
                                </div>
                                </div>
                            </div>
                        </div>
                    <div id="vehicle" class="tab-pane">
                        <div class="chart-section">
                            <div class="chart-card">
                                <div class="chart-header">
                            <h3>Vehicle Wise Fuel Quantity</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="vehicle-wise-chart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="vehicle-category" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Fuel Type Distribution</h3>
                            </div>
                                <div class="chart-container">
                                    <canvas id="fuel-type-chart"></canvas>
                        </div>
                    </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Filling Source Distribution</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="filling-source-chart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
    <script>
        function toggleSidebar() {
            const sidebar = document.querySelector('.sidebar');
            sidebar.classList.toggle('collapsed');
        }
    </script>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>
<!-- Firebase SDK -->
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>

<!-- Chart.js Library -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/2.9.4/Chart.js"></script>

<!-- SheetJS Library for Excel Export -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>

<script>
// Firebase Configuration
const firebaseConfig = {
    databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com"
};

// Initialize Firebase
if (!firebase.apps.length) {
    firebase.initializeApp(firebaseConfig);
}

const database = firebase.database();
const adminId = "cahj3jJqe5Z16pQdLGSWrNgm84o1";

// Global variables
let allFuelData = [];
let chartInstances = {};
let currentDateRange = 7; // Default to last 7 days
let customStartDate = null;
let customEndDate = null;

// Fetch fuel log data from Firebase
function fetchFuelData() {
    const fuelRef = database.ref(`GlobalFuelLogs/admins/${adminId}`);
    
    // Show loading state
    showLoadingState(true);
    
    fuelRef.once('value')
        .then((snapshot) => {
            allFuelData = [];
            const data = snapshot.val();
            
            console.log('Raw Firebase data:', data);
            
            if (data) {
                Object.keys(data).forEach(key => {
                    const fuelLog = data[key];
                    // Accept any object that has at least country, equipmentName, or fuelType
                    // This handles both formats: with and without 'id' field
                    if (fuelLog && typeof fuelLog === 'object' && 
                        (fuelLog.country || fuelLog.equipmentName || fuelLog.fuelType || fuelLog.id)) {
                        allFuelData.push({
                            firebaseKey: key,
                            id: fuelLog.id || key, // Use Firebase key if id not present
                            ...fuelLog
                        });
                    }
                });
            }
            
            console.log('Fetched fuel logs:', allFuelData.length);
            console.log('Sample fuel log:', allFuelData[0]);
            
            if (allFuelData.length === 0) {
                showNotification('No fuel log data found', 'info');
            } else {
                showNotification(`Loaded ${allFuelData.length} fuel log(s)`, 'success');
            }
            
            updateDashboard();
            showLoadingState(false);
        })
        .catch((error) => {
            console.error('Error fetching data:', error);
            showNotification('Error fetching fuel data: ' + error.message, 'error');
            showLoadingState(false);
        });
}

// Show/hide loading state
function showLoadingState(isLoading) {
    const loadingOverlay = document.getElementById('loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.style.display = isLoading ? 'flex' : 'none';
    }
}

// Filter data by date range
function filterDataByDateRange(days) {
    // Handle custom date range
    if (days === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate);
            start.setHours(0, 0, 0, 0);
            const end = new Date(customEndDate);
            end.setHours(23, 59, 59, 999);
            
            return allFuelData.filter(fuelLog => {
                const createdAt = fuelLog.createdAt;
                if (!createdAt) return false;
                
                const logDate = new Date(createdAt);
                return logDate >= start && logDate <= end;
            });
        }
        // If no custom dates selected, return all data
        return allFuelData;
    }
    
    const now = new Date();
    const startDate = new Date(now);
    
    if (days === 'today') {
        startDate.setHours(0, 0, 0, 0);
    } else {
        startDate.setDate(now.getDate() - parseInt(days));
    }
    
    return allFuelData.filter(fuelLog => {
        const createdAt = fuelLog.createdAt;
        if (!createdAt) return false;
        
        const logDate = new Date(createdAt);
        return logDate >= startDate && logDate <= now;
    });
}

// Update KPI cards
function updateKPICards(filteredData) {
    const totalLogs = filteredData.length;
    
    // Calculate total fuel quantity
    const totalFuelQuantity = filteredData.reduce((sum, log) => {
        const quantity = parseFloat(log.fuelQuantity) || 0;
        return sum + quantity;
    }, 0);
    
    // Calculate outstation fuel quantity
    const outstationQuantity = filteredData.reduce((sum, log) => {
        const source = log.selectedSource || log.fillingSource || log.selectedBunker || '';
        if (source === 'Outstation' || source.toLowerCase().includes('outstation')) {
            const quantity = parseFloat(log.fuelQuantity) || 0;
            return sum + quantity;
        }
        return sum;
    }, 0);
    
    // Calculate average fuel quantity
    const averageQuantity = totalLogs > 0 ? (totalFuelQuantity / totalLogs) : 0;
    
    // Calculate total fuel cost
    const totalFuelCost = filteredData.reduce((sum, log) => {
        const cost = parseFloat(log.fuelCost) || 0;
        return sum + cost;
    }, 0);
    
    // Calculate outstation cost
    const outstationCost = filteredData.reduce((sum, log) => {
        const source = log.selectedSource || log.fillingSource || log.selectedBunker || '';
        if (source === 'Outstation' || source.toLowerCase().includes('outstation')) {
            const cost = parseFloat(log.fuelCost) || 0;
            return sum + cost;
        }
        return sum;
    }, 0);
    
    // Update card values
    document.querySelector('[data-kpi="fuel-logs"] .kpi-value').textContent = totalLogs;
    document.querySelector('[data-kpi="fuel-quantity"] .kpi-value').textContent = totalFuelQuantity.toFixed(2);
    document.querySelector('[data-kpi="outstation-quantity"] .kpi-value').textContent = outstationQuantity.toFixed(2);
    document.querySelector('[data-kpi="average-quantity"] .kpi-value').textContent = averageQuantity.toFixed(2);
    document.querySelector('[data-kpi="fuel-cost"] .kpi-value').textContent = '$' + totalFuelCost.toFixed(2);
    document.querySelector('[data-kpi="outstation-cost"] .kpi-value').textContent = '$' + outstationCost.toFixed(2);
    
    // Update additional KPI cards
    const fuelTypes = {};
    const bunkers = {};
    
    filteredData.forEach(log => {
        const fuelType = log.fuelType || 'Unknown';
        const bunker = log.selectedSource || log.selectedBunker || log.fillingSource || 'Unknown';
        
        fuelTypes[fuelType] = (fuelTypes[fuelType] || 0) + parseFloat(log.fuelQuantity || 0);
        bunkers[bunker] = (bunkers[bunker] || 0) + parseFloat(log.fuelQuantity || 0);
    });
    
    // Find most used fuel type and bunker
    const topFuelType = Object.keys(fuelTypes).length > 0 ? 
        Object.keys(fuelTypes).reduce((a, b) => fuelTypes[a] > fuelTypes[b] ? a : b) : 
        'No Data';
    const topBunker = Object.keys(bunkers).length > 0 ? 
        Object.keys(bunkers).reduce((a, b) => bunkers[a] > bunkers[b] ? a : b) : 
        'No Data';
    
    const additionalCards = document.querySelectorAll('.additional-kpi-cards .kpi-card');
    if (additionalCards[0]) {
        additionalCards[0].querySelector('.kpi-value').textContent = topBunker;
        additionalCards[0].querySelector('.kpi-value').style.fontSize = '1.5rem';
    }
    if (additionalCards[1]) {
        additionalCards[1].querySelector('.kpi-value').textContent = topFuelType;
        additionalCards[1].querySelector('.kpi-value').style.fontSize = '1.5rem';
    }
}

// Update all dashboard components
function updateDashboard() {
    const filteredData = filterDataByDateRange(currentDateRange);
    
    updateKPICards(filteredData);
    updateDatewiseFuelChart(filteredData);
    updateVehicleWiseChart(filteredData);
    updateFuelTypeChart(filteredData);
    updateFillingSourceChart(filteredData);
}

// Update datewise fuel quantity chart
function updateDatewiseFuelChart(data) {
    const dateRange = currentDateRange === 'today' ? 1 : 
                     currentDateRange === 'custom' ? 30 : 
                     parseInt(currentDateRange) || 7;
    const dates = [];
    const quantities = {};
    
    // Generate date labels
    for (let i = dateRange - 1; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        const dateStr = date.toLocaleDateString('en-GB');
        dates.push(dateStr);
        quantities[dateStr] = 0;
    }
    
    // Count fuel quantity by date
    data.forEach(log => {
        const createdAt = log.createdAt;
        if (createdAt) {
            const date = new Date(createdAt).toLocaleDateString('en-GB');
            if (quantities.hasOwnProperty(date)) {
                quantities[date] += parseFloat(log.fuelQuantity || 0);
            }
        }
    });
    
    updateChart('running', dates, dates.map(d => quantities[d]),
                ['#0055A5', '#5899DA', '#DC0D0E', '#945ECF', '#10b981', '#FFA500']);
}

// Update vehicle-wise fuel chart
function updateVehicleWiseChart(data) {
    const vehicles = {};
    data.forEach(log => {
        const vehicle = log.equipmentName || log.equipmentNo || 'Unknown';
        vehicles[vehicle] = (vehicles[vehicle] || 0) + parseFloat(log.fuelQuantity || 0);
    });
    
    updateChart('vehicle-wise-chart', Object.keys(vehicles), Object.values(vehicles),
                ['#0055A5', '#5899DA', '#DC0D0E', '#945ECF', '#10b981', '#FFA500']);
}

// Update fuel type chart
function updateFuelTypeChart(data) {
    const fuelTypes = {};
    data.forEach(log => {
        const fuelType = log.fuelType || 'Unknown';
        fuelTypes[fuelType] = (fuelTypes[fuelType] || 0) + parseFloat(log.fuelQuantity || 0);
    });
    
    updateChart('fuel-type-chart', Object.keys(fuelTypes), Object.values(fuelTypes),
                ['#0055A5', '#10b981', '#FFA500', '#DC0D0E']);
}

// Update filling source chart
function updateFillingSourceChart(data) {
    const sources = {};
    data.forEach(log => {
        const source = log.selectedSource || log.selectedBunker || log.fillingSource || 'Unknown';
        sources[source] = (sources[source] || 0) + parseFloat(log.fuelQuantity || 0);
    });
    
    updateChart('filling-source-chart', Object.keys(sources), Object.values(sources),
                ['#945ECF', '#5899DA', '#0055A5', '#10b981']);
}

// Generic chart update function
function updateChart(chartId, labels, data, colors) {
    const ctx = document.getElementById(chartId);
    if (!ctx) return;
    
    // Destroy existing chart
    if (chartInstances[chartId]) {
        chartInstances[chartId].destroy();
    }
    
    // Handle empty data
    if (!labels || labels.length === 0 || !data || data.length === 0) {
        labels = ['No Data'];
        data = [0];
        colors = ['#e0e0e0'];
    }
    
    // Create new chart
    chartInstances[chartId] = new Chart(chartId, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                backgroundColor: colors,
                data: data
            }]
        },
            options: {
            legend: { display: false },
            scales: {
                yAxes: [{ ticks: { beginAtZero: true } }]
            },
                title: {
                    display: true,
                text: ctx.closest('.chart-card')?.querySelector('h3')?.textContent || ''
            },
            tooltips: {
                enabled: labels[0] !== 'No Data',
                callbacks: {
                    label: function(tooltipItem, data) {
                        const value = data.datasets[0].data[tooltipItem.index];
                        // Format based on chart type
                        if (chartId.includes('cost')) {
                            return '$' + value.toFixed(2);
                        } else {
                            return value.toFixed(2) + ' Liters';
                        }
                    }
                }
            }
        }
    });
}

// Export to Excel functionality
function exportToExcel() {
    const filteredData = filterDataByDateRange(currentDateRange);
    
    if (filteredData.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    // Prepare data for Excel
    const excelData = filteredData.map((log, index) => {
        return {
            'S.No': index + 1,
            'Request Number': log.requestNumber || 'N/A',
            'Country': log.country || 'N/A',
            'Project': log.project || 'N/A',
            'Equipment Name': log.equipmentName || 'N/A',
            'Equipment No': log.equipmentNo || 'N/A',
            'Equipment ID': log.equipmentId || 'N/A',
            'Fuel Type': log.fuelType || 'N/A',
            'Fuel Quantity': log.fuelQuantity || 0,
            'Fuel Unit': log.fuelUnit || 'Liters',
            'Fuel Cost': log.fuelCost || 0,
            'Filling Source': log.selectedSource || log.selectedBunker || log.fillingSource || 'N/A',
            'Receiver Name': log.receiverName || 'N/A',
            'Meter Reading': log.meterReading || 'N/A',
            'Meter Unit': log.meterUnit || 'N/A',
            'Notes': log.note || 'N/A',
            'Created At': log.createdAt ? new Date(log.createdAt).toLocaleString() : 'N/A',
            'Updated At': log.updatedAt ? new Date(log.updatedAt).toLocaleString() : 'N/A'
        };
    });
    
    // Create workbook and worksheet
    const wb = XLSX.utils.book_new();
    const ws = XLSX.utils.json_to_sheet(excelData);
    
    // Set column widths
    const colWidths = [
        { wch: 6 },  // S.No
        { wch: 20 }, // Request Number
        { wch: 12 }, // Country
        { wch: 15 }, // Project
        { wch: 20 }, // Equipment Name
        { wch: 15 }, // Equipment No
        { wch: 12 }, // Equipment ID
        { wch: 12 }, // Fuel Type
        { wch: 15 }, // Fuel Quantity
        { wch: 12 }, // Fuel Unit
        { wch: 12 }, // Fuel Cost
        { wch: 18 }, // Filling Source
        { wch: 18 }, // Receiver Name
        { wch: 20 }, // Meter Reading
        { wch: 12 }, // Meter Unit
        { wch: 30 }, // Notes
        { wch: 20 }, // Created At
        { wch: 20 }  // Updated At
    ];
    ws['!cols'] = colWidths;
    
    // Add worksheet to workbook
    XLSX.utils.book_append_sheet(wb, ws, 'Fuel Logs');
    
    // Create summary sheet
    const totalQuantity = filteredData.reduce((sum, log) => sum + parseFloat(log.fuelQuantity || 0), 0);
    const totalCost = filteredData.reduce((sum, log) => sum + parseFloat(log.fuelCost || 0), 0);
    const avgQuantity = filteredData.length > 0 ? totalQuantity / filteredData.length : 0;
    
    const summaryData = [
        { 'Metric': 'Total Fuel Logs', 'Value': filteredData.length },
        { 'Metric': 'Total Fuel Quantity', 'Value': totalQuantity.toFixed(2) + ' Liters' },
        { 'Metric': 'Total Fuel Cost', 'Value': '$' + totalCost.toFixed(2) },
        { 'Metric': 'Average Quantity per Log', 'Value': avgQuantity.toFixed(2) + ' Liters' },
        { 'Metric': 'Date Range', 'Value': getCurrentDateRangeText() },
        { 'Metric': 'Export Date', 'Value': new Date().toLocaleString() }
    ];
    
    const wsSummary = XLSX.utils.json_to_sheet(summaryData);
    wsSummary['!cols'] = [{ wch: 25 }, { wch: 30 }];
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    // Generate filename
    const dateRangeText = getCurrentDateRangeText().replace(/\s+/g, '_');
    const timestamp = new Date().toISOString().slice(0, 10);
    const filename = `Fuel_Management_KPI_${dateRangeText}_${timestamp}.xlsx`;
    
    // Export file
    XLSX.writeFile(wb, filename);
    
    showNotification(`Exported ${filteredData.length} fuel log(s) to Excel successfully`, 'success');
}

// Get current date range as text for filename
function getCurrentDateRangeText() {
    if (currentDateRange === 'today') {
        return 'Today';
    } else if (currentDateRange === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            const end = new Date(customEndDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            return `Custom_${start}_to_${end}`;
        }
        return 'All_Data';
    } else {
        return `Last_${currentDateRange}_Days`;
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', () => {
    fetchFuelData();
    
    // Add export to Excel button listener
    const exportBtnHeader = document.getElementById('export-excel-btn-header');
    if (exportBtnHeader) {
        exportBtnHeader.addEventListener('click', () => {
            exportToExcel();
        });
    }
    
    // Add refresh button listener
    const refreshBtnHeader = document.getElementById('refresh-data-btn-header');
    if (refreshBtnHeader) {
        refreshBtnHeader.addEventListener('click', () => {
            showNotification('Refreshing data...', 'info');
            fetchFuelData();
        });
    }
    
    // Custom date range functionality
    const customDateRange = document.getElementById('custom-date-range');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');
    const applyDatesBtn = document.getElementById('apply-custom-dates');
    const clearDatesBtn = document.getElementById('clear-custom-dates');
    
    // Apply custom dates
    if (applyDatesBtn) {
        applyDatesBtn.addEventListener('click', () => {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                showNotification('Please select both start and end dates', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                showNotification('Start date cannot be after end date', 'error');
                return;
            }
            
            customStartDate = startDate;
            customEndDate = endDate;
            currentDateRange = 'custom';
            
            // Update active button
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === 'custom') {
                    btn.classList.add('active');
                }
            });
            
            updateDashboard();
            showNotification(`Showing data from ${new Date(startDate).toLocaleDateString()} to ${new Date(endDate).toLocaleDateString()}`, 'success');
        });
    }
    
    // Clear custom dates
    if (clearDatesBtn) {
        clearDatesBtn.addEventListener('click', () => {
            startDateInput.value = '';
            endDateInput.value = '';
            customStartDate = null;
            customEndDate = null;
            
            // Switch back to default (7 days)
            currentDateRange = 7;
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === '7days') {
                    btn.classList.add('active');
                }
            });
            
            // Hide custom date range picker
            if (customDateRange) {
                customDateRange.style.display = 'none';
            }
            
            updateDashboard();
            showNotification('Custom date range cleared', 'info');
        });
    }
    
    // Set max date to today for date inputs
    const today = new Date().toISOString().split('T')[0];
    if (startDateInput) startDateInput.max = today;
    if (endDateInput) endDateInput.max = today;
    
    // When start date changes, update end date minimum
    if (startDateInput && endDateInput) {
        startDateInput.addEventListener('change', () => {
            if (startDateInput.value) {
                endDateInput.min = startDateInput.value;
                if (endDateInput.value && new Date(endDateInput.value) < new Date(startDateInput.value)) {
                    endDateInput.value = '';
                }
            }
        });
        
        // Set default dates when custom date picker is shown
        const periodButtons = document.querySelectorAll('.period-btn');
        periodButtons.forEach(btn => {
            if (btn.getAttribute('data-period') === 'custom') {
                btn.addEventListener('click', () => {
                    if (!startDateInput.value || !endDateInput.value) {
                        const end = new Date();
                        const start = new Date();
                        start.setDate(end.getDate() - 7);
                        
                        startDateInput.value = start.toISOString().split('T')[0];
                        endDateInput.value = end.toISOString().split('T')[0];
                    }
                });
            }
        });
    }
});
</script>

<style>
/* CSS Variables for consistent theming - Matching Workorder */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Inter', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
    --z-index-modal: 1055;
}

/* Import Google Fonts for better typography */
@import url('https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap');

/* ===== MAIN CONTAINER ===== */
.fuel-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
    color: var(--text-primary);
}

/* ===== HEADER STYLES - Matching Workorder Design ===== */
.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.create-btn i {
    font-size: 18px;
}

/* ===== MAIN CONTENT ===== */
.fuel-kpi-content {
    padding: 30px 20px;
    background: var(--bg-light);
    min-height: auto;
    overflow: visible;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    font-family: var(--font-family);
    margin-bottom: 1rem;
    margin: 0;
    overflow: hidden;
}

/* ===== TIME PERIOD SELECTOR ===== */
.time-period-selector {
    margin-bottom: 2rem;
}

.period-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.period-btn {
    padding: 12px 24px;
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.period-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.period-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

/* Custom Date Range Picker */
.custom-date-range {
    margin-top: 1rem;
    padding: 1.5rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.date-range-inputs {
    display: flex;
    align-items: end;
    gap: 1rem;
    flex-wrap: wrap;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.date-input-group label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.date-input {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
    transition: var(--transition-base);
    min-width: 180px;
    cursor: pointer;
}

.date-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-apply-dates,
.btn-clear-dates {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.btn-apply-dates {
    background: var(--primary-color);
    color: white;
}

.btn-apply-dates:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-clear-dates {
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-clear-dates:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

/* ===== MODERN TABS ===== */
.modern-tabs {
    margin-bottom: 2rem;
}

.tab-navigation {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 0;
    margin-bottom: 2rem;
    overflow-x: auto;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: transparent;
    color: var(--text-secondary);
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    position: relative;
    border-bottom: 2px solid transparent;
}

.tab-btn:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-btn.active {
    color: var(--primary-color);
    background: var(--bg-light);
    border-bottom-color: var(--primary-color);
}

.tab-btn.active::after {
    content: none;
}

/* ===== TAB CONTENT ===== */
.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

/* ===== KPI CARDS ===== */
.kpi-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.kpi-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 1rem;
}

.kpi-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.kpi-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--bg-white);
    flex-shrink: 0;
}

.kpi-icon.success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}

.kpi-icon.warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
}

.kpi-icon.info {
    background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%);
}

.kpi-icon.secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
}

.kpi-content {
    flex: 1;
}

.kpi-content h3 {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0 0 8px 0;
}

.kpi-value {
    font-size: 32px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0 0 4px 0;
    line-height: 1;
}

.kpi-label {
    font-size: 12px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

/* ===== OVERVIEW BOTTOM SECTION ===== */
.overview-bottom-section {
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 1.5rem;
    margin-top: 1.5rem;
}

.additional-kpi-cards {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.chart-section {
    display: flex;
    flex-direction: column;
}

/* ===== CHARTS SECTION ===== */
.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.chart-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
    border-color: var(--primary-color);
}

.chart-header {
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 1px solid var(--border-light);
}

.chart-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.chart-container {
    position: relative;
    height: 250px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chart-container canvas {
    max-width: 100%;
    max-height: 100%;
}

/* ===== RESPONSIVE DESIGN ===== */
@media (max-width: 992px) {
    .kpi-cards-grid {
        grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
        gap: 16px;
    }
    
    .charts-grid {
        grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
        gap: 16px;
    }
    
    .overview-bottom-section {
        grid-template-columns: 1fr;
        gap: 16px;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .header-right {
        gap: 10px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .fuel-kpi-container,
    .fuel-kpi-content {
        padding: 20px 15px;
    }
    
    .content-box {
        padding: 20px;
        margin: 0;
        border-radius: var(--radius-lg);
    }
    
    .time-period-selector {
        margin-bottom: 1.5rem;
    }
    
    .period-buttons {
        display: grid;
        grid-template-columns: repeat(auto-fit, minmax(120px, 1fr));
        gap: 8px;
    }
    
    .period-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
        text-align: center;
    }
    
    .custom-date-range {
        padding: 1rem;
    }
    
    .date-range-inputs {
        flex-direction: column;
        gap: 0.75rem;
    }
    
    .date-input-group {
        width: 100%;
    }
    
    .date-input {
        width: 100%;
        min-width: auto;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        width: 100%;
        justify-content: center;
    }
    
    .tab-navigation {
        flex-direction: column;
        gap: 0;
        border-bottom: 1px solid var(--border-color);
    }
    
    .tab-btn {
        justify-content: flex-start;
        padding: 12px 16px;
        font-size: 14px;
        min-height: 44px;
        border-bottom: 1px solid var(--border-color);
    }
    
    .kpi-cards-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .kpi-card {
        padding: 16px;
        flex-direction: column;
        text-align: center;
        gap: 12px;
    }
    
    .kpi-icon {
        width: 48px;
        height: 48px;
    }
    
    .kpi-content h3 {
        font-size: 14px;
    }
    
    .kpi-value {
        font-size: 24px;
    }
    
    .kpi-label {
        font-size: 12px;
    }
    
    .overview-bottom-section {
        grid-template-columns: 1fr;
        gap: 16px;
    }
    
    .charts-grid {
        grid-template-columns: 1fr;
        gap: 12px;
    }
    
    .chart-card {
        padding: 16px;
    }
    
    .chart-header h3 {
        font-size: 16px;
    }
    
    .chart-container {
        height: 200px;
    }
}

@media (max-width: 576px) {
    .header-title {
        font-size: 20px;
    }
    
    .header-right {
        flex-direction: column;
        gap: 8px;
        align-items: stretch;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
        padding: 10px 12px;
        font-size: 13px;
    }
    
    .content-box {
        padding: 14px;
    }
    
    .period-buttons {
        grid-template-columns: repeat(auto-fit, minmax(100px, 1fr));
        gap: 6px;
    }
    
    .period-btn {
        padding: 8px 10px;
        font-size: 12px;
        min-height: 36px;
    }
    
    .custom-date-range {
        padding: 0.875rem;
    }
    
    .date-input {
        padding: 10px;
        font-size: 14px;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        padding: 10px 14px;
        font-size: 13px;
    }
    
    .tab-btn {
        padding: 10px 14px;
        font-size: 13px;
        min-height: 40px;
    }
    
    .kpi-card {
        padding: 14px;
    }
    
    .kpi-icon {
        width: 44px;
        height: 44px;
    }
    
    .kpi-content h3 {
        font-size: 13px;
    }
    
    .kpi-value {
        font-size: 22px;
    }
    
    .kpi-label {
        font-size: 11px;
    }
    
    .chart-container {
        height: 190px;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .header-title {
        font-size: 18px;
    }
    
    .back-button {
        font-size: 18px;
    }
    
    .header-right {
        width: 100%;
    }
    
    .create-btn {
        font-size: 12px;
        padding: 9px 12px;
    }
    
    .content-box {
        padding: 12px;
    }
    
    .period-buttons {
        grid-template-columns: 1fr;
        gap: 6px;
    }
    
    .period-btn {
        padding: 8px 12px;
        font-size: 12px;
        min-height: 36px;
    }
    
    .custom-date-range {
        padding: 0.75rem;
    }
    
    .date-input {
        padding: 9px;
        font-size: 13px;
    }
    
    .btn-apply-dates,
    .btn-clear-dates {
        padding: 9px 12px;
        font-size: 12px;
    }
    
    .tab-btn {
        padding: 10px 12px;
        font-size: 13px;
        min-height: 40px;
    }
    
    .kpi-card {
        padding: 12px;
        flex-direction: column;
        text-align: center;
        gap: 8px;
    }
    
    .kpi-icon {
        width: 40px;
        height: 40px;
    }
    
    .kpi-content h3 {
        font-size: 13px;
    }
    
    .kpi-value {
        font-size: 20px;
    }
    
    .kpi-label {
        font-size: 11px;
    }
    
    .chart-card {
        padding: 12px;
    }
    
    .chart-header h3 {
        font-size: 14px;
    }
    
    .chart-container {
        height: 180px;
    }
}

/* ===== LOADING SPINNER ===== */
@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

/* Loading/Sync Modal - Matching Workorder Design */
.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    font-size: 40px;
    animation: spin 1s linear infinite;
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

/* ===== ACCESSIBILITY ===== */
@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

/* ===== FOCUS STYLES ===== */
button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}

/* ===== PRINT STYLES ===== */
@media print {
    .header-gradient,
    .period-buttons,
    .tab-navigation {
        display: none !important;
    }
    
    .fuel-kpi-container {
        background: white;
    }
    
    .kpi-card,
    .chart-card {
        border: 1px solid #000;
        break-inside: avoid;
    }
}
</style>

<script>
// Optimized Fuel Management KPI Functionality
(function() {
    'use strict';
    
    // Configuration
    const CONFIG = {
        notificationTimeout: 5000,
        selectors: {
            tabButtons: '.tab-btn',
            tabPanes: '.tab-pane',
            periodButtons: '.period-btn',
            kpiCards: '.kpi-card',
            chartContainers: '.chart-container'
        }
    };

    // Cached DOM elements
    let elements = {};
    
    // Initialize DOM elements
    function initElements() {
        elements = {
            tabButtons: document.querySelectorAll(CONFIG.selectors.tabButtons),
            tabPanes: document.querySelectorAll(CONFIG.selectors.tabPanes),
            periodButtons: document.querySelectorAll(CONFIG.selectors.periodButtons),
            kpiCards: document.querySelectorAll(CONFIG.selectors.kpiCards),
            chartContainers: document.querySelectorAll(CONFIG.selectors.chartContainers)
        };
    }

    // Tab switching functionality
    function initTabSwitching() {
        elements.tabButtons.forEach(button => {
            button.addEventListener('click', function() {
                const targetTab = this.getAttribute('data-tab');

                // Remove active class from all tabs and panes
                elements.tabButtons.forEach(btn => btn.classList.remove('active'));
                elements.tabPanes.forEach(pane => pane.classList.remove('active'));

                // Add active class to clicked tab and corresponding pane
                this.classList.add('active');
                const targetPane = document.getElementById(targetTab);
                if (targetPane) {
                    targetPane.classList.add('active');
                }
            });
        });
    }

    // Time period selector functionality
    function initPeriodSelector() {
        const customDateRange = document.getElementById('custom-date-range');
        
        elements.periodButtons.forEach(button => {
            button.addEventListener('click', function() {
                // Remove active class from all period buttons
                elements.periodButtons.forEach(btn => btn.classList.remove('active'));
                
                // Add active class to clicked button
                this.classList.add('active');
                
                const period = this.getAttribute('data-period');
                console.log('Period changed to:', period);
                
                // Show/hide custom date range picker
                if (period === 'custom') {
                    if (customDateRange) {
                        customDateRange.style.display = 'block';
                    }
                    currentDateRange = 'custom';
                    return;
                } else {
                    // Hide custom date range picker for other periods
                    if (customDateRange) {
                        customDateRange.style.display = 'none';
                    }
                    
                    // Update current date range
                    if (period === 'today') {
                        currentDateRange = 'today';
                    } else if (period === '7days') {
                        currentDateRange = 7;
                    } else if (period === '30days') {
                        currentDateRange = 30;
                    } else if (period === '90days') {
                        currentDateRange = 90;
                    }
                    
                    // Update dashboard with new date range
                    updateDashboard();
                }
            });
        });
    }

    // KPI card hover effects
    function initKpiCardEffects() {
        elements.kpiCards.forEach(card => {
            card.addEventListener('mouseenter', function() {
                this.style.transform = 'translateY(-4px)';
            });
            
            card.addEventListener('mouseleave', function() {
                this.style.transform = 'translateY(0)';
            });
        });
    }

    // Chart container responsive handling
    function initChartContainers() {
        elements.chartContainers.forEach(container => {
            const canvas = container.querySelector('canvas');
            if (canvas) {
                // Ensure canvas is responsive
                canvas.style.maxWidth = '100%';
                canvas.style.height = 'auto';
            }
        });
    }

    // Notification system for user feedback
    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span>${message}</span>
                <button class="notification-close" aria-label="Close notification">&times;</button>
            </div>
        `;
        
        document.body.appendChild(notification);
        
        // Auto remove after timeout
        setTimeout(() => removeNotification(notification), CONFIG.notificationTimeout);
        
        // Close button functionality
        notification.querySelector('.notification-close').addEventListener('click', () => {
            removeNotification(notification);
        });
    }
    
    function removeNotification(notification) {
        if (notification && notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }
    
    // Add notification styles - Matching workorder design
    function addNotificationStyles() {
        const notificationStyles = document.createElement('style');
        notificationStyles.textContent = `
            .notification {
                position: fixed;
                top: 20px;
                right: 20px;
                z-index: var(--z-index-notification);
                padding: 16px;
                border-radius: 8px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
                font-family: var(--font-family);
                font-weight: 500;
                max-width: 400px;
                min-width: 300px;
                animation: slideIn 0.3s ease;
                background: white;
            }
            
            .notification-success {
                border-left: 4px solid #28a745;
            }
            
            .notification-error {
                border-left: 4px solid #dc3545;
            }
            
            .notification-info {
                border-left: 4px solid #17a2b8;
            }
            
            .notification-content {
                display: flex;
                justify-content: space-between;
                align-items: center;
                gap: 1rem;
            }
            
            .notification-content span {
                color: #333;
                font-size: 14px;
            }
            
            .notification-close {
                background: none;
                border: none;
                font-size: 18px;
                color: #999;
                cursor: pointer;
                padding: 0;
                line-height: 1;
                margin-left: 12px;
            }
            
            .notification-close:hover {
                color: #666;
            }
            
            @keyframes slideIn {
                from {
                    transform: translateX(100%);
                    opacity: 0;
                }
                to {
                    transform: translateX(0);
                    opacity: 1;
                }
            }
        `;
        document.head.appendChild(notificationStyles);
    }

    // Initialize everything
    function init() {
        initElements();
        initTabSwitching();
        initPeriodSelector();
        initKpiCardEffects();
        initChartContainers();
        addNotificationStyles();
    }

    // Start when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
</script>
@endsection