@extends('welcome')

@section('title', 'Employees Who Closed Jobs')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="employees-closed-container">
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Employee Data...</div>
        </div>
    </div>
    
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ route('dashboard.team_kpi.home') }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Employees Who Completed or Rejected Jobs</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn">
                    <i class="fas fa-file-excel"></i>
                    Export
                </button>
                <button type="button" class="create-btn" id="refresh-btn">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="employees-closed-content">
        <div class="content-box">
            <!-- Date Range Picker -->
            <div class="date-picker-container">
                <div class="date-inputs">
                    <div class="date-input-group">
                        <label>From Date:</label>
                        <input type="date" id="from-date" class="date-input">
                    </div>
                    <div class="date-input-group">
                        <label>To Date:</label>
                        <input type="date" id="to-date" class="date-input">
                    </div>
                    <button class="btn-clear-dates" id="clear-dates">Clear</button>
                </div>
            </div>
            
            <!-- Summary Stats -->
            <div class="summary-header">
                <div class="summary-stat">
                    <i class="fas fa-users"></i>
                    <div>
                        <div class="stat-label">Total Employees</div>
                        <div class="stat-value" id="total-employees">0</div>
                    </div>
                </div>
                <div class="summary-stat">
                    <i class="fas fa-clipboard-check"></i>
                    <div>
                        <div class="stat-label">Total Jobs Closed</div>
                        <div class="stat-value" id="total-jobs">0</div>
                    </div>
                </div>
                <div class="summary-stat">
                    <i class="fas fa-tasks"></i>
                    <div>
                        <div class="stat-label">Total Tasks</div>
                        <div class="stat-value" id="total-tasks">0</div>
                    </div>
                </div>
            </div>
            
            <!-- Employees List -->
            <div id="employees-list" class="employees-list">
                <div class="loading-state">
                    <i class="fas fa-spinner fa-spin"></i>
                    <span>Loading employees data...</span>
                </div>
            </div>
        </div>
    </div>
</div>

<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>

<script>
const firebaseConfig = { databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com" };
if (!firebase.apps.length) firebase.initializeApp(firebaseConfig);
const database = firebase.database();

let userTaskMap = [];
let fromDate = null;
let toDate = null;

function showLoadingState(isLoading) {
    document.getElementById('loading-overlay').style.display = isLoading ? 'flex' : 'none';
}

async function fetchData() {
    showLoadingState(true);
    try {
        // 1. Fetch all job cards globally (recursive)
        const jobCardsSnap = await database.ref('/GlobalWorkOrders').once('value');
        const jobRootData = jobCardsSnap.val() || {};
        const allJobCards = [];
        
        function extractJobCards(obj, path = []) {
            if (!obj || typeof obj !== 'object') return;
            
            Object.entries(obj).forEach(([key, item]) => {
                if (key === 'lastRequestNumber' || typeof item !== 'object' || item === null) return;
                
                // Check if this looks like a job card
                const hasJobCardFields = item.requestNumber || item.title || item.jobCardNumber || item.status || item.equipment || item.items || item.createdAt;
                if (hasJobCardFields) {
                    allJobCards.push({ 
                        ...item, 
                        adminUid: path[0] || 'unknown', 
                        jobId: key 
                    });
                } else {
                    // Recurse deeper
                    extractJobCards(item, [...path, key]);
                }
            });
        }
        
        extractJobCards(jobRootData);
        
        console.log('Total job cards found:', allJobCards.length);
        
        // 2. Build user-task map
        const userMap = {};
        
        for (const job of allJobCards) {
            // Apply date filtering if dates are set
            if (fromDate || toDate) {
                const createdAt = job.createdAt ? new Date(job.createdAt) : null;
                
                if (fromDate && createdAt) {
                    const from = new Date(fromDate);
                    from.setHours(0, 0, 0, 0);
                    const jobDate = new Date(createdAt);
                    jobDate.setHours(0, 0, 0, 0);
                    if (jobDate < from) continue;
                }
                
                if (toDate && createdAt) {
                    const to = new Date(toDate);
                    to.setHours(23, 59, 59, 999);
                    if (createdAt > to) continue;
                }
            }
            
            const jobCardNumber = job.requestNumber || job.title || job.jobCardNumber || job.jobId;
            const inspectionNumber = job.linkInspection?.name || job.linkInspection?.id || '-';
            const jobCardStatus = job.status || job.jobStatus || job.state || '-';
            const items = Array.isArray(job.items) ? job.items : [];
            
            for (const task of items) {
                if (Array.isArray(task.assignTo)) {
                    for (const user of task.assignTo) {
                        if (!user?.id) continue;
                        
                        if (!userMap[user.id]) {
                            userMap[user.id] = {
                                user,
                                jobs: [],
                            };
                        }
                        
                        // Find if this job is already added for this user
                        let jobEntry = userMap[user.id].jobs.find(j => j.jobCardNumber === jobCardNumber);
                        if (!jobEntry) {
                            jobEntry = {
                                jobCardNumber,
                                inspectionNumber,
                                adminUid: job.adminUid,
                                jobId: job.jobId,
                                jobCardStatus,
                                createdAt: job.createdAt,
                                tasks: [],
                            };
                            userMap[user.id].jobs.push(jobEntry);
                        }
                        
                        jobEntry.tasks.push({
                            taskId: task.id,
                            text: task.text,
                            note: task.note,
                            status: task.status,
                        });
                    }
                }
            }
        }
        
        // 3. Fetch user details for all user IDs (recursive)
        const userIds = Object.keys(userMap);
        const usersSnap = await database.ref('/Globalusers').once('value');
        const usersRootData = usersSnap.val() || {};
        
        // Extract all users recursively
        const allUsersMap = {};
        
        function extractUsers(obj, path = []) {
            if (!obj || typeof obj !== 'object') return;
            
            Object.entries(obj).forEach(([key, item]) => {
                if (typeof item !== 'object' || item === null) return;
                
                // Check if this looks like a user object
                const hasUserFields = item.email || item.fullName || item.name || item.employeeNumber || item.role;
                if (hasUserFields) {
                    allUsersMap[key] = item;
                } else {
                    // Recurse deeper
                    extractUsers(item, [...path, key]);
                }
            });
        }
        
        extractUsers(usersRootData);
        
        console.log('Total users found:', Object.keys(allUsersMap).length);
        
        // Update user details
        userIds.forEach(uid => {
            const userDetails = allUsersMap[uid];
            if (userDetails) {
                userMap[uid].user = {
                    ...userMap[uid].user,
                    fullName: userDetails.fullName || userDetails.name || userMap[uid].user.fullName || userMap[uid].user.name,
                    employeeNumber: userDetails.employeeNumber || userMap[uid].user.employeeNumber,
                    mobile: userDetails.mobile || userDetails.phone || '-',
                };
            }
        });
        
        userTaskMap = Object.values(userMap);
        
        console.log('Users with tasks:', userTaskMap.length);
        
        renderEmployeesList();
        updateSummary();
        
        showLoadingState(false);
        showNotification(`Loaded ${userTaskMap.length} employees`, 'success');
    } catch (error) {
        console.error('Error fetching data:', error);
        showNotification('Error loading data', 'error');
        showLoadingState(false);
        document.getElementById('employees-list').innerHTML = '<div class="no-data">Error loading data</div>';
    }
}

function renderEmployeesList() {
    const container = document.getElementById('employees-list');
    
    if (userTaskMap.length === 0) {
        container.innerHTML = '<div class="no-data">No participation data found</div>';
        return;
    }
    
    const html = userTaskMap.map((entry, idx) => {
        // Only include jobs with completed or rejected status
        const filteredJobs = entry.jobs.filter(job => {
            const s = (job.jobCardStatus || '').toLowerCase();
            return s === 'completed' || s === 'complete' || s === 'success' || s === 'rejected';
        });
        
        if (filteredJobs.length === 0) return '';
        
        // Most recent job for header status
        const mainJob = filteredJobs[filteredJobs.length - 1];
        let mainStatus = mainJob?.jobCardStatus || '-';
        let statusColor = getStatusColor(mainStatus);
        
        const jobsHtml = filteredJobs.map((job, jdx) => {
            const jobStatusColor = getStatusColor(job.jobCardStatus);
            
            return `
                <div class="job-card">
                    <div class="job-info-row">
                        <strong><i class="fas fa-file-alt"></i> Job Card:</strong> ${job.jobCardNumber || '-'}
                    </div>
                    <div class="job-info-row">
                        <strong><i class="fas fa-clipboard-check"></i> Inspection Report:</strong> ${job.inspectionNumber || '-'}
                    </div>
                    <div class="job-info-row">
                        <strong><i class="fas fa-info-circle"></i> Job Card Status:</strong> 
                        <span class="status-badge" style="background-color: ${jobStatusColor}; color: white">${job.jobCardStatus || '-'}</span>
                    </div>
                    ${job.createdAt ? `
                        <div class="job-info-row">
                            <strong><i class="fas fa-calendar"></i> Created:</strong> ${new Date(job.createdAt).toLocaleString('en-GB')}
                        </div>
                    ` : ''}
                    <div class="job-info-row">
                        <strong><i class="fas fa-tasks"></i> Tasks:</strong>
                    </div>
                    ${job.tasks.map((task, tidx) => `
                        <div class="task-item">
                            <i class="fas fa-check-circle"></i>
                            ${task.text} <span class="task-status">(${task.status})</span>
                            ${task.note ? `<br><span class="task-note">Note: ${task.note}</span>` : ''}
                        </div>
                    `).join('')}
                </div>
            `;
        }).join('');
        
        return `
            <div class="user-card">
                <div class="user-header">
                    <div class="user-name">${entry.user.fullName || entry.user.name || 'Unknown'}</div>
                    <div class="status-badge-header" style="background-color: ${statusColor}; color: white">${mainStatus}</div>
                </div>
                <div class="user-info">
                    <span><i class="fas fa-id-card"></i> Emp#: ${entry.user.employeeNumber || '-'}</span>
                    <span><i class="fas fa-phone"></i> Mobile: ${entry.user.mobile || '-'}</span>
                    <span><i class="fas fa-briefcase"></i> Jobs: ${filteredJobs.length}</span>
                </div>
                ${jobsHtml}
            </div>
        `;
    }).filter(Boolean).join('');
    
    container.innerHTML = html || '<div class="no-data">No employees with completed or rejected jobs found</div>';
}

function getStatusColor(status) {
    if (typeof status !== 'string') return '#888';
    
    const s = status.toLowerCase();
    if (s === 'completed' || s === 'complete' || s === 'success') return '#2ecc40';
    if (s === 'rejected') return '#d81b60';
    if (s === 'processing' || s === 'in progress') return '#ff9800';
    if (s === 'initialized' || s === 'on hold') return '#fbc02d';
    return '#888';
}

function updateSummary() {
    let totalJobs = 0;
    let totalTasks = 0;
    
    const employeesWithClosedJobs = userTaskMap.filter(entry => {
        const filteredJobs = entry.jobs.filter(job => {
            const s = (job.jobCardStatus || '').toLowerCase();
            return s === 'completed' || s === 'complete' || s === 'success' || s === 'rejected';
        });
        
        if (filteredJobs.length > 0) {
            totalJobs += filteredJobs.length;
            filteredJobs.forEach(job => {
                totalTasks += job.tasks.length;
            });
            return true;
        }
        return false;
    });
    
    document.getElementById('total-employees').textContent = employeesWithClosedJobs.length;
    document.getElementById('total-jobs').textContent = totalJobs;
    document.getElementById('total-tasks').textContent = totalTasks;
}

function exportToExcel() {
    const employeesWithClosedJobs = userTaskMap.filter(entry => {
        const filteredJobs = entry.jobs.filter(job => {
            const s = (job.jobCardStatus || '').toLowerCase();
            return s === 'completed' || s === 'complete' || s === 'success' || s === 'rejected';
        });
        return filteredJobs.length > 0;
    });
    
    if (employeesWithClosedJobs.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    const excelData = [];
    
    employeesWithClosedJobs.forEach(entry => {
        const filteredJobs = entry.jobs.filter(job => {
            const s = (job.jobCardStatus || '').toLowerCase();
            return s === 'completed' || s === 'complete' || s === 'success' || s === 'rejected';
        });
        
        filteredJobs.forEach(job => {
            job.tasks.forEach((task, taskIdx) => {
                excelData.push({
                    'Employee Name': entry.user.fullName || entry.user.name || '-',
                    'Employee Number': entry.user.employeeNumber || '-',
                    'Mobile': entry.user.mobile || '-',
                    'Job Card Number': job.jobCardNumber || '-',
                    'Inspection Number': job.inspectionNumber || '-',
                    'Job Card Status': job.jobCardStatus || '-',
                    'Job Created Date': job.createdAt ? new Date(job.createdAt).toLocaleString('en-GB') : '-',
                    'Task': task.text || '-',
                    'Task Status': task.status || '-',
                    'Task Note': task.note || '-'
                });
            });
        });
    });
    
    const ws = XLSX.utils.json_to_sheet(excelData);
    ws['!cols'] = [
        { wch: 25 }, { wch: 18 }, { wch: 15 }, { wch: 20 }, { wch: 20 },
        { wch: 18 }, { wch: 20 }, { wch: 30 }, { wch: 15 }, { wch: 30 }
    ];
    const wb = XLSX.utils.book_new();
    XLSX.utils.book_append_sheet(wb, ws, 'Employees Closed Jobs');
    
    // Add summary sheet
    const totalJobs = employeesWithClosedJobs.reduce((sum, e) => sum + e.jobs.filter(j => {
        const s = (j.jobCardStatus || '').toLowerCase();
        return s === 'completed' || s === 'complete' || s === 'success' || s === 'rejected';
    }).length, 0);
    
    const summary = [
        { 'Metric': 'Total Employees', 'Value': employeesWithClosedJobs.length },
        { 'Metric': 'Total Jobs Closed', 'Value': totalJobs },
        { 'Metric': 'Total Tasks', 'Value': excelData.length },
        { 'Metric': 'Date Range From', 'Value': fromDate ? fromDate.toLocaleDateString('en-GB') : 'Not Set' },
        { 'Metric': 'Date Range To', 'Value': toDate ? toDate.toLocaleDateString('en-GB') : 'Not Set' },
        { 'Metric': 'Export Date', 'Value': new Date().toLocaleString('en-GB') }
    ];
    const wsSummary = XLSX.utils.json_to_sheet(summary);
    wsSummary['!cols'] = [{ wch: 25 }, { wch: 30 }];
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    const timestamp = new Date().toISOString().split('T')[0];
    XLSX.writeFile(wb, `Employees_Closed_Jobs_${timestamp}.xlsx`);
    showNotification('Data exported successfully', 'success');
}

function showNotification(message, type) {
    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    notification.innerHTML = `<div class="notification-content"><span>${message}</span><button class="notification-close">&times;</button></div>`;
    document.body.appendChild(notification);
    setTimeout(() => notification.remove(), 5000);
    notification.querySelector('.notification-close').addEventListener('click', () => notification.remove());
}

document.addEventListener('DOMContentLoaded', () => {
    // Initial fetch without date filter (shows all data)
    fetchData();
    
    // Date range
    document.getElementById('from-date').addEventListener('change', function() {
        if (this.value) {
            fromDate = new Date(this.value);
            fromDate.setHours(0, 0, 0, 0);
            document.getElementById('to-date').min = this.value;
        } else {
            fromDate = null;
        }
        fetchData();
    });
    
    document.getElementById('to-date').addEventListener('change', function() {
        if (this.value) {
            toDate = new Date(this.value);
            toDate.setHours(23, 59, 59, 999);
        } else {
            toDate = null;
        }
        fetchData();
    });
    
    document.getElementById('clear-dates').addEventListener('click', () => {
        document.getElementById('from-date').value = '';
        document.getElementById('to-date').value = '';
        fromDate = null;
        toDate = null;
        fetchData();
        showNotification('Date range cleared - showing all data', 'info');
    });
    
    // Refresh
    document.getElementById('refresh-btn').addEventListener('click', () => {
        showNotification('Refreshing data...', 'info');
        fetchData();
    });
    
    // Export
    document.getElementById('export-excel-btn').addEventListener('click', exportToExcel);
});
</script>

<style>
@import url('https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap');
:root{--primary-color:#0055a5;--primary-hover:#003f7d;--text-primary:#212529;--text-secondary:#6c757d;--bg-light:#f8fafc;--bg-white:#ffffff;--border-light:#e2e8f0;--shadow-sm:0 .125rem .25rem rgba(0,0,0,.075);--shadow-md:0 .5rem 1rem rgba(0,0,0,.15);--shadow-lg:0 1rem 3rem rgba(0,0,0,.175);--radius-md:.375rem;--radius-lg:.5rem;--transition-base:all .15s ease-in-out;--font-family:'Jost',-apple-system,BlinkMacSystemFont,'Segoe UI',Roboto,sans-serif}
.employees-closed-container{min-height:100vh;background:var(--bg-light);font-family:var(--font-family)}
.header-gradient{background:#0055a5;padding:20px 0;box-shadow:0 4px 12px rgba(0,85,165,.3)}
.header-content{width:100%;padding:0 20px;display:flex;align-items:center;justify-content:space-between}
.header-left{display:flex;align-items:center;gap:15px}
.back-button{color:white;font-size:20px;text-decoration:none;padding:8px;border-radius:50%;transition:var(--transition-base)}
.back-button:hover{background-color:rgba(255,255,255,.1)}
.header-title{color:white;font-size:28px;font-weight:700;margin:0}
.header-right{display:flex;gap:15px}
.create-btn{background:white;color:var(--primary-color);border:none;padding:12px 24px;border-radius:var(--radius-md);font-size:16px;font-weight:600;cursor:pointer;transition:var(--transition-base);box-shadow:var(--shadow-sm);display:inline-flex;align-items:center;gap:8px}
.create-btn:hover{background:rgba(255,255,255,.9);transform:translateY(-2px);box-shadow:var(--shadow-md)}
.employees-closed-content{padding:30px 20px}
.content-box{background:white;padding:30px;border-radius:var(--radius-lg);box-shadow:var(--shadow-md);border:1px solid var(--border-light)}
.date-picker-container{margin-bottom:20px;display:flex;justify-content:center}
.date-inputs{display:flex;gap:15px;align-items:end;flex-wrap:wrap;justify-content:center}
.date-input-group{display:flex;flex-direction:column;gap:8px}
.date-input-group label{font-size:14px;font-weight:500;color:var(--text-primary)}
.date-input{padding:12px 16px;border:1px solid var(--border-light);border-radius:var(--radius-md);font-size:14px;font-family:var(--font-family);min-width:180px}
.date-input:focus{outline:none;border-color:var(--primary-color);box-shadow:0 0 0 3px rgba(0,85,165,.1)}
.btn-clear-dates{padding:12px 24px;background:white;color:var(--text-primary);border:1px solid var(--border-light);border-radius:var(--radius-md);font-size:14px;font-weight:600;cursor:pointer;transition:var(--transition-base)}
.btn-clear-dates:hover{background:var(--bg-light);border-color:var(--primary-color);color:var(--primary-color)}
.summary-header{display:grid;grid-template-columns:repeat(auto-fit,minmax(250px,1fr));gap:20px;margin-bottom:30px}
.summary-stat{display:flex;align-items:center;gap:15px;background:linear-gradient(135deg,#f8fafc 0%,#e3f2fd 100%);padding:20px;border-radius:var(--radius-lg);border:1px solid var(--border-light)}
.summary-stat i{font-size:32px;color:var(--primary-color)}
.stat-label{font-size:12px;color:var(--text-secondary);font-weight:500;text-transform:uppercase;margin-bottom:4px}
.stat-value{font-size:28px;color:var(--primary-color);font-weight:700}
.employees-list{display:grid;gap:20px}
.user-card{background:#F5F7FA;border-radius:10px;padding:20px;border:1px solid #E0E0E0;box-shadow:var(--shadow-sm);transition:var(--transition-base)}
.user-card:hover{box-shadow:var(--shadow-md);transform:translateY(-2px)}
.user-header{display:flex;justify-content:space-between;align-items:center;margin-bottom:12px;padding-bottom:12px;border-bottom:2px solid #E0E0E0}
.user-name{font-size:18px;font-weight:700;color:#015185}
.status-badge-header{font-size:13px;font-weight:700;padding:6px 14px;border-radius:12px}
.user-info{display:flex;gap:20px;margin-bottom:16px;font-size:14px;color:#2267A2;flex-wrap:wrap}
.user-info span{display:flex;align-items:center;gap:6px}
.user-info i{font-size:14px}
.job-card{background:white;border-radius:8px;padding:16px;margin-bottom:12px;border:1px solid #E0E0E0;box-shadow:var(--shadow-sm)}
.job-card:last-child{margin-bottom:0}
.job-info-row{margin-bottom:8px;font-size:14px;color:#015185;display:flex;align-items:center;gap:8px;flex-wrap:wrap}
.job-info-row strong{color:#025383;display:flex;align-items:center;gap:6px}
.job-info-row i{font-size:13px;color:#0055a5}
.status-badge{font-weight:600;font-size:14px;padding:4px 12px;border-radius:12px}
.task-item{margin-left:20px;margin-top:6px;font-size:13px;color:#444;display:flex;align-items:flex-start;gap:8px}
.task-item i{color:#2ecc40;font-size:12px;margin-top:2px}
.task-status{color:#666;font-weight:500}
.task-note{color:#888;font-size:12px;font-style:italic;margin-left:28px;display:block;margin-top:2px}
.loading-state,.no-data{text-align:center;padding:60px;color:var(--text-secondary);font-size:18px}
.loading-state i{margin-right:10px;font-size:24px}
@keyframes spin{0%{transform:rotate(0deg)}100%{transform:rotate(360deg)}}
.sync-modal{position:fixed;top:0;left:0;right:0;bottom:0;background:rgba(0,0,0,.5);display:flex;align-items:center;justify-content:center;z-index:10001}
.sync-content{background:white;border-radius:var(--radius-lg);padding:40px;text-align:center;max-width:300px;box-shadow:var(--shadow-lg)}
.sync-icon{width:40px;height:40px;margin:0 auto 20px;font-size:40px;animation:spin 1s linear infinite}
.sync-text{font-size:18px;font-weight:600;margin-bottom:10px}
.sync-subtext{font-size:14px;color:var(--text-secondary)}
.notification{position:fixed;top:20px;right:20px;z-index:10000;padding:16px;border-radius:8px;box-shadow:0 4px 12px rgba(0,0,0,.15);font-family:var(--font-family);max-width:400px;background:white;animation:slideIn .3s}
.notification-success{border-left:4px solid #28a745}
.notification-error{border-left:4px solid #dc3545}
.notification-info{border-left:4px solid #17a2b8}
.notification-content{display:flex;justify-content:space-between;align-items:center}
.notification-close{background:none;border:none;font-size:18px;color:#999;cursor:pointer}
@keyframes slideIn{from{transform:translateX(100%);opacity:0}to{transform:translateX(0);opacity:1}}
@media (max-width:768px){.header-title{font-size:20px}.summary-header{grid-template-columns:1fr}.header-right{flex-direction:column;width:100%}.create-btn{width:100%;justify-content:center}.date-inputs{flex-direction:column;align-items:stretch}.btn-clear-dates,.date-input{width:100%}.user-info{flex-direction:column;gap:8px}}
@media (max-width:480px){.header-content{flex-direction:column;gap:15px;align-items:flex-start}.header-right{width:100%}.create-btn{width:100%}}
</style>
@endsection
