@extends('welcome')

@section('title', 'Maintenance KPI')

@section('styles')
<link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
<link href="https://fonts.googleapis.com/css2?family=Jost:wght@300;400;500;600;700&display=swap" rel="stylesheet">
@endsection

@section('content')
<div class="maintenance-kpi-container">
    <!-- Loading Overlay -->
    <div id="loading-overlay" class="sync-modal" style="display: none;">
        <div class="sync-content">
            <div class="sync-icon">⏳</div>
            <div class="sync-text">Please Wait</div>
            <div class="sync-subtext">Loading Maintenance Data...</div>
        </div>
    </div>
    
    <!-- Header -->
    <div class="header-gradient">
        <div class="header-content">
            <div class="header-left">
                <a href="{{ url()->previous() }}" class="back-button">
                    <i class="fas fa-arrow-left"></i>
                </a>
                <h1 class="header-title">Maintenance KPI Dashboard</h1>
            </div>
            <div class="header-right">
                <button type="button" class="create-btn" id="export-excel-btn-header">
                    <i class="fas fa-file-excel"></i>
                    Export to Excel
                </button>
                <button type="button" class="create-btn" id="refresh-data-btn-header">
                    <i class="fas fa-sync-alt"></i>
                    Refresh
                </button>
            </div>
        </div>
    </div>
    
    <div class="maintenance-kpi-content">
        <div class="content-box">
            <div class="time-period-selector">
                <div class="period-buttons">
                    <button class="period-btn" data-period="today">Today</button>
                    <button class="period-btn active" data-period="7days">Last 7 Days</button>
                    <button class="period-btn" data-period="30days">Last 30 Days</button>
                    <button class="period-btn" data-period="90days">Last 90 Days</button>
                    <button class="period-btn" data-period="custom">Custom Date</button>
                </div>
                
                <!-- Custom Date Range Picker -->
                <div class="custom-date-range" id="custom-date-range" style="display: none;">
                    <div class="date-range-inputs">
                        <div class="date-input-group">
                            <label for="start-date">Start Date:</label>
                            <input type="date" id="start-date" class="date-input">
                        </div>
                        <div class="date-input-group">
                            <label for="end-date">End Date:</label>
                            <input type="date" id="end-date" class="date-input">
                        </div>
                        <button class="btn-apply-dates" id="apply-custom-dates">Apply</button>
                        <button class="btn-clear-dates" id="clear-custom-dates">Clear</button>
                    </div>
                </div>
            </div>
            
            <div class="modern-tabs">
                <div class="tab-navigation">
                    <button class="tab-btn active" data-tab="overview">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="3" width="18" height="18" rx="2" ry="2"/>
                            <path d="M9 9h6v6H9z"/>
                        </svg>
                        Overview
                    </button>
                    <button class="tab-btn" data-tab="request-maintenance">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                            <polyline points="14,2 14,8 20,8"/>
                        </svg>
                        Request Maintenance
                    </button>
                    <button class="tab-btn" data-tab="service-schedule">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                            <line x1="16" y1="2" x2="16" y2="6"/>
                            <line x1="8" y1="2" x2="8" y2="6"/>
                            <line x1="3" y1="10" x2="21" y2="10"/>
                        </svg>
                        Service Schedule
                    </button>
                    <button class="tab-btn" data-tab="status-analysis">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M9 12l2 2 4-4"/>
                            <circle cx="12" cy="12" r="10"/>
                        </svg>
                        Status Analysis
                    </button>
                    <button class="tab-btn" data-tab="datewise">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                            <line x1="16" y1="2" x2="16" y2="6"/>
                            <line x1="8" y1="2" x2="8" y2="6"/>
                            <line x1="3" y1="10" x2="21" y2="10"/>
                        </svg>
                        Date Analysis
                    </button>
                    <button class="tab-btn" data-tab="equipment-analysis">
                        <svg xmlns="http://www.w3.org/2000/svg" width="18" height="18" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                            <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                        </svg>
                        Equipment
                    </button>
                </div>
                <div class="tab-content">
                    <div id="overview" class="tab-pane active">
                        <div class="kpi-cards-grid">
                            <div class="kpi-card" data-kpi="total-requests">
                                <div class="kpi-icon">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8z"/>
                                        <polyline points="14,2 14,8 20,8"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Requests</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="total-schedules">
                                <div class="kpi-icon info">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <rect x="3" y="4" width="18" height="18" rx="2" ry="2"/>
                                        <line x1="16" y1="2" x2="16" y2="6"/>
                                        <line x1="8" y1="2" x2="8" y2="6"/>
                                        <line x1="3" y1="10" x2="21" y2="10"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Total Schedules</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="pending-requests">
                                <div class="kpi-icon warning">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <circle cx="12" cy="12" r="10"/>
                                        <polyline points="12 6 12 12 16 14"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Pending Requests</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                            
                            <div class="kpi-card" data-kpi="completed-maintenance">
                                <div class="kpi-icon success">
                                    <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                        <path d="M20 6L9 17l-5-5"/>
                                    </svg>
                                </div>
                                <div class="kpi-content">
                                    <h3>Completed</h3>
                                    <div class="kpi-value">00</div>
                                    <div class="kpi-label">Count</div>
                                </div>
                            </div>
                        </div>
                        <div class="overview-bottom-section">
                            <div class="additional-kpi-cards">
                                <div class="kpi-card">
                                    <div class="kpi-icon secondary">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <path d="M19 17h2l.64-2.54A6 6 0 0 0 20 7a6 6 0 0 0-6-6H6a6 6 0 0 0-6 6c0 2.34 1.95 4.18 4.35 4.46L7 17h2"/>
                                        </svg>
                                    </div>
                                    <div class="kpi-content">
                                        <h3>Unique Equipment</h3>
                                        <div class="kpi-value">00</div>
                                        <div class="kpi-label">Count</div>
                                    </div>
                                </div>
                                
                                <div class="kpi-card">
                                    <div class="kpi-icon danger">
                                        <svg xmlns="http://www.w3.org/2000/svg" width="32" height="32" viewBox="0 0 24 24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                            <circle cx="12" cy="12" r="10"/>
                                            <line x1="15" y1="9" x2="9" y2="15"/>
                                            <line x1="9" y1="9" x2="15" y2="15"/>
                                        </svg>
                                    </div>
                                    <div class="kpi-content">
                                        <h3>Overdue</h3>
                                        <div class="kpi-value">00</div>
                                        <div class="kpi-label">Count</div>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="chart-section">
                                <div class="chart-card">
                                    <div class="chart-header">
                                        <h3>Maintenance Status Distribution</h3>
                                    </div>
                                    <div class="chart-container">
                                        <canvas id="status-distribution"></canvas>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="request-maintenance" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Request Maintenance by Status</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="request-by-status"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Request Maintenance by Priority</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="request-by-priority"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Request Maintenance Count Over Time</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="request-over-time"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="service-schedule" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Service Schedule by Type</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="schedule-by-type"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Service Schedule by Status</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="schedule-by-status"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Upcoming vs Completed Schedules</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="schedule-upcoming-completed"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="status-analysis" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Combined Status Distribution</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="combined-status"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Priority Distribution</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="priority-distribution"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Request vs Schedule Comparison</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="request-vs-schedule"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="datewise" class="tab-pane">
                        <div class="charts-grid">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Date wise Request Count</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="datewise-request"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Date wise Schedule Count</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="datewise-schedule"></canvas>
                                </div>
                            </div>
                            
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Combined Daily Trend</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="datewise-combined"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                    
                    <div id="equipment-analysis" class="tab-pane">
                        <div class="filter-section">
                            <div class="filter-form">
                                <div class="form-group">
                                    <label for="equipment-filter">Select Particular Equipment:</label>
                                    <select class="form-select" id="equipment-filter" name="equipment">
                                        <option value="">All Equipment</option>
                                    </select>
                                </div>
                                <button class="btn-go">Go</button>
                            </div>
                        </div>
                        
                        <div class="chart-section">
                            <div class="chart-card">
                                <div class="chart-header">
                                    <h3>Maintenance Count by Equipment</h3>
                                </div>
                                <div class="chart-container">
                                    <canvas id="by-equipment-chart"></canvas>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>
<script>
    function toggleSidebar() {
        const sidebar = document.querySelector('.sidebar');
        sidebar.classList.toggle('collapsed');
    }
</script>
</div>
<script>
    var dropdown = document.getElementsByClassName("dropdown-btn");
    var i;

    for (i = 0; i < dropdown.length; i++) {
        dropdown[i].addEventListener("click", function() {
            this.classList.toggle("active");
            var dropdownContent = this.nextElementSibling;
            if (dropdownContent.style.display === "block") {
                dropdownContent.style.display = "none";
            } else {
                dropdownContent.style.display = "block";
            }
        });
    }
</script>
<!-- Firebase SDK -->
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-app-compat.js"></script>
<script src="https://www.gstatic.com/firebasejs/9.22.0/firebase-database-compat.js"></script>

<!-- Chart.js Library -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/Chart.js/2.9.4/Chart.js"></script>
    
<!-- SheetJS Library for Excel Export -->
<script src="https://cdnjs.cloudflare.com/ajax/libs/xlsx/0.18.5/xlsx.full.min.js"></script>
<script>
// Firebase Configuration
const firebaseConfig = {
    databaseURL: "https://titan-drilling-1f8e9-default-rtdb.firebaseio.com"
};

// Initialize Firebase
if (!firebase.apps.length) {
    firebase.initializeApp(firebaseConfig);
}

const database = firebase.database();

// Global variables
let allRequestMaintenanceData = [];
let allServiceScheduleData = [];
let chartInstances = {};
let currentDateRange = 7;
let customStartDate = null;
let customEndDate = null;

// Fetch Request Maintenance data - from ALL UIDs and nested folders
function fetchRequestMaintenanceData() {
    const requestRef = database.ref(`GlobalRequestMaintenancereport`);
    
    return requestRef.once('value')
        .then((snapshot) => {
            allRequestMaintenanceData = [];
            const data = snapshot.val();
            
            if (data) {
                // Recursive function to extract all request data from any depth
                function extractRequests(obj, uid = '', path = []) {
                    if (!obj || typeof obj !== 'object') return;
                    
                    Object.keys(obj).forEach(key => {
                        const item = obj[key];
                        
                        if (typeof item !== 'object' || item === null) {
                            return;
                        }
                        
                        // Check if this looks like a request object (has request-related fields)
                        const hasRequestFields = item.title || item.scheduleId || item.country || item.project || item.equipment || item.status;
                        
                        if (hasRequestFields) {
                            // This is a request object
                            allRequestMaintenanceData.push({
                                id: key,
                                uid: uid || path[0] || 'unknown',
                                ...item
                            });
                        } else {
                            // This might be a folder, go deeper
                            extractRequests(item, uid || key, [...path, key]);
                        }
                    });
                }
                
                extractRequests(data);
            }
            
            console.log('Fetched request maintenance from all folders:', allRequestMaintenanceData.length);
            return allRequestMaintenanceData.length;
        });
}

// Fetch Service Schedule data - from ALL UIDs, going inside folders
function fetchServiceScheduleData() {
    const scheduleRef = database.ref(`GlobalServiceScheduled`);
    
    return scheduleRef.once('value')
        .then((snapshot) => {
            allServiceScheduleData = [];
            const data = snapshot.val();
            
            if (data) {
                // Recursive function to extract all schedule data
                function extractSchedules(obj, uid = '', path = []) {
                    if (!obj || typeof obj !== 'object') return;
                    
                    Object.keys(obj).forEach(key => {
                        const item = obj[key];
                        
                        if (typeof item !== 'object' || item === null) {
                            return;
                        }
                        
                        // Check if this looks like a schedule object
                        const hasScheduleFields = item.scheduleTitle || item.equipmentName || item.serviceType || item.scheduledDate || item.title;
                        
                        if (hasScheduleFields) {
                            allServiceScheduleData.push({
                                id: key,
                                uid: uid || path[0] || 'unknown',
                                ...item
                            });
                        } else {
                            // Go deeper
                            extractSchedules(item, uid || key, [...path, key]);
                        }
                    });
                }
                
                extractSchedules(data);
            }
            
            console.log('Fetched service schedules:', allServiceScheduleData.length);
            return allServiceScheduleData.length;
        });
}

// Fetch all data
function fetchAllData() {
    showLoadingState(true);
    
    Promise.all([
        fetchRequestMaintenanceData(),
        fetchServiceScheduleData()
    ])
    .then(([requestCount, scheduleCount]) => {
        if (requestCount === 0 && scheduleCount === 0) {
            showNotification('No maintenance data found', 'info');
        } else {
            showNotification(`Loaded ${requestCount} request(s) and ${scheduleCount} schedule(s) from all users`, 'success');
        }
        
        updateDashboard();
        showLoadingState(false);
    })
    .catch((error) => {
        console.error('Error fetching data:', error);
        showNotification('Error fetching maintenance data: ' + error.message, 'error');
        showLoadingState(false);
    });
}

// Show/hide loading state
function showLoadingState(isLoading) {
    const loadingOverlay = document.getElementById('loading-overlay');
    if (loadingOverlay) {
        loadingOverlay.style.display = isLoading ? 'flex' : 'none';
    }
}

// Filter data by date range
function filterDataByDateRange(data, days) {
    if (days === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate);
            start.setHours(0, 0, 0, 0);
            const end = new Date(customEndDate);
            end.setHours(23, 59, 59, 999);
            
            return data.filter(item => {
                const createdAt = item.createdAt || item.dateCreated || item.scheduledDate;
                if (!createdAt) return false;
                
                const itemDate = new Date(createdAt);
                return itemDate >= start && itemDate <= end;
            });
        }
        return data;
    }
    
    const now = new Date();
    const startDate = new Date(now);
    
    if (days === 'today') {
        startDate.setHours(0, 0, 0, 0);
    } else {
        startDate.setDate(now.getDate() - parseInt(days));
    }
    
    return data.filter(item => {
        const createdAt = item.createdAt || item.dateCreated || item.scheduledDate;
        if (!createdAt) return false;
        
        const itemDate = new Date(createdAt);
        return itemDate >= startDate && itemDate <= now;
    });
}

// Update KPI cards
function updateKPICards(filteredRequests, filteredSchedules) {
    const totalRequests = filteredRequests.length;
    const totalSchedules = filteredSchedules.length;
    
    const pendingRequests = filteredRequests.filter(r => 
        r.status === 'Pending' || r.status === 'Open' || !r.status
    ).length;
    
    const completed = [...filteredRequests, ...filteredSchedules].filter(item => 
        item.status === 'Completed' || item.status === 'Complete' || item.status === 'Closed'
    ).length;
    
    const uniqueEquipment = new Set();
    [...filteredRequests, ...filteredSchedules].forEach(item => {
        const equipmentName = item.equipmentName || item.equipment?.equipmentName;
        if (equipmentName) uniqueEquipment.add(equipmentName);
    });
    
    const now = new Date();
    const overdue = [...filteredRequests, ...filteredSchedules].filter(item => {
        const dueDate = item.dueDate || item.scheduledDate;
        if (!dueDate) return false;
        const due = new Date(dueDate);
        return due < now && item.status !== 'Completed' && item.status !== 'Complete';
    }).length;
    
    document.querySelector('[data-kpi="total-requests"] .kpi-value').textContent = String(totalRequests).padStart(2, '0');
    document.querySelector('[data-kpi="total-schedules"] .kpi-value').textContent = String(totalSchedules).padStart(2, '0');
    document.querySelector('[data-kpi="pending-requests"] .kpi-value').textContent = String(pendingRequests).padStart(2, '0');
    document.querySelector('[data-kpi="completed-maintenance"] .kpi-value').textContent = String(completed).padStart(2, '0');
    
    const additionalCards = document.querySelectorAll('.additional-kpi-cards .kpi-card');
    if (additionalCards[0]) {
        additionalCards[0].querySelector('.kpi-value').textContent = String(uniqueEquipment.size).padStart(2, '0');
    }
    if (additionalCards[1]) {
        additionalCards[1].querySelector('.kpi-value').textContent = String(overdue).padStart(2, '0');
    }
}

// Update dashboard
function updateDashboard() {
    const filteredRequests = filterDataByDateRange(allRequestMaintenanceData, currentDateRange);
    const filteredSchedules = filterDataByDateRange(allServiceScheduleData, currentDateRange);
    
    updateKPICards(filteredRequests, filteredSchedules);
    updateStatusDistributionChart(filteredRequests, filteredSchedules);
    updateRequestMaintenanceCharts(filteredRequests);
    updateServiceScheduleCharts(filteredSchedules);
    updateStatusAnalysisCharts(filteredRequests, filteredSchedules);
    updateDatewiseCharts(filteredRequests, filteredSchedules);
    updateEquipmentChart(filteredRequests, filteredSchedules);
    
    populateEquipmentDropdown([...allRequestMaintenanceData, ...allServiceScheduleData]);
}

// Chart functions
function updateStatusDistributionChart(requests, schedules) {
    const combined = [...requests, ...schedules];
    const statusCounts = {};
    
    combined.forEach(item => {
        const status = item.status || 'Pending';
        statusCounts[status] = (statusCounts[status] || 0) + 1;
    });
    
    updateChart('status-distribution', Object.keys(statusCounts), Object.values(statusCounts),
                ['#0055A5', '#FFA500', '#10b981', '#DC0D0E', '#6c757d']);
}

function updateRequestMaintenanceCharts(requests) {
    const statusCounts = {};
    requests.forEach(r => {
        const status = r.status || 'Pending';
        statusCounts[status] = (statusCounts[status] || 0) + 1;
    });
    updateChart('request-by-status', Object.keys(statusCounts), Object.values(statusCounts),
                ['#0055A5', '#FFA500', '#10b981', '#DC0D0E']);
    
    const priorityCounts = {};
    requests.forEach(r => {
        const priority = r.priority || 'Medium';
        priorityCounts[priority] = (priorityCounts[priority] || 0) + 1;
    });
    updateChart('request-by-priority', Object.keys(priorityCounts), Object.values(priorityCounts),
                ['#DC0D0E', '#FFA500', '#10b981']);
    
    const dateRange = currentDateRange === 'today' ? 1 : parseInt(currentDateRange) || 7;
    const dates = [];
    const counts = {};
    
    for (let i = dateRange - 1; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        const dateStr = date.toLocaleDateString('en-GB');
        dates.push(dateStr);
        counts[dateStr] = 0;
    }
    
    requests.forEach(r => {
        const createdAt = r.createdAt || r.dateCreated;
        if (createdAt) {
            const date = new Date(createdAt).toLocaleDateString('en-GB');
            if (counts.hasOwnProperty(date)) {
                counts[date]++;
            }
        }
    });
    
    updateChart('request-over-time', dates, dates.map(d => counts[d]),
                ['#0055A5', '#945ECF', '#E8743B']);
}

function updateServiceScheduleCharts(schedules) {
    const typeCounts = {};
    schedules.forEach(s => {
        const type = s.serviceType || s.scheduleType || 'General';
        typeCounts[type] = (typeCounts[type] || 0) + 1;
    });
    updateChart('schedule-by-type', Object.keys(typeCounts), Object.values(typeCounts),
                ['#0055A5', '#945ECF', '#E8743B', '#10b981']);
    
    const statusCounts = {};
    schedules.forEach(s => {
        const status = s.status || 'Scheduled';
        statusCounts[status] = (statusCounts[status] || 0) + 1;
    });
    updateChart('schedule-by-status', Object.keys(statusCounts), Object.values(statusCounts),
                ['#0055A5', '#FFA500', '#10b981']);
    
    const now = new Date();
    let upcoming = 0;
    let completed = 0;
    
    schedules.forEach(s => {
        if (s.status === 'Completed' || s.status === 'Complete') {
            completed++;
        } else {
            const scheduledDate = new Date(s.scheduledDate || s.dueDate);
            if (scheduledDate > now) {
                upcoming++;
            }
        }
    });
    
    updateChart('schedule-upcoming-completed', ['Upcoming', 'Completed'], [upcoming, completed],
                ['#FFA500', '#10b981']);
}

function updateStatusAnalysisCharts(requests, schedules) {
    const combined = [...requests, ...schedules];
    const statusCounts = {};
    combined.forEach(item => {
        const status = item.status || 'Pending';
        statusCounts[status] = (statusCounts[status] || 0) + 1;
    });
    updateChart('combined-status', Object.keys(statusCounts), Object.values(statusCounts),
                ['#0055A5', '#FFA500', '#10b981', '#DC0D0E', '#945ECF']);
    
    const priorityCounts = {};
    combined.forEach(item => {
        const priority = item.priority || 'Medium';
        priorityCounts[priority] = (priorityCounts[priority] || 0) + 1;
    });
    updateChart('priority-distribution', Object.keys(priorityCounts), Object.values(priorityCounts),
                ['#DC0D0E', '#FFA500', '#10b981']);
    
    updateChart('request-vs-schedule', ['Requests', 'Schedules'], 
                [requests.length, schedules.length],
                ['#0055A5', '#10b981']);
}

function updateDatewiseCharts(requests, schedules) {
    const dateRange = currentDateRange === 'today' ? 1 : parseInt(currentDateRange) || 7;
    const dates = [];
    const requestCounts = {};
    const scheduleCounts = {};
    
    for (let i = dateRange - 1; i >= 0; i--) {
        const date = new Date();
        date.setDate(date.getDate() - i);
        const dateStr = date.toLocaleDateString('en-GB');
        dates.push(dateStr);
        requestCounts[dateStr] = 0;
        scheduleCounts[dateStr] = 0;
    }
    
    requests.forEach(r => {
        const createdAt = r.createdAt || r.dateCreated;
        if (createdAt) {
            const date = new Date(createdAt).toLocaleDateString('en-GB');
            if (requestCounts.hasOwnProperty(date)) {
                requestCounts[date]++;
            }
        }
    });
    
    schedules.forEach(s => {
        const createdAt = s.createdAt || s.dateCreated || s.scheduledDate;
        if (createdAt) {
            const date = new Date(createdAt).toLocaleDateString('en-GB');
            if (scheduleCounts.hasOwnProperty(date)) {
                scheduleCounts[date]++;
            }
        }
    });
    
    updateChart('datewise-request', dates, dates.map(d => requestCounts[d]),
                ['#0055A5', '#945ECF']);
    updateChart('datewise-schedule', dates, dates.map(d => scheduleCounts[d]),
                ['#10b981', '#E8743B']);
    
    const ctx = document.getElementById('datewise-combined');
    if (ctx && chartInstances['datewise-combined']) {
        chartInstances['datewise-combined'].destroy();
    }
    
    chartInstances['datewise-combined'] = new Chart('datewise-combined', {
        type: 'bar',
        data: {
            labels: dates,
            datasets: [
                {
                    label: 'Requests',
                    backgroundColor: '#0055A5',
                    data: dates.map(d => requestCounts[d])
                },
                {
                    label: 'Schedules',
                    backgroundColor: '#10b981',
                    data: dates.map(d => scheduleCounts[d])
                }
            ]
        },
        options: {
            legend: { display: true },
            scales: {
                xAxes: [{ stacked: true }],
                yAxes: [{ stacked: true, ticks: { beginAtZero: true } }]
            }
        }
    });
}

function updateEquipmentChart(requests, schedules, selectedEquipment = null) {
    let combined = [...requests, ...schedules];
    
    if (selectedEquipment) {
        combined = combined.filter(item => {
            const equipmentName = item.equipmentName || item.equipment?.equipmentName;
            return equipmentName === selectedEquipment;
        });
    }
    
    const equipmentCounts = {};
    combined.forEach(item => {
        const equipmentName = item.equipmentName || item.equipment?.equipmentName || 'Unknown';
        equipmentCounts[equipmentName] = (equipmentCounts[equipmentName] || 0) + 1;
    });
    
    updateChart('by-equipment-chart', Object.keys(equipmentCounts), Object.values(equipmentCounts),
                ['#0055A5', '#945ECF', '#E8743B', '#10b981']);
}

// Generic chart update
function updateChart(chartId, labels, data, colors) {
    const ctx = document.getElementById(chartId);
    if (!ctx) return;
    
    if (chartInstances[chartId]) {
        chartInstances[chartId].destroy();
    }
    
    if (!labels || labels.length === 0 || !data || data.length === 0) {
        labels = ['No Data'];
        data = [0];
        colors = ['#e0e0e0'];
    }
    
    chartInstances[chartId] = new Chart(chartId, {
        type: 'bar',
        data: {
            labels: labels,
            datasets: [{
                backgroundColor: colors,
                data: data
            }]
        },
        options: {
            legend: { display: false },
            scales: {
                yAxes: [{ ticks: { beginAtZero: true } }]
            },
            title: {
                display: true,
                text: ctx.closest('.chart-card')?.querySelector('h3')?.textContent || ''
            },
            tooltips: {
                enabled: labels[0] !== 'No Data'
            }
        }
    });
}

// Populate equipment dropdown
function populateEquipmentDropdown(data) {
    const equipment = new Set();
    data.forEach(item => {
        const equipmentName = item.equipmentName || item.equipment?.equipmentName;
        if (equipmentName) equipment.add(equipmentName);
    });
    
    const dropdown = document.getElementById('equipment-filter');
    if (dropdown) {
        dropdown.innerHTML = '<option value="">All Equipment</option>';
        Array.from(equipment).sort().forEach(eq => {
            const option = document.createElement('option');
            option.value = eq;
            option.textContent = eq;
            dropdown.appendChild(option);
        });
    }
}

// Export to Excel
function exportToExcel() {
    const filteredRequests = filterDataByDateRange(allRequestMaintenanceData, currentDateRange);
    const filteredSchedules = filterDataByDateRange(allServiceScheduleData, currentDateRange);
    
    if (filteredRequests.length === 0 && filteredSchedules.length === 0) {
        showNotification('No data to export', 'error');
        return;
    }
    
    // Prepare Request Maintenance data with all fields
    const requestData = filteredRequests.map((item, index) => {
        const equipment = item.equipment || {};
        const notifyBefore = item.notifyBefore || {};
        
        return {
            'S.No': index + 1,
            'Type': 'Request Maintenance',
            'ID': item.id || 'N/A',
            'Schedule ID': item.scheduleId || 'N/A',
            'Title': item.title || 'N/A',
            'Status': item.status || 'N/A',
            'Priority': item.priority || 'N/A',
            'Country': item.country || 'N/A',
            'Project': item.project || 'N/A',
            'Equipment Name': equipment.equipmentName || 'N/A',
            'Equipment Number': equipment.equipmentNumber || 'N/A',
            'Equipment Entry Number': equipment.entryNumber || 'N/A',
            'Meter Reading': item.meterReading || equipment.meterReading || 'N/A',
            'Meter Unit': item.meterUnit || equipment.meterUnit || 'N/A',
            'Last Service Date': item.lastServiceDate || 'N/A',
            'Days To Go': item.daysToGo || 'N/A',
            'Hours To Go': item.hoursToGo || 'N/A',
            'Due Hours': item.dueHours || 'N/A',
            'Interval Unit': item.intervalUnit || 'N/A',
            'Repeat Type': item.repeatType || 'N/A',
            'Tasks': Array.isArray(item.tasks) ? item.tasks.join(', ') : 'N/A',
            'Note': item.note || 'N/A',
            'Notify Before Value': notifyBefore.value || 'N/A',
            'Notify Before Time': notifyBefore.time || 'N/A',
            'Notify Before Unit': notifyBefore.unit || 'N/A',
            'Created At': item.createdAt ? new Date(item.createdAt).toLocaleString('en-GB') : 'N/A'
        };
    });
    
    // Prepare Service Schedule data with all fields
    const scheduleData = filteredSchedules.map((item, index) => {
        const equipment = item.equipment || {};
        const notifyBefore = item.notifyBefore || {};
        
        return {
            'S.No': index + 1,
            'Type': 'Service Schedule',
            'ID': item.id || 'N/A',
            'Schedule ID': item.scheduleId || 'N/A',
            'Title': item.title || 'N/A',
            'Status': item.status || 'N/A',
            'Priority': item.priority || 'N/A',
            'Country': item.country || 'N/A',
            'Project': item.project || 'N/A',
            'Equipment Name': equipment.equipmentName || 'N/A',
            'Equipment Number': equipment.equipmentNumber || 'N/A',
            'Equipment Entry Number': equipment.entryNumber || 'N/A',
            'Meter Reading': item.meterReading || equipment.meterReading || 'N/A',
            'Meter Unit': item.meterUnit || equipment.meterUnit || 'N/A',
            'Last Service Date': item.lastServiceDate || 'N/A',
            'Days To Go': item.daysToGo || 'N/A',
            'Hours To Go': item.hoursToGo || 'N/A',
            'Due Hours': item.dueHours || 'N/A',
            'Interval Unit': item.intervalUnit || 'N/A',
            'Repeat Type': item.repeatType || 'N/A',
            'Tasks': Array.isArray(item.tasks) ? item.tasks.join(', ') : 'N/A',
            'Note': item.note || 'N/A',
            'Notify Before Value': notifyBefore.value || 'N/A',
            'Notify Before Time': notifyBefore.time || 'N/A',
            'Notify Before Unit': notifyBefore.unit || 'N/A',
            'Created At': item.createdAt ? new Date(item.createdAt).toLocaleString('en-GB') : 'N/A'
        };
    });
    
    const wb = XLSX.utils.book_new();
    
    // Add Request Maintenance sheet
    if (requestData.length > 0) {
        const wsRequests = XLSX.utils.json_to_sheet(requestData);
        wsRequests['!cols'] = [
            { wch: 6 }, { wch: 20 }, { wch: 15 }, { wch: 18 }, { wch: 30 },
            { wch: 12 }, { wch: 10 }, { wch: 12 }, { wch: 15 }, { wch: 25 },
            { wch: 18 }, { wch: 20 }, { wch: 12 }, { wch: 12 }, { wch: 18 },
            { wch: 12 }, { wch: 12 }, { wch: 12 }, { wch: 15 }, { wch: 15 },
            { wch: 30 }, { wch: 30 }, { wch: 15 }, { wch: 15 }, { wch: 15 },
            { wch: 20 }
        ];
        XLSX.utils.book_append_sheet(wb, wsRequests, 'Request Maintenance');
    }
    
    // Add Service Schedule sheet
    if (scheduleData.length > 0) {
        const wsSchedules = XLSX.utils.json_to_sheet(scheduleData);
        wsSchedules['!cols'] = [
            { wch: 6 }, { wch: 20 }, { wch: 15 }, { wch: 18 }, { wch: 30 },
            { wch: 12 }, { wch: 10 }, { wch: 12 }, { wch: 15 }, { wch: 25 },
            { wch: 18 }, { wch: 20 }, { wch: 12 }, { wch: 12 }, { wch: 18 },
            { wch: 12 }, { wch: 12 }, { wch: 12 }, { wch: 15 }, { wch: 15 },
            { wch: 30 }, { wch: 30 }, { wch: 15 }, { wch: 15 }, { wch: 15 },
            { wch: 20 }
        ];
        XLSX.utils.book_append_sheet(wb, wsSchedules, 'Service Schedule');
    }
    
    // Add Summary sheet
    const summaryData = [
        { 'Metric': 'Total Request Maintenance', 'Value': filteredRequests.length },
        { 'Metric': 'Total Service Schedules', 'Value': filteredSchedules.length },
        { 'Metric': 'Total Records', 'Value': filteredRequests.length + filteredSchedules.length },
        { 'Metric': 'Date Range', 'Value': getCurrentDateRangeText() },
        { 'Metric': 'Export Date', 'Value': new Date().toLocaleString('en-GB') }
    ];
    
    const wsSummary = XLSX.utils.json_to_sheet(summaryData);
    wsSummary['!cols'] = [{ wch: 30 }, { wch: 30 }];
    XLSX.utils.book_append_sheet(wb, wsSummary, 'Summary');
    
    const dateRangeText = getCurrentDateRangeText().replace(/\s+/g, '_');
    const timestamp = new Date().toISOString().slice(0, 10);
    const filename = `Maintenance_KPI_${dateRangeText}_${timestamp}.xlsx`;
    
    XLSX.writeFile(wb, filename);
    
    showNotification(`Exported ${filteredRequests.length} request(s) and ${filteredSchedules.length} schedule(s) successfully`, 'success');
}

function getCurrentDateRangeText() {
    if (currentDateRange === 'today') {
        return 'Today';
    } else if (currentDateRange === 'custom') {
        if (customStartDate && customEndDate) {
            const start = new Date(customStartDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            const end = new Date(customEndDate).toLocaleDateString('en-GB').replace(/\//g, '-');
            return `Custom_${start}_to_${end}`;
        }
        return 'All_Data';
    } else {
        return `Last_${currentDateRange}_Days`;
    }
}

// Initialize
document.addEventListener('DOMContentLoaded', () => {
    fetchAllData();
    
    const exportBtnHeader = document.getElementById('export-excel-btn-header');
    if (exportBtnHeader) {
        exportBtnHeader.addEventListener('click', () => {
            exportToExcel();
        });
    }
    
    const refreshBtnHeader = document.getElementById('refresh-data-btn-header');
    if (refreshBtnHeader) {
        refreshBtnHeader.addEventListener('click', () => {
            showNotification('Refreshing data...', 'info');
            fetchAllData();
        });
    }
    
    const customDateRange = document.getElementById('custom-date-range');
    const startDateInput = document.getElementById('start-date');
    const endDateInput = document.getElementById('end-date');
    const applyDatesBtn = document.getElementById('apply-custom-dates');
    const clearDatesBtn = document.getElementById('clear-custom-dates');
    
    if (applyDatesBtn) {
        applyDatesBtn.addEventListener('click', () => {
            const startDate = startDateInput.value;
            const endDate = endDateInput.value;
            
            if (!startDate || !endDate) {
                showNotification('Please select both start and end dates', 'error');
                return;
            }
            
            if (new Date(startDate) > new Date(endDate)) {
                showNotification('Start date cannot be after end date', 'error');
                return;
            }
            
            customStartDate = startDate;
            customEndDate = endDate;
            currentDateRange = 'custom';
            
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === 'custom') {
                    btn.classList.add('active');
                }
            });
            
            updateDashboard();
            showNotification(`Showing data from ${new Date(startDate).toLocaleDateString('en-GB')} to ${new Date(endDate).toLocaleDateString('en-GB')}`, 'success');
        });
    }
    
    if (clearDatesBtn) {
        clearDatesBtn.addEventListener('click', () => {
            startDateInput.value = '';
            endDateInput.value = '';
            customStartDate = null;
            customEndDate = null;
            
            currentDateRange = 7;
            document.querySelectorAll('.period-btn').forEach(btn => {
                btn.classList.remove('active');
                if (btn.getAttribute('data-period') === '7days') {
                    btn.classList.add('active');
                }
            });
            
            if (customDateRange) {
                customDateRange.style.display = 'none';
            }
            
            updateDashboard();
            showNotification('Custom date range cleared', 'info');
        });
    }
    
    const today = new Date().toISOString().split('T')[0];
    if (startDateInput) startDateInput.max = today;
    if (endDateInput) endDateInput.max = today;
    
    if (startDateInput && endDateInput) {
        startDateInput.addEventListener('change', () => {
            if (startDateInput.value) {
                endDateInput.min = startDateInput.value;
                if (endDateInput.value && new Date(endDateInput.value) < new Date(startDateInput.value)) {
                    endDateInput.value = '';
                }
            }
        });
    }
    
    const equipmentFilter = document.getElementById('equipment-filter');
    const equipmentGoBtn = document.querySelector('#equipment-analysis .btn-go');
    if (equipmentGoBtn && equipmentFilter) {
        equipmentGoBtn.addEventListener('click', () => {
            const selectedEquipment = equipmentFilter.value;
            const filteredRequests = filterDataByDateRange(allRequestMaintenanceData, currentDateRange);
            const filteredSchedules = filterDataByDateRange(allServiceScheduleData, currentDateRange);
            updateEquipmentChart(filteredRequests, filteredSchedules, selectedEquipment || null);
            if (selectedEquipment) {
                showNotification(`Equipment chart filtered: ${selectedEquipment}`, 'success');
            } else {
                showNotification('Showing all equipment', 'info');
            }
        });
    }
});
</script>

<style>
/* CSS Variables */
:root {
    --primary-color: #0055a5;
    --primary-hover: #003f7d;
    --secondary-color: #6c757d;
    --success-color: #28a745;
    --warning-color: #ffc107;
    --danger-color: #dc3545;
    --info-color: #17a2b8;
    --text-primary: #212529;
    --text-secondary: #6c757d;
    --text-muted: #64748b;
    --text-light: #cbd5e1;
    --bg-primary: #ffffff;
    --bg-secondary: #f8f9fa;
    --bg-light: #f8fafc;
    --bg-white: #ffffff;
    --border-color: #dee2e6;
    --border-light: #e2e8f0;
    --shadow-sm: 0 0.125rem 0.25rem rgba(0, 0, 0, 0.075);
    --shadow-md: 0 0.5rem 1rem rgba(0, 0, 0, 0.15);
    --shadow-lg: 0 1rem 3rem rgba(0, 0, 0, 0.175);
    --radius-sm: 0.25rem;
    --radius-md: 0.375rem;
    --radius-lg: 0.5rem;
    --transition-base: all 0.15s ease-in-out;
    --font-family: 'Jost', -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
    --z-index-notification: 10000;
    --z-index-modal: 1055;
}

.maintenance-kpi-container {
    min-height: 100vh;
    background: var(--bg-light);
    font-family: var(--font-family);
    padding: 0;
    margin: 0;
    overflow-x: hidden;
    overflow-y: visible;
    color: var(--text-primary);
}

.header-gradient {
    background: #0055a5;
    padding: 20px 0;
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.header-content {
    width: 100%;
    padding: 0 20px;
    display: flex;
    align-items: center;
    justify-content: space-between;
}

.header-left {
    display: flex;
    align-items: center;
    gap: 15px;
}

.back-button {
    color: white;
    font-size: 20px;
    text-decoration: none;
    padding: 8px;
    border-radius: 50%;
    transition: var(--transition-base);
}

.back-button:hover {
    background-color: rgba(255, 255, 255, 0.1);
    color: white;
}

.header-title {
    color: white;
    font-size: 28px;
    font-weight: 700;
    margin: 0;
}

.header-right {
    display: flex;
    align-items: center;
    gap: 15px;
}

.create-btn {
    background: white;
    color: var(--primary-color);
    border: none;
    padding: 12px 24px;
    border-radius: var(--radius-md);
    font-size: 16px;
    font-weight: 600;
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    display: inline-flex;
    align-items: center;
    gap: 8px;
}

.create-btn:hover {
    background: rgba(255, 255, 255, 0.9);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.create-btn i {
    font-size: 18px;
}

.maintenance-kpi-content {
    padding: 30px 20px;
    background: var(--bg-light);
    min-height: auto;
    overflow: visible;
}

.content-box {
    background: white;
    padding: 30px;
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    border: 1px solid var(--border-light);
    font-family: var(--font-family);
    margin: 0;
    overflow: hidden;
}

.time-period-selector {
    margin-bottom: 2rem;
}

.period-buttons {
    display: flex;
    gap: 0.5rem;
    flex-wrap: wrap;
}

.period-btn {
    padding: 12px 24px;
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.period-btn:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.period-btn.active {
    background: var(--primary-color);
    color: white;
    border-color: var(--primary-color);
    box-shadow: 0 4px 12px rgba(0, 85, 165, 0.3);
}

.custom-date-range {
    margin-top: 1rem;
    padding: 1.5rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    box-shadow: var(--shadow-md);
    animation: slideDown 0.3s ease-out;
}

@keyframes slideDown {
    from {
        opacity: 0;
        transform: translateY(-10px);
    }
    to {
        opacity: 1;
        transform: translateY(0);
    }
}

.date-range-inputs {
    display: flex;
    align-items: end;
    gap: 1rem;
    flex-wrap: wrap;
}

.date-input-group {
    display: flex;
    flex-direction: column;
    gap: 8px;
}

.date-input-group label {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-primary);
    font-family: var(--font-family);
}

.date-input {
    padding: 12px 16px;
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    font-family: var(--font-family);
    transition: var(--transition-base);
    min-width: 180px;
    cursor: pointer;
}

.date-input:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-apply-dates,
.btn-clear-dates {
    padding: 12px 24px;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    box-shadow: var(--shadow-sm);
}

.btn-apply-dates {
    background: var(--primary-color);
    color: white;
}

.btn-apply-dates:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

.btn-clear-dates {
    background: white;
    color: var(--text-primary);
    border: 1px solid var(--border-color);
}

.btn-clear-dates:hover {
    background: var(--bg-light);
    border-color: var(--primary-color);
    color: var(--primary-color);
    transform: translateY(-1px);
}

.modern-tabs {
    margin-bottom: 2rem;
}

.tab-navigation {
    display: flex;
    gap: 5px;
    flex-wrap: wrap;
    border-bottom: 1px solid var(--border-light);
    padding-bottom: 0;
    margin-bottom: 2rem;
    overflow-x: auto;
}

.tab-btn {
    display: flex;
    align-items: center;
    gap: 8px;
    padding: 12px 16px;
    background: transparent;
    color: var(--text-secondary);
    border: none;
    border-radius: 0;
    font-size: 14px;
    font-weight: 500;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    white-space: nowrap;
    position: relative;
    border-bottom: 2px solid transparent;
}

.tab-btn:hover {
    color: var(--primary-color);
    background: var(--bg-light);
}

.tab-btn.active {
    color: var(--primary-color);
    background: var(--bg-light);
    border-bottom-color: var(--primary-color);
}

.tab-content {
    min-height: 400px;
}

.tab-pane {
    display: none;
}

.tab-pane.active {
    display: block;
}

.kpi-cards-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(280px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.kpi-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
    display: flex;
    align-items: center;
    gap: 1rem;
}

.kpi-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-2px);
    border-color: var(--primary-color);
}

.kpi-icon {
    width: 56px;
    height: 56px;
    background: linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%);
    border-radius: var(--radius-md);
    display: flex;
    align-items: center;
    justify-content: center;
    color: var(--bg-white);
    flex-shrink: 0;
}

.kpi-icon.success {
    background: linear-gradient(135deg, #10b981 0%, #059669 100%);
}

.kpi-icon.warning {
    background: linear-gradient(135deg, #f59e0b 0%, #d97706 100%);
}

.kpi-icon.info {
    background: linear-gradient(135deg, #06b6d4 0%, #0891b2 100%);
}

.kpi-icon.secondary {
    background: linear-gradient(135deg, #6b7280 0%, #4b5563 100%);
}

.kpi-icon.danger {
    background: linear-gradient(135deg, #dc3545 0%, #c82333 100%);
}

.kpi-content {
    flex: 1;
}

.kpi-content h3 {
    font-size: 14px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0 0 8px 0;
}

.kpi-value {
    font-size: 32px;
    font-weight: 700;
    color: var(--primary-color);
    margin: 0 0 4px 0;
    line-height: 1;
}

.kpi-label {
    font-size: 12px;
    font-weight: 500;
    color: var(--text-secondary);
    margin: 0;
    text-transform: uppercase;
    letter-spacing: 0.5px;
}

.overview-bottom-section {
    display: grid;
    grid-template-columns: 1fr 2fr;
    gap: 1.5rem;
    margin-top: 1.5rem;
}

.additional-kpi-cards {
    display: flex;
    flex-direction: column;
    gap: 1rem;
}

.chart-section {
    display: flex;
    flex-direction: column;
}

.charts-grid {
    display: grid;
    grid-template-columns: repeat(auto-fit, minmax(300px, 1fr));
    gap: 1.5rem;
    margin-bottom: 2rem;
}

.chart-card {
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
    transition: var(--transition-base);
}

.chart-card:hover {
    box-shadow: var(--shadow-md);
    transform: translateY(-1px);
    border-color: var(--primary-color);
}

.chart-header {
    margin-bottom: 1rem;
    padding-bottom: 0.75rem;
    border-bottom: 1px solid var(--border-light);
}

.chart-header h3 {
    font-size: 16px;
    font-weight: 600;
    color: var(--text-primary);
    margin: 0;
}

.chart-container {
    position: relative;
    height: 250px;
    display: flex;
    align-items: center;
    justify-content: center;
}

.chart-container canvas {
    max-width: 100%;
    max-height: 100%;
}

.filter-section {
    margin-bottom: 2rem;
}

.filter-form {
    display: flex;
    align-items: end;
    gap: 1rem;
    background: white;
    border: 1px solid var(--border-light);
    border-radius: var(--radius-lg);
    padding: 1.5rem;
    box-shadow: var(--shadow-sm);
}

.form-group {
    flex: 1;
    display: flex;
    flex-direction: column;
    margin-bottom: 0;
}

.form-group label {
    font-weight: 500;
    color: var(--text-primary);
    margin-bottom: 8px;
    font-size: 14px;
}

.form-select {
    border: 1px solid var(--border-color);
    border-radius: var(--radius-md);
    padding: 12px 16px;
    font-size: 14px;
    font-weight: 400;
    color: var(--text-primary);
    background: white;
    transition: var(--transition-base);
    font-family: var(--font-family);
}

.form-select:focus {
    outline: none;
    border-color: var(--primary-color);
    box-shadow: 0 0 0 3px rgba(0, 85, 165, 0.1);
}

.btn-go {
    display: inline-flex;
    align-items: center;
    gap: 8px;
    padding: 12px 24px;
    background: var(--primary-color);
    color: white;
    border: none;
    border-radius: var(--radius-md);
    font-size: 14px;
    font-weight: 600;
    font-family: var(--font-family);
    cursor: pointer;
    transition: var(--transition-base);
    box-shadow: var(--shadow-sm);
    white-space: nowrap;
    height: fit-content;
    min-height: 44px;
}

.btn-go:hover {
    background: var(--primary-hover);
    transform: translateY(-2px);
    box-shadow: var(--shadow-md);
}

@keyframes spin {
    0% { transform: rotate(0deg); }
    100% { transform: rotate(360deg); }
}

.sync-modal {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0, 0, 0, 0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 10001;
}

.sync-content {
    background: white;
    border-radius: var(--radius-lg);
    padding: 40px;
    text-align: center;
    max-width: 300px;
    box-shadow: var(--shadow-lg);
}

.sync-icon {
    width: 40px;
    height: 40px;
    margin: 0 auto 20px;
    font-size: 40px;
    animation: spin 1s linear infinite;
}

.sync-text {
    font-size: 18px;
    font-weight: 600;
    color: var(--text-primary);
    margin-bottom: 10px;
}

.sync-subtext {
    font-size: 14px;
    color: var(--text-secondary);
}

@media (max-width: 992px) {
    .overview-bottom-section {
        grid-template-columns: 1fr;
    }
}

@media (max-width: 768px) {
    .header-content {
        padding: 0 15px;
    }
    
    .header-title {
        font-size: 24px;
    }
    
    .create-btn {
        padding: 10px 16px;
        font-size: 14px;
    }
    
    .maintenance-kpi-content {
        padding: 20px 15px;
    }
    
    .content-box {
        padding: 20px;
    }
    
    .kpi-cards-grid,
    .charts-grid {
        grid-template-columns: 1fr;
    }
    
    .tab-navigation {
        flex-direction: column;
        gap: 0;
    }
    
    .filter-form {
        flex-direction: column;
        align-items: stretch;
    }
    
    .btn-go {
        width: 100%;
        justify-content: center;
    }
}

@media (max-width: 480px) {
    .header-content {
        flex-direction: column;
        align-items: flex-start;
        gap: 15px;
    }
    
    .header-right {
        width: 100%;
        flex-direction: column;
    }
    
    .create-btn {
        width: 100%;
        justify-content: center;
    }
}

.notification {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: var(--z-index-notification);
    padding: 16px;
    border-radius: 8px;
    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
    font-family: var(--font-family);
    font-weight: 500;
    max-width: 400px;
    min-width: 300px;
    animation: slideIn 0.3s ease;
    background: white;
}

.notification-success {
    border-left: 4px solid #28a745;
}

.notification-error {
    border-left: 4px solid #dc3545;
}

.notification-info {
    border-left: 4px solid #17a2b8;
}

.notification-content {
    display: flex;
    justify-content: space-between;
    align-items: center;
    gap: 1rem;
}

.notification-content span {
    color: #333;
    font-size: 14px;
}

.notification-close {
    background: none;
    border: none;
    font-size: 18px;
    color: #999;
    cursor: pointer;
    padding: 0;
    line-height: 1;
    margin-left: 12px;
}

.notification-close:hover {
    color: #666;
}

@keyframes slideIn {
    from {
        transform: translateX(100%);
        opacity: 0;
    }
    to {
        transform: translateX(0);
        opacity: 1;
    }
}

@media (prefers-reduced-motion: reduce) {
    * {
        animation-duration: 0.01ms !important;
        animation-iteration-count: 1 !important;
        transition-duration: 0.01ms !important;
    }
}

button:focus-visible,
input:focus-visible,
select:focus-visible {
    outline: 2px solid #3b82f6;
    outline-offset: 2px;
}
</style>

<script>
(function() {
    'use strict';
    
    const CONFIG = {
        notificationTimeout: 5000,
        selectors: {
            tabButtons: '.tab-btn',
            tabPanes: '.tab-pane',
            periodButtons: '.period-btn',
            kpiCards: '.kpi-card'
        }
    };

    let elements = {};
    
    function initElements() {
        elements = {
            tabButtons: document.querySelectorAll(CONFIG.selectors.tabButtons),
            tabPanes: document.querySelectorAll(CONFIG.selectors.tabPanes),
            periodButtons: document.querySelectorAll(CONFIG.selectors.periodButtons),
            kpiCards: document.querySelectorAll(CONFIG.selectors.kpiCards)
        };
    }

    function initTabSwitching() {
        elements.tabButtons.forEach(button => {
            button.addEventListener('click', function() {
                const targetTab = this.getAttribute('data-tab');
                elements.tabButtons.forEach(btn => btn.classList.remove('active'));
                elements.tabPanes.forEach(pane => pane.classList.remove('active'));
                this.classList.add('active');
                const targetPane = document.getElementById(targetTab);
                if (targetPane) {
                    targetPane.classList.add('active');
                }
            });
        });
    }

    function initPeriodSelector() {
        const customDateRange = document.getElementById('custom-date-range');
        
        elements.periodButtons.forEach(button => {
            button.addEventListener('click', function() {
                elements.periodButtons.forEach(btn => btn.classList.remove('active'));
                this.classList.add('active');
                
                const period = this.getAttribute('data-period');
                
                if (period === 'custom') {
                    if (customDateRange) {
                        customDateRange.style.display = 'block';
                    }
                    currentDateRange = 'custom';
                    return;
                } else {
                    if (customDateRange) {
                        customDateRange.style.display = 'none';
                    }
                    
                    if (period === 'today') {
                        currentDateRange = 'today';
                    } else if (period === '7days') {
                        currentDateRange = 7;
                    } else if (period === '30days') {
                        currentDateRange = 30;
                    } else if (period === '90days') {
                        currentDateRange = 90;
                    }
                    
                    updateDashboard();
                }
            });
        });
    }

    function showNotification(message, type = 'info') {
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="notification-content">
                <span>${message}</span>
                <button class="notification-close" aria-label="Close notification">&times;</button>
            </div>
        `;
        
        document.body.appendChild(notification);
        setTimeout(() => removeNotification(notification), CONFIG.notificationTimeout);
        
        notification.querySelector('.notification-close').addEventListener('click', () => {
            removeNotification(notification);
        });
    }
    
    function removeNotification(notification) {
        if (notification && notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }
    
    window.showNotification = showNotification;

    function init() {
        initElements();
        initTabSwitching();
        initPeriodSelector();
    }

    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
})();
</script>
@endsection

