/**
 * Import function triggers from their respective submodules:
 *
 * const {onCall} = require("firebase-functions/v2/https");
 * const {onDocumentWritten} = require("firebase-functions/v2/firestore");
 *
 * See a full list of supported triggers at https://firebase.google.com/docs/functions
 */

const functions = require('firebase-functions/v1');
const admin = require('firebase-admin');
admin.initializeApp();
const nodemailer = require('nodemailer');

// Import inspection report email notification function
const inspectionReportEmailNotify = require('./InspectionReportEmailNotify');

// Import DVIR report email notification function
const dvirReportEmailNotify = require('./GlobalDvirNotify');

// Import Incident Report notification function
const incidentReportNotify = require('./GlobalIncidentReportNotify');

// Import Request Maintenance notification function
const requestMaintenanceNotify = require('./GlobalRequestmaintenanceNotify');

// Import Job Card notification function
const workOrderNotify = require('./GlobalWorkOrderNotify');

// Import HR Application notification function
const hrApplicationNotify = require('./AdminHrAppliApplyNotify');

// Import HR Application Approval/Rejection notification function
const approveApplicationSentNotify = require('./ApproveApplicationSentNotify');

// Import New User Creation notification function
const newUserCreateNotify = require('./NewUserCreateNotify');

// Import Filter Delete Unused User Data function
const filterDeleteUnusedUserData = require('./FilterDeleteUnusedUserData');

// For cost control, you can set the maximum number of containers that can be
// running at the same time. This helps mitigate the impact of unexpected
// traffic spikes by instead downgrading performance. This limit is a
// per-function limit. You can override the limit for each function using the
// `maxInstances` option in the function's options, e.g.
// `onRequest({ maxInstances: 5 }, (req, res) => { ... })`.
// NOTE: setGlobalOptions does not apply to functions using the v1 API. V1
// functions should each use functions.runWith({ maxInstances: 10 }) instead.
// In the v1 API, each function can only serve one request per container, so
// this will be the maximum concurrent request count.
// Remove setGlobalOptions usage for v1 functions to avoid undefined errors

// Create and deploy your first functions
// https://firebase.google.com/docs/functions/get-started

// v1 RTDB trigger to process password updates
exports.processAuthPasswordUpdates = functions.database
  .ref('/AdminTasks/authPasswordUpdates/{reqId}')
  .onCreate(async (snap, ctx) => {
    const payload = snap.val() || {};
    const { email, newPassword, userId } = payload;
    const now = new Date().toISOString();

    if (!email || !newPassword || !userId) {
      await snap.ref.update({ status: 'error', error: 'Missing fields', processedAt: now });
      return;
    }

    try {
      // Get or create the Auth user
      let uid;
      try {
        const user = await admin.auth().getUserByEmail(email);
        uid = user.uid;
        await admin.auth().updateUser(uid, { password: newPassword });
      } catch (e) {
        if (e.code === 'auth/user-not-found') {
          const created = await admin.auth().createUser({ email, password: newPassword });
          uid = created.uid;
        } else {
          throw e;
        }
      }

      // Write back authUid for mapping in RTDB
      await admin.database().ref(`/Globalusers/admins/${userId}/authUid`).set(uid);

      // Mark as done
      await snap.ref.update({ status: 'done', authUid: uid, processedAt: now });
    } catch (e) {
      await snap.ref.update({ status: 'error', error: e.message || String(e), processedAt: now });
    }
  });

// Create Auth user without signing in the client
exports.createAuthUser = functions.database
  .ref('/AdminTasks/authUserCreates/{reqId}')
  .onCreate(async (snap, ctx) => {
    const payload = snap.val() || {};
    const { email, password, userId } = payload;
    const now = new Date().toISOString();

    if (!email || !password || !userId) {
      await snap.ref.update({ status: 'error', error: 'Missing fields', processedAt: now });
      await admin.database().ref(`/Globalusers/admins/${userId}/authCreate`).set({ status: 'error', error: 'Missing fields', processedAt: now }).catch(() => {});
      return;
    }

    try {
      // Check if user exists by email
      let uid;
      try {
        const user = await admin.auth().getUserByEmail(email);
        uid = user.uid;
        // Do not change password here; only map existing
        await snap.ref.update({ status: 'exists', authUid: uid, processedAt: now });
        await admin.database().ref(`/Globalusers/admins/${userId}/authCreate`).set({ status: 'exists', authUid: uid, processedAt: now }).catch(() => {});
      } catch (e) {
        if (e.code === 'auth/user-not-found') {
          const created = await admin.auth().createUser({ email, password });
          uid = created.uid;
          await snap.ref.update({ status: 'created', authUid: uid, processedAt: now });
          await admin.database().ref(`/Globalusers/admins/${userId}/authCreate`).set({ status: 'created', authUid: uid, processedAt: now }).catch(() => {});
        } else {
          throw e;
        }
      }

      // Write back mapping in RTDB
      if (uid) {
        await admin.database().ref(`/Globalusers/admins/${userId}/authUid`).set(uid);
      }
    } catch (e) {
      await snap.ref.update({ status: 'error', error: e.message || String(e), processedAt: now });
      await admin.database().ref(`/Globalusers/admins/${userId}/authCreate`).set({ status: 'error', error: e.message || String(e), processedAt: now }).catch(() => {});
    }
  });

// Bulk delete Auth users and RTDB admins except whitelisted emails
exports.bulkDeleteUsersExcept = functions.database
  .ref('/AdminTasks/bulkDeleteUsers/{reqId}')
  .onCreate(async (snap, ctx) => {
    const payload = snap.val() || {};
    const keepEmails = Array.isArray(payload.keepEmails) ? payload.keepEmails.map((e) => String(e).toLowerCase().trim()) : [];
    const now = new Date().toISOString();

    if (!keepEmails || keepEmails.length === 0) {
      await snap.ref.update({ status: 'error', error: 'keepEmails array required', processedAt: now });
      return;
    }

    try {
      // 1) Collect all Auth UIDs to delete (except keepEmails)
      const uidsToDelete = [];
      let nextPageToken = undefined;
      do {
        const res = await admin.auth().listUsers(1000, nextPageToken);
        res.users.forEach((u) => {
          const email = (u.email || '').toLowerCase();
          if (email && !keepEmails.includes(email)) {
            uidsToDelete.push(u.uid);
          }
        });
        nextPageToken = res.pageToken;
      } while (nextPageToken);

      // Delete in batches of up to 1000
      let authDeletedCount = 0;
      for (let i = 0; i < uidsToDelete.length; i += 1000) {
        const batch = uidsToDelete.slice(i, i + 1000);
        const result = await admin.auth().deleteUsers(batch);
        authDeletedCount += result.successCount || 0;
      }

      // 2) Clean RTDB under /Globalusers/admins: delete everything except nodes whose email is whitelisted
      const adminsSnap = await admin.database().ref('/Globalusers/admins').once('value');
      const adminsVal = adminsSnap.val() || {};
      const updates = {};
      let rtdbDeletedCount = 0;
      Object.keys(adminsVal).forEach((key) => {
        const node = adminsVal[key];
        const email = (node && node.email ? String(node.email) : '').toLowerCase();
        const shouldKeep = email && keepEmails.includes(email);
        if (!shouldKeep) {
          updates[`/Globalusers/admins/${key}`] = null;
          rtdbDeletedCount += 1;
        }
      });
      if (Object.keys(updates).length > 0) {
        await admin.database().ref('/').update(updates);
      }

      await snap.ref.update({
        status: 'done',
        processedAt: now,
        keepEmails,
        summary: { authDeletedCount, rtdbDeletedCount },
      });
    } catch (e) {
      await snap.ref.update({ status: 'error', error: e.message || String(e), processedAt: now });
    }
  });

// Transport for SMTP emails (provided credentials)
const mailTransport = nodemailer.createTransport({
  host: 'smtp.gmail.com',
  port: 587,
  secure: false,
  auth: {
    user: 'mandeep.owninfotech@gmail.com',
    pass: 'aeqa purb mxtl hkhh',
  },
});

// Helper to send email
async function sendEmail(toEmail, subject, html) {
  const mailOptions = {
    from: 'Titan Drilling <support@titandrillingzm.com>',
    to: toEmail,
    subject,
    html,
  };
  return mailTransport.sendMail(mailOptions);
}

// Helper to send FCM push
async function sendFcmToToken(token, title, body, data = {}) {
  const message = {
    token,
    notification: { title, body },
    data: Object.entries(data).reduce((acc, [k, v]) => ({ ...acc, [k]: String(v) }), {}),
    android: { priority: 'high' },
    apns: { payload: { aps: { sound: 'default' } } },
  };
  return admin.messaging().send(message);
}

// Helper to send FCM to a topic (e.g., per-user topic)
async function sendFcmToTopic(topic, title, body, data = {}) {
  const message = {
    topic,
    notification: { title, body },
    data: Object.entries(data).reduce((acc, [k, v]) => ({ ...acc, [k]: String(v) }), {}),
    android: { priority: 'high' },
    apns: { payload: { aps: { sound: 'default' } } },
  };
  return admin.messaging().send(message);
}

// DISABLED: Queue node: /AdminTasks/notifyProfileUpdated/{reqId}
// This function is now DISABLED to prevent duplicate notifications
// Profile updates are now handled by the dedicated sendProfileUpdateNotification function
exports.notifyProfileUpdated = functions.database
  .ref('/AdminTasks/notifyProfileUpdated/{reqId}')
  .onCreate(async (snap, ctx) => {
    // DISABLED: This function no longer sends notifications
    // Profile updates are now handled by /AdminTasks/profileUpdateNotifications
    // This prevents duplicate notifications and email spam
    
    console.log('Profile update notification request received but function is disabled');
    await snap.ref.update({ 
      status: 'disabled', 
      message: 'Function disabled - use profileUpdateNotifications instead',
      processedAt: new Date().toISOString() 
    });
    return null;
  });

// DISABLED: Auto-send notification and email when an admin profile is updated
// This function is now DISABLED to prevent duplicate notifications
// Profile updates are now handled by the dedicated sendProfileUpdateNotification function
exports.onAdminProfileUpdated = functions.database
  .ref('/Globalusers/admins/{userId}')
  .onUpdate(async (change, ctx) => {
    // DISABLED: This function no longer sends notifications
    // Profile updates are now handled by /AdminTasks/profileUpdateNotifications
    // This prevents duplicate notifications and email spam
    console.log('Profile update detected but notifications are now handled by dedicated system');
    return null;
  });

// Queue: update Auth account for an existing user by UID (no creation)
// Payload: { userId: <authUid>, email?: <newEmail>, password?: <newPassword> }
exports.processAuthAccountUpdates = functions.database
  .ref('/AdminTasks/authAccountUpdates/{reqId}')
  .onCreate(async (snap, ctx) => {
    const payload = snap.val() || {};
    const { userId, email, password } = payload;
    const now = new Date().toISOString();

    if (!userId || (!email && !password)) {
      await snap.ref.update({ status: 'error', error: 'userId and at least one of email/password required', processedAt: now });
      return;
    }

    try {
      const update = {};
      if (email) update.email = String(email).trim();
      if (password) update.password = String(password);

      await admin.auth().updateUser(userId, update);

      // Keep RTDB email in sync if changed
      if (email) {
        await admin.database().ref(`/Globalusers/admins/${userId}/email`).set(update.email).catch(() => {});
      }

      await snap.ref.update({ status: 'done', processedAt: now });
    } catch (e) {
      await snap.ref.update({ status: 'error', error: e.message || String(e), processedAt: now });
    }
  });

// Import the notification service
const notificationService = require('./BackendNotificationService');
const noticeBoardService = require('./GlobalNoticeBoardNotifications');

// HTTP function to send test push notification
exports.sendTestPushNotification = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { fcmToken, highPriority = false } = req.body;

    if (!fcmToken) {
      res.status(400).json({ success: false, error: 'FCM token is required' });
      return;
    }

    let result;
    if (highPriority) {
      result = await notificationService.sendHighPriorityNotification(
        fcmToken,
        '🚨 High Priority Test',
        'This is a HIGH PRIORITY test notification!',
        { type: 'high_priority_test', timestamp: Date.now().toString() }
      );
    } else {
      result = await notificationService.sendTestPushNotification(fcmToken);
    }

    res.json(result);
  } catch (error) {
    console.error('Error in sendTestPushNotification:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to validate FCM token
exports.validateFCMToken = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { fcmToken } = req.body;

    if (!fcmToken) {
      res.status(400).json({ success: false, error: 'FCM token is required' });
      return;
    }

    const result = await notificationService.validateFCMToken(fcmToken);
    res.json({ success: true, ...result });
  } catch (error) {
    console.error('Error in validateFCMToken:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to send high-priority notification
exports.sendHighPriorityNotification = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { fcmToken, title, body, data } = req.body;

    if (!fcmToken || !title || !body) {
      res.status(400).json({ success: false, error: 'FCM token, title, and body are required' });
      return;
    }

    const result = await notificationService.sendHighPriorityNotification(fcmToken, title, body, data);
    res.json(result);
  } catch (error) {
    console.error('Error in sendHighPriorityNotification:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to send push notification by email
exports.sendPushNotificationByEmail = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { email, title, body, data } = req.body;

    if (!email) {
      res.status(400).json({ success: false, error: 'Email is required' });
      return;
    }

    const result = await notificationService.sendPushNotificationByEmail(email, title, body, data);
    res.json(result);
  } catch (error) {
    console.error('Error in sendPushNotificationByEmail:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to send Job Card email
exports.sendWorkOrderEmail = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { to, subject, html, type } = req.body;

    if (!to || !subject || !html) {
      res.status(400).json({ success: false, error: 'to, subject, and html are required' });
      return;
    }

    // Send email using the existing sendEmail function
    const result = await sendEmail(to, subject, html);
    
    console.log(`📧 Job Card email sent to: ${to}`);
    res.json({ success: true, messageId: result });
  } catch (error) {
    console.error('Error in sendWorkOrderEmail:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to get FCM token by email
exports.getFCMTokenByEmail = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { email } = req.body;

    if (!email) {
      res.status(400).json({ success: false, error: 'Email is required' });
      return;
    }

    const result = await notificationService.getFCMTokenByEmail(email);
    res.json(result);
  } catch (error) {
    console.error('Error in getFCMTokenByEmail:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to store FCM token for a user
exports.storeFCMToken = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { userId, fcmToken } = req.body;

    if (!userId || !fcmToken) {
      res.status(400).json({ success: false, error: 'User ID and FCM token are required' });
      return;
    }

    const result = await notificationService.storeFCMToken(userId, fcmToken);
    res.json(result);
  } catch (error) {
    console.error('Error in storeFCMToken:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to get all users with FCM tokens
exports.getAllUsersWithFCMTokens = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'GET') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const result = await notificationService.getAllUsersWithFCMTokens();
    res.json(result);
  } catch (error) {
    console.error('Error in getAllUsersWithFCMTokens:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to send notification to all devices of a user
exports.sendNotificationToAllUserDevices = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { userId, title, body, data } = req.body;

    if (!userId || !title || !body) {
      res.status(400).json({ success: false, error: 'User ID, title, and body are required' });
      return;
    }

    const result = await notificationService.sendNotificationToAllUserDevices(userId, title, body, data);
    res.json(result);
  } catch (error) {
    console.error('Error in sendNotificationToAllUserDevices:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to send notice board notification to all admins
exports.sendNoticeBoardNotification = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { title, description, imageUrl, noticeId, createdAt } = req.body;

    if (!title || !description) {
      res.status(400).json({ success: false, error: 'Title and description are required' });
      return;
    }

    const noticeData = {
      noticeId: noticeId || `notice_${Date.now()}`,
      title: title,
      description: description,
      imageUrl: imageUrl || null,
      createdAt: createdAt || new Date().toISOString()
    };

    console.log('📢 Sending notice board notification to all admins...');
    const result = await noticeBoardService.sendNoticeBoardNotificationToAllAdmins(noticeData);
    
    res.json(result);
  } catch (error) {
    console.error('Error in sendNoticeBoardNotification:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to debug FCM token storage for a user
exports.debugFCMTokenStorage = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { userId } = req.body;

    if (!userId) {
      res.status(400).json({ success: false, error: 'User ID is required' });
      return;
    }

    const result = await notificationService.debugFCMTokenStorage(userId);
    res.json(result);
  } catch (error) {
    console.error('Error in debugFCMTokenStorage:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to send test notice board notification
exports.sendTestNoticeBoardNotification = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    console.log('🧪 Sending test notice board notification...');
    const result = await noticeBoardService.sendTestNoticeBoardNotification();
    
    res.json(result);
  } catch (error) {
    console.error('Error in sendTestNoticeBoardNotification:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to get notice board notification statistics
exports.getNoticeBoardNotificationStats = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'GET') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const result = await noticeBoardService.getNoticeBoardNotificationStats();
    res.json(result);
  } catch (error) {
    console.error('Error in getNoticeBoardNotificationStats:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to refresh and validate FCM tokens for a user
exports.refreshAndValidateFCMTokens = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { userId } = req.body;

    if (!userId) {
      res.status(400).json({ success: false, error: 'User ID is required' });
      return;
    }

    const result = await notificationService.refreshAndValidateFCMTokens(userId);
    res.json(result);
  } catch (error) {
    console.error('Error in refreshAndValidateFCMTokens:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to clean up invalid FCM tokens
exports.cleanupInvalidFCMTokens = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const result = await notificationService.cleanupInvalidFCMTokens();
    res.json(result);
  } catch (error) {
    console.error('Error in cleanupInvalidFCMTokens:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// HTTP function to get FCM token statistics for all users
exports.getFCMTokenStatistics = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'GET') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const result = await notificationService.getFCMTokenStatistics();
    res.json(result);
  } catch (error) {
    console.error('Error in getFCMTokenStatistics:', error);
    res.status(500).json({ success: false, error: error.message });
  }
});

// NEW: Dedicated function for Edit Team Member profile updates
// This function sends BOTH email AND push notifications when profile is updated
exports.sendEditTeamMemberNotification = functions.database
  .ref('/AdminTasks/editTeamMemberUpdates/{reqId}')
  .onCreate(async (snap, ctx) => {
    const payload = snap.val() || {};
    const { userId, updateData, adminName } = payload;
    const now = new Date().toISOString();

    if (!userId || !updateData || !adminName) {
      await snap.ref.update({ 
        status: 'error', 
        error: 'Missing required fields: userId, updateData, or adminName', 
        processedAt: now 
      });
      return;
    }

    try {
      console.log(`🚀 Sending Edit Team Member notification to user ${userId}...`);
      
      // Get user data to send personalized notifications
      const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
      if (!userSnapshot.exists()) {
        await snap.ref.update({ 
          status: 'error', 
          error: 'User not found in database', 
          processedAt: now 
        });
        return;
      }
      
      const userData = userSnapshot.val();
      const userName = userData.name || 'Team Member';
      const userEmail = userData.email || '';
      
      // Prepare notification message
      const { changedFields, emailChanged, previousEmail, newEmail } = updateData;
      const baseMsg = `Your profile has been updated by ${adminName}.`;
      const details = changedFields && changedFields.length > 0 ? ` Changes: ${changedFields.join(', ')}` : '';
      const emailChangeMsg = emailChanged ? ` Email changed from ${previousEmail || 'N/A'} to ${newEmail}.` : '';
      const notificationMessage = `${baseMsg}${details}${emailChangeMsg}`.trim();

      // 1. SEND PUSH NOTIFICATION
      let pushNotificationResult = { success: false, error: 'No devices found' };
      
      if (userData.devices && Object.keys(userData.devices).length > 0) {
        // Multi-device system
        let successCount = 0;
        let failureCount = 0;
        
        for (const [deviceKey, deviceData] of Object.entries(userData.devices)) {
          if (deviceData.fcmStatus === 'active' && deviceData.token) {
            try {
              const result = await admin.messaging().send({
                token: deviceData.token,
                notification: {
                  title: 'Profile Updated',
                  body: notificationMessage,
                },
                data: {
                  type: 'profile_updated',
                  updatedBy: adminName,
                  emailChanged: emailChanged ? 'true' : 'false',
                  previousEmail: previousEmail || '',
                  newEmail: newEmail || '',
                  changedFields: JSON.stringify(changedFields || []),
                  timestamp: now
                },
                android: {
                  priority: 'high',
                  notification: {
                    channelId: 'profile_updates',
                    priority: 'high'
                  }
                },
                apns: {
                  payload: {
                    aps: {
                      sound: 'default',
                      badge: 1
                    }
                  }
                }
              });
              
              if (result) {
                successCount++;
                console.log(`Push notification sent to device ${deviceKey} for user ${userId}`);
              }
            } catch (error) {
              failureCount++;
              console.error(`Failed to send push notification to device ${deviceKey}:`, error);
            }
          }
        }
        
        if (successCount > 0) {
          pushNotificationResult = { success: true, devicesNotified: successCount, devicesFailed: failureCount };
        }
      } else if (userData.NotificationFCMtoken) {
        // Legacy single token system
        try {
          const result = await admin.messaging().send({
            token: userData.NotificationFCMtoken,
            notification: {
              title: 'Profile Updated',
              body: notificationMessage,
            },
            data: {
              type: 'profile_updated',
              updatedBy: adminName,
              emailChanged: emailChanged ? 'true' : 'false',
              previousEmail: previousEmail || '',
              newEmail: newEmail || '',
              changedFields: JSON.stringify(changedFields || []),
              timestamp: now
            },
            android: {
              priority: 'high',
              notification: {
                channelId: 'profile_updates',
                priority: 'high'
              }
            },
            apns: {
              payload: {
                aps: {
                  sound: 'default',
                  badge: 1
                }
              }
            }
          });
          
          if (result) {
            pushNotificationResult = { success: true, devicesNotified: 1, devicesFailed: 0 };
            console.log(`Push notification sent to legacy token for user ${userId}`);
          }
        } catch (error) {
          pushNotificationResult = { success: false, error: error.message };
          console.error(`Failed to send push notification to legacy token:`, error);
        }
      }

      // 2. SEND EMAIL NOTIFICATION
      let emailResult = { success: false, error: 'No email found' };
      
      if (userEmail) {
        try {
          // Create detailed email content
          const emailSubject = 'Your Titan Drilling Profile Has Been Updated';
          const emailHtml = `
            <div style="font-family: Arial, sans-serif; font-size: 14px; color: #333; max-width: 600px; margin: 0 auto;">
              <div style="background: linear-gradient(135deg, #015185 0%, #3481BC 100%); padding: 20px; text-align: center;">
                <h1 style="color: white; margin: 0; font-size: 24px;">Profile Update Notification</h1>
              </div>
              
              <div style="padding: 20px; background: #f9f9f9;">
                <p>Dear <strong>${userName}</strong>,</p>
                
                <p>Your profile has been successfully updated by <strong>${adminName}</strong>.</p>
                
                <div style="background: white; padding: 15px; border-radius: 8px; border-left: 4px solid #015185; margin: 20px 0;">
                  <h3 style="color: #015185; margin-top: 0;">Update Summary</h3>
                  <ul style="margin: 10px 0; padding-left: 20px;">
                    ${changedFields && changedFields.length > 0 ? 
                      changedFields.map(field => `<li><strong>${field}</strong></li>`).join('') : 
                      '<li>Profile information updated</li>'
                    }
                  </ul>
                </div>
                
                ${emailChanged ? `
                <div style="background: #fff3cd; padding: 15px; border-radius: 8px; border-left: 4px solid #ffc107; margin: 20px 0;">
                  <h3 style="color: #856404; margin-top: 0;">⚠️ Email Address Changed</h3>
                  <p><strong>Previous Email:</strong> ${previousEmail || 'N/A'}</p>
                  <p><strong>New Email:</strong> ${newEmail || 'N/A'}</p>
                  <p style="font-size: 12px; color: #856404;">Please use your new email address for future logins.</p>
                </div>
                ` : ''}
                
                <p>If you did not request these changes or have any questions, please contact your administrator immediately.</p>
                
                <div style="background: #e7f3ff; padding: 15px; border-radius: 8px; border-left: 4px solid #015185; margin: 20px 0;">
                  <h3 style="color: #015185; margin-top: 0;">Next Steps</h3>
                  <ul style="margin: 10px 0; padding-left: 20px;">
                    <li>Review your updated profile information</li>
                    <li>Update your login credentials if needed</li>
                    <li>Contact support if you notice any discrepancies</li>
                  </ul>
                </div>
                
                <p>Best regards,<br><strong>Titan Drilling Team</strong></p>
                
                <hr style="border: none; border-top: 1px solid #ddd; margin: 20px 0;">
                <p style="font-size: 12px; color: #666; text-align: center;">
                  This is an automated notification. Please do not reply to this email.
                </p>
              </div>
            </div>
          `;
          
          // Send email using the existing sendEmail function
          const emailId = await sendEmail(userEmail, emailSubject, emailHtml);
          
          if (emailId) {
            emailResult = { success: true, emailId: emailId };
            console.log(`Email notification sent successfully to ${userEmail} for user ${userId}`);
          }
        } catch (error) {
          emailResult = { success: false, error: error.message };
          console.error(`Failed to send email notification to ${userEmail}:`, error);
        }
      }

      // Update the task status with results
      const finalResult = {
        pushNotification: pushNotificationResult,
        email: emailResult,
        summary: {
          userId: userId,
          userName: userName,
          adminName: adminName,
          changedFields: changedFields || [],
          emailChanged: emailChanged || false,
          processedAt: now
        }
      };

      await snap.ref.update({ 
        status: 'completed', 
        processedAt: now,
        result: finalResult
      });
      
      console.log(`✅ Edit Team Member notification completed for user ${userId}!`);
      console.log(`📱 Push: ${pushNotificationResult.success ? 'Success' : 'Failed'}`);
      console.log(`📧 Email: ${emailResult.success ? 'Success' : 'Failed'}`);
      
    } catch (error) {
      await snap.ref.update({ 
        status: 'error', 
        error: error.message || String(error), 
        processedAt: now 
      });
      console.error(`❌ Error sending Edit Team Member notification to user ${userId}:`, error);
    }
  });

// Inspection Report Email Notification Function
exports.sendInspectionReportEmails = functions
  .runWith({ memory: '512MB', timeoutSeconds: 120 })
  .database
  .ref('/GlobalInspectionReport/admins/{userId}/{reportId}')
  .onCreate(async (snap, ctx) => {
    const reportData = snap.val();
    const { userId, reportId } = ctx.params;
    const now = new Date().toISOString();

    if (!reportData) {
      console.log('No report data found');
      return null;
    }

    try {
      // Get the user who created the inspection
      const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
      const userData = userSnapshot.val();
      
      if (!userData) {
        console.log('User data not found for userId:', userId);
        return null;
      }

      const { email: creatorEmail, name: creatorName, countries, projects } = userData;
      const { step1, step2, step3 } = reportData;
      
      if (!step1 || !step3) {
        console.log('Incomplete report data');
        return null;
      }

      const { country, project, inspector, selectedEquipment, requestNumber } = step1;
      const { equipmentStatus, faultyItems, pdfDownloadUrl } = step3;
      const checklistTitle = step2?.checklistTitle || 'Inspection Report';
      
      // Debug logging
      console.log('=== MAIN EMAIL FUNCTION DEBUG ===');
      console.log('Report data structure:', { step1: !!step1, step2: !!step2, step3: !!step3 });
      console.log('Extracted data:', {
        country,
        project,
        inspector,
        requestNumber,
        checklistTitle,
        equipmentName: selectedEquipment?.equipmentName
      });

      // Find all supervisors in the same country and project
      const supervisorsSnapshot = await admin.database()
        .ref('/Globalusers/admins')
        .orderByChild('role')
        .equalTo('Supervisor')
        .once('value');
      
      const supervisorsData = supervisorsSnapshot.val() || {};
      const relevantSupervisors = [];

      Object.entries(supervisorsData).forEach(([supervisorId, supervisorData]) => {
        if (supervisorData.countries && supervisorData.projects) {
          const hasMatchingCountry = supervisorData.countries.includes(country);
          const hasMatchingProject = supervisorData.projects.includes(project);
          
          if (hasMatchingCountry && hasMatchingProject) {
            relevantSupervisors.push({
              id: supervisorId,
              email: supervisorData.email,
              name: supervisorData.name || 'Supervisor'
            });
          }
        }
      });

      // Prepare email content
      const emailSubject = `New Inspection Report: ${requestNumber || 'N/A'}`;
      
      const emailHtml = `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <div style="background-color: #3481BC; color: white; padding: 20px; text-align: center;">
            <h1 style="margin: 0;">Inspection Report Notification</h1>
          </div>
          
          <div style="padding: 20px; background-color: #f9f9f9;">
            <h2 style="color: #333;">New Inspection Report Created</h2>
            
            <div style="background-color: white; padding: 15px; border-radius: 5px; margin: 15px 0;">
              <h3 style="color: #3481BC; margin-top: 0;">Report Details</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Request Number:</td>
                  <td style="padding: 8px;">${requestNumber || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Country:</td>
                  <td style="padding: 8px;">${country || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Project:</td>
                  <td style="padding: 8px;">${project || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Inspector:</td>
                  <td style="padding: 8px;">${inspector || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Equipment:</td>
                  <td style="padding: 8px;">${selectedEquipment?.equipmentName || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Status:</td>
                  <td style="padding: 8px;">${equipmentStatus || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Faulty Items:</td>
                  <td style="padding: 8px;">${faultyItems || 0}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created By:</td>
                  <td style="padding: 8px;">${creatorName || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created At:</td>
                  <td style="padding: 8px;">${new Date(step3.createdAt).toLocaleString()}</td>
                </tr>
              </table>
            </div>
            
            ${pdfDownloadUrl ? `
            <div style="text-align: center; margin: 20px 0;">
              <a href="${pdfDownloadUrl}" style="background-color: #3481BC; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block;">
                View Inspection Report
              </a>
            </div>
            ` : ''}
            
            <div style="background-color: #e8f4fd; padding: 15px; border-radius: 5px; border-left: 4px solid #3481BC;">
              <p style="margin: 0; color: #333;">
                <strong>Note:</strong> This is an automated notification. The inspection report has been created and is now available for review.
              </p>
            </div>
          </div>
          
          <div style="background-color: #f0f0f0; padding: 15px; text-align: center; color: #666; font-size: 12px;">
            <p>© ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
          </div>
        </div>
      `;

      // Send email to creator
      if (creatorEmail) {
        try {
          await sendEmail(creatorEmail, emailSubject, emailHtml);
          console.log('Email sent to creator:', creatorEmail);
        } catch (error) {
          console.error('Failed to send email to creator:', error);
        }
      }

      // Send email to relevant supervisors
      const supervisorEmails = [];
      for (const supervisor of relevantSupervisors) {
        if (supervisor.email && supervisor.email !== creatorEmail) {
          try {
            await sendEmail(supervisor.email, emailSubject, emailHtml);
            supervisorEmails.push(supervisor.email);
            console.log('Email sent to supervisor:', supervisor.email);
          } catch (error) {
            console.error('Failed to send email to supervisor:', supervisor.email, error);
          }
        }
      }

              // Send push notifications to filtered users (ProAdmin, Supervisors, Managers)
        try {
          console.log('🚀 Starting inspection report push notifications...');
          
          // Prepare inspection report data
          const inspectionReportData = {
            reportId: reportId || 'unknown',
            title: `🔍 New Inspection Report: ${requestNumber || 'N/A'}`,
            description: `${inspector || 'Unknown'} created inspection report for ${project || 'Unknown Project'} - ${selectedEquipment?.equipmentName || 'Equipment'}`,
            country: country || 'Unknown',
            project: project || 'Unknown',
            inspector: inspector || 'Unknown',
            requestNumber: requestNumber || 'N/A',
            equipmentName: selectedEquipment?.equipmentName || 'N/A',
            equipmentStatus: equipmentStatus || 'Unknown',
            faultyItems: faultyItems || 0,
            creatorEmail: creatorEmail || 'unknown@example.com',
            creatorName: creatorName || 'Unknown',
            createdAt: now,
            noticeId: reportId || 'unknown',
            imageUrl: null
          };
          
          console.log('🔍 Inspection report data:', JSON.stringify(inspectionReportData, null, 2));
          
          // Get filtered users for inspection report notifications
          const filteredUsers = await getFilteredUsersForInspectionReport(country, project);
          console.log('📋 Filtered users for inspection report:', filteredUsers.length);

          if (filteredUsers && filteredUsers.length > 0) {
            console.log(`📊 Found ${filteredUsers.length} eligible users for inspection report notifications`);
            
            let totalSuccess = 0;
            let totalFailed = 0;
            let totalInvalidTokens = 0;
            const notificationResults = [];

            // Send notification to each eligible user
            for (const user of filteredUsers) {
              try {
                console.log(`📤 Sending inspection notification to user: ${user.name} (${user.email})`);
                
                // Save notification to user's database for bell icon
                const notificationRef = admin.database().ref(`/Globalusers/admins/${user.userId}/notifications`).push();
                await notificationRef.set({
                  title: inspectionReportData.title,
                  body: inspectionReportData.description.substring(0, 100) + (inspectionReportData.description.length > 100 ? '...' : ''),
                  type: 'inspection_report',
                  requestNumber: inspectionReportData.requestNumber,
                  project: inspectionReportData.project,
                  equipmentName: inspectionReportData.equipmentName,
                  data: {
                    type: 'inspection_report',
                    reportId: inspectionReportData.reportId,
                    noticeId: inspectionReportData.noticeId,
                    title: inspectionReportData.title,
                    description: inspectionReportData.description,
                    imageUrl: inspectionReportData.imageUrl,
                    createdAt: inspectionReportData.createdAt,
                    country: inspectionReportData.country,
                    project: inspectionReportData.project,
                    inspector: inspectionReportData.inspector,
                    requestNumber: inspectionReportData.requestNumber,
                    equipmentName: inspectionReportData.equipmentName,
                    equipmentStatus: inspectionReportData.equipmentStatus,
                    faultyItems: inspectionReportData.faultyItems.toString(),
                    creatorEmail: inspectionReportData.creatorEmail,
                    creatorName: inspectionReportData.creatorName,
                    action: 'view_inspection_report',
                  },
                  timestamp: Date.now(),
                  isRead: false,
                  createdAt: now,
                });
                console.log(`✅ Notification saved to database for user ${user.userId}`);
                
                // Send FCM notification to all user's devices
                let deviceSuccessCount = 0;
                let deviceFailedCount = 0;
                let invalidTokenCount = 0;
                
                for (const device of user.devices) {
                  try {
                    await admin.messaging().send({
                      token: device.fcmToken,
                      notification: {
                        title: inspectionReportData.title,
                        body: inspectionReportData.description.substring(0, 100) + (inspectionReportData.description.length > 100 ? '...' : ''),
                      },
                      data: {
                        type: 'inspection_report',
                        reportId: inspectionReportData.reportId,
                        noticeId: inspectionReportData.noticeId,
                        title: inspectionReportData.title,
                        description: inspectionReportData.description,
                        imageUrl: inspectionReportData.imageUrl || '',
                        createdAt: inspectionReportData.createdAt,
                        country: inspectionReportData.country,
                        project: inspectionReportData.project,
                        inspector: inspectionReportData.inspector,
                        requestNumber: inspectionReportData.requestNumber,
                        equipmentName: inspectionReportData.equipmentName,
                        equipmentStatus: inspectionReportData.equipmentStatus,
                        faultyItems: inspectionReportData.faultyItems.toString(),
                        creatorEmail: inspectionReportData.creatorEmail,
                        creatorName: inspectionReportData.creatorName,
                        action: 'view_inspection_report',
                      },
                      android: {
                        priority: 'high',
                        notification: {
                          channelId: 'inspection_reports',
                          priority: 'high',
                          sound: 'default',
                        }
                      },
                      apns: {
                        payload: {
                          aps: {
                            sound: 'default',
                            badge: 1,
                          }
                        }
                      }
                    });
                    deviceSuccessCount++;
                    console.log(`✅ Notification sent to device ${device.deviceKey}`);
                  } catch (deviceError) {
                    if (deviceError.code === 'messaging/invalid-registration-token' || 
                        deviceError.code === 'messaging/registration-token-not-registered') {
                      invalidTokenCount++;
                      console.log(`🔄 Invalid token for device ${device.deviceKey}`);
                      // Mark device as inactive
                      await admin.database().ref(`/Globalusers/admins/${user.userId}/devices/${device.deviceKey}/fcmStatus`).set('inactive');
                    } else {
                      deviceFailedCount++;
                      console.error(`❌ Failed to send to device ${device.deviceKey}:`, deviceError);
                    }
                  }
                }
                
                if (deviceSuccessCount > 0) {
                  totalSuccess += deviceSuccessCount;
                  notificationResults.push({
                    user: user.name,
                    email: user.email,
                    status: 'success',
                    devices: deviceSuccessCount,
                    failed: deviceFailedCount,
                    invalidTokens: invalidTokenCount,
                  });
                } else {
                  totalFailed += user.totalDevices || 0;
                  notificationResults.push({
                    user: user.name,
                    email: user.email,
                    status: 'failed',
                    error: 'All devices failed',
                  });
                }
              } catch (userError) {
                console.error(`❌ Error sending inspection notification to user ${user.name}:`, userError);
                totalFailed += user.totalDevices || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'error',
                  error: userError.message,
                });
              }
            }

            // Log comprehensive results
            console.log('📊 Inspection report push notification results:', {
              totalUsers: filteredUsers.length,
              totalSuccess,
              totalFailed,
              totalInvalidTokens,
              results: notificationResults,
            });

            console.log(`✅ Inspection report push notifications completed: ${totalSuccess} successful, ${totalFailed} failed, ${totalInvalidTokens} invalid tokens`);
            
          } else {
            console.log('⚠️ No eligible users found for inspection report push notifications');
          }

        } catch (error) {
          console.error('❌ Error sending inspection report push notifications:', error);
        }

      // Store email records in Globalinspectionemails
      const emailRecord = {
        reportId,
        userId,
        creatorEmail,
        creatorName,
        country,
        project,
        requestNumber: requestNumber,
        equipmentName: selectedEquipment?.equipmentName,
        equipmentStatus,
        faultyItems,
        supervisorEmails,
        emailSent: true,
        emailSentAt: now,
        emailContent: emailHtml,
        pushNotificationsSent: true,
        pushNotificationsSentAt: now,
        status: 'sent'
      };

      await admin.database().ref(`/Globalinspectionemails/${reportId}`).set(emailRecord);

      console.log(`Inspection report notifications sent successfully. Emails: Creator: ${creatorEmail}, Supervisors: ${supervisorEmails.join(', ')}. Push notifications sent to all users.`);
      
      return {
        success: true,
        creatorEmail,
        supervisorEmails,
        totalEmailsSent: 1 + supervisorEmails.length,
        pushNotificationsSent: true,
        message: 'Emails and push notifications sent successfully'
      };

    } catch (error) {
      console.error('Error sending inspection report emails:', error);
      
      // Store error record
      const errorRecord = {
        reportId,
        userId,
        error: error.message,
        errorAt: now,
        status: 'error'
      };
      
      await admin.database().ref(`/Globalinspectionemails/${reportId}`).set(errorRecord);
      
      return {
        success: false,
        error: error.message
      };
    }
  });

// Function to get filtered users for inspection report notifications
async function getFilteredUsersForInspectionReport(inspectionCountry, inspectionProject) {
  try {
    console.log('🔍 Filtering users for inspection report notifications...');
    console.log('🔍 Inspection Country:', inspectionCountry, '(type:', typeof inspectionCountry, ')');
    console.log('🔍 Inspection Project:', inspectionProject, '(type:', typeof inspectionProject, ')');
    
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      console.log('❌ No admin users found');
      return [];
    }

    const admins = adminsSnapshot.val();
    const eligibleUsers = [];

    for (const [userId, userData] of Object.entries(admins)) {
      const userRole = userData.role || 'Unknown';
      const userName = userData.name || 'Unknown';
      const userEmail = userData.email || 'No email';
      const userCountries = userData.countries || [];
      const userProjects = userData.projects || [];
      
      console.log(`👤 Checking user: ${userName} (${userRole})`);
      console.log(`   Raw user data:`, JSON.stringify(userData, null, 2));
      console.log(`   Countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
      console.log(`   Projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
      
      // Check if user has active FCM tokens
      const devices = userData.devices || {};
      const activeDevices = Object.entries(devices).filter(([_, device]) => 
        device.token && device.fcmStatus === 'active'
      );
      
      if (activeDevices.length === 0) {
        console.log(`   ❌ No active FCM tokens for ${userName}`);
        continue;
      }
      
      // Check role eligibility
      const isEligibleRole = isRoleEligibleForInspectionReport(userRole);
      if (!isEligibleRole) {
        console.log(`   ❌ Role ${userRole} not eligible for inspection reports`);
        continue;
      }
      
      // Check if role should always get notifications (no country/project check)
      if (isAlwaysEligibleRole(userRole)) {
        console.log(`   ✅ ${userRole} ${userName} is always eligible (no country/project check needed)`);
      }
      // For supervisors and all manager types, check country and project matching
      else if (userRole === 'Supervisor' || isManagerRole(userRole)) {
        console.log(`   🔍 Checking country/project match for ${userRole} ${userName}:`);
        console.log(`      User countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
        console.log(`      User projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
        console.log(`      Inspection country: "${inspectionCountry}" (type: ${typeof inspectionCountry})`);
        console.log(`      Inspection project: "${inspectionProject}" (type: ${typeof inspectionProject})`);
        
        // Ensure userCountries and userProjects are arrays
        const userCountriesArray = Array.isArray(userCountries) ? userCountries : [];
        const userProjectsArray = Array.isArray(userProjects) ? userProjects : [];
        
        // For supervisors, check both country AND project match
        // For managers, check both country AND project match
        const hasMatchingCountry = userCountriesArray.some(country => 
          country && country.toString().toLowerCase() === inspectionCountry.toLowerCase()
        );
        
        const hasMatchingProject = userProjectsArray.some(project => 
          project && project.toString().toLowerCase() === inspectionProject.toLowerCase()
        );
        
        console.log(`      Country match result: ${hasMatchingCountry}`);
        console.log(`      Project match result: ${hasMatchingProject}`);
        
        if (!hasMatchingCountry || !hasMatchingProject) {
          console.log(`   ❌ ${userRole} ${userName} doesn't match country/project`);
          console.log(`      Has country "${inspectionCountry}": ${hasMatchingCountry}`);
          console.log(`      Has project "${inspectionProject}": ${hasMatchingProject}`);
          continue;
        }
        
        console.log(`   ✅ ${userRole} ${userName} matches country and project`);
      } else {
        console.log(`   ❌ ${userRole} ${userName} is not eligible for inspection report notifications`);
        continue;
      }
      
      // Add user to eligible list
      eligibleUsers.push({
        userId,
        name: userName,
        email: userEmail,
        role: userRole,
        totalDevices: activeDevices.length,
        devices: activeDevices.map(([deviceKey, device]) => ({
          deviceKey,
          fcmToken: device.token,
          deviceId: device.deviceInfo?.deviceId || 'unknown',
          platform: device.deviceInfo?.platform || 'unknown',
          lastTokenUpdate: device.lastTokenUpdate || 'Unknown',
          isCurrentDevice: device.isCurrentDevice || false
        }))
      });
    }

    console.log(`✅ Found ${eligibleUsers.length} eligible users for inspection report notifications`);
    return eligibleUsers;
    
  } catch (error) {
    console.error('❌ Error filtering users for inspection report:', error);
    return [];
  }
}

// Function to check if a role is eligible for inspection report notifications
function isRoleEligibleForInspectionReport(role) {
  const eligibleRoles = [
    'admin',
    'ProAdmin',
    'Supervisor',
    'CEO',
    'Country Manager',
    'Manager',
    'Project Manager',
    'Operations Manager',
    'Site Manager',
    'Field Manager',
    'Regional Manager',
    'Area Manager',
    'Department Manager',
    'Team Manager',
    'Senior Manager',
    'General Manager',
    'Executive Manager'
  ];
  
  // Check exact match first
  if (eligibleRoles.includes(role)) {
    return true;
  }
  
  // Check if role contains "Manager" (case insensitive)
  if (role && role.toLowerCase().includes('manager')) {
    return true;
  }
  
  return false;
}

// Function to check if a role is a manager type (needs country/project filtering)
function isManagerRole(role) {
  const managerRoles = [
    'CEO',
    'Country Manager',
    'Manager',
    'Project Manager',
    'Operations Manager',
    'Site Manager',
    'Field Manager',
    'Regional Manager',
    'Area Manager',
    'Department Manager',
    'Team Manager',
    'Senior Manager',
    'General Manager',
    'Executive Manager'
  ];
  
  // Check exact match first
  if (managerRoles.includes(role)) {
    return true;
  }
  
  // Check if role contains "Manager" (case insensitive)
  if (role && role.toLowerCase().includes('manager')) {
    return true;
  }
  
  return false;
}

// Function to check if a role should always get notifications (no country/project check)
function isAlwaysEligibleRole(role) {
  const alwaysEligibleRoles = [
    'admin',
    'ProAdmin',
    'CEO'
  ];
  
  return alwaysEligibleRoles.includes(role);
}

// Function to get filtered users for request maintenance notifications
async function getFilteredUsersForRequestMaintenance(maintenanceCountry, maintenanceProject) {
  try {
    console.log('🔍 Filtering users for request maintenance notifications...');
    console.log('🔍 Maintenance Country:', maintenanceCountry, '(type:', typeof maintenanceCountry, ')');
    console.log('🔍 Maintenance Project:', maintenanceProject, '(type:', typeof maintenanceProject, ')');
    
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      console.log('❌ No admin users found');
      return [];
    }

    const admins = adminsSnapshot.val();
    const eligibleUsers = [];

    for (const [userId, userData] of Object.entries(admins)) {
      const userRole = userData.role || 'Unknown';
      const userName = userData.name || 'Unknown';
      const userEmail = userData.email || 'No email';
      const userCountries = userData.countries || [];
      const userProjects = userData.projects || [];
      
      console.log(`👤 Checking user: ${userName} (${userRole})`);
      console.log(`   Raw user data:`, JSON.stringify(userData, null, 2));
      console.log(`   Countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
      console.log(`   Projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
      
      // Check if user has active FCM tokens
      const devices = userData.devices || {};
      const activeDevices = Object.entries(devices).filter(([_, device]) => 
        device.token && device.fcmStatus === 'active'
      );
      
      if (activeDevices.length === 0) {
        console.log(`   ❌ No active FCM tokens for ${userName}`);
        continue;
      }
      
      // Check role eligibility
      const isEligibleRole = isRoleEligibleForRequestMaintenance(userRole);
      if (!isEligibleRole) {
        console.log(`   ❌ Role ${userRole} not eligible for request maintenance`);
        continue;
      }
      
      // Check if role should always get notifications (no country/project check)
      if (isAlwaysEligibleRole(userRole)) {
        console.log(`   ✅ ${userRole} ${userName} is always eligible (no country/project check needed)`);
      }
      // For supervisors and all manager types, check country and project matching
      else if (userRole === 'Supervisor' || isManagerRole(userRole)) {
        console.log(`   🔍 Checking country/project match for ${userRole} ${userName}:`);
        console.log(`      User countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
        console.log(`      User projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
        console.log(`      Maintenance country: "${maintenanceCountry}" (type: ${typeof maintenanceCountry})`);
        console.log(`      Maintenance project: "${maintenanceProject}" (type: ${typeof maintenanceProject})`);
        
        // Ensure userCountries and userProjects are arrays
        const userCountriesArray = Array.isArray(userCountries) ? userCountries : [];
        const userProjectsArray = Array.isArray(userProjects) ? userProjects : [];
        
        // For supervisors, check both country AND project match
        // For managers, check both country AND project match
        const hasMatchingCountry = userCountriesArray.some(country => 
          country && country.toString().toLowerCase() === maintenanceCountry.toLowerCase()
        );
        
        const hasMatchingProject = userProjectsArray.some(project => 
          project && project.toString().toLowerCase() === maintenanceProject.toLowerCase()
        );
        
        console.log(`      Country match result: ${hasMatchingCountry}`);
        console.log(`      Project match result: ${hasMatchingProject}`);
        
        if (!hasMatchingCountry || !hasMatchingProject) {
          console.log(`   ❌ ${userRole} ${userName} doesn't match country/project`);
          console.log(`      Has country "${maintenanceCountry}": ${hasMatchingCountry}`);
          console.log(`      Has project "${maintenanceProject}": ${hasMatchingProject}`);
          continue;
        }
        
        console.log(`   ✅ ${userRole} ${userName} matches country and project`);
      } else {
        console.log(`   ❌ ${userRole} ${userName} is not eligible for request maintenance notifications`);
        continue;
      }
      
      // Add user to eligible list
      eligibleUsers.push({
        userId,
        name: userName,
        email: userEmail,
        role: userRole,
        totalDevices: activeDevices.length,
        devices: activeDevices.map(([deviceKey, device]) => ({
          deviceKey,
          fcmToken: device.token,
          deviceId: device.deviceInfo?.deviceId || 'unknown',
          platform: device.deviceInfo?.platform || 'unknown',
          lastTokenUpdate: device.lastTokenUpdate || 'Unknown',
          isCurrentDevice: device.isCurrentDevice || false
        }))
      });
    }

    console.log(`✅ Found ${eligibleUsers.length} eligible users for request maintenance notifications`);
    return eligibleUsers;
    
  } catch (error) {
    console.error('❌ Error filtering users for request maintenance:', error);
    return [];
  }
}

// Function to check if a role is eligible for request maintenance notifications
function isRoleEligibleForRequestMaintenance(role) {
  const eligibleRoles = [
    'admin',
    'ProAdmin',
    'Supervisor',
    'CEO',
    'Country Manager',
    'Manager',
    'Project Manager',
    'Operations Manager',
    'Site Manager',
    'Field Manager',
    'Regional Manager',
    'Area Manager',
    'Department Manager',
    'Team Manager',
    'Senior Manager',
    'General Manager',
    'Executive Manager'
  ];
  
  // Check exact match first
  if (eligibleRoles.includes(role)) {
    return true;
  }
  
  // Check if role contains "Manager" (case insensitive)
  if (role && role.toLowerCase().includes('manager')) {
    return true;
  }
  
  return false;
}

// Note: Removed old inspection report notification functions
// Now using filtered users based on role and country/project matching

// Function to send inspection report notification to a specific admin (EXACT COPY OF NOTICE BOARD PATTERN)
async function sendInspectionReportNotificationToAdmin(adminId, adminData, inspectionReportData, devices) {
  try {
    console.log(`📱 Sending inspection report notification to admin: ${adminData.name || adminId}`);
    
    // Save notification to admin's database
    const saveResult = await saveInspectionReportNotificationToAdmin(adminId, inspectionReportData);
    if (!saveResult.success) {
      console.warn(`⚠️ Failed to save inspection report notification to admin ${adminId} database`);
    }

    // Send FCM notification to all devices
    const deviceResults = [];
    let deviceSuccessCount = 0;
    let deviceFailureCount = 0;
    let invalidTokenCount = 0;

    for (const [deviceKey, deviceData] of Object.entries(devices)) {
      if (deviceData.token && deviceData.fcmStatus === 'active') {
        try {
          const fcmResult = await sendInspectionReportFCMNotification(
            deviceData.token, 
            inspectionReportData, 
            adminData, 
            deviceKey
          );
          
          if (fcmResult.success) {
            deviceSuccessCount++;
            deviceResults.push({ 
              deviceKey, 
              status: 'success', 
              messageId: fcmResult.messageId 
            });
          } else {
            deviceFailureCount++;
            deviceResults.push({ 
              deviceKey, 
              status: 'failed', 
              error: fcmResult.error 
            });
            
            // If token is invalid, mark device as inactive
            if (fcmResult.code === 'messaging/invalid-registration-token' || 
                fcmResult.code === 'messaging/registration-token-not-registered') {
              await markDeviceAsInactive(adminId, deviceKey);
              invalidTokenCount++;
            }
          }
        } catch (error) {
          deviceFailureCount++;
          deviceResults.push({ 
            deviceKey, 
            status: 'error', 
            error: error.message 
          });
        }
      }
    }

    return {
      adminId,
      adminName: adminData.name || 'Unknown',
      status: deviceSuccessCount > 0 ? 'success' : 'failed',
      devices: deviceResults,
      deviceSuccessCount,
      deviceFailureCount,
      invalidTokens: invalidTokenCount
    };

  } catch (error) {
    console.error(`❌ Error sending inspection report notification to admin ${adminId}:`, error);
    return {
      adminId,
      adminName: adminData.name || 'Unknown',
      status: 'error',
      error: error.message
    };
  }
}

// Function to save inspection report notification to admin's database (EXACT COPY OF NOTICE BOARD PATTERN)
async function saveInspectionReportNotificationToAdmin(adminId, inspectionReportData) {
  try {
    const notificationRef = admin.database().ref(`/Globalusers/admins/${adminId}/notifications`);
    const newNotificationRef = notificationRef.push();
    
    const notificationRecord = {
      title: `🔍 New Inspection Report: ${inspectionReportData.requestNumber}`,
      body: inspectionReportData.description.substring(0, 100) + (inspectionReportData.description.length > 100 ? '...' : ''),
      data: {
        type: 'inspection_report_created',
        reportId: inspectionReportData.reportId,
        noticeId: inspectionReportData.noticeId || inspectionReportData.reportId,
        title: inspectionReportData.title,
        description: inspectionReportData.description,
        imageUrl: inspectionReportData.imageUrl || null,
        country: inspectionReportData.country,
        project: inspectionReportData.project,
        inspector: inspectionReportData.inspector,
        requestNumber: inspectionReportData.requestNumber,
        equipmentName: inspectionReportData.equipmentName,
        equipmentStatus: inspectionReportData.equipmentStatus,
        faultyItems: inspectionReportData.faultyItems,
        creatorEmail: inspectionReportData.creatorEmail,
        creatorName: inspectionReportData.creatorName,
        createdAt: inspectionReportData.createdAt,
        source: 'inspection_report',
        priority: 'high'
      },
      timestamp: Date.now(),
      isRead: false,
      createdAt: new Date().toISOString(),
    };
    
    await newNotificationRef.set(notificationRecord);
    console.log(`✅ Inspection report notification saved to admin ${adminId} database`);
    
    return { success: true, notificationId: newNotificationRef.key };
  } catch (error) {
    console.error(`❌ Error saving inspection report notification to admin ${adminId} database:`, error);
    return { success: false, error: error.message };
  }
}

// Function to send inspection report FCM notification (EXACT COPY OF NOTICE BOARD PATTERN)
async function sendInspectionReportFCMNotification(fcmToken, inspectionReportData, adminData, deviceKey) {
  try {
    console.log('🔍 FCM Notification Data:', {
      title: inspectionReportData.title,
      description: inspectionReportData.description,
      requestNumber: inspectionReportData.requestNumber
    });
    
    const message = {
      token: fcmToken,
      notification: {
        title: `🔍 New Inspection Report: ${inspectionReportData.requestNumber}`,
        body: inspectionReportData.description.substring(0, 100) + (inspectionReportData.description.length > 100 ? '...' : ''),
      },
      data: {
        type: 'inspection_report_created',
        reportId: inspectionReportData.reportId || 'unknown',
        noticeId: inspectionReportData.noticeId || inspectionReportData.reportId || 'unknown',
        title: inspectionReportData.title,
        description: inspectionReportData.description,
        imageUrl: inspectionReportData.imageUrl || '',
        country: inspectionReportData.country || '',
        project: inspectionReportData.project || '',
        inspector: inspectionReportData.inspector || '',
        requestNumber: inspectionReportData.requestNumber || '',
        equipmentName: inspectionReportData.equipmentName || '',
        equipmentStatus: inspectionReportData.equipmentStatus || '',
        faultyItems: inspectionReportData.faultyItems || '',
        creatorEmail: inspectionReportData.creatorEmail || '',
        creatorName: inspectionReportData.creatorName || '',
        createdAt: inspectionReportData.createdAt || Date.now().toString(),
        source: 'inspection_report',
        priority: 'high',
        deviceKey: deviceKey,
        adminName: adminData.name || 'Admin',
        adminRole: adminData.role || 'Admin'
      },
      android: {
        notification: {
          channelId: 'inspection_reports',
          importance: 'high',
          priority: 'high',

          icon: 'ic_notification',
          color: '#FF6B35',
          sound: 'default',
          vibrateTimingsMillis: [0, 250, 250, 250],
          lightSettings: {
            color: '#FF6B35',
            lightOnDurationMillis: 1000,
            lightOffDurationMillis: 1000,
          },
        },
        priority: 'high',
      },
      apns: {
        payload: {
          aps: {
            'content-available': 1,
            'mutable-content': 1,
            sound: 'default',
            badge: 1,
            alert: {
              title: `🔍 New Inspection Report: ${inspectionReportData.requestNumber}`,
              body: inspectionReportData.description.substring(0, 100) + (inspectionReportData.description.length > 100 ? '...' : ''),
            },
            category: 'INSPECTION_REPORT',
          },
        },
        headers: {
          'apns-priority': '10',
          'apns-push-type': 'alert',
        },
      },
    };

    console.log(`📱 Sending inspection report FCM notification to device ${deviceKey} with token:`, fcmToken.substring(0, 20) + '...');
    console.log(`📱 FCM Message structure:`, {
      notification: message.notification,
      data: message.data,
      android: message.android?.notification?.channelId,
      apns: message.apns?.payload?.aps?.category
    });
    const response = await admin.messaging().send(message);
    
    console.log(`✅ FCM inspection report notification sent successfully to device ${deviceKey}`);
    return { success: true, messageId: response };
    
  } catch (error) {
    console.error(`❌ Error sending FCM inspection report notification to device ${deviceKey}:`, error);
    
    // Handle specific FCM errors
    if (error.code === 'messaging/invalid-registration-token' || 
        error.code === 'messaging/registration-token-not-registered') {
      console.log(`🔄 FCM token is invalid or expired for device ${deviceKey}, marking as inactive`);
      return { success: false, error: 'Invalid or expired FCM token', code: error.code };
    }
    
    return { success: false, error: error.message, code: error.code };
  }
}

// Function to mark device as inactive when FCM token is invalid (EXACT COPY OF NOTICE BOARD PATTERN)
async function markDeviceAsInactive(adminId, deviceKey) {
  try {
    const deviceRef = admin.database().ref(`/Globalusers/admins/${adminId}/devices/${deviceKey}`);
    await deviceRef.update({
      fcmStatus: 'inactive',
      lastError: 'Invalid FCM token',
      lastErrorAt: new Date().toISOString()
    });
    console.log(`🔄 Marked device ${deviceKey} as inactive for admin ${adminId}`);
  } catch (error) {
    console.error(`❌ Error marking device ${deviceKey} as inactive for admin ${adminId}:`, error);
  }
}

// Test function to send simple push notification
async function sendTestInspectionPushNotification() {
  try {
    console.log('🧪 Starting test push notification...');
    
    // Get all users with FCM tokens
    const allUsersSnapshot = await admin.database()
      .ref('/Globalusers/admins')
      .once('value');
    
    if (!allUsersSnapshot.exists()) {
      console.log('🧪 No admin users found in database');
      return { success: false, error: 'No admin users found' };
    }
    
    const allUsers = allUsersSnapshot.val() || {};
    let totalUsers = 0;
    let usersWithTokens = 0;
    let testNotificationsSent = 0;
    
    // Check each user for FCM tokens
    for (const [userId, userData] of Object.entries(allUsers)) {
      totalUsers++;
      
      let hasToken = false;
      
      // Check new multi-device structure
      if (userData.devices && Object.keys(userData.devices).length > 0) {
        hasToken = true;
        usersWithTokens++;
        console.log(`🧪 User ${userId} has ${Object.keys(userData.devices).length} devices`);
      }
      // Check legacy single token
      else if (userData.NotificationFCMtoken) {
        hasToken = true;
        usersWithTokens++;
        console.log(`🧪 User ${userId} has legacy FCM token`);
      }
      // Check fcmTokens array
      else if (userData.fcmTokens && Array.isArray(userData.fcmTokens) && userData.fcmTokens.length > 0) {
        hasToken = true;
        usersWithTokens++;
        console.log(`🧪 User ${userId} has ${userData.fcmTokens.length} FCM tokens in array`);
      }
      
      // Send test notification to first user with token
      if (hasToken && testNotificationsSent === 0) {
        try {
          let testToken = null;
          
          if (userData.devices && Object.keys(userData.devices).length > 0) {
            const firstDevice = Object.values(userData.devices)[0];
            testToken = firstDevice.token;
          } else if (userData.NotificationFCMtoken) {
            testToken = userData.NotificationFCMtoken;
          } else if (userData.fcmTokens && userData.fcmTokens.length > 0) {
            testToken = userData.fcmTokens[0];
          }
          
          if (testToken) {
            const testMessage = {
              token: testToken,
              notification: {
                title: '🧪 Test Inspection Report',
                body: 'This is a test push notification for inspection reports'
              },
              data: {
                type: 'test_inspection_report',
                message: 'Test notification'
              }
            };
            
            const response = await admin.messaging().send(testMessage);
            console.log(`🧪 Test notification sent successfully to user ${userId}:`, response);
            testNotificationsSent++;
          }
        } catch (error) {
          console.error(`🧪 Test notification failed for user ${userId}:`, error);
        }
      }
    }
    
    console.log(`🧪 Test results: ${totalUsers} total users, ${usersWithTokens} with tokens, ${testNotificationsSent} test notifications sent`);
    
    return {
      success: true,
      totalUsers,
      usersWithTokens,
      testNotificationsSent
    };
    
  } catch (error) {
    console.error('🧪 Test push notification error:', error);
    return { success: false, error: error.message };
  }
}

// Export inspection report email notification functions
exports.inspectionReportEmailNotify = inspectionReportEmailNotify.inspectionReportEmailNotify;
exports.manualInspectionReportEmail = inspectionReportEmailNotify.manualInspectionReportEmail;
exports.testUserFiltering = inspectionReportEmailNotify.testUserFiltering;
exports.debugUserData = inspectionReportEmailNotify.debugUserData;
exports.showAllRoles = inspectionReportEmailNotify.showAllRoles;

// Export DVIR report email notification functions
exports.dvirReportEmailNotify = dvirReportEmailNotify.dvirReportEmailNotify;
exports.manualDvirReportEmail = dvirReportEmailNotify.manualDvirReportEmail;
exports.testDvirUserData = dvirReportEmailNotify.testDvirUserData;

// Export Incident Report notification functions
exports.GlobalIncidentReportNotify = incidentReportNotify.GlobalIncidentReportNotify;

// Export Request Maintenance notification functions
exports.GlobalRequestmaintenanceNotify = requestMaintenanceNotify.GlobalRequestmaintenanceNotify;

// Export Job Card notification functions
exports.GlobalWorkOrderNotify = workOrderNotify.GlobalWorkOrderNotify;

// Export HR Application notification functions
exports.AdminHrAppliApplyNotify = hrApplicationNotify.AdminHrAppliApplyNotify;

// Export HR Application Approval/Rejection notification function
exports.ApproveApplicationSentNotify = approveApplicationSentNotify.ApproveApplicationSentNotify;

// Export New User Creation notification function
exports.NewUserCreateNotify = newUserCreateNotify.NewUserCreateNotify;

// Test HR Application Email Function
exports.testHrApplicationEmail = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { email, applicationType = 'leave_application', fullName = 'Test User', employeeNo = '12345' } = req.body;

    if (!email) {
      res.status(400).json({ success: false, error: 'Email is required' });
      return;
    }

    // Test email content
    const testEmailSubject = `Test HR Application: ${applicationType} - Submission Confirmed`;
    const testEmailHtml = `
      <html>
        <body style="font-family: Arial, sans-serif; line-height: 1.6; color: #333;">
          <div style="max-width: 600px; margin: 0 auto; padding: 20px;">
            <div style="background-color: #015185; color: white; padding: 20px; text-align: center; border-radius: 8px 8px 0 0;">
              <h1 style="margin: 0; font-size: 24px;">Test HR Application Confirmation</h1>
              <p style="margin: 10px 0 0 0; font-size: 16px;">${applicationType}</p>
            </div>
            
            <div style="background-color: #f8f9fa; padding: 20px; border: 1px solid #dee2e6;">
              <h2 style="color: #015185; margin-top: 0;">Test Email - Application Submitted Successfully</h2>
              <p style="background-color: #d4edda; padding: 10px; border-left: 4px solid #28a745; margin-bottom: 20px;">
                <strong>This is a test email:</strong> Your HR application notification system is working correctly.
              </p>
              
              <table style="width: 100%; border-collapse: collapse; margin-bottom: 20px;">
                <tr style="border-bottom: 1px solid #dee2e6;">
                  <td style="padding: 8px; font-weight: bold; width: 40%;">Application Type:</td>
                  <td style="padding: 8px;">${applicationType}</td>
                </tr>
                <tr style="border-bottom: 1px solid #dee2e6;">
                  <td style="padding: 8px; font-weight: bold;">Employee Name:</td>
                  <td style="padding: 8px;">${fullName}</td>
                </tr>
                <tr style="border-bottom: 1px solid #dee2e6;">
                  <td style="padding: 8px; font-weight: bold;">Employee Number:</td>
                  <td style="padding: 8px;">${employeeNo}</td>
                </tr>
                <tr style="border-bottom: 1px solid #dee2e6;">
                  <td style="padding: 8px; font-weight: bold;">Test Email:</td>
                  <td style="padding: 8px;">${email}</td>
                </tr>
                <tr style="border-bottom: 1px solid #dee2e6;">
                  <td style="padding: 8px; font-weight: bold;">Status:</td>
                  <td style="padding: 8px;"><span style="background-color: #28a745; color: white; padding: 2px 8px; border-radius: 4px;">Test Successful</span></td>
                </tr>
                <tr style="border-bottom: 1px solid #dee2e6;">
                  <td style="padding: 8px; font-weight: bold;">Test Date:</td>
                  <td style="padding: 8px;">${new Date().toLocaleDateString('en-GB')} ${new Date().toLocaleTimeString()}</td>
                </tr>
              </table>
              
              <div style="background-color: #e9ecef; padding: 15px; border-radius: 5px; margin-top: 20px;">
                <p style="margin: 0; font-size: 14px; color: #6c757d; text-align: center;">
                  This is a test notification from the Titan Drilling HR Management System.<br>
                  If you received this email, the notification system is working correctly.
                </p>
              </div>
            </div>
          </div>
        </body>
      </html>
    `;

    // Send test email
    console.log(`📧 Sending test HR application email to: ${email}`);
    
    const mailOptions = {
      from: "Titan Drilling HR Management System <support@titandrillingzm.com>",
      to: email,
      subject: testEmailSubject,
      html: testEmailHtml
    };

    const info = await mailTransport.sendMail(mailOptions);
    
    console.log(`✅ Test email sent successfully to ${email} (Message ID: ${info.messageId})`);
    
    res.json({
      success: true,
      message: 'Test HR application email sent successfully',
      email: email,
      messageId: info.messageId,
      timestamp: new Date().toISOString()
    });
    
  } catch (error) {
    console.error('Error sending test HR application email:', error);
    res.status(500).json({ 
      success: false, 
      error: error.message,
      details: 'Failed to send test HR application email'
    });
  }
});

// Debug HR Application Supervisors Function
exports.debugHrSupervisors = functions.https.onRequest(async (req, res) => {
  // Enable CORS
  res.set('Access-Control-Allow-Origin', '*');
  res.set('Access-Control-Allow-Methods', 'GET, POST');
  res.set('Access-Control-Allow-Headers', 'Content-Type');

  if (req.method === 'OPTIONS') {
    res.status(204).send('');
    return;
  }

  if (req.method !== 'POST') {
    res.status(405).send('Method Not Allowed');
    return;
  }

  try {
    const { country = 'Zambia', project = 'Kansanshi' } = req.body;

    console.log(`🔍 Debugging supervisors for Country: ${country}, Project: ${project}`);

    // Get all users
    const usersSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    const users = usersSnapshot.val() || {};
    
    const allUsers = [];
    const supervisors = [];
    const matchingSupervisors = [];

    for (const [userId, userData] of Object.entries(users)) {
      if (!userData) continue;

      const userInfo = {
        userId,
        name: userData.name || 'Unknown',
        email: userData.email || 'No email',
        role: userData.role || 'No role',
        countries: userData.countries || [],
        projects: userData.projects || []
      };

      allUsers.push(userInfo);

      // Check if supervisor
      const userRole = userData.role || '';
      const isSupervisor = userRole.toLowerCase().includes('supervisor') || 
                          userRole.toLowerCase().includes('manager') || 
                          userRole.toLowerCase().includes('lead') ||
                          userRole.toLowerCase().includes('foreman') ||
                          userRole.toLowerCase().includes('superintendent');

      if (isSupervisor) {
        supervisors.push(userInfo);

        // Check if matches country and project
        const userCountries = Array.isArray(userData.countries) ? userData.countries : [];
        const userProjects = Array.isArray(userData.projects) ? userData.projects : [];
        
        const countryMatch = userCountries.includes(country);
        const projectMatch = userProjects.includes(project);
        
        if (countryMatch && projectMatch) {
          matchingSupervisors.push({
            ...userInfo,
            countryMatch,
            projectMatch
          });
        }
      }
    }

    const debugInfo = {
      searchCriteria: { country, project },
      totalUsers: allUsers.length,
      totalSupervisors: supervisors.length,
      matchingSupervisors: matchingSupervisors.length,
      allUsers: allUsers.slice(0, 10), // First 10 users for debugging
      supervisors: supervisors,
      matchingSupervisors: matchingSupervisors
    };

    console.log('🔍 Debug Results:', debugInfo);

    res.json({
      success: true,
      debugInfo: debugInfo
    });
    
  } catch (error) {
    console.error('Error debugging HR supervisors:', error);
    res.status(500).json({ 
      success: false, 
      error: error.message 
    });
  }
});

// Service Schedule Email and Push Notification Function
exports.sendServiceScheduleEmails = functions.database
  .ref('/GlobalServiceScheduled/admins/{userId}/serverschedule/{scheduleId}')
  .onCreate(async (snap, ctx) => {
    const scheduleData = snap.val();
    const { userId, scheduleId } = ctx.params;
    const now = new Date().toISOString();

    if (!scheduleData) {
      console.log('No service schedule data found');
      return null;
    }

    try {
      // Get the user who created the service schedule
      const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
      const userData = userSnapshot.val();
      
      if (!userData) {
        console.log('User data not found for userId:', userId);
        return null;
      }

      const { email: creatorEmail, name: creatorName } = userData;
      const { title, scheduleId: scheduleNumber, country, project, equipment, repeatType, tasks } = scheduleData;
      
      if (!title || !scheduleNumber) {
        console.log('Incomplete service schedule data');
        return null;
      }

      // Debug logging
      console.log('=== SERVICE SCHEDULE EMAIL FUNCTION DEBUG ===');
      console.log('Extracted data:', {
        country,
        project,
        title,
        scheduleNumber,
        equipment: equipment ? equipment.equipmentName : 'N/A',
        repeatType,
        tasks: tasks ? tasks.length : 0
      });

      // Find all supervisors in the same country and project
      const supervisorsSnapshot = await admin.database()
        .ref('/Globalusers/admins')
        .orderByChild('role')
        .equalTo('Supervisor')
        .once('value');
      
      const supervisorsData = supervisorsSnapshot.val() || {};
      const relevantSupervisors = [];

      Object.entries(supervisorsData).forEach(([supervisorId, supervisorData]) => {
        if (supervisorData.countries && supervisorData.projects) {
          const hasMatchingCountry = supervisorData.countries.includes(country);
          const hasMatchingProject = supervisorData.projects.includes(project);
          
          if (hasMatchingCountry && hasMatchingProject) {
            relevantSupervisors.push({
              id: supervisorId,
              email: supervisorData.email,
              name: supervisorData.name || 'Supervisor'
            });
          }
        }
      });

      // Prepare email content
      const emailSubject = `New Service Schedule: ${scheduleNumber || 'N/A'}`;
      
      const emailHtml = `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <div style="background-color: #3481BC; color: white; padding: 20px; text-align: center;">
            <h1 style="margin: 0;">Service Schedule Notification</h1>
          </div>
          
          <div style="padding: 20px; background-color: #f9f9f9;">
            <h2 style="color: #333;">New Service Schedule Created</h2>
            
            <div style="background-color: white; padding: 15px; border-radius: 5px; margin: 15px 0;">
              <h3 style="color: #3481BC; margin-top: 0;">Schedule Details</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Schedule ID:</td>
                  <td style="padding: 8px;">${scheduleNumber || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Title:</td>
                  <td style="padding: 8px;">${title || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Country:</td>
                  <td style="padding: 8px;">${country || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Project:</td>
                  <td style="padding: 8px;">${project || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Equipment:</td>
                  <td style="padding: 8px;">${equipment ? equipment.equipmentName : 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Repeat Type:</td>
                  <td style="padding: 8px;">${repeatType || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Tasks:</td>
                  <td style="padding: 8px;">${tasks ? tasks.length : 0} task(s)</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created By:</td>
                  <td style="padding: 8px;">${creatorName || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created At:</td>
                  <td style="padding: 8px;">${new Date(scheduleData.createdAt || now).toLocaleString()}</td>
                </tr>
              </table>
            </div>
            
            <div style="background-color: #e8f4fd; padding: 15px; border-radius: 5px; border-left: 4px solid #3481BC;">
              <p style="margin: 0; color: #333;">
                <strong>Note:</strong> This is an automated notification. The service schedule has been created and is now available for review.
              </p>
            </div>
          </div>
          
          <div style="background-color: #f0f0f0; padding: 15px; text-align: center; color: #666; font-size: 12px;">
            <p>© ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
          </div>
        </div>
      `;

      // Send email to creator
      if (creatorEmail) {
        try {
          await sendEmail(creatorEmail, emailSubject, emailHtml);
          console.log('Email sent to creator:', creatorEmail);
        } catch (error) {
          console.error('Failed to send email to creator:', error);
        }
      }

      // Send email to relevant supervisors
      const supervisorEmails = [];
      for (const supervisor of relevantSupervisors) {
        if (supervisor.email && supervisor.email !== creatorEmail) {
          try {
            await sendEmail(supervisor.email, emailSubject, emailHtml);
            supervisorEmails.push(supervisor.email);
            console.log('Email sent to supervisor:', supervisor.email);
          } catch (error) {
            console.error('Failed to send email to supervisor:', supervisor.email, error);
          }
        }
      }

      // Send push notifications using the same pattern as Job Cards
      try {
        console.log('🚀 Starting service schedule push notifications...');
        
        // Prepare service schedule data
        const scheduleNotificationData = {
          reportId: scheduleId || 'unknown',
          title: `📅 New Service Schedule: ${scheduleNumber || 'N/A'}`,
          description: `${creatorName || 'Unknown'} created service schedule for ${project || 'Unknown Project'} - ${equipment ? equipment.equipmentName : 'Equipment'}`,
          country: country || 'Unknown',
          project: project || 'Unknown',
          scheduleNumber: scheduleNumber || 'N/A',
          scheduleTitle: title || 'N/A',
          repeatType: repeatType || 'Unknown',
          equipment: equipment ? equipment.equipmentName : 'N/A',
          tasks: tasks ? tasks.length : 0,
          creatorEmail: creatorEmail || 'unknown@example.com',
          creatorName: creatorName || 'Unknown',
          createdAt: now,
          noticeId: scheduleId || 'unknown',
          imageUrl: null
        };
        
        console.log('📅 Service schedule data being sent:', JSON.stringify(scheduleNotificationData, null, 2));
        
        // Get filtered users for service schedule notifications
        const filteredUsers = await getFilteredUsersForServiceSchedule(country, project);
        console.log('📋 Filtered users for service schedule:', filteredUsers);

        if (filteredUsers && filteredUsers.length > 0) {
          console.log(`📊 Found ${filteredUsers.length} eligible users for service schedule notifications`);
          
          let totalSuccess = 0;
          let totalFailed = 0;
          let totalInvalidTokens = 0;
          const notificationResults = [];

          // Send notification to each eligible user
          for (const user of filteredUsers) {
            try {
              console.log(`📤 Sending service schedule notification to user: ${user.name} (${user.email})`);
              
              const notificationResponse = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendNotificationToAllUserDevices', {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                  userId: user.userId,
                  title: `📅 ${scheduleNotificationData.title}`,
                  body: scheduleNotificationData.description.length > 100 ? scheduleNotificationData.description.substring(0, 100) + '...' : scheduleNotificationData.description,
                  data: {
                    type: 'service_schedule_created',
                    reportId: scheduleNotificationData.reportId,
                    noticeId: scheduleNotificationData.noticeId,
                    title: scheduleNotificationData.title,
                    description: scheduleNotificationData.description,
                    imageUrl: scheduleNotificationData.imageUrl,
                    createdAt: scheduleNotificationData.createdAt,
                    country: scheduleNotificationData.country,
                    project: scheduleNotificationData.project,
                    scheduleNumber: scheduleNotificationData.scheduleNumber,
                    scheduleTitle: scheduleNotificationData.scheduleTitle,
                    repeatType: scheduleNotificationData.repeatType,
                    equipment: scheduleNotificationData.equipment,
                    tasks: scheduleNotificationData.tasks,
                    creatorEmail: scheduleNotificationData.creatorEmail,
                    creatorName: scheduleNotificationData.creatorName,
                    action: 'view_service_schedule',
                  },
                }),
              });

              const notificationResult = await notificationResponse.json();
              console.log(`📋 Service schedule notification result for ${user.name}:`, notificationResult);

              if (notificationResult.success) {
                totalSuccess += notificationResult.successfulDevices || 0;
                totalFailed += notificationResult.failedDevices || 0;
                totalInvalidTokens += notificationResult.invalidTokens || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'success',
                  devices: notificationResult.successfulDevices || 0,
                  failed: notificationResult.failedDevices || 0,
                  invalidTokens: notificationResult.invalidTokens || 0,
                });
              } else {
                totalFailed += user.totalDevices || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'failed',
                  error: notificationResult.error,
                });
              }
            } catch (userError) {
              console.error(`❌ Error sending service schedule notification to user ${user.name}:`, userError);
              totalFailed += user.totalDevices || 0;
              notificationResults.push({
                user: user.name,
                email: user.email,
                status: 'error',
                error: userError.message,
              });
            }
          }

          // Log comprehensive results
          console.log('📊 Service schedule push notification results:', {
            totalUsers: filteredUsers.length,
            totalSuccess,
            totalFailed,
            totalInvalidTokens,
            results: notificationResults,
          });

          console.log(`✅ Service schedule push notifications completed: ${totalSuccess} successful, ${totalFailed} failed, ${totalInvalidTokens} invalid tokens`);
          
        } else {
          console.log('⚠️ No eligible users found for service schedule push notifications');
        }

      } catch (error) {
        console.error('❌ Error sending service schedule push notifications:', error);
      }

      // Store email records in Globalservicescheduleemails
      const emailRecord = {
        scheduleId,
        userId,
        creatorEmail,
        creatorName,
        country,
        project,
        title,
        scheduleNumber,
        repeatType,
        equipment: equipment ? equipment.equipmentName : 'N/A',
        tasks: tasks ? tasks.length : 0,
        supervisorEmails,
        emailSent: true,
        emailSentAt: now,
        emailContent: emailHtml,
        pushNotificationsSent: true,
        pushNotificationsSentAt: now,
        status: 'sent'
      };

      await admin.database().ref(`/Globalservicescheduleemails/${scheduleId}`).set(emailRecord);

      console.log(`Service schedule notifications sent successfully. Emails: Creator: ${creatorEmail}, Supervisors: ${supervisorEmails.join(', ')}. Push notifications sent to all users.`);
      
      return {
        success: true,
        creatorEmail,
        supervisorEmails,
        totalEmailsSent: 1 + supervisorEmails.length,
        pushNotificationsSent: true,
        message: 'Emails and push notifications sent successfully'
      };

    } catch (error) {
      console.error('Error sending service schedule emails:', error);
      
      // Store error record
      const errorRecord = {
        scheduleId,
        userId,
        error: error.message,
        errorAt: now,
        status: 'error'
      };
      
      await admin.database().ref(`/Globalservicescheduleemails/${scheduleId}`).set(errorRecord);
      
      return {
        success: false,
        error: error.message
      };
    }
  });

// Function to get filtered users for service schedule notifications
async function getFilteredUsersForServiceSchedule(scheduleCountry, scheduleProject) {
  try {
    console.log('🔍 Filtering users for service schedule notifications...');
    console.log('🔍 Schedule Country:', scheduleCountry, '(type:', typeof scheduleCountry, ')');
    console.log('🔍 Schedule Project:', scheduleProject, '(type:', typeof scheduleProject, ')');
    
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      console.log('❌ No admin users found');
      return [];
    }

    const admins = adminsSnapshot.val();
    const eligibleUsers = [];

    for (const [userId, userData] of Object.entries(admins)) {
      const userRole = userData.role || 'Unknown';
      const userName = userData.name || 'Unknown';
      const userEmail = userData.email || 'No email';
      const userCountries = userData.countries || [];
      const userProjects = userData.projects || [];
      
      console.log(`👤 Checking user: ${userName} (${userRole})`);
      console.log(`   Countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
      console.log(`   Projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
      
      // Check if user has active FCM tokens
      const devices = userData.devices || {};
      const activeDevices = Object.entries(devices).filter(([_, device]) => 
        device.token && device.fcmStatus === 'active'
      );
      
      if (activeDevices.length === 0) {
        console.log(`   ❌ No active FCM tokens for ${userName}`);
        continue;
      }
      
      // Check role eligibility for service schedules
      const isEligibleRole = isRoleEligibleForServiceSchedule(userRole);
      if (!isEligibleRole) {
        console.log(`   ❌ Role ${userRole} not eligible for service schedules`);
        continue;
      }
      
      // Check if role should always get notifications (no country/project check)
      if (isAlwaysEligibleRole(userRole)) {
        console.log(`   ✅ ${userRole} ${userName} is always eligible (no country/project check needed)`);
      }
      // For supervisors and all manager types, check country and project matching
      else if (userRole === 'Supervisor' || isManagerRole(userRole)) {
        console.log(`   🔍 Checking country/project match for ${userRole} ${userName}:`);
        console.log(`      User countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
        console.log(`      User projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
        console.log(`      Schedule country: "${scheduleCountry}" (type: ${typeof scheduleCountry})`);
        console.log(`      Schedule project: "${scheduleProject}" (type: ${typeof scheduleProject})`);
        
        // Ensure userCountries and userProjects are arrays
        const userCountriesArray = Array.isArray(userCountries) ? userCountries : [];
        const userProjectsArray = Array.isArray(userProjects) ? userProjects : [];
        
        // For supervisors, check both country AND project match
        // For managers, check both country AND project match
        const hasMatchingCountry = userCountriesArray.some(country => 
          country && country.toString().toLowerCase() === scheduleCountry.toLowerCase()
        );
        
        const hasMatchingProject = userProjectsArray.some(project => 
          project && project.toString().toLowerCase() === scheduleProject.toLowerCase()
        );
        
        console.log(`      Country match result: ${hasMatchingCountry}`);
        console.log(`      Project match result: ${hasMatchingProject}`);
        
        if (!hasMatchingCountry || !hasMatchingProject) {
          console.log(`   ❌ ${userRole} ${userName} doesn't match country/project`);
          console.log(`      Has country "${scheduleCountry}": ${hasMatchingCountry}`);
          console.log(`      Has project "${scheduleProject}": ${hasMatchingProject}`);
          continue;
        }
        
        console.log(`   ✅ ${userRole} ${userName} matches country and project`);
      } else {
        console.log(`   ❌ ${userRole} ${userName} is not eligible for service schedule notifications`);
        continue;
      }
      
      // Add user to eligible list
      eligibleUsers.push({
        userId,
        name: userName,
        email: userEmail,
        role: userRole,
        totalDevices: activeDevices.length,
        devices: activeDevices.map(([deviceKey, device]) => ({
          deviceKey,
          fcmToken: device.token,
          deviceId: device.deviceInfo?.deviceId || 'unknown',
          platform: device.deviceInfo?.platform || 'unknown',
          lastTokenUpdate: device.lastTokenUpdate || 'Unknown',
          isCurrentDevice: device.isCurrentDevice || false
        }))
      });
    }

    console.log(`✅ Found ${eligibleUsers.length} eligible users for service schedule notifications`);
    return eligibleUsers;
    
  } catch (error) {
    console.error('❌ Error filtering users for service schedule:', error);
    return [];
  }
}

// Function to check if a role is eligible for service schedule notifications
function isRoleEligibleForServiceSchedule(role) {
  const eligibleRoles = [
    'admin',
    'ProAdmin',
    'Supervisor',
    'CEO',
    'Country Manager',
    'Manager',
    'Project Manager',
    'Operations Manager',
    'Site Manager',
    'Field Manager',
    'Regional Manager',
    'Area Manager',
    'Department Manager',
    'Team Manager',
    'Senior Manager',
    'General Manager',
    'Executive Manager'
  ];
  
  // Check exact match first
  if (eligibleRoles.includes(role)) {
    return true;
  }
  
  // Check if role contains "Manager" (case insensitive)
  if (role && role.toLowerCase().includes('manager')) {
    return true;
  }
  
  return false;
}

// Job Card Email and Push Notification Function (similar to incident reports)
exports.sendWorkOrderEmails = functions.database
  .ref('/GlobalWorkOrders/admins/{userId}/{workOrderId}')
  .onCreate(async (snap, ctx) => {
    const workOrderData = snap.val();
    const { userId, workOrderId } = ctx.params;
    const now = new Date().toISOString();

    if (!workOrderData) {
      console.log('No Job Card data found');
      return null;
    }

    try {
      // Get the user who created the Job Card
      const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
      const userData = userSnapshot.val();
      
      if (!userData) {
        console.log('User data not found for userId:', userId);
        return null;
      }

      const { email: creatorEmail, name: creatorName } = userData;
      const { title, requestNumber, country, project, priority, equipment, workOrderType, createdAt } = workOrderData;
      
      if (!title || !requestNumber) {
        console.log('Incomplete Job Card data');
        return null;
      }

      // Debug logging
      console.log('=== Job Card EMAIL FUNCTION DEBUG ===');
      console.log('Extracted data:', {
        country,
        project,
        title,
        requestNumber,
        priority,
        equipment: equipment ? equipment.equipmentName : 'N/A',
        workOrderType
      });

      // Find all supervisors in the same country and project
      const supervisorsSnapshot = await admin.database()
        .ref('/Globalusers/admins')
        .orderByChild('role')
        .equalTo('Supervisor')
        .once('value');
      
      const supervisorsData = supervisorsSnapshot.val() || {};
      const relevantSupervisors = [];

      Object.entries(supervisorsData).forEach(([supervisorId, supervisorData]) => {
        if (supervisorData.countries && supervisorData.projects) {
          const hasMatchingCountry = supervisorData.countries.includes(country);
          const hasMatchingProject = supervisorData.projects.includes(project);
          
          if (hasMatchingCountry && hasMatchingProject) {
            relevantSupervisors.push({
              id: supervisorId,
              email: supervisorData.email,
              name: supervisorData.name || 'Supervisor'
            });
          }
        }
      });

      // Prepare email content
      const emailSubject = `New Job Card: ${requestNumber || 'N/A'}`;
      
      const emailHtml = `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <div style="background-color: #4CAF50; color: white; padding: 20px; text-align: center;">
            <h1 style="margin: 0;">Job Card Notification</h1>
          </div>
          
          <div style="padding: 20px; background-color: #f9f9f9;">
            <h2 style="color: #333;">New Job Card Created</h2>
            
            <div style="background-color: white; padding: 15px; border-radius: 5px; margin: 15px 0;">
              <h3 style="color: #4CAF50; margin-top: 0;">Job Card Details</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Request Number:</td>
                  <td style="padding: 8px;">${requestNumber || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Title:</td>
                  <td style="padding: 8px;">${title || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Country:</td>
                  <td style="padding: 8px;">${country || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Project:</td>
                  <td style="padding: 8px;">${project || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Priority:</td>
                  <td style="padding: 8px;">${priority || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Equipment:</td>
                  <td style="padding: 8px;">${equipment ? equipment.equipmentName : 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Type:</td>
                  <td style="padding: 8px;">${workOrderType || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created By:</td>
                  <td style="padding: 8px;">${creatorName || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created At:</td>
                  <td style="padding: 8px;">${new Date(createdAt || now).toLocaleString()}</td>
                </tr>
              </table>
            </div>
            
            <div style="background-color: #e8f4fd; padding: 15px; border-radius: 5px; border-left: 4px solid #4CAF50;">
              <p style="margin: 0; color: #333;">
                <strong>Note:</strong> This is an automated notification. The job card has been created and is now available for review.
              </p>
            </div>
          </div>
          
          <div style="background-color: #f0f0f0; padding: 15px; text-align: center; color: #666; font-size: 12px;">
            <p>© ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
          </div>
        </div>
      `;

      // Send email to creator
      if (creatorEmail) {
        try {
          await sendEmail(creatorEmail, emailSubject, emailHtml);
          console.log('Email sent to creator:', creatorEmail);
        } catch (error) {
          console.error('Failed to send email to creator:', error);
        }
      }

      // Send email to relevant supervisors
      const supervisorEmails = [];
      for (const supervisor of relevantSupervisors) {
        if (supervisor.email && supervisor.email !== creatorEmail) {
          try {
            await sendEmail(supervisor.email, emailSubject, emailHtml);
            supervisorEmails.push(supervisor.email);
            console.log('Email sent to supervisor:', supervisor.email);
          } catch (error) {
            console.error('Failed to send email to supervisor:', supervisor.email, error);
          }
        }
      }

      // Send push notifications using the same pattern as incident reports
      try {
        console.log('🚀 Starting Job Card push notifications...');
        
        // Prepare Job Card data
        const workOrderNotificationData = {
          reportId: workOrderId || 'unknown',
          title: `🔧 New Job Card: ${requestNumber || 'N/A'}`,
          description: `${creatorName || 'Unknown'} created job card for ${project || 'Unknown Project'} - ${equipment ? equipment.equipmentName : 'Equipment'}`,
          country: country || 'Unknown',
          project: project || 'Unknown',
          requestNumber: requestNumber || 'N/A',
          workOrderTitle: title || 'N/A',
          priority: priority || 'Unknown',
          equipment: equipment ? equipment.equipmentName : 'N/A',
          workOrderType: workOrderType || 'N/A',
          creatorEmail: creatorEmail || 'unknown@example.com',
          creatorName: creatorName || 'Unknown',
          createdAt: now,
          noticeId: workOrderId || 'unknown',
          imageUrl: null
        };
        
        console.log('🔧 Job Card data being sent:', JSON.stringify(workOrderNotificationData, null, 2));
        
        // Get filtered users for Job Card notifications
        const filteredUsers = await getFilteredUsersForWorkOrder(country, project);
        console.log('📋 Filtered users for Job Card:', filteredUsers);

        if (filteredUsers && filteredUsers.length > 0) {
          console.log(`📊 Found ${filteredUsers.length} eligible users for Job Card notifications`);
          
          let totalSuccess = 0;
          let totalFailed = 0;
          let totalInvalidTokens = 0;
          const notificationResults = [];

          // Send notification to each eligible user
          for (const user of filteredUsers) {
            try {
              console.log(`📤 Sending Job Card notification to user: ${user.name} (${user.email})`);
              
              const notificationResponse = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendNotificationToAllUserDevices', {
                method: 'POST',
                headers: {
                  'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                  userId: user.userId,
                  title: `🔧 ${workOrderNotificationData.title}`,
                  body: workOrderNotificationData.description.length > 100 ? workOrderNotificationData.description.substring(0, 100) + '...' : workOrderNotificationData.description,
                  data: {
                    type: 'work_order_created',
                    reportId: workOrderNotificationData.reportId,
                    noticeId: workOrderNotificationData.noticeId,
                    title: workOrderNotificationData.title,
                    description: workOrderNotificationData.description,
                    imageUrl: workOrderNotificationData.imageUrl,
                    createdAt: workOrderNotificationData.createdAt,
                    country: workOrderNotificationData.country,
                    project: workOrderNotificationData.project,
                    requestNumber: workOrderNotificationData.requestNumber,
                    workOrderTitle: workOrderNotificationData.workOrderTitle,
                    priority: workOrderNotificationData.priority,
                    equipment: workOrderNotificationData.equipment,
                    workOrderType: workOrderNotificationData.workOrderType,
                    creatorEmail: workOrderNotificationData.creatorEmail,
                    creatorName: workOrderNotificationData.creatorName,
                    action: 'view_work_order',
                  },
                }),
              });

              const notificationResult = await notificationResponse.json();
              console.log(`📋 Job Card notification result for ${user.name}:`, notificationResult);

              if (notificationResult.success) {
                totalSuccess += notificationResult.successfulDevices || 0;
                totalFailed += notificationResult.failedDevices || 0;
                totalInvalidTokens += notificationResult.invalidTokens || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'success',
                  devices: notificationResult.successfulDevices || 0,
                  failed: notificationResult.failedDevices || 0,
                  invalidTokens: notificationResult.invalidTokens || 0,
                });
              } else {
                totalFailed += user.totalDevices || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'failed',
                  error: notificationResult.error,
                });
              }
            } catch (userError) {
              console.error(`❌ Error sending Job Card notification to user ${user.name}:`, userError);
              totalFailed += user.totalDevices || 0;
              notificationResults.push({
                user: user.name,
                email: user.email,
                status: 'error',
                error: userError.message,
              });
            }
          }

          // Log comprehensive results
          console.log('📊 Job Card push notification results:', {
            totalUsers: filteredUsers.length,
            totalSuccess,
            totalFailed,
            totalInvalidTokens,
            results: notificationResults,
          });

          console.log(`✅ Job Card push notifications completed: ${totalSuccess} successful, ${totalFailed} failed, ${totalInvalidTokens} invalid tokens`);
          
        } else {
          console.log('⚠️ No eligible users found for Job Card push notifications');
        }

      } catch (error) {
        console.error('❌ Error sending Job Card push notifications:', error);
      }

      // Store email records in Globalworkorderemails
      const emailRecord = {
        workOrderId,
        userId,
        creatorEmail,
        creatorName,
        country,
        project,
        title,
        requestNumber,
        priority,
        equipment: equipment ? equipment.equipmentName : 'N/A',
        workOrderType,
        supervisorEmails,
        emailSent: true,
        emailSentAt: now,
        emailContent: emailHtml,
        pushNotificationsSent: true,
        pushNotificationsSentAt: now,
        status: 'sent'
      };

      await admin.database().ref(`/Globalworkorderemails/${workOrderId}`).set(emailRecord);

      console.log(`Job Card notifications sent successfully. Emails: Creator: ${creatorEmail}, Supervisors: ${supervisorEmails.join(', ')}. Push notifications sent to all users.`);
      
      return {
        success: true,
        creatorEmail,
        supervisorEmails,
        totalEmailsSent: 1 + supervisorEmails.length,
        pushNotificationsSent: true,
        message: 'Emails and push notifications sent successfully'
      };

    } catch (error) {
      console.error('Error sending Job Card emails:', error);
      
      // Store error record
      const errorRecord = {
        workOrderId,
        userId,
        error: error.message,
        errorAt: now,
        status: 'error'
      };
      
      await admin.database().ref(`/Globalworkorderemails/${workOrderId}`).set(errorRecord);
      
      return {
        success: false,
        error: error.message
      };
    }
  });

// Function to get filtered users for Job Card notifications
async function getFilteredUsersForWorkOrder(workOrderCountry, workOrderProject) {
  try {
    console.log('🔍 Filtering users for Job Card notifications...');
    console.log('🔍 Job Card Country:', workOrderCountry, '(type:', typeof workOrderCountry, ')');
    console.log('🔍 Job Card Project:', workOrderProject, '(type:', typeof workOrderProject, ')');
    
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      console.log('❌ No admin users found');
      return [];
    }

    const admins = adminsSnapshot.val();
    const eligibleUsers = [];

    for (const [userId, userData] of Object.entries(admins)) {
      const userRole = userData.role || 'Unknown';
      const userName = userData.name || 'Unknown';
      const userEmail = userData.email || 'No email';
      const userCountries = userData.countries || [];
      const userProjects = userData.projects || [];
      
      console.log(`👤 Checking user: ${userName} (${userRole})`);
      console.log(`   Countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
      console.log(`   Projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
      
      // Check if user has active FCM tokens
      const devices = userData.devices || {};
      const activeDevices = Object.entries(devices).filter(([_, device]) => 
        device.token && device.fcmStatus === 'active'
      );
      
      if (activeDevices.length === 0) {
        console.log(`   ❌ No active FCM tokens for ${userName}`);
        continue;
      }
      
      // Check role eligibility for Job Cards
      const isEligibleRole = isRoleEligibleForWorkOrder(userRole);
      if (!isEligibleRole) {
        console.log(`   ❌ Role ${userRole} not eligible for Job Cards`);
        continue;
      }
      
      // Check if role should always get notifications (no country/project check)
      if (isAlwaysEligibleRole(userRole)) {
        console.log(`   ✅ ${userRole} ${userName} is always eligible (no country/project check needed)`);
      }
      // For supervisors and all manager types, check country and project matching
      else if (userRole === 'Supervisor' || isManagerRole(userRole)) {
        console.log(`   🔍 Checking country/project match for ${userRole} ${userName}:`);
        console.log(`      User countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
        console.log(`      User projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
        console.log(`      Job Card country: "${workOrderCountry}" (type: ${typeof workOrderCountry})`);
        console.log(`      Job Card project: "${workOrderProject}" (type: ${typeof workOrderProject})`);
        
        // Ensure userCountries and userProjects are arrays
        const userCountriesArray = Array.isArray(userCountries) ? userCountries : [];
        const userProjectsArray = Array.isArray(userProjects) ? userProjects : [];
        
        // For supervisors, check both country AND project match
        // For managers, check both country AND project match
        const hasMatchingCountry = userCountriesArray.some(country => 
          country && country.toString().toLowerCase() === workOrderCountry.toLowerCase()
        );
        
        const hasMatchingProject = userProjectsArray.some(project => 
          project && project.toString().toLowerCase() === workOrderProject.toLowerCase()
        );
        
        console.log(`      Country match result: ${hasMatchingCountry}`);
        console.log(`      Project match result: ${hasMatchingProject}`);
        
        if (!hasMatchingCountry || !hasMatchingProject) {
          console.log(`   ❌ ${userRole} ${userName} doesn't match country/project`);
          console.log(`      Has country "${workOrderCountry}": ${hasMatchingCountry}`);
          console.log(`      Has project "${workOrderProject}": ${hasMatchingProject}`);
          continue;
        }
        
        console.log(`   ✅ ${userRole} ${userName} matches country and project`);
      } else {
        console.log(`   ❌ ${userRole} ${userName} is not eligible for Job Card notifications`);
        continue;
      }
      
      // Add user to eligible list
      eligibleUsers.push({
        userId,
        name: userName,
        email: userEmail,
        role: userRole,
        totalDevices: activeDevices.length,
        devices: activeDevices.map(([deviceKey, device]) => ({
          deviceKey,
          fcmToken: device.token,
          deviceId: device.deviceInfo?.deviceId || 'unknown',
          platform: device.deviceInfo?.platform || 'unknown',
          lastTokenUpdate: device.lastTokenUpdate || 'Unknown',
          isCurrentDevice: device.isCurrentDevice || false
        }))
      });
    }

    console.log(`✅ Found ${eligibleUsers.length} eligible users for Job Card notifications`);
    return eligibleUsers;
    
  } catch (error) {
    console.error('❌ Error filtering users for Job Card:', error);
    return [];
  }
}

// Function to check if a role is eligible for Job Card notifications
function isRoleEligibleForWorkOrder(role) {
  const eligibleRoles = [
    'admin',
    'ProAdmin',
    'Supervisor',
    'CEO',
    'Country Manager',
    'Manager',
    'Project Manager',
    'Operations Manager',
    'Site Manager',
    'Field Manager',
    'Regional Manager',
    'Area Manager',
    'Department Manager',
    'Team Manager',
    'Senior Manager',
    'General Manager',
    'Executive Manager'
  ];
  
  // Check exact match first
  if (eligibleRoles.includes(role)) {
    return true;
  }
  
  // Check if role contains "Manager" (case insensitive)
  if (role && role.toLowerCase().includes('manager')) {
    return true;
  }
  
  return false;
}

// Request Maintenance Email Notification Function (similar to incident reports)
exports.sendRequestMaintenanceEmails = functions
  .runWith({ memory: '512MB', timeoutSeconds: 120 })
  .database
  .ref('/GlobalRequestMaintenancereport/admins/{userId}/{reportId}')
  .onCreate(async (snap, ctx) => {
    const reportData = snap.val();
    const { userId, reportId } = ctx.params;
    const now = new Date().toISOString();

    if (!reportData) {
      console.log('No maintenance request data found');
      return null;
    }

    try {
      // Get the user who created the maintenance request
      const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
      const userData = userSnapshot.val();
      
      if (!userData) {
        console.log('User data not found for userId:', userId);
        return null;
      }

      const { email: creatorEmail, name: creatorName } = userData;
      const { workOrderTitle, requestNumber, country, project, priority, assignTo, equipment, tasks, createdAt } = reportData;
      
      if (!workOrderTitle || !requestNumber) {
        console.log('Incomplete maintenance request data');
        return null;
      }

      // Debug logging
      console.log('=== REQUEST MAINTENANCE EMAIL FUNCTION DEBUG ===');
      console.log('Extracted data:', {
        country,
        project,
        workOrderTitle,
        requestNumber,
        priority,
        equipment: equipment ? equipment.equipmentName : 'N/A',
        assignTo: assignTo ? assignTo.map(a => a.fullName || a.name).join(', ') : 'Unassigned',
        tasks: tasks ? tasks.length : 0
      });

      // Find all supervisors in the same country and project
      const supervisorsSnapshot = await admin.database()
        .ref('/Globalusers/admins')
        .orderByChild('role')
        .equalTo('Supervisor')
        .once('value');
      
      const supervisorsData = supervisorsSnapshot.val() || {};
      const relevantSupervisors = [];

      Object.entries(supervisorsData).forEach(([supervisorId, supervisorData]) => {
        if (supervisorData.countries && supervisorData.projects) {
          const hasMatchingCountry = supervisorData.countries.includes(country);
          const hasMatchingProject = supervisorData.projects.includes(project);
          
          if (hasMatchingCountry && hasMatchingProject) {
            relevantSupervisors.push({
              id: supervisorId,
              email: supervisorData.email,
              name: supervisorData.name || 'Supervisor'
            });
          }
        }
      });

      // Prepare email content
      const emailSubject = `New Request Maintenance: ${requestNumber || 'N/A'}`;
      
      const emailHtml = `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <div style="background-color: #015185; color: white; padding: 20px; text-align: center;">
            <h1 style="margin: 0;">Request Maintenance Notification</h1>
          </div>
          
          <div style="padding: 20px; background-color: #f9f9f9;">
            <h2 style="color: #333;">New Request Maintenance Created</h2>
            
            <div style="background-color: white; padding: 15px; border-radius: 5px; margin: 15px 0;">
              <h3 style="color: #015185; margin-top: 0;">Request Details</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Request Number:</td>
                  <td style="padding: 8px;">${requestNumber || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Title:</td>
                  <td style="padding: 8px;">${workOrderTitle || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Country:</td>
                  <td style="padding: 8px;">${country || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Project:</td>
                  <td style="padding: 8px;">${project || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Priority:</td>
                  <td style="padding: 8px;">${priority || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Equipment:</td>
                  <td style="padding: 8px;">${equipment ? equipment.equipmentName : 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Assigned To:</td>
                  <td style="padding: 8px;">${assignTo ? assignTo.map(a => a.fullName || a.name).join(', ') : 'Unassigned'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Tasks:</td>
                  <td style="padding: 8px;">${tasks ? tasks.length : 0} task(s)</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created By:</td>
                  <td style="padding: 8px;">${creatorName || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created At:</td>
                  <td style="padding: 8px;">${new Date(createdAt || now).toLocaleString()}</td>
                </tr>
              </table>
            </div>
            
            <div style="background-color: #e8f4fd; padding: 15px; border-radius: 5px; border-left: 4px solid #015185;">
              <p style="margin: 0; color: #333;">
                <strong>Note:</strong> This is an automated notification. The request maintenance has been created and is now available for review.
              </p>
            </div>
          </div>
          
          <div style="background-color: #f0f0f0; padding: 15px; text-align: center; color: #666; font-size: 12px;">
            <p>© ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
          </div>
        </div>
      `;

      // Send email to creator
      if (creatorEmail) {
        try {
          await sendEmail(creatorEmail, emailSubject, emailHtml);
          console.log('Email sent to creator:', creatorEmail);
        } catch (error) {
          console.error('Failed to send email to creator:', error);
        }
      }

      // Send email to relevant supervisors
      const supervisorEmails = [];
      for (const supervisor of relevantSupervisors) {
        if (supervisor.email && supervisor.email !== creatorEmail) {
          try {
            await sendEmail(supervisor.email, emailSubject, emailHtml);
            supervisorEmails.push(supervisor.email);
            console.log('Email sent to supervisor:', supervisor.email);
          } catch (error) {
            console.error('Failed to send email to supervisor:', supervisor.email, error);
          }
        }
      }

      // Send push notifications using the same pattern as incident reports
      try {
        console.log('🚀 Starting request maintenance push notifications...');
        
        // Prepare request maintenance data
        const requestMaintenanceData = {
          reportId: reportId || 'unknown',
          title: `🔧 New Request Maintenance: ${requestNumber || 'N/A'}`,
          description: `${creatorName || 'Unknown'} created request maintenance for ${project || 'Unknown Project'} - ${equipment ? equipment.equipmentName : 'Equipment'}`,
          country: country || 'Unknown',
          project: project || 'Unknown',
          workOrderTitle: workOrderTitle || 'N/A',
          requestNumber: requestNumber || 'N/A',
          priority: priority || 'Unknown',
          equipment: equipment ? equipment.equipmentName : 'N/A',
          assignTo: assignTo ? assignTo.map(a => a.fullName || a.name).join(', ') : 'Unassigned',
          tasks: tasks ? tasks.length : 0,
          creatorEmail: creatorEmail || 'unknown@example.com',
          creatorName: creatorName || 'Unknown',
          createdAt: now,
          noticeId: reportId || 'unknown',
          imageUrl: null
        };
        
        console.log('🔧 Request maintenance data being sent:', JSON.stringify(requestMaintenanceData, null, 2));
        
        // Get filtered users for request maintenance notifications
        const filteredUsers = await getFilteredUsersForRequestMaintenance(country, project);
        console.log('📋 Filtered users for request maintenance:', filteredUsers);

        if (filteredUsers && filteredUsers.length > 0) {
          console.log(`📊 Found ${filteredUsers.length} eligible users for request maintenance notifications`);
          
          let totalSuccess = 0;
          let totalFailed = 0;
          let totalInvalidTokens = 0;
          const notificationResults = [];

          // Send notification to each eligible user (SAME AS INSPECTION REPORTS)
          for (const user of filteredUsers) {
            try {
              console.log(`📤 Sending request maintenance notification to user: ${user.name} (${user.email})`);
              
              // Save notification to user's database for bell icon (SAME AS INSPECTION REPORTS)
              const notificationRef = admin.database().ref(`/Globalusers/admins/${user.userId}/notifications`).push();
              await notificationRef.set({
                title: requestMaintenanceData.title,
                body: requestMaintenanceData.description.substring(0, 100) + (requestMaintenanceData.description.length > 100 ? '...' : ''),
                type: 'request_maintenance_created',
                requestNumber: requestMaintenanceData.requestNumber,
                country: requestMaintenanceData.country,
                project: requestMaintenanceData.project,
                workOrderTitle: requestMaintenanceData.workOrderTitle,
                data: {
                  type: 'request_maintenance_created',
                  reportId: requestMaintenanceData.reportId,
                  noticeId: requestMaintenanceData.noticeId,
                  title: requestMaintenanceData.title,
                  description: requestMaintenanceData.description,
                  imageUrl: requestMaintenanceData.imageUrl,
                  createdAt: requestMaintenanceData.createdAt,
                  country: requestMaintenanceData.country,
                  project: requestMaintenanceData.project,
                  workOrderTitle: requestMaintenanceData.workOrderTitle,
                  requestNumber: requestMaintenanceData.requestNumber,
                  priority: requestMaintenanceData.priority,
                  equipment: requestMaintenanceData.equipment,
                  assignTo: requestMaintenanceData.assignTo,
                  tasks: requestMaintenanceData.tasks.toString(),
                  creatorEmail: requestMaintenanceData.creatorEmail,
                  creatorName: requestMaintenanceData.creatorName,
                  action: 'view_request_maintenance',
                },
                timestamp: Date.now(),
                isRead: false,
                createdAt: now,
              });
              console.log(`✅ Notification saved to database for user ${user.userId}`);
              
              // Send FCM notification to all user's devices (DIRECT - SAME AS INSPECTION REPORTS!)
              let deviceSuccessCount = 0;
              let deviceFailedCount = 0;
              let invalidTokenCount = 0;
              
              for (const device of user.devices) {
                try {
                  await admin.messaging().send({
                    token: device.fcmToken,
                    notification: {
                      title: requestMaintenanceData.title,
                      body: requestMaintenanceData.description.substring(0, 100) + (requestMaintenanceData.description.length > 100 ? '...' : ''),
                    },
                    data: {
                      type: 'request_maintenance_created',
                      reportId: requestMaintenanceData.reportId,
                      noticeId: requestMaintenanceData.noticeId,
                      title: requestMaintenanceData.title,
                      description: requestMaintenanceData.description,
                      imageUrl: requestMaintenanceData.imageUrl || '',
                      createdAt: requestMaintenanceData.createdAt,
                      country: requestMaintenanceData.country,
                      project: requestMaintenanceData.project,
                      workOrderTitle: requestMaintenanceData.workOrderTitle,
                      requestNumber: requestMaintenanceData.requestNumber,
                      priority: requestMaintenanceData.priority,
                      equipment: requestMaintenanceData.equipment,
                      assignTo: requestMaintenanceData.assignTo,
                      tasks: requestMaintenanceData.tasks.toString(),
                      creatorEmail: requestMaintenanceData.creatorEmail,
                      creatorName: requestMaintenanceData.creatorName,
                      action: 'view_request_maintenance',
                    },
                    android: {
                      priority: 'high',
                      notification: {
                        channelId: 'request_maintenance',
                        priority: 'high',
                        sound: 'default',
                      }
                    },
                    apns: {
                      payload: {
                        aps: {
                          sound: 'default',
                          badge: 1,
                        }
                      }
                    }
                  });
                  deviceSuccessCount++;
                  console.log(`✅ Notification sent to device ${device.deviceKey}`);
                } catch (deviceError) {
                  if (deviceError.code === 'messaging/invalid-registration-token' || 
                      deviceError.code === 'messaging/registration-token-not-registered') {
                    invalidTokenCount++;
                    console.log(`🔄 Invalid token for device ${device.deviceKey}`);
                    // Mark device as inactive
                    await admin.database().ref(`/Globalusers/admins/${user.userId}/devices/${device.deviceKey}/fcmStatus`).set('inactive');
                  } else {
                    deviceFailedCount++;
                    console.error(`❌ Failed to send to device ${device.deviceKey}:`, deviceError);
                  }
                }
              }
              
              if (deviceSuccessCount > 0) {
                totalSuccess += deviceSuccessCount;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'success',
                  devices: deviceSuccessCount,
                  failed: deviceFailedCount,
                  invalidTokens: invalidTokenCount,
                });
              } else {
                totalFailed += user.totalDevices || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'failed',
                  error: 'All devices failed',
                });
              }
            } catch (userError) {
              console.error(`❌ Error sending request maintenance notification to user ${user.name}:`, userError);
              totalFailed += user.totalDevices || 0;
              notificationResults.push({
                user: user.name,
                email: user.email,
                status: 'error',
                error: userError.message,
              });
            }
          }

          // Log comprehensive results
          console.log('📊 Request maintenance push notification results:', {
            totalUsers: filteredUsers.length,
            totalSuccess,
            totalFailed,
            totalInvalidTokens,
            results: notificationResults,
          });

          console.log(`✅ Request maintenance push notifications completed: ${totalSuccess} successful, ${totalFailed} failed, ${totalInvalidTokens} invalid tokens`);
          
        } else {
          console.log('⚠️ No eligible users found for request maintenance push notifications');
        }

      } catch (error) {
        console.error('❌ Error sending request maintenance push notifications:', error);
      }

      // Store email records in Globalrequestmaintenanceemails
      const emailRecord = {
        reportId,
        userId,
        creatorEmail,
        creatorName,
        country,
        project,
        workOrderTitle,
        requestNumber,
        priority,
        equipment: equipment ? equipment.equipmentName : 'N/A',
        assignTo: assignTo ? assignTo.map(a => a.fullName || a.name).join(', ') : 'Unassigned',
        tasks: tasks ? tasks.length : 0,
        supervisorEmails,
        emailSent: true,
        emailSentAt: now,
        emailContent: emailHtml,
        pushNotificationsSent: true,
        pushNotificationsSentAt: now,
        status: 'sent'
      };

      await admin.database().ref(`/Globalrequestmaintenanceemails/${reportId}`).set(emailRecord);

      console.log(`Request maintenance notifications sent successfully. Emails: Creator: ${creatorEmail}, Supervisors: ${supervisorEmails.join(', ')}. Push notifications sent to all users.`);
      
      return {
        success: true,
        creatorEmail,
        supervisorEmails,
        totalEmailsSent: 1 + supervisorEmails.length,
        pushNotificationsSent: true,
        message: 'Emails and push notifications sent successfully'
      };

    } catch (error) {
      console.error('Error sending request maintenance emails:', error);
      
      // Store error record
      const errorRecord = {
        reportId,
        userId,
        error: error.message,
        errorAt: now,
        status: 'error'
      };
      
      await admin.database().ref(`/Globalrequestmaintenanceemails/${reportId}`).set(errorRecord);
      
      return {
        success: false,
        error: error.message
      };
    }
  });

// Incident Report Email Notification Function (similar to inspection reports)
exports.sendIncidentReportEmails = functions
  .runWith({ memory: '512MB', timeoutSeconds: 120 })
  .database
  .ref('/GlobalIncidentReports/admins/{userId}/{reportId}')
  .onCreate(async (snap, ctx) => {
    const reportData = snap.val();
    const { userId, reportId } = ctx.params;
    const now = new Date().toISOString();

    if (!reportData) {
      console.log('No incident report data found');
      return null;
    }

    try {
      // Get the user who created the incident report
      const userSnapshot = await admin.database().ref(`/Globalusers/admins/${userId}`).once('value');
      const userData = userSnapshot.val();
      
      if (!userData) {
        console.log('User data not found for userId:', userId);
        return null;
      }

      const { email: creatorEmail, name: creatorName } = userData;
      const { step1, step2 } = reportData;
      
      if (!step1) {
        console.log('Incomplete incident report data');
        return null;
      }

      const { country, project, incidentNumber, incidentDate, severity, incidentArea, vehicleName } = step1;
      const { pdfDownloadUrl } = step2 || {};
      
      // Debug logging
      console.log('=== INCIDENT REPORT EMAIL FUNCTION DEBUG ===');
      console.log('Report data structure:', { step1: !!step1, step2: !!step2 });
      console.log('Extracted data:', {
        country,
        project,
        incidentNumber,
        incidentDate,
        severity,
        incidentArea,
        vehicleName
      });

      // Get all users and filter for email recipients (SAME AS INSPECTION REPORTS)
      const allAdminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
      const allAdmins = allAdminsSnapshot.val() || {};
      
      const proAdmins = [];
      const relevantSupervisorsAndManagers = [];
      
      // Filter users for email recipients
      Object.entries(allAdmins).forEach(([adminId, adminData]) => {
        if (!adminData || !adminData.email) return;
        
        const userRole = adminData.role || '';
        const userCountries = adminData.countries || [];
        const userProjects = adminData.projects || [];
        
        // ProAdmins and CEO always get emails (no country/project check)
        if (userRole === 'ProAdmin' || userRole === 'CEO') {
          proAdmins.push({
            id: adminId,
            email: adminData.email,
            name: adminData.name || userRole
          });
        }
        // Supervisors and ALL Manager types - check country AND project match
        else if (userRole === 'Supervisor' || userRole.toLowerCase().includes('manager')) {
          const hasMatchingCountry = Array.isArray(userCountries) && userCountries.includes(country);
          const hasMatchingProject = Array.isArray(userProjects) && userProjects.includes(project);
          
          if (hasMatchingCountry && hasMatchingProject) {
            relevantSupervisorsAndManagers.push({
              id: adminId,
              email: adminData.email,
              name: adminData.name || userRole,
              role: userRole
            });
          }
        }
      });

      console.log(`📧 Email recipients breakdown:`);
      console.log(`   ProAdmins/CEO (always): ${proAdmins.length}`);
      console.log(`   Supervisors/Managers (matching ${country}+${project}): ${relevantSupervisorsAndManagers.length}`);

      // Prepare email content
      const emailSubject = `New Incident Report: ${incidentNumber || 'N/A'}`;
      
      const emailHtml = `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <div style="background-color: #F44336; color: white; padding: 20px; text-align: center;">
            <h1 style="margin: 0;">Incident Report Notification</h1>
          </div>
          
          <div style="padding: 20px; background-color: #f9f9f9;">
            <h2 style="color: #333;">New Incident Report Created</h2>
            
            <div style="background-color: white; padding: 15px; border-radius: 5px; margin: 15px 0;">
              <h3 style="color: #F44336; margin-top: 0;">Report Details</h3>
              <table style="width: 100%; border-collapse: collapse;">
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Incident Number:</td>
                  <td style="padding: 8px;">${incidentNumber || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Country:</td>
                  <td style="padding: 8px;">${country || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Project:</td>
                  <td style="padding: 8px;">${project || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Incident Date:</td>
                  <td style="padding: 8px;">${incidentDate ? new Date(incidentDate).toLocaleDateString() : 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Severity:</td>
                  <td style="padding: 8px;">${severity || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Incident Area:</td>
                  <td style="padding: 8px;">${incidentArea || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Vehicle:</td>
                  <td style="padding: 8px;">${vehicleName || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created By:</td>
                  <td style="padding: 8px;">${creatorName || 'N/A'}</td>
                </tr>
                <tr>
                  <td style="padding: 8px; font-weight: bold; color: #666;">Created At:</td>
                  <td style="padding: 8px;">${new Date(now).toLocaleString()}</td>
                </tr>
              </table>
            </div>
            
            ${pdfDownloadUrl ? `
            <div style="text-align: center; margin: 20px 0;">
              <a href="${pdfDownloadUrl}" style="background-color: #F44336; color: white; padding: 12px 24px; text-decoration: none; border-radius: 5px; display: inline-block;">
                View Incident Report
              </a>
            </div>
            ` : ''}
            
            <div style="background-color: #e8f4fd; padding: 15px; border-radius: 5px; border-left: 4px solid #F44336;">
              <p style="margin: 0; color: #333;">
                <strong>Note:</strong> This is an automated notification. The incident report has been created and is now available for review.
              </p>
            </div>
          </div>
          
          <div style="background-color: #f0f0f0; padding: 15px; text-align: center; color: #666; font-size: 12px;">
            <p>© ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
          </div>
        </div>
      `;

      // Send email to creator
      if (creatorEmail) {
        try {
          await sendEmail(creatorEmail, emailSubject, emailHtml);
          console.log('✅ Email sent to creator:', creatorEmail);
        } catch (error) {
          console.error('❌ Failed to send email to creator:', error);
        }
      }

      // Send email to ALL ProAdmins and CEOs (no country/project filtering)
      const proAdminEmails = [];
      for (const proAdmin of proAdmins) {
        if (proAdmin.email && proAdmin.email !== creatorEmail) {
          try {
            await sendEmail(proAdmin.email, emailSubject, emailHtml);
            proAdminEmails.push(proAdmin.email);
            console.log('✅ Email sent to ProAdmin/CEO:', proAdmin.email);
          } catch (error) {
            console.error('❌ Failed to send email to ProAdmin/CEO:', proAdmin.email, error);
          }
        }
      }

      // Send email to relevant supervisors and managers (with country+project match)
      const supervisorEmails = [];
      for (const user of relevantSupervisorsAndManagers) {
        if (user.email && user.email !== creatorEmail) {
          try {
            await sendEmail(user.email, emailSubject, emailHtml);
            supervisorEmails.push(user.email);
            console.log(`✅ Email sent to ${user.role}:`, user.email);
          } catch (error) {
            console.error(`❌ Failed to send email to ${user.role}:`, user.email, error);
          }
        }
      }

      // Send push notifications using the same pattern as inspection reports
      try {
        console.log('🚀 Starting incident report push notifications...');
        
        // Prepare incident report data
        const incidentReportData = {
          reportId: reportId || 'unknown',
          title: `🚨 New Incident Report: ${incidentNumber || 'N/A'}`,
          description: `${creatorName || 'Unknown'} created incident report for ${project || 'Unknown Project'} - ${vehicleName || incidentArea || 'Area'}`,
          country: country || 'Unknown',
          project: project || 'Unknown',
          incidentNumber: incidentNumber || 'N/A',
          incidentDate: incidentDate || 'N/A',
          severity: severity || 'Unknown',
          incidentArea: incidentArea || 'N/A',
          vehicleName: vehicleName || 'N/A',
          creatorEmail: creatorEmail || 'unknown@example.com',
          creatorName: creatorName || 'Unknown',
          createdAt: now,
          noticeId: reportId || 'unknown',
          imageUrl: null
        };
        
        console.log('🚨 Incident report data being sent:', JSON.stringify(incidentReportData, null, 2));
        
        // Get filtered users for incident report notifications
        const filteredUsers = await getFilteredUsersForIncidentReport(country, project);
        console.log('📋 Filtered users for incident report:', filteredUsers);

        if (filteredUsers && filteredUsers.length > 0) {
          console.log(`📊 Found ${filteredUsers.length} eligible users for incident report notifications`);
          
          let totalSuccess = 0;
          let totalFailed = 0;
          let totalInvalidTokens = 0;
          const notificationResults = [];

          // Send notification to each eligible user
          for (const user of filteredUsers) {
            try {
              console.log(`📤 Sending incident notification to user: ${user.name} (${user.email})`);
              
              // Save notification to user's database for bell icon (SAME AS INSPECTION REPORTS)
              const notificationRef = admin.database().ref(`/Globalusers/admins/${user.userId}/notifications`).push();
              await notificationRef.set({
                title: incidentReportData.title,
                body: incidentReportData.description.substring(0, 100) + (incidentReportData.description.length > 100 ? '...' : ''),
                type: 'incident_report',
                incidentNumber: incidentReportData.incidentNumber,
                country: incidentReportData.country,
                project: incidentReportData.project,
                adminName: incidentReportData.creatorName,
                data: {
                  type: 'incident_report_created',
                  reportId: incidentReportData.reportId,
                  noticeId: incidentReportData.noticeId,
                  title: incidentReportData.title,
                  description: incidentReportData.description,
                  imageUrl: incidentReportData.imageUrl,
                  createdAt: incidentReportData.createdAt,
                  country: incidentReportData.country,
                  project: incidentReportData.project,
                  incidentNumber: incidentReportData.incidentNumber,
                  incidentDate: incidentReportData.incidentDate,
                  severity: incidentReportData.severity,
                  incidentArea: incidentReportData.incidentArea,
                  vehicleName: incidentReportData.vehicleName,
                  creatorEmail: incidentReportData.creatorEmail,
                  creatorName: incidentReportData.creatorName,
                  action: 'view_incident_report',
                },
                timestamp: Date.now(),
                isRead: false,
                createdAt: now,
              });
              console.log(`✅ Notification saved to database for user ${user.userId}`);
              
              // Send FCM notification to all user's devices (DIRECT - NO FETCH!)
              let deviceSuccessCount = 0;
              let deviceFailedCount = 0;
              let invalidTokenCount = 0;
              
              for (const device of user.devices) {
                try {
                  await admin.messaging().send({
                    token: device.fcmToken,
                    notification: {
                      title: incidentReportData.title,
                      body: incidentReportData.description.substring(0, 100) + (incidentReportData.description.length > 100 ? '...' : ''),
                    },
                  data: {
                    type: 'incident_report_created',
                    reportId: incidentReportData.reportId,
                    noticeId: incidentReportData.noticeId,
                    title: incidentReportData.title,
                    description: incidentReportData.description,
                    imageUrl: incidentReportData.imageUrl || '',
                    createdAt: incidentReportData.createdAt,
                    country: incidentReportData.country,
                    project: incidentReportData.project,
                    incidentNumber: incidentReportData.incidentNumber,
                    incidentDate: incidentReportData.incidentDate,
                    severity: incidentReportData.severity,
                    incidentArea: incidentReportData.incidentArea,
                    vehicleName: incidentReportData.vehicleName,
                    creatorEmail: incidentReportData.creatorEmail,
                    creatorName: incidentReportData.creatorName,
                    action: 'view_incident_report',
                  },
                    android: {
                      priority: 'high',
                      notification: {
                        channelId: 'incident_reports',
                        priority: 'high',
                        sound: 'default',
                      }
                    },
                    apns: {
                      payload: {
                        aps: {
                          sound: 'default',
                          badge: 1,
                        }
                      }
                    }
                  });
                  deviceSuccessCount++;
                  console.log(`✅ Notification sent to device ${device.deviceKey}`);
                } catch (deviceError) {
                  if (deviceError.code === 'messaging/invalid-registration-token' || 
                      deviceError.code === 'messaging/registration-token-not-registered') {
                    invalidTokenCount++;
                    console.log(`🔄 Invalid token for device ${device.deviceKey}`);
                    // Mark device as inactive
                    await admin.database().ref(`/Globalusers/admins/${user.userId}/devices/${device.deviceKey}/fcmStatus`).set('inactive');
                  } else {
                    deviceFailedCount++;
                    console.error(`❌ Failed to send to device ${device.deviceKey}:`, deviceError);
                  }
                }
              }
              
              if (deviceSuccessCount > 0) {
                totalSuccess += deviceSuccessCount;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'success',
                  devices: deviceSuccessCount,
                  failed: deviceFailedCount,
                  invalidTokens: invalidTokenCount,
                });
              } else {
                totalFailed += user.totalDevices || 0;
                notificationResults.push({
                  user: user.name,
                  email: user.email,
                  status: 'failed',
                  error: 'All devices failed',
                });
              }
            } catch (userError) {
              console.error(`❌ Error sending incident notification to user ${user.name}:`, userError);
              totalFailed += user.totalDevices || 0;
              notificationResults.push({
                user: user.name,
                email: user.email,
                status: 'error',
                error: userError.message,
              });
            }
          }

          // Log comprehensive results
          console.log('📊 Incident report push notification results:', {
            totalUsers: filteredUsers.length,
            totalSuccess,
            totalFailed,
            totalInvalidTokens,
            results: notificationResults,
          });

          console.log(`✅ Incident report push notifications completed: ${totalSuccess} successful, ${totalFailed} failed, ${totalInvalidTokens} invalid tokens`);
          
        } else {
          console.log('⚠️ No eligible users found for incident report push notifications');
        }

      } catch (error) {
        console.error('❌ Error sending incident report push notifications:', error);
      }

      // Store email records in Globalincidentemails (includes both ProAdmin and Supervisor emails)
      const emailRecord = {
        reportId,
        userId,
        creatorEmail,
        creatorName,
        country,
        project,
        incidentNumber: incidentNumber,
        incidentDate,
        severity,
        incidentArea,
        vehicleName,
        proAdminEmails,
        supervisorEmails,
        emailSent: true,
        emailSentAt: now,
        emailContent: emailHtml,
        pushNotificationsSent: true,
        pushNotificationsSentAt: now,
        status: 'sent'
      };

      await admin.database().ref(`/Globalincidentemails/${reportId}`).set(emailRecord);

      console.log(`✅ Incident report notifications sent successfully!`);
      console.log(`   Creator: ${creatorEmail}`);
      console.log(`   ProAdmins: ${proAdminEmails.length > 0 ? proAdminEmails.join(', ') : 'None'}`);
      console.log(`   Supervisors/Managers: ${supervisorEmails.length > 0 ? supervisorEmails.join(', ') : 'None'}`);
      
      return {
        success: true,
        creatorEmail,
        proAdminEmails,
        supervisorEmails,
        totalEmailsSent: 1 + proAdminEmails.length + supervisorEmails.length,
        pushNotificationsSent: true,
        message: 'Emails and push notifications sent successfully'
      };

    } catch (error) {
      console.error('Error sending incident report emails:', error);
      
      // Store error record
      const errorRecord = {
        reportId,
        userId,
        error: error.message,
        errorAt: now,
        status: 'error'
      };
      
      await admin.database().ref(`/Globalincidentemails/${reportId}`).set(errorRecord);
      
      return {
        success: false,
        error: error.message
      };
    }
  });

// Function to get filtered users for incident report notifications
async function getFilteredUsersForIncidentReport(incidentCountry, incidentProject) {
  try {
    console.log('🔍 Filtering users for incident report notifications...');
    console.log('🔍 Incident Country:', incidentCountry, '(type:', typeof incidentCountry, ')');
    console.log('🔍 Incident Project:', incidentProject, '(type:', typeof incidentProject, ')');
    
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      console.log('❌ No admin users found');
      return [];
    }

    const admins = adminsSnapshot.val();
    const eligibleUsers = [];

    for (const [userId, userData] of Object.entries(admins)) {
      const userRole = userData.role || 'Unknown';
      const userName = userData.name || 'Unknown';
      const userEmail = userData.email || 'No email';
      const userCountries = userData.countries || [];
      const userProjects = userData.projects || [];
      
      console.log(`👤 Checking user: ${userName} (${userRole})`);
      console.log(`   Raw user data:`, JSON.stringify(userData, null, 2));
      console.log(`   Countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
      console.log(`   Projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
      
      // Check if user has active FCM tokens
      const devices = userData.devices || {};
      const activeDevices = Object.entries(devices).filter(([_, device]) => 
        device.token && device.fcmStatus === 'active'
      );
      
      if (activeDevices.length === 0) {
        console.log(`   ❌ No active FCM tokens for ${userName}`);
        continue;
      }
      
      // Check role eligibility
      const isEligibleRole = isRoleEligibleForIncidentReport(userRole);
      if (!isEligibleRole) {
        console.log(`   ❌ Role ${userRole} not eligible for incident reports`);
        continue;
      }
      
      // Check if role should always get notifications (no country/project check)
      if (isAlwaysEligibleRole(userRole)) {
        console.log(`   ✅ ${userRole} ${userName} is always eligible (no country/project check needed)`);
      }
      // For supervisors and all manager types, check country and project matching
      else if (userRole === 'Supervisor' || isManagerRole(userRole)) {
        console.log(`   🔍 Checking country/project match for ${userRole} ${userName}:`);
        console.log(`      User countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
        console.log(`      User projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
        console.log(`      Incident country: "${incidentCountry}" (type: ${typeof incidentCountry})`);
        console.log(`      Incident project: "${incidentProject}" (type: ${typeof incidentProject})`);
        
        // Ensure userCountries and userProjects are arrays
        const userCountriesArray = Array.isArray(userCountries) ? userCountries : [];
        const userProjectsArray = Array.isArray(userProjects) ? userProjects : [];
        
        // For supervisors, check both country AND project match
        // For managers, check both country AND project match
        const hasMatchingCountry = userCountriesArray.some(country => 
          country && country.toString().toLowerCase() === incidentCountry.toLowerCase()
        );
        
        const hasMatchingProject = userProjectsArray.some(project => 
          project && project.toString().toLowerCase() === incidentProject.toLowerCase()
        );
        
        console.log(`      Country match result: ${hasMatchingCountry}`);
        console.log(`      Project match result: ${hasMatchingProject}`);
        
        if (!hasMatchingCountry || !hasMatchingProject) {
          console.log(`   ❌ ${userRole} ${userName} doesn't match country/project`);
          console.log(`      Has country "${incidentCountry}": ${hasMatchingCountry}`);
          console.log(`      Has project "${incidentProject}": ${hasMatchingProject}`);
          continue;
        }
        
        console.log(`   ✅ ${userRole} ${userName} matches country and project`);
      } else {
        console.log(`   ❌ ${userRole} ${userName} is not eligible for incident report notifications`);
        continue;
      }
      
      // Add user to eligible list
      eligibleUsers.push({
        userId,
        name: userName,
        email: userEmail,
        role: userRole,
        totalDevices: activeDevices.length,
        devices: activeDevices.map(([deviceKey, device]) => ({
          deviceKey,
          fcmToken: device.token,
          deviceId: device.deviceInfo?.deviceId || 'unknown',
          platform: device.deviceInfo?.platform || 'unknown',
          lastTokenUpdate: device.lastTokenUpdate || 'Unknown',
          isCurrentDevice: device.isCurrentDevice || false
        }))
      });
    }

    console.log(`✅ Found ${eligibleUsers.length} eligible users for incident report notifications`);
    return eligibleUsers;
    
  } catch (error) {
    console.error('❌ Error filtering users for incident report:', error);
    return [];
  }
}

// Function to check if a role is eligible for incident report notifications
function isRoleEligibleForIncidentReport(role) {
  const eligibleRoles = [
    'admin',
    'ProAdmin',
    'Supervisor',
    'CEO',
    'Country Manager',
    'Manager',
    'Project Manager',
    'Operations Manager',
    'Site Manager',
    'Field Manager',
    'Regional Manager',
    'Area Manager',
    'Department Manager',
    'Team Manager',
    'Senior Manager',
    'General Manager',
    'Executive Manager'
  ];
  
  // Check exact match first
  if (eligibleRoles.includes(role)) {
    return true;
  }
  
  // Check if role contains "Manager" (case insensitive)
  if (role && role.toLowerCase().includes('manager')) {
    return true;
  }
  
  return false;
}

// Function to check if a role is a manager type (needs country/project filtering)
function isManagerRole(role) {
  const managerRoles = [
    'CEO',
    'Country Manager',
    'Manager',
    'Project Manager',
    'Operations Manager',
    'Site Manager',
    'Field Manager',
    'Regional Manager',
    'Area Manager',
    'Department Manager',
    'Team Manager',
    'Senior Manager',
    'General Manager',
    'Executive Manager'
  ];
  
  // Check exact match first
  if (managerRoles.includes(role)) {
    return true;
  }
  
  // Check if role contains "Manager" (case insensitive)
  if (role && role.toLowerCase().includes('manager')) {
    return true;
  }
  
  return false;
}

// Function to check if a role should always get notifications (no country/project check)
function isAlwaysEligibleRole(role) {
  const alwaysEligibleRoles = [
    'admin',
    'ProAdmin',
    'CEO'
  ];
  
  return alwaysEligibleRoles.includes(role);
}

// Function to get filtered users for request maintenance notifications
async function getFilteredUsersForRequestMaintenance(maintenanceCountry, maintenanceProject) {
  try {
    console.log('🔍 Filtering users for request maintenance notifications...');
    console.log('🔍 Maintenance Country:', maintenanceCountry, '(type:', typeof maintenanceCountry, ')');
    console.log('🔍 Maintenance Project:', maintenanceProject, '(type:', typeof maintenanceProject, ')');
    
    const adminsSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    
    if (!adminsSnapshot.exists()) {
      console.log('❌ No admin users found');
      return [];
    }

    const admins = adminsSnapshot.val();
    const eligibleUsers = [];

    for (const [userId, userData] of Object.entries(admins)) {
      const userRole = userData.role || 'Unknown';
      const userName = userData.name || 'Unknown';
      const userEmail = userData.email || 'No email';
      const userCountries = userData.countries || [];
      const userProjects = userData.projects || [];
      
      console.log(`👤 Checking user: ${userName} (${userRole})`);
      console.log(`   Raw user data:`, JSON.stringify(userData, null, 2));
      console.log(`   Countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
      console.log(`   Projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
      
      // Check if user has active FCM tokens
      const devices = userData.devices || {};
      const activeDevices = Object.entries(devices).filter(([_, device]) => 
        device.token && device.fcmStatus === 'active'
      );
      
      if (activeDevices.length === 0) {
        console.log(`   ❌ No active FCM tokens for ${userName}`);
        continue;
      }
      
      // Check role eligibility
      const isEligibleRole = isRoleEligibleForRequestMaintenance(userRole);
      if (!isEligibleRole) {
        console.log(`   ❌ Role ${userRole} not eligible for request maintenance`);
        continue;
      }
      
      // Check if role should always get notifications (no country/project check)
      if (isAlwaysEligibleRole(userRole)) {
        console.log(`   ✅ ${userRole} ${userName} is always eligible (no country/project check needed)`);
      }
      // For supervisors and all manager types, check country and project matching
      else if (userRole === 'Supervisor' || isManagerRole(userRole)) {
        console.log(`   🔍 Checking country/project match for ${userRole} ${userName}:`);
        console.log(`      User countries: ${JSON.stringify(userCountries)} (type: ${typeof userCountries})`);
        console.log(`      User projects: ${JSON.stringify(userProjects)} (type: ${typeof userProjects})`);
        console.log(`      Maintenance country: "${maintenanceCountry}" (type: ${typeof maintenanceCountry})`);
        console.log(`      Maintenance project: "${maintenanceProject}" (type: ${typeof maintenanceProject})`);
        
        // Ensure userCountries and userProjects are arrays
        const userCountriesArray = Array.isArray(userCountries) ? userCountries : [];
        const userProjectsArray = Array.isArray(userProjects) ? userProjects : [];
        
        // For supervisors, check both country AND project match
        // For managers, check both country AND project match
        const hasMatchingCountry = userCountriesArray.some(country => 
          country && country.toString().toLowerCase() === maintenanceCountry.toLowerCase()
        );
        
        const hasMatchingProject = userProjectsArray.some(project => 
          project && project.toString().toLowerCase() === maintenanceProject.toLowerCase()
        );
        
        console.log(`      Country match result: ${hasMatchingCountry}`);
        console.log(`      Project match result: ${hasMatchingProject}`);
        
        if (!hasMatchingCountry || !hasMatchingProject) {
          console.log(`   ❌ ${userRole} ${userName} doesn't match country/project`);
          console.log(`      Has country "${maintenanceCountry}": ${hasMatchingCountry}`);
          console.log(`      Has project "${maintenanceProject}": ${hasMatchingProject}`);
          continue;
        }
        
        console.log(`   ✅ ${userRole} ${userName} matches country and project`);
      } else {
        console.log(`   ❌ ${userRole} ${userName} is not eligible for request maintenance notifications`);
        continue;
      }
      
      // Add user to eligible list
      eligibleUsers.push({
        userId,
        name: userName,
        email: userEmail,
        role: userRole,
        totalDevices: activeDevices.length,
        devices: activeDevices.map(([deviceKey, device]) => ({
          deviceKey,
          fcmToken: device.token,
          deviceId: device.deviceInfo?.deviceId || 'unknown',
          platform: device.deviceInfo?.platform || 'unknown',
          lastTokenUpdate: device.lastTokenUpdate || 'Unknown',
          isCurrentDevice: device.isCurrentDevice || false
        }))
      });
    }

    console.log(`✅ Found ${eligibleUsers.length} eligible users for request maintenance notifications`);
    return eligibleUsers;
    
  } catch (error) {
    console.error('❌ Error filtering users for request maintenance:', error);
    return [];
  }
}

// Function to check if a role is eligible for request maintenance notifications
function isRoleEligibleForRequestMaintenance(role) {
  const eligibleRoles = [
    'admin',
    'ProAdmin',
    'Supervisor',
    'CEO',
    'Country Manager',
    'Manager',
    'Project Manager',
    'Operations Manager',
    'Site Manager',
    'Field Manager',
    'Regional Manager',
    'Area Manager',
    'Department Manager',
    'Team Manager',
    'Senior Manager',
    'General Manager',
    'Executive Manager'
  ];
  
  // Check exact match first
  if (eligibleRoles.includes(role)) {
    return true;
  }
  
  // Check if role contains "Manager" (case insensitive)
  if (role && role.toLowerCase().includes('manager')) {
    return true;
  }
  
  return false;
}


// Export the filter delete unused user data functions
exports.filterDeleteUnusedUserData = filterDeleteUnusedUserData.filterDeleteUnusedUserData;
exports.cleanupUnusedAdminUsers = filterDeleteUnusedUserData.cleanupUnusedAdminUsers;
// Export HSE notification function
const hseFuncNotify = require('./HSE_func_notify');
exports.sendHseReportNotify = hseFuncNotify.sendHseReportNotify;
