# Inspection Report Email Notification System

This Firebase function automatically sends email notifications when inspection reports are created, with PDF attachments and proper user permission filtering.

## 📋 Overview

The `InspectionReportEmailNotify` function:
- **Triggers automatically** when a new inspection report is created in Firebase
- **Sends emails with PDF attachments** to relevant users based on their permissions
- **Filters recipients** by country, project, and module permissions
- **Includes supervisors and proadmins** in the email notifications
- **Logs all email activity** for tracking and debugging

## 🔧 Setup Requirements

### 1. Email Configuration

Configure Firebase environment variables:

```bash
# Set up email credentials
firebase functions:config:set email.user="your-email@gmail.com"
firebase functions:config:set email.pass="your-app-password"
```

**Note:** Use an App Password for Gmail, not your regular password.

### 2. Firebase Database Structure

The function expects this data structure:

```
/Globalusers/admins/{userId}/
├── email: "user@example.com"
├── name: "User Name"
├── role: "Admin|Supervisor|ProAdmin|User"
├── countries: ["Zambia", "DRC"]
├── projects: ["Kobold", "Kansanshi"]
└── modules: [
    {
      module: "inspectionReport",
      permissions: ["read", "write"]
    }
  ]

/GlobalInspectionReport/admins/{adminId}/{reportId}/
├── country: "Zambia"
├── project: "Kobold"
├── inspector: "John Doe"
├── selectedEquipment: {...}
├── checklistTitle: "Daily Checklist"
├── requestNumber: "ZAM-KOB-IR-001"
├── createdAt: "2024-01-01T00:00:00.000Z"
├── coordinates: "lat, lng"
└── gpsAddress: "Location address"
```

### 3. Dependencies

Ensure these packages are in `functions/package.json`:

```json
{
  "dependencies": {
    "firebase-admin": "^12.6.0",
    "firebase-functions": "^6.0.1",
    "nodemailer": "^6.9.14"
  }
}
```

## 🚀 How It Works

### Automatic Trigger

1. **Report Creation**: When a new inspection report is saved to Firebase
2. **Function Trigger**: Firebase function automatically executes
3. **User Filtering**: Finds users with matching country/project/module permissions
4. **Email Generation**: Creates HTML email with report details and PDF link
5. **Email Sending**: Sends to users and supervisors
6. **Logging**: Records all email activity in Firebase

### Permission-Based Filtering

Users receive emails if they have:
- ✅ Access to the report's **country**
- ✅ Access to the report's **project** 
- ✅ **Inspection Report module** permissions

Supervisors and ProAdmins receive emails if they have:
- ✅ Access to the report's **country**
- ✅ Access to the report's **project**
- ✅ Role: "Supervisor", "ProAdmin", or "Admin"

### **Recipient Categories**

The system sends emails to **4 categories** of users:

1. **Users with Permissions**: All users who have `inspectionReport` module access for the specific country+project
2. **Supervisors & ProAdmins**: Users with supervisor or proadmin roles for the specific country+project  
3. **Admin & Supervisor Users**: Users with admin or supervisor roles who have matching country+project permissions AND inspection report module access
4. **Report Creator**: The user who created the inspection report (always receives a copy)

**Total Recipients**: Sum of all 4 categories

## 📧 Email Template

The email includes:

### Header
- Company branding
- Report request number

### Report Summary Table
- Request Number
- Country & Project
- Inspector Name
- Equipment Details
- Checklist Used
- Date & Location

### Inspection Results
- Visual summary with pass/fail counts
- Total items inspected

### PDF Download
- Direct link to full PDF report (7-day expiry)

### Footer
- Company copyright
- Automated notification disclaimer

## 🔍 Testing

### Manual Testing

You can test the email function manually:

```javascript
// Call this from your app or Firebase console
const testEmailNotification = functions.httpsCallable('manualInspectionReportEmail');

testEmailNotification({
  reportId: 'your-report-id',
  adminId: 'your-admin-id'
}).then((result) => {
  console.log('Test result:', result.data);
}).catch((error) => {
  console.error('Test error:', error);
});
```

## 📊 Monitoring

### Email Logs

All email activity is logged in Firebase:

```
/emailLogs/inspectionReports/{logId}/
├── recipient: "user@example.com"
├── recipientName: "User Name"
├── recipientRole: "Supervisor"
├── emailType: "Inspection Report Notification"
├── messageId: "email-message-id"
├── sentAt: timestamp
└── status: "sent"
```

### Function Logs

Monitor function execution in Firebase Console:
1. Go to Firebase Console
2. Functions → Logs
3. Filter by `inspectionReportEmailNotify`

## 🛠️ Troubleshooting

### Common Issues

1. **Emails not sending**
   - Check email configuration: `firebase functions:config:get`
   - Verify Gmail App Password is correct
   - Check function logs for errors

2. **Users not receiving emails**
   - Verify user has correct country/project permissions
   - Check that user has `inspectionReport` module access
   - Ensure user's email is valid in Firebase

3. **PDF links not working**
   - Check Firebase Storage permissions
   - Verify PDF was uploaded successfully
   - Check that storage path is correct

### Debug Commands

```bash
# View current configuration
firebase functions:config:get

# View function logs
firebase functions:log --only inspectionReportEmailNotify

# Deploy functions
firebase deploy --only functions
```

## 🔒 Security

- **Email credentials** are stored securely in Firebase Functions config
- **PDF links** have 7-day expiry for security
- **User permissions** are checked before sending emails
- **Email logs** track all notifications for audit trail

## 📝 Customization

### Email Template

To customize the email template, edit the `prepareEmailContent()` function in `InspectionReportEmailNotify.js`:

```javascript
// Modify email subject
const emailSubject = `Custom Subject: ${requestNumber}`;

// Modify email body HTML
const emailBody = `
  <!-- Your custom HTML template -->
`;
```

### Permission Logic

To modify who receives emails, edit the `checkUserPermissions()` and `getSupervisorsAndAdmins()` functions.

### Email Provider

To use a different email provider, modify the `emailConfig` object:

```javascript
const emailConfig = {
  service: 'outlook', // or other provider
  auth: {
    user: functions.config().email?.user,
    pass: functions.config().email?.pass
  }
};
```

## 📞 Support

For issues or questions:
1. Check Firebase Function logs
2. Verify email configuration
3. Test with manual trigger function
4. Review permission structure in Firebase Database

---

**Note**: This system automatically sends emails when inspection reports are created. Ensure your email quotas and Firebase usage limits are appropriate for your expected volume.
