const functions = require('firebase-functions/v1');
const admin = require('firebase-admin');

/**
 * Filter and delete unused user data from /Globalusers/admins/
 * This function removes admin user data that doesn't exist in Firebase Authentication
 */
exports.filterDeleteUnusedUserData = functions.https.onCall(async (data, context) => {
  try {
    // Check if user is authenticated
    if (!context.auth) {
      throw new functions.https.HttpsError('unauthenticated', 'User must be authenticated');
    }

    // Check if user has admin privileges (optional - you can modify this check)
    // if (!context.auth.token.admin) {
    //   throw new functions.https.HttpsError('permission-denied', 'Admin access required');
    // }

    console.log('Starting filterDeleteUnusedUserData function...');

    // Get all users from Firebase Authentication
    const authUsers = await admin.auth().listUsers();
    const authUserUids = new Set(authUsers.users.map(user => user.uid));
    
    console.log(`Found ${authUserUids.size} users in Authentication`);

    // Get all admin users from Realtime Database
    const adminUsersSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    const adminUsers = adminUsersSnapshot.val();

    if (!adminUsers) {
      console.log('No admin users found in Realtime Database');
      return {
        success: true,
        message: 'No admin users found in Realtime Database',
        deletedCount: 0,
        remainingCount: 0
      };
    }

    const adminUserUids = Object.keys(adminUsers);
    console.log(`Found ${adminUserUids.length} admin users in Realtime Database`);

    // Find UIDs that exist in Realtime Database but not in Authentication
    const unusedUids = adminUserUids.filter(uid => !authUserUids.has(uid));
    
    console.log(`Found ${unusedUids.length} unused admin user UIDs to delete`);

    if (unusedUids.length === 0) {
      console.log('No unused admin users found');
      return {
        success: true,
        message: 'No unused admin users found',
        deletedCount: 0,
        remainingCount: adminUserUids.length
      };
    }

    // Delete unused admin users from Realtime Database
    const updates = {};
    unusedUids.forEach(uid => {
      updates[`/Globalusers/admins/${uid}`] = null;
    });

    await admin.database().ref().update(updates);

    console.log(`Successfully deleted ${unusedUids.length} unused admin users`);
    console.log('Deleted UIDs:', unusedUids);

    return {
      success: true,
      message: `Successfully deleted ${unusedUids.length} unused admin users`,
      deletedCount: unusedUids.length,
      remainingCount: adminUserUids.length - unusedUids.length,
      deletedUids: unusedUids
    };

  } catch (error) {
    console.error('Error in filterDeleteUnusedUserData:', error);
    throw new functions.https.HttpsError('internal', `Error filtering unused user data: ${error.message}`);
  }
});

/**
 * HTTP function to manually trigger the cleanup
 * You can call this via HTTP request or from your app
 */
exports.cleanupUnusedAdminUsers = functions.https.onRequest(async (req, res) => {
  try {
    console.log('Starting cleanupUnusedAdminUsers HTTP function...');

    // Get all users from Firebase Authentication
    const authUsers = await admin.auth().listUsers();
    const authUserUids = new Set(authUsers.users.map(user => user.uid));
    
    console.log(`Found ${authUserUids.size} users in Authentication`);

    // Get all admin users from Realtime Database
    const adminUsersSnapshot = await admin.database().ref('/Globalusers/admins').once('value');
    const adminUsers = adminUsersSnapshot.val();

    if (!adminUsers) {
      console.log('No admin users found in Realtime Database');
      res.status(200).json({
        success: true,
        message: 'No admin users found in Realtime Database',
        deletedCount: 0,
        remainingCount: 0
      });
      return;
    }

    const adminUserUids = Object.keys(adminUsers);
    console.log(`Found ${adminUserUids.length} admin users in Realtime Database`);

    // Find UIDs that exist in Realtime Database but not in Authentication
    const unusedUids = adminUserUids.filter(uid => !authUserUids.has(uid));
    
    console.log(`Found ${unusedUids.length} unused admin user UIDs to delete`);

    if (unusedUids.length === 0) {
      console.log('No unused admin users found');
      res.status(200).json({
        success: true,
        message: 'No unused admin users found',
        deletedCount: 0,
        remainingCount: adminUserUids.length
      });
      return;
    }

    // Delete unused admin users from Realtime Database
    const updates = {};
    unusedUids.forEach(uid => {
      updates[`/Globalusers/admins/${uid}`] = null;
    });

    await admin.database().ref().update(updates);

    console.log(`Successfully deleted ${unusedUids.length} unused admin users`);
    console.log('Deleted UIDs:', unusedUids);

    res.status(200).json({
      success: true,
      message: `Successfully deleted ${unusedUids.length} unused admin users`,
      deletedCount: unusedUids.length,
      remainingCount: adminUserUids.length - unusedUids.length,
      deletedUids: unusedUids
    });

  } catch (error) {
    console.error('Error in cleanupUnusedAdminUsers:', error);
    res.status(500).json({
      success: false,
      error: error.message
    });
  }
});
