import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  Animated,
  ActivityIndicator,
  Modal,
  ScrollView,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
  ToastAndroid,
  Alert,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import auth from '@react-native-firebase/auth';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import LinearGradient from 'react-native-linear-gradient';
import NetInfo from '@react-native-community/netinfo';
import { TextInput } from 'react-native-paper';
import { GoogleSignin } from '@react-native-google-signin/google-signin';
import Toast from 'react-native-toast-message';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import DropDownPicker from 'react-native-dropdown-picker';

// Get screen dimensions for responsive design
const { width } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style, isLoading }) => {
  return (
    <LinearGradient colors={['#015185', '#3481BC']} style={[styles.gradientButton, style]}>
      <TouchableOpacity style={styles.buttonTouchable} onPress={onPress} activeOpacity={0.8} disabled={isLoading}>
        {isLoading ? (
          <ActivityIndicator size="small" color="#fff" />
        ) : (
          <Text style={styles.googleButtonText}>{title}</Text>
        )}
      </TouchableOpacity>
    </LinearGradient>
  );
};

// Custom Animated Input Component
const AnimatedInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  keyboardType,
  autoCapitalize,
  leftIconName,
  rightIconName,
  onRightIconPress,
  onBlur,
  maxLength,
  inputRef,
  hasError,
  onFocus,
}) => {
  const [isFocused, setIsFocused] = useState(false);
  const animatedLabel = useRef(new Animated.Value(value ? 1 : 0)).current;
  const animatedBorderWidth = useRef(new Animated.Value(1)).current;
  const animatedBorderColor = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    Animated.parallel([
      Animated.timing(animatedLabel, {
        toValue: isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderWidth, {
        toValue: isFocused ? 2 : 1,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderColor, {
        toValue: hasError ? 2 : isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
    ]).start();
  }, [isFocused, value, hasError]);

  const labelStyle = {
    position: 'absolute',
    left: scale(45),
    top: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(18), scale(-8)],
    }),
    fontSize: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(16), scale(12)],
    }),
    color: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: ['#888', '#015185'],
    }),
    backgroundColor: '#fff',
    paddingHorizontal: scale(4),
    zIndex: 2,
  };

  const borderColor = animatedBorderColor.interpolate({
    inputRange: [0, 1, 2],
    outputRange: ['#ddd', '#015185', '#ff0000'],
  });

  return (
    <View style={styles.inputContainer} ref={inputRef}>
      <Animated.View
        style={[
          styles.inputGradient,
          {
            borderWidth: animatedBorderWidth,
            borderColor: borderColor,
          },
        ]}
      >
        <Animated.Text style={labelStyle}>{label}</Animated.Text>
        <TextInput
          mode="outlined"
          value={value}
          onChangeText={onChangeText}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          style={styles.input}
          onFocus={() => {
            setIsFocused(true);
            onFocus && onFocus();
          }}
          onBlur={() => {
            setIsFocused(false);
            onBlur && onBlur();
          }}
          left={
            leftIconName && (
              <TextInput.Icon
                icon={() => <MaterialIcons name={leftIconName} size={scale(24)} color="#015185" />}
              />
            )
          }
          right={
            rightIconName && (
              <TextInput.Icon
                icon={() => (
                  <TouchableOpacity onPress={onRightIconPress}>
                    <MaterialIcons name={rightIconName} size={scale(24)} color="#015185" />
                  </TouchableOpacity>
                )}
              />
            )
          }
          outlineStyle={{
            borderWidth: 0,
            borderRadius: scale(8),
          }}
          theme={{
            colors: {
              primary: 'transparent',
              text: '#333',
              placeholder: '#888',
            },
          }}
          maxLength={maxLength}
        />
      </Animated.View>
    </View>
  );
};

// SyncModal Component with customizable message
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function for platform-specific toasts
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.showWithGravity(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

const AdminSignUp = () => {
  const navigation = useNavigation();
  // Authentication state
  const [name, setName] = useState('');
  const [email, setEmail] = useState('');
  const [password, setPassword] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [nameError, setNameError] = useState(false);
  const [nameErrorMessage, setNameErrorMessage] = useState('');
  const [emailError, setEmailError] = useState(false);
  const [emailErrorMessage, setEmailErrorMessage] = useState('');
  const [passwordError, setPasswordError] = useState(false);
  const [passwordErrorMessage, setPasswordErrorMessage] = useState('');
  const [isLoadingSignUp, setIsLoadingSignUp] = useState(false);
  const [isNetworkError, setIsNetworkError] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [syncModalMessage, setSyncModalMessage] = useState('');
  const networkErrorAnim = useRef(new Animated.Value(-scale(50))).current;
  const scrollViewRef = useRef(null);
  const nameInputRef = useRef(null);
  const emailInputRef = useRef(null);
  const passwordInputRef = useRef(null);

  // Dropdown state
  const [countries, setCountries] = useState([]);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems, setCountryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);

  const [projects, setProjects] = useState([]);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState([]);

  const [modules, setModules] = useState([]);
  const [moduleOpen, setModuleOpen] = useState(false);
  const [moduleItems, setModuleItems] = useState([
    { label: 'Equipments', value: 'Equipments' },
    { label: 'Documents', value: 'Documents' },
    { label: 'Team Employ', value: 'teamEmploy' },
    { label: 'Inspection Report', value: 'inspectionReport' },
    { label: 'DVIR Report', value: 'dvirReport' },
    { label: 'Checklist', value: 'checklist' },
    { label: 'Incident Report', value: 'incidentReport' },
    { label: 'Daily Report', value: 'dailyReport' },
    { label: 'Request Maintenance', value: 'requestMaintenance' },
    { label: 'Jobcard', value: 'jobcard' },
    { label: 'Service Schedule', value: 'serviceSchedule' },
    { label: 'Fuel Log', value: 'fuelLog' },
    { label: 'Logbook', value: 'logbook' },
    { label: 'HR Form', value: 'hrForm' },
    { label: 'HR Applications', value: 'hrApplications' },
    { label: 'Team KPI', value: 'teamKpi' },
  ]);

  const permissionOptions = [
    { label: 'All Permissions', value: 'all' },
    { label: 'Create', value: 'create' },
    { label: 'Edit', value: 'edit' },
    { label: 'Delete', value: 'delete' },
    { label: 'View', value: 'view' },
    { label: 'Update', value: 'update' },
    { label: 'Only Mine View', value: 'onlyMineView' },
  ];

  const individualPermissions = ['create', 'edit', 'delete', 'view', 'update', 'onlyMineView'];

  const [modulePermissions, setModulePermissions] = useState({});
  const [permissionOpen, setPermissionOpen] = useState({});
  const [errors, setErrors] = useState({
    country: false,
    project: false,
    module: false,
    permissions: {},
  });

  // Network error animation
  const showNetworkError = () => {
    setIsNetworkError(true);
    Animated.timing(networkErrorAnim, {
      toValue: 0,
      duration: 300,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }, 3000);
  };

  // Configure Google Sign-In
  useEffect(() => {
    const initialize = async () => {
      try {
        await GoogleSignin.configure({
          webClientId: '749788854975-rhkmepo20c1n038ftnreustj8br62qbj.apps.googleusercontent.com',
          androidClientId: '749788854975-1vqr2rje9a19hl56ohheqvr32hl6045p.apps.googleusercontent.com',
          iosClientId: '749788854975-fsan3f653h4fffue813uqc38h6mu7ipe.apps.googleusercontent.com',
          offlineAccess: true,
        });
        await checkNetworkStatus();
      } catch (error) {
        console.error('Initialization error:', error);
        showNetworkError();
        showToast('error', 'Initialization Error', 'Failed to initialize Google Sign-In or network check');
      }
    };

    initialize();

    return () => {
      GoogleSignin.signOut().catch((error) => console.error('Google Sign-In cleanup error:', error));
    };
  }, []);

  // Populate projects based on countries
  useEffect(() => {
    const projectMap = {
      Zambia: [
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ],
      DRC: [
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ],
    };

    const updatedProjects = countries
      .flatMap((country) => projectMap[country] || [])
      .filter((proj, index, self) => index === self.findIndex((p) => p.value === proj.value));

    setProjectItems(updatedProjects);
    const validProjects = projects.filter((proj) => updatedProjects.some((item) => item.value === proj));
    if (validProjects.length !== projects.length) {
      setProjects(validProjects);
    }
  }, [countries]);

  // Update module permissions
  useEffect(() => {
    const updatedPermissions = { ...modulePermissions };
    const updatedErrors = { ...errors.permissions };

    modules.forEach((mod) => {
      updatedPermissions[mod] = updatedPermissions[mod] || [];
      updatedErrors[mod] = updatedPermissions[mod].length === 0;
    });

    Object.keys(updatedPermissions).forEach((mod) => {
      if (!modules.includes(mod)) {
        delete updatedPermissions[mod];
        delete updatedErrors[mod];
      }
    });

    setModulePermissions(updatedPermissions);
    setErrors((prev) => ({ ...prev, permissions: updatedErrors }));
  }, [modules]);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showNetworkError();
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      }
      return true;
    } catch (error) {
      console.error('Network check error:', error);
      showNetworkError();
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

  // Validation functions
  const validateName = (name) => {
    const nameEmpty = !name.trim();
    const nameTooLong = name.length > 13;

    setNameError(nameEmpty || nameTooLong);
    if (nameEmpty) {
      setNameErrorMessage('Name is required');
    } else if (nameTooLong) {
      setNameErrorMessage('Name cannot exceed 13 characters');
    } else {
      setNameErrorMessage('');
    }

    return !(nameEmpty || nameTooLong);
  };

  const validateEmail = (email) => {
    const emailEmpty = !email.trim();
    const emailInvalid = emailEmpty ? false : !/\S+@\S+\.\S+/.test(email);
    const emailTooLong = email.length > 40;

    setEmailError(emailEmpty || emailInvalid || emailTooLong);
    if (emailEmpty) {
      setEmailErrorMessage('Email is required');
    } else if (emailInvalid) {
      setEmailErrorMessage('Enter a valid email address');
    } else if (emailTooLong) {
      setEmailErrorMessage('Email cannot exceed 40 characters');
    } else {
      setEmailErrorMessage('');
    }

    return !(emailEmpty || emailInvalid || emailTooLong);
  };

  const validatePassword = (password) => {
    const passwordEmpty = !password.trim();
    const passwordShort = passwordEmpty ? false : password.length < 8;
    const passwordNoSpecialChar = passwordEmpty ? false : !/[!@#$%^&*(),.?":{}|<>]/.test(password);
    const passwordTooLong = password.length > 15;

    setPasswordError(passwordEmpty || passwordShort || passwordNoSpecialChar || passwordTooLong);
    if (passwordEmpty) {
      setPasswordErrorMessage('Password is required');
    } else if (passwordShort) {
      setPasswordErrorMessage('Password must be at least 8 characters');
    } else if (passwordNoSpecialChar) {
      setPasswordErrorMessage('Password needs a special character');
    } else if (passwordTooLong) {
      setPasswordErrorMessage('Password cannot exceed 15 characters');
    } else {
      setPasswordErrorMessage('');
    }

    return !(passwordEmpty || passwordShort || passwordNoSpecialChar || passwordTooLong);
  };

  const validateDropdowns = () => {
    const newErrors = {
      country: countries.length === 0,
      project: projects.length === 0,
      module: modules.length === 0,
      permissions: {},
    };

    modules.forEach((mod) => {
      const perms = modulePermissions[mod] || [];
      newErrors.permissions[mod] = perms.length === 0;
    });

    setErrors(newErrors);

    return (
      !newErrors.country &&
      !newErrors.project &&
      !newErrors.module &&
      Object.values(newErrors.permissions).every((v) => v === false)
    );
  };

  // Handle permission changes
  const handlePermissionChange = (module, callback) => {
    const selected = typeof callback === 'function' ? callback(modulePermissions[module] || []) : callback;
    let newPermissions = Array.isArray(selected) ? [...selected] : [];

    if (newPermissions.includes('all') && !modulePermissions[module]?.includes('all')) {
      newPermissions = [...new Set(['all', ...individualPermissions])];
    } else if (!newPermissions.includes('all') && modulePermissions[module]?.includes('all')) {
      newPermissions = newPermissions.filter((perm) => perm !== 'all');
    } else {
      const hasAllIndividual = individualPermissions.every((perm) => newPermissions.includes(perm));
      if (hasAllIndividual && !newPermissions.includes('all')) {
        newPermissions = [...newPermissions, 'all'];
      } else if (!hasAllIndividual && newPermissions.includes('all')) {
        newPermissions = newPermissions.filter((perm) => perm !== 'all');
      }
    }

    setModulePermissions((prev) => ({
      ...prev,
      [module]: newPermissions,
    }));

    setErrors((prev) => ({
      ...prev,
      permissions: {
        ...prev.permissions,
        [module]: newPermissions.length === 0,
      },
    }));
  };

  // Handle email/password signup
  const handleEmailPasswordSignUp = async () => {
    const isNameValid = validateName(name);
    const isEmailValid = validateEmail(email);
    const isPasswordValid = validatePassword(password);
    const isDropdownValid = validateDropdowns();

    if (!isNameValid || !isEmailValid || !isPasswordValid || !isDropdownValid) {
      showToast('error', 'Validation Error', 'Please correct all input errors');
      Alert.alert('Error', 'Please fill all required fields correctly.');
      return;
    }

    setIsLoadingSignUp(true);
    setSyncModalVisible(true);
    setSyncModalMessage('Processing Sign Up');

    try {
      if (!(await checkNetworkStatus())) {
        throw new Error('No internet connection');
      }

      const userCredential = await auth().createUserWithEmailAndPassword(email, password);
      const userId = userCredential.user.uid;

      const userData = {
        name,
        email,
        loginStatus: false,
        lastLogin: null,
        countries,
        projects,
        modules: modules.map((mod) => ({
          module: mod,
          permissions: modulePermissions[mod].filter((perm) => perm !== 'all'),
        })),
      };

      await database().ref(`/Globalusers/admins/${userId}`).set(userData);

      try {
        await Promise.all([
          AsyncStorage.setItem('userUid', userId),
          AsyncStorage.setItem('userEmail', email),
          AsyncStorage.setItem('Loginsuccess-01', 'true'),
        ]);
      } catch (storageError) {
        console.error('AsyncStorage error:', storageError);
        showToast('error', 'Storage Error', 'Failed to save user data locally');
      }

      showToast('success', 'Success', 'Sign-up successful');
      navigation.replace('MainApp');
    } catch (error) {
      console.error('Sign-up error:', {
        message: error.message,
        code: error.code,
        details: JSON.stringify(error, null, 2),
      });
      const errorMessages = {
        'auth/email-already-in-use': 'This email is already in use',
        'auth/invalid-email': 'Invalid email address',
        'auth/weak-password': 'Password is too weak',
        'auth/too-many-requests': 'Too many attempts. Please try again later',
        default: error.message || 'An unexpected error occurred',
      };
      showToast('error', 'Sign-Up Error', errorMessages[error.code] || errorMessages.default);
    } finally {
      setIsLoadingSignUp(false);
      setSyncModalVisible(false);
    }
  };

  // Handle Google Sign-In
  const handleGoogleSignUp = async () => {
    const isDropdownValid = validateDropdowns();

    if (!isDropdownValid) {
      showToast('error', 'Validation Error', 'Please select all required dropdowns');
      Alert.alert('Error', 'Please fill all required fields correctly.');
      return;
    }

    setIsLoadingSignUp(true);
    setSyncModalVisible(true);
    setSyncModalMessage('Processing Google Sign Up');

    try {
      if (!(await checkNetworkStatus())) {
        throw new Error('No internet connection');
      }

      await GoogleSignin.hasPlayServices({ showPlayServicesUpdateDialog: true });
      const signInResult = await GoogleSignin.signIn();
      const idToken = signInResult.data?.idToken || signInResult.idToken;

      if (!idToken) {
        throw new Error('No ID token found in Google Sign-In response.');
      }

      const googleCredential = auth.GoogleAuthProvider.credential(idToken);
      const userCredential = await auth().signInWithCredential(googleCredential);
      const userId = userCredential.user.uid;

      const userRef = database().ref(`/Globalusers/admins/${userId}`);
      const snapshot = await userRef.once('value');
      const isNewUser = !snapshot.exists();

      const userData = {
        name: userCredential.user.displayName || 'Unknown',
        email: userCredential.user.email,
        loginStatus: true,
        lastLogin: new Date().toISOString(),
        countries,
        projects,
        modules: modules.map((mod) => ({
          module: mod,
          permissions: modulePermissions[mod].filter((perm) => perm !== 'all'),
        })),
      };

      if (isNewUser) {
        await userRef.set(userData);
        showToast('success', 'Sign-Up Success', 'New user created with Google');
      } else {
        await userRef.update({
          loginStatus: true,
          lastLogin: new Date().toISOString(),
          countries,
          projects,
          modules: userData.modules,
        });
        showToast('success', 'Sign-In Success', 'Logged in with Google');
      }

      try {
        await Promise.all([
          AsyncStorage.setItem('userUid', userId),
          AsyncStorage.setItem('userEmail', userCredential.user.email),
          AsyncStorage.setItem('Loginsuccess-01', 'true'),
        ]);
      } catch (storageError) {
        console.error('AsyncStorage error:', storageError);
        showToast('error', 'Storage Error', 'Failed to save user data locally');
      }

      navigation.replace('MainApp');
    } catch (error) {
      console.error('Google Sign-In error:', {
        message: error.message,
        code: error.code,
        details: JSON.stringify(error, null, 2),
      });
      await GoogleSignin.signOut();
      await auth().signOut();
      const errorMessages = {
        'auth/account-exists-with-different-credential': 'Account exists with different credential.',
        'auth/invalid-credential': 'Invalid Google credential.',
        'auth/operation-not-allowed': 'Google Sign-In not enabled in Firebase.',
        10: 'Developer Error: Check SHA-1 fingerprint in Firebase Console.',
        'auth/network-request-failed': 'Network error: Check your internet connection.',
        default: error.message || 'Unexpected error during Google Sign-In.',
      };
      showToast('error', 'Google Sign-Up Error', errorMessages[error.code] || errorMessages.default);
    } finally {
      setIsLoadingSignUp(false);
      setSyncModalVisible(false);
    }
  };

  const toggleShowPassword = () => {
    setShowPassword(!showPassword);
  };

  // Handle input focus to scroll
  const handleInputFocus = (inputRef) => {
    setTimeout(() => {
      inputRef.current?.measureLayout(
        scrollViewRef.current.getScrollResponder(),
        (x, y) => {
          scrollViewRef.current.scrollTo({ y: y - scale(100), animated: true });
        },
        (error) => {
          console.error('Error measuring layout:', error);
        }
      );
    }, 300);
  };

  return (
    <SafeAreaView style={styles.container}>
      {isNetworkError && (
        <Animated.View
          style={[styles.networkErrorBanner, { transform: [{ translateY: networkErrorAnim }] }]}
        >
          <Text style={styles.networkErrorText}>Internet connection not available</Text>
        </Animated.View>
      )}

      <KeyboardAvoidingView
        style={{ flex: 1 }}
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? scale(100) : scale(20)}
      >
        <ScrollView
          ref={scrollViewRef}
          contentContainerStyle={styles.scrollContent}
          keyboardShouldPersistTaps="handled"
          nestedScrollEnabled={true}
          disableScrollViewPanResponder={true}
          showsVerticalScrollIndicator={false}
        >
          <View style={styles.header}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={scale(24)} color="#015185" />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.syncButton}
              onPress={() => {
                setSyncModalVisible(true);
                setSyncModalMessage('Syncing Cloud Data');
                setTimeout(() => {
                  setSyncModalVisible(false);
                  showToast('success', 'Sync Complete', 'Data synchronized successfully');
                }, 2000);
              }}
            >
              <Image
                source={require('../../../Images/adminhome/equp/cloud_1.png')}
                style={styles.cloudIcon}
              />
            </TouchableOpacity>
          </View>

          <View style={styles.contentContainer}>
            <View style={styles.logoContainer}>
              <Image
                source={require('../../../Images/adminlogin/logo.png')}
                style={styles.logo}
              />
              <Text style={styles.loginText}>Sign Up</Text>
            </View>

            <View style={styles.formContainer}>
              {/* Authentication Inputs */}
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={nameInputRef}
                  label="Name"
                  value={name}
                  onChangeText={(text) => {
                    setName(text);
                    validateName(text);
                  }}
                  autoCapitalize="words"
                  leftIconName="person"
                  hasError={nameError}
                  onBlur={() => validateName(name)}
                  maxLength={13}
                  onFocus={() => handleInputFocus(nameInputRef)}
                />
                {nameError && <Text style={styles.errorText}>{nameErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={emailInputRef}
                  label="Email"
                  value={email}
                  onChangeText={(text) => {
                    setEmail(text);
                    validateEmail(text);
                  }}
                  keyboardType="email-address"
                  autoCapitalize="none"
                  leftIconName="email"
                  hasError={emailError}
                  onBlur={() => validateEmail(email)}
                  maxLength={40}
                  onFocus={() => handleInputFocus(emailInputRef)}
                />
                {emailError && <Text style={styles.errorText}>{emailErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={passwordInputRef}
                  label="Password"
                  value={password}
                  onChangeText={(text) => {
                    setPassword(text);
                    validatePassword(text);
                  }}
                  secureTextEntry={!showPassword}
                  autoCapitalize="none"
                  leftIconName="lock"
                  rightIconName={showPassword ? 'visibility' : 'visibility-off'}
                  onRightIconPress={toggleShowPassword}
                  hasError={passwordError}
                  onBlur={() => validatePassword(password)}
                  maxLength={15}
                  onFocus={() => handleInputFocus(passwordInputRef)}
                />
                {passwordError && <Text style={styles.errorText}>{passwordErrorMessage}</Text>}
              </View>

              {/* Dropdown Inputs */}
              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Countries <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  multiple={true}
                  open={countryOpen}
                  value={countries}
                  items={countryItems}
                  setOpen={(open) => {
                    setCountryOpen(open);
                    if (open) {
                      setProjectOpen(false);
                      setModuleOpen(false);
                      setPermissionOpen({});
                    }
                  }}
                  setValue={setCountries}
                  setItems={setCountryItems}
                  placeholder="Select Countries"
                  zIndex={10000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[inputStyles.input, errors.country ? inputStyles.inputError : null]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.country && (
                  <Text style={inputStyles.errorText}>At least one country is required</Text>
                )}
                <View style={styles.chipContainer}>
                  {countries.length > 0 ? (
                    countries.map((country, index) => (
                      <View key={index} style={styles.chip}>
                        <Text style={styles.chipText}>{country}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.noItemsText}>No countries selected</Text>
                  )}
                </View>
              </View>

              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Projects <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  multiple={true}
                  open={projectOpen}
                  value={projects}
                  items={projectItems}
                  setOpen={(open) => {
                    setProjectOpen(open);
                    if (open) {
                      setCountryOpen(false);
                      setModuleOpen(false);
                      setPermissionOpen({});
                    }
                  }}
                  setValue={setProjects}
                  setItems={setProjectItems}
                  placeholder="Select Projects"
                  disabled={countries.length === 0}
                  disabledStyle={{ opacity: 0.6 }}
                  zIndex={9000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[
                    inputStyles.input,
                    countries.length === 0 && styles.disabledDropdown,
                    errors.project ? inputStyles.inputError : null,
                  ]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.project && (
                  <Text style={inputStyles.errorText}>At least one project is required</Text>
                )}
                <View style={styles.chipContainer}>
                  {projects.length > 0 ? (
                    projects.map((project, index) => (
                      <View key={index} style={styles.chip}>
                        <Text style={styles.chipText}>{project}</Text>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.noItemsText}>No projects selected</Text>
                  )}
                </View>
              </View>

              <View style={inputStyles.container}>
                <Text style={inputStyles.label}>
                  Modules <Text style={inputStyles.requiredError}>*</Text>
                </Text>
                <DropDownPicker
                  multiple={true}
                  open={moduleOpen}
                  value={modules}
                  items={moduleItems}
                  setOpen={(open) => {
                    setModuleOpen(open);
                    if (open) {
                      setCountryOpen(false);
                      setProjectOpen(false);
                      setPermissionOpen({});
                    }
                  }}
                  setValue={setModules}
                  setItems={setModuleItems}
                  placeholder="Select Modules"
                  zIndex={8000}
                  listMode="SCROLLVIEW"
                  scrollViewProps={{
                    nestedScrollEnabled: true,
                  }}
                  style={[inputStyles.input, errors.module ? inputStyles.inputError : null]}
                  dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                  textStyle={styles.dropdownText}
                  placeholderStyle={styles.placeholderStyle}
                  badgeDotColors={['#015185']}
                  badgeTextStyle={styles.badgeText}
                  badgeStyle={styles.badge}
                />
                {errors.module && (
                  <Text style={inputStyles.errorText}>At least one module is required</Text>
                )}
                <View style={styles.chipContainer}>
                  {modules.length > 0 ? (
                    modules.map((module, index) => (
                      <View key={module} style={styles.moduleChipContainer}>
                        <View style={styles.chip}>
                          <Text style={styles.chipText}>{module}</Text>
                        </View>
                        <View style={styles.permissionContainer}>
                          <Text style={styles.label}>
                            Permissions for {module} <Text style={styles.requiredError}>*</Text>
                          </Text>
                          <DropDownPicker
                            multiple={true}
                            open={permissionOpen[module] || false}
                            value={modulePermissions[module] || []}
                            items={permissionOptions}
                            setOpen={() =>
                              setPermissionOpen((prev) => ({
                                ...Object.keys(prev).reduce((acc, key) => ({ ...acc, [key]: false }), {}),
                                [module]: !prev[module],
                              }))
                            }
                            setValue={(callback) => handlePermissionChange(module, callback)}
                            setItems={() => {}}
                            placeholder={`Select Permissions for ${module}`}
                            zIndex={8000 - (index + 1)}
                            listMode="SCROLLVIEW"
                            scrollViewProps={{
                              nestedScrollEnabled: true,
                            }}
                            style={[
                              inputStyles.input,
                              errors.permissions[module] ? inputStyles.inputError : null,
                            ]}
                            dropDownContainerStyle={[styles.dropdownContainer, { maxHeight: scale(200) }]}
                            textStyle={styles.dropdownText}
                            placeholderStyle={styles.placeholderStyle}
                            badgeDotColors={['#015185']}
                            badgeTextStyle={styles.badgeText}
                            badgeStyle={styles.badge}
                          />
                          {errors.permissions[module] && (
                            <Text style={inputStyles.errorText}>
                              At least one permission is required for {module}
                            </Text>
                          )}
                          <View style={styles.chipContainer}>
                            {modulePermissions[module]?.length > 0 ? (
                              modulePermissions[module].map((permission, permIndex) => (
                                <View key={permIndex} style={styles.chip}>
                                  <Text style={styles.chipText}>{permission}</Text>
                                </View>
                              ))
                            ) : (
                              <Text style={styles.noItemsText}>No permissions selected</Text>
                            )}
                          </View>
                        </View>
                      </View>
                    ))
                  ) : (
                    <Text style={styles.noItemsText}>No modules selected</Text>
                  )}
                </View>
              </View>

              <GradientButton
                title="Sign Up with Email"
                onPress={handleEmailPasswordSignUp}
                isLoading={isLoadingSignUp}
              />

              <View style={styles.dividerContainer}>
                <View style={styles.borderLine} />
                <Text style={styles.dividerText}>Other ways to Sign Up</Text>
                <View style={styles.borderLine} />
              </View>

              <GradientButton
                title="Sign Up with Google"
                onPress={handleGoogleSignUp}
                isLoading={isLoadingSignUp}
              />
            </View>
          </View>
        </ScrollView>
      </KeyboardAvoidingView>

      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
        message={syncModalMessage}
      />
      <Toast />
    </SafeAreaView>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: scale(20),
    marginHorizontal: scale(10),
  },
  label: {
    fontSize: scale(14),
    color: '#015185',
    marginBottom: scale(4),
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: scale(5),
    paddingHorizontal: scale(12),
    paddingVertical: scale(14),
    fontSize: scale(16),
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: scale(12),
    marginTop: scale(2),
    marginLeft: scale(2),
  },
});

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollContent: {
    paddingHorizontal: scale(5),
    paddingBottom: scale(50),
    flexGrow: 1,
  },
  header: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    padding: scale(10),
    marginTop: scale(10),
  },
  syncButton: {
    padding: scale(5),
  },
  cloudIcon: {
    width: scale(28),
    height: scale(28),
  },
  contentContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: scale(20),
  },
  logo: {
    width: '80%',
    height: scale(100),
    resizeMode: 'contain',
  },
  loginText: {
    fontSize: scale(24),
    fontWeight: 'bold',
    color: '#015185',
    marginTop: 8,
  },
  formContainer: {
    width: '100%',
  },
  inputContainer: {
    marginBottom: scale(20),
    marginHorizontal: scale(8),
  },
  inputGradient: {
    borderRadius: scale(8),
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  input: {
    backgroundColor: '#fff',
    height: scale(50),
    borderRadius: scale(8),
    fontSize: scale(16),
  },
  gradientButton: {
    borderRadius: scale(6),
    marginVertical: scale(8),
    width: '95%',
    justifyContent: 'center',
    alignSelf: 'center',
  },
  buttonTouchable: {
    alignItems: 'center',
    justifyContent: 'center',
    height: scale(48),
    borderRadius: scale(6),
    width: '100%',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  dividerContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    width: '100%',
    marginVertical: scale(20),
  },
  borderLine: {
    flex: 1,
    height: 1,
    borderStyle: 'dashed',
    borderWidth: 1,
    borderColor: '#ccc',
  },
  dividerText: {
    fontSize: scale(14),
    color: '#888',
    fontWeight: '600',
    paddingHorizontal: scale(10),
  },
  networkErrorBanner: {
    backgroundColor: '#ff4444',
    paddingVertical: scale(10),
    alignItems: 'center',
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: scale(14),
    fontWeight: 'bold',
  },
  errorText: {
    color: 'red',
    fontSize: scale(12),
    marginLeft: scale(10),
    marginTop: scale(4),
    marginBottom: scale(8),
    position: 'absolute',
    bottom: scale(-20),
  },
  mainemailpass: {
    position: 'relative',
    marginBottom: scale(30),
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: scale(18),
    color: '#333',
    marginBottom: scale(10),
    fontWeight: '600',
  },
  syncIcon: {
    width: scale(24),
    height: scale(24),
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#555',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: scale(5),
    backgroundColor: '#fff',
  },
  dropdownText: {
    fontSize: scale(14),
    color: '#333',
  },
  placeholderStyle: {
    fontSize: scale(14),
    color: '#888',
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  badge: {
    backgroundColor: '#e6f0fa',
    borderRadius: scale(12),
    paddingHorizontal: scale(8),
    paddingVertical: scale(4),
    margin: scale(2),
    borderWidth: 1,
    borderColor: '#015185',
  },
  badgeText: {
    fontSize: scale(12),
    color: '#015185',
    fontWeight: '500',
  },
  chipContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginTop: scale(8),
  },
  chip: {
    backgroundColor: '#e6f0fa',
    borderRadius: scale(12),
    paddingHorizontal: scale(10),
    paddingVertical: scale(6),
    marginRight: scale(8),
    marginBottom: scale(8),
    borderWidth: 1,
    borderColor: '#015185',
  },
  chipText: {
    fontSize: scale(12),
    color: '#015185',
    fontWeight: '500',
  },
  noItemsText: {
    fontSize: scale(14),
    color: '#888',
    marginTop: scale(8),
  },
  moduleChipContainer: {
    marginBottom: scale(20),
    width: '100%',
  },
  permissionContainer: {
    marginTop: scale(8),
    marginLeft: scale(12),
  },
  label: {
    fontSize: scale(14),
    color: '#015185',
    marginBottom: scale(4),
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
});

export default AdminSignUp;