import React from 'react';
import { View, TextInput, TouchableOpacity, Image, Animated, StyleSheet, Text } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';

// Animated Floating Label Input Component
const FloatingLabelInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  icon,
  rightIcon,
  onRightIconPress,
  keyboardType,
  autoCapitalize,
  style,
}) => {
  const [isFocused, setIsFocused] = React.useState(false);
  const animatedIsFocused = React.useRef(new Animated.Value(value ? 1 : 0)).current;

  React.useEffect(() => {
    Animated.timing(animatedIsFocused, {
      toValue: isFocused || value ? 1 : 0,
      duration: 200,
      useNativeDriver: false,
    }).start();
  }, [isFocused, value]);

  const labelStyle = {
    position: 'absolute',
    left: 44,
    top: animatedIsFocused.interpolate({ inputRange: [0, 1], outputRange: [18, -8] }),
    fontSize: animatedIsFocused.interpolate({ inputRange: [0, 1], outputRange: [16, 12] }),
    color: '#015185',
    backgroundColor: '#fff',
    paddingHorizontal: 4,
    zIndex: 2,
    fontWeight: '400',
    
  };

  return (
    <View style={[styles.inputContainer, { marginBottom: 24 }]}>
      {icon && <Image source={icon} style={styles.inputIcon} />}
      <Animated.Text style={labelStyle}>{label}</Animated.Text>
      <LinearGradient
        colors={isFocused ? ['#015185', '#3481BC'] : ['#ddd', '#ddd']}
        start={{ x: 0, y: 0 }}
        end={{ x: 1, y: 0 }}
        style={styles.inputGradient}
      >
        <TextInput
          value={value}
          onChangeText={onChangeText}
          style={[styles.inputWithIcon, style]}
          onFocus={() => setIsFocused(true)}
          onBlur={() => setIsFocused(false)}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          placeholder={!isFocused && !value ? label : ''}
          placeholderTextColor="#888"
        />
      </LinearGradient>
      {rightIcon && (
        <TouchableOpacity style={styles.eyeButton} onPress={onRightIconPress}>
          <Image source={rightIcon} style={styles.eyeIcon} />
        </TouchableOpacity>
      )}
    </View>
  );
};

// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style }) => {
  return (
    <LinearGradient colors={["#015185", "#3481BC"]} style={[styles.gradientButton, style]}>
      <TouchableOpacity
        style={styles.buttonTouchable}
        onPress={onPress}
        activeOpacity={0.8}
      >
        <Text style={styles.googleButtonText}>{title}</Text>
      </TouchableOpacity>
    </LinearGradient>
  );
};

// Styles for both components
const styles = StyleSheet.create({
  // FloatingLabelInput styles
  inputContainer: {
    position: 'relative',
    marginBottom: 15,
    marginHorizontal: 10,
    marginBottom:20,
    // borderWidth:1
    
  },
  inputGradient: {
    borderRadius: 3,
    borderWidth: 1, // Add borderWidth to ensure gradient border is visible on all sides
    borderColor: 'transparent', // Make the default border transparent so the gradient takes over
  },
  inputWithIcon: {
    height: 50,
    backgroundColor: '#fff', // Keep background white for the input area
    borderRadius: 3,
    paddingHorizontal: 15,
    paddingLeft: 45,
    fontSize: 16,
    color: '#333',
    width: '100%',
    textAlignVertical: 'center',
    borderWidth: 0, // Ensure no inner border conflicts with the gradient
  },
  inputIcon: {
    width: 24,
    height: 24,
    position: 'absolute',
    left: 10,
    top: '50%',
    transform: [{ translateY: -12 }],
    tintColor: '#015185',
    zIndex: 2,
  },
  eyeButton: {
    position: 'absolute',
    right: 10,
    top: '50%',
    transform: [{ translateY: -12 }],
    zIndex: 2,
  },
  eyeIcon: {
    width: 24,
    height: 24,
    tintColor: '#015185',
  },
  // GradientButton styles
  gradientButton: {
    borderRadius: 6,
    marginTop: 10,
    marginBottom: 10,
    width: '95%',
    justifyContent: "center",
    alignSelf: "center",
  },
  buttonTouchable: {
    alignItems: 'center',
    justifyContent: 'center',
    height: 48,
    borderRadius: 6,
    width: '100%',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export { FloatingLabelInput, GradientButton };