import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  Image,
  StyleSheet,
  SafeAreaView,
  TouchableOpacity,
  Animated,
  ActivityIndicator,
  Modal,
  Platform,
  ToastAndroid,
  ScrollView,
  KeyboardAvoidingView,
  Dimensions,
  PermissionsAndroid,
  AppState,
} from 'react-native';
import { useNavigation } from '@react-navigation/native';
import auth from '@react-native-firebase/auth';
import database from '@react-native-firebase/database';
import messaging from '@react-native-firebase/messaging';
import AsyncStorage from '@react-native-async-storage/async-storage';
import LinearGradient from 'react-native-linear-gradient';
import NetInfo from '@react-native-community/netinfo';
import { TextInput } from 'react-native-paper';
import { GoogleSignin } from '@react-native-google-signin/google-signin';
import Toast from 'react-native-toast-message';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import notifee, { AndroidImportance } from '@notifee/react-native';
import iosNotificationManager from '../Utils/IOSNotificationManager';

// Get screen dimensions for responsive design
const { width } = Dimensions.get('window');
const scale = (size) => (width / 375) * size;

// Detect if running on emulator - lazy evaluation with multiple fallbacks
let _isEmulatorCache = null;
const isEmulator = () => {
  if (_isEmulatorCache !== null) {
    return _isEmulatorCache;
  }
  
  try {
    // For Android, check common emulator indicators
    if (Platform.OS === 'android') {
      const DeviceInfo = require('react-native-device-info');
      _isEmulatorCache = DeviceInfo.isEmulatorSync();
      return _isEmulatorCache;
    }
    // For iOS, always return false as iOS simulators work fine
    _isEmulatorCache = false;
    return false;
  } catch (error) {
    console.warn('Could not detect emulator, assuming real device:', error);
    _isEmulatorCache = false;
    return false;
  }
};

// Reusable Gradient Button Component
const GradientButton = ({ onPress, title, style, isLoading }) => (
  <LinearGradient colors={['#015185', '#3481BC']} style={[styles.gradientButton, style]}>
    <TouchableOpacity style={styles.buttonTouchable} onPress={onPress} activeOpacity={0.8} disabled={isLoading}>
      {isLoading ? (
        <ActivityIndicator size="small" color="#fff" />
      ) : (
        <Text style={styles.googleButtonText}>{title}</Text>
      )}
    </TouchableOpacity>
  </LinearGradient>
);

// Custom Animated Input Component
const AnimatedInput = ({
  label,
  value,
  onChangeText,
  secureTextEntry,
  keyboardType,
  autoCapitalize,
  leftIconName,
  rightIconName,
  onRightIconPress,
  hasError,
  onBlur,
  inputRef,
  onFocus,
}) => {
  // Move Dimensions inside component to fix Hermes runtime error
  const { width } = Dimensions.get('window');
  const scale = (size) => (width / 375) * size;

  const [isFocused, setIsFocused] = useState(false);
  const animatedLabel = useRef(new Animated.Value(value ? 1 : 0)).current;
  const animatedBorderWidth = useRef(new Animated.Value(1)).current;
  const animatedBorderColor = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    Animated.parallel([
      Animated.timing(animatedLabel, {
        toValue: isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderWidth, {
        toValue: isFocused ? 2 : 1,
        duration: 200,
        useNativeDriver: false,
      }),
      Animated.timing(animatedBorderColor, {
        toValue: hasError ? 2 : isFocused || value ? 1 : 0,
        duration: 200,
        useNativeDriver: false,
      }),
    ]).start();
  }, [isFocused, value, hasError]);

  const labelStyle = {
    position: 'absolute',
    left: scale(45),
    top: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(18), scale(-8)],
    }),
    fontSize: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: [scale(16), scale(12)],
    }),
    color: animatedLabel.interpolate({
      inputRange: [0, 1],
      outputRange: ['#888', '#015185'],
    }),
    backgroundColor: '#fff',
    paddingHorizontal: scale(4),
    zIndex: 2,
  };

  const borderColor = animatedBorderColor.interpolate({
    inputRange: [0, 1, 2],
    outputRange: ['#ddd', '#015185', '#ff0000'],
  });

  return (
    <View style={styles.inputContainer} ref={inputRef}>
      <Animated.View
        style={[
          styles.inputGradient,
          {
            borderWidth: animatedBorderWidth,
            borderColor: borderColor,
          },
        ]}
      >
        <Animated.Text style={labelStyle}>{label}</Animated.Text>
        <TextInput
          mode="outlined"
          value={value}
          onChangeText={onChangeText}
          secureTextEntry={secureTextEntry}
          keyboardType={keyboardType}
          autoCapitalize={autoCapitalize}
          style={styles.input}
          onFocus={() => {
            setIsFocused(true);
            onFocus && onFocus();
          }}
          onBlur={() => {
            setIsFocused(false);
            onBlur && onBlur();
          }}
          left={
            leftIconName && (
              <TextInput.Icon
                icon={() => <MaterialIcons name={leftIconName} size={scale(24)} color="#015185" />}
              />
            )
          }
          right={
            rightIconName && (
              <TextInput.Icon
                icon={() => (
                  <TouchableOpacity onPress={onRightIconPress}>
                    <MaterialIcons name={rightIconName} size={scale(24)} color="#015185" />
                  </TouchableOpacity>
                )}
              />
            )
          }
          outlineStyle={{
            borderWidth: 0,
            borderRadius: scale(8),
          }}
          theme={{
            colors: {
              primary: 'transparent',
              text: '#333',
              placeholder: '#888',
            },
          }}
        />
      </Animated.View>
    </View>
  );
};

// SyncModal Component
const SyncModal = ({ visible, onClose, message }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
            />
            <Text style={styles.syncModalTextadd}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

// Helper function to show toast messages based on platform
const showToast = (type, title, message) => {
  if (Platform.OS === 'android') {
    ToastAndroid.showWithGravity(message, ToastAndroid.LONG, ToastAndroid.BOTTOM);
  } else {
    Toast.show({
      type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  }
};

// Helper function to trigger welcome notification
const triggerWelcomeNotification = async (userName, userRole, userEmail) => {
  try {
    console.log('🎉 Triggering welcome notification for:', userName, userRole, userEmail);

    // Platform-specific notification handling
    if (Platform.OS === 'ios') {
      // Prefer native iOS manager if available, otherwise fall back safely
      try {
        if (iosNotificationManager?.isAvailable && iosNotificationManager.isAvailable()) {
          console.log('📱 iOS: Using native notification manager...');
          await iosNotificationManager.ensurePermissions();
          await iosNotificationManager.displayWelcomeNotification(userName, userRole);
          console.log('✅ iOS: Welcome notification displayed via native manager');
          showToast('success', 'Welcome!', `Hello ${userName || 'Admin'}! Welcome notification displayed.`);
        } else {
          console.warn('ℹ️ iOS notification bridge not available - falling back to notifee/toast');
          // Try notifee only if available
          await displayNotifeeNotification(userName, userRole, userEmail).catch(() => {
            showToast('success', 'Welcome!', `Hello ${userName || 'Admin'}! Welcome to Titan Drilling!`);
          });
        }
      } catch (iosError) {
        console.warn('⚠️ iOS native notification failed, falling back to notifee/toast:', iosError);
        await displayNotifeeNotification(userName, userRole, userEmail).catch(() => {
          showToast('success', 'Welcome!', `Hello ${userName || 'Admin'}! Welcome to Titan Drilling!`);
        });
      }
    } else {
      // Use notifee for Android if available
      await displayNotifeeNotification(userName, userRole, userEmail).catch((err) => {
        console.warn('⚠️ notifee notification failed:', err);
        try { showToast('success', 'Welcome!', `Hello ${userName || 'Admin'}!`); } catch (_) {}
      });
    }

    // Also send a push notification from Firebase to test FCM (best-effort)
    try {
      console.log('📱 Sending welcome push notification via FCM...');
      const response = await fetch('https://us-central1-titan-drilling-1f8e9.cloudfunctions.net/sendTestPushNotification', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          title: '🎉 Welcome to Titan Drilling!',
          body: `Hello ${userName || 'Admin'}! You are logged in as ${userRole || 'User'}`,
          data: {
            type: 'welcome_login',
            userName: userName || 'Admin',
            userRole: userRole || 'User',
            userEmail: userEmail || '',
            loginTime: new Date().toISOString(),
          },
        }),
      });

      const result = await response.json();
      if (result?.success) {
        console.log('✅ Welcome push notification sent successfully via FCM');
        showToast('success', 'Welcome!', 'Welcome notification sent successfully!');
      } else {
        console.warn('⚠️ Welcome push notification failed:', result?.error);
        showToast('info', 'Welcome', 'Welcome notification failed, but login was successful');
      }
    } catch (fcmError) {
      console.warn('⚠️ Could not send welcome push notification:', fcmError);
      showToast('info', 'Welcome', 'Welcome notification failed, but login was successful');
    }

  } catch (error) {
    console.error('❌ Error creating welcome notification:', error);
    try { showToast('success', 'Welcome!', `Hello ${userName || 'Admin'}! Welcome to Titan Drilling!`); } catch (_) {}
  }
};

// Helper function to display notifee notification (fallback for iOS, primary for Android)
const displayNotifeeNotification = async (userName, userRole, userEmail) => {
  try {
    // Create notification channels if they don't exist
    await notifee.createChannel({
      id: 'default',
      name: 'Default Channel',
      importance: AndroidImportance.HIGH,
      sound: 'default',
      vibration: true,
      lights: true,
    });

    console.log('✅ Default notification channel created/verified');
    
    // Create welcome notification with user details - use default channel for reliability
    await notifee.displayNotification({
      title: '🎉 Welcome to Titan Drilling!',
      body: `Hello ${userName || 'Admin'}! You are logged in as ${userRole || 'User'}`,
      android: {
        channelId: 'default', // Use default channel for better reliability
        importance: AndroidImportance.HIGH,
        pressAction: { id: 'welcome' },
        color: '#015185',
        largeIcon: 'ic_launcher',
        showTimestamp: true,
        timestamp: Date.now(),
        // Add custom data
        data: {
          type: 'welcome',
          userName: userName || 'Admin',
          userRole: userRole || 'User',
          userEmail: userEmail || '',
          loginTime: new Date().toISOString(),
        },
        // Make notification more prominent
        smallIcon: 'ic_notification',
        largeIcon: 'ic_launcher',
        color: '#015185',
        // Add sound and vibration
        sound: 'default',
        vibration: true,
        // Make it persistent
        ongoing: false,
        autoCancel: true,
        // Add priority
        priority: 'high',
      },
      ios: {
        categoryId: 'welcome',
        threadId: 'titan-drilling-welcome',
        // iOS specific settings
        sound: 'default',
        badge: 1,
      },
    });

    console.log('✅ Welcome notification displayed successfully via notifee');
    showToast('success', 'Welcome!', `Hello ${userName || 'Admin'}! Welcome notification displayed.`);
    
  } catch (error) {
    console.error('❌ Error creating notifee notification:', error);
    throw error;
  }
};

// Request notification permission (Android 13+ runtime + iOS authorization)
const requestNotificationPermission = async () => {
  try {
    console.log('🔐 Requesting notification permissions...');
    
    // Skip on emulator to prevent crashes
    if (isEmulator()) {
      console.log('ℹ️ Emulator detected - skipping notification permission request');
      return false;
    }
    
    if (Platform.OS === 'android' && Platform.Version >= 33) {
      console.log('📱 Android 13+ detected, requesting POST_NOTIFICATIONS permission...');
      try {
        const granted = await PermissionsAndroid.request(
          PermissionsAndroid.PERMISSIONS.POST_NOTIFICATIONS
        );
        console.log('📱 POST_NOTIFICATIONS permission result:', granted);
        if (granted !== PermissionsAndroid.RESULTS.GRANTED) {
          console.log('❌ POST_NOTIFICATIONS permission denied');
          return false;
        }
      } catch (permError) {
        console.warn('⚠️ Permission request failed:', permError.message);
        return false;
      }
    }
    
    console.log('📱 Requesting FCM permission...');
    try {
      const authStatus = await messaging().requestPermission();
      console.log('📱 FCM permission result:', authStatus);
      
      const enabled =
        authStatus === messaging.AuthorizationStatus.AUTHORIZED ||
        authStatus === messaging.AuthorizationStatus.PROVISIONAL;
      
      console.log('📱 FCM permission enabled:', enabled);
      return enabled;
    } catch (fcmError) {
      console.warn('⚠️ FCM permission request failed:', fcmError.message);
      return false;
    }
  } catch (error) {
    console.error('❌ Notification permission error:', error);
    return false;
  }
};

// Get FCM token and save under /Globalusers/admins/{userId} with multi-device support
// Only stores FCM token once per device per user - no refresh needed
const ensureFcmTokenAndSave = async (userId) => {
  try {
    console.log('🔐 Starting FCM token setup for user:', userId);
    
    // Check if running on emulator
    const runningOnEmulator = isEmulator();
    if (runningOnEmulator) {
      console.log('⚠️ Running on emulator - FCM token setup may be skipped or fail gracefully');
    }
    
    // Request notification permission
    const permission = await requestNotificationPermission();
    if (!permission) {
      console.log('❌ Notification permission denied');
      if (runningOnEmulator) {
        console.log('ℹ️ This is normal on emulator - continuing without FCM token');
        return { token: null, isEmulator: true };
      }
      return null;
    }
    console.log('✅ Notification permission granted');
    
    // Get FCM token with emulator-friendly error handling
    console.log('📱 Getting FCM token...');
    let token = null;
    try {
      token = await messaging().getToken();
    } catch (tokenError) {
      console.warn('⚠️ Failed to get FCM token:', tokenError.message);
      if (runningOnEmulator) {
        console.log('ℹ️ FCM token unavailable on emulator - continuing without it');
        return { token: null, isEmulator: true };
      }
      throw tokenError;
    }
    
    if (!token) {
      console.log('❌ Failed to get FCM token');
      if (runningOnEmulator) {
        console.log('ℹ️ This is normal on emulator - continuing without FCM token');
        return { token: null, isEmulator: true };
      }
      return null;
    }
    console.log('✅ FCM token obtained:', token.substring(0, 20) + '...');
    console.log('📱 Full FCM token length:', token.length);
    
    // Verify token format
    if (token.length < 100) {
      console.warn('⚠️ FCM token seems too short, might be invalid');
    }
    
    // Get device identifier (unique for this device)
    const deviceId = await getDeviceIdentifier();
    console.log('📱 Device ID:', deviceId);
    
    // Check if this device already has a stored FCM token
    const storedDeviceKey = await AsyncStorage.getItem('deviceKey');
    const storedDeviceId = await AsyncStorage.getItem('deviceId');
    const storedToken = await AsyncStorage.getItem('fcmToken');
    
    console.log('📱 Stored device info:', {
      deviceKey: storedDeviceKey,
      deviceId: storedDeviceId,
      hasToken: !!storedToken
    });
    
    // Save FCM token to database with multi-device support
    const userRef = database().ref(`/Globalusers/admins/${userId}`);
    
    try {
      // ⚡ FAST PATH: If we have stored device info and token matches, skip ALL database operations
      if (storedDeviceKey && storedDeviceId && storedToken && token === storedToken) {
        console.log('⚡ Fast path: Same device, same token - skipping all database operations for speed');
        return { 
          token: storedToken, 
          deviceKey: storedDeviceKey, 
          deviceId: storedDeviceId,
          isExistingDevice: true 
        };
      }
      
      // Need to check database for existing devices or token changed
      console.log('📊 Fetching existing user data...');
      const userSnapshot = await userRef.once('value');
      const userData = userSnapshot.val() || {};
      console.log('📋 User data retrieved:', Object.keys(userData));
      
      // Check if this device already exists and has the same token
      const existingDevices = userData.devices || {};
      let deviceKey = 'FCM'; // Default key
      let isNewDevice = false;
      let isTokenChanged = false;
      
      // Check if this is the same device with stored info (token must be different since fast path didn't catch it)
      if (storedDeviceKey && storedDeviceId && storedToken) {
        const existingDevice = existingDevices[storedDeviceKey];
        if (existingDevice && existingDevice.deviceInfo?.deviceId === storedDeviceId) {
          // Same device, different token - update token
          console.log('🔄 Same device, token changed - updating token');
          deviceKey = storedDeviceKey;
          isTokenChanged = true;
        } else {
          // Device info mismatch - treat as new device
          console.log('🔄 Device info mismatch - treating as new device');
          isNewDevice = true;
        }
      } else {
        // No stored device info - this is a new device
        console.log('🆕 No stored device info - this is a new device');
        isNewDevice = true;
      }
      
      // If it's a new device, find the next available device key
      if (isNewDevice) {
        // Find the next available device key (FCM, FCM2, FCM3, etc.)
        let counter = 1;
        while (existingDevices[`FCM${counter}`]) {
          counter++;
        }
        deviceKey = counter === 1 ? 'FCM' : `FCM${counter}`;
        console.log('🆕 Adding new device:', deviceKey);
      }
      
      // Prepare device data with improved iOS compatibility
      const deviceData = {
        token: token,
        lastTokenUpdate: new Date().toISOString(),
        deviceInfo: {
          platform: Platform.OS,
          version: Platform.Version,
          deviceId: deviceId,
          timestamp: new Date().toISOString(),
          // iOS specific info
          iosVersion: Platform.OS === 'ios' ? Platform.Version : null,
          deviceModel: Platform.OS === 'ios' ? 'iOS Device' : null,
          // Android specific info
          androidVersion: Platform.OS === 'android' ? Platform.Version : null,
          sdkVersion: Platform.OS === 'android' ? Platform.Version : null,
        },
        fcmStatus: 'active',
        lastLogin: new Date().toISOString(),
        isCurrentDevice: true, // Mark this as the current device
        // Additional metadata for better tracking
        appVersion: '1.0.0', // You can get this from your app config
        buildNumber: '1', // You can get this from your app config
        lastTokenValidation: new Date().toISOString(),
      };
      
      console.log('📱 Device data prepared:', deviceData);
      
      // Update the specific device using set() instead of update() for dynamic paths
      console.log(`🔄 Saving device data to path: /Globalusers/admins/${userId}/devices/${deviceKey}`);
      const deviceRef = userRef.child('devices').child(deviceKey);
      
      // Test database write permission first
      console.log('🧪 Testing database write permission...');
      await deviceRef.set(deviceData);
      console.log('✅ Device data saved successfully!');
      
      // Calculate active devices count correctly
      if (isNewDevice) {
        const activeDevicesCount = Object.keys(existingDevices).length + 1;
        console.log(`📊 Active devices count: ${activeDevicesCount}`);
        
        // Update user profile fields
        console.log('🔄 Updating user profile fields...');
        await userRef.update({
          lastLogin: new Date().toISOString(),
          activeDevices: activeDevicesCount,
          // Keep legacy token for backward compatibility
          NotificationFCMtoken: token,
          lastTokenUpdate: new Date().toISOString(),
        });
        console.log('✅ User profile updated successfully!');
      } else {
        // Just update lastLogin for existing device
        await userRef.update({
          lastLogin: new Date().toISOString(),
          // Keep legacy token updated for backward compatibility
          NotificationFCMtoken: token,
          lastTokenUpdate: new Date().toISOString(),
        });
        console.log('✅ User profile updated for existing device');
      }
      
      // Mark other devices as not current (do it in background for speed)
      setTimeout(() => {
        (async () => {
          try {
            for (const key of Object.keys(existingDevices)) {
              if (key !== deviceKey) {
                await userRef.child('devices').child(key).child('isCurrentDevice').set(false);
              }
            }
            console.log('✅ Other devices marked as inactive');
          } catch (markError) {
            console.warn('⚠️ Warning: Could not mark other devices as inactive:', markError);
          }
        })();
      }, 500);
      
      console.log(`✅ FCM token ${isNewDevice ? 'saved' : 'updated'} to database for device: ${deviceKey}`);
      
      // Store in AsyncStorage for local access
      try {
        await AsyncStorage.setItem('fcmToken', token);
        await AsyncStorage.setItem('fcmTokenTimestamp', new Date().toISOString());
        await AsyncStorage.setItem('deviceKey', deviceKey);
        await AsyncStorage.setItem('deviceId', deviceId);
        console.log('✅ FCM token stored locally');
      } catch (storageError) {
        console.warn('⚠️ Failed to store FCM token locally:', storageError);
      }
      
      return { 
        token, 
        deviceKey, 
        deviceId, 
        isExistingDevice: !isNewDevice,
        isTokenChanged 
      };
      
    } catch (dbError) {
      console.error('❌ Database operation failed:', dbError);
      console.error('❌ Error code:', dbError.code);
      console.error('❌ Error message:', dbError.message);
      
      // Check if it's a permission error
      if (dbError.code === 'PERMISSION_DENIED') {
        console.error('🚨 PERMISSION DENIED: Firebase database rules are blocking writes');
        console.error('🚨 Check your Firebase Console > Database > Rules');
      }
      
      throw dbError; // Re-throw to be caught by outer catch
    }
    
  } catch (error) {
    console.error('❌ Error ensuring FCM token:', error);
    console.error('❌ Full error details:', JSON.stringify(error, null, 2));
    return null;
  }
};

// Function to generate unique device identifier
const getDeviceIdentifier = async () => {
  try {
    // Try to get existing device ID from storage
    let deviceId = await AsyncStorage.getItem('deviceId');
    
    if (!deviceId) {
      // Generate new device ID if none exists
      deviceId = `device_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
      await AsyncStorage.setItem('deviceId', deviceId);
      console.log('🆔 Generated new device ID:', deviceId);
    }
    
    return deviceId;
  } catch (error) {
    console.error('❌ Error getting device identifier:', error);
    // Fallback device ID
    return `fallback_${Date.now()}`;
  }
};











export default function AdminLoginScreen() {
  const navigation = useNavigation();
  const [emailOrEmployeeId, setEmailOrEmployeeId] = useState('');
  const [password, setPassword] = useState('');
  const [showPassword, setShowPassword] = useState(false);
  const [emailOrEmployeeIdError, setEmailOrEmployeeIdError] = useState(false);
  const [emailOrEmployeeIdErrorMessage, setEmailOrEmployeeIdErrorMessage] = useState('');
  const [passwordError, setPasswordError] = useState(false);
  const [passwordErrorMessage, setPasswordErrorMessage] = useState('');
  const [isLoadingEmail, setIsLoadingEmail] = useState(false);
  const [isLoadingGoogle, setIsLoadingGoogle] = useState(false);
  const [isNetworkError, setIsNetworkError] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);

  const networkErrorAnim = useRef(new Animated.Value(-scale(50))).current;
  const scrollViewRef = useRef(null);
  const emailOrEmployeeIdInputRef = useRef(null);
  const passwordInputRef = useRef(null);

  const showNetworkError = () => {
    setIsNetworkError(true);
    Animated.timing(networkErrorAnim, {
      toValue: 0,
      duration: 300,
      useNativeDriver: true,
    }).start();

    setTimeout(() => {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }, 3000);
  };

  const hideNetworkError = () => {
    if (isNetworkError) {
      Animated.timing(networkErrorAnim, {
        toValue: -scale(50),
        duration: 300,
        useNativeDriver: true,
      }).start(() => setIsNetworkError(false));
    }
  };

  useEffect(() => {
    const initialize = async () => {
      try {
        // Wrap emulator check in try-catch to prevent crashes
        let runningOnEmulator = false;
        try {
          runningOnEmulator = isEmulator();
          console.log('📱 Running on emulator:', runningOnEmulator);
        } catch (emulatorCheckError) {
          console.warn('⚠️ Could not check if emulator:', emulatorCheckError.message);
        }
        
        // Configure Google Sign-in with emulator-friendly error handling
        if (!runningOnEmulator) {
          try {
            await GoogleSignin.configure({
              webClientId: '749788854975-rhkmepo20c1n038ftnreustj8br62qbj.apps.googleusercontent.com',
              androidClientId: '749788854975-1vqr2rje9a19hl56ohheqvr32hl6045p.apps.googleusercontent.com',
              iosClientId: '749788854975-fsan3f653h4fffue813uqc38h6mu7ipe.apps.googleusercontent.com',
              offlineAccess: true,
            });
            console.log('✅ Google Sign-in configured successfully');
          } catch (googleError) {
            console.warn('⚠️ Google Sign-in configuration failed (may be normal on emulator):', googleError.message);
            // Don't throw error - allow app to continue
          }
        } else {
          console.log('ℹ️ Skipping Google Sign-in configuration on emulator');
        }
        
        try {
          await checkNetworkStatus();
        } catch (networkError) {
          console.warn('⚠️ Network check failed:', networkError.message);
        }
        
        // FCM token is now stored once per device per user - no need to check/refresh on startup
        console.log('ℹ️ FCM token will be verified/stored on next login if needed');
        

      } catch (error) {
        console.error('Initialization error:', error);
        // Don't show error on emulator for initialization issues
        try {
          if (!isEmulator()) {
            showNetworkError();
            showToast('error', 'Initialization Error', 'Failed to initialize');
          }
        } catch (toastError) {
          console.error('Could not show error toast:', toastError);
        }
      }
    };

    initialize();

    // Listen for app state changes to refresh FCM token
    const handleAppStateChange = (nextAppState) => {
      if (nextAppState === 'active') {
        console.log('🔄 App became active, checking FCM token...');
        // No need to refresh FCM token - it's stored once per device per user
        // Just log the app state change for debugging
        console.log('ℹ️ App state changed to active - FCM token refresh not needed');
      }
    };

    const subscription = AppState.addEventListener('change', handleAppStateChange);

    return () => {
      GoogleSignin.signOut().catch((error) => console.error('Google Sign-In cleanup error:', error));
      subscription?.remove();
    };
  }, []);

  // Function to check and refresh FCM token on app startup
  const checkAndRefreshFCMToken = async () => {
    try {
      console.log('🔄 Checking FCM token on app startup...');
      
      // Check if we have a stored FCM token
      const storedToken = await AsyncStorage.getItem('fcmToken');
      const storedTimestamp = await AsyncStorage.getItem('fcmTokenTimestamp');
      
      if (storedToken && storedTimestamp) {
        console.log('✅ FCM token already stored for this device');
        console.log(`📱 Token stored on: ${storedTimestamp}`);
        
        // No need to refresh - FCM tokens are stored once per device per user
        // Firebase handles token refresh automatically when needed
        console.log('ℹ️ FCM token refresh not needed - stored once per device');
      } else {
        console.log('ℹ️ No stored FCM token found - will be stored on next login');
      }
    } catch (error) {
      console.error('❌ Error checking FCM token:', error);
    }
  };

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) {
        showNetworkError();
        showToast('error', 'Network Error', 'Internet connection not available');
        return false;
      } else {
        // Hide network error if network is available
        hideNetworkError();
        return true;
      }
    } catch (error) {
      console.error('Network check error:', error);
      showNetworkError();
      showToast('error', 'Network Error', 'Failed to check network status');
      return false;
    }
  };

  const validateEmailOrEmployeeId = (input) => {
    const inputEmpty = !input.trim();
    const isEmail = /\S+@\S+\.\S+/.test(input);
    const isEmployeeId = input.length >= 1; // Any text or number with at least 1 character
    const inputTooLong = input.length > 100; // Increased limit for flexibility

    setEmailOrEmployeeIdError(inputEmpty || (!isEmail && !isEmployeeId) || inputTooLong);
    if (inputEmpty) {
      setEmailOrEmployeeIdErrorMessage('Email/Employee ID is required');
    } else if (!isEmail && !isEmployeeId) {
      setEmailOrEmployeeIdErrorMessage('Enter a valid email address or employee ID');
    } else if (inputTooLong) {
      setEmailOrEmployeeIdErrorMessage('Input cannot exceed 100 characters');
    } else {
      setEmailOrEmployeeIdErrorMessage('');
    }

    return !(inputEmpty || (!isEmail && !isEmployeeId) || inputTooLong);
  };

  const validatePassword = (password) => {
    const passwordEmpty = !password.trim();

    setPasswordError(passwordEmpty);
    if (passwordEmpty) {
      setPasswordErrorMessage('Password is required');
    } else {
      setPasswordErrorMessage('');
    }

    return !passwordEmpty;
  };

  const findUserByEmailOrEmployeeId = async (input) => {
    try {
      const adminsRef = database().ref('/Globalusers/admins');
      
      // ⚡ OPTIMIZED: Use indexed query for faster search (only on Android devices)
      console.log('🔍 Searching for user:', input);
      
      // First, try to search by email using orderByChild query (much faster than fetching all)
      try {
        const emailQuery = adminsRef.orderByChild('email').equalTo(input).limitToFirst(1);
        const emailSnapshot = await emailQuery.once('value');
        
        if (emailSnapshot.exists()) {
          const userId = Object.keys(emailSnapshot.val())[0];
          const userData = emailSnapshot.val()[userId];
          console.log('✅ User found by email:', userId);
          return { userId, userData };
        }
      } catch (emailError) {
        console.warn('⚠️ Email query failed, trying employee number:', emailError.message);
      }
      
      // If not found by email, try employee number
      try {
        const empQuery = adminsRef.orderByChild('employeeNumber').equalTo(input).limitToFirst(1);
        const empSnapshot = await empQuery.once('value');
        
        if (empSnapshot.exists()) {
          const userId = Object.keys(empSnapshot.val())[0];
          const userData = empSnapshot.val()[userId];
          console.log('✅ User found by employee number:', userId);
          return { userId, userData };
        }
      } catch (empError) {
        console.warn('⚠️ Employee number query failed:', empError.message);
      }
      
      console.log('❌ User not found');
      return null;
    } catch (error) {
      console.error('❌ Error finding user:', error);
      return null;
    }
  };

  const handleLogin = async () => {
    const isInputValid = validateEmailOrEmployeeId(emailOrEmployeeId);
    const isPasswordValid = validatePassword(password);

    if (!isInputValid || !isPasswordValid) {
      return;
    }

    setIsLoadingEmail(true);
    setSyncModalVisible(true);

    try {
      const startTime = Date.now();
      console.log('⏱️ Login started at:', new Date().toISOString());
      
      if (!(await checkNetworkStatus())) {
        setIsLoadingEmail(false);
        setSyncModalVisible(false);
        return;
      }
      console.log('⏱️ Network check took:', Date.now() - startTime, 'ms');

      // First, try to find user by email or employee ID
      const searchStartTime = Date.now();
      const userInfo = await findUserByEmailOrEmployeeId(emailOrEmployeeId);
      console.log('⏱️ User search took:', Date.now() - searchStartTime, 'ms');
      
      if (!userInfo) {
        showToast('error', 'Login Error', 'User not found with provided email/employee ID');
        setIsLoadingEmail(false);
        setSyncModalVisible(false);
        return;
      }

      const { userId, userData } = userInfo;

      // Try to authenticate with Firebase using the found email
      const authStartTime = Date.now();
      console.log('⏱️ Starting Firebase authentication...');
      const userCredential = await auth().signInWithEmailAndPassword(userData.email, password);
      console.log('⏱️ Firebase auth took:', Date.now() - authStartTime, 'ms');
      
      if (userCredential.user.uid !== userId) {
        throw new Error('User authentication mismatch');
      }

      const userRole = userData.role;

      if (userRole !== 'ProAdmin') {
        await auth().signOut();
        showToast('error', 'Access Denied', 'Please login from user dashboard');
        setIsLoadingEmail(false);
        setSyncModalVisible(false);
        return;
      }

      // Store session data immediately for quick login
      const storageStartTime = Date.now();
      await Promise.all([
        AsyncStorage.setItem('userUid', userId),
        AsyncStorage.setItem('userEmail', userData.email),
        AsyncStorage.setItem('Loginsuccess-01', 'true'),
        AsyncStorage.setItem('ProAdminLogined', 'true'),
        AsyncStorage.setItem('adminPassword', password),
        AsyncStorage.removeItem('TeamMemberLogined'),
      ]);
      console.log('⏱️ Storage operations took:', Date.now() - storageStartTime, 'ms');
      console.log('⏱️ Total login time:', Date.now() - startTime, 'ms');

      // Show success and navigate immediately
      showToast('success', 'Success', 'Logged in successfully');
      setIsLoadingEmail(false);
      setSyncModalVisible(false);
      console.log('⏱️ Navigating to MainApp...');
      navigation.replace('MainApp');

      // Run all background tasks AFTER navigation (fire and forget)
      setTimeout(() => {
        (async () => {
          try {
            console.log('🔄 Running background tasks after login...');
            
            const userRef = database().ref(`/Globalusers/admins/${userId}`);
            
            // Update login status
            await userRef.update({
              loginStatus: true,
              lastLogin: new Date().toISOString(),
              firstLoginCompleted: true,
            });

            // Quick FCM token update (optimized)
            console.log('🔄 Updating FCM token in background...');
            const fcmResult = await ensureFcmTokenAndSave(userId);
            if (fcmResult && fcmResult.token) {
              console.log(`✅ FCM token ${fcmResult.isExistingDevice ? 'verified' : 'saved'} for device: ${fcmResult.deviceKey}`);
            }

            // Subscribe to topic
            if (!isEmulator()) {
              try { 
                await messaging().subscribeToTopic(`uid_${userId}`); 
                console.log('✅ Subscribed to topic:', `uid_${userId}`);
              } catch (e) { 
                console.warn('⚠️ subscribeToTopic error:', e?.message || e); 
              }
            }

            // Trigger welcome notification (no await, fire and forget)
            triggerWelcomeNotification(userData.name, userData.role, userData.email).catch(err => {
              console.warn('⚠️ Welcome notification failed:', err.message);
            });

          } catch (bgError) {
            console.warn('⚠️ Background task error:', bgError);
          }
        })();
      }, 100); // Small delay to ensure navigation completes first

    } catch (error) {
      console.error('Login error:', error);
      const errorMessages = {
        'auth/invalid-email': 'Invalid email address.',
        'auth/wrong-password': 'Incorrect password.',
        'auth/user-not-found': 'No user found with this email.',
        'auth/too-many-requests': 'Too many attempts. Please try again later.',
        'auth/invalid-credential': 'Invalid credentials provided.',
        default: error.message || 'An unexpected error occurred.',
      };
      showToast('error', 'Login Error', errorMessages[error.code] || errorMessages.default);
      setIsLoadingEmail(false);
      setSyncModalVisible(false);
    }
  };

  const handleGoogleLogin = async () => {
    setIsLoadingGoogle(true);
    setSyncModalVisible(true);

    try {
      if (!(await checkNetworkStatus())) {
        setIsLoadingGoogle(false);
        setSyncModalVisible(false);
        return;
      }

      // Check Google Play Services with emulator-friendly error handling
      try {
        await GoogleSignin.hasPlayServices({ showPlayServicesUpdateDialog: true });
      } catch (playServicesError) {
        console.error('Play Services error:', playServicesError);
        if (isEmulator()) {
          showToast('error', 'Emulator Limitation', 'Google Sign-in not available on emulator. Please use email/password login.');
        } else {
          showToast('error', 'Google Play Services', 'Google Play Services not available. Please update Google Play Services.');
        }
        setIsLoadingGoogle(false);
        setSyncModalVisible(false);
        return;
      }
      
      const signInResult = await GoogleSignin.signIn();
      const idToken = signInResult.data?.idToken || signInResult.idToken;

      if (!idToken) {
        throw new Error('No ID token found in Google Sign-In response');
      }

      const googleCredential = auth.GoogleAuthProvider.credential(idToken);
      const userCredential = await auth().signInWithCredential(googleCredential);
      const userId = userCredential.user.uid;

      const userRef = database().ref(`/Globalusers/admins/${userId}`);
      const snapshot = await userRef.once('value');

      if (!snapshot.exists()) {
        throw new Error('User not found in admin database');
      }

      const userData = snapshot.val();

      // Role check
      const userRole = userData.role;
      if (userRole !== 'ProAdmin') {
        await GoogleSignin.signOut();
        await auth().signOut();
        showToast('error', 'Access Denied', 'Please login from user dashboard');
        setIsLoadingGoogle(false);
        setSyncModalVisible(false);
        return;
      }

      // Store session data immediately for quick login
      await Promise.all([
        AsyncStorage.setItem('userUid', userId),
        AsyncStorage.setItem('userEmail', userCredential.user.email),
        AsyncStorage.setItem('Loginsuccess-01', 'true'),
        AsyncStorage.setItem('ProAdminLogined', 'true'),
        AsyncStorage.removeItem('TeamMemberLogined'),
      ]);

      // Show success and navigate immediately
      showToast('success', 'Success', 'Logged in with Google successfully');
      setIsLoadingGoogle(false);
      setSyncModalVisible(false);
      navigation.replace('MainApp');

      // Run all background tasks AFTER navigation (fire and forget)
      setTimeout(() => {
        (async () => {
          try {
            console.log('🔄 Running background tasks after Google login...');
            
            // Update login status
            await userRef.update({
              loginStatus: true,
              lastLogin: new Date().toISOString(),
              firstLoginCompleted: true,
            });

            // Quick FCM token update (optimized)
            console.log('🔄 Updating FCM token in background...');
            const fcmResult = await ensureFcmTokenAndSave(userId);
            if (fcmResult && fcmResult.token) {
              console.log(`✅ FCM token ${fcmResult.isExistingDevice ? 'verified' : 'saved'} for device: ${fcmResult.deviceKey}`);
            }

            // Subscribe to topic
            if (!isEmulator()) {
              try { 
                await messaging().subscribeToTopic(`uid_${userId}`); 
                console.log('✅ Subscribed to topic:', `uid_${userId}`);
              } catch (e) { 
                console.warn('⚠️ subscribeToTopic error:', e?.message || e); 
              }
            }

            // Trigger welcome notification (no await, fire and forget)
            triggerWelcomeNotification(userData.name, userData.role, userData.email).catch(err => {
              console.warn('⚠️ Welcome notification failed:', err.message);
            });

          } catch (bgError) {
            console.warn('⚠️ Background task error:', bgError);
          }
        })();
      }, 100); // Small delay to ensure navigation completes first

    } catch (error) {
      console.error('Google Login error:', error);
      await GoogleSignin.signOut();
      await auth().signOut();
      const errorMessages = {
        'auth/account-exists-with-different-credential': 'Account exists with different credential.',
        'auth/invalid-credential': 'Invalid Google credential.',
        'auth/operation-not-allowed': 'Google Sign-In not enabled.',
        10: 'Developer Error: Check SHA-1 fingerprint in Firebase Console.',
        'auth/network-request-failed': 'Network error: Please check your internet connection.',
        default: error.message || 'An unexpected error occurred.',
      };
      showToast('error', 'Google Login Error', errorMessages[error.code] || errorMessages.default);
      setIsLoadingGoogle(false);
      setSyncModalVisible(false);
    }
  };

  const toggleShowPassword = () => {
    setShowPassword(!showPassword);
  };

  const handleInputFocus = (inputRef) => {
    if (inputRef.current && scrollViewRef.current) {
      inputRef.current.measureLayout(
        scrollViewRef.current.getScrollResponder(),
        (x, y) => {
          scrollViewRef.current.scrollTo({ y: y - scale(100), animated: true });
        },
        (error) => {
          console.warn('Failed to measure layout:', error);
          scrollViewRef.current.scrollTo({ y: 0, animated: true });
        }
      );
    }
  };

  return (
    <SafeAreaView style={styles.container}>
      {isNetworkError && (
        <Animated.View
          style={[styles.networkErrorBanner, { transform: [{ translateY: networkErrorAnim }] }]}
        >
          <Text style={styles.networkErrorText}>Internet connection not available</Text>
        </Animated.View>
      )}

      <KeyboardAvoidingView
        behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
        keyboardVerticalOffset={Platform.OS === 'ios' ? scale(60) : scale(20)}
        style={styles.keyboardAvoidingContainer}
      >
        <ScrollView
          ref={scrollViewRef}
          contentContainerStyle={styles.scrollContentContainer}
          showsVerticalScrollIndicator={false}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.contentContainer}>
            <View style={styles.logoContainer}>
              <Image
                source={require('../../Images/adminlogin/logo.png')}
                style={styles.logo}
                resizeMode="contain"
              />
              <Text style={styles.loginText}>Admin Login</Text>
            </View>
            

            
            <View style={styles.formContainer}>
              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={emailOrEmployeeIdInputRef}
                  label="Email/Employee ID"
                  value={emailOrEmployeeId}
                  onChangeText={(text) => {
                    setEmailOrEmployeeId(text);
                    validateEmailOrEmployeeId(text);
                  }}
                  keyboardType="email-address"
                  autoCapitalize="none"
                  leftIconName="person"
                  hasError={emailOrEmployeeIdError}
                  onBlur={() => validateEmailOrEmployeeId(emailOrEmployeeId)}
                  onFocus={() => handleInputFocus(emailOrEmployeeIdInputRef)}
                />
                {emailOrEmployeeIdError && <Text style={styles.errorText}>{emailOrEmployeeIdErrorMessage}</Text>}
              </View>

              <View style={styles.mainemailpass}>
                <AnimatedInput
                  inputRef={passwordInputRef}
                  label="Password"
                  value={password}
                  onChangeText={(text) => {
                    setPassword(text);
                    validatePassword(text);
                  }}
                  secureTextEntry={!showPassword}
                  autoCapitalize="none"
                  leftIconName="lock"
                  rightIconName={showPassword ? 'visibility' : 'visibility-off'}
                  onRightIconPress={toggleShowPassword}
                  hasError={passwordError}
                  onBlur={() => validatePassword(password)}
                  onFocus={() => handleInputFocus(passwordInputRef)}
                />
                {passwordError && <Text style={styles.errorText}>{passwordErrorMessage}</Text>}
              </View>

              <GradientButton title="Login" onPress={handleLogin} isLoading={isLoadingEmail} />
            

            
              <View style={styles.dividerContainer}>
                <View style={styles.borderLine} />
                <TouchableOpacity onPress={() => navigation.navigate('SignUp')}>
                  <Text style={styles.dividerText}>Need an account? Sign Up</Text>
                </TouchableOpacity>
                <View style={styles.borderLine} />
              </View>
            </View>
          </View>
        </ScrollView>
      </KeyboardAvoidingView>

      <SyncModal
        visible={syncModalVisible}
        onClose={() => setSyncModalVisible(false)}
        message="Processing Login"
      />
      <Toast />
    </SafeAreaView>
  );
}

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
    paddingHorizontal: scale(6),
    marginTop: scale(24),
  },
  keyboardAvoidingContainer: {
    flex: 1,
  },
  scrollContentContainer: {
    flexGrow: 1,
    justifyContent: 'center',
    paddingBottom: scale(20),
  },
  contentContainer: {
    flex: 1,
    justifyContent: 'center',
  },
  logoContainer: {
    alignItems: 'center',
    marginBottom: scale(20),
  },
  logo: {
    width: '80%',
    height: scale(100),
    resizeMode: 'contain',
  },
  loginText: {
    fontSize: scale(24),
    fontWeight: 'bold',
    color: '#015185',
    marginTop: scale(10),
  },
  formContainer: {
    width: '100%',
  },
  inputContainer: {
    marginBottom: scale(8),
    marginHorizontal: scale(10),
  },
  inputGradient: {
    borderRadius: scale(8),
    backgroundColor: '#fff',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
  },
  input: {
    backgroundColor: '#fff',
    height: scale(56),
    borderRadius: scale(8),
    fontSize: scale(16),
  },
  gradientButton: {
    borderRadius: scale(6),
    marginVertical: scale(10),
    width: '95%',
    justifyContent: 'center',
    alignSelf: 'center',
  },
  buttonTouchable: {
    alignItems: 'center',
    justifyContent: 'center',
    height: scale(48),
    borderRadius: scale(6),
    width: '100%',
  },
  googleButtonText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  dividerContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    width: '100%',
    marginVertical: scale(20),
  },
  borderLine: {
    flex: 1,
    height: 1,
    borderStyle: 'dashed',
    borderWidth: 1,
    borderColor: '#888',
  },
  dividerText: {
    fontSize: scale(16),
    color: '#888',
    fontWeight: 'bold',
    textAlign: 'center',
    paddingHorizontal: scale(10),
  },
  networkErrorBanner: {
    position: 'absolute',
    top: 0,
    left: 0,
    right: 0,
    backgroundColor: '#ff0000',
    paddingVertical: scale(10),
    alignItems: 'center',
    zIndex: 1000,
  },
  networkErrorText: {
    color: '#fff',
    fontSize: scale(16),
    fontWeight: 'bold',
  },
  errorText: {
    color: 'red',
    fontSize: scale(14),
    marginLeft: scale(10),
    marginTop: scale(4),
    position: 'absolute',
    bottom: scale(-10),
  },
  mainemailpass: {
    marginBottom: scale(12),
    position: 'relative',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: '#fff',
    borderRadius: scale(10),
    padding: scale(20),
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  syncModalText: {
    fontSize: scale(18),
    marginBottom: scale(10),
    color: '#333',
    fontWeight: '600',
  },
  syncIcon: {
    width: scale(24),
    height: scale(24),
    marginRight: scale(8),
  },
  syncModalTextadd: {
    fontSize: scale(14),
    color: '#rijk333',
    fontWeight: '500',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },

});