import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  Dimensions,
  TouchableOpacity,
  ScrollView,
  Modal,
  Animated,
  StatusBar,
} from 'react-native';
import { useNavigation, useRoute } from '@react-navigation/native';
import DropDownPicker from 'react-native-dropdown-picker';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import Toast from 'react-native-toast-message';
import NetInfo from '@react-native-community/netinfo';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.3;

const TEpremissionR = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { member } = route.params || {}; // Get the selected team member

  const [countries, setCountries] = useState([]);
  const [countryOpen, setCountryOpen] = useState(false);
  const [countryItems, setCountryItems] = useState([
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ]);

  const [projects, setProjects] = useState([]);
  const [projectOpen, setProjectOpen] = useState(false);
  const [projectItems, setProjectItems] = useState([]);

  const [modules, setModules] = useState([]);
  const [moduleOpen, setModuleOpen] = useState(false);
  const [moduleItems, setModuleItems] = useState([
    { label: 'Equipments', value: 'Equipments' },
    { label: 'Documents', value: 'Documents' },
    { label: 'Team Employ', value: 'teamEmploy' },
    { label: 'Inspection Report', value: 'inspectionReport' },
    { label: 'HSE Inspection', value: 'hseInspection' },
    { label: 'DVIR Report', value: 'dvirReport' },
    { label: 'Checklist', value: 'checklist' },
    { label: 'Incident Report', value: 'incidentReport' },
    { label: 'Daily Report', value: 'dailyReport' },
    { label: 'Request Maintenance', value: 'requestMaintenance' },
    { label: 'Jobcard', value: 'jobcard' },
    { label: 'Service Schedule', value: 'serviceSchedule' },
    { label: 'Fuel Log', value: 'fuelLog' },
    { label: 'Logbook', value: 'logbook' },
    { label: 'HR Form', value: 'hrForm' },
    { label: 'HR Applications', value: 'hrApplications' },
    { label: 'Team KPI', value: 'teamKpi' },
  ]);

  const permissionOptions = [
    { label: 'All Permissions', value: 'all' },
    { label: 'Create', value: 'create' },
    { label: 'Edit', value: 'edit' },
    { label: 'Delete', value: 'delete' },
    { label: 'View', value: 'view' },
    { label: 'Update', value: 'update' },
    { label: 'Only Mine View', value: 'onlyMineView' },
  ];

  const individualPermissions = ['create', 'edit', 'delete', 'view', 'update', 'onlyMineView'];

  const [modulePermissions, setModulePermissions] = useState({});
  const [permissionOpen, setPermissionOpen] = useState({});
  const permissionRefs = useRef({});
  const [errors, setErrors] = useState({
    country: false,
    project: false,
    module: false,
    permissions: {},
  });
  const [isOnline, setIsOnline] = useState(true);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [isLoadingPermissions, setIsLoadingPermissions] = useState(true);
  const [showModuleSelectionModal, setShowModuleSelectionModal] = useState(false);
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const showToast = (type, title, message) => {
    Toast.show({
      type: type,
      text1: title,
      text2: message,
      position: 'bottom',
    });
  };

  // Load existing permissions for the team member
  useEffect(() => {
    const initialize = async () => {
      console.log('TEpremissionR initialized with member:', member);
      console.log('Member ID:', member?.id);
      console.log('Member data structure:', JSON.stringify(member, null, 2));
      
      if (!member?.id) {
        showToast('error', 'Error', 'No team member selected.');
        navigation.goBack();
        return;
      }

      // Initialize with default modules if none exist
      const initializeDefaultModules = () => {
        const defaultModules = [
          'Equipments', 'Documents', 'teamEmploy', 'inspectionReport', 
          'hseInspection', 'dvirReport', 'checklist', 'incidentReport', 
          'dailyReport', 'requestMaintenance', 'jobcard', 'serviceSchedule', 
          'fuelLog', 'logbook', 'hrForm', 'hrApplications', 'teamKpi'
        ];
        
        if (modules.length === 0) {
          setModules(defaultModules);
          const defaultPermissions = {};
          defaultModules.forEach(mod => {
            defaultPermissions[mod] = [];
          });
          setModulePermissions(defaultPermissions);
        }
      };

      const online = await checkNetworkStatus();
      setIsOnline(online);

      if (online) {
        const adminUid = await AsyncStorage.getItem('userUid');
        if (!adminUid) {
          showToast('error', 'Error', 'Admin not authenticated.');
          navigation.navigate('AdminLoginScreen');
          return;
        }

        // Load existing permissions from the main user database
        const userRef = database().ref(`/Globalusers/admins/${member.id}`);
        userRef.once('value', (snapshot) => {
          const data = snapshot.val();
          if (data) {
            console.log('Raw data from Firebase:', data);
            
            // Set existing countries and projects - ensure they are arrays
            const existingCountries = Array.isArray(data.countries) ? data.countries : [];
            const existingProjects = Array.isArray(data.projects) ? data.projects : [];
            
            setCountries(existingCountries);
            setProjects(existingProjects);
            
            // Set existing modules - handle both old and new formats
            // Show all modules that exist in the user's data, even if no permissions
            let existingModules = [];
            const permissions = {};
            
            if (data.modules && Array.isArray(data.modules)) {
              data.modules.forEach((mod) => {
                const moduleName = mod.module || mod;
                if (moduleName) {
                  existingModules.push(moduleName);
                  permissions[moduleName] = mod.permissions || [];
                }
              });
            }
            
            setModules(existingModules);
            setModulePermissions(permissions);
            
            // Initialize refs with loaded permissions
            existingModules.forEach((mod) => {
              permissionRefs.current[mod] = permissions[mod] || [];
            });
            
            console.log('Processed existing permissions:', {
              countries: existingCountries,
              projects: existingProjects,
              modules: existingModules,
              permissions: permissions,
              memberId: member.id
            });
          } else {
            console.log('No data found for member:', member.id);
          }
          
          // Initialize default modules if none exist
          initializeDefaultModules();
          setIsLoadingPermissions(false);
        }).catch((error) => {
          console.error('Error loading permissions:', error);
          setIsLoadingPermissions(false);
        });
      } else {
        // Try to load from cached user data first
        const cachedUserData = await AsyncStorage.getItem(`user_${member.id}`);
        if (cachedUserData) {
          const data = JSON.parse(cachedUserData);
          console.log('Cached user data:', data);
          
          // Ensure arrays for countries and projects
          const existingCountries = Array.isArray(data.countries) ? data.countries : [];
          const existingProjects = Array.isArray(data.projects) ? data.projects : [];
          
          setCountries(existingCountries);
          setProjects(existingProjects);
          
          // Handle both old and new module formats
          // Show all modules that exist in the user's data, even if no permissions
          let existingModules = [];
          const permissions = {};
          
          if (data.modules && Array.isArray(data.modules)) {
            data.modules.forEach((mod) => {
              const moduleName = mod.module || mod;
              if (moduleName) {
                existingModules.push(moduleName);
                permissions[moduleName] = mod.permissions || [];
              }
            });
          }
          
          setModules(existingModules);
          setModulePermissions(permissions);
          
          // Initialize refs with loaded permissions
          existingModules.forEach((mod) => {
            permissionRefs.current[mod] = permissions[mod] || [];
          });
          
          // Initialize default modules if none exist
          initializeDefaultModules();
          showToast('info', 'Offline Mode', 'Loaded cached user data.');
        } else {
          // Fallback to old permissions cache
          const cachedPermissions = await AsyncStorage.getItem(`permissions_${member.id}`);
          if (cachedPermissions) {
            const data = JSON.parse(cachedPermissions);
            console.log('Old cached permissions:', data);
            
            // Ensure arrays for countries and projects
            const existingCountries = Array.isArray(data.countries) ? data.countries : [];
            const existingProjects = Array.isArray(data.projects) ? data.projects : [];
            
            setCountries(existingCountries);
            setProjects(existingProjects);
            
                      // Handle both old and new module formats
          // Show all modules that exist in the user's data, even if no permissions
          let existingModules = [];
          const permissions = {};
          
          if (data.modules && Array.isArray(data.modules)) {
            data.modules.forEach((mod) => {
              const moduleName = mod.module || mod;
              if (moduleName) {
                existingModules.push(moduleName);
                permissions[moduleName] = mod.permissions || [];
              }
            });
          }
          
          setModules(existingModules);
          setModulePermissions(permissions);
          
          // Initialize refs with loaded permissions
          existingModules.forEach((mod) => {
            permissionRefs.current[mod] = permissions[mod] || [];
          });
          
          // Initialize default modules if none exist
          initializeDefaultModules();
          showToast('info', 'Offline Mode', 'Loaded cached permissions.');
          }
        }
        setIsLoadingPermissions(false);
      }

      const unsubscribeNetInfo = NetInfo.addEventListener(async (state) => {
        const newOnlineStatus = state.isConnected;
        setIsOnline(newOnlineStatus);
        if (newOnlineStatus) {
          await syncPendingPermissions();
        }
      });

      return () => unsubscribeNetInfo();
    };

    initialize();
  }, [member, navigation]);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const syncPendingPermissions = async () => {
    if (!isOnline) return;

    try {
      const storedPending = await AsyncStorage.getItem('pendingPermissions');
      const actions = storedPending ? JSON.parse(storedPending) : [];

      if (actions.length === 0) return;

      const adminUid = await AsyncStorage.getItem('userUid');
      for (const action of actions) {
        if (action.memberId === member.id) {
          // Save to the main user database
          await database()
            .ref(`/Globalusers/admins/${action.memberId}`)
            .update({
              countries: action.permissions.countries,
              projects: action.permissions.projects,
              modules: action.permissions.modules
            });
          
          // Also save to team member info for backward compatibility
          await database()
            .ref(`/GlobalTeamMemberInformation/${action.memberId}`)
            .update({
              countries: action.permissions.countries,
              projects: action.permissions.projects,
              modules: action.permissions.modules
            });
        }
      }

      await AsyncStorage.setItem('pendingPermissions', JSON.stringify([]));
      showToast('success', 'Success', 'Pending permissions synced.');
    } catch (error) {
      console.error('Error syncing pending permissions:', error);
      showToast('error', 'Error', 'Failed to sync permissions.');
    }
  };

  // Populate projects based on countries
  useEffect(() => {
    const projectMap = {
      Zambia: [
        { label: 'Kansanshi', value: 'Kansanshi' },
        { label: 'Kalumbila', value: 'Kalumbila' },
        { label: 'Kimteto', value: 'Kimteto' },
        { label: 'Kobold', value: 'Kobold' },
        { label: 'FQM Exploration', value: 'FQM Exploration' },
        { label: 'Mimosa', value: 'Mimosa' },
      ],
      DRC: [
        { label: 'Lubumbashi', value: 'Lubumbashi' },
        { label: 'Musompo', value: 'Musompo' },
        { label: 'IME', value: 'IME' },
        { label: 'Kamoa', value: 'Kamoa' },
      ],
    };

    const updatedProjects = countries
      .map((country) => projectMap[country] || [])
      .flat()
      .filter(
        (proj, index, self) =>
          index === self.findIndex((p) => p.value === proj.value)
      );

    setProjectItems(updatedProjects);
    const validProjects = projects.filter((proj) =>
      updatedProjects.some((item) => item.value === proj)
    );
    setProjects(validProjects);
  }, [countries]);

  // Update module permissions structure
  useEffect(() => {
    const updatedPermissions = { ...modulePermissions };
    const updatedErrors = { ...errors.permissions };

    modules.forEach((mod) => {
      updatedPermissions[mod] = updatedPermissions[mod] || [];
      updatedErrors[mod] = updatedPermissions[mod].length === 0;
      // Always update the ref
      permissionRefs.current[mod] = updatedPermissions[mod] || [];
    });

    Object.keys(updatedPermissions).forEach((mod) => {
      if (!modules.includes(mod)) {
        delete updatedPermissions[mod];
        delete updatedErrors[mod];
        delete permissionRefs.current[mod];
      }
    });

    setModulePermissions(updatedPermissions);
    setErrors((prev) => ({ ...prev, permissions: updatedErrors }));

    // Save current state to AsyncStorage for offline access
    const saveCurrentState = async () => {
      try {
        const currentState = {
          countries,
          projects,
          modules: modules.map(mod => ({
            module: mod,
            permissions: updatedPermissions[mod] || []
          }))
        };
        await AsyncStorage.setItem(`user_${member?.id}`, JSON.stringify(currentState));
      } catch (error) {
        console.error('Error saving current state:', error);
      }
    };
    
    if (member?.id) {
      saveCurrentState();
    }
  }, [modules, countries, projects, member?.id]);

  // Debug permission changes and sync refs
  useEffect(() => {
    console.log('Module permissions updated:', modulePermissions);
    
    // Keep refs in sync with state
    Object.keys(modulePermissions).forEach((module) => {
      permissionRefs.current[module] = modulePermissions[module] || [];
    });
  }, [modulePermissions]);

  const handlePermissionChange = (module, callback) => {
    let newPermissions = [];
    
    // Handle the callback properly for multiple selection
    if (typeof callback === 'function') {
      // Get current permissions for this module from ref to avoid stale state
      const currentPermissions = permissionRefs.current[module] || [];
      // Call the callback function to get the new selection
      newPermissions = callback(currentPermissions);
    } else {
      // If it's a direct value, use it
      newPermissions = Array.isArray(callback) ? [...callback] : [];
    }

    // Ensure newPermissions is always an array
    if (!Array.isArray(newPermissions)) {
      newPermissions = [];
    }

    // Handle "All Permissions" logic
    if (newPermissions.includes('all')) {
      // If "All Permissions" is selected, include all individual permissions
      newPermissions = ['all', ...individualPermissions];
    }

    console.log('Permission change for module:', module, 'Current:', permissionRefs.current[module], 'New:', newPermissions);

    // Update the ref immediately
    permissionRefs.current[module] = newPermissions;

    setModulePermissions((prev) => ({
      ...prev,
      [module]: newPermissions,
    }));

    setErrors((prev) => ({
      ...prev,
      permissions: {
        ...prev.permissions,
        [module]: false, // Don't require permissions initially
      },
    }));
  };

  // Function to add a new module to the list
  const addModule = (moduleName) => {
    if (!modules.includes(moduleName)) {
      setModules(prev => [...prev, moduleName]);
      setModulePermissions(prev => ({
        ...prev,
        [moduleName]: []
      }));
      permissionRefs.current[moduleName] = [];
    }
  };

  // Function to remove a module from the list
  const removeModule = (moduleName) => {
    setModules(prev => prev.filter(mod => mod !== moduleName));
    setModulePermissions(prev => {
      const newPermissions = { ...prev };
      delete newPermissions[moduleName];
      return newPermissions;
    });
    delete permissionRefs.current[moduleName];
  };

  // Function to add individual modules
  const addIndividualModule = () => {
    const availableModules = moduleItems
      .filter(item => !modules.includes(item.value))
      .map(item => item.label);
    
    if (availableModules.length === 0) {
      showToast('info', 'Info', 'All available modules are already added.');
      return;
    }
    
    // Show module selection modal
    setShowModuleSelectionModal(true);
  };

  const validateAndSubmit = async () => {
    const newErrors = {
      country: countries.length === 0,
      project: projects.length === 0,
      module: modules.length === 0,
      permissions: {},
    };

    // Require permissions for each module
    modules.forEach((mod) => {
      const perms = modulePermissions[mod] || [];
      newErrors.permissions[mod] = perms.length === 0;
    });

    setErrors(newErrors);

    const isValid =
      !newErrors.country &&
      !newErrors.project &&
      !newErrors.module &&
      Object.values(newErrors.permissions).every((v) => v === false);

    if (!isValid) {
      showToast('error', 'Error', 'Please fill all required fields.');
      return;
    }

    const finalData = {
      countries,
      projects,
      modules: modules.map((mod) => ({
        module: mod,
        permissions: modulePermissions[mod].filter((perm) => perm !== 'all'),
      })),
    };

    try {
      const adminUid = await AsyncStorage.getItem('userUid');
      if (!adminUid) {
        showToast('error', 'Error', 'Admin not authenticated.');
        navigation.navigate('AdminLoginScreen');
        return;
      }

      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      if (!isOnline) {
        const pendingPermissions = await AsyncStorage.getItem('pendingPermissions');
        const actions = pendingPermissions ? JSON.parse(pendingPermissions) : [];
        actions.push({ memberId: member.id, permissions: finalData });
        await AsyncStorage.setItem('pendingPermissions', JSON.stringify(actions));
        
        // Cache the updated user data
        await AsyncStorage.setItem(`user_${member.id}`, JSON.stringify({
          ...member,
          countries: finalData.countries,
          projects: finalData.projects,
          modules: finalData.modules
        }));
        showToast('info', 'Offline Mode', 'Permissions will sync when online.');
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        navigation.goBack();
        return;
      }

      // Save to the main user database
      await database()
        .ref(`/Globalusers/admins/${member.id}`)
        .update({
          countries: finalData.countries,
          projects: finalData.projects,
          modules: finalData.modules
        });

      // Also save to team member info for backward compatibility
      await database()
        .ref(`/GlobalTeamMemberInformation/${member.id}`)
        .update({
          countries: finalData.countries,
          projects: finalData.projects,
          modules: finalData.modules
        });

      // Cache the updated user data
      await AsyncStorage.setItem(`user_${member.id}`, JSON.stringify({
        ...member,
        countries: finalData.countries,
        projects: finalData.projects,
        modules: finalData.modules
      }));
      showToast('success', 'Success', 'Permissions saved successfully.');
      navigation.goBack();
    } catch (error) {
      console.error('Error saving permissions:', error);
      showToast('error', 'Error', 'Failed to save permissions.');
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
    }
  };

  // Ensure only one dropdown is open at a time
  const handleDropdownOpen = (type, value, module = null) => {
    if (type === 'country') {
      setCountryOpen(value);
      setProjectOpen(false);
      setModuleOpen(false);
      setPermissionOpen({});
    } else if (type === 'project') {
      setCountryOpen(false);
      setProjectOpen(value);
      setModuleOpen(false);
      setPermissionOpen({});
    } else if (type === 'module') {
      setCountryOpen(false);
      setProjectOpen(false);
      setModuleOpen(value);
      setPermissionOpen({});
    } else if (type === 'permission') {
      setCountryOpen(false);
      setProjectOpen(false);
      setModuleOpen(false);
      setPermissionOpen((prev) => ({
        ...Object.keys(prev).reduce((acc, key) => ({ ...acc, [key]: false }), {}),
        [module]: value,
      }));
    }
  };

  return (
    <ScrollView
      style={styles.scrollContainer}
      contentContainerStyle={styles.scrollContentContainer}
      keyboardShouldPersistTaps="handled"
    >
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Assign Permissions</Text>
          </View>
          <View style={styles.headerRight}>
            <Text style={styles.memberName}>{member?.name || 'Team Member'}</Text>
            <TouchableOpacity 
              style={styles.refreshButton}
              onPress={() => {
                setIsLoadingPermissions(true);
                // Reload permissions
                const initialize = async () => {
                  try {
                    const adminUid = await AsyncStorage.getItem('userUid');
                    if (adminUid) {
                      const userRef = database().ref(`/Globalusers/admins/${member.id}`);
                      userRef.once('value', (snapshot) => {
                        const data = snapshot.val();
                        if (data) {
                          console.log('Refreshed data from Firebase:', data);
                          
                          const existingCountries = Array.isArray(data.countries) ? data.countries : [];
                          const existingProjects = Array.isArray(data.projects) ? data.projects : [];
                          
                          setCountries(existingCountries);
                          setProjects(existingProjects);
                          
                          // Show all modules that exist in the user's data, even if no permissions
                          let existingModules = [];
                          const permissions = {};
                          
                          if (data.modules && Array.isArray(data.modules)) {
                            data.modules.forEach((mod) => {
                              const moduleName = mod.module || mod;
                              if (moduleName) {
                                existingModules.push(moduleName);
                                permissions[moduleName] = mod.permissions || [];
                              }
                            });
                          }
                          
                          setModules(existingModules);
                          setModulePermissions(permissions);
                          
                          // Initialize refs with refreshed permissions
                          existingModules.forEach((mod) => {
                            permissionRefs.current[mod] = permissions[mod] || [];
                          });
                          
                          // Initialize default modules if none exist
                          initializeDefaultModules();
                          showToast('success', 'Refreshed', 'Permissions data refreshed successfully');
                        }
                        setIsLoadingPermissions(false);
                      });
                    }
                  } catch (error) {
                    console.error('Error refreshing permissions:', error);
                    setIsLoadingPermissions(false);
                  }
                };
                initialize();
              }}
            >
              <MaterialIcons name="refresh" size={width * 0.05} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>
      </LinearGradient>

      <View style={styles.container}>
        
        {isLoadingPermissions && (
          <View style={styles.loadingContainer}>
            <Text style={styles.loadingText}>Loading existing permissions...</Text>
          </View>
        )}



        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Countries <Text style={inputStyles.requiredError}>*</Text>
          </Text>
          <DropDownPicker
            multiple
            open={countryOpen}
            value={countries}
            items={countryItems}
            setOpen={(value) => handleDropdownOpen('country', value)}
            setValue={setCountries}
            setItems={setCountryItems}
            placeholder="Select Countries"
            zIndex={5000}
            style={[inputStyles.input, errors.country ? inputStyles.inputError : null]}
            dropDownContainerStyle={styles.dropdownContainer}
            textStyle={styles.dropdownText}
            placeholderStyle={styles.placeholderStyle}
            badgeDotColors={['#015185']}
            badgeTextStyle={styles.badgeText}
            badgeStyle={styles.badge}
            listMode="SCROLLVIEW"
            scrollViewProps={{ nestedScrollEnabled: true }}
          />
          {errors.country && <Text style={inputStyles.errorText}>At least one country is required</Text>}
          <View style={styles.chipContainer}>
            {countries.length > 0 ? (
              countries.map((country, index) => (
                <View key={index} style={styles.chip}>
                  <Text style={styles.chipText}>{country}</Text>
                </View>
              ))
            ) : (
              <Text style={styles.noItemsText}>No countries selected</Text>
            )}
          </View>
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Projects <Text style={inputStyles.requiredError}>*</Text>
          </Text>
          <DropDownPicker
            multiple
            open={projectOpen}
            value={projects}
            items={projectItems}
            setOpen={(value) => handleDropdownOpen('project', value)}
            setValue={setProjects}
            setItems={setProjectItems}
            placeholder="Select Projects"
            disabled={countries.length === 0}
            disabledStyle={{ opacity: 0.6 }}
            zIndex={4000}
            style={[
              inputStyles.input,
              countries.length === 0 && styles.disabledDropdown,
              errors.project ? inputStyles.inputError : null,
            ]}
            dropDownContainerStyle={styles.dropdownContainer}
            textStyle={styles.dropdownText}
            placeholderStyle={styles.placeholderStyle}
            badgeDotColors={['#015185']}
            badgeTextStyle={styles.badgeText}
            badgeStyle={styles.badge}
            listMode="SCROLLVIEW"
            scrollViewProps={{ nestedScrollEnabled: true }}
          />
          {errors.project && <Text style={inputStyles.errorText}>At least one project is required</Text>}
          <View style={styles.chipContainer}>
            {projects.length > 0 ? (
              projects.map((project, index) => (
                <View key={index} style={styles.chip}>
                  <Text style={styles.chipText}>{project}</Text>
                </View>
              ))
            ) : (
              <Text style={styles.noItemsText}>No projects selected</Text>
            )}
          </View>
        </View>

        <View style={inputStyles.container}>
          <Text style={inputStyles.label}>
            Modules <Text style={inputStyles.requiredError}>*</Text>
          </Text>
          <View style={styles.moduleHeaderContainer}>
            <DropDownPicker
              multiple
              open={moduleOpen}
              value={modules}
              items={moduleItems}
              setOpen={(value) => handleDropdownOpen('module', value)}
              setValue={setModules}
              setItems={setModuleItems}
              placeholder="Select Modules"
              zIndex={3000}
              style={[inputStyles.input, errors.module ? inputStyles.inputError : null]}
              dropDownContainerStyle={styles.dropdownContainer}
              textStyle={styles.dropdownText}
              placeholderStyle={styles.placeholderStyle}
              badgeDotColors={['#015185']}
              badgeTextStyle={styles.badgeText}
              badgeStyle={styles.badge}
              listMode="SCROLLVIEW"
              scrollViewProps={{ nestedScrollEnabled: true }}
            />
            <TouchableOpacity
              style={styles.addAllModulesButton}
              onPress={() => {
                const allModules = moduleItems.map(item => item.value);
                setModules(allModules);
                allModules.forEach(mod => {
                  if (!modulePermissions[mod]) {
                    setModulePermissions(prev => ({
                      ...prev,
                      [mod]: []
                    }));
                    permissionRefs.current[mod] = [];
                  }
                });
              }}
            >
              <Text style={styles.addAllModulesButtonText}>Add All</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.addModuleButton}
              onPress={addIndividualModule}
            >
              <Text style={styles.addModuleButtonText}>Add Module</Text>
            </TouchableOpacity>
          </View>
          {errors.module && <Text style={inputStyles.errorText}>At least one module is required</Text>}
                    <View style={styles.chipContainer}>
            {modules.length > 0 ? (
              modules.map((module, index) => (
                <View key={index} style={styles.moduleChipContainer}>
                  <View style={styles.moduleHeaderRow}>
                    <View style={styles.chip}>
                      <Text style={styles.chipText}>{module}</Text>
                    </View>
                    <TouchableOpacity
                      style={styles.removeModuleButton}
                      onPress={() => removeModule(module)}
                    >
                      <MaterialIcons name="remove-circle" size={20} color="#FF4444" />
                    </TouchableOpacity>
                  </View>
                  
                  <View style={styles.permissionContainer}>
                    <View style={styles.permissionHeaderRow}>
                      <Text style={inputStyles.label}>
                        Permissions for {module} <Text style={inputStyles.requiredError}>*</Text>
                      </Text>
                      <Text style={styles.currentPermissionsText}>
                        Current: {modulePermissions[module]?.length > 0 ? modulePermissions[module].join(', ') : 'None'}
                      </Text>
                    </View>
                    
                    <DropDownPicker
                      multiple
                      open={permissionOpen[module] || false}
                      value={modulePermissions[module] || []}
                      items={permissionOptions}
                      setOpen={(value) => handleDropdownOpen('permission', value, module)}
                      setValue={(callback) => {
                        console.log('Permission change for module:', module, 'callback:', callback);
                        handlePermissionChange(module, callback);
                      }}
                      setItems={() => {}}
                      placeholder={`Select Permissions for ${module}`}
                      zIndex={2000 - index}
                      style={[
                        inputStyles.input,
                        errors.permissions[module] ? inputStyles.inputError : null,
                      ]}
                      dropDownContainerStyle={styles.dropdownContainer}
                      textStyle={styles.dropdownText}
                      placeholderStyle={styles.placeholderStyle}
                      badgeDotColors={['#015185']}
                      badgeTextStyle={styles.badgeText}
                      badgeStyle={styles.badge}
                      listMode="SCROLLVIEW"
                      scrollViewProps={{ nestedScrollEnabled: true }}
                      closeAfterSelecting={false}
                      min={0}
                      max={permissionOptions.length}
                      searchable={false}
                      autoScroll={true}
                      multipleText={modulePermissions[module]?.length > 0 ? `${modulePermissions[module].length} items selected` : ''}
                      showTickIcon={true}
                      tickIconStyle={{
                        width: width * 0.05,
                        height: width * 0.05,
                        tintColor: '#015185',
                      }}
                    />
                    
                    {errors.permissions[module] && (
                      <Text style={inputStyles.errorText}>
                        At least one permission is required for {module}
                      </Text>
                    )}
                    
                    <View style={styles.permissionChipsContainer}>
                      {modulePermissions[module] && modulePermissions[module].length > 0 ? (
                        modulePermissions[module].map((permission, permIndex) => (
                          <View key={permIndex} style={styles.permissionChip}>
                            <Text style={styles.permissionChipText}>{permission}</Text>
                            <TouchableOpacity
                              style={styles.removePermissionButton}
                              onPress={() => {
                                const newPermissions = modulePermissions[module].filter(p => p !== permission);
                                handlePermissionChange(module, newPermissions);
                              }}
                            >
                              <MaterialIcons name="close" size={16} color="#FF4444" />
                            </TouchableOpacity>
                          </View>
                        ))
                      ) : (
                        <Text style={styles.noItemsText}>No permissions selected</Text>
                      )}
                    </View>
                  </View>
                </View>
              ))
            ) : (
              <Text style={styles.noItemsText}>No modules selected</Text>
            )}
          </View>
        </View>

                  {/* Summary Section */}
          <View style={styles.summaryContainer}>
            <Text style={styles.summaryTitle}>Summary</Text>
            <View style={styles.summaryRow}>
              <Text style={styles.summaryLabel}>Countries:</Text>
              <Text style={styles.summaryValue}>{countries.join(', ') || 'None'}</Text>
            </View>
            <View style={styles.summaryRow}>
              <Text style={styles.summaryLabel}>Projects:</Text>
              <Text style={styles.summaryValue}>{projects.join(', ') || 'None'}</Text>
            </View>
            <View style={styles.summaryRow}>
              <Text style={styles.summaryLabel}>Modules:</Text>
              <Text style={styles.summaryValue}>{modules.length} modules selected</Text>
            </View>
            <View style={styles.permissionSummary}>
              {modules.map(module => (
                <Text key={module} style={styles.permissionItem}>
                  • {module}: {modulePermissions[module]?.length > 0 ? modulePermissions[module].join(', ') : 'No permissions'}
                </Text>
              ))}
            </View>
          </View>

          {/* Debug Section - Remove in production */}
          <View style={styles.debugContainer}>
            <Text style={styles.debugTitle}>Debug Info</Text>
            <Text style={styles.debugText}>Member ID: {member?.id}</Text>
            <Text style={styles.debugText}>Modules Count: {modules.length}</Text>
            <Text style={styles.debugText}>Countries Count: {countries.length}</Text>
            <Text style={styles.debugText}>Projects Count: {projects.length}</Text>
            <Text style={styles.debugText}>Online: {isOnline ? 'Yes' : 'No'}</Text>
          </View>

        <View style={styles.submitButtonContainer}>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <TouchableOpacity style={styles.submitButton} onPress={validateAndSubmit}>
              <Text style={styles.submitButtonText}>Submit</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      </View>

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Saving Permissions</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Module Selection Modal */}
      <Modal visible={showModuleSelectionModal} transparent animationType="fade">
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Module</Text>
            <Text style={styles.modalSubtitle}>Select a module to add</Text>
            
            <ScrollView style={styles.modalScrollView}>
              {moduleItems
                .filter(item => !modules.includes(item.value))
                .map((item, index) => (
                  <TouchableOpacity
                    key={index}
                    style={styles.modalRow}
                    onPress={() => {
                      addModule(item.value);
                      setShowModuleSelectionModal(false);
                      showToast('success', 'Success', `${item.label} module added`);
                    }}
                  >
                    <Text style={styles.modalOptionText}>{item.label}</Text>
                    <MaterialIcons name="add-circle" size={24} color="#4CAF50" />
                  </TouchableOpacity>
                ))}
            </ScrollView>
            
            <TouchableOpacity
              style={styles.modalCancelButton}
              onPress={() => setShowModuleSelectionModal(false)}
            >
              <Text style={styles.modalCancelButtonText}>Cancel</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Toast />
    </ScrollView>
  );
};

const inputStyles = StyleSheet.create({
  container: {
    marginBottom: width * 0.04,
  },
  label: {
    fontSize: width * 0.038,
    color: '#015185',
    marginBottom: 4,
    fontWeight: '500',
  },
  requiredError: {
    color: 'red',
    fontWeight: 'bold',
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    paddingHorizontal: 12,
    paddingVertical: 14,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
  },
  inputError: {
    borderColor: 'red',
  },
  errorText: {
    color: 'red',
    fontSize: width * 0.032,
    marginTop: 2,
    marginLeft: 2,
  },
});

const styles = StyleSheet.create({
  scrollContainer: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollContentContainer: {
    flexGrow: 1,
    paddingBottom: width * 0.1,
    paddingTop: 0,
  },
  container: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingTop: width * 0.02,
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  memberName: {
    color: '#fff',
    fontSize: width * 0.035,
    fontWeight: '500',
    textAlign: 'center',
  },
  refreshButton: {
    marginTop: width * 0.01,
    padding: width * 0.01,
    backgroundColor: 'rgba(255, 255, 255, 0.2)',
    borderRadius: width * 0.02,
  },
  dropdownContainer: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 5,
    backgroundColor: '#fff',
    maxHeight: width * 0.5,
  },
  dropdownText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  placeholderStyle: {
    fontSize: width * 0.04,
    color: '#888',
  },
  disabledDropdown: {
    backgroundColor: '#e0e0e0',
  },
  badge: {
    backgroundColor: '#e6f0fa',
    borderRadius: 12,
    paddingHorizontal: 8,
    paddingVertical: 4,
    margin: 2,
    borderWidth: 1,
    borderColor: '#015185',
  },
  badgeText: {
    fontSize: width * 0.032,
    color: '#015185',
    fontWeight: '500',
  },
  chipContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginTop: 8,
  },
  loadingContainer: {
    alignItems: 'center',
    paddingVertical: width * 0.03,
    backgroundColor: '#f8f9fa',
    borderRadius: 8,
    marginBottom: width * 0.02,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  loadingText: {
    fontSize: width * 0.035,
    color: '#6c757d',
    fontWeight: '500',
  },
  summaryContainer: {
    backgroundColor: '#e8f4fd',
    borderRadius: 8,
    padding: width * 0.03,
    marginBottom: width * 0.03,
    borderWidth: 1,
    borderColor: '#b3d9f2',
  },
  summaryTitle: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.02,
    textAlign: 'center',
  },
  summaryRow: {
    marginBottom: width * 0.015,
  },
  summaryLabel: {
    fontSize: width * 0.035,
    fontWeight: '600',
    color: '#333',
    marginBottom: width * 0.005,
  },
  summaryValue: {
    fontSize: width * 0.035,
    color: '#666',
    lineHeight: width * 0.045,
  },
  permissionSummary: {
    marginTop: width * 0.01,
  },
  permissionItem: {
    fontSize: width * 0.032,
    color: '#666',
    marginBottom: width * 0.01,
    marginLeft: width * 0.02,
    lineHeight: width * 0.04,
  },
  debugContainer: {
    backgroundColor: '#fff3cd',
    borderRadius: 6,
    padding: width * 0.02,
    marginBottom: width * 0.02,
    borderWidth: 1,
    borderColor: '#ffeaa7',
  },
  debugTitle: {
    fontSize: width * 0.032,
    fontWeight: 'bold',
    color: '#856404',
    marginBottom: width * 0.01,
  },
  debugText: {
    fontSize: width * 0.03,
    color: '#856404',
    marginBottom: width * 0.005,
    fontFamily: 'monospace',
  },
  chip: {
    backgroundColor: '#e6f0fa',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 6,
    marginRight: 8,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#015185',
  },
  chipText: {
    fontSize: width * 0.035,
    color: '#015185',
    fontWeight: '500',
  },
  noItemsText: {
    fontSize: width * 0.035,
    color: '#888',
    marginTop: 2,
  },
  moduleChipContainer: {
    marginBottom: 12,
    width: '100%',
  },
  permissionContainer: {
    marginTop: 8,
    marginLeft: 12,
  },
  moduleHeaderContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    gap: 10,
  },
  addAllModulesButton: {
    backgroundColor: '#4CAF50',
    paddingHorizontal: 15,
    paddingVertical: 10,
    borderRadius: 5,
    justifyContent: 'center',
    alignItems: 'center',
  },
  addAllModulesButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  addModuleButton: {
    backgroundColor: '#FF9800',
    paddingHorizontal: 15,
    paddingVertical: 10,
    borderRadius: 5,
    justifyContent: 'center',
    alignItems: 'center',
  },
  addModuleButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  moduleHeaderRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    width: '100%',
  },
  removeModuleButton: {
    padding: 5,
    marginLeft: 10,
  },
  permissionHeaderRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 8,
  },
  currentPermissionsText: {
    fontSize: width * 0.03,
    color: '#666',
    fontStyle: 'italic',
    textAlign: 'right',
    flex: 1,
    marginLeft: 10,
  },
  permissionChipsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    marginTop: 8,
  },
  permissionChip: {
    backgroundColor: '#e6f0fa',
    borderRadius: 12,
    paddingHorizontal: 10,
    paddingVertical: 6,
    marginRight: 8,
    marginBottom: 8,
    borderWidth: 1,
    borderColor: '#015185',
    flexDirection: 'row',
    alignItems: 'center',
  },
  permissionChipText: {
    fontSize: width * 0.032,
    color: '#015185',
    fontWeight: '500',
    marginRight: 5,
  },
  removePermissionButton: {
    padding: 2,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    width: '80%',
    maxHeight: '70%',
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 10,
    textAlign: 'center',
  },
  modalSubtitle: {
    fontSize: 14,
    color: '#666',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalScrollView: {
    maxHeight: 200,
    marginBottom: 15,
  },
  modalRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingVertical: 12,
    paddingHorizontal: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  modalOptionText: {
    fontSize: 16,
    color: '#333',
  },
  modalCancelButton: {
    backgroundColor: '#FF4444',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
    alignSelf: 'center',
  },
  modalCancelButtonText: {
    color: '#fff',
    fontSize: 14,
    fontWeight: 'bold',
  },
  submitButtonContainer: {
    marginTop: width * 0.05,
    marginBottom: width * 0.05,
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  submitButton: {
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
    alignItems: 'center',
  },
  submitButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
  },
  syncIcon: {
    width: width * 0.075,
    height: width * 0.075,
    marginRight: width * 0.025,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    marginBottom: width * 0.05,
  },
});

export default TEpremissionR;