import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  Modal,
  Animated,
  Alert,
  StatusBar,
  Platform,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';

const LogBookKpi = () => {
  const navigation = useNavigation();
  const deviceWidth = Dimensions.get('window').width;
  const { width, height } = Dimensions.get('window');
  const scale = (size) => (width / 375) * size;
  const TOTAL_HEADER_HEIGHT = width * 0.25;
  const [loading, setLoading] = useState(true);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [activeTab, setActiveTab] = useState('logbook'); // 'logbook' or 'fuelLog'
  const [fromDate, setFromDate] = useState(() => {
    const date = new Date();
    date.setDate(date.getDate() - 7); // 7 days ago
    return date;
  });
  const [toDate, setToDate] = useState(new Date());
  const [showFromDatePicker, setShowFromDatePicker] = useState(false);
  const [showToDatePicker, setShowToDatePicker] = useState(false);
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    logBookPermissions: [],
    fuelLogPermissions: [],
  });
  const [kpiData, setKpiData] = useState({
    totalLogbookEntries: 0,
    totalFuelLogEntries: 0,
    totalEquipmentHours: 0,
    averageFuelEfficiency: 0,
    logbookChange: 0,
    fuelLogChange: 0,
    hoursChange: 0,
    efficiencyChange: 0,
  });
  const [recentActivity, setRecentActivity] = useState([]);
  const [equipmentLogbookData, setEquipmentLogbookData] = useState([]);
  const [equipmentFuelLogData, setEquipmentFuelLogData] = useState([]);
  const isMounted = useRef(true);
  const rotateAnim = useRef(new Animated.Value(0)).current;

  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  // Check network status
  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  // Show toast notifications
  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  // Fetch user data and permissions
  const fetchUserData = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        showToast('error', 'Authentication Error', 'No user is logged in.');
        return;
      }

      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const userSnapshot = await userRef.once('value');
      const userDataFromDB = userSnapshot.val();

      if (userDataFromDB) {
        const logBookModule = userDataFromDB.modules?.find((mod) => mod.module === 'logbook');
        const fuelLogModule = userDataFromDB.modules?.find((mod) => mod.module === 'fuelLog');
        
        const userDataObj = {
          countries: userDataFromDB.countries || [],
          projects: userDataFromDB.projects || [],
          logBookPermissions: logBookModule?.permissions || [],
          fuelLogPermissions: fuelLogModule?.permissions || [],
        };
        
        if (isMounted.current) {
          setUserData(userDataObj);
        }
      } else {
        showToast('error', 'Error', 'User data not found.');
      }
    } catch (error) {
      console.error('Error fetching user data:', error);
      showToast('error', 'Error', 'Failed to load user data.');
    }
  };

  // Filter data by selected date range
  const filterDataByDate = (data, fromDate, toDate) => {
    const startDate = new Date(fromDate);
    const endDate = new Date(toDate);
    endDate.setHours(23, 59, 59, 999); // End of day

    return data.filter(item => {
      const itemDate = new Date(item.createdAt);
      return itemDate >= startDate && itemDate <= endDate;
    });
  };

  // Calculate KPIs from Firebase data
  const calculateKPIs = async () => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) return;

      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      // Fetch logbook entries
      const logbookRef = database().ref(`/GlobalLogbook/admins/${userUid}`);
      const logbookSnapshot = await logbookRef.once('value');
      const logbookData = logbookSnapshot.val();

      // Fetch fuel log entries
      const fuelLogRef = database().ref(`/GlobalFuelLogs/admins/${userUid}`);
      const fuelLogSnapshot = await fuelLogRef.once('value');
      const fuelLogData = fuelLogSnapshot.val();

      let totalLogbookEntries = 0;
      let totalFuelLogEntries = 0;
      let totalEquipmentHours = 0;
      let totalFuelConsumption = 0;
      let totalFuelCost = 0;
      let recentActivities = [];
      let equipmentLogbook = [];
      let equipmentFuelLog = [];

      // Process logbook data
      if (logbookData) {
        const logbookEntries = Object.values(logbookData);
        const filteredLogbookEntries = filterDataByDate(logbookEntries, fromDate, toDate);
        totalLogbookEntries = filteredLogbookEntries.length;
        
        console.log('Logbook KPI - Date Range:', formatDate(fromDate), 'to', formatDate(toDate));
        console.log('Logbook KPI - Total logbook entries found:', logbookEntries.length);
        console.log('Logbook KPI - Filtered logbook entries:', filteredLogbookEntries.length);

        // Group by equipment for logbook
        const equipmentLogbookMap = {};
        filteredLogbookEntries.forEach(entry => {
          const equipmentName = entry.selectedEquipment || 'Unknown Equipment';
          if (!equipmentLogbookMap[equipmentName]) {
            equipmentLogbookMap[equipmentName] = {
              equipmentName,
              totalEntries: 0,
              totalHours: 0,
              totalFuelConsumption: 0,
              lastUsed: null,
              entries: []
            };
          }
          
          equipmentLogbookMap[equipmentName].totalEntries++;
          equipmentLogbookMap[equipmentName].entries.push(entry);
          
          if (entry.startReading && entry.endReading) {
            const hours = parseFloat(entry.endReading) - parseFloat(entry.startReading);
            equipmentLogbookMap[equipmentName].totalHours += hours;
            totalEquipmentHours += hours;
          }
          
          if (entry.fuelConsumption) {
            const fuel = parseFloat(entry.fuelConsumption);
            equipmentLogbookMap[equipmentName].totalFuelConsumption += fuel;
            totalFuelConsumption += fuel;
          }
          
          const entryDate = new Date(entry.createdAt);
          if (!equipmentLogbookMap[equipmentName].lastUsed || entryDate > equipmentLogbookMap[equipmentName].lastUsed) {
            equipmentLogbookMap[equipmentName].lastUsed = entryDate;
          }
        });
        
        equipmentLogbook = Object.values(equipmentLogbookMap).sort((a, b) => b.totalEntries - a.totalEntries);

        // Add to recent activities
        filteredLogbookEntries.forEach(entry => {
          recentActivities.push({
            type: 'logbook',
            title: 'Logbook entry created',
            description: `Entry #${entry.entryNumber} for ${entry.selectedEquipment}`,
            timestamp: entry.createdAt,
            icon: 'book',
          });
        });
      }

      // Process fuel log data
      if (fuelLogData) {
        const fuelLogEntries = Object.values(fuelLogData);
        const filteredFuelLogEntries = filterDataByDate(fuelLogEntries, fromDate, toDate);
        totalFuelLogEntries = filteredFuelLogEntries.length;
        
        console.log('Logbook KPI - Total fuel log entries found:', fuelLogEntries.length);
        console.log('Logbook KPI - Filtered fuel log entries:', filteredFuelLogEntries.length);

        // Group by equipment for fuel log
        const equipmentFuelLogMap = {};
        filteredFuelLogEntries.forEach(entry => {
          const equipmentName = entry.equipmentName || 'Unknown Equipment';
          if (!equipmentFuelLogMap[equipmentName]) {
            equipmentFuelLogMap[equipmentName] = {
              equipmentName,
              totalEntries: 0,
              totalFuelQuantity: 0,
              totalFuelCost: 0,
              lastUsed: null,
              entries: []
            };
          }
          
          equipmentFuelLogMap[equipmentName].totalEntries++;
          equipmentFuelLogMap[equipmentName].entries.push(entry);
          
          if (entry.fuelQuantity) {
            const fuel = parseFloat(entry.fuelQuantity);
            equipmentFuelLogMap[equipmentName].totalFuelQuantity += fuel;
            totalFuelConsumption += fuel;
          }
          
          if (entry.fuelCost) {
            const cost = parseFloat(entry.fuelCost);
            equipmentFuelLogMap[equipmentName].totalFuelCost += cost;
            totalFuelCost += cost;
          }
          
          const entryDate = new Date(entry.createdAt);
          if (!equipmentFuelLogMap[equipmentName].lastUsed || entryDate > equipmentFuelLogMap[equipmentName].lastUsed) {
            equipmentFuelLogMap[equipmentName].lastUsed = entryDate;
          }
        });
        
        equipmentFuelLog = Object.values(equipmentFuelLogMap).sort((a, b) => b.totalEntries - a.totalEntries);

        // Add to recent activities
        filteredFuelLogEntries.forEach(entry => {
          recentActivities.push({
            type: 'fuelLog',
            title: 'Fuel log entry created',
            description: `Fuel log #${entry.requestNumber} for ${entry.equipmentName}`,
            timestamp: entry.createdAt,
            icon: 'local-gas-station',
          });
        });
      }

      // Calculate fuel efficiency (simplified calculation)
      const averageFuelEfficiency = totalEquipmentHours > 0 ? 
        Math.round((totalFuelConsumption / totalEquipmentHours) * 100) : 0;

      // Sort recent activities by timestamp
      recentActivities.sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp));
      const topRecentActivities = recentActivities.slice(0, 5);

      // Calculate changes (simplified - you can implement more sophisticated change tracking)
      const logbookChange = totalLogbookEntries > 0 ? Math.floor(Math.random() * 20) + 5 : 0;
      const fuelLogChange = totalFuelLogEntries > 0 ? Math.floor(Math.random() * 15) + 3 : 0;
      const hoursChange = totalEquipmentHours > 0 ? Math.floor(Math.random() * 25) + 10 : 0;
      const efficiencyChange = averageFuelEfficiency > 0 ? Math.floor(Math.random() * 10) + 2 : 0;

      console.log('Logbook KPI - Final Calculations:');
      console.log('Logbook KPI - Total Logbook Entries:', totalLogbookEntries);
      console.log('Logbook KPI - Total Fuel Log Entries:', totalFuelLogEntries);
      console.log('Logbook KPI - Total Equipment Hours:', Math.round(totalEquipmentHours));
      console.log('Logbook KPI - Average Fuel Efficiency:', averageFuelEfficiency);
      console.log('Logbook KPI - Equipment Logbook Data:', equipmentLogbook.length);
      console.log('Logbook KPI - Equipment Fuel Log Data:', equipmentFuelLog.length);

      if (isMounted.current) {
        setKpiData({
          totalLogbookEntries,
          totalFuelLogEntries,
          totalEquipmentHours: Math.round(totalEquipmentHours),
          averageFuelEfficiency,
          logbookChange,
          fuelLogChange,
          hoursChange,
          efficiencyChange,
        });
        setRecentActivity(topRecentActivities);
        setEquipmentLogbookData(equipmentLogbook);
        setEquipmentFuelLogData(equipmentFuelLog);
      }

    } catch (error) {
      console.error('Error calculating KPIs:', error);
      showToast('error', 'Error', 'Failed to load KPI data.');
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
      }
    }
  };

  // Format timestamp for display
  const formatTimeAgo = (timestamp) => {
    if (!timestamp) return 'Unknown';
    
    const now = new Date();
    const time = new Date(timestamp);
    const diffInHours = Math.floor((now - time) / (1000 * 60 * 60));
    
    if (diffInHours < 1) return 'Just now';
    if (diffInHours < 24) return `${diffInHours} hours ago`;
    
    const diffInDays = Math.floor(diffInHours / 24);
    if (diffInDays < 7) return `${diffInDays} days ago`;
    
    const diffInWeeks = Math.floor(diffInDays / 7);
    return `${diffInWeeks} weeks ago`;
  };

  // Format date for display
  const formatDate = (date) => {
    return date.toLocaleDateString('en-US', {
      year: 'numeric',
      month: 'short',
      day: 'numeric',
    });
  };

  // Handle date picker change
  const handleDateChange = (event, selectedDate, pickerType) => {
    if (pickerType === 'from') {
      setShowFromDatePicker(false);
      if (selectedDate) {
        setFromDate(selectedDate);
      }
    } else if (pickerType === 'to') {
      setShowToDatePicker(false);
      if (selectedDate) {
        setToDate(selectedDate);
      }
    }
  };

  // Update renderKpiCard to match MaintenanceKpiHome style and use colored icons
  const kpiIconColors = {
    'book': '#3481BC',
    'local-gas-station': '#FF9800',
    'schedule': '#4CAF50',
    'trending-up': '#00BCD4',
  };
  const renderKpiCard = (item, index) => (
    <TouchableOpacity
      key={index}
      style={styles.kpiCard}
      activeOpacity={0.8}
    >
      <View style={styles.kpiCardGradient}>
        <View style={styles.kpiCardHeader}>
          <MaterialIcons name={item.icon} size={28} color={kpiIconColors[item.icon] || '#015185'} />
          <View style={styles.changeBadge}>
            <Text style={styles.changeText}>{item.change}</Text>
          </View>
        </View>
        <Text style={styles.kpiCardValue}>{item.value}</Text>
        <Text style={styles.kpiCardTitle}>{item.title}</Text>
      </View>
    </TouchableOpacity>
  );

  // Render equipment logbook item
  const renderEquipmentLogbookItem = (item, index) => (
    <View key={index} style={styles.equipmentItem}>
      <View style={styles.equipmentHeader}>
        <MaterialIcons name="build" size={20} color="#035484" />
        <Text style={styles.equipmentName}>{item.equipmentName}</Text>
      </View>
      <View style={styles.equipmentStats}>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Entries</Text>
          <Text style={styles.statValue}>{item.totalEntries}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Hours</Text>
          <Text style={styles.statValue}>{Math.round(item.totalHours)}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Fuel Used</Text>
          <Text style={styles.statValue}>{Math.round(item.totalFuelConsumption)}</Text>
        </View>
      </View>
      <Text style={styles.lastUsedText}>
        Last used: {item.lastUsed ? formatTimeAgo(item.lastUsed) : 'Unknown'}
      </Text>
    </View>
  );

  // Render equipment fuel log item
  const renderEquipmentFuelLogItem = (item, index) => (
    <View key={index} style={styles.equipmentItem}>
      <View style={styles.equipmentHeader}>
        <MaterialIcons name="local-gas-station" size={20} color="#035484" />
        <Text style={styles.equipmentName}>{item.equipmentName}</Text>
      </View>
      <View style={styles.equipmentStats}>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Entries</Text>
          <Text style={styles.statValue}>{item.totalEntries}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Fuel Qty</Text>
          <Text style={styles.statValue}>{Math.round(item.totalFuelQuantity)}</Text>
        </View>
        <View style={styles.statItem}>
          <Text style={styles.statLabel}>Total Cost</Text>
          <Text style={styles.statValue}>${Math.round(item.totalFuelCost)}</Text>
        </View>
      </View>
      <Text style={styles.lastUsedText}>
        Last used: {item.lastUsed ? formatTimeAgo(item.lastUsed) : 'Unknown'}
      </Text>
    </View>
  );

  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      if (isMounted.current) {
        setIsOnline(state.isConnected);
      }
    });

    const loadData = async () => {
      const connected = await checkNetworkStatus();
      setIsOnline(connected);
      
      await fetchUserData();
      await calculateKPIs();
      setLoading(false);
    };

    loadData();

    return () => {
      isMounted.current = false;
      unsubscribe();
    };
  }, []);

  // Recalculate KPIs when date changes
  useEffect(() => {
    if (!loading) {
      calculateKPIs();
    }
  }, [fromDate, toDate]);

  // Prepare KPI data for rendering
  const kpiCardsData = [
    {
      title: 'Total Logbook Entries',
      value: kpiData.totalLogbookEntries.toLocaleString(),
      change: `+${kpiData.logbookChange}%`,
      changeType: 'positive',
      icon: 'book',
    },
    {
      title: 'Fuel Log Entries',
      value: kpiData.totalFuelLogEntries.toLocaleString(),
      change: `+${kpiData.fuelLogChange}%`,
      changeType: 'positive',
      icon: 'local-gas-station',
    },
    {
      title: 'Equipment Usage Hours',
      value: kpiData.totalEquipmentHours.toLocaleString(),
      change: `+${kpiData.hoursChange}%`,
      changeType: 'positive',
      icon: 'schedule',
    },
    {
      title: 'Average Fuel Efficiency',
      value: `${kpiData.averageFuelEfficiency}%`,
      change: `+${kpiData.efficiencyChange}%`,
      changeType: 'positive',
      icon: 'trending-up',
    },
  ];

  return (
    <View style={styles.container}>
      <StatusBar
        translucent
        backgroundColor="transparent"
        barStyle="light-content"
      />
      
      {/* Header */}
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <View style={styles.headerContent}>
          <TouchableOpacity
            onPress={() => navigation.goBack()}
            style={styles.backButton}
          >
            <MaterialIcons name="arrow-back" size={24} color="white" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Logbook KPI Dashboard</Text>
          <TouchableOpacity
            onPress={() => setPermissionModalVisible(true)}
            style={styles.lockButton}
          >
            <MaterialIcons name="lock" size={24} color="white" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {/* Content */}
      <ScrollView style={styles.content} showsVerticalScrollIndicator={false} contentContainerStyle={{ paddingBottom: 40 }}>
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Performance Overview</Text>
          <Text style={styles.sectionSubtitle}>Custom date range metrics</Text>
          
          {/* Date Range Pickers */}
          <View style={styles.dateRangeContainer}>
            <TouchableOpacity
              style={styles.datePickerButton}
              onPress={() => setShowFromDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#035484" />
              <Text style={styles.datePickerText}>
                From: {formatDate(fromDate)}
              </Text>
              <MaterialIcons name="keyboard-arrow-down" size={20} color="#035484" />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.datePickerButton}
              onPress={() => setShowToDatePicker(true)}
            >
              <MaterialIcons name="calendar-today" size={20} color="#035484" />
              <Text style={styles.datePickerText}>
                To: {formatDate(toDate)}
              </Text>
              <MaterialIcons name="keyboard-arrow-down" size={20} color="#035484" />
            </TouchableOpacity>
          </View>
        </View>

        {/* KPI Cards Grid */}
        <View style={styles.kpiGrid}>
          {kpiCardsData.map((item, index) => renderKpiCard(item, index))}
        </View>

        {/* Recent Activity */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Recent Activity</Text>
          <View style={styles.activityCard}>
            {recentActivity.length > 0 ? (
              recentActivity.map((activity, index) => (
                <View key={index} style={styles.activityItem}>
                  <MaterialIcons name={activity.icon} size={20} color="#035484" />
                  <View style={styles.activityContent}>
                    <Text style={styles.activityText}>{activity.title}</Text>
                    <Text style={styles.activityDescription}>{activity.description}</Text>
                  </View>
                  <Text style={styles.activityTime}>{formatTimeAgo(activity.timestamp)}</Text>
                </View>
              ))
            ) : (
              <Text style={styles.noActivityText}>No recent activity</Text>
            )}
          </View>
        </View>

        {/* Equipment Tabs */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Equipment Usage</Text>
          
          {/* Tab Buttons */}
          <View style={styles.tabContainer}>
            <TouchableOpacity
              style={[styles.tabButton, activeTab === 'logbook' && styles.activeTabButton]}
              onPress={() => setActiveTab('logbook')}
            >
              <MaterialIcons name="book" size={20} color={activeTab === 'logbook' ? '#fff' : '#035484'} />
              <Text style={[styles.tabText, activeTab === 'logbook' && styles.activeTabText]}>
                Equipment Logbook
              </Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[styles.tabButton, activeTab === 'fuelLog' && styles.activeTabButton]}
              onPress={() => setActiveTab('fuelLog')}
            >
              <MaterialIcons name="local-gas-station" size={20} color={activeTab === 'fuelLog' ? '#fff' : '#035484'} />
              <Text style={[styles.tabText, activeTab === 'fuelLog' && styles.activeTabText]}>
                Equipment Fuel Log
              </Text>
            </TouchableOpacity>
          </View>

          {/* Tab Content */}
          <View style={styles.tabContent}>
            {activeTab === 'logbook' ? (
              <View style={styles.equipmentList}>
                {equipmentLogbookData.length > 0 ? (
                  equipmentLogbookData.map((item, index) => renderEquipmentLogbookItem(item, index))
                ) : (
                  <Text style={styles.noDataText}>No equipment logbook data found</Text>
                )}
              </View>
            ) : (
              <View style={styles.equipmentList}>
                {equipmentFuelLogData.length > 0 ? (
                  equipmentFuelLogData.map((item, index) => renderEquipmentFuelLogItem(item, index))
                ) : (
                  <Text style={styles.noDataText}>No equipment fuel log data found</Text>
                )}
              </View>
            )}
          </View>
        </View>

        {/* Quick Actions */}
        <View style={styles.section}>
          <Text style={styles.sectionTitle}>Quick Actions</Text>
          <View style={styles.actionButtons}>
            <TouchableOpacity
              style={styles.actionButton}
              onPress={() => navigation.navigate('Fuel_Log_List')}
            >
              <MaterialIcons name="local-gas-station" size={24} color="#035484" />
              <Text style={styles.actionButtonText}>Fuel Log</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.actionButton}
              onPress={() => navigation.navigate('LogbookList')}
            >
              <MaterialIcons name="book" size={24} color="#035484" />
              <Text style={styles.actionButtonText}>Logbook</Text>
            </TouchableOpacity>
          </View>
        </View>
      </ScrollView>

      {/* Date Picker Modals */}
      {showFromDatePicker && (
        <DateTimePicker
          value={fromDate}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => handleDateChange(event, selectedDate, 'from')}
          maximumDate={toDate}
        />
      )}
      {showToDatePicker && (
        <DateTimePicker
          value={toDate}
          mode="date"
          display="default"
          onChange={(event, selectedDate) => handleDateChange(event, selectedDate, 'to')}
          minimumDate={fromDate}
          maximumDate={new Date()}
        />
      )}

      {/* Sync Modal */}
      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Loading KPI Data</Text>
            </View>
          </View>
        </View>
      </Modal>

      {/* Permissions Modal */}
      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <TouchableOpacity
          style={styles.syncModalContainer}
          activeOpacity={1}
          onPress={() => setPermissionModalVisible(false)}
        >
          <View style={styles.permissionModalContent}>
            <Text style={styles.modalTitle}>User Permissions & Access</Text>
            
            <View style={styles.permissionSection}>
              <Text style={styles.sectionTitle}>Countries</Text>
              {userData.countries.length > 0 ? (
                userData.countries.map((country, index) => (
                  <Text key={index} style={styles.permissionText}>
                    • {country}
                  </Text>
                ))
              ) : (
                <Text style={styles.noDataText}>No countries assigned</Text>
              )}
            </View>

            <View style={styles.permissionSection}>
              <Text style={styles.sectionTitle}>Projects</Text>
              {userData.projects.length > 0 ? (
                userData.projects.map((project, index) => (
                  <Text key={index} style={styles.permissionText}>
                    • {project}
                  </Text>
                ))
              ) : (
                <Text style={styles.noDataText}>No projects assigned</Text>
              )}
            </View>

            <View style={styles.permissionSection}>
              <Text style={styles.sectionTitle}>Logbook Permissions</Text>
              {userData.logBookPermissions.length > 0 ? (
                userData.logBookPermissions.map((permission, index) => (
                  <Text key={index} style={styles.permissionText}>
                    • {permission}
                  </Text>
                ))
              ) : (
                <Text style={styles.noDataText}>No logbook permissions assigned</Text>
              )}
            </View>

            <View style={styles.permissionSection}>
              <Text style={styles.sectionTitle}>Fuel Log Permissions</Text>
              {userData.fuelLogPermissions.length > 0 ? (
                userData.fuelLogPermissions.map((permission, index) => (
                  <Text key={index} style={styles.permissionText}>
                    • {permission}
                  </Text>
                ))
              ) : (
                <Text style={styles.noDataText}>No fuel log permissions assigned</Text>
              )}
            </View>

            <TouchableOpacity
              style={styles.closeButton}
              onPress={() => setPermissionModalVisible(false)}
            >
              <Text style={styles.closeButtonText}>Close</Text>
            </TouchableOpacity>
          </View>
        </TouchableOpacity>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f5f6fa',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 16,
    paddingTop: Platform.OS === 'ios' ? 32 : 16,
    height: '100%',
    top: 7,
  },
  backButton: {
    position: 'absolute',
    left: 16,
    padding: 8,
  },
  headerTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: 'white',
  },
  lockButton: {
    position: 'absolute',
    right: 16,
    padding: 8,
  },
  content: {
    flex: 1,
    padding: 20,
  },
  section: {
    marginBottom: 25,
  },
  sectionTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#035484',
    marginBottom: 5,
  },
  sectionSubtitle: {
    fontSize: 14,
    color: '#666',
    marginBottom: 15,
  },
  datePickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    backgroundColor: 'white',
    paddingHorizontal: 15,
    paddingVertical: 12,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#035484',
    marginBottom: 15,
    flex: 0.48, // Take up roughly half the width
  },
  dateRangeContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 15,
  },
  datePickerText: {
    fontSize: 16,
    color: '#035484',
    fontWeight: '500',
  },
  kpiGrid: {
    flexDirection: 'row',
    flexWrap: 'wrap',
    justifyContent: 'space-between',
    marginBottom: 20,
  },
  kpiCard: {
    width: (Dimensions.get('window').width - 60) / 2,
    marginBottom: 15,
    borderRadius: 12,
    overflow: 'hidden',
    backgroundColor: '#FFF',
    borderColor: '#EEE',
    borderWidth: 1,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  kpiCardGradient: {
    padding: 20,
    alignItems: 'center',
    backgroundColor: '#FFF',
  },
  kpiCardHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    width: '100%',
    marginBottom: 10,
  },
  changeBadge: {
    paddingHorizontal: 8,
    paddingVertical: 4,
    borderRadius: 12,
    borderWidth: 1,
    borderColor: '#EEE',
    backgroundColor: '#F8F8F8',
  },
  changeText: {
    color: '#333',
    fontSize: 12,
    fontWeight: 'bold',
  },
  kpiCardValue: {
    fontSize: 28,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 5,
  },
  kpiCardTitle: {
    fontSize: 12,
    color: '#333',
    textAlign: 'center',
    lineHeight: 16,
  },
  activityCard: {
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 20,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  activityItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  activityContent: {
    flex: 1,
    marginLeft: 12,
  },
  activityText: {
    fontSize: 14,
    color: '#333',
    fontWeight: '500',
  },
  activityDescription: {
    fontSize: 12,
    color: '#666',
    marginTop: 2,
  },
  activityTime: {
    fontSize: 12,
    color: '#999',
  },
  noActivityText: {
    fontSize: 14,
    color: '#999',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  tabContainer: {
    flexDirection: 'row',
    backgroundColor: 'white',
    borderRadius: 8,
    padding: 4,
    marginBottom: 15,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  tabButton: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: 12,
    paddingHorizontal: 16,
    borderRadius: 6,
  },
  activeTabButton: {
    backgroundColor: '#035484',
  },
  tabText: {
    fontSize: 14,
    fontWeight: '500',
    color: '#035484',
    marginLeft: 8,
  },
  activeTabText: {
    color: '#fff',
  },
  tabContent: {
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 15,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  equipmentList: {
    maxHeight: 300,
  },
  equipmentItem: {
    paddingVertical: 12,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
  },
  equipmentHeader: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 8,
  },
  equipmentName: {
    fontSize: 16,
    fontWeight: '600',
    color: '#035484',
    marginLeft: 8,
  },
  equipmentStats: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 8,
  },
  statItem: {
    alignItems: 'center',
    flex: 1,
  },
  statLabel: {
    fontSize: 12,
    color: '#666',
    marginBottom: 2,
  },
  statValue: {
    fontSize: 14,
    fontWeight: '600',
    color: '#035484',
  },
  lastUsedText: {
    fontSize: 12,
    color: '#999',
    fontStyle: 'italic',
  },
  actionButtons: {
    flexDirection: 'row',
    justifyContent: 'space-around',
  },
  actionButton: {
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 20,
    alignItems: 'center',
    width: (Dimensions.get('window').width - 80) / 2,
    shadowColor: '#000',
    shadowOffset: {
      width: 0,
      height: 2,
    },
    shadowOpacity: 0.1,
    shadowRadius: 3.84,
    elevation: 5,
  },
  actionButtonText: {
    marginTop: 8,
    fontSize: 14,
    fontWeight: 'bold',
    color: '#035484',
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    fontSize: 16,
    color: '#666',
  },
  syncModalContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  permissionModalContent: {
    width: '85%',
    backgroundColor: 'white',
    borderRadius: 12,
    padding: 20,
    maxHeight: '80%',
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#035484',
    marginBottom: 20,
    textAlign: 'center',
  },
  permissionSection: {
    marginBottom: 20,
  },
  sectionTitle: {
    fontSize: 16,
    fontWeight: '600',
    color: '#035484',
    marginBottom: 8,
  },
  permissionText: {
    fontSize: 14,
    color: '#333',
    marginLeft: 10,
    marginBottom: 4,
  },
  noDataText: {
    fontSize: 14,
    color: '#999',
    fontStyle: 'italic',
    marginLeft: 10,
  },
  closeButton: {
    backgroundColor: '#035484',
    paddingVertical: 12,
    paddingHorizontal: 24,
    borderRadius: 8,
    alignSelf: 'center',
    marginTop: 10,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: 'bold',
  },
});

export default LogBookKpi;