// Email Configuration and Templates
export const EMAIL_CONFIG = {
  // Email service configuration
  service: {
    name: 'Titan Drilling Email Service',
    from: 'noreply@titandrilling.com',
    replyTo: 'support@titandrilling.com',
    supportEmail: 'support@titandrilling.com'
  },

  // Retry configuration
  retry: {
    maxAttempts: 3,
    delayBetweenAttempts: 5000, // 5 seconds
    maxDelay: 30000 // 30 seconds
  },

  // Queue configuration
  queue: {
    maxConcurrentEmails: 5,
    processInterval: 1000, // 1 second
    maxQueueSize: 1000
  },

  // Template configuration
  templates: {
    inspectionCreated: 'inspection_created',
    inspectionNotification: 'inspection_notification',
    inspectionUpdated: 'inspection_updated',
    inspectionCompleted: 'inspection_completed'
  }
};

// Email template styles
export const EMAIL_STYLES = {
  colors: {
    primary: '#015185',
    secondary: '#3481BC',
    success: '#28a745',
    warning: '#ffc107',
    danger: '#dc3545',
    info: '#17a2b8',
    light: '#f8f9fa',
    dark: '#343a40'
  },
  
  fonts: {
    primary: 'Arial, sans-serif',
    secondary: 'Helvetica, sans-serif'
  },
  
  spacing: {
    small: '10px',
    medium: '20px',
    large: '30px'
  },
  
  borderRadius: '5px',
  boxShadow: '0 2px 4px rgba(0,0,0,0.1)'
};

// Email subjects
export const EMAIL_SUBJECTS = {
  inspectionCreated: (requestNumber) => `Inspection Report Created - ${requestNumber}`,
  inspectionNotification: (requestNumber) => `New Inspection Report - ${requestNumber}`,
  inspectionUpdated: (requestNumber) => `Inspection Report Updated - ${requestNumber}`,
  inspectionCompleted: (requestNumber) => `Inspection Report Completed - ${requestNumber}`,
  inspectionRejected: (requestNumber) => `Inspection Report Rejected - ${requestNumber}`,
  inspectionApproved: (requestNumber) => `Inspection Report Approved - ${requestNumber}`
};

// Email footer template
export const EMAIL_FOOTER = `
  <div style="margin-top: 30px; padding-top: 20px; border-top: 1px solid #eee; text-align: center; color: #666; font-size: 12px;">
    <p>This is an automated email from Titan Drilling System.</p>
    <p>Please do not reply to this email. For support, contact: ${EMAIL_CONFIG.service.supportEmail}</p>
    <p>&copy; ${new Date().getFullYear()} Titan Drilling. All rights reserved.</p>
  </div>
`;

// Email header template
export const EMAIL_HEADER = `
  <div style="background: linear-gradient(135deg, ${EMAIL_STYLES.colors.primary}, ${EMAIL_STYLES.colors.secondary}); padding: 20px; text-align: center; border-radius: ${EMAIL_STYLES.borderRadius} ${EMAIL_STYLES.borderRadius} 0 0;">
    <h1 style="color: white; margin: 0; font-size: 24px; font-weight: bold;">Titan Drilling System</h1>
    <p style="color: white; margin: 5px 0 0 0; font-size: 16px;">Inspection Report Management</p>
  </div>
`;

// Status colors for equipment status
export const STATUS_COLORS = {
  'Operational': EMAIL_STYLES.colors.success,
  'Maintenance Required': EMAIL_STYLES.colors.warning,
  'Out of Service': EMAIL_STYLES.colors.danger,
  'Under Review': EMAIL_STYLES.colors.info,
  'Pending': EMAIL_STYLES.colors.warning
};

// Priority colors
export const PRIORITY_COLORS = {
  'Low': EMAIL_STYLES.colors.success,
  'Medium': EMAIL_STYLES.colors.warning,
  'High': EMAIL_STYLES.colors.danger,
  'Critical': '#8B0000'
};

// Country and project mappings
export const LOCATION_MAPPINGS = {
  countries: {
    'Zambia': {
      code: 'ZAM',
      flag: '🇿🇲',
      timezone: 'Africa/Lusaka'
    },
    'DRC': {
      code: 'DRC',
      flag: '🇨🇩',
      timezone: 'Africa/Kinshasa'
    }
  },
  
  projects: {
    'Kobold': {
      location: 'Zambia',
      type: 'Mining',
      status: 'Active'
    },
    'Kalumbila': {
      location: 'Zambia',
      type: 'Mining',
      status: 'Active'
    },
    'Kansanshi': {
      location: 'Zambia',
      type: 'Mining',
      status: 'Active'
    },
    'Kimiteto': {
      location: 'Zambia',
      type: 'Mining',
      status: 'Active'
    },
    'Mimosa': {
      location: 'Zambia',
      type: 'Mining',
      status: 'Active'
    },
    'Musompo': {
      location: 'DRC',
      type: 'Mining',
      status: 'Active'
    },
    'IME': {
      location: 'DRC',
      type: 'Mining',
      status: 'Active'
    }
  }
};

// Email validation patterns
export const EMAIL_VALIDATION = {
  email: /^[^\s@]+@[^\s@]+\.[^\s@]+$/,
  phone: /^\+?[\d\s\-\(\)]+$/,
  employeeNumber: /^[A-Z0-9]{3,10}$/
};

// Default email templates
export const DEFAULT_TEMPLATES = {
  // Template for when inspection is created
  inspectionCreated: (data) => `
    ${EMAIL_HEADER}
    <div style="max-width: 600px; margin: 0 auto; padding: 20px; background: white; border-radius: 0 0 ${EMAIL_STYLES.borderRadius} ${EMAIL_STYLES.borderRadius}; box-shadow: ${EMAIL_STYLES.boxShadow};">
      <h2 style="color: ${EMAIL_STYLES.colors.primary}; margin-bottom: 20px;">Inspection Report Created Successfully</h2>
      
      <p>Dear <strong>${data.creator.name}</strong>,</p>
      
      <p>Your inspection report has been created and submitted successfully. The report is now available for review by supervisors and stakeholders.</p>
      
      <div style="background-color: ${EMAIL_STYLES.colors.light}; padding: 15px; border-radius: ${EMAIL_STYLES.borderRadius}; margin: 20px 0; border-left: 4px solid ${EMAIL_STYLES.colors.primary};">
        <h3 style="margin-top: 0; color: ${EMAIL_STYLES.colors.primary};">Report Summary</h3>
        <table style="width: 100%; border-collapse: collapse;">
          <tr><td style="padding: 5px 0; font-weight: bold;">Request Number:</td><td style="padding: 5px 0;">${data.requestNumber}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Country:</td><td style="padding: 5px 0;">${data.country}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Project:</td><td style="padding: 5px 0;">${data.project}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Equipment:</td><td style="padding: 5px 0;">${data.equipment}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Status:</td><td style="padding: 5px 0; color: ${STATUS_COLORS[data.status] || EMAIL_STYLES.colors.dark};">${data.status}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Faulty Items:</td><td style="padding: 5px 0;">${data.faultyItems}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Created:</td><td style="padding: 5px 0;">${new Date(data.createdAt).toLocaleString()}</td></tr>
        </table>
      </div>
      
      <p><strong>Next Steps:</strong></p>
      <ul style="margin: 15px 0; padding-left: 20px;">
        <li>Your report has been automatically shared with all supervisors in the ${data.country} - ${data.project} region</li>
        <li>Supervisors will review the report and may request additional information if needed</li>
        <li>You will be notified of any updates or actions required</li>
      </ul>
      
      <p>If you need to make any changes to the report, please contact your supervisor or the system administrator.</p>
    </div>
    ${EMAIL_FOOTER}
  `,

  // Template for supervisor notifications
  supervisorNotification: (data, supervisor) => `
    ${EMAIL_HEADER}
    <div style="max-width: 600px; margin: 0 auto; padding: 20px; background: white; border-radius: 0 0 ${EMAIL_STYLES.borderRadius} ${EMAIL_STYLES.borderRadius}; box-shadow: ${EMAIL_STYLES.boxShadow};">
      <h2 style="color: ${EMAIL_STYLES.colors.primary}; margin-bottom: 20px;">New Inspection Report Available</h2>
      
      <p>Dear <strong>${supervisor.name}</strong>,</p>
      
      <p>A new inspection report has been created in your jurisdiction that requires your attention and review.</p>
      
      <div style="background-color: ${EMAIL_STYLES.colors.light}; padding: 15px; border-radius: ${EMAIL_STYLES.borderRadius}; margin: 20px 0; border-left: 4px solid ${EMAIL_STYLES.colors.info};">
        <h3 style="margin-top: 0; color: ${EMAIL_STYLES.colors.primary};">Report Details</h3>
        <table style="width: 100%; border-collapse: collapse;">
          <tr><td style="padding: 5px 0; font-weight: bold;">Request Number:</td><td style="padding: 5px 0;">${data.requestNumber}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Country:</td><td style="padding: 5px 0;">${data.country}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Project:</td><td style="padding: 5px 0;">${data.project}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Inspector:</td><td style="padding: 5px 0;">${data.inspector}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Equipment:</td><td style="padding: 5px 0;">${data.equipment}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Status:</td><td style="padding: 5px 0; color: ${STATUS_COLORS[data.status] || EMAIL_STYLES.colors.dark};">${data.status}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Faulty Items:</td><td style="padding: 5px 0;">${data.faultyItems}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Created:</td><td style="padding: 5px 0;">${new Date(data.createdAt).toLocaleString()}</td></tr>
          <tr><td style="padding: 5px 0; font-weight: bold;">Created By:</td><td style="padding: 5px 0;">${data.creator.name} (${data.creator.employeeNumber})</td></tr>
        </table>
      </div>
      
      <p><strong>Action Required:</strong></p>
      <ul style="margin: 15px 0; padding-left: 20px;">
        <li>Review the inspection report for accuracy and completeness</li>
        <li>Verify that all safety requirements have been met</li>
        <li>Approve, reject, or request additional information as needed</li>
        <li>Ensure compliance with company policies and regulations</li>
      </ul>
      
      <p>Please log into the Titan Drilling System to review the complete report and take appropriate action.</p>
      
      <div style="text-align: center; margin: 30px 0;">
        <a href="#" style="background-color: ${EMAIL_STYLES.colors.primary}; color: white; padding: 12px 24px; text-decoration: none; border-radius: ${EMAIL_STYLES.borderRadius}; display: inline-block;">Review Report</a>
      </div>
    </div>
    ${EMAIL_FOOTER}
  `
};

export default {
  EMAIL_CONFIG,
  EMAIL_STYLES,
  EMAIL_SUBJECTS,
  EMAIL_FOOTER,
  EMAIL_HEADER,
  STATUS_COLORS,
  PRIORITY_COLORS,
  LOCATION_MAPPINGS,
  EMAIL_VALIDATION,
  DEFAULT_TEMPLATES
};
