import React, { useRef, useEffect } from 'react';
import { View, Text, StyleSheet, Platform, Dimensions, TouchableOpacity, StatusBar } from 'react-native';
import SignatureCanvas from 'react-native-signature-canvas';
import Toast from 'react-native-toast-message';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';

const { width, height } = Dimensions.get('window');

const SignatureCaptureScreen = ({ route, navigation }) => {
  const { role, onSave } = route.params;
  const ref = useRef(null);
  const capturedSignature = useRef(null);

  // Log when the screen mounts
  useEffect(() => {
    console.log('SignatureCaptureScreen mounted with params:', { role, onSave: !!onSave });
    return () => {
      console.log('SignatureCaptureScreen unmounted');
      capturedSignature.current = null;
    };
  }, []);

  const isValidSignatureUri = (uri) => {
    return uri && typeof uri === 'string' && uri.startsWith('data:image');
  };

  const handleSave = async () => {
    try {
      if (!capturedSignature.current) {
        console.warn('No signature provided');
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Please draw a signature before saving.',
        });
        return;
      }

      if (!isValidSignatureUri(capturedSignature.current)) {
        console.warn('Invalid signature format:', capturedSignature.current);
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Invalid signature format. Please try again.',
        });
        return;
      }

      console.log('Signature is valid, calling onSave with:', capturedSignature.current.substring(0, 50) + '...');
      onSave(capturedSignature.current);
      Toast.show({
        type: 'success',
        text1: 'Success',
        text2: `${role} signature saved successfully.`,
      });
      navigation.goBack();
    } catch (error) {
      console.error('Unexpected error in handleSave:', error.message, error.stack);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'An unexpected error occurred while saving the signature.',
      });
    }
  };

  const handleClear = () => {
    try {
      if (!ref.current) {
        console.error('SignatureCanvas ref is not set');
        Toast.show({
          type: 'error',
          text1: 'Error',
          text2: 'Signature component not initialized.',
        });
        return;
      }
      ref.current.clearSignature();
      capturedSignature.current = null;
      console.log('Signature cleared successfully');
      Toast.show({
        type: 'info',
        text1: 'Cleared',
        text2: 'Signature has been cleared.',
      });
    } catch (error) {
      console.error('Error clearing signature:', error.message, error.stack);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to clear signature.',
      });
    }
  };

  const canvasStyle = Platform.select({
    ios: `
      .m-signature-pad { height: ${height * 0.4}px; width: 100%; margin: 0; padding: 0; }
      .m-signature-pad--body { border: 1px solid #ddd; height: 100%; width: 100%; }
      canvas { height: 100% !important; width: 100% !important; }
    `,
    android: `
      .m-signature-pad { height: ${height * 0.4}px; width: 100%; margin: 0; padding: 0; }
      .m-signature-pad--body { border: 1px solid #ddd; height: 100%; width: 100%; }
      canvas { height: 100% !important; width: 100% !important; }
    `,
  });

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: styles.headerHeight }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>{role} Signature</Text>
          </View>
          <View style={styles.headerRight} />
        </View>
      </LinearGradient>
      <View style={styles.canvasContainer}>
        <SignatureCanvas
          ref={ref}
          onOK={(signature) => {
            console.log('Signature captured via onOK:', signature ? signature.substring(0, 50) + '...' : signature);
            capturedSignature.current = signature;
            handleSave();
          }}
          onEmpty={() => {
            console.log('Signature canvas is empty');
            capturedSignature.current = null;
          }}
          onBegin={() => console.log('User started drawing')}
          onEnd={() => console.log('User finished drawing')}
          descriptionText="Sign here"
          clearText="Clear"
          confirmText="Confirm"
          webStyle={canvasStyle}
          penColor="#000000"
          backgroundColor="#ffffff"
          minStrokeWidth={2}
          maxStrokeWidth={4}
          onError={(error) => {
            console.error('SignatureCanvas WebView error:', error);
            Toast.show({
              type: 'error',
              text1: 'Error',
              text2: 'Failed to load signature canvas.',
            });
          }}
        />
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerHeight: width * 0.25,
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  canvasContainer: {
    flex: 1,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 5,
    overflow: 'hidden',
  },
});

export default SignatureCaptureScreen;