import { View, Text, TouchableOpacity, TextInput, StyleSheet, Dimensions, ScrollView } from 'react-native';
import React, { useState } from 'react';
import { useNavigation, useRoute } from '@react-navigation/native';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');

const ApplicationApprovReject = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { application, applicationType, action, approverRole, applicationId } = route.params;
  const [rejectReason, setRejectReason] = useState('');
  const [loading, setLoading] = useState(false);

  const handleSubmit = async () => {
    setLoading(true);
    try {
      const user = auth().currentUser;
      if (!user) throw new Error('User not authenticated');
      const userName = user.displayName || user.email || (approverRole === 'HrManager' ? 'HR Manager' : approverRole === 'FinanceManager' ? 'Finance Manager' : 'Supervisor');
      const userUid = user.uid;
      // Path to update
      const appRef = database().ref(`/GlobalHrApplications/admins/${applicationType}/${application.userUid || userUid}/${applicationId}`);
      // Build update object
      let updateObj = {};
      if (approverRole === 'HrManager') {
        updateObj.HrManager = {
          status: action === 'approve' ? 'Approved' : 'Rejected',
          name: userName,
          timestamp: new Date().toISOString(),
        };
        if (action === 'reject') {
          updateObj.HrManager.reason = rejectReason;
        }
      } else if (approverRole === 'FinanceManager') {
        updateObj.FinanceManager = {
          status: action === 'approve' ? 'Approved' : 'Rejected',
          name: userName,
          timestamp: new Date().toISOString(),
        };
        if (action === 'reject') {
          updateObj.FinanceManager.reason = rejectReason;
        }
      } else {
        updateObj.Supervisor = {
          status: action === 'approve' ? 'Approved' : 'Rejected',
          name: userName,
          timestamp: new Date().toISOString(),
        };
        if (action === 'reject') {
          updateObj.Supervisor.reason = rejectReason;
        }
      }
      await appRef.update(updateObj);
      // Fetch latest data from Firebase
      const snapshot = await appRef.once('value');
      const updatedApp = snapshot.val();
      // Determine new status
      let newStatus = 'Pending';
      // Check for rejection first
      if (
        updatedApp.Supervisor?.status === 'Rejected' ||
        updatedApp.HrManager?.status === 'Rejected' ||
        updatedApp.FinanceManager?.status === 'Rejected'
      ) {
        newStatus = 'Rejected';
      } else {
        // Only consider roles that exist for this application
        const roles = [];
        if (updatedApp.Supervisor) roles.push(updatedApp.Supervisor.status);
        if (updatedApp.HrManager) roles.push(updatedApp.HrManager.status);
        if (updatedApp.FinanceManager) roles.push(updatedApp.FinanceManager.status);
        if (roles.length > 0 && roles.every((s) => s === 'Approved')) {
          newStatus = 'Approved';
        }
      }
      // Update the main status field if changed
      await appRef.update({ status: newStatus });
      Toast.show({
        type: 'success',
        text1: `Application ${action === 'approve' ? 'Approved' : 'Rejected'}`,
        position: 'bottom',
      });
      // Go back to ApplicationDetail, which will refresh on focus
      navigation.goBack();
    } catch (e) {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: e.message || 'Failed to update application',
        position: 'bottom',
      });
    } finally {
      setLoading(false);
    }
  };

  const getDetails = () => {
    const details = [
      { label: 'Full Name', value: application.fullName },
      { label: 'Employee No', value: application.employeeNo },
    ];
    switch (applicationType) {
      case 'salary_advance_application':
        details.push(
          { label: 'Amount', value: application.salaryAdvanceAmount },
          { label: 'Department', value: application.salaryAdvanceDepartment },
          { label: 'Recovery Period', value: application.selectedRecoveryPeriod },
          { label: 'Sites', value: application.selectedSites?.join(', ') },
          { label: 'Reasons', value: application.selectedReasons?.join(', ') },
        );
        break;
      case 'leave_application':
        details.push(
          { label: 'Start Date', value: application.leaveStartDate },
          { label: 'End Date', value: application.leaveEndDate },
          { label: 'Reason', value: application.reasonOfLeave },
        );
        break;
      case 'funeral_grant_form':
        details.push(
          { label: 'Deceased Name', value: application.deceasedName },
          { label: 'Relationship', value: application.deceasedRelationship },
          { label: 'Deceased Date', value: application.deceasedDate },
        );
        break;
      case 'application_for_appeal':
        details.push(
          { label: 'Offence', value: application.appealOffence },
          { label: 'Offence Date', value: application.appealOffenceDate },
        );
        break;
      case 'grievance_record':
        details.push(
          { label: 'Place of Work', value: application.grievancePlaceOfWork },
          { label: 'Details', value: application.grievanceDetails },
        );
        break;
      case 'hospital_attendance_form':
        details.push(
          { label: 'Clinics', value: application.selectedClinics?.join(', ') },
          { label: 'Shift', value: application.selectedShift },
        );
        break;
      default:
        break;
    }
    return details;
  };

  return (
    <ScrollView contentContainerStyle={styles.container}>
      <Text style={styles.title}>{action === 'approve' ? 'Approve' : 'Reject'} Application</Text>
      <View style={styles.card}>
        {getDetails().map((item, idx) => (
          <Text key={idx} style={styles.label}>
            {item.label}: <Text style={styles.value}>{item.value || 'N/A'}</Text>
          </Text>
        ))}
        <Text style={styles.label}>
          Status: <Text style={styles.value}>
            {approverRole === 'HrManager'
              ? (application.HrManager?.status || 'Pending')
              : approverRole === 'FinanceManager'
              ? (application.FinanceManager?.status || 'Pending')
              : (application.Supervisor?.status || 'Pending')}
          </Text>
        </Text>
      </View>
      {action === 'reject' && (
        <View style={styles.inputContainer}>
          <Text style={styles.label}>Reason for Rejection <Text style={{ color: 'red' }}>*</Text></Text>
          <TextInput
            style={styles.input}
            placeholder="Enter reason"
            value={rejectReason}
            onChangeText={setRejectReason}
            multiline
            numberOfLines={3}
          />
        </View>
      )}
      {/* Disable approve/reject if already fully approved or rejected */}
      <TouchableOpacity
        style={[styles.submitBtn, {
          backgroundColor: action === 'approve' ? '#34C759' : '#FF3B30',
          opacity: loading || (action === 'reject' && !rejectReason.trim()) || application.status === 'Approved' || application.status === 'Rejected' ? 0.6 : 1
        }]}
        onPress={handleSubmit}
        disabled={loading || (action === 'reject' && !rejectReason.trim()) || application.status === 'Approved' || application.status === 'Rejected'}
      >
        <Text style={styles.submitBtnText}>{action === 'approve' ? 'Approve' : 'Reject'}</Text>
      </TouchableOpacity>
      <Toast />
    </ScrollView>
  );
};

const styles = StyleSheet.create({
  container: {
    flexGrow: 1,
    padding: width * 0.06,
    backgroundColor: '#f5f5f5',
    alignItems: 'center',
    justifyContent: 'center',
  },
  title: {
    fontSize: width * 0.055,
    fontWeight: 'bold',
    color: '#025383',
    marginBottom: width * 0.06,
    textAlign: 'center',
  },
  card: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: width * 0.05,
    marginBottom: width * 0.06,
    width: '100%',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  label: {
    fontSize: width * 0.04,
    fontWeight: '600',
    color: '#333',
    marginBottom: 6,
  },
  value: {
    fontWeight: '400',
    color: '#015185',
  },
  inputContainer: {
    width: '100%',
    marginBottom: width * 0.04,
  },
  input: {
    borderWidth: 1,
    borderColor: '#e0e0e0',
    borderRadius: 6,
    padding: 12,
    fontSize: width * 0.04,
    backgroundColor: '#fff',
    minHeight: 60,
    textAlignVertical: 'top',
  },
  submitBtn: {
    width: '100%',
    borderRadius: 6,
    paddingVertical: 14,
    alignItems: 'center',
    marginTop: width * 0.04,
  },
  submitBtnText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: width * 0.045,
  },
});

export default ApplicationApprovReject;
