import React, { useState } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  FlatList,
  ActivityIndicator,
  Dimensions,
  Modal,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import database from '@react-native-firebase/database';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import LinearGradient from 'react-native-linear-gradient';

const { width, height } = Dimensions.get('window');

const ApplicationList = ({ applications, isLoading, applicationType, navigation }) => {
  const [menuVisibleId, setMenuVisibleId] = useState(null);
  const [localApplications, setLocalApplications] = useState(applications);
  const [modalVisible, setModalVisible] = useState(false);
  const [modalLoading, setModalLoading] = useState(false);
  const [modalError, setModalError] = useState(null);
  const [userDetails, setUserDetails] = useState(null);
  const [allowedCountries, setAllowedCountries] = useState([]);
  const [allowedProjects, setAllowedProjects] = useState([]);

  // Fetch allowed countries and projects on mount
  React.useEffect(() => {
    const fetchUserPermissions = async () => {
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) return;
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        const snapshot = await userRef.once('value');
        const data = snapshot.val();
        if (data) {
          setAllowedCountries(Array.isArray(data.countries) ? data.countries : (data.countries ? [data.countries] : []));
          setAllowedProjects(Array.isArray(data.projects) ? data.projects : (data.projects ? [data.projects] : []));
        }
      } catch (e) {}
    };
    fetchUserPermissions();
  }, []);

  // Filter applications based on allowed countries and projects
  React.useEffect(() => {
    if (allowedCountries.length === 0 && allowedProjects.length === 0) {
      setLocalApplications(applications);
    } else {
      setLocalApplications(applications.filter(app => {
        const countryOk = allowedCountries.length === 0 || allowedCountries.includes(app.country);
        const projectOk = allowedProjects.length === 0 || allowedProjects.includes(app.project);
        return countryOk && projectOk;
      }));
    }
  }, [applications, allowedCountries, allowedProjects]);

  const formatDate = (dateString) => {
    if (!dateString) return 'N/A';
    const date = new Date(dateString);
    return date.toISOString().split('T')[0];
  };

  const getSummary = (item) => {
    switch (applicationType) {
      case 'leave_application':
        return `From: ${formatDate(item.leaveStartDate)} To: ${formatDate(item.leaveEndDate)}`;
      case 'funeral_grant_form':
        return `Deceased: ${item.deceasedName || 'N/A'}, Relationship: ${item.deceasedRelationship || 'N/A'}`;
      case 'salary_advance_application':
        return `Amount: ${item.salaryAdvanceAmount || 'N/A'}, Recovery: ${item.selectedRecoveryPeriod || 'N/A'}`;
      case 'application_for_appeal':
        return `Offence: ${item.appealOffence || 'N/A'}, Date: ${formatDate(item.appealOffenceDate)}`;
      case 'grievance_record':
        return `Complainants: ${item.grievanceComplaints?.length || 0}, Place: ${item.grievancePlaceOfWork || 'N/A'}`;
      case 'hospital_attendance_form':
        return `Clinics: ${item.selectedClinics?.join(', ') || 'N/A'}, Shift: ${item.selectedShift || 'N/A'}`;
      default:
        return 'N/A';
    }
  };

  const handleDelete = async (item) => {
    try {
      // Remove from Firebase
      const userUid = await AsyncStorage.getItem('userUid');
      if (userUid && item.applicationId) {
        await database()
          .ref(`/GlobalHrApplications/admins/${applicationType}/${userUid}/${item.applicationId}`)
          .remove();
      }
      // Remove from local state
      setLocalApplications((prev) => prev.filter((app) => app.applicationId !== item.applicationId));
      Toast.show({ type: 'success', text1: 'Deleted', text2: 'Application deleted successfully.', position: 'bottom' });
      setMenuVisibleId(null);
    } catch (error) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to delete application.', position: 'bottom' });
    }
  };

  const handleLockPress = async () => {
    setModalVisible(true);
    setModalLoading(true);
    setModalError(null);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) {
        setModalError('User ID not found. Please log in again.');
        setModalLoading(false);
        return;
      }
      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const snapshot = await userRef.once('value');
      const data = snapshot.val();
      if (!data) {
        setUserDetails(null);
        setModalError('User details not found.');
      } else {
        const country = Array.isArray(data.countries) && data.countries.length > 0
          ? data.countries.join(', ')
          : (data.selectedCountry || data.country || 'N/A');
        const project = Array.isArray(data.projects) && data.projects.length > 0
          ? data.projects.join(', ')
          : (data.selectedProject || data.project || 'N/A');
        let hrFormPermission = 'N/A';
        if (Array.isArray(data.modules)) {
          const hrModule = data.modules.find(m => m.module === 'hrForm');
          if (hrModule && hrModule.permissions) {
            hrFormPermission = hrModule.permissions.length > 0
              ? hrModule.permissions.join(', ')
              : 'No Permissions';
          } else {
            hrFormPermission = 'No Permissions';
          }
        }
        setUserDetails({ country, project, hrFormPermission });
      }
    } catch (err) {
      setModalError('Failed to fetch user details.');
      setUserDetails(null);
    }
    setModalLoading(false);
  };

  const renderItem = ({ item }) => {
    // Determine status display
    let statusText = item.status || 'Pending';
    let statusColor = '#F57C00';
    if (statusText === 'Approved') statusColor = '#4CAF50';
    else if (statusText === 'Rejected') statusColor = '#D81B60';

    // Supervisor status logic
    let supervisorStatus = '';
    if (statusText === 'Pending' && item.Supervisor) {
      if (item.Supervisor.status === 'Approved') {
        supervisorStatus = 'Supervisor Approved, Pending';
      } else if (item.Supervisor.status === 'Rejected') {
        supervisorStatus = 'Supervisor Rejected, Pending';
      }
    }

    return (
      <View style={styles.applicationItem}>
        <TouchableOpacity
          style={{ flex: 1 }}
          onPress={() => navigation.navigate('ApplicationDetail', { application: item, applicationType })}
          accessibilityLabel={`View details for ${applicationType} application by ${item.fullName}`}
          accessibilityRole="button"
        >
          <View style={styles.applicationContent}>
            <View style={{ flex: 1 }}>
              <Text style={styles.applicationTitle}>{item.fullName || 'Unknown'}</Text>
              {item.email ? (
                <Text style={styles.applicationEmail}>{item.email}</Text>
              ) : null}
              <Text style={styles.applicationSubtitle}>Employee No: {item.employeeNo || 'N/A'}</Text>
              <Text style={styles.applicationSubtitle}>Submitted: {formatDate(item.submittedAt)}</Text>
              <Text style={styles.applicationSubtitle}>{getSummary(item)}</Text>
              <Text style={[styles.applicationStatus, { color: statusColor }]}>Status: {statusText}</Text>
              {supervisorStatus ? (
                <Text style={[styles.applicationStatus, { color: '#015185', fontSize: width * 0.032, marginTop: 2 }]}>{supervisorStatus}</Text>
              ) : null}
            </View>
            <TouchableOpacity
              onPress={() => setMenuVisibleId(menuVisibleId === item.applicationId ? null : item.applicationId)}
              style={{ padding: width * 0.01, marginLeft: width * 0.01 }}
              accessibilityLabel="Show options"
              accessibilityRole="button"
            >
              <MaterialIcons name="more-vert" size={width * 0.07} color="#333" />
            </TouchableOpacity>
          </View>
        </TouchableOpacity>
        {menuVisibleId === item.applicationId && (
          <View style={styles.menuDropdown}>
            <TouchableOpacity
              style={styles.menuItem}
              onPress={() => {
                setMenuVisibleId(null);
                navigation.navigate('ApplicationDetail', { application: item, applicationType });
              }}
            >
              <MaterialIcons name="visibility" size={width * 0.05} color="#015185" style={{ marginRight: 8 }} />
              <Text style={styles.menuText}>View</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.menuItem}
              onPress={() => handleDelete(item)}
            >
              <MaterialIcons name="delete" size={width * 0.05} color="#D81B60" style={{ marginRight: 8 }} />
              <Text style={[styles.menuText, { color: '#D81B60' }]}>Delete</Text>
            </TouchableOpacity>
          </View>
        )}
      </View>
    );
  };

  return (
    <View style={styles.listContainer}>
      {/* Header with lock icon */}
      <View style={styles.headerRow}>
        <Text style={styles.headerTitle}>Applications</Text>
        <TouchableOpacity style={styles.lockButton} onPress={handleLockPress} accessibilityLabel="Show user details" accessibilityRole="button">
          <MaterialIcons name="lock" size={20} color="#015185" />
        </TouchableOpacity>
      </View>
      {/* Modal for user details */}
      <Modal
        visible={modalVisible}
        transparent
        animationType="slide"
        onRequestClose={() => setModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>User Details</Text>
            {modalLoading ? (
              <ActivityIndicator size="large" color="#015185" />
            ) : modalError ? (
              <Text style={styles.modalError}>{modalError}</Text>
            ) : userDetails ? (
              <View style={styles.modalDetailsBox}>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>Country:</Text> {userDetails.country}</Text>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>Project:</Text> {userDetails.project}</Text>
                <Text style={styles.modalDetail}><Text style={styles.modalLabel}>HR Form Permission:</Text> {String(userDetails.hrFormPermission)}</Text>
              </View>
            ) : null}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setModalVisible(false)}
              accessibilityLabel="Close user details"
              accessibilityRole="button"
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>
      {isLoading ? (
        <ActivityIndicator size="large" color="#015185" style={styles.loader} />
      ) : localApplications.length === 0 ? (
        <Text style={styles.noApplicationsText}>No applications found</Text>
      ) : (
        <FlatList
          data={localApplications}
          renderItem={renderItem}
          keyExtractor={(item) => item.applicationId}
          showsVerticalScrollIndicator={false}
          extraData={menuVisibleId}
        />
      )}
    </View>
  );
};

const styles = StyleSheet.create({
  listContainer: {
    flex: 1,
    padding: width * 0.04,
  },
  headerRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    marginBottom: width * 0.03,
  },
  headerTitle: {
    fontSize: width * 0.055,
    fontWeight: 'bold',
    color: '#015185',
  },
  lockButton: {
    padding: width * 0.01,
  },
  applicationItem: {
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    marginBottom: width * 0.03,
    padding: width * 0.04,
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  applicationContent: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
  },
  applicationTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#333',
  },
  applicationSubtitle: {
    fontSize: width * 0.035,
    color: '#666',
    marginTop: width * 0.01,
  },
  applicationStatus: {
    fontSize: width * 0.035,
    fontWeight: '600',
    marginTop: width * 0.01,
  },
  noApplicationsText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: height * 0.2,
  },
  loader: {
    marginTop: height * 0.2,
  },
  menuDropdown: {
    position: 'absolute',
    top: width * 0.13,
    right: width * 0.04,
    backgroundColor: '#fff',
    borderRadius: 8,
    elevation: 6,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 4,
    zIndex: 100,
    paddingVertical: 4,
    minWidth: width * 0.32,
  },
  menuItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 16,
  },
  menuText: {
    fontSize: width * 0.04,
    color: '#015185',
    fontWeight: '500',
  },
  applicationEmail: {
    fontSize: width * 0.033,
    color: '#888',
    marginTop: 2,
    marginBottom: 2,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.4)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 24,
    alignItems: 'center',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 8,
    elevation: 5,
  },
  modalTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    marginBottom: 16,
    color: '#015185',
  },
  modalDetailsBox: {
    marginBottom: 16,
    alignSelf: 'stretch',
  },
  modalDetail: {
    fontSize: 16,
    marginBottom: 8,
    color: '#222',
  },
  modalLabel: {
    fontWeight: 'bold',
    color: '#015185',
  },
  modalCloseButton: {
    marginTop: 8,
    backgroundColor: '#015185',
    borderRadius: 8,
    paddingVertical: 8,
    paddingHorizontal: 24,
  },
  modalCloseText: {
    color: '#fff',
    fontWeight: 'bold',
    fontSize: 16,
  },
  modalError: {
    color: '#FF6B6B',
    fontSize: 16,
    marginBottom: 12,
    textAlign: 'center',
  },
});

export default ApplicationList;