import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  StatusBar,
  TextInput,
  Image,
  Dimensions,
  Animated,
  FlatList,
  Modal,
  TouchableWithoutFeedback,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import { useNavigation } from '@react-navigation/native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import database from '@react-native-firebase/database';
import NetInfo from '@react-native-community/netinfo';
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;

const Fuel_Log_List = () => {
  const navigation = useNavigation();
  const [searchQuery, setSearchQuery] = useState('');
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [permissionModalVisible, setPermissionModalVisible] = useState(false);
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [fuelLogs, setFuelLogs] = useState([]);
  const [filteredLogs, setFilteredLogs] = useState([]);
  const [dropdownVisible, setDropdownVisible] = useState(null);
  const [selectedItem, setSelectedItem] = useState(null);
  const [dropdownPosition, setDropdownPosition] = useState({ x: 0, y: 0, openUp: false });
  const [sortBy, setSortBy] = useState('createdAt');
  const [userData, setUserData] = useState({
    countries: [],
    projects: [],
    fuelLogPermissions: [],
  });
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const isMounted = useRef(true);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const DROPDOWN_WIDTH = 220;
  const DROPDOWN_ITEM_HEIGHT = width * 0.13;
  const DROPDOWN_ICON_SIZE = 22;
  const DROPDOWN_FONT_SIZE = 16;

  const dropdownData = [
    { label: 'Delete', icon: 'delete', iconColor: '#F44336', action: 'handleDelete' },
  ];

  useEffect(() => {
    if (syncModalVisible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [syncModalVisible]);

  useEffect(() => {
    fetchFuelLogs();
    return () => {
      isMounted.current = false;
    };
  }, []);

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      if (!netInfo.isConnected) throw new Error('Internet connection not available');
      return true;
    } catch (error) {
      throw new Error('Failed to check network status: ' + error.message);
    }
  };

  const showToast = (type, text1, text2) => {
    if (!isMounted.current) return;
    Toast.show({
      type,
      text1,
      text2,
      position: 'bottom',
      visibilityTime: 3000,
      autoHide: true,
    });
  };

  const fetchFuelLogs = async () => {
    setSyncModalVisible(true);
    setIsRefreshing(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) throw new Error('No user is logged in');

      if (!(await checkNetworkStatus())) throw new Error('Internet connection not available');

      // Fetch user data
      const userRef = database().ref(`/Globalusers/admins/${userUid}`);
      const userSnapshot = await userRef.once('value');
      const userDataFromDB = userSnapshot.val();

      if (userDataFromDB) {
        const fuelLogModule = userDataFromDB.modules?.find((mod) => mod.module === 'fuelLog');
        setUserData({
          countries: userDataFromDB.countries || [],
          projects: userDataFromDB.projects || [],
          fuelLogPermissions: fuelLogModule?.permissions || [],
        });
      } else {
        showToast('error', 'Error', 'User data not found.');
        setUserData({ countries: [], projects: [], fuelLogPermissions: [] });
      }

      // Fetch fuel logs
      const logsRef = database().ref(`/GlobalFuelLogs/admins/${userUid}`);
      const snapshot = await logsRef.once('value');
      const data = snapshot.val();

      if (data) {
        const logsArray = Object.keys(data).map((id) => ({
          id,
          ...data[id],
        }));
        logsArray.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        setFuelLogs(logsArray);
        setFilteredLogs(logsArray);
        showToast('success', 'Success', 'Fuel logs loaded successfully.');
      } else {
        setFuelLogs([]);
        setFilteredLogs([]);
        showToast('info', 'No Data', 'No fuel logs found.');
      }
    } catch (error) {
      console.error('Error fetching data from Firebase:', error);
      let errorMessage = 'Failed to load data.';
      if (error.message.includes('Internet connection not available')) {
        errorMessage = 'No internet connection. Please connect to the internet.';
      } else if (error.message.includes('No user is logged in')) {
        errorMessage = 'User not authenticated. Please log in.';
        navigation.navigate('AdminLoginScreen');
      } else if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      if (isMounted.current) {
        setSyncModalVisible(false);
        setIsRefreshing(false);
      }
    }
  };

  const handleSearch = (text) => {
    setSearchQuery(text);
    if (text) {
      const filtered = fuelLogs.filter(
        (log) =>
          (log.equipmentName?.toLowerCase().includes(text.toLowerCase()) ||
          log.fuelType?.toLowerCase().includes(text.toLowerCase()) ||
          log.equipmentNo?.toLowerCase().includes(text.toLowerCase()))
      );
      setFilteredLogs(filtered);
    } else {
      setFilteredLogs(fuelLogs);
    }
  };

  const handleSort = (criteria) => {
    setSortBy(criteria);
    let sortedLogs = [...filteredLogs];
    switch (criteria) {
      case 'createdAt':
        sortedLogs.sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
        break;
      case 'equipmentNo':
        sortedLogs.sort((a, b) => a.equipmentNo.localeCompare(b.equipmentNo));
        break;
      case 'equipmentName':
        sortedLogs.sort((a, b) => a.equipmentName.localeCompare(b.equipmentName));
        break;
      case 'fuelQuantity':
        sortedLogs.sort((a, b) => parseFloat(b.fuelQuantity) - parseFloat(a.fuelQuantity));
        break;
      case 'fuelCost':
        sortedLogs.sort((a, b) => parseFloat(b.fuelCost || 0) - parseFloat(a.fuelCost || 0));
        break;
      default:
        break;
    }
    setFilteredLogs(sortedLogs);
    setFilterModalVisible(false);
  };

  const handleRefresh = () => {
    fetchFuelLogs();
  };

  const handleActionPress = (item, event) => {
    setSelectedItem(item);
    setDropdownVisible(true);

    const { pageX, pageY } = event.nativeEvent;
    const iconHeight = 22;
    const dropdownHeight = DROPDOWN_ITEM_HEIGHT * dropdownData.length;
    const screenHeight = Dimensions.get('window').height;

    let top = pageY + 2;
    let openUp = false;
    if (pageY + 2 + dropdownHeight + 20 > screenHeight) {
      top = pageY - dropdownHeight - 2;
      openUp = true;
    }

    setDropdownPosition({ x: pageX - DROPDOWN_WIDTH + iconHeight, y: top, openUp });
  };

  const handleViewReport = () => {
    if (selectedItem) {
      showToast('info', 'View Report', 'View Report functionality not implemented.');
    }
    setDropdownVisible(false);
  };

  const handleShare = () => {
    if (selectedItem) {
      showToast('info', 'Share', 'Share functionality not implemented.');
    }
    setDropdownVisible(false);
  };

  const handleDelete = async () => {
    if (!selectedItem) {
      showToast('error', 'Error', 'No item selected to delete.');
      return;
    }

    setSyncModalVisible(true);
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid) throw new Error('No user is logged in');
      if (!(await checkNetworkStatus())) throw new Error('Internet connection not available');

      const logRef = database().ref(`/GlobalFuelLogs/admins/${userUid}/${selectedItem.id}`);
      await logRef.remove();

      const updatedLogs = fuelLogs.filter(log => log.id !== selectedItem.id);
      setFuelLogs(updatedLogs);
      setFilteredLogs(updatedLogs);
      showToast('success', 'Success', 'Fuel log deleted successfully.');
    } catch (error) {
      console.error('Error deleting fuel log:', error);
      let errorMessage = 'Failed to delete fuel log.';
      if (error.message.includes('Internet connection not available')) {
        errorMessage = 'No internet connection. Please try again later.';
      } else if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Please contact support.';
      }
      showToast('error', 'Error', errorMessage);
    } finally {
      setSyncModalVisible(false);
      setDropdownVisible(null);
    }
  };

  const handleAction = (action) => {
    if (!selectedItem) {
      showToast('error', 'Error', 'No item selected.');
      return;
    }

    switch (action) {
      case 'Delete':
        handleDelete();
        break;
      case 'Receipt':
        showToast('info', 'Receipt', 'Receipt functionality not implemented.');
        setDropdownVisible(null);
        break;
      case 'Fuel Browser':
        showToast('info', 'Fuel Browser', 'Fuel Browser functionality not implemented.');
        setDropdownVisible(null);
        break;
      default:
        setDropdownVisible(null);
    }
  };

  const renderFuelLog = ({ item }) => (
    <View style={styles.logCard}>
      <View style={styles.logTable}>
        {/* Request Number Row */}
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={[styles.tableHeader, { color: '#015185', fontWeight: 'bold' }]}>Request#</Text>
          <Text style={[styles.tableData, { color: '#015185', fontWeight: 'bold' }]}>{item.requestNumber || 'N/A'}</Text>
        </View>
        {/* Equipment Number Row */}
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Equipment #</Text>
          <Text style={styles.tableData}>{item.equipmentNo || 'N/A'}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Equipment Name</Text>
          <Text style={styles.tableData}>{item.equipmentName || 'N/A'}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Fuel Type</Text>
          <Text style={styles.tableData}>{item.fuelType || 'N/A'}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Fuel Quantity</Text>
          <Text style={styles.tableData}>{item.fuelQuantity} {item.fuelUnit || ''}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Meter Reading</Text>
          <Text style={styles.tableData}>{item.meterReading} {item.meterUnit || ''}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Total</Text>
          <Text style={styles.tableData}>{item.fuelCost || 'N/A'}</Text>
        </View>
        <View style={[styles.tableRow, styles.tableRowWithBorder]}>
          <Text style={styles.tableHeader}>Date</Text>
          <Text style={styles.tableData}>
            {new Date(item.createdAt).toLocaleString('en-GB', {
              day: '2-digit',
              month: 'short',
              year: 'numeric',
              hour: '2-digit',
              minute: '2-digit',
              hour12: true,
            })}
          </Text>
        </View>
      </View>
      {/* Three dots icon for dropdown */}
      <View style={{ alignItems: 'flex-end', marginTop: 8 }}>
        <TouchableOpacity
          onPress={(event) => handleActionPress(item, event)}
          style={styles.threeDotsContainer}
        >
          <MaterialIcons name="more-vert" size={width * 0.06} color="#015185" />
        </TouchableOpacity>
      </View>
    </View>
  );

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Fuel Log</Text>
          </View>
          <View style={styles.headerRight}>
            <TouchableOpacity onPress={handleRefresh} style={styles.refreshButton}>
              <Image
                source={require('../../../Images/adminhome/equp/cloud.png')}
                style={styles.refreshIcon}
              />
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.iconButton}
              onPress={() => setPermissionModalVisible(true)}
            >
              <MaterialIcons name="lock" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
          </View>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder="Search..."
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons name="search" size={width * 0.05} color="#888" style={styles.bladeIcon} />
          </View>
          <TouchableOpacity style={styles.filterButton} onPress={() => setFilterModalVisible(true)}>
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      <FlatList
        data={filteredLogs}
        renderItem={renderFuelLog}
        keyExtractor={(item) => item.id}
        contentContainerStyle={styles.mainContent}
        ListEmptyComponent={
          <Text style={styles.defaultText}>No fuel logs found.</Text>
        }
        refreshing={isRefreshing}
        onRefresh={handleRefresh}
      />

      {dropdownVisible && (
        <Modal
          visible={dropdownVisible}
          transparent={true}
          animationType="fade"
          onRequestClose={() => setDropdownVisible(false)}
        >
          <TouchableOpacity
            style={styles.dropdownOverlay}
            onPress={() => setDropdownVisible(false)}
            activeOpacity={1}
          >
            <View
              style={[
                styles.dropdownContainer,
                {
                  position: 'absolute',
                  top: dropdownPosition.y + 8,
                  left: dropdownPosition.x,
                },
              ]}
            >
              <TouchableOpacity
                style={[styles.dropdownItem, { borderBottomWidth: 0 }]}
                onPress={() => {
                  setDropdownVisible(false);
                  setShowDeleteConfirm(true);
                }}
              >
                <MaterialIcons name="delete" size={DROPDOWN_ICON_SIZE} color="#F44336" />
                <Text style={styles.dropdownText}>Delete</Text>
              </TouchableOpacity>
            </View>
          </TouchableOpacity>
        </Modal>
      )}

      {/* Delete confirmation modal */}
      <Modal
        visible={!!showDeleteConfirm}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setShowDeleteConfirm(false)}
      >
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Confirm Delete</Text>
            <Text style={styles.syncModalTextadd}>Are you sure you want to delete this fuel log?</Text>
            <View style={{ flexDirection: 'row', marginTop: 20 }}>
              <TouchableOpacity
                style={[styles.closeButton, { marginRight: 10, backgroundColor: '#888' }]}
                onPress={() => setShowDeleteConfirm(false)}
              >
                <Text style={styles.closeButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.closeButton, { backgroundColor: '#F44336' }]}
                onPress={async () => {
                  setShowDeleteConfirm(false);
                  await handleDelete();
                }}
              >
                <Text style={styles.closeButtonText}>Delete</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {userData.fuelLogPermissions.includes('create') && (
        <View style={styles.buttonContainer}>
          <LinearGradient colors={['#3481BC', '#025383']} style={styles.buttonGradient}>
            <TouchableOpacity
              style={styles.startButton}
              onPress={() => navigation.navigate('CreateFuelLog')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.buttonText}>Fuel Log Entry</Text>
            </TouchableOpacity>
          </LinearGradient>
        </View>
      )}

      <Modal visible={syncModalVisible} transparent animationType="fade">
        <View style={styles.loadingModalContainer}>
          <View style={styles.loadingModalContent}>
            <Text style={styles.loadingModalText}>Please Wait</Text>
            <View style={styles.loadingIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.loadingIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.loadingModalTextadd}>Processing of cloud data syncing</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal visible={filterModalVisible} transparent animationType="fade">
        <TouchableWithoutFeedback onPress={() => setFilterModalVisible(false)}>
          <View style={styles.filterModalContainer}>
            <View style={styles.filterModalContent}>
              <TouchableOpacity onPress={() => handleSort('createdAt')}>
                <Text style={styles.filterText}>Created Date</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={() => handleSort('equipmentNo')}>
                <Text style={styles.filterText}>Equipment Number</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={() => handleSort('equipmentName')}>
                <Text style={styles.filterText}>Equipment Name</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={() => handleSort('fuelQuantity')}>
                <Text style={styles.filterText}>Fuel Quantity</Text>
              </TouchableOpacity>
              <TouchableOpacity onPress={() => handleSort('fuelCost')}>
                <Text style={styles.filterText}>Total Cost</Text>
              </TouchableOpacity>
              <View style={styles.buttonWrapper}>
                <TouchableOpacity
                  style={styles.closeButton}
                  onPress={() => setFilterModalVisible(false)}
                >
                  <Text style={styles.closeButtonText}>Close</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </TouchableWithoutFeedback>
      </Modal>

      <Modal
        visible={permissionModalVisible}
        transparent
        animationType="fade"
        onRequestClose={() => setPermissionModalVisible(false)}
      >
        <TouchableWithoutFeedback onPress={() => setPermissionModalVisible(false)}>
          <View style={styles.filterModalContainer}>
            <View style={styles.filterModalContent}>
              <Text style={styles.modalTitle}>User Permissions</Text>
              <View style={styles.permissionSection}>
                <Text style={styles.sectionTitle}>Countries</Text>
                {userData.countries.length > 0 ? (
                  userData.countries.map((country, index) => (
                    <Text key={index} style={styles.permissionText}>
                      • {country}
                    </Text>
                  ))
                ) : (
                  <Text style={styles.noDataText}>No countries assigned</Text>
                )}
              </View>
              <View style={styles.permissionSection}>
                <Text style={styles.sectionTitle}>Projects</Text>
                {userData.projects.length > 0 ? (
                  userData.projects.map((project, index) => (
                    <Text key={index} style={styles.permissionText}>
                      • {project}
                    </Text>
                  ))
                ) : (
                  <Text style={styles.noDataText}>No projects assigned</Text>
                )}
              </View>
              <View style={styles.permissionSection}>
                <Text style={styles.sectionTitle}>Fuel Log Permissions</Text>
                {userData.fuelLogPermissions.length > 0 ? (
                  userData.fuelLogPermissions.map((permission, index) => (
                    <Text key={index} style={styles.permissionText}>
                      • {permission}
                    </Text>
                  ))
                ) : (
                  <Text style={styles.noDataText}>No permissions assigned</Text>
                )}
              </View>
              <View style={styles.buttonWrapper}>
                <TouchableOpacity
                  style={styles.closeButton}
                  onPress={() => setPermissionModalVisible(false)}
                >
                  <Text style={styles.closeButtonText}>Close</Text>
                </TouchableOpacity>
              </View>
            </View>
          </View>
        </TouchableWithoutFeedback>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#ECF0F3',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerRight: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    marginLeft: width * 0.025,
  },
  refreshButton: {
    padding: width * 0.015,
    justifyContent: 'center',
    alignItems: 'center',
  },
  refreshIcon: {
    width: width * 0.07,
    height: width * 0.07,
    resizeMode: 'contain',
  },
  iconButton: {
    padding: width * 0.012,
    marginLeft: width * 0.025,
  },
  icons12: {
    width: width * 0.07,
    height: width * 0.06,
  },
  searchContainer: {
    flexDirection: 'row',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    alignItems: 'center',
    width: '100%',
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
    marginRight: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    color: '#333',
    fontSize: width * 0.04,
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    padding: width * 0.012,
  },
  mainContent: {
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
    paddingBottom: width * 0.2,
  },
  defaultText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginVertical: width * 0.05,
  },
  logCard: {
    backgroundColor: '#fff',
    borderRadius: width * 0.02,
    padding: width * 0.04,
    marginBottom: width * 0.03,
    borderWidth: 1,
    borderColor: '#ddd',
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.1,
    shadowRadius: 4,
    elevation: 2,
  },
  logTable: {
    backgroundColor: '#fff',
  },
  tableRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: width * 0.015,
  },
  tableRowWithBorder: {
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingBottom: width * 0.015,
  },
  equipmentRow: {
    backgroundColor: '#015185',
  },
  tableHeader: {
    fontSize: width * 0.035,
    color: '#025383',
    fontWeight: 'bold',
    width: '40%',
  },
  equipmentText: {
    color: '#fff',
  },
  tableData: {
    fontSize: width * 0.035,
    color: '#333',
    width: '60%',
    textAlign: 'right',
  },
  noteContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'flex-end',
    width: '60%',
  },
  threeDotsContainer: {
    padding: width * 0.01,
    marginLeft: width * 0.02,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.2)',
  },
  dropdownContainer: {
    backgroundColor: '#fff',
    borderWidth: 1,
    borderColor: '#eee',
    width: 220,
    elevation: 10,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.12,
    shadowRadius: 8,
    paddingVertical: 4,
    zIndex: 999,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 10,
    paddingHorizontal: 16,
    borderBottomWidth: 1,
    borderBottomColor: '#f0f0f0',
    backgroundColor: '#fff',
  },
  dropdownText: {
    fontSize: 16,
    color: '#222',
    marginLeft: 16,
    fontWeight: '500',
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  buttonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  startButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  buttonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  loadingModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.25,
    shadowRadius: 4,
  },
  loadingModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
    fontWeight: '600',
  },
  loadingIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.02,
  },
  loadingModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
    fontWeight: '500',
  },
  loadingIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  filterModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  filterModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'flex-start',
  },
  filterText: {
    fontSize: width * 0.04,
    color: '#025383',
    marginBottom: 10,
  },
  buttonWrapper: {
    width: '100%',
    alignItems: 'flex-end',
    marginTop: 10,
  },
  closeButton: {
    backgroundColor: '#025383',
    paddingVertical: 8,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  closeButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  modalTitle: {
    fontSize: width * 0.05,
    fontWeight: 'bold',
    color: '#025383',
    marginBottom: width * 0.04,
    textAlign: 'center',
  },
  permissionSection: {
    marginBottom: width * 0.04,
    width: '100%',
  },
  sectionTitle: {
    fontSize: width * 0.045,
    fontWeight: '600',
    color: '#025383',
    marginBottom: width * 0.02,
  },
  permissionText: {
    fontSize: width * 0.04,
    color: '#333',
    marginLeft: width * 0.02,
    marginBottom: width * 0.01,
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#888',
    fontStyle: 'italic',
  },
});

export default Fuel_Log_List;