import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  ScrollView,
  Dimensions,
  Image,
  Platform,
  SafeAreaView,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import CheckBox from '@react-native-community/checkbox';
import Toast from 'react-native-toast-message';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import { SafeAreaProvider, useSafeAreaInsets } from 'react-native-safe-area-context';
import NetInfo from '@react-native-community/netinfo';
import Modal from 'react-native-modal';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import debounce from 'lodash.debounce';

const { width, height } = Dimensions.get('window');

const DvirCorrectiveActionStep2 = ({
  report,
  setActiveStep,
  navigation,
  subItemStatuses,
  setSubItemStatuses,
  resetSubItemStatuses,
}) => {
  const [sessionState, setSessionState] = useState({
    isSatisfactory: null,
    role: null,
    mechanicName: '',
    operatorName: '',
  });
  const [isSubmitting, setIsSubmitting] = useState(false);
  const [mechanicSignatureUri, setMechanicSignatureUri] = useState(null);
  const [operatorSignatureUri, setOperatorSignatureUri] = useState(null);
  const [signatureRefreshKey, setSignatureRefreshKey] = useState(0);
  const [savePdf, setSavePdf] = useState(true);
  const [errors, setErrors] = useState({
    isSatisfactory: '',
    role: '',
    mechanicName: '',
    operatorName: '',
  });
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [combinedPdfDownloadUrl, setCombinedPdfDownloadUrl] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const isMounted = useRef(true);
  const insets = useSafeAreaInsets();

  useEffect(() => {
    isMounted.current = true;
    const loadStoredData = async () => {
      setIsLoading(true);
      if (!report?.id) {
        console.warn('No report ID provided for loading data');
        Toast.show({ type: 'error', text1: 'Error', text2: 'Missing report information.' });
        return;
      }
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) {
          Toast.show({ type: 'error', text1: 'Authentication Error', text2: 'Please log in again.' });
          navigation.replace('LoginScreens');
          return;
        }
        
        console.log('Loading corrective action data for report:', report.id);
        
        // Read from new path structure
        const correctiveRef = database().ref(`/GlobalDVirCorectiveActionR/${userUid}/${report.id}`);
        const snapshot = await correctiveRef.once('value');
        const correctiveData = snapshot.val();
        
        console.log('Corrective action data found:', !!correctiveData);
        if (correctiveData) {
          console.log('Setting session state with data:', {
            isSatisfactory: correctiveData.isSatisfactory,
            role: correctiveData.role,
            mechanicName: correctiveData.mechanicName,
            operatorName: correctiveData.operatorName,
            subItemStatusesCount: Object.keys(correctiveData.subItemStatuses || {}).length
          });
          
          setSessionState({
            isSatisfactory: correctiveData.isSatisfactory !== undefined ? correctiveData.isSatisfactory : null,
            role: correctiveData.role || null,
            mechanicName: correctiveData.mechanicName || '',
            operatorName: correctiveData.operatorName || '',
          });
          setMechanicSignatureUri(correctiveData.mechanicSignatureUri || null);
          setOperatorSignatureUri(correctiveData.operatorSignatureUri || null);
          
          // Only update subItemStatuses if we have data and it's different from current
          if (correctiveData.subItemStatuses && Object.keys(correctiveData.subItemStatuses).length > 0) {
            console.log('Updating subItemStatuses with:', correctiveData.subItemStatuses);
            setSubItemStatuses(correctiveData.subItemStatuses);
          }
          
          setSignatureRefreshKey((prev) => prev + 1);
          setCombinedPdfDownloadUrl(correctiveData.pdfDownloadUrl || null);
        } else {
          console.log('No corrective action data found, keeping existing state');
        }
      } catch (error) {
        console.error('Error loading corrective action data:', error);
        setIsOnline(false);
        Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to load stored data. Working offline.' });
      } finally {
        setIsLoading(false);
      }
    };
    loadStoredData();

    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => {
      isMounted.current = false;
      unsubscribe();
    };
  }, [report?.id, report?.userUid, setSubItemStatuses, navigation]);

  const isValidSignatureUri = useCallback((uri) => uri && typeof uri === 'string' && uri.startsWith('data:image'), []);

  // Handle session state changes with logging
  const handleSessionStateChange = useCallback((updates) => {
    console.log('Session state changing from:', sessionState, 'to:', { ...sessionState, ...updates });
    setSessionState(prev => ({ ...prev, ...updates }));
  }, [sessionState]);

  const validateFields = useCallback(() => {
    const newErrors = {
      isSatisfactory: sessionState.isSatisfactory === null ? 'Please select if the corrective action is satisfactory' : '',
      role: sessionState.role === null ? 'Please select your role' : '',
      mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') && !sessionState.mechanicName.trim() ? 'Mechanic name is required' : '',
      operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') && !sessionState.operatorName.trim() ? 'Operator name is required' : '',
    };
    if ((sessionState.role === 'Mechanic' || sessionState.role === 'Both') && !isValidSignatureUri(mechanicSignatureUri)) {
      newErrors.mechanicName = 'Mechanic signature is required';
    }
    if ((sessionState.role === 'Operator' || sessionState.role === 'Both') && !isValidSignatureUri(operatorSignatureUri)) {
      newErrors.operatorName = 'Operator signature is required';
    }
    setErrors(newErrors);
    return Object.values(newErrors).every((error) => error === '');
  }, [sessionState, mechanicSignatureUri, operatorSignatureUri]);

  const debouncedValidate = useCallback(debounce(validateFields, 300), [validateFields]);

  useEffect(() => {
    debouncedValidate();
    return () => debouncedValidate.cancel();
  }, [sessionState, mechanicSignatureUri, operatorSignatureUri, debouncedValidate]);

  // Auto-save session state changes to Firebase
  useEffect(() => {
    const autoSaveSessionState = async () => {
      if (!report?.id || !isOnline) return;
      
      try {
        const userUid = await AsyncStorage.getItem('userUid');
        if (!userUid) return;
        
        // Only save if we have meaningful data
        if (sessionState.role || sessionState.mechanicName || sessionState.operatorName || sessionState.isSatisfactory !== null) {
          console.log('Auto-saving session state:', sessionState);
          
          const updateData = {
            isSatisfactory: sessionState.isSatisfactory,
            role: sessionState.role,
            mechanicName: sessionState.mechanicName,
            operatorName: sessionState.operatorName,
            mechanicSignatureUri,
            operatorSignatureUri,
            updatedAt: new Date().toISOString(),
          };
          
          await database()
            .ref(`/GlobalDVirCorectiveActionR/${userUid}/${report.id}`)
            .update(updateData);
        }
      } catch (error) {
        console.error('Error auto-saving session state:', error);
      }
    };
    
    // Debounce the auto-save to avoid too many Firebase calls
    const timeoutId = setTimeout(autoSaveSessionState, 1000);
    return () => clearTimeout(timeoutId);
  }, [sessionState, mechanicSignatureUri, operatorSignatureUri, report?.id, isOnline]);

  const generateCombinedPDFContent = () => {
    // Inspection Items
    let inspectionItemsHtml = '';
    if (Array.isArray(report.step2?.items)) {
      inspectionItemsHtml = report.step2.items
        .filter((item) => item.status && item.status !== '' && item.status !== 'N/A')
        .map(
          (item) => `
            <tr>
              <td>${item.sectionName || 'N/A'}</td>
              <td>${item.label || item.name || 'N/A'}</td>
              <td>${item.isInspectionRequired ? 'A' : 'B'}</td>
              <td style="text-align: center;">${item.status === 'Okay' ? '✓' : ''}</td>
              <td style="text-align: center;">${item.status === 'Faulty' ? 'X' : ''}</td>
              <td style="text-align: center;">${item.status === 'N/A' ? '—' : ''}</td>
              <td>${item.note || item.description || ''}</td>
            </tr>
          `
        )
        .join('');
    } else if (report.step2?.itemStates) {
      inspectionItemsHtml = Object.keys(report.step2.itemStates)
        .filter((key) => {
          const status = report.step2.itemStates[key].status;
          return status === 'No' || status === 'Fail' || status === 'faulty';
        })
        .map((key) => {
          const [sectionKey, itemId] = key.split('_');
          const itemName = report.step2.itemStates[key].name || `${sectionKey}_${itemId}`;
          const status = report.step2.itemStates[key].status;
          return `
            <tr>
              <td>${sectionKey}</td>
              <td>${itemName}</td>
              <td>${report.step2.itemStates[key].isInspectionRequired ? 'A' : 'B'}</td>
              <td style="text-align: center;">${status === 'Yes' ? '✓' : ''}</td>
              <td style="text-align: center;">${status === 'No' || status === 'Fail' || status === 'faulty' ? 'X' : ''}</td>
              <td style="text-align: center;">${status === '' ? '—' : ''}</td>
              <td>${report.step2.itemStates[key].note || ''}</td>
            </tr>
          `;
        })
        .join('');
    }

    // Corrective Action Items
    let correctiveItemsHtml = '';
    if (Array.isArray(report.step2?.items)) {
      correctiveItemsHtml = report.step2.items
        .filter((item) => item.status && item.status !== '' && item.status !== 'N/A')
        .map(
          (item) => `
            <tr>
              <td>${item.sectionName || 'N/A'}</td>
              <td>${item.label || item.name || 'N/A'}</td>
              <td>${item.category || 'N/A'}</td>
              <td>${subItemStatuses[item.itemId || item.id] || 'Pending'}</td>
            </tr>
          `
        )
        .join('');
    } else if (report.step2?.itemStates) {
      correctiveItemsHtml = Object.keys(report.step2.itemStates)
        .filter((key) => {
          const status = report.step2.itemStates[key].status;
          return status === 'No' || status === 'Fail' || status === 'faulty';
        })
        .map((key) => {
          const [sectionKey, itemId] = key.split('_');
          const itemName = report.step2.itemStates[key].name || `${sectionKey}_${itemId}`;
          return `
            <tr>
              <td>${sectionKey}</td>
              <td>${itemName}</td>
              <td>${report.step2.itemStates[key].category || 'N/A'}</td>
              <td>${subItemStatuses[key] || 'Pending'}</td>
            </tr>
          `;
        })
        .join('');
    }

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
        <meta charset="UTF-8"/>
        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
        <title>DVIR Corrective Action Report</title>
        <style>
          body {
            font-family: Arial, sans-serif;
            margin: 20px;
            font-size: 12px;
          }
          .header-row {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin-bottom: 10px;
          }
          .logo-left {
            display: flex;
            align-items: flex-start;
          }
          .logo-left img {
            width: 140px;
            height: 50px;
            object-fit: contain;
          }
          .doc-info-box {
            width: 340px;
          }
          .doc-info-box table {
            width: 100%;
            border: 1px solid #222;
            border-collapse: collapse;
            font-size: 12px;
          }
          .doc-info-box td {
            border: 1px solid #222;
            padding: 4px 8px;
            text-align: left;
          }
          .title-section {
            text-align: center;
            margin: 10px 0 0 0;
          }
          .title-section h2 {
            margin: 0;
            font-size: 18px;
            font-weight: bold;
          }
          .request-number {
            font-size: 14px;
            font-weight: bold;
            margin-top: 4px;
          }
          .info-boxes-row {
            display: flex;
            justify-content: space-between;
            align-items: flex-start;
            margin: 16px 0 10px 0;
          }
          .info-box {
            width: 45%;
          }
          .info-box table {
            width: 100%;
            border: 1px solid #222;
            border-collapse: collapse;
            font-size: 12px;
          }
          .info-box td {
            border: 1px solid #222;
            padding: 4px 8px;
            text-align: left;
          }
          .section-break {
            margin-top: 20px;
            border-top: 2px solid #015185;
            padding-top: 10px;
          }
          table.checklist, table.corrective-table {
            width: 100%;
            border-collapse: collapse;
            margin-bottom: 20px;
          }
          table.checklist th, table.checklist td, table.corrective-table th, table.corrective-table td {
            border: 1px solid #222;
            padding: 8px;
            text-align: center;
          }
          table.checklist th, table.corrective-table th {
            background-color: #f2f2f2;
            font-weight: bold;
          }
          .summary-section-table {
            width: 100%;
            margin-top: 30px;
            border: none;
          }
          .summary-section-table td {
            border: none;
            padding: 8px;
            font-size: 14px;
            text-align: left;
          }
          .summary-section-table .label {
            font-weight: bold;
            width: 180px;
          }
          .summary-section-table .signature-cell {
            text-align: right;
            vertical-align: middle;
            padding-right: 30px;
          }
          .signature img {
            max-width: 200px;
            height: auto;
          }
        </style>
      </head>
      <body>
        <div class="header-row">
          <div class="logo-left">
            <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Logo"/>
          </div>
          <div class="doc-info-box">
            <table>
              <tr><td>Document Name</td><td>DVIR Corrective Action Report</td></tr>
              <tr><td>Report ID</td><td>CA-${report.step1?.requestNumber || 'N/A'}</td></tr>
              <tr><td>Date</td><td>${new Date().toLocaleDateString()}</td></tr>
              <tr><td>Country</td><td>${report.step1?.country || 'N/A'}</td></tr>
              <tr><td>Project</td><td>${report.step1?.project || 'N/A'}</td></tr>
            </table>
          </div>
        </div>
        <div class="title-section">
          <h2>DVIR CORRECTIVE ACTION REPORT</h2>
          <div class="request-number">NO: CA-${report.step1?.requestNumber || 'N/A'}</div>
        </div>
        <div class="info-boxes-row">
          <div class="info-box">
            <table>
              <tr><td>DATE</td><td>${new Date().toLocaleDateString()}</td></tr>
              <tr><td>SITE / PROJECT</td><td>${report.step1?.project || 'N/A'}</td></tr>
            </table>
          </div>
          <div class="info-box">
            <table>
              <tr><td>VEHICLE No</td><td>${report.step1?.selectedEquipment?.equipmentName || 'N/A'}</td></tr>
              <tr><td>HOURS</td><td>${report.step1?.meterReading || 'N/A'} ${report.step1?.selectedEquipment?.meterUnit || 'Hours'}</td></tr>
            </table>
          </div>
        </div>
        <div class="section-break"></div>
        <h2>Inspection Items</h2>
        <table class="checklist">
          <tr>
            <th>Section</th>
            <th>Item</th>
            <th>Class</th>
            <th>Okay</th>
            <th>Faulty</th>
            <th>N/A</th>
            <th>Comment</th>
          </tr>
          ${inspectionItemsHtml || '<tr><td colspan="7">No inspection items available</td></tr>'}
        </table>
        <div class="section-break"></div>
        <h2>Corrective Action Items</h2>
        <div>
          <p><strong>Equipment Condition:</strong> ${sessionState.isSatisfactory ? 'Satisfactory' : 'Unsatisfactory'}</p>
          <p><strong>Role:</strong> ${sessionState.role || 'N/A'}</p>
          <p><strong>Mechanic Name:</strong> ${sessionState.mechanicName || 'N/A'}</p>
          <p><strong>Operator Name:</strong> ${sessionState.operatorName || 'N/A'}</p>
        </div>
        <table class="corrective-table">
          <tr>
            <th>Section</th>
            <th>Item</th>
            <th>Category</th>
            <th>Status</th>
          </tr>
          ${correctiveItemsHtml || '<tr><td colspan="4">No corrective action items available</td></tr>'}
        </table>
        <table class="summary-section-table">
          <tr>
            <td class="label">Mechanic Signature</td>
            <td>
              ${(sessionState.role === 'Mechanic' || sessionState.role === 'Both') && isValidSignatureUri(mechanicSignatureUri)
                ? `<img src="${mechanicSignatureUri}" alt="Mechanic Signature" style="width: 120px; height: 60px;"/>`
                : 'Not Provided'}
            </td>
            <td class="label">Operator Signature</td>
            <td>
              ${(sessionState.role === 'Operator' || sessionState.role === 'Both') && isValidSignatureUri(operatorSignatureUri)
                ? `<img src="${operatorSignatureUri}" alt="Operator Signature" style="width: 120px; height: 60px;"/>`
                : 'Not Provided'}
            </td>
          </tr>
        </table>
      </body>
      </html>
    `;
  };

  const generateAndSavePDF = async () => {
    if (!validateFields()) {
      Toast.show({ type: 'error', text1: 'Incomplete Data', text2: 'Please fill all required fields.' });
      return null;
    }
    setIsLoading(true);
    try {
      const combinedHtml = generateCombinedPDFContent();
      const options = {
        html: combinedHtml,
        fileName: `DVIRReport_${report.step1?.requestNumber || 'Unknown'}_${Date.now()}`,
        directory: Platform.OS === 'android' ? 'Documents' : 'DocumentDirectory',
      };
      const { filePath } = await RNHTMLtoPDF.convert(options);
      console.log('DVIR PDF generated at:', filePath);

      const userId = await AsyncStorage.getItem('userUid');
      const storageRef = storage().ref(`GlobalCorrectiveActions/${report.id}/pdfs/DVIRReport_${report.step1?.requestNumber || 'Unknown'}_${Date.now()}.pdf`);
      await storageRef.putFile(filePath);
      const downloadUrl = await storageRef.getDownloadURL();
      await RNFS.unlink(filePath).catch((err) => console.warn('Failed to clean up PDF:', err));

      return downloadUrl;
    } catch (error) {
      console.error('Error generating or uploading PDF:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to generate or upload PDF.' });
      return null;
    } finally {
      setIsLoading(false);
    }
  };

  const queueOperation = async (operation) => {
    try {
      let pendingOperations = [];
      const pendingOperationsRaw = await AsyncStorage.getItem('pendingOperations');
      pendingOperations = pendingOperationsRaw ? JSON.parse(pendingOperationsRaw) : [];
      pendingOperations.push(operation);
      await AsyncStorage.setItem('pendingOperations', JSON.stringify(pendingOperations));
      Toast.show({
        type: 'info',
        text1: 'Offline Mode',
        text2: 'Corrective action data queued for sync when online.',
      });
    } catch (error) {
      console.error('Error queuing operation:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to queue corrective action data.',
      });
    }
  };

  const saveToFirebase = async (pdfUrl) => {
    try {
      const userUid = await AsyncStorage.getItem('userUid');
      if (!userUid || !report?.id) return false;

      const correctiveActionData = {
        reportId: report.id,
        userUid: userUid,
        dvirNumber: report.dvirNumber || report.step1?.requestNumber,
        subItemStatuses,
        isSatisfactory: sessionState.isSatisfactory,
        role: sessionState.role,
        mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
        operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
        mechanicSignatureUri,
        operatorSignatureUri,
        pdfDownloadUrl: pdfUrl,
        createdAt: new Date().toISOString(),
        updatedAt: new Date().toISOString(),
        faultyCount: Object.keys(subItemStatuses).length,
        status: 'completed',
        // Add tracking fields
        originalReportData: {
          step1: report.step1,
          step2: report.step2,
          step3: report.step3,
        }
      };

      // Save to new corrective action path structure
      await database()
        .ref(`/GlobalDVirCorectiveActionR/${userUid}/${report.id}`)
        .set(correctiveActionData);

      // Update the main report with corrective action reference and PDF URL
      await database()
        .ref(`/GlobalDvirReports/admins/${userUid}/inspectionReports/${report.id}/correctiveAction`)
        .update({
          hasCorrectiveAction: true,
          correctiveActionId: report.id,
          correctiveActionNumber: `CA-${report.id}`,
          pdfDownloadUrl: pdfUrl,
          status: 'completed',
          updatedAt: new Date().toISOString(),
          faultyCount: Object.keys(subItemStatuses).length,
          isSatisfactory: sessionState.isSatisfactory,
          role: sessionState.role,
          mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
          operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
        });

      // Update local storage
      let storedReports = [];
      const storedData = await AsyncStorage.getItem('inspectionReports');
      storedReports = storedData ? JSON.parse(storedData) : [];
      
      const updatedReports = storedReports.map((r) =>
        r.id === report.id
          ? { 
              ...r, 
              correctiveAction: { 
                ...r.correctiveAction, 
                hasCorrectiveAction: true,
                correctiveActionId: report.id,
                correctiveActionNumber: `CA-${report.id}`,
                pdfDownloadUrl: pdfUrl,
                status: 'completed',
                updatedAt: new Date().toISOString(),
                faultyCount: Object.keys(subItemStatuses).length,
                isSatisfactory: sessionState.isSatisfactory,
                role: sessionState.role,
                mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
                operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
              } 
            }
          : r
      );
      
      if (!storedReports.some((r) => r.id === report.id)) {
        updatedReports.push({ 
          ...report, 
          correctiveAction: {
            hasCorrectiveAction: true,
            correctiveActionId: report.id,
            correctiveActionNumber: `CA-${report.id}`,
            pdfDownloadUrl: pdfUrl,
            status: 'completed',
            updatedAt: new Date().toISOString(),
            faultyCount: Object.keys(subItemStatuses).length,
            isSatisfactory: sessionState.isSatisfactory,
            role: sessionState.role,
            mechanicName: (sessionState.role === 'Mechanic' || sessionState.role === 'Both') ? sessionState.mechanicName : null,
            operatorName: (sessionState.role === 'Operator' || sessionState.role === 'Both') ? sessionState.operatorName : null,
          }
        });
      }
      
      await AsyncStorage.setItem('inspectionReports', JSON.stringify(updatedReports));

      // Process any pending operations
      let pendingOperations = [];
      const pendingData = await AsyncStorage.getItem('pendingOperations');
      pendingOperations = pendingData ? JSON.parse(pendingData) : [];
      
      for (const operation of pendingOperations) {
        if (operation.type === 'correctiveAction' && operation.reportId === report.id) {
          await database()
            .ref(`/GlobalDVirCorectiveActionR/${userUid}/${operation.reportId}`)
            .update(operation.correctiveAction);
        }
      }
      await AsyncStorage.setItem('pendingOperations', JSON.stringify([]));

      return true;
    } catch (error) {
      console.error('Error in saveToFirebase:', error);
      setIsOnline(false);
      await queueOperation({
        type: 'correctiveAction',
        reportId: report.id,
        userId: await AsyncStorage.getItem('userUid') || 'unknown',
        correctiveAction: { 
          reportId: report.id,
          userUid: await AsyncStorage.getItem('userUid'),
          dvirNumber: report.dvirNumber || report.step1?.requestNumber,
          subItemStatuses, 
          pdfDownloadUrl: pdfUrl,
          createdAt: new Date().toISOString(),
          updatedAt: new Date().toISOString(),
          faultyCount: Object.keys(subItemStatuses).length,
          status: 'completed'
        },
        path: `/GlobalDVirCorectiveActionR/${userUid}/${report.id}`,
      });
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save corrective action data. Saved offline.',
      });
      return false;
    }
  };

  const handleDone = async () => {
    if (isSubmitting || !isMounted.current) return;
    if (!validateFields()) return;
    setIsSubmitting(true);
    setIsLoading(true);
    try {
      let pdfUrl = null;
      if (savePdf && isOnline) {
        pdfUrl = await generateAndSavePDF();
        setCombinedPdfDownloadUrl(pdfUrl);
      }
      const saveSuccess = await saveToFirebase(pdfUrl);
      if (saveSuccess) {
        Toast.show({
          type: 'success',
          text1: 'Success',
          text2: pdfUrl ? 'Report saved and PDF uploaded' : 'Report saved successfully.',
        });
        if (isMounted.current && savePdf && pdfUrl) {
          setShowSuccessModal(true);
        } else {
          // Navigate back to DVIR home screen
          navigation.navigate('DviranddotInHome');
        }
      }
    } catch (error) {
      console.error('Error submitting corrective action:', error);
      Toast.show({ type: 'error', text1: 'Error', text2: 'Failed to submit corrective action. Please try again.' });
    } finally {
      if (isMounted.current) {
        setIsSubmitting(false);
        setIsLoading(false);
      }
    }
  };

  const handleMechanicSignature = () => {
    navigation.navigate('SignatureCaptureScreen', {
      role: 'Mechanic',
      onSave: (signature) => {
        setMechanicSignatureUri(signature);
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleOperatorSignature = () => {
    navigation.navigate('SignatureCaptureScreen', {
      role: 'Operator',
      onSave: (signature) => {
        setOperatorSignatureUri(signature);
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleClearSignature = (type) => {
    if (type === 'Mechanic') {
      setMechanicSignatureUri(null);
      setSessionState((prev) => ({ ...prev, mechanicName: '' }));
      setErrors((prev) => ({ ...prev, mechanicName: '' }));
    } else if (type === 'Operator') {
      setOperatorSignatureUri(null);
      setSessionState((prev) => ({ ...prev, operatorName: '' }));
      setErrors((prev) => ({ ...prev, operatorName: '' }));
    }
    setSignatureRefreshKey((prev) => prev + 1);
  };

  const handleImageError = (type) => {
    console.error(`${type} signature image failed to load`);
    Toast.show({ type: 'error', text1: 'Error', text2: `Failed to load ${type.toLowerCase()} signature. Please try again.` });
    handleClearSignature(type);
  };

  const handleRolePress = useCallback((value) => {
    const newRole = sessionState.role === value ? null : value;
    const updates = {
      role: newRole,
      mechanicName: newRole !== 'Mechanic' && newRole !== 'Both' ? '' : sessionState.mechanicName,
      operatorName: newRole !== 'Operator' && newRole !== 'Both' ? '' : sessionState.operatorName,
    };
    
    handleSessionStateChange(updates);
    setErrors((prev) => ({ ...prev, role: '' }));
    if (value !== 'Mechanic' && value !== 'Both') setMechanicSignatureUri(null);
    if (value !== 'Operator' && value !== 'Both') setOperatorSignatureUri(null);
    setSignatureRefreshKey((prev) => prev + 1);
  }, [sessionState.role, sessionState.mechanicName, sessionState.operatorName, handleSessionStateChange]);

  const handleSatisfactoryPress = useCallback((value) => {
    const newValue = sessionState.isSatisfactory === value ? null : value;
    handleSessionStateChange({ isSatisfactory: newValue });
    setErrors((prev) => ({ ...prev, isSatisfactory: '' }));
  }, [sessionState.isSatisfactory, handleSessionStateChange]);

  const showMechanicSection = sessionState.role === 'Mechanic' || sessionState.role === 'Both';
  const showOperatorSection = sessionState.role === 'Operator' || sessionState.role === 'Both';

  const handleViewPdf = () => {
    if (!combinedPdfDownloadUrl) {
      Toast.show({ type: 'error', text1: 'Error', text2: 'PDF not available for viewing.' });
      return;
    }
    navigation.navigate('PdfviewPage', { 
      pdfUrl: combinedPdfDownloadUrl, 
      onGoBack: () => {
        navigation.reset({
          index: 1,
          routes: [
            { name: 'MainApp' },
            { name: 'DviranddotInHome' },
          ],
        });
      } 
    });
    setShowSuccessModal(false);
  };

  const ProcessorModal = ({ visible }) => {
    const rotateAnim = useRef(new Animated.Value(0)).current;

    useEffect(() => {
      if (visible) {
        Animated.loop(
          Animated.timing(rotateAnim, {
            toValue: 1,
            duration: 900,
            useNativeDriver: true,
          })
        ).start();
      } else {
        rotateAnim.setValue(0);
      }
    }, [visible]);

    const rotate = rotateAnim.interpolate({
      inputRange: [0, 1],
      outputRange: ['0deg', '360deg'],
    });

    return (
      <Modal isVisible={visible} style={{ margin: 0 }} backdropOpacity={0.4}>
        <View style={styles.processorModalOuterCenter}>
          <View style={styles.processorModalContainer}>
            <Text style={styles.processorModalTitle}>Please Wait</Text>
            <View style={styles.processorRowCenter}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate }] }]}
              />
              <Text style={styles.processorModalText}>Loading</Text>
            </View>
          </View>
        </View>
      </Modal>
    );
  };

  if (!report || !report.step1 || !report.step2) {
    return (
      <SafeAreaProvider>
        <SafeAreaView style={styles.container} edges={['top', 'left', 'right']}>
          <Text style={styles.noDataText}>No report data available</Text>
        </SafeAreaView>
      </SafeAreaProvider>
    );
  }

  return (
    <SafeAreaProvider>
      <SafeAreaView style={styles.container} edges={['top', 'left', 'right']}>
        <ScrollView
          contentContainerStyle={[styles.contentContainer, { paddingBottom: insets.bottom + 60 }]}
          showsVerticalScrollIndicator={false}
          keyboardShouldPersistTaps="handled"
        >
          <View style={styles.sectionNoBG}>
            <Text style={styles.sectionTitleLargeBlue}>Corrective of vehicle is satisfactory</Text>
            <View style={styles.buttonRowFullWidth}>
              <TouchableOpacity
                style={[
                  styles.fullWidthButton,
                  sessionState.isSatisfactory === true && styles.fullWidthButtonSelected,
                ]}
                onPress={() => handleSatisfactoryPress(true)}
              >
                <Text
                  style={[
                    styles.fullWidthButtonText,
                    sessionState.isSatisfactory === true && styles.fullWidthButtonTextSelected,
                  ]}
                >
                  Yes
                </Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[
                  styles.fullWidthButton,
                  sessionState.isSatisfactory === false && styles.fullWidthButtonSelected,
                ]}
                onPress={() => handleSatisfactoryPress(false)}
              >
                <Text
                  style={[
                    styles.fullWidthButtonText,
                    sessionState.isSatisfactory === false && styles.fullWidthButtonTextSelected,
                  ]}
                >
                  No
                </Text>
              </TouchableOpacity>
            </View>
            {errors.isSatisfactory ? <Text style={styles.errorText}>{errors.isSatisfactory}</Text> : null}
          </View>

          <View style={styles.sectionNoBG}>
            <Text style={styles.sectionTitleLargeBlue}>Are you?</Text>
            <View style={styles.buttonRowFullWidth3}>
              {['Mechanic', 'Operator', 'Both'].map((role) => (
                <TouchableOpacity
                  key={role}
                  style={[styles.fullWidthButton3, sessionState.role === role && styles.fullWidthButtonSelected]}
                  onPress={() => handleRolePress(role)}
                >
                  <Text
                    style={[
                      styles.fullWidthButtonText,
                      sessionState.role === role && styles.fullWidthButtonTextSelected,
                    ]}
                  >
                    {role}
                  </Text>
                </TouchableOpacity>
              ))}
            </View>
            {errors.role ? <Text style={styles.errorText}>{errors.role}</Text> : null}
          </View>

          {showMechanicSection && (
            <View style={styles.sectionNoBG}>
              <Text style={styles.signatureLabelCenteredBlue}>Mechanic Signature</Text>
              <TextInput
                style={styles.signatureInput}
                placeholder="Mechanic Name"
                placeholderTextColor="#888"
                value={sessionState.mechanicName}
                onChangeText={(text) => handleSessionStateChange({ mechanicName: text })}
              />
              {errors.mechanicName ? <Text style={styles.errorText}>{errors.mechanicName}</Text> : null}
              <View style={{ height: 12 }} />
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(mechanicSignatureUri) ? (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleMechanicSignature}>
                    <Image
                      source={{ uri: mechanicSignatureUri }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={() => handleImageError('Mechanic')}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleMechanicSignature}>
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          )}

          {showMechanicSection && (
            <View style={styles.acknowledgeRowCenteredColumn}>
              <View style={styles.acknowledgeRowCenteredRow}>
                <MaterialIcons name="check-box" size={22} color="#28a745" style={{ alignSelf: 'center' }} />
                <Text style={styles.acknowledgeTitleCentered}>MECHANIC'S ACKNOWLEDGEMENT</Text>
              </View>
              <Text style={styles.acknowledgeDescCentered}>
                By signing, I acknowledge that I have checked all corrective action taken on vehicle.
              </Text>
            </View>
          )}

          {showOperatorSection && (
            <View style={styles.sectionNoBG}>
              <Text style={styles.signatureLabelCenteredBlue}>Operator Signature</Text>
              <TextInput
                style={styles.signatureInput}
                placeholder="Operator Name"
                placeholderTextColor="#888"
                value={sessionState.operatorName}
                onChangeText={(text) => handleSessionStateChange({ operatorName: text })}
              />
              {errors.operatorName ? <Text style={styles.errorText}>{errors.operatorName}</Text> : null}
              <View style={{ height: 12 }} />
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(operatorSignatureUri) ? (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleOperatorSignature}>
                    <Image
                      source={{ uri: operatorSignatureUri }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={() => handleImageError('Operator')}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity style={styles.plusIconCircleContainer} onPress={handleOperatorSignature}>
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          )}

          <View style={styles.section}>
            <View style={styles.checkboxRow}>
              <CheckBox value={savePdf} onValueChange={setSavePdf} tintColors={{ true: '#015185', false: '#888' }} />
              <Text style={styles.checkboxLabel}>Save as PDF</Text>
            </View>
          </View>

          <View style={styles.buttonContainer}>
            <TouchableOpacity
              style={styles.cancelButton}
              onPress={() => setActiveStep(1)}
              disabled={isSubmitting}
            >
              <Text style={styles.buttonText}>Back</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={[styles.submitButton, isSubmitting && styles.disabledButton]}
              onPress={handleDone}
              disabled={isSubmitting}
            >
              <Text style={[styles.buttonText, { color: '#fff' }]}>Done</Text>
            </TouchableOpacity>
          </View>
        </ScrollView>

        <Toast />

        <Modal isVisible={showSuccessModal} style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Success</Text>
            <MaterialIcons
              name="check"
              size={width * 0.1}
              color="#015185"
              style={{ alignSelf: 'center', marginVertical: width * 0.02 }}
            />
            <Text style={styles.modalSubtitle}>Report Generated and Saved</Text>
            <View style={styles.modalButtonContainer}>
              <TouchableOpacity
                style={styles.modalButton}
                                  onPress={() => {

                                    navigation.reset({
                                      index: 1,
                                      routes: [
                                        { name: 'MainApp' },
                                        { name: 'DviranddotInHome' },
                                      ],
                                    });


                
                  }}
              >
                <Text style={styles.modalButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity style={styles.modalButton} onPress={handleViewPdf}>
                <Text style={styles.modalButtonText}>View Report</Text>
              </TouchableOpacity>
            </View>
          </View>
        </Modal>

        <ProcessorModal visible={isLoading} />
      </SafeAreaView>
    </SafeAreaProvider>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  contentContainer: {
    paddingHorizontal: width * 0.04,
    paddingTop: width * 0.025,
    paddingBottom: width * 0.1,
  },
  section: {
    marginVertical: width * 0.025,
    padding: width * 0.025,
    backgroundColor: '#f0f0f0',
    borderRadius: width * 0.012,
  },
  sectionNoBG: {
    marginVertical: width * 0.025,
  },
  sectionTitleLargeBlue: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#1676b7',
    marginBottom: width * 0.01,
  },
  buttonRowFullWidth: {
    flexDirection: 'row',
    width: '100%',
    marginBottom: width * 0.01,
    gap: 8,
  },
  fullWidthButton: {
    flex: 1,
    backgroundColor: '#f2f6fa',
    borderRadius: 2,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.025,
  },
  fullWidthButtonSelected: {
    backgroundColor: '#1676b7',
  },
  fullWidthButtonText: {
    color: '#222',
    fontWeight: 'bold',
    fontSize: width * 0.04,
  },
  fullWidthButtonTextSelected: {
    color: '#fff',
  },
  buttonRowFullWidth3: {
    flexDirection: 'row',
    width: '100%',
    marginBottom: width * 0.01,
    gap: 8,
  },
  fullWidthButton3: {
    flex: 1,
    backgroundColor: '#f2f6fa',
    borderRadius: 2,
    alignItems: 'center',
    justifyContent: 'center',
    paddingVertical: width * 0.025,
  },
  signatureLabelCenteredBlue: {
    color: '#1676b7',
    fontWeight: 'bold',
    fontSize: width * 0.045,
    textAlign: 'center',
    marginBottom: 4,
    marginTop: width * 0.04,
  },
  signatureInput: {
    backgroundColor: '#f0f0f0',
    borderRadius: 2,
    paddingHorizontal: width * 0.025,
    paddingVertical: width * 0.025,
    fontSize: width * 0.038,
    color: '#333',
    marginBottom: 4,
    borderWidth: 0,
    width: '100%',
    alignSelf: 'center',
  },
  signatureBoxImage: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
    marginVertical: width * 0.025,
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: '100%',
    height: '100%',
    justifyContent: 'center',
    alignItems: 'center',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  acknowledgeRowCenteredColumn: {
    alignItems: 'center',
  },
  acknowledgeRowCenteredRow: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  acknowledgeTitleCentered: {
    fontSize: width * 0.04,
    fontWeight: 'bold',
    color: '#28a745',
    marginLeft: width * 0.02,
  },
  acknowledgeDescCentered: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
    marginTop: width * 0.015,
  },
  checkboxRow: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
  },
  checkboxLabel: {
    fontSize: width * 0.04,
    color: '#666',
    marginLeft: width * 0.02,
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.02,
  },
  cancelButton: {
    paddingVertical: width * 0.012,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.008,
    borderWidth: 1,
    borderColor: '#015185',
    alignItems: 'center',
    width: '40%',
  },
  submitButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.012,
    paddingHorizontal: width * 0.025,
    borderRadius: width * 0.008,
    alignItems: 'center',
    width: '40%',
  },
  disabledButton: {
    backgroundColor: '#ccc',
  },
  buttonText: {
    fontSize: width * 0.032,
    fontWeight: '500',
    color: '#333',
  },
  errorText: {
    color: '#ff4444',
    fontSize: width * 0.035,
    marginLeft: width * 0.015,
    marginBottom: width * 0.015,
  },
  noDataText: {
    fontSize: width * 0.04,
    color: '#666',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  modalOverlay: {
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    padding: width * 0.025,
    width: '80%',
    alignItems: 'center',
  },
  modalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: width * 0.015,
  },
  modalSubtitle: {
    fontSize: width * 0.035,
    color: '#333',
    textAlign: 'center',
    marginBottom: width * 0.025,
  },
  modalButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
    flexWrap: 'nowrap',
    gap: 10,
  },
  modalButton: {
    backgroundColor: '#015185',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.012,
    alignItems: 'center',
    flex: 1,
    marginHorizontal: 2,
    marginBottom: width * 0.015,
    minWidth: 0,
  },
  modalButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: '500',
  },
  processorModalOuterCenter: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  processorModalContainer: {
    backgroundColor: '#fff',
    borderRadius: 14,
    padding: 28,
    alignItems: 'center',
    justifyContent: 'center',
    alignSelf: 'center',
    width: width * 0.7,
  },
  processorModalTitle: {
    fontSize: 20,
    fontWeight: '500',
    color: '#222',
    marginBottom: 10,
    textAlign: 'center',
  },
  processorModalText: {
    fontSize: 16,
    color: '#015185',
    marginTop: 6,
    textAlign: 'center',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 12,
  },
});

export default DvirCorrectiveActionStep2;