import React, { useState, useEffect } from 'react';
import { View, Text, StyleSheet, StatusBar, Platform, Dimensions, TouchableOpacity, Modal, ActivityIndicator } from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import LinearGradient from 'react-native-linear-gradient';
import { useNavigation } from '@react-navigation/native';
import auth from '@react-native-firebase/auth';
import firestore from '@react-native-firebase/firestore';

const ViewProfilepage = () => {
  const navigation = useNavigation();
  const { width } = Dimensions.get('window');
  const TOTAL_HEADER_HEIGHT = width * 0.25;
  const [loading, setLoading] = useState(true);
  const [profileData, setProfileData] = useState({
    countries: [],
    projects: [],
    name: '',
    role: '',
  });

  useEffect(() => {
    // Fetch profile data using the same pattern as MainNavigation
    const fetchProfileData = async () => {
      try {
        setLoading(true);
        
        // First try to get data from AsyncStorage
        const email = await AsyncStorage.getItem('userEmail');
        const name = await AsyncStorage.getItem('userName');
        const role = await AsyncStorage.getItem('userRole');
        
        // Set initial data from AsyncStorage
        if (name || role) {
          setProfileData(prev => ({
            ...prev,
            name: name || '',
            role: role || '',
          }));
        }
        
        // Get Firebase auth user
        const user = auth().currentUser;
        if (user) {
          // Update with Firebase auth data if available
          if (user.displayName && !name) {
            setProfileData(prev => ({ ...prev, name: user.displayName }));
            await AsyncStorage.setItem('userName', user.displayName);
          }
          
          // Try to get additional data from Firestore users collection
          try {
            const userDoc = await firestore().collection('users').doc(user.uid).get();
            if (userDoc.exists) {
              const userData = userDoc.data();
              setProfileData({
                countries: userData.countries || [],
                projects: userData.projects || [],
                name: userData.name || user.displayName || name || '',
                role: userData.role || role || '',
              });
            } else {
              // If no Firestore document, use auth data
              setProfileData({
                countries: [],
                projects: [],
                name: user.displayName || name || '',
                role: role || '',
              });
            }
          } catch (firestoreError) {
            console.log('Firestore error, using auth data:', firestoreError);
            // Fallback to auth data if Firestore fails
            setProfileData({
              countries: [],
              projects: [],
              name: user.displayName || name || '',
              role: role || '',
            });
          }
        } else {
          // No auth user, use AsyncStorage data
          setProfileData({
            countries: [],
            projects: [],
            name: name || '',
            role: role || '',
          });
        }
        
      } catch (error) {
        console.error('Error fetching profile data:', error);
        // Set fallback data
        setProfileData({
          countries: [],
          projects: [],
          name: 'User',
          role: 'User',
        });
      } finally {
        setLoading(false);
      }
    };

    fetchProfileData();
  }, []);

  return (
    <View style={styles.container}>
      <StatusBar
        translucent
        backgroundColor="transparent"
        barStyle="light-content"
      />
      {/* Header with gradient */}
      <LinearGradient colors={['#3481BC', '#025383']} style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}>
        <View style={styles.headerContent}>
          <TouchableOpacity style={styles.backButton} onPress={() => navigation.goBack()}>
            <MaterialIcons name="arrow-back" size={24} color="white" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>User Profile</Text>
        </View>
      </LinearGradient>
      
      {/* Loading Modal */}
      <Modal
        transparent={true}
        visible={loading}
        animationType="fade"
      >
        <View style={styles.loadingModal}>
          <View style={styles.loadingContent}>
            <ActivityIndicator size="large" color="#3481BC" />
            <Text style={styles.loadingText}>Loading Profile...</Text>
          </View>
        </View>
      </Modal>

      {/* Profile Fields */}
      <View style={styles.content}>
        {/* Location/Country */}
        {/* <View style={styles.fieldContainer}>
          <MaterialIcons name="location-on" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>LOCATION/COUNTRY</Text>
            <Text style={styles.value}>
              {profileData.countries && profileData.countries.length > 0 
                ? profileData.countries.join(', ') 
                : 'Not set'}
            </Text>
          </View>
        </View> */}
        {/* Project */}
        {/* <View style={styles.fieldContainer}>
          <MaterialIcons name="work" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>PROJECT</Text>
            <Text style={styles.value}>
              {profileData.projects && profileData.projects.length > 0 
                ? profileData.projects.join(', ') 
                : 'Not set'}
            </Text>
          </View>
        </View> */}
        {/* Name */}
        <View style={styles.fieldContainer}>
          <MaterialIcons name="person" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>NAME</Text>
            <Text style={styles.value}>{profileData.name || 'Not set'}</Text>
          </View>
        </View>
        {/* Role */}
        <View style={styles.fieldContainer}>
          <MaterialIcons name="badge" size={24} color="#3481BC" style={styles.icon} />
          <View style={styles.fieldContent}>
            <Text style={styles.label}>ROLE</Text>
            <Text style={styles.value}>{profileData.role || 'Not set'}</Text>
          </View>
        </View>
      </View>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#f6f7f9',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: 16,
    paddingTop: Platform.OS === 'ios' ? 32 : 16,
    height: '100%',
    top: 7,
  },
  backButton: {
    padding: 8,
  },
  headerTitle: {
    color: 'white',
    fontSize: 20,
    fontWeight: 'bold',
    flex: 1,
    textAlign: 'center',
  },
  content: {
    flex: 1,
    padding: 20,
  },
  fieldContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    marginBottom: 22,
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 12,
    shadowColor: '#000',
    shadowOpacity: 0.04,
    shadowRadius: 4,
    shadowOffset: { width: 0, height: 2 },
    elevation: 2,
  },
  icon: {
    marginRight: 15,
  },
  fieldContent: {
    flex: 1,
    borderBottomWidth: 0,
    paddingBottom: 0,
  },
  label: {
    fontSize: 12,
    color: '#888',
    marginBottom: 5,
    fontWeight: 'bold',
    letterSpacing: 0.5,
  },
  value: {
    fontSize: 16,
    color: '#29486A',
    fontWeight: '500',
  },
  loadingModal: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  loadingContent: {
    backgroundColor: '#fff',
    padding: 20,
    borderRadius: 10,
    alignItems: 'center',
  },
  loadingText: {
    marginTop: 10,
    fontSize: 16,
    color: '#333',
  },
});

export default ViewProfilepage;