import React, { useState, useCallback } from 'react';
import { View, Text, StyleSheet, TouchableOpacity, Linking, FlatList, ScrollView, StatusBar, Dimensions, Platform } from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import { Dropdown } from 'react-native-element-dropdown';
import contactsData from './ViewNotification/Contact_Us_Json.json'; // Adjust path as needed
import Toast from 'react-native-toast-message';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.25;

const Conatct_US_AC = () => {
  const navigation = useNavigation();
  const [selectedCountry, setSelectedCountry] = useState(null);
  const [selectedProject, setSelectedProject] = useState(null);

  // Dropdown data
  const countries = [
    { label: 'Zambia', value: 'Zambia' },
    { label: 'DRC', value: 'DRC' },
  ];

  const getProjects = useCallback(() => {
    if (!selectedCountry) return [];
    return contactsData[selectedCountry]?.projects.map(project => ({
      label: project.projectName,
      value: project.projectName,
    })) || [];
  }, [selectedCountry]);

  // Get all contacts for the selected project and additional sections
  const getContacts = useCallback(() => {
    if (!selectedCountry || !selectedProject) return [];

    const projectContacts = contactsData[selectedCountry]?.projects.find(
      project => project.projectName === selectedProject
    )?.contacts || [];

    const additionalContacts = [
      ...(contactsData[selectedCountry]?.hr || []),
      ...(contactsData[selectedCountry]?.finance || []),
      ...(contactsData[selectedCountry]?.incidentNotification || []),
      ...(contactsData[selectedCountry]?.clerk || []),
    ];

    // Combine and remove duplicates based on email
    const allContacts = [...projectContacts, ...additionalContacts];
    const uniqueContacts = [];
    const seenEmails = new Set();

    allContacts.forEach(contact => {
      if (!seenEmails.has(contact.email)) {
        uniqueContacts.push(contact);
        seenEmails.add(contact.email);
      }
    });

    return uniqueContacts;
  }, [selectedCountry, selectedProject]);

  const handleLink = useCallback((url) => {
    Linking.openURL(url).catch(err => {
      console.error('Failed to open URL:', err);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to open link.',
        position: 'bottom',
      });
    });
  }, []);

  const renderContact = ({ item }) => (
    <TouchableOpacity
      style={styles.row}
      onPress={() => handleLink(`mailto:${item.email}`)}
      activeOpacity={0.7}
      accessibilityRole="button"
      accessibilityLabel={`Email ${item.name}`}
      accessibilityHint={`Send an email to ${item.name}`}
    >
      <MaterialIcons name="email" size={24} color="#035484" style={styles.icon} />
      <View>
        <Text style={styles.linkText}>{item.name}</Text>
        <Text style={styles.positionText}>{item.position}</Text>
        <Text style={styles.emailText}>{item.email}</Text>
      </View>
    </TouchableOpacity>
  );

  return (
    <View style={styles.screen}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar translucent backgroundColor="transparent" barStyle="light-content" />
        <View style={styles.headerContent}>
          <TouchableOpacity
            style={styles.backButton}
            onPress={() => navigation.goBack()}
            activeOpacity={0.8}
          >
            <MaterialIcons name="arrow-back" size={24} color="#fff" />
          </TouchableOpacity>
          <Text style={styles.headerTitle}>Contact Us</Text>
          <View style={{ width: 24 }} />
        </View>
      </LinearGradient>
      <ScrollView
        style={styles.scrollContainer}
        contentContainerStyle={styles.scrollContent}
        showsVerticalScrollIndicator={false}
      >
        <View style={styles.card}>
          <Text style={styles.title}>How can we help you?</Text>
       

          {/* Country Dropdown */}
          <Text style={styles.dropdownLabel}>Select Country</Text>
          <Dropdown
            style={styles.dropdown}
            data={countries}
            labelField="label"
            valueField="value"
            placeholder="Select a country"
            value={selectedCountry}
            onChange={item => {
              setSelectedCountry(item.value);
              setSelectedProject(null); // Reset project selection
            }}
            containerStyle={styles.dropdownContainer}
            selectedTextStyle={styles.dropdownText}
            placeholderStyle={styles.dropdownPlaceholder}
            accessibilityLabel="Select Country"
            accessibilityHint="Choose a country to view projects"
          />

          {/* Project Dropdown */}
          {selectedCountry && (
            <>
              <Text style={styles.dropdownLabel}>Select Project</Text>
              <Dropdown
                style={styles.dropdown}
                data={getProjects()}
                labelField="label"
                valueField="value"
                placeholder="Select a project"
                value={selectedProject}
                onChange={item => setSelectedProject(item.value)}
                containerStyle={styles.dropdownContainer}
                selectedTextStyle={styles.dropdownText}
                placeholderStyle={styles.dropdownPlaceholder}
                accessibilityLabel="Select Project"
                accessibilityHint="Choose a project to view contacts"
              />
            </>
          )}

          {/* Contacts List */}
          {selectedProject && (
            <FlatList
              data={getContacts()}
              renderItem={renderContact}
              keyExtractor={(item, index) => `${item.email}-${index}`}
              style={styles.contactsList}
              nestedScrollEnabled={false} // Disable FlatList scrolling to let ScrollView handle it
              ListEmptyComponent={<Text style={styles.noContactsText}>No contacts available for this project.</Text>}
            />
          )}
        </View>
        <View>
          <Text>
          {/* text */}

          </Text>
          <Text style={styles.subtitle}>{"\n"}</Text>
        </View>
      </ScrollView>
    </View>
  );
};

const styles = StyleSheet.create({
  screen: {
    flex: 1,
    backgroundColor: '#f6f7f9',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
    elevation: 4,
    shadowColor: '#000',
    shadowOpacity: 0.1,
    shadowRadius: 8,
    shadowOffset: { width: 0, height: 4 },
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: Platform.OS === 'ios' ? width * 0.08 : width * 0.04,
    height: '100%',
    top:6
  },
  backButton: {
    padding: 4,
    marginRight: 8,
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.05,
    fontWeight: 'bold',
    letterSpacing: 0.5,
    textAlign: 'center',
    flex: 1,
  },
  scrollContainer: {
    flex: 1,
    width: '100%',
  },
  scrollContent: {
    paddingHorizontal: 16,
    paddingBottom: 20, // Ensure padding at the bottom for scrollable content
  },
  card: {
    width: '100%',
    borderRadius: 14,
    paddingVertical: 18,
    paddingHorizontal: 0,
    alignItems: 'center',
  },
  title: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 2,
  },
  subtitle: {
    fontSize: 15,
    color: '#29486A',
    textAlign: 'center',
    marginBottom: 10,
  },
  dropdownLabel: {
    fontSize: 15,
    color: '#29486A',
    fontWeight: 'bold',
    marginBottom: 6,
    marginLeft: 2,
    alignSelf: 'flex-start',
    width: '100%',
  },
  dropdown: {
    borderWidth: 1,
    borderColor: '#bcd2ee',
    borderRadius: 8,
    paddingHorizontal: 14,
    paddingVertical: 12,
    backgroundColor: '#f6f7fa',
    marginBottom: 18,
    minHeight: 48,
    justifyContent: 'center',
    width: '100%',
  },
  dropdownContainer: {
    borderRadius: 8,
    backgroundColor: '#fff',
    borderWidth: 0,
  },
  dropdownText: {
    fontSize: 16,
    color: '#222',
    textAlignVertical: 'center',
    includeFontPadding: false,
  },
  dropdownPlaceholder: {
    fontSize: 16,
    color: '#888',
    textAlignVertical: 'center',
    includeFontPadding: false,
  },
  contactsList: {
    width: '100%',
    marginVertical: 12,
  },
  row: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 8,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  icon: {
    marginRight: 10,
  },
  linkText: {
    color: '#3481BC',
    fontSize: 15,
    fontWeight: 'bold',
  },
  positionText: {
    fontSize: 14,
    color: '#666',
  },
  emailText: {
    color: '#3481BC',
    fontSize: 14,
    textDecorationLine: 'underline',
  },
  noContactsText: {
    fontSize: 14,
    color: '#888',
    textAlign: 'center',
    marginVertical: 10,
  },
});

export default Conatct_US_AC;