import React, { useState, useEffect, useRef, useCallback } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  StatusBar,
  ScrollView,
  Modal,
  Animated,
  Image,
  ToastAndroid,
  Platform,
  ActivityIndicator,
  Dimensions,
} from 'react-native';
import LinearGradient from 'react-native-linear-gradient';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import Ionicons from 'react-native-vector-icons/Ionicons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import { useNavigation, useFocusEffect, useRoute } from '@react-navigation/native';
import { Dropdown } from 'react-native-element-dropdown';
import Toast from 'react-native-toast-message';
import database from '@react-native-firebase/database';
import auth from '@react-native-firebase/auth';
import NetInfo from '@react-native-community/netinfo';

const { width } = Dimensions.get('window');
const TOTAL_HEADER_HEIGHT = width * 0.45;
const INITIAL_ITEMS_TO_LOAD = 15;
const ITEMS_PER_PAGE = 10;

const HSEselecteq = () => {
  const navigation = useNavigation();
  const route = useRoute();
  const { selectedEquipment: initialSelected, onSelect, source } = route.params || {};
  
  // Debug: Log route parameters
  console.log('Service_select_equipment route params:', route.params);
  console.log('onSelect callback:', onSelect);
  console.log('source:', source);
  const [searchQuery, setSearchQuery] = useState('');
  const [firebaseEquipment, setFirebaseEquipment] = useState([]);
  const [filteredEquipment, setFilteredEquipment] = useState([]);
  const [displayedEquipment, setDisplayedEquipment] = useState([]);
  const [page, setPage] = useState(1);
  const [isLoadingMore, setIsLoadingMore] = useState(false);
  const [isLoadingInitial, setIsLoadingInitial] = useState(true);
  const [hasMore, setHasMore] = useState(true);
  const [actionModalVisible, setActionModalVisible] = useState(false);
  const [statusModalVisible, setStatusModalVisible] = useState(false);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [showDeleteConfirm, setShowDeleteConfirm] = useState(false);
  const [selectedEquipment, setSelectedEquipment] = useState(initialSelected || null);
  const [newStatus, setNewStatus] = useState('Available');
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [filterModalVisible, setFilterModalVisible] = useState(false);
  const [selectedFilter, setSelectedFilter] = useState(null);
  const [selectedCountry, setSelectedCountry] = useState('All Countries');
  const [countryDropdownVisible, setCountryDropdownVisible] = useState(false);
  const [countryDropdownPosition, setCountryDropdownPosition] = useState({ x: 0, y: 0, width: 0 });
  const [userInfoModalVisible, setUserInfoModalVisible] = useState(false);
  const [userInfo, setUserInfo] = useState({
    countries: [],
    projects: [],
    equipmentPermissions: [],
    employeeNumber: '',
    name: '',
    userUid: '',
  });
  const countryButtonRef = useRef(null);
  const permissionsButtonRef = useRef(null);
  const onSelectCallbackRef = useRef(onSelect);

  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });

  const checkNetworkStatus = async () => {
    try {
      const netInfo = await NetInfo.fetch();
      return netInfo.isConnected;
    } catch (error) {
      console.error('Error checking network status:', error);
      return false;
    }
  };

  const showToast = (type, text1, text2) => {
    if (Platform.OS === 'android') {
      ToastAndroid.show(`${text1}: ${text2}`, ToastAndroid.LONG);
    } else {
      Toast.show({ type, text1, text2, position: 'bottom' });
    }
  };

  const fetchUserInfo = async () => {
    try {
      const user = auth().currentUser;
      if (!user) {
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return null;
      }

      let userUid = await AsyncStorage.getItem('userUid');
      let userData = null;

      if (userUid) {
        const userRef = database().ref(`/Globalusers/admins/${userUid}`);
        const snapshot = await userRef.once('value');
        userData = snapshot.val();
      }

      if (!userData) {
        const email = user.email.replace(/\./g, '_');
        const userRef = database().ref(`/Globalusers/admins/${email}`);
        const snapshot = await userRef.once('value');
        userData = snapshot.val();
        if (userData && userData.userUid) {
          userUid = userData.userUid;
          await AsyncStorage.setItem('userUid', userUid);
        } else {
          userUid = email; // Fallback to email if no userUid
          await AsyncStorage.setItem('userUid', userUid);
        }
      }

      if (!userData) {
        showToast('error', 'Error', 'User data not found.');
        return null;
      }

      const equipmentModule = userData.modules?.find(
        (module) => module.module?.toLowerCase() === 'equipments'
      ) || { permissions: [] };

      const userInfo = {
        countries: userData.countries || [],
        projects: userData.projects || [],
        equipmentPermissions: equipmentModule.permissions || [],
        employeeNumber: userData.employeeNumber || '',
        name: userData.name || '',
        userUid: userUid || user.uid,
      };

      await AsyncStorage.setItem('userInfo', JSON.stringify(userInfo));
      setUserInfo(userInfo);
      return userInfo;
    } catch (error) {
      console.error('Error fetching user info:', error);
      showToast('error', 'Error', 'Failed to fetch user permissions.');
      return null;
    }
  };

  const filterEquipmentByPermission = useCallback((equipmentArr, userInfoObj) => {
    if (!equipmentArr || !userInfoObj) return [];

    if (userInfoObj.equipmentPermissions.includes('onlyMineView')) {
      const creatorIdentifier = userInfoObj.userUid;
      return equipmentArr
        .filter(
          (item) =>
            (item.createdBy === creatorIdentifier || !item.createdBy) && // Fallback if createdBy is missing
            userInfoObj.countries.includes(item.country)
        )
        .map((item) => ({
          ...item,
          createdBy: item.createdBy || userInfoObj.userUid, // Add createdBy if missing
          equipmentName: item.equipmentName
            ? item.equipmentName.charAt(0).toUpperCase() + item.equipmentName.slice(1).toLowerCase()
            : item.equipmentName,
        }));
    } else if (userInfoObj.equipmentPermissions.includes('view')) {
      return equipmentArr
        .filter((item) => userInfoObj.countries.includes(item.country))
        .map((item) => ({
          ...item,
          equipmentName: item.equipmentName
            ? item.equipmentName.charAt(0).toUpperCase() + item.equipmentName.slice(1).toLowerCase()
            : item.equipmentName,
        }));
    } else {
      return [];
    }
  }, []);

  const loadData = async (isRefresh = false) => {
    setSyncModalVisible(true);
    Animated.loop(
      Animated.timing(rotateAnim, {
        toValue: 1,
        duration: 1000,
        useNativeDriver: true,
      })
    ).start();

    try {
      const user = auth().currentUser;
      if (!user) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'User not authenticated. Please log in.');
        navigation.replace('AdminLogin');
        return;
      }

      const userInfoData = await fetchUserInfo();
      if (!userInfoData) {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        return;
      }

      console.log('User Info:', userInfoData); // Debug: Check user info and permissions
      const isOnline = await checkNetworkStatus();
      let parsedData = [];

      if (isOnline) {
        if (userInfoData.equipmentPermissions.includes('onlyMineView')) {
          const equipmentRef = database().ref(`/equipment/${userInfoData.userUid}`);
          const snapshot = await equipmentRef.once('value');
          const equipmentData = snapshot.val() || {};

          console.log('Raw Equipment Data:', equipmentData); // Debug: Check raw data from Firebase

          parsedData = Object.keys(equipmentData).map((key) => ({
            ...equipmentData[key],
            firebaseKey: key,
          }));

          console.log('Parsed Equipment Data:', parsedData); // Debug: Check parsed data

          const userRef = database().ref(`/Globalusers/admins/${userInfoData.userUid}`);
          const userSnapshot = await userRef.once('value');
          const userData = userSnapshot.val();
          if (!userData || !userData.countries || userData.countries.length === 0) {
            parsedData = [];
            console.log('No user data or countries found, setting parsedData to empty');
          } else {
            parsedData = parsedData.filter((item) =>
              userData.countries.includes(item.country)
            );
            console.log('Filtered by countries:', parsedData);
          }
        } else {
          const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
          const snapshot = await globalRef.once('value');
          const data = snapshot.val() || [];

          parsedData = Array.isArray(data)
            ? data.map((item, index) => ({
                ...item,
                firebaseKey: `equip_${index}`,
              }))
            : [];
        }

        parsedData = parsedData
          .filter((item, index, self) => self.findIndex((t) => t.id === item.id) === index)
          .sort((a, b) => {
            const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
            const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
            return dateB - dateA;
          });

        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        console.log('Data after filterEquipmentByPermission:', filteredData); // Debug: Check final filtered data

        parsedData = filteredData;
        await AsyncStorage.setItem('equipment', JSON.stringify(parsedData));
      } else {
        const savedData = await AsyncStorage.getItem('equipment');
        parsedData = savedData ? JSON.parse(savedData) : [];
        const filteredData = filterEquipmentByPermission(parsedData, userInfoData);
        console.log('Offline Data after filtering:', filteredData); // Debug: Check offline data
        parsedData = filteredData;
        parsedData.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        showToast('info', 'Offline', 'Equipment data refreshed from local storage.');
      }

      if (isRefresh) {
        setFirebaseEquipment(parsedData);
      } else {
        setFirebaseEquipment((prev) => [...prev, ...parsedData]);
      }

      setFilteredEquipment(parsedData);
      setDisplayedEquipment(parsedData.slice(0, INITIAL_ITEMS_TO_LOAD));
      setHasMore(parsedData.length > INITIAL_ITEMS_TO_LOAD);

      const storedCountry = await AsyncStorage.getItem('selectedCountry');
      setSelectedCountry(storedCountry || 'All Countries');
    } catch (error) {
      console.error('Error loading equipment:', error.code, error.message);
      let errorMessage = 'Failed to load equipment.';
      if (error.code === 'PERMISSION_DENIED') {
        errorMessage = 'Permission denied. Check Firebase security rules.';
      } else if (error.code === 'NETWORK_ERROR') {
        errorMessage = 'Offline: Loading from local storage.';
        const savedData = await AsyncStorage.getItem('equipment');
        const parsedData = savedData ? JSON.parse(savedData) : [];
        const userInfoData = await AsyncStorage.getItem('userInfo');
        const userInfoParsed = userInfoData ? JSON.parse(userInfoData) : userInfo;
        parsedData.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        const filteredData = filterEquipmentByPermission(parsedData, userInfoParsed);
        setFirebaseEquipment(filteredData);
        setFilteredEquipment(filteredData);
        setDisplayedEquipment(filteredData.slice(0, INITIAL_ITEMS_TO_LOAD));
        setHasMore(filteredData.length > INITIAL_ITEMS_TO_LOAD);
      }
      showToast(
        error.code === 'NETWORK_ERROR' ? 'info' : 'error',
        error.code === 'NETWORK_ERROR' ? 'Offline' : 'Error',
        errorMessage
      );
    } finally {
      setSyncModalVisible(false);
      rotateAnim.setValue(0);
      setIsRefreshing(false);
      setIsLoadingMore(false);
      setIsLoadingInitial(false);
    }
  };

  useEffect(() => {
    loadData(true);
  }, []);

  // Update callback ref when route params change
  useEffect(() => {
    onSelectCallbackRef.current = onSelect;
  }, [onSelect]);

  useFocusEffect(
    useCallback(() => {
      handleRefresh();
    }, [])
  );

  useEffect(() => {
    const syncPendingChanges = async () => {
      const isOnline = await checkNetworkStatus();
      if (!isOnline) return;

      const user = auth().currentUser;
      if (!user) return;

      try {
        const existing = await AsyncStorage.getItem('equipment');
        const equipment = existing ? JSON.parse(existing) : [];
        const pending = equipment.filter((item) => item.pendingSync);

        if (pending.length > 0) {
          setSyncModalVisible(true);
          Animated.loop(
            Animated.timing(rotateAnim, {
              toValue: 1,
              duration: 1000,
              useNativeDriver: true,
            })
          ).start();

          const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
          const globalSnapshot = await globalRef.once('value');
          let globalData = globalSnapshot.val() || [];

          for (const item of pending) {
            if (item.pendingDelete) {
              globalData = globalData.filter((g) => g.id !== item.id);
            } else {
              const updatedItem = { ...item, pendingSync: false, pendingDelete: false };
              const globalIndex = globalData.findIndex((g) => g.id === item.id);
              if (globalIndex !== -1) {
                globalData[globalIndex] = updatedItem;
              } else {
                globalData.push(updatedItem);
              }
            }
          }

          await globalRef.set(globalData);

          const updatedEquipment = equipment
            .filter((item) => !item.pendingDelete)
            .map((item) => ({
              ...item,
              pendingSync: false,
              pendingDelete: false,
            }));
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedEquipment));

          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('success', 'Success', 'Pending changes synced to database.');
        }
      } catch (error) {
        console.error('Error syncing pending changes:', error);
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        showToast('error', 'Error', 'Failed to sync pending changes.');
      }
    };

    const unsubscribe = NetInfo.addEventListener((state) => {
      if (state.isConnected) {
        syncPendingChanges();
      }
    });

    return () => unsubscribe();
  }, []);

  const handleSearch = (query) => {
    setSearchQuery(query);
    setPage(1);
    filterEquipment(query, selectedFilter, selectedCountry);
  };

  const filterEquipment = (query, filterType, country) => {
    let filtered = filterEquipmentByPermission(firebaseEquipment, userInfo);

    if (country && country !== 'All Countries') {
      filtered = filtered.filter((item) => item.country === country);
    }

    if (query) {
      filtered = filtered.filter((item) => {
        switch (filterType) {
          case 'Created Date':
            return new Date(item.createdAt)
              .toLocaleDateString()
              .toLowerCase()
              .includes(query.toLowerCase());
          case 'Equipment Number':
            return (item.equipmentNumber?.toLowerCase() || '').includes(query.toLowerCase());
          case 'Equipment Name':
            return (item.equipmentName?.toLowerCase() || '').includes(query.toLowerCase());
          case 'Equipment Categories':
            return (item.mainCategory?.toLowerCase() || '').includes(query.toLowerCase());
          case 'Meter Reading':
            return `${item?.meterReading || ''}`.toLowerCase().includes(query.toLowerCase());
          case 'Status':
            return (item.status?.toLowerCase() || '').includes(query.toLowerCase());
          default:
            return (
              (item?.equipmentName?.toLowerCase() || '').includes(query.toLowerCase()) ||
              (item?.equipmentNumber?.toLowerCase() || '').includes(query.toLowerCase()) ||
              (item?.mainCategory?.toLowerCase() || '').includes(query.toLowerCase()) ||
              (item?.status?.toLowerCase() || '').includes(query.toLowerCase())
            );
        }
      });
    }

    setFilteredEquipment(filtered);
    setDisplayedEquipment(filtered.slice(0, INITIAL_ITEMS_TO_LOAD));
    setHasMore(filtered.length > INITIAL_ITEMS_TO_LOAD);
    setPage(1);
  };

  const handleCountryFilterPress = () => {
    if (countryButtonRef.current) {
      countryButtonRef.current.measureInWindow((x, y, width, height) => {
        setCountryDropdownPosition({ x, y: y + height + 4, width });
        setCountryDropdownVisible(true);
      });
    }
  };

  const handleCountrySelect = (country) => {
    setSelectedCountry(country);
    AsyncStorage.setItem('selectedCountry', country);
    filterEquipment(searchQuery, selectedFilter, country);
    setCountryDropdownVisible(false);
  };

  const handleFilterSelect = (filter) => {
    setSelectedFilter(filter);
    setFilterModalVisible(false);
    filterEquipment(searchQuery, filter, selectedCountry);
  };

  const handleRefresh = async () => {
    setIsRefreshing(true);
    setPage(1);
    setHasMore(true);
    await loadData(true);
  };

  const handleScroll = ({ nativeEvent }) => {
    const { layoutMeasurement, contentOffset, contentSize } = nativeEvent;
    const isBottom = layoutMeasurement.height + contentOffset.y >= contentSize.height - 20;
    if (isBottom && hasMore && !isLoadingMore) {
      setIsLoadingMore(true);
      setPage((prev) => prev + 1);
    }
  };

  useEffect(() => {
    const startIndex = (page - 1) * ITEMS_PER_PAGE;
    const endIndex = startIndex + ITEMS_PER_PAGE;
    setDisplayedEquipment(filteredEquipment.slice(0, INITIAL_ITEMS_TO_LOAD + endIndex));
    setHasMore(filteredEquipment.length > INITIAL_ITEMS_TO_LOAD + endIndex);
  }, [page, filteredEquipment]);

  const handleActionPress = (item) => {
    if (!userInfo.equipmentPermissions.includes('view') && !userInfo.equipmentPermissions.includes('onlyMineView')) {
      showToast('error', 'Permission Denied', 'You do not have permission to view equipment.');
      return;
    }
    setSelectedEquipment(item);
    setActionModalVisible(true);
    setNewStatus(item.status || 'Available');
  };

  const handleStatusChange = async () => {
    if (!userInfo.equipmentPermissions.includes('update')) {
      showToast('error', 'Permission Denied', 'You do not have permission to update equipment.');
      return;
    }
    if (selectedEquipment) {
      const updatedEquipment = {
        ...selectedEquipment,
        status: newStatus,
        updatedAt: new Date().toISOString(),
        createdBy: userInfo.userUid, // Ensure createdBy is set during updates
      };
      let updatedFirebaseList;

      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      try {
        const user = auth().currentUser;
        if (!user) {
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('error', 'Error', 'User not authenticated. Please log in.');
          navigation.replace('AdminLogin');
          return;
        }

        const isOnline = await checkNetworkStatus();

        updatedFirebaseList = firebaseEquipment.map((item) =>
          item.id === updatedEquipment.id ? updatedEquipment : item
        );
        updatedFirebaseList.sort((a, b) => {
          const dateA = a.updatedAt ? new Date(a.updatedAt) : new Date(a.createdAt);
          const dateB = b.updatedAt ? new Date(b.updatedAt) : new Date(b.createdAt);
          return dateB - dateA;
        });
        setFirebaseEquipment(updatedFirebaseList);
        setFilteredEquipment(updatedFirebaseList);
        setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));

        if (isOnline) {
          const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
          const globalSnapshot = await globalRef.once('value');
          let globalData = globalSnapshot.val() || [];
          const globalIndex = globalData.findIndex((g) => g.id === updatedEquipment.id);
          if (globalIndex !== -1) {
            globalData[globalIndex] = updatedEquipment;
          } else {
            globalData.push(updatedEquipment);
          }
          await globalRef.set(globalData);

          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          showToast('success', 'Success', 'Status updated in database.');
        } else {
          updatedFirebaseList = updatedFirebaseList.map((item) =>
            item.id === updatedEquipment.id ? { ...item, pendingSync: true } : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          showToast('info', 'Offline', 'Status updated locally. Will sync when online.');
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
        }
      } catch (error) {
        console.error('Error updating status:', error.code, error.message);
        let errorMessage = 'Failed to update status.';
        if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Check Firebase security rules.';
        } else if (error.code === 'NETWORK_ERROR') {
          errorMessage = 'Offline: Status updated locally, will sync when online.';
          updatedFirebaseList = firebaseEquipment.map((item) =>
            item.id === updatedEquipment.id ? { ...updatedEquipment, pendingSync: true } : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
          showToast('info', 'Offline', errorMessage);
          return;
        }
        showToast('error', 'Error', errorMessage);
      } finally {
        setSyncModalVisible(false);
        rotateAnim.setValue(0);
        setStatusModalVisible(false);
        setActionModalVisible(false);
      }
    }
  };

  const handleDelete = async () => {
    if (!userInfo.equipmentPermissions.includes('delete')) {
      showToast('error', 'Permission Denied', 'You do not have permission to delete equipment.');
      return;
    }
    if (selectedEquipment) {
      let updatedFirebaseList;

      setSyncModalVisible(true);
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();

      try {
        const user = auth().currentUser;
        if (!user) {
          setSyncModalVisible(false);
          rotateAnim.setValue(0);
          showToast('error', 'Error', 'User not authenticated. Please log in.');
          navigation.replace('AdminLogin');
          return;
        }

        const isOnline = await checkNetworkStatus();

        updatedFirebaseList = firebaseEquipment.filter((item) => item.id !== selectedEquipment.id);
        setFirebaseEquipment(updatedFirebaseList);
        setFilteredEquipment(updatedFirebaseList);
        setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));

        if (isOnline) {
          const globalRef = database().ref('EquipmentsDataJSON/GlobalEquipmentJsonData');
          const globalSnapshot = await globalRef.once('value');
          let globalData = globalSnapshot.val() || [];
          globalData = globalData.filter((g) => g.id !== selectedEquipment.id);
          await globalRef.set(globalData);

          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          showToast('success', 'Success', 'Equipment deleted from database.');
        } else {
          updatedFirebaseList = firebaseEquipment.map((item) =>
            item.id === selectedEquipment.id
              ? { ...item, pendingSync: true, pendingDelete: true }
              : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          showToast('info', 'Offline', 'Equipment deleted locally. Will sync when online.');
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
        }
      } catch (error) {
        console.error('Error deleting equipment:', error.code, error.message);
        let errorMessage = 'Failed to delete equipment.';
        if (error.code === 'PERMISSION_DENIED') {
          errorMessage = 'Permission denied. Check Firebase security rules.';
        } else if (error.code === 'NETWORK_ERROR') {
          errorMessage = 'Offline: Equipment deleted locally, will sync when online.';
          updatedFirebaseList = firebaseEquipment.map((item) =>
            item.id === selectedEquipment.id
              ? { ...item, pendingSync: true, pendingDelete: true }
              : item
          );
          await AsyncStorage.setItem('equipment', JSON.stringify(updatedFirebaseList));
          setFirebaseEquipment(updatedFirebaseList);
          setFilteredEquipment(updatedFirebaseList);
          setDisplayedEquipment(updatedFirebaseList.slice(0, INITIAL_ITEMS_TO_LOAD + ((page - 1) * ITEMS_PER_PAGE)));
          showToast('info', 'Offline', errorMessage);
          return;
        }
        showToast('error', 'Error', errorMessage);
      } finally {
        setSyncModalVisible(false);
        setShowDeleteConfirm(false);
        rotateAnim.setValue(0);
        setActionModalVisible(false);
      }
    }
  };

  const getStatusColor = (status) => {
    switch (status) {
      case 'Available':
        return '#2ecc71';
      case 'Maintenance Required':
        return '#f1c40f';
      case 'In Maintenance':
        return '#e67e22';
      case 'Breakdown':
        return '#e74c3c';
      case 'Deploy':
        return '#3498db';
      default:
        return '#666';
    }
  };

  const getDynamicFieldLabels = (mainCategory) => {
    switch (mainCategory) {
      case 'Light Vehicles TDC':
      case 'Heavy Vehicles TDC':
        return { field1: 'Vehicle Number', field2: 'Registration', field3: 'VIN Number' };
      case 'TDC Rigs':
        return { field1: 'Rig Name', field2: 'Truck Type', field3: 'Chassis Number' };
      case 'Machinery':
        return { field1: 'Equipment Number', field2: 'Equipment Name', field3: 'VIN Number' };
      case 'Mud Mixer Trailers':
        return { field1: 'Number', field2: 'Serial Number', field3: 'Type' };
      case 'Trailers':
        return { field1: 'Power', field2: 'Type', field3: 'Equipment' };
      case 'Appliances':
      case 'Generators':
        return { field1: 'Type', field2: 'Type', field3: 'Serial' };
      case 'Workshop Machinery':
      case 'Pumps':
      case 'Tents':
        return { field1: 'Type', field2: 'Model', field3: 'Serial' };
      case 'Containers':
        return { field1: 'Purpose', field2: 'Container Type', field3: 'Container' };
      case 'Lightning Shelters':
        return { field1: 'Shelter', field2: 'Shelter', field3: 'Shelter' };
      default:
        return { field1: 'Field 1', field2: 'Field 2', field3: 'Field 3' };
    }
  };

  const renderAdditionalFields = (item) => {
    const fields = [];
    const { field1, field2, field3 } = getDynamicFieldLabels(item.mainCategory);
    if (item.dynamicField1 && item.dynamicField1 !== 'N/A') {
      fields.push({ label: field1, value: item.dynamicField1 });
    }
    if (item?.dynamicField2 && item?.dynamicField2 !== 'N/A') {
      fields.push({ label: field2, value: item.dynamicField2 });
    }
    if (item?.dynamicField3 && item?.dynamicField3 !== '') {
      fields.push({ label: field3, value: item.dynamicField3 });
    }

    return fields.map((field, index) => (
      <View key={index} style={styles.equipmentDetailItem}>
        <Text style={styles.detailLabel}>{field.label}</Text>
        <Text style={styles.detailValue}>{field.value}</Text>
      </View>
    ));
  };

  const getCountryOptions = () => {
    const countries = ['All Countries', ...new Set(firebaseEquipment.map(item => item.country).filter(Boolean))];
    const countryCounts = countries.reduce((acc, country) => {
      if (country === 'All Countries') {
        acc[country] = firebaseEquipment.length;
      } else {
        acc[country] = firebaseEquipment.filter(item => item.country === country).length;
      }
      return acc;
    }, {});
    return { countries, countryCounts };
  };

  const getStatusOptions = () => {
    const statuses = [...new Set(firebaseEquipment.map(item => item.status).filter(Boolean))];
    return statuses.map(status => ({ label: status, value: status }));
  };

  const getActionOptions = () => {
    const baseOptions = [];
    if (userInfo.equipmentPermissions.includes('view') || userInfo.equipmentPermissions.includes('onlyMineView')) {
      baseOptions.push(
        { label: 'Start Inspection', action: () => navigation.navigate('InsreportHeader', { equipment: selectedEquipment }) },
        { label: 'Equipment Overview', action: () => navigation.navigate('EquipmentOverview', { equipment: selectedEquipment }) }
      );
    }
    if (userInfo.equipmentPermissions.includes('update')) {
      baseOptions.push({
        label: 'Update',
        action: () => navigation.navigate('Add_equipment_update', { equipment: selectedEquipment }),
      });
    }
    if (userInfo.equipmentPermissions.includes('delete')) {
      baseOptions.push({
        label: 'Delete',
        action: () => setShowDeleteConfirm(true),
      });
    }
    return baseOptions;
  };

  const renderCountryDropdown = () => {
    const { countries, countryCounts } = getCountryOptions();
    return (
      <Modal visible={countryDropdownVisible} transparent animationType="none">
        <TouchableOpacity
          style={styles.dropdownOverlay}
          onPress={() => setCountryDropdownVisible(false)}
          activeOpacity={1}
        >
          <View
            style={[
              styles.dropdownContainer,
              {
                top: countryDropdownPosition.y,
                left: countryDropdownPosition.x,
                width: countryDropdownPosition.width,
                minWidth: 80,
                maxWidth: 300,
              },
            ]}
          >
            <ScrollView
              nestedScrollEnabled={true}
              showsVerticalScrollIndicator={true}
              style={{ maxHeight: 200 }}
            >
              {countries.map((country) => (
                <TouchableOpacity
                  key={country}
                  style={[
                    styles.dropdownItem,
                    selectedCountry === country && styles.dropdownItemActive,
                  ]}
                  onPress={() => handleCountrySelect(country)}
                >
                  <MaterialIcons
                    name="public"
                    size={width * 0.05}
                    color={selectedCountry === country ? '#3481BC' : '#333'}
                  />
                  <Text
                    style={[
                      styles.dropdownText,
                      selectedCountry === country && styles.dropdownTextActive,
                    ]}
                  >
                    {country} ({countryCounts[country] || 0})
                  </Text>
                </TouchableOpacity>
              ))}
            </ScrollView>
          </View>
        </TouchableOpacity>
      </Modal>
    );
  };

  const renderFilterBar = () => {
    const { countryCounts } = getCountryOptions();
    return (
      <View style={{ backgroundColor: '#f5f5f5' }}>
        <ScrollView
          horizontal
          showsHorizontalScrollIndicator={false}
          contentContainerStyle={styles.filterBar}
        >
          <TouchableOpacity
            ref={countryButtonRef}
            style={styles.filterButton}
            onPress={handleCountryFilterPress}
          >
            <MaterialIcons name="public" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>
              Country: {selectedCountry} ({countryCounts[selectedCountry] || 0})
            </Text>
            <MaterialIcons name="arrow-drop-down" size={width * 0.05} color="#333" />
          </TouchableOpacity>
          <TouchableOpacity
            ref={permissionsButtonRef}
            style={styles.filterButton}
            onPress={() => setUserInfoModalVisible(true)}
          >
            <MaterialIcons name="lock" size={width * 0.05} color="#333" style={styles.filterIcon} />
            <Text style={styles.filterButtonText}>Permissions</Text>
          </TouchableOpacity>
        </ScrollView>
      </View>
    );
  };

  // When selecting an equipment from the list, always ensure equipmentNumber is set and meterReading is in Miles
  const handleSelectEquipment = async (item) => {
    console.log('Original equipment item:', item);
    
    // Clean the equipment data to ensure all properties are strings or numbers
    const cleanEquipmentData = (obj) => {
      const cleaned = {};
      Object.keys(obj).forEach(key => {
        const value = obj[key];
        if (typeof value === 'string' || typeof value === 'number' || typeof value === 'boolean') {
          cleaned[key] = value;
        } else if (value === null || value === undefined) {
          cleaned[key] = '';
        } else if (typeof value === 'object') {
          // Convert objects to JSON strings
          try {
            cleaned[key] = JSON.stringify(value);
          } catch (error) {
            cleaned[key] = '[Object]';
          }
        } else {
          cleaned[key] = String(value);
        }
      });
      return cleaned;
    };

    let equipmentNumber = item.equipmentNumber;
    if (!equipmentNumber && item.entryNumber) {
      equipmentNumber = item.entryNumber;
    }

    // Always convert meterReading to Miles if needed
    let meterReading = item.meterReading;
    let meterUnit = 'Miles';
    if (item.meterUnit && item.meterUnit.toLowerCase() === 'km') {
      meterReading = (parseFloat(item.meterReading) * 0.621371).toFixed(2);
    }

    const correctedEquipment = cleanEquipmentData({
      ...item,
      equipmentNumber,
      meterReading,
      meterUnit,
    });

    console.log('Cleaned equipment data:', correctedEquipment);
    setSelectedEquipment(correctedEquipment);

    // Save to AsyncStorage for persistence
    try {
      const savedHse = await AsyncStorage.getItem('hseStep1');
      let hseData = savedHse ? JSON.parse(savedHse) : {};
      hseData.selectedEquipment = correctedEquipment;
      await AsyncStorage.setItem('hseStep1', JSON.stringify(hseData));
    } catch (error) {
      console.error('Error saving selected equipment:', error);
    }

    navigation.setParams({ selectedEquipment: correctedEquipment });
    navigation.goBack();
  };

  return (
    <View style={styles.container}>
      <LinearGradient
        colors={['#3481BC', '#025383']}
        style={[styles.header, { height: TOTAL_HEADER_HEIGHT }]}
      >
        <StatusBar
          translucent
          backgroundColor="transparent"
          barStyle="light-content"
        />
        <View style={styles.headerContent}>
          <View style={styles.headerLeft}>
            <TouchableOpacity onPress={() => navigation.goBack()}>
              <MaterialIcons name="arrow-back" size={width * 0.06} color="#fff" />
            </TouchableOpacity>
            <Text style={styles.headerTitle}>Equipments</Text>
          </View>
          <TouchableOpacity onPress={handleRefresh}>
            <Image
              source={require('../../../Images/adminhome/equp/cloud_1.png')}
              style={styles.icons12}
            />
          </TouchableOpacity>
        </View>

        <View style={styles.searchContainer}>
          <View style={styles.searchWrapper}>
            <TextInput
              style={styles.searchInput}
              placeholder={`Search${selectedFilter ? ' by ' + selectedFilter : '...'}`}
              placeholderTextColor="#888"
              value={searchQuery}
              onChangeText={handleSearch}
            />
            <MaterialIcons
              name="search"
              size={width * 0.05}
              color="#888"
              style={styles.bladeIcon}
            />
          </View>
          <TouchableOpacity
            style={styles.filterButton}
            onPress={() => setFilterModalVisible(true)}
          >
            <MaterialIcons name="filter-list" size={width * 0.06} color="#fff" />
          </TouchableOpacity>
        </View>
      </LinearGradient>

      {renderFilterBar()}
      {renderCountryDropdown()}

      {isLoadingInitial ? (
        <View style={styles.loadingContainer}>
          <ActivityIndicator size="large" color="#3481BC" />
          <Text style={styles.loadingText}>Loading Equipment...</Text>
        </View>
      ) : (
        <ScrollView
          style={styles.mainContent}
          onScroll={handleScroll}
          scrollEventThrottle={16}
        >
          {selectedEquipment && (
            <View style={[styles.equipmentRow, { backgroundColor: '#e6f0ff', borderColor: '#015185', borderWidth: 2, marginBottom: 10 }]}> 
              <View style={styles.equipmentInfo}>
                <View style={styles.equipmentDetailItemheader}>
                  <Text style={styles.detailLabelheader}>{selectedEquipment.equipmentName}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Category</Text>
                  <Text style={styles.detailValue}>{selectedEquipment.mainCategory}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Model</Text>
                  <Text style={styles.detailValue}>{selectedEquipment.model || 'N/A'}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Meter Reading</Text>
                  <Text style={styles.detailValue}>{selectedEquipment.meterReading} {selectedEquipment.meterUnit}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Equipment Number</Text>
                  <Text style={styles.detailValue}>{selectedEquipment.equipmentNumber || selectedEquipment.entryNumber || 'N/A'}</Text>
                </View>
                <View style={styles.equipmentDetailItem}>
                  <Text style={styles.detailLabel}>Created Date</Text>
                  <Text style={styles.detailValue}>{new Date(selectedEquipment.createdAt).toLocaleDateString()}</Text>
                </View>
                {renderAdditionalFields(selectedEquipment)}
                <View style={styles.equipmentDetailItemNoBorder}>
                  <Text style={[styles.detailLabelstatuss, { color: getStatusColor(selectedEquipment.status) }]}>{selectedEquipment.status}</Text>
                </View>
              </View>
              <TouchableOpacity
                style={{ backgroundColor: '#28a745', borderRadius: 5, padding: 10, alignSelf: 'center', marginLeft: 10 }}
                disabled
              >
                <Text style={{ color: '#fff', fontWeight: 'bold' }}>Selected</Text>
              </TouchableOpacity>
            </View>
          )}
          {displayedEquipment.filter(item => !selectedEquipment || item.id !== selectedEquipment.id).length > 0 ? (
            displayedEquipment.filter(item => !selectedEquipment || item.id !== selectedEquipment.id).map((item) => (
              <View key={item.id} style={styles.equipmentRow}>
                <View style={styles.equipmentInfo}>
                  <View style={styles.equipmentDetailItemheader}>
                    <Text style={styles.detailLabelheader}>{item.equipmentName}</Text>
                  </View>
                  <View style={styles.equipmentDetailItem}>
                    <Text style={styles.detailLabel}>Category</Text>
                    <Text style={styles.detailValue}>{item.mainCategory}</Text>
                  </View>
                  <View style={styles.equipmentDetailItem}>
                    <Text style={styles.detailLabel}>Model</Text>
                    <Text style={styles.detailValue}>{item.model || 'N/A'}</Text>
                  </View>
                  <View style={styles.equipmentDetailItem}>
                    <Text style={styles.detailLabel}>Meter Reading</Text>
                    <Text style={styles.detailValue}>{item.meterReading} {item.meterUnit}</Text>
                  </View>
                  <View style={styles.equipmentDetailItem}>
                    <Text style={styles.detailLabel}>Equipment Number</Text>
                    <Text style={styles.detailValue}>{item.equipmentNumber || item.entryNumber || 'N/A'}</Text>
                  </View>
                  <View style={styles.equipmentDetailItem}>
                    <Text style={styles.detailLabel}>Created Date</Text>
                    <Text style={styles.detailValue}>{new Date(item.createdAt).toLocaleDateString()}</Text>
                  </View>
                  {renderAdditionalFields(item)}
                  <View style={styles.equipmentDetailItemNoBorder}>
                    <Text style={[styles.detailLabelstatuss, { color: getStatusColor(item.status) }]}>{item.status}</Text>
                  </View>
                </View>
                <TouchableOpacity
                  style={{ backgroundColor: '#015185', borderRadius: 5, padding: 10, alignSelf: 'center', marginLeft: 10 }}
                  onPress={() => handleSelectEquipment(item)}
                >
                  <Text style={{ color: '#fff', fontWeight: 'bold' }}>Select</Text>
                </TouchableOpacity>
              </View>
            ))
          ) : (
            <Text style={styles.placeholderText}>
              {searchQuery ? 'No matching equipment found' : 'No equipment available'}
            </Text>
          )}
          {isLoadingMore && (
            <ActivityIndicator size="large" color="#3481BC" style={{ marginVertical: 20 }} />
          )}
        </ScrollView>
      )}

      <View style={styles.buttonContainer}>
        <LinearGradient
          colors={['#3481BC', '#025383']}
          style={styles.mapButtonGradient}
        >
          <TouchableOpacity style={styles.mapButton}>
            <Image
              source={require('../../../Images/adminhome/equp/mapview_1.png')}
              style={styles.mapIcon}
            />
            <Text style={styles.addButtonTextmap}>Map View</Text>
          </TouchableOpacity>
        </LinearGradient>
        {userInfo.equipmentPermissions.includes('create') && (
          <LinearGradient
            colors={['#3481BC', '#025383']}
            style={styles.addButtonGradient}
          >
            <TouchableOpacity
              style={styles.addButton}
              onPress={() => navigation.navigate('Add_equipment')}
            >
              <Ionicons name="add" size={width * 0.06} color="#fff" />
              <Text style={styles.addButtonText}>Add Equipment</Text>
            </TouchableOpacity>
          </LinearGradient>
        )}
      </View>

      <Modal
        visible={actionModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setActionModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.actionModalContent}>
            <Text style={styles.actionModalTitle}>Actions</Text>
            {getActionOptions().map((option, index) => (
              <TouchableOpacity
                key={index}
                style={styles.actionItem}
                onPress={() => {
                  setActionModalVisible(false);
                  option.action();
                }}
              >
                <Text>{option.label}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setActionModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={statusModalVisible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setStatusModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.statusModalContent}>
            <LinearGradient
              colors={['#3481BC', '#025383']}
              style={styles.statusModalHeader}
            >
              <Text style={styles.statusModalTitle}>Change Status</Text>
            </LinearGradient>
            <View style={styles.statusDropdownWrapper}>
              <Dropdown
                style={styles.dropdown}
                containerStyle={styles.dropdownContainer}
                data={getStatusOptions()}
                labelField="label"
                valueField="value"
                placeholder="Select Status"
                value={newStatus}
                onChange={(item) => setNewStatus(item.value)}
                placeholderStyle={styles.dropdownPlaceholder}
                selectedTextStyle={styles.dropdownSelectedText}
                itemTextStyle={styles.dropdownItemText}
                maxHeight={200}
              />
            </View>
            <TouchableOpacity
              style={styles.applyButton}
              onPress={handleStatusChange}
            >
              <Text style={styles.applyButtonText}>Apply</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={filterModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setFilterModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.filterModalContent}>
            <Text style={styles.filterModalTitle}>Filter By</Text>
            {[
              'Created Date',
              'Equipment Number',
              'Equipment Name',
              'Equipment Categories',
              'Meter Reading',
              'Status',
            ].map((filter) => (
              <TouchableOpacity
                key={filter}
                style={styles.filterItem}
                onPress={() => handleFilterSelect(filter)}
              >
                <Text style={styles.filterItemText}>{filter}</Text>
              </TouchableOpacity>
            ))}
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setFilterModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={userInfoModalVisible}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setUserInfoModalVisible(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.userInfoModalContent}>
            <Text style={styles.filterModalTitle}>User Permissions</Text>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Countries:</Text>
              <Text style={styles.userInfoValue}>{userInfo.countries.join(', ') || 'None'}</Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Projects:</Text>
              <Text style={styles.userInfoValue}>{userInfo.projects.join(', ') || 'None'}</Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Equipment Permissions:</Text>
              <Text style={styles.userInfoValue}>{userInfo.equipmentPermissions.join(', ') || 'None'}</Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Employee Number:</Text>
              <Text style={styles.userInfoValue}>{userInfo.employeeNumber || 'N/A'}</Text>
            </View>
            <View style={styles.userInfoItem}>
              <Text style={styles.userInfoLabel}>Name:</Text>
              <Text style={styles.userInfoValue}>{userInfo.name || 'N/A'}</Text>
            </View>
            <TouchableOpacity
              style={styles.modalCloseButton}
              onPress={() => setUserInfoModalVisible(false)}
            >
              <Text style={styles.modalCloseText}>Close</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      <Modal
        visible={syncModalVisible}
        transparent
        animationType="fade"
      >
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Processing Equipment Data</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Modal
        visible={showDeleteConfirm}
        transparent={true}
        animationType="fade"
        onRequestClose={() => setShowDeleteConfirm(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.deleteConfirmModalContent}>
            <Text style={styles.deleteConfirmTitle}>Confirm Delete</Text>
            <Text style={styles.deleteConfirmText}>
              Are you sure you want to delete this equipment?
            </Text>
            <View style={styles.deleteConfirmButtonContainer}>
              <TouchableOpacity
                style={[styles.deleteConfirmButton, { backgroundColor: '#666' }]}
                onPress={() => setShowDeleteConfirm(false)}
              >
                <Text style={styles.deleteConfirmButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.deleteConfirmButton, { backgroundColor: '#3481BC' }]}
                onPress={handleDelete}
              >
                <Text style={styles.deleteConfirmButtonText}>OK</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  header: {
    width: '100%',
    justifyContent: 'center',
  },
  headerContent: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    paddingHorizontal: width * 0.04,
    paddingTop: StatusBar.currentHeight || width * 0.1,
  },
  headerLeft: {
    flexDirection: 'row',
    alignItems: 'center',
  },
  headerTitle: {
    color: '#fff',
    fontSize: width * 0.045,
    fontWeight: '500',
    marginLeft: width * 0.025,
  },
  searchContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingHorizontal: width * 0.04,
    paddingVertical: width * 0.025,
  },
  searchWrapper: {
    flex: 1,
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#fff',
    borderRadius: width * 0.09,
    paddingHorizontal: width * 0.025,
  },
  searchInput: {
    flex: 1,
    height: width * 0.1,
    fontSize: width * 0.04,
    color: '#333',
  },
  bladeIcon: {
    marginLeft: width * 0.012,
  },
  filterButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.015,
    paddingHorizontal: width * 0.03,
    borderRadius: width * 0.02,
    marginLeft: width * 0.025,
    borderWidth: 1,
    borderColor: '#ddd',
  },
  filterBar: {
    flexDirection: 'row',
    paddingVertical: width * 0.02,
    paddingHorizontal: width * 0.04,
  },
  filterButtonText: {
    fontSize: width * 0.035,
    color: '#333',
    marginLeft: width * 0.015,
    marginRight: width * 0.01,
  },
  filterIcon: {
    marginRight: width * 0.01,
  },
  dropdownOverlay: {
    flex: 1,
    backgroundColor: 'transparent',
  },
  dropdownContainer: {
    position: 'absolute',
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 5,
  },
  dropdownItem: {
    flexDirection: 'row',
    alignItems: 'center',
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  dropdownItemActive: {
    backgroundColor: '#e6f0ff',
  },
  dropdownText: {
    marginLeft: width * 0.025,
    fontSize: width * 0.04,
    color: '#333',
  },
  dropdownTextActive: {
    color: '#3481BC',
    fontWeight: 'bold',
  },
  mainContent: {
    flex: 1,
  },
  loadingContainer: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
  },
  loadingText: {
    marginTop: 10,
    fontSize: width * 0.04,
    color: '#333',
  },
  placeholderText: {
    fontSize: width * 0.045,
    color: '#888',
    textAlign: 'center',
    marginTop: width * 0.05,
  },
  equipmentRow: {
    marginHorizontal: width * 0.04,
    marginVertical: width * 0.01,
    backgroundColor: '#F2F2F2',
    borderRadius: 5,
    borderWidth: 1,
    borderColor: '#eee',
  },
  equipmentInfo: {
    flex: 1,
  },
  equipmentDetailItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
    paddingHorizontal: width * 0.05,
    paddingRight: width * 0.1,
  },
  equipmentDetailItemNoBorder: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    paddingLeft: width * 0.05,
  },
  equipmentDetailItemheader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: 5,
    backgroundColor: '#E9E9E9',
    paddingHorizontal: 5,
  },
  detailLabel: {
    fontSize: width * 0.035,
    color: '#666',
    fontWeight: '400',
    width: '40%',
    marginVertical: 3,
  },
  detailLabelstatuss: {
    fontSize: width * 0.035,
    fontWeight: '400',
    width: '60%',
    marginVertical: 3,
  },
  detailLabelheader: {
    fontSize: width * 0.045,
    color: '#015185',
    fontWeight: '600',
    width: '100%',
  },
  detailValue: {
    fontSize: width * 0.04,
    color: '#015185',
    width: '60%',
    textAlign: 'right',
    fontWeight: '500',
  },
  menuButton: {
    position: 'absolute',
    right: 10,
    bottom: 1,
    padding: 5,
  },
  buttonContainer: {
    position: 'absolute',
    bottom: '10%',
    width: '100%',
    alignItems: 'center',
  },
  mapButtonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
    marginBottom: 10,
  },
  mapButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  mapIcon: {
    width: width * 0.06,
    height: width * 0.06,
    marginRight: width * 0.025,
  },
  addButtonGradient: {
    borderRadius: width * 0.012,
    overflow: 'hidden',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: width * 0.025,
    paddingHorizontal: width * 0.05,
  },
  addButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
    marginLeft: width * 0.012,
  },
  addButtonTextmap: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  modalOverlay: {
    flex: 1,
    justifyContent: 'center',
    alignItems: 'center',
    backgroundColor: 'rgba(0,0,0,0.5)',
  },
  actionModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
  },
  actionModalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    marginBottom: width * 0.04,
    color: '#333',
  },
  actionItem: {
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  modalCloseButton: {
    marginTop: width * 0.04,
    alignItems: 'center',
  },
  modalCloseText: {
    color: '#3481BC',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  statusModalContent: {
    width: '80%',
    height: 220,
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    overflow: 'hidden',
    alignSelf: 'center',
  },
  statusModalHeader: {
    padding: 10,
  },
  statusModalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    color: '#fff',
    height: 40,
    textAlign: 'center',
    lineHeight: 40,
  },
  statusDropdownWrapper: {
    paddingHorizontal: width * 0.04,
    paddingVertical: 10,
    zIndex: 1000,
    alignItems: 'center',
  },
  dropdown: {
    backgroundColor: '#f0f0f0',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#ddd',
    height: 40,
    paddingHorizontal: 10,
    width: '100%',
  },
  dropdownContainer: {
    width: '65%',
    backgroundColor: '#fff',
    borderRadius: width * 0.012,
    borderWidth: 1,
    borderColor: '#ddd',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 2,
  },
  dropdownPlaceholder: {
    fontSize: width * 0.035,
    color: '#888',
  },
  dropdownSelectedText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  dropdownItemText: {
    fontSize: width * 0.035,
    color: '#333',
  },
  applyButton: {
    backgroundColor: '#3481BC',
    padding: width * 0.025,
    borderRadius: width * 0.012,
    alignItems: 'center',
    marginHorizontal: width * 0.04,
    marginVertical: 10,
  },
  applyButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
  filterModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
  },
  filterModalTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    marginBottom: width * 0.04,
    color: '#333',
  },
  filterItem: {
    padding: width * 0.025,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  filterItemText: {
    fontSize: width * 0.04,
    color: '#333',
  },
  userInfoModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
  },
  userInfoItem: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    paddingVertical: width * 0.02,
    borderBottomWidth: 1,
    borderBottomColor: '#eee',
  },
  userInfoLabel: {
    fontSize: width * 0.04,
    color: '#666',
    fontWeight: '500',
    width: '40%',
  },
  userInfoValue: {
    fontSize: width * 0.04,
    color: '#333',
    width: '60%',
    textAlign: 'right',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: width * 0.045,
    marginBottom: width * 0.025,
    color: '#333',
  },
  syncIcon: {
    width: width * 0.055,
    height: width * 0.055,
    marginRight: width * 0.025,
  },
  syncModalTextadd: {
    fontSize: width * 0.035,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  icons12: {
    width: width * 0.06,
    height: width * 0.06,
  },
  deleteConfirmModalContent: {
    width: '80%',
    backgroundColor: '#fff',
    borderRadius: width * 0.025,
    padding: width * 0.05,
    alignItems: 'center',
  },
  deleteConfirmTitle: {
    fontSize: width * 0.045,
    fontWeight: 'bold',
    marginBottom: width * 0.04,
    color: '#333',
  },
  deleteConfirmText: {
    fontSize: width * 0.04,
    color: '#666',
    marginBottom: width * 0.05,
    textAlign: 'center',
  },
  deleteConfirmButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    width: '100%',
  },
  deleteConfirmButton: {
    flex: 1,
    padding: width * 0.025,
    borderRadius: width * 0.012,
    marginHorizontal: width * 0.012,
    alignItems: 'center',
  },
  deleteConfirmButtonText: {
    color: '#fff',
    fontSize: width * 0.04,
    fontWeight: 'bold',
  },
});

export default HSEselecteq;