import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TextInput,
  ScrollView,
  TouchableOpacity,
  Dimensions,
  KeyboardAvoidingView,
  Platform,
  Modal,
  Image,
  Alert,
  Animated,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';
import Toast from 'react-native-toast-message';
import DateTimePicker from '@react-native-community/datetimepicker';
import { launchCamera, launchImageLibrary } from 'react-native-image-picker';
import { useNavigation } from '@react-navigation/native';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';

const { width, height } = Dimensions.get('window');

const HSE_158Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const rotateAnim = useRef(new Animated.Value(0)).current;
  const rotateInterpolate = rotateAnim.interpolate({
    inputRange: [0, 1],
    outputRange: ['0deg', '360deg'],
  });
  const [entries, setEntries] = useState([
    {
      id: 1,
      timeIn: '',
      area: '',
      step1Status: '',
      step2Status: '',
      comment: '',
      picSignature: '',
      picSignatureImage: null,
      picTimeOut: '',
    }
  ]);

  const [version, setVersion] = useState('2');
  const [approved, setApproved] = useState('HSM');
  const [showTimePicker, setShowTimePicker] = useState(false);
  const [currentTimeField, setCurrentTimeField] = useState('');
  const [currentEntryId, setCurrentEntryId] = useState(null);
  const [selectedTime, setSelectedTime] = useState(new Date());
  const [showImageOptions, setShowImageOptions] = useState(false);
  const [currentImageField, setCurrentImageField] = useState('');
  const [currentImageEntryId, setCurrentImageEntryId] = useState(null);
  const [isLoading, setIsLoading] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [pdfDownloadUrl, setPdfDownloadUrl] = useState(null);

  useEffect(() => {
    loadSavedData();
  }, []);

  // Animation effect for loading modal
  useEffect(() => {
    if (isLoading) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [isLoading]);

  // Monitor network connectivity
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => unsubscribe();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse158Step3');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setEntries(parsedData.entries || entries);
        setVersion(parsedData.version || '2');
        setApproved(parsedData.approved || 'HSM');
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async (data) => {
    try {
      await AsyncStorage.setItem('hse158Step3', JSON.stringify(data));
    } catch (error) {
      console.error('Error saving data:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to save data',
        position: 'top',
      });
    }
  };

  const handleInputChange = (id, field, value) => {
    const updatedEntries = entries.map(entry => 
      entry.id === id ? { ...entry, [field]: value } : entry
    );
    setEntries(updatedEntries);
    saveData({ entries: updatedEntries, version, approved });
  };

  const addNewEntry = () => {
    const newEntry = {
      id: Date.now(),
      timeIn: '',
      area: '',
      step1Status: '',
      step2Status: '',
      comment: '',
      picSignature: '',
      picSignatureImage: null,
      picTimeOut: '',
    };
    const updatedEntries = [...entries, newEntry];
    setEntries(updatedEntries);
    saveData({ entries: updatedEntries, version, approved });
  };

  const removeEntry = (id) => {
    if (entries.length > 1) {
      const updatedEntries = entries.filter(entry => entry.id !== id);
      setEntries(updatedEntries);
      saveData({ entries: updatedEntries, version, approved });
    }
  };

  const showTimePickerModal = (entryId, field) => {
    setCurrentEntryId(entryId);
    setCurrentTimeField(field);
    setShowTimePicker(true);
  };

  const handleTimeChange = (event, selectedDate) => {
    setShowTimePicker(false);
    if (selectedDate && currentEntryId && currentTimeField) {
      const timeString = selectedDate.toLocaleTimeString('en-US', {
        hour: '2-digit',
        minute: '2-digit',
        hour12: false
      });
      handleInputChange(currentEntryId, currentTimeField, timeString);
    }
  };

  const formatTime = (timeString) => {
    if (!timeString) return 'Select Time';
    return timeString;
  };

  const showImagePickerOptions = (entryId, field) => {
    setCurrentImageEntryId(entryId);
    setCurrentImageField(field);
    setShowImageOptions(true);
  };

  const handleImageSelection = async (type) => {
    setShowImageOptions(false);
    
    try {
      let result;
      if (type === 'camera') {
        result = await launchCamera({
          mediaType: 'photo',
          quality: 0.8,
          includeBase64: true,
        });
      } else {
        result = await launchImageLibrary({
          mediaType: 'photo',
          quality: 0.8,
          includeBase64: true,
        });
      }

      if (result.assets && result.assets[0]) {
        const image = result.assets[0];
        const imageData = {
          uri: image.uri,
          base64: image.base64,
          fileName: image.fileName || `signature_${Date.now()}.jpg`,
          type: image.type || 'image/jpeg',
        };

        if (currentImageField === 'picSignatureImage') {
          handleInputChange(currentImageEntryId, 'picSignatureImage', imageData);
          handleInputChange(currentImageEntryId, 'picSignature', 'Image uploaded');
        }
      }
    } catch (error) {
      console.error('Error selecting image:', error);
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'Failed to select image',
        position: 'top',
      });
    }
  };

  const removeImage = (entryId, field) => {
    if (field === 'picSignatureImage') {
      handleInputChange(entryId, 'picSignatureImage', null);
      handleInputChange(entryId, 'picSignature', '');
    }
  };

  const generatePDFContent = async () => {
    try {
      // Load data from AsyncStorage
      const savedHse1 = await AsyncStorage.getItem('hseStep1');
      const savedHse2 = await AsyncStorage.getItem('hse158Step2');
      const savedHse3 = await AsyncStorage.getItem('hse158Step3');
      
      const step1Data = savedHse1 ? JSON.parse(savedHse1) : {};
      const step2Data = savedHse2 ? JSON.parse(savedHse2) : {};
      const step3Data = savedHse3 ? JSON.parse(savedHse3) : {};
      
      const checklistTitle = step1Data.selectedChecklist?.document_name || 'HSE-STD-8.1-1 Fatigue Ops-Normal Checklist';
      const documentId = step1Data.selectedChecklist?.document_id || 'HSE-158';
      const versionDate = step1Data.selectedChecklist?.version_date || '24/08/2025';
      const version = step1Data.selectedChecklist?.version || '2';
      const approvedBy = step1Data.selectedChecklist?.approved_by || 'HSM';
      
      // Get crew details from step2 - access through formData
      const crewNames = step2Data.crewNames || step2Data.formData?.crewNames || '';
      const dateAndShift = step2Data.dateAndShift || step2Data.formData?.dateAndShift || '';
      const supervisorName = step2Data.supervisorName || step2Data.formData?.supervisorName || '';
      const shiftType = step2Data.shiftType || step2Data.formData?.shiftType || 'DAY';
      
      // Get step1 assessment data - access through formData
      const approachObservation = step2Data.approachObservation || step2Data.formData?.approachObservation || '';
      const physicalSymptoms = step2Data.physicalSymptoms || step2Data.formData?.physicalSymptoms || {};
      const mentalFunctioning = step2Data.mentalFunctioning || step2Data.formData?.mentalFunctioning || {};
      const emotionMotivation = step2Data.emotionMotivation || step2Data.formData?.emotionMotivation || {};
      const riskAssessment = step2Data.riskAssessment || step2Data.formData?.riskAssessment || {};
      const fatigueLevel = step2Data.fatigueLevel || step2Data.formData?.fatigueLevel || {};
      
      // Get step3 entries
      const entries = step3Data.entries || [];
      
      // Generate step1 table rows
      let step1Rows = '';
      
      // Approach row
      step1Rows += `
        <tr>
          <td class="category">APPROACH</td>
          <td class="checkboxes">${approachObservation || 'Was there anything observed on arrival at the site?'}</td>
        </tr>
      `;
      
      // Physical symptoms row
      const physicalSymptomsHtml = [
        `<div class="checkbox-group"><input type="checkbox" ${physicalSymptoms.eyesBloodshot ? 'checked' : ''}><label>Eyes bloodshot</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${physicalSymptoms.poorCoordination ? 'checked' : ''}><label>Poor co-ordination</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${physicalSymptoms.slowerMovements ? 'checked' : ''}><label>Slower movements</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${physicalSymptoms.other ? 'checked' : ''}><label>Other</label></div>`
      ].join('');
      
      step1Rows += `
        <tr>
          <td class="category">Physical symptoms</td>
          <td class="checkboxes">${physicalSymptomsHtml}</td>
        </tr>
      `;
      
      // Mental functioning row
      const mentalFunctioningHtml = [
        `<div class="checkbox-group"><input type="checkbox" ${mentalFunctioning.distractedFromTask ? 'checked' : ''}><label>Distracted from task</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${mentalFunctioning.shortTermMemoryLoss ? 'checked' : ''}><label>Short term memory loss (forgets instruction)</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${mentalFunctioning.zombieLikeBehaviour ? 'checked' : ''}><label>Zombie like behaviour</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${mentalFunctioning.poorConcentration ? 'checked' : ''}><label>Poor concentration</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${mentalFunctioning.noddingOffMomentarily ? 'checked' : ''}><label>Nodding off momentarily</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${mentalFunctioning.other ? 'checked' : ''}><label>Other</label></div>`
      ].join('');
      
      step1Rows += `
        <tr>
          <td class="category">Mental functioning</td>
          <td class="checkboxes">${mentalFunctioningHtml}</td>
        </tr>
      `;
      
      // Emotion/motivation row
      const emotionMotivationHtml = [
        `<div class="checkbox-group"><input type="checkbox" ${emotionMotivation.seemsDepressed ? 'checked' : ''}><label>Seems depressed like behaviour</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${emotionMotivation.easilyFrustrated ? 'checked' : ''}><label>Easily frustrated with tasks</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${emotionMotivation.irritable ? 'checked' : ''}><label>Irritable</label></div>`,
        `<div class="checkbox-group"><input type="checkbox" ${emotionMotivation.other ? 'checked' : ''}><label>Other</label></div>`
      ].join('');
      
      step1Rows += `
        <tr>
          <td class="category">Emotion / motivation</td>
          <td class="checkboxes">${emotionMotivationHtml}</td>
        </tr>
      `;
      
      // Generate step3 observation log rows
      let observationRows = '';
      entries.forEach((entry, index) => {
        const signatureImage = entry.picSignatureImage ? 
          `<img src="${entry.picSignatureImage.uri}" style="max-width: 80px; max-height: 40px; border: 1px solid #ccc; border-radius: 4px;" alt="Signature"/>` : 
          '';
        
        observationRows += `
          <tr>
            <td>${entry.timeIn || ''}</td>
            <td>${entry.area || ''}</td>
            <td>${entry.step1Status || ''}</td>
            <td>${entry.step2Status || ''}</td>
            <td>${entry.comment || ''}</td>
            <td style="text-align: center;">${signatureImage}</td>
            <td>${entry.picTimeOut || ''}</td>
          </tr>
        `;
      });
      
      // Add empty rows if less than 12 entries
      const emptyRows = Math.max(0, 12 - entries.length);
      for (let i = 0; i < emptyRows; i++) {
        observationRows += `
          <tr><td></td><td></td><td></td><td></td><td></td><td></td><td></td></tr>
        `;
      }
      
      // Generate fatigue level opinion
      const fatigueOpinion = [
        `<input type="checkbox" ${fatigueLevel.unsafeToWork ? 'checked' : ''}> Unsafe to work`,
        `<input type="checkbox" ${fatigueLevel.safeToWork ? 'checked' : ''}> Safe to work`,
        `<input type="checkbox" ${fatigueLevel.restrictedDuties ? 'checked' : ''}> Restricted duties`
      ].join(' ');
      
      // Get risk assessment data from step2
      const fatigueIncident = riskAssessment.fatigueIncidentOccurred || '';
      const sleepReason = riskAssessment.reasonsForPoorSleep || '';
      const employeeRisk = riskAssessment.employeeAtRisk || '';
      const tasksPerforming = riskAssessment.tasksPerforming || '';
      const functioningRisk = riskAssessment.riskAssociated || '';
      const highRiskTasks = riskAssessment.highRiskTasks || '';
      const employeeExplanation = riskAssessment.employeeExplanation || '';
      const taskDuration = riskAssessment.howLongOnTask || '';
      const hoursSinceSleep = riskAssessment.hoursSinceLastSlept || '';
      const breakDuration = riskAssessment.breakDuration || '';
      const lastBreak = riskAssessment.lastBreak || '';
      const lastHydration = riskAssessment.lastWaterFood || '';
      const fatiguePrevention = riskAssessment.howPreventFatigue || '';
      
      return `
        <!DOCTYPE html>
        <html lang="en">
        <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>Titan Drilling Checklists</title>
          <style>
            body {
              font-family: Arial, sans-serif;
              margin: 0;
              padding: 0;
              background-color: #f9f9f9;
            }
            .container {
              width: 100%;
              max-width: 800px;
              margin: 0 auto;
              padding: 20px;
              position: relative;
            }
            .logo {
              position: absolute;
              top: 20px;
              left: 20px;
            }
            .logo img {
              width: 50px;
            }
            .header {
              text-align: left;
              margin-bottom: 20px;
              margin-left: 70px;
            }
            .header h1 {
              margin: 0;
              font-size: 24px;
              color: #000;
            }
            .table-section {
              background-color: #fff;
              border: 1px solid #000;
              padding: 10px;
              margin-bottom: 20px;
            }
            table {
              width: 100%;
              border-collapse: collapse;
              font-size: 12px;
            }
            th, td {
              border: 1px solid #000;
              padding: 5px;
              text-align: left;
              vertical-align: top;
            }
            th {
              background-color: #f2f2f2;
            }
            .document-info {
              float: right;
              font-size: 10px;
              text-align: right;
              width: 300px;
            }
            .document-info table {
              width: 100%;
              border-collapse: collapse;
              font-size: 10px;
            }
            .document-info th, .document-info td {
              border: 1px solid #000;
              padding: 2px;
              text-align: left;
            }
            .checkbox-group {
              margin: 0;
              display: flex;
              align-items: center;
            }
            .checkbox-group input {
              margin-right: 5px;
            }
            .checkbox-group label {
              margin-left: 0;
            }
            .step1-header {
              font-weight: bold;
              padding: 5px;
              background-color: #f2f2f2;
              border-bottom: 1px solid #000;
            }
            .step1-table {
              width: 100%;
              border-collapse: collapse;
            }
            .step1-table td {
              border: 1px solid #000;
              padding: 5px;
            }
            .step1-table .category {
              width: 30%;
              background-color: #e6e6e6;
            }
            .step1-table .checkboxes {
              width: 70%;
            }
            .risk-data {
              font-weight: bold;
              color: #333;
            }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="table-section">
               <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 200px; height: auto;">
              <div class="document-info">
                <table>
                  <tr>
                    <th>Document Name:</th>
                    <td>${checklistTitle}</td>
                  </tr>
                  <tr>
                    <th>Document ID:</th>
                    <td>${documentId}</td>
                  </tr>
                  <tr>
                    <th>Version Date:</th>
                    <td>${versionDate}</td>
                  </tr>
                  <tr>
                    <th>Version:</th>
                    <td>${version}</td>
                  </tr>
                  <tr>
                    <th>Approved:</th>
                    <td>${approvedBy}</td>
                  </tr>
                </table>
              </div>
              <table>
                <tr>
                  <th>Crew Name(s):</th>
                  <td colspan="2">${crewNames}</td>
                  <th>Date &amp; Shift:</th>
                  <td>${dateAndShift}</td>
                  <th>${shiftType}</th>
                </tr>
                <tr>
                  <th>Supervisor Name:</th>
                  <td colspan="2">${supervisorName}</td>
                  <th>Supervisor Signature:</th>
                  <td colspan="2">
                    ${step2Data.supervisorSignature ? 
                      `<img src="${step2Data.supervisorSignature}" style="max-width: 100px; max-height: 50px; border: 1px solid #ccc; border-radius: 4px;" alt="Supervisor Signature"/>` : 
                      ''
                    }
                  </td>
                </tr>
                <tr>
                  <td colspan="6" class="step1-header">STEP 1: SIGNS AND SYMPTOMS (please tick)</td>
                </tr>
                <tr>
                  <td colspan="6">Has there been a change in the crew's behavior? If so, what have you observed and who?</td>
                </tr>
                <tr>
                  <td colspan="6">
                    <table class="step1-table">
                      ${step1Rows}
                    </table>
                  </td>
                </tr>
                <tr>
                  <td colspan="6">STEP 2: RISK</td>
                </tr>
                <tr>
                  <td colspan="3">Has a fatigue related incident occurred? <span class="risk-data">${fatigueIncident}</span></td>
                  <td colspan="3">Is there a reason for not enough sleep or poor sleep (stress, relationship, childcare, substance use etc)? <span class="risk-data">${sleepReason}</span></td>
                </tr>
                <tr>
                  <td colspan="3">Has an employee self-report/ by another person/ been identified as at risk of fatigue? <span class="risk-data">${employeeRisk}</span></td>
                  <td colspan="3">What tasks have they been performing? <span class="risk-data">${tasksPerforming}</span></td>
                </tr>
                <tr>
                  <td colspan="3">Is there a risk associated with an employee&#39;s functioning/ behaviour? <span class="risk-data">${functioningRisk}</span></td>
                  <td colspan="3">Are those tasks &#39;high-risk&#39; for fatigue (e.g. repetitious or entail not constant activity)? <span class="risk-data">${highRiskTasks}</span></td>
                </tr>
                <tr>
                  <td colspan="3">What is the employee&#39;s explanation for the behaviour you have observed? <span class="risk-data">${employeeExplanation}</span></td>
                  <td colspan="3">If yes, how long have they been working on that task? <span class="risk-data">${taskDuration}</span></td>
                </tr>
                <tr>
                  <td colspan="3">How many hours since the employee last slept? <span class="risk-data">${hoursSinceSleep}</span></td>
                  <td colspan="3">How long was the break? <span class="risk-data">${breakDuration}</span></td>
                </tr>
                <tr>
                  <td colspan="3">When did they last have a break? <span class="risk-data">${lastBreak}</span></td>
                  <td colspan="3">when did they last drink water or eat something (dehydrated or hungry)? <span class="risk-data">${lastHydration}</span></td>
                </tr>
                <tr>
                  <td colspan="3">How do they usually prevent fatigue? <span class="risk-data">${fatiguePrevention}</span></td>
                  <td colspan="3">How do they usually prevent fatigue? <span class="risk-data">${fatiguePrevention}</span></td>
                </tr>
                <tr>
                  <td colspan="6">Opinion regarding level of fatigue: ${fatigueOpinion}</td>
                </tr>
              </table>
            </div>
            <div class="table-section">
              <table>
                <tr>
                  <th>TIME IN</th>
                  <th>Area:</th>
                  <th>Step 1 Status</th>
                  <th>Step 2 Status</th>
                  <th>COMMENT</th>
                  <th>PC Signature</th>
                  <th>PC TIME OUT</th>
                </tr>
                ${observationRows}
              </table>
            </div>
          </div>
        </body>
        </html>
      `;
    } catch (error) {
      console.error('Error generating PDF content:', error);
      throw error;
    }
  };

  const generateAndSavePDF = async () => {
    try {
      setIsLoading(true);
      
      // Check if required data exists
      const savedHse2 = await AsyncStorage.getItem('hse158Step2');
      const savedHse3 = await AsyncStorage.getItem('hse158Step3');
      
      if (!savedHse2 || !savedHse3) {
        throw new Error('Required inspection data is missing. Please complete steps 2 and 3 first.');
      }
      
      const htmlContent = await generatePDFContent();
      const options = {
        html: htmlContent,
        fileName: `HSE_158_Report_${Date.now()}`,
        directory: Platform.OS === 'android' ? 'Documents' : 'DocumentDirectory',
      };
      
      const result = await RNHTMLtoPDF.convert(options);
      if (!result || !result.filePath) {
        throw new Error('Failed to generate PDF file');
      }
      const filePath = result.filePath;
      console.log('PDF generated at:', filePath);

      if (isOnline) {
        // Upload to Firebase Storage
        const userEmail = await AsyncStorage.getItem('userEmail');
        if (!userEmail) {
          throw new Error('User email not found. Please log in again.');
        }
        
        const userSnapshot = await database()
          .ref('/Globalusers/admins')
          .orderByChild('email')
          .equalTo(userEmail)
          .once('value');
        const userData = userSnapshot.val();
        if (!userData) {
          throw new Error('User not found in admin database.');
        }
        const userId = Object.keys(userData)[0];
        
        const storageRef = storage().ref(`HSE_158_Reports/${userId}/${Date.now()}.pdf`);
        await storageRef.putFile(filePath);
        const downloadUrl = await storageRef.getDownloadURL();
        
        if (!downloadUrl) {
          throw new Error('Failed to get PDF download URL');
        }
        
        // Save to Firebase Database
        const reportData = {
          checklistType: 'HSE-158',
          entries: entries,
          version: version,
          approved: approved,
          pdfDownloadUrl: downloadUrl,
          createdAt: new Date().toISOString(),
          userId: userId,
        };
        
        const reportId = Date.now().toString();
        const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${reportId}`);
        await dbRef.set(reportData);
        
        console.log('Report saved to database with ID:', reportId);
        
        setPdfDownloadUrl(downloadUrl);
        
        // Clean up local file
        await RNFS.unlink(filePath).catch(err => console.warn('Failed to delete local PDF:', err));
        
        return downloadUrl;
      } else {
        // Save locally for offline sync
        const offlineData = {
          checklistType: 'HSE-158',
          entries: entries,
          version: version,
          approved: approved,
          pdfPath: filePath,
          createdAt: new Date().toISOString(),
        };
        
        const offlineReports = JSON.parse(await AsyncStorage.getItem('offlineHSE158Reports') || '[]');
        offlineReports.push(offlineData);
        await AsyncStorage.setItem('offlineHSE158Reports', JSON.stringify(offlineReports));
        
        return null;
      }
    } catch (error) {
      console.error('Error generating PDF:', error);
      throw error;
    } finally {
      setIsLoading(false);
    }
  };

  const handleNext = async () => {
    // Validate required fields
    const hasEmptyRequiredFields = entries.some(entry => 
      !entry.timeIn || !entry.area || !entry.step1Status || !entry.step2Status
    );
    
    if (hasEmptyRequiredFields) {
      Toast.show({
        type: 'error',
        text1: 'Required Fields Missing',
        text2: 'Please fill all required fields in each entry',
        position: 'top',
      });
      return;
    }
    
    Alert.alert(
      'Generate Report',
      'Are you sure you want to generate the HSE-158 inspection report?',
      [
        { text: 'Cancel', style: 'cancel' },
        {
          text: 'Generate',
          style: 'default',
          onPress: async () => {
            try {
              setIsLoading(true);
              
              // Generate PDF and save to database
              const pdfUrl = await generateAndSavePDF();
    
    Toast.show({
      type: 'success',
      text1: 'Success',
                text2: 'HSE-158 inspection report generated and saved successfully!',
                position: 'bottom',
              });
              
              // Show success modal with PDF view option
              if (pdfUrl) {
                setShowSuccessModal(true);
              } else {
                // Clear stored data after successful generation
                setTimeout(async () => {
                  try {
                    await AsyncStorage.removeItem('hse158Step2');
                    await AsyncStorage.removeItem('hse158Step3');
                    navigation.navigate('HomeScreen');
                  } catch (error) {
                    console.error('Error clearing data:', error);
                    navigation.navigate('HomeScreen');
                  }
                }, 2000);
              }
            } catch (error) {
              console.error('Error generating report:', error);
              Toast.show({
                type: 'error',
                text1: 'Error',
                text2: error.message || 'Failed to generate report.',
                position: 'bottom',
              });
            } finally {
              setIsLoading(false);
            }
          },
        },
      ]
    );
  };

  const handleViewPdf = () => {
    if (pdfDownloadUrl) {
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfDownloadUrl,
        onGoBack: () => {
          setShowSuccessModal(false);
        }
      });
    } else {
      Toast.show({
        type: 'error',
        text1: 'Error',
        text2: 'PDF not available for viewing.',
        position: 'bottom',
      });
    }
  };

  const handleBackToHome = () => {
    setShowSuccessModal(false);
    // Clear stored data
    AsyncStorage.removeItem('hse158Step2');
    AsyncStorage.removeItem('hse158Step3');
    navigation.navigate('HomeScreen');
  };

  const handleBack = () => {
    setActiveStep(2);
  };

  const renderEntryRow = (entry, index) => (
    <View key={entry.id} style={styles.entryRow}>
      <View style={styles.rowHeader}>
        <Text style={styles.rowNumber}>Row {index + 1}</Text>
        {entries.length > 1 && (
          <TouchableOpacity 
            style={styles.removeButton} 
            onPress={() => removeEntry(entry.id)}
          >
            <MaterialIcons name="delete" size={20} color="#dc3545" />
          </TouchableOpacity>
        )}
      </View>
      
      <View style={styles.inputRow}>
        <View style={styles.inputColumn}>
          <Text style={styles.columnLabel}>TIME IN *</Text>
          <TouchableOpacity
            style={styles.timeInput}
            onPress={() => showTimePickerModal(entry.id, 'timeIn')}
          >
            <Text style={[
              styles.timeInputText,
              !entry.timeIn && styles.timePlaceholderText
            ]}>
              {formatTime(entry.timeIn)}
            </Text>
            <MaterialIcons name="access-time" size={20} color="#015185" />
          </TouchableOpacity>
        </View>
        
        <View style={styles.inputColumn}>
          <Text style={styles.columnLabel}>Area: *</Text>
          <TextInput
            style={styles.input}
            value={entry.area}
            onChangeText={(value) => handleInputChange(entry.id, 'area', value)}
            placeholder="Enter area"
            placeholderTextColor="#999"
          />
        </View>
      </View>

      <View style={styles.inputRow}>
        <View style={styles.inputColumn}>
          <Text style={styles.columnLabel}>Step 1 Status *</Text>
          <TextInput
            style={styles.input}
            value={entry.step1Status}
            onChangeText={(value) => handleInputChange(entry.id, 'step1Status', value)}
            placeholder="Enter status"
            placeholderTextColor="#999"
          />
        </View>
        
        <View style={styles.inputColumn}>
          <Text style={styles.columnLabel}>Step 2 Status *</Text>
          <TextInput
            style={styles.input}
            value={entry.step2Status}
            onChangeText={(value) => handleInputChange(entry.id, 'step2Status', value)}
            placeholder="Enter status"
            placeholderTextColor="#999"
          />
        </View>
      </View>

      <View style={styles.commentSection}>
        <Text style={styles.columnLabel}>COMMENT</Text>
        <TextInput
          style={styles.commentInput}
          value={entry.comment}
          onChangeText={(value) => handleInputChange(entry.id, 'comment', value)}
          placeholder="Enter comment"
          placeholderTextColor="#999"
          multiline
          numberOfLines={3}
        />
      </View>

      <View style={styles.inputRow}>
        <View style={styles.inputColumn}>
          <Text style={styles.columnLabel}>PIC Signature</Text>
          <View style={styles.signatureContainer}>
            {entry.picSignatureImage ? (
              <View style={styles.imageContainer}>
                <Image 
                  source={{ uri: entry.picSignatureImage.uri }} 
                  style={styles.signatureImage}
                  resizeMode="contain"
                />
                <TouchableOpacity 
                  style={styles.removeImageButton}
                  onPress={() => removeImage(entry.id, 'picSignatureImage')}
                >
                  <MaterialIcons name="close" size={16} color="#fff" />
                </TouchableOpacity>
              </View>
            ) : (
              <TouchableOpacity
                style={styles.uploadButton}
                onPress={() => showImagePickerOptions(entry.id, 'picSignatureImage')}
              >
                <MaterialIcons name="add-a-photo" size={24} color="#015185" />
                <Text style={styles.uploadButtonText}>Upload Image</Text>
              </TouchableOpacity>
            )}
          </View>
        </View>
        
        <View style={styles.inputColumn}>
          <Text style={styles.columnLabel}>PIC TIME OUT</Text>
          <TouchableOpacity
            style={styles.timeInput}
            onPress={() => showTimePickerModal(entry.id, 'picTimeOut')}
          >
            <Text style={[
              styles.timeInputText,
              !entry.picTimeOut && styles.timePlaceholderText
            ]}>
              {formatTime(entry.picTimeOut)}
            </Text>
            <MaterialIcons name="access-time" size={20} color="#015185" />
          </TouchableOpacity>
        </View>
      </View>
    </View>
  );

  return (
    <KeyboardAvoidingView 
      style={styles.container} 
      behavior={Platform.OS === 'ios' ? 'padding' : 'height'}
      keyboardVerticalOffset={Platform.OS === 'ios' ? 0 : 20}
    >
      <ScrollView style={styles.scrollView} contentContainerStyle={styles.scrollContent}>
        {/* Header */}
       

       

        {/* Form Content */}
        <View style={styles.formContainer}>
          {/* Table Header */}
          <View style={styles.tableHeader}>
            <Text style={styles.tableTitle}>Observation Log</Text>
            <Text style={styles.tableSubtitle}>Record all observations and activities</Text>
          </View>

          {/* Entries */}
          {entries.map((entry, index) => renderEntryRow(entry, index))}

          {/* Add New Entry Button */}
          <TouchableOpacity style={styles.addButton} onPress={addNewEntry}>
            <MaterialIcons name="add" size={24} color="#fff" />
            <Text style={styles.addButtonText}>Add New Entry</Text>
          </TouchableOpacity>
        </View>
      </ScrollView>

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Generate Report</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Time Picker Modal */}
      {showTimePicker && (
        <DateTimePicker
          value={selectedTime}
          mode="time"
          is24Hour={true}
          display="default"
          onChange={handleTimeChange}
        />
      )}

      {/* Image Options Modal */}
      <Modal
        visible={showImageOptions}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setShowImageOptions(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Select Image Source</Text>
            <TouchableOpacity
              style={styles.modalOption}
              onPress={() => handleImageSelection('camera')}
            >
              <MaterialIcons name="camera-alt" size={24} color="#015185" />
              <Text style={styles.modalOptionText}>Take Photo</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.modalOption}
              onPress={() => handleImageSelection('gallery')}
            >
              <MaterialIcons name="photo-library" size={24} color="#015185" />
              <Text style={styles.modalOptionText}>Choose from Gallery</Text>
            </TouchableOpacity>
            <TouchableOpacity
              style={styles.modalCancelButton}
              onPress={() => setShowImageOptions(false)}
            >
              <Text style={styles.modalCancelText}>Cancel</Text>
            </TouchableOpacity>
          </View>
        </View>
      </Modal>

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <MaterialIcons name="check-circle" size={64} color="#4CAF50" />
            <Text style={styles.successModalTitle}>Report Generated Successfully!</Text>
            <Text style={styles.successModalText}>
              Your HSE-158 Inspection Report has been generated and saved successfully.
            </Text>
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={handleBackToHome}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPdf}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Loading Modal */}
      <Modal visible={isLoading} transparent animationType="fade">
        <View style={styles.syncModalContainer}>
          <View style={styles.syncModalContent}>
            <Text style={styles.syncModalText}>Please Wait</Text>
            <View style={styles.syncIcondiv}>
              <Animated.Image
                source={require('../../../../Images/adminlogin/sync.png')}
                style={[styles.syncIcon, { transform: [{ rotate: rotateInterpolate }] }]}
              />
              <Text style={styles.syncModalTextadd}>Generating Report</Text>
            </View>
          </View>
        </View>
      </Modal>

      <Toast />
    </KeyboardAvoidingView>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  scrollView: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  header: {
    backgroundColor: '#f8f9fa',
    padding: 20,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  headerTitle: {
    fontSize: 24,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    marginBottom: 10,
  },
  headerSubtitle: {
    fontSize: 16,
    color: '#666',
    textAlign: 'center',
    fontStyle: 'italic',
  },
  infoSection: {
    backgroundColor: '#fff',
    padding: 15,
    margin: 20,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    flexDirection: 'row',
    justifyContent: 'space-around',
  },
  infoRow: {
    alignItems: 'center',
  },
  infoLabel: {
    fontSize: 14,
    color: '#666',
    marginBottom: 5,
  },
  infoValue: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
  },
  formContainer: {
    padding: 20,
  },
  tableHeader: {
    alignItems: 'center',
    marginBottom: 20,
  },
  tableTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 5,
  },
  tableSubtitle: {
    fontSize: 14,
    color: '#666',
    fontStyle: 'italic',
  },
  entryRow: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 15,
    marginBottom: 20,
    borderWidth: 1,
    borderColor: '#e9ecef',
    elevation: 2,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 1 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
  },
  rowHeader: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    marginBottom: 15,
    paddingBottom: 10,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  rowNumber: {
    fontSize: 16,
    fontWeight: 'bold',
    color: '#015185',
  },
  removeButton: {
    padding: 5,
  },
  inputRow: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    marginBottom: 15,
  },
  inputColumn: {
    width: '48%',
  },
  columnLabel: {
    fontSize: 14,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 14,
    backgroundColor: '#fff',
    color: '#333',
  },
  timeInput: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
  },
  timeInputText: {
    fontSize: 14,
    color: '#333',
  },
  timePlaceholderText: {
    color: '#999',
  },
  signatureContainer: {
    height: 80, // Match the new image height
    justifyContent: 'center',
  },
  uploadButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    height: 80, // Match the new image height
  },
  uploadButtonText: {
    color: '#015185',
    fontSize: 14,
    fontWeight: '600',
    marginLeft: 8,
  },
  imageContainer: {
    position: 'relative',
    alignItems: 'center',
    justifyContent: 'center',
    height: 80, // Increased height for better image preview
    width: '100%',
  },
  signatureImage: {
    width: '100%',
    height: 80, // Increased height for better image preview
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  removeImageButton: {
    position: 'absolute',
    top: -8,
    right: -8,
    backgroundColor: '#dc3545',
    borderRadius: 12,
    width: 24,
    height: 24,
    alignItems: 'center',
    justifyContent: 'center',
    elevation: 3,
  },
  commentSection: {
    marginBottom: 15,
  },
  commentInput: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 8,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 14,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 80,
    textAlignVertical: 'top',
  },
  addButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    backgroundColor: '#28a745',
    paddingVertical: 15,
    paddingHorizontal: 20,
    borderRadius: 8,
    marginTop: 10,
    elevation: 3,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.2,
    shadowRadius: 3,
  },
  addButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
    marginLeft: 8,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 12,
    padding: 20,
    width: '80%',
    alignItems: 'center',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 20,
  },
  modalOption: {
    flexDirection: 'row',
    alignItems: 'center',
    paddingVertical: 15,
    paddingHorizontal: 20,
    width: '100%',
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  modalOptionText: {
    fontSize: 16,
    color: '#333',
    marginLeft: 12,
  },
  modalCancelButton: {
    marginTop: 15,
    paddingVertical: 12,
    paddingHorizontal: 25,
  },
  modalCancelText: {
    fontSize: 16,
    color: '#666',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0,0,0,0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    width: '70%',
    backgroundColor: 'white',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
  },
  syncModalText: {
    fontSize: 18,
    marginBottom: 10,
    color: '#333',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginRight: 10,
  },
  syncModalTextadd: {
    fontSize: 14,
    color: '#333',
  },
  syncIcondiv: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 30,
    height: 30,
    marginRight: 10,
  },
});

export default HSE_158Step3;