
import React, { useState, useEffect, useRef } from 'react';
import {
  View,
  Text,
  StyleSheet,
  TouchableOpacity,
  TextInput,
  Alert,
  Dimensions,
  Modal,
  ScrollView,
  Platform,
  Image,
  Animated,
} from 'react-native';
import AsyncStorage from '@react-native-async-storage/async-storage';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import { useNavigation } from '@react-navigation/native';
import Toast from 'react-native-toast-message';
import { Dropdown } from 'react-native-element-dropdown';
import DateTimePicker from '@react-native-community/datetimepicker';
import RNHTMLtoPDF from 'react-native-html-to-pdf';
import RNFS from 'react-native-fs';
import database from '@react-native-firebase/database';
import storage from '@react-native-firebase/storage';
import NetInfo from '@react-native-community/netinfo';
import { Linking } from 'react-native';

const { width, height } = Dimensions.get('window');

const SyncModal = ({ visible, onClose, message = 'Downloading PDF...' }) => {
  const rotateAnim = useRef(new Animated.Value(0)).current;

  useEffect(() => {
    if (visible) {
      Animated.loop(
        Animated.timing(rotateAnim, {
          toValue: 1,
          duration: 1000,
          useNativeDriver: true,
        })
      ).start();
    } else {
      rotateAnim.setValue(0);
    }
  }, [visible]);

  return (
    <Modal visible={visible} transparent animationType="fade">
      <View style={styles.syncModalContainer}>
        <View style={styles.syncModalContent}>
          <Text style={styles.syncModalText}>Please Wait</Text>
          <View style={styles.syncIcondiv}>
            <Animated.Image
              source={require('../../../../Images/adminlogin/sync.png')}
              style={[styles.syncIcon, { transform: [{ rotate: rotateAnim.interpolate({ inputRange: [0, 1], outputRange: ['0deg', '360deg'] }) }] }]}
            />
            <Text style={styles.syncModalText}>{message}</Text>
          </View>
        </View>
      </View>
    </Modal>
  );
};

const HSE_123Step3 = ({ setActiveStep }) => {
  const navigation = useNavigation();
  const [formData, setFormData] = useState({
    date: '',
    fleetNumber: '',
    servicedBy: '',
    mileage: '',
    comments: '',
    nextServiceDue: '',
  });
  const [signature, setSignature] = useState('');
  const [signatureRefreshKey, setSignatureRefreshKey] = useState(0);
  const [isLoading, setIsLoading] = useState(false);
  const [isOnline, setIsOnline] = useState(true);
  const [showSuccessModal, setShowSuccessModal] = useState(false);
  const [pdfDownloadUrl, setPdfDownloadUrl] = useState(null);
  const [showDatePicker, setShowDatePicker] = useState(false);
  const [selectedDate, setSelectedDate] = useState(new Date());
  const [generatedPdfPath, setGeneratedPdfPath] = useState(null);
  const [requestNumber, setRequestNumber] = useState('');
  const [hseId, setHseId] = useState(null);
  const [syncModalVisible, setSyncModalVisible] = useState(false);
  const [uploadProgress, setUploadProgress] = useState(0);

  // Get admin user ID function
  const getAdminUserId = async () => {
    const userEmail = await AsyncStorage.getItem('userEmail');
    if (!userEmail) throw new Error('User email not found. Please log in again.');
    const userSnapshot = await database()
      .ref('/Globalusers/admins')
      .orderByChild('email')
      .equalTo(userEmail)
      .once('value');
    const userData = userSnapshot.val();
    if (!userData) throw new Error('User not found in admin database.');
    return Object.keys(userData)[0];
  };

  // Monitor network connectivity
  useEffect(() => {
    const unsubscribe = NetInfo.addEventListener((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    NetInfo.fetch().then((state) => {
      setIsOnline(state.isConnected && state.isInternetReachable !== false);
    });

    return () => unsubscribe();
  }, []);

  const isValidSignatureUri = (uri) => uri && typeof uri === 'string' && uri.startsWith('data:image');

  const handleSignaturePress = () => {
    navigation.navigate('HSE41_42_signature', {
      role: 'Technician',
      onSave: (signature) => {
        setSignature(signature);
        setSignatureRefreshKey((prev) => prev + 1);
      },
    });
  };

  const handleClearSignature = () => {
    setSignature(null);
    setSignatureRefreshKey((prev) => prev + 1);
  };

  const handleImageError = () => {
    console.error('Signature image failed to load');
    Toast.show({ 
      type: 'error', 
      text1: 'Error', 
      text2: 'Failed to load signature. Please try again.', 
      position: 'bottom' 
    });
    handleClearSignature();
  };

  const handleDateChange = (event, selectedDate) => {
    setShowDatePicker(false);
    if (selectedDate) {
      setSelectedDate(selectedDate);
      setFormData(prev => ({ ...prev, date: selectedDate.toLocaleDateString() }));
    }
  };

  const showDatePickerModal = () => {
    setShowDatePicker(true);
  };

  useEffect(() => {
    loadSavedData();
  }, []);

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse123Step3');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setFormData(parsedData.formData || {});
        setSignature(parsedData.signature || '');
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        formData,
        signature,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse123Step3', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const handleInputChange = (field, value) => {
    const updatedData = { ...formData, [field]: value };
    setFormData(updatedData);
    saveData();
  };

  const handleSubmit = async () => {
    // Validate required fields
    if (!formData.date || !formData.fleetNumber || !formData.servicedBy || !formData.mileage) {
      Alert.alert('Error', 'Please fill in all required fields (Date, Fleet Number, Serviced By, Mileage)');
      return;
    }

    // Validate signature
    if (!signature) {
      Alert.alert('Error', 'Please add your signature');
      return;
    }

    try {
      await saveData();
      
      // Generate PDF
      await generatePDF();
      
      // Show success modal with options
      setShowSuccessModal(true);
    } catch (error) {
      console.error('Error submitting form:', error);
      Alert.alert('Error', 'Failed to submit form. Please try again.');
    }
  };

  const handleBack = () => {
    setActiveStep(2);
  };



  const handleViewPDF = async () => {
    if (!requestNumber) {
      Alert.alert('Error', 'Report ID not available.');
      return;
    }

    try {
      const userId = await getAdminUserId(); // Get actual user ID
      const reportRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${requestNumber}`);
      const snapshot = await reportRef.once('value');
      const reportData = snapshot.val();

      if (!reportData) {
        Alert.alert('Error', 'Report not found in the database.');
        return;
      }

      const pdfUrl = reportData.pdfDownloadUrl;
      if (!pdfUrl) {
        Alert.alert('Error', 'PDF URL not available in the report.');
        return;
      }

      // Close modal first
      setShowSuccessModal(false);
      
      // Navigate to HSEViewPDF screen
      navigation.navigate('HSEViewPDF', {
        pdfUrl: pdfUrl,
        onGoBack: () => {
          // Handle going back from PDF view
          navigation.navigate('HSEInspectionHome');
        }
      });
    } catch (error) {
      console.error('Error fetching PDF URL:', error);
      Alert.alert('Error', 'Could not fetch the PDF URL. Please try again.');
    }
  };

  const generatePDF = async () => {
    try {
      setIsLoading(true);
      setSyncModalVisible(true);
      setUploadProgress(10);
      
      // Load data from all steps
      const step1Data = await loadStep1Data();
      const step2Data = await loadStep2Data();
      const step3Data = formData;
      
      // Ensure signature is available - try multiple sources
      let signatureData = step3Data.signature;
      if (!signatureData) {
        // Try to get signature from AsyncStorage as backup
        try {
          const savedStep3 = await AsyncStorage.getItem('hse123Step3');
          if (savedStep3) {
            const parsedStep3 = JSON.parse(savedStep3);
            signatureData = parsedStep3.signature || signatureData;
          }
        } catch (error) {
          console.log('Could not load signature from AsyncStorage:', error);
        }
      }
      
      // Update step3Data with signature
      const finalStep3Data = {
        ...step3Data,
        signature: signatureData
      };
      
      // Debug: Log the data being used
      console.log('Step 1 Data:', step1Data);
      console.log('Step 2 Data:', step2Data);
      console.log('Step 3 Data:', finalStep3Data);
      
      // Debug: Check signature specifically
      console.log('Original Signature Data:', step3Data.signature);
      console.log('Final Signature Data:', finalStep3Data.signature);
      console.log('Signature Type:', typeof finalStep3Data.signature);
      console.log('Signature Length:', finalStep3Data.signature ? finalStep3Data.signature.length : 'No signature');
      
      // Additional debugging for position-specific data
      if (step2Data.selections) {
        console.log('Position-specific selections found:');
        Object.keys(step2Data.selections).forEach(key => {
          if (key.includes('_')) {
            console.log(`${key}: ${step2Data.selections[key]}`);
          }
        });
      }
      
      // Get the request number from step1
      const dbId = step1Data.requestNumber;
      console.log('Step1 Data:', step1Data);
      console.log('Request Number from Step1:', dbId);
      
      if (!dbId) {
        console.error('Step1 Data Keys:', Object.keys(step1Data));
        throw new Error('Request number is missing from step 1 data. Please go back to step 1 and ensure all fields are filled.');
      }

      // Set the request number for this report
      setRequestNumber(dbId);
      setHseId(dbId);

      // Set the request number for this report
      setRequestNumber(dbId);
      setHseId(dbId);

      const html = generatePDFHTML(step1Data, step2Data, finalStep3Data);
      
      const options = {
        html,
        fileName: `HSE_123_${dbId}`,
        directory: 'Documents',
      };

      setUploadProgress(30);
      const file = await RNHTMLtoPDF.convert(options);
      
      if (file.filePath) {
        setGeneratedPdfPath(file.filePath);
        setUploadProgress(50);
        
        // Upload PDF to Firebase Storage
        try {
          const userId = await getAdminUserId(); // Get actual user ID
          const storageRef = storage().ref(`HSE_Reports/${userId}/${dbId}.pdf`);
          const response = await fetch(`file://${file.filePath}`);
          const blob = await response.blob();
          setUploadProgress(70);
          await storageRef.put(blob);
          const downloadUrl = await storageRef.getDownloadURL();
          setUploadProgress(90);
          
          // Save report data to Firebase Database
          const reportData = {
            id: dbId,
            rig: step1Data.selectedEquipment ? (step1Data.selectedEquipment.equipmentName || step1Data.selectedEquipment.equipmentNumber || 'N/A') : 'N/A',
            location: step1Data.gpsAddress || step1Data.coordinates || 'N/A',
            date: step3Data.date,
            supervisor: step3Data.servicedBy,
            selectedRole: 'Technician',
            supervisorSignature: isValidSignatureUri(step3Data.signature) ? step3Data.signature : null,
            managerSignature: null,
            pdfDownloadUrl: downloadUrl,
            createdAt: new Date().toISOString(),
            userId: userId,
            step1: step1Data,
            step2: step2Data,
            step3: {
              date: step3Data.date,
              fleetNumber: step3Data.fleetNumber,
              servicedBy: step3Data.servicedBy,
              mileage: step3Data.mileage,
              comments: step3Data.comments,
              nextServiceDue: step3Data.nextServiceDue,
              signature: isValidSignatureUri(step3Data.signature) ? step3Data.signature : null,
            }
          };
          
          const dbRef = database().ref(`/GlobalHSEInspections/admins/${userId}/${dbId}`);
          await dbRef.set(reportData);
          
          // Increment the report counter
          await incrementReportCounter();
          setUploadProgress(100);
          
          // Remove data from all steps after successful generation
          await flushStepData(dbId);
          
          // Save offline data
          const offlineData = {
            id: dbId,
            requestNumber: dbId,
            pdfPath: file.filePath,
            downloadUrl: downloadUrl,
            timestamp: new Date().toISOString(),
          };
          await AsyncStorage.setItem(`hse123_offline_${dbId}`, JSON.stringify(offlineData));
          
          Toast.show({
            type: 'success',
            text1: 'PDF Generated & Saved',
            text2: `Report saved with ID: ${dbId}`,
            position: 'bottom'
          });
        } catch (dbError) {
          console.error('Error saving to database:', dbError);
          Toast.show({
            type: 'warning',
            text1: 'PDF Generated',
            text2: 'PDF saved locally. Database sync failed.',
            position: 'bottom'
          });
        }
      }
    } catch (error) {
      console.error('Error generating PDF:', error);
      Toast.show({
        type: 'error',
        text1: 'PDF Generation Failed',
        text2: error.message || 'Please try again',
        position: 'bottom'
      });
    } finally {
      setIsLoading(false);
      setSyncModalVisible(false);
      setUploadProgress(0);
    }
  };

  const loadStep1Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hseStep1');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 1 data:', error);
      return {};
    }
  };

  const getCurrentReportNumber = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      const currentNumber = snapshot.val() || 0;
      return currentNumber;
    } catch (error) {
      console.error('Error getting current report number:', error);
      return 0;
    }
  };

  const incrementReportCounter = async () => {
    try {
      const counterRef = database().ref('/GlobalHSEInspections/admins/Reportidnum');
      const snapshot = await counterRef.once('value');
      let currentNumber = snapshot.val() || 0;
      currentNumber++;
      await counterRef.set(currentNumber);
      return currentNumber;
    } catch (error) {
      console.error('Error incrementing report counter:', error);
      return 0;
    }
  };

  const flushStepData = async (hseId) => {
    try {
      await AsyncStorage.removeItem('hseStep1');
      await AsyncStorage.removeItem('hse123Step2');
      await AsyncStorage.removeItem('hse123Step3');
      await AsyncStorage.removeItem(`hse123_offline_${hseId}`);
      console.log('Successfully cleared all step data for HSE123');
    } catch (error) {
      console.error('Error flushing step data:', error);
      throw new Error('Failed to clear local data');
    }
  };

  const loadStep2Data = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse123Step2');
      return savedData ? JSON.parse(savedData) : {};
    } catch (error) {
      console.error('Error loading step 2 data:', error);
      return {};
    }
  };

  const generatePDFHTML = (step1, step2, step3) => {
    const { selections = {}, notes = {} } = step2;
    
    // Helper function to get position-specific data
    const getPositionData = (baseKey, positions) => {
      let positionData = '';
      positions.forEach(position => {
        const positionKey = `${baseKey}_${position.replace(/\s+/g, '_')}`;
        const positionValue = selections[positionKey] || '';
        const positionNote = notes[positionKey] || '';
        if (positionValue || positionNote) {
          positionData += `<div class="sub-comment"><strong>${position}:</strong> ${positionValue}${positionNote ? ` - ${positionNote}` : ''}</div>`;
        }
      });
      return positionData;
    };
    
    // Generate checklist rows with complete structure
    let checklistRows = '';
    
    // 15000km Service
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">15000km Service</td>
      </tr>
      <tr>
        <td>Engine oil changed</td>
        <td>${selections['engineOilChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineOilChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineOilChanged'] || ''}</td>
      </tr>
      <tr>
        <td>Engine oil filter changed</td>
        <td>${selections['engineOilFilterChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineOilFilterChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineOilFilterChanged'] || ''}</td>
      </tr>
      <tr>
        <td>Engine fuel filter changed</td>
        <td>${selections['engineFuelFilterChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineFuelFilterChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineFuelFilterChanged'] || ''}</td>
      </tr>
      <tr>
        <td>Air filter</td>
        <td>${selections['airFilter'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airFilter'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airFilter'] || ''}</td>
      </tr>
      <tr>
        <td>Radiator coolant checked</td>
        <td>${selections['radiatorCoolantChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['radiatorCoolantChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['radiatorCoolantChecked'] || ''}</td>
      </tr>
    `;
    
    // 30000km or 12 Month Service
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">30000km or 12 Month Service</td>
      </tr>
      <tr>
        <td>Radiator coolant changed</td>
        <td>${selections['radiatorCoolantChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['radiatorCoolantChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['radiatorCoolantChanged'] || ''}</td>
      </tr>
      <tr>
        <td>Power steering fluid checked</td>
        <td>${selections['powerSteeringFluidChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['powerSteeringFluidChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['powerSteeringFluidChecked'] || ''}</td>
      </tr>
    `;
    
    // 50000km Service
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">50000km Service</td>
      </tr>
      <tr>
        <td>Power steering fluid changed</td>
        <td>${selections['powerSteeringFluidChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['powerSteeringFluidChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['powerSteeringFluidChanged'] || ''}</td>
      </tr>
      <tr>
        <td>Windscreen washer fluid checked</td>
        <td>${selections['windscreenWasherFluidChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['windscreenWasherFluidChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['windscreenWasherFluidChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Battery</td>
        <td>${selections['battery'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['battery'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['battery'] || ''}</td>
      </tr>
      <tr>
        <td>Battery clamp secured</td>
        <td>${selections['batteryClampSecured'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['batteryClampSecured'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['batteryClampSecured'] || ''}</td>
      </tr>
      <tr>
        <td>Fan/aircon belts checked</td>
        <td>${selections['fanAirconBeltsChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['fanAirconBeltsChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['fanAirconBeltsChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Air compressor</td>
        <td>${selections['airCompressor'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airCompressor'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airCompressor'] || ''}</td>
      </tr>
      <tr>
        <td>Idler bearings checked</td>
        <td>${selections['idlerBearingsChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['idlerBearingsChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['idlerBearingsChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Viscous fan/fan blades checked</td>
        <td>${selections['viscousFanFanBladesChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['viscousFanFanBladesChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['viscousFanFanBladesChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Engine brake/retarder check/trailer brake test</td>
        <td>${selections['engineBrakeRetarderCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineBrakeRetarderCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineBrakeRetarderCheck'] || ''}</td>
      </tr>
      <tr>
        <td>Brake fluid/pedal checked (if applicable)</td>
        <td>${selections['brakeFluidPedalChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['brakeFluidPedalChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['brakeFluidPedalChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Clutch fluid/pedal checked (if applicable)</td>
        <td>${selections['clutchFluidPedalChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['clutchFluidPedalChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['clutchFluidPedalChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Aircon filter checked</td>
        <td>${selections['airconFilterChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airconFilterChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airconFilterChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Aircon filter changed</td>
        <td>${selections['airconFilterChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airconFilterChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airconFilterChanged'] || ''}</td>
      </tr>
      <tr>
        <td>Aircon/heater working</td>
        <td>${selections['airconHeaterWorking'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['airconHeaterWorking'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['airconHeaterWorking'] || ''}</td>
      </tr>
      <tr>
        <td>Engine bay leaks/loose items check</td>
        <td>${selections['engineBayLeaksLooseItemsCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['engineBayLeaksLooseItemsCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['engineBayLeaksLooseItemsCheck'] || ''}</td>
      </tr>
      <tr>
        <td>Windscreen checked</td>
        <td>${selections['windscreenChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['windscreenChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['windscreenChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Windscreen wipers checked/changed</td>
        <td>${selections['windscreenWipersCheckedChanged'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['windscreenWipersCheckedChanged'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['windscreenWipersCheckedChanged'] || ''}</td>
      </tr>
      <tr>
        <td>Gearbox oil level check</td>
        <td>${selections['gearboxOilLevelCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['gearboxOilLevelCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['gearboxOilLevelCheck'] || ''}</td>
      </tr>
    `;
    
    // 50000km or 18 Months Service
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">50000km or 18 Months Service</td>
      </tr>
      <tr>
        <td>Gearbox oil change</td>
        <td>${selections['gearboxOilChange'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['gearboxOilChange'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['gearboxOilChange'] || ''}</td>
      </tr>
      <tr>
        <td>Splitter box check (if applicable)</td>
        <td>${selections['splitterBoxCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['splitterBoxCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['splitterBoxCheck'] || ''}</td>
      </tr>
      <tr>
        <td>Transfer box oil level check (if applicable)</td>
        <td>${selections['transferBoxOilLevelCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['transferBoxOilLevelCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['transferBoxOilLevelCheck'] || ''}</td>
      </tr>
      <tr>
        <td>Diff oil level check</td>
        <td>${selections['diffOilLevelCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['diffOilLevelCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['diffOilLevelCheck'] || ''}</td>
      </tr>
      <tr>
        <td>Diff oil change</td>
        <td>${selections['diffOilChange'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['diffOilChange'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['diffOilChange'] || ''}</td>
      </tr>
      <tr>
        <td>King pin play check</td>
        <td>${selections['kingPinPlayCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['kingPinPlayCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('kingPinPlayCheck', ['1ST AXLE', 'MID AXLE', 'REAR AXLE'])}
          ${notes['kingPinPlayCheck'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['kingPinPlayCheck']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>Wheel bearing check</td>
        <td>${selections['wheelBearingCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['wheelBearingCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('wheelBearingCheck', ['1ST AXLE', 'MID AXLE', 'REAR AXLE'])}
          ${notes['wheelBearingCheck'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['wheelBearingCheck']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>Brake pads/shoes check</td>
        <td>${selections['brakePadsShoesCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['brakePadsShoesCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('brakePadsShoesCheck', ['1ST AXLE %', 'MID AXLE %', 'REAR AXLE %'])}
          ${notes['brakePadsShoesCheck'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['brakePadsShoesCheck']}</div>` : ''}
        </td>
      </tr>
    `;
    
    // Other Checks
    checklistRows += `
      <tr class="section-header">
        <td colspan="4">Other Checks</td>
      </tr>
      <tr>
        <td>Handbrake shoes check/adjustment</td>
        <td>${selections['handbrakeShoesCheckAdjustment'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['handbrakeShoesCheckAdjustment'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['handbrakeShoesCheckAdjustment'] || ''}</td>
      </tr>
      <tr>
        <td>Shock absorbers/dampers check</td>
        <td>${selections['shockAbsorbersDampersCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['shockAbsorbersDampersCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['shockAbsorbersDampersCheck'] || ''}</td>
      </tr>
      <tr>
        <td>Suspension springs splayed/checked</td>
        <td>${selections['suspensionSpringsSplayedChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['suspensionSpringsSplayedChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['suspensionSpringsSplayedChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Hangers checked</td>
        <td>${selections['hangersChecked'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['hangersChecked'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['hangersChecked'] || ''}</td>
      </tr>
      <tr>
        <td>Shackle pins/bushes</td>
        <td>${selections['shacklePinsBushes'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['shacklePinsBushes'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['shacklePinsBushes'] || ''}</td>
      </tr>
      <tr>
        <td>Tie rod end/ball joint change/check</td>
        <td>${selections['tieRodEndBallJointChangeCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['tieRodEndBallJointChangeCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['tieRodEndBallJointChangeCheck'] || ''}</td>
      </tr>
      <tr>
        <td>Wheel nuts secure</td>
        <td>${selections['wheelNutsSecure'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['wheelNutsSecure'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('wheelNutsSecure', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'])}
          ${notes['wheelNutsSecure'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['wheelNutsSecure']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>Tyre wear/pressure</td>
        <td>${selections['tyreWearPressure'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['tyreWearPressure'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('tyreWearPressure', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'])}
          ${notes['tyreWearPressure'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['tyreWearPressure']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>Head/tail/signal/brake lights/hazards</td>
        <td>${selections['headTailSignalBrakeLightsHazards'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['headTailSignalBrakeLightsHazards'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('headTailSignalBrakeLightsHazards', ['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'])}
          ${notes['headTailSignalBrakeLightsHazards'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['headTailSignalBrakeLightsHazards']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>Beacon</td>
        <td>${selections['beacon'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['beacon'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['beacon'] || ''}</td>
      </tr>
      <tr>
        <td>Horn</td>
        <td>${selections['horn'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['horn'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['horn'] || ''}</td>
      </tr>
      <tr>
        <td>Door handles</td>
        <td>${selections['doorHandles'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['doorHandles'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('doorHandles', ['LEFT FRONT', 'RIGHT FRONT'])}
          ${notes['doorHandles'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['doorHandles']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>Window winding mechanism</td>
        <td>${selections['windowWindingMechanism'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['windowWindingMechanism'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('windowWindingMechanism', ['LEFT FRONT', 'LEFT REAR', 'RIGHT FRONT', 'RIGHT REAR'])}
          ${notes['windowWindingMechanism'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['windowWindingMechanism']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>Exterior (dents/paintwork)</td>
        <td>${selections['exteriorDentsPaintwork'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['exteriorDentsPaintwork'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['exteriorDentsPaintwork'] || ''}</td>
      </tr>
      <tr>
        <td>Greased</td>
        <td>${selections['greased'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['greased'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>
          ${getPositionData('greased', ['1ST PROPSHAFT', '2ND PROPSHAFT', 'AXLES', 'SHACKLES'])}
          ${notes['greased'] ? `<div class="sub-comment"><strong>General Note:</strong> ${notes['greased']}</div>` : ''}
        </td>
      </tr>
      <tr>
        <td>Buggy whip</td>
        <td>${selections['buggyWhip'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['buggyWhip'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['buggyWhip'] || ''}</td>
      </tr>
      <tr>
        <td>5th wheel tested (if applicable)</td>
        <td>${selections['fifthWheelTested'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['fifthWheelTested'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['fifthWheelTested'] || ''}</td>
      </tr>
      <tr>
        <td>Chevron</td>
        <td>${selections['chevron'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['chevron'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['chevron'] || ''}</td>
      </tr>
      <tr>
        <td>Fire extinguisher check/pressure check</td>
        <td>${selections['fireExtinguisherCheck'] === 'YES/OK' ? '✓' : ''}</td>
        <td>${selections['fireExtinguisherCheck'] === 'NO/DEF' ? '✓' : ''}</td>
        <td>${notes['fireExtinguisherCheck'] || ''}</td>
      </tr>
    `;

    return `
      <!DOCTYPE html>
      <html lang="en">
      <head>
          <meta charset="UTF-8">
          <meta name="viewport" content="width=device-width, initial-scale=1.0">
          <title>HSE-FOR-8.1.1 Workshop Truck Check Sheet</title>
          <style>
              body {
                  font-family: Arial, sans-serif;
                  margin: 10px;
                  padding: 0;
                  font-size: 14px;
              }
              .page-title {
                  border-bottom: 3px solid #015185;
                  margin-bottom: 20px;
                  padding-bottom: 15px;
              }
              h1 {
                  text-align: center;
                  margin-bottom: 10px;
                  font-size: 1.5em;
              }
              .header-container {
                  display: flex;
                  flex-wrap: wrap;
                  justify-content: space-between;
                  align-items: flex-start;
                  margin-bottom: 15px;
                  gap: 20px;
              }
              .logo-section {
                  flex: 0 0 auto;
                  text-align: center;
                  padding: 10px;
              }
              .header-info-section {
                  flex: 1;
                  display: flex;
                  justify-content: flex-end;
              }
              .header-table {
                  border-collapse: collapse;
                  width: 100%;
                  max-width: 250px;
              }
              .header-table th, .header-table td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
              }
              .header-table th {
                  background-color: #f2f2f2;
              }
              h2 {
                  text-align: center;
                  font-size: 1.2em;
                  margin: 15px 0;
              }
              table.checklist {
                  width: 100%;
                  border-collapse: collapse;
                  margin-bottom: 15px;
              }
              .checklist th, .checklist td {
                  border: 1px solid #ddd;
                  padding: 6px;
                  text-align: left;
                  font-size: 0.9em;
              }
              .checklist th {
                  background-color: #f2f2f2;
              }
              .section-header td {
                  background-color: #e0e0e0;
                  font-weight: bold;
                  text-align: center;
              }
              .info-row td {
                  padding: 8px;
              }
              .sub-comment {
                  margin: 4px 0;
                  padding: 4px;
                  border-top: 1px solid #eee;
                  font-size: 0.85em;
                  background-color: #f8f9fa;
                  border-radius: 3px;
                  padding: 6px;
              }
              .sub-comment strong {
                  color: #015185;
                  font-weight: 600;
              }
              .signature-section {
                  text-align: center;
                  margin-top: 20px;
                  padding: 15px;
                  border-top: 2px solid #015185;
                  background-color: #f8f9fa;
                  border-radius: 8px;
              }
              .signature-section p {
                  margin-bottom: 10px;
                  font-weight: bold;
                  color: #015185;
                  font-size: 16px;
              }
              .signature-section img {
                  max-width: 200px;
                  max-height: 100px;
                  border: 1px solid #ddd;
                  border-radius: 5px;
                  box-shadow: 0 2px 4px rgba(0,0,0,0.1);
              }
              .checklist th:nth-child(1), .checklist td:nth-child(1) {
                  width: 55%;
              }
              .checklist th:nth-child(2), .checklist td:nth-child(2) {
                  width: 15%;
                  text-align: center;
              }
              .checklist th:nth-child(3), .checklist td:nth-child(3) {
                  width: 15%;
                  text-align: center;
              }
              .checklist th:nth-child(4), .checklist td:nth-child(4) {
                  width: 15%;
              }
              .footer-table {
                  width: 100%;
                  border-collapse: collapse;
                  margin-top: 15px;
              }
              .footer-table td {
                  border: 1px solid #ddd;
                  padding: 8px;
                  font-size: 0.9em;
              }
          </style>
      </head>
      <body>
         
          <div class="header-container">
              <div class="logo-section">
                  <img src="https://www.titanafricadrilling.com/wp-content/uploads/2022/10/titan-drlling_blue.png" alt="Titan Drilling Logo" style="width: 200px; height: auto;">
              </div>
              <div class="header-info-section">
                  <table class="header-table">
                      <tr>
                          <th colspan="2">Header Information</th>
                      </tr>
                      <tr>
                          <td><strong>Document ID:</strong></td>
                          <td>HSE-123</td>
                      </tr>
                      <tr>
                          <td><strong>Version Date:</strong></td>
                          <td>19/01/2024</td>
                      </tr>
                      <tr>
                          <td><strong>Version:</strong></td>
                          <td>1</td>
                      </tr>
                      <tr>
                          <td><strong>Approved By:</strong></td>
                          <td>HSM</td>
                      </tr>
                  </table>
              </div>
          </div>

          <table class="checklist">
              <tr class="info-row">
                  <td><strong>Date:</strong> ${step3.date || ''}</td>
                  <td><strong>Fleet Number:</strong> ${step3.fleetNumber || ''}</td>
                  <td><strong>Mileage:</strong> ${step3.mileage || ''}</td>
                  <td><strong>Serviced By:</strong> ${step3.servicedBy || ''}</td>
              </tr>
              <tr class="info-row">
                  <td><strong>Country:</strong> ${step1.country || ''}</td>
                  <td><strong>Project:</strong> ${step1.project || ''}</td>
                  <td><strong>Equipment:</strong> ${step1.selectedEquipment ? (step1.selectedEquipment.equipmentName || step1.selectedEquipment.equipmentNumber || 'N/A') : ''}</td>
                  <td><strong>Request #:</strong> ${step1.requestNumber || ''}</td>
              </tr>
          </table>
        

          <h2>Checklist</h2>

          <table class="checklist">
              <tr>
                  <th>Task</th>
                  <th>Yes</th>
                  <th>No</th>
                  <th>Comments</th>
              </tr>
              ${checklistRows || '<tr><td colspan="4" style="text-align: center;">No checklist items available</td></tr>'}
          </table>

          <table class="footer-table">
              <tr class="info-row">
                  <td><strong>Comments:</strong> ${step3.comments || ''}</td>
                  <td><strong>Next Service Due (km):</strong> ${step3.nextServiceDue || ''}</td>
                  <td><strong>Signature:</strong> ${step3.signature && step3.signature.length > 0 ? `
                      <div style="text-align: center;">
                          <img src="${step3.signature}" alt="Technician Signature" style="max-width: 80px; max-height: 40px; border: 1px solid #015185; border-radius: 4px; box-shadow: 0 2px 4px rgba(0,0,0,0.1);" />
                          <br><small style="color: #4CAF50; font-weight: bold;">✓ Signed</small>
                      </div>
                  ` : `
                      <span style="color: #FF6B6B; font-weight: bold;">⚠ Not Signed</span>
                  `}</td>
              </tr>
          </table>
          

          

          

      </body>
      </html>
    `;
  };

  return (
    <View style={styles.container}>
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
        <Text style={styles.pageTitle}>Workshop Truck Check Sheet - Review & Submit</Text>
        
        {isLoading && (
          <View style={styles.loadingContainer}>
            <MaterialIcons name="hourglass-empty" size={40} color="#015185" />
            <Text style={styles.loadingText}>Generating PDF Report...</Text>
            <Text style={styles.loadingSubtext}>Please wait while we compile your data</Text>
          </View>
        )}

        {/* Form Section */}
        <View style={styles.section}>
          <View style={styles.sectionHeader}>
            <Text style={styles.sectionTitle}>FORM DETAILS</Text>
          </View>
          
          <View style={styles.formFields}>
            {/* DATE Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>DATE:</Text>
              <TouchableOpacity style={styles.pickerButton} onPress={showDatePickerModal}>
                <Text style={styles.pickerButtonText}>
                  {formData.date || selectedDate.toLocaleDateString()}
                </Text>
                <MaterialIcons name="calendar-today" size={20} color="#015185" />
              </TouchableOpacity>
            </View>

            {/* FLEET # Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>FLEET #:</Text>
              <TextInput
                style={styles.input}
                value={formData.fleetNumber}
                onChangeText={(value) => handleInputChange('fleetNumber', value)}
                placeholder="Enter fleet number"
                placeholderTextColor="#999"
              />
            </View>

            {/* SERVICED BY Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>SERVICED BY:</Text>
              <TextInput
                style={styles.input}
                value={formData.servicedBy}
                onChangeText={(value) => handleInputChange('servicedBy', value)}
                placeholder="Enter service technician name"
                placeholderTextColor="#999"
              />
            </View>

            {/* MILEAGE Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>MILEAGE:</Text>
              <TextInput
                style={styles.input}
                value={formData.mileage}
                onChangeText={(value) => handleInputChange('mileage', value)}
                placeholder="Enter current mileage"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
            </View>

            {/* COMMENTS Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>COMMENTS:</Text>
              <TextInput
                style={styles.input}
                value={formData.comments}
                onChangeText={(value) => handleInputChange('comments', value)}
                placeholder="Enter any additional comments"
                placeholderTextColor="#999"
                multiline
                numberOfLines={4}
                textAlignVertical="top"
              />
            </View>

            {/* NEXT SERVICE DUE Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>NEXT SERVICE DUE:</Text>
              <TextInput
                style={styles.input}
                value={formData.nextServiceDue}
                onChangeText={(value) => handleInputChange('nextServiceDue', value)}
                placeholder="Enter next service due mileage"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
            </View>

            {/* SIGNATURE Field */}
            <View style={styles.fieldContainer}>
              <Text style={styles.fieldLabel}>SIGNATURE:</Text>
              <View style={styles.signatureBoxImage}>
                {isValidSignatureUri(signature) ? (
                  <TouchableOpacity 
                    style={styles.plusIconCircleContainer} 
                    onPress={handleSignaturePress}
                    key={signatureRefreshKey}
                  >
                    <Image
                      source={{ uri: signature }}
                      style={styles.signatureImage}
                      resizeMode="contain"
                      onError={handleImageError}
                    />
                  </TouchableOpacity>
                ) : (
                  <TouchableOpacity 
                    style={styles.plusIconCircleContainer} 
                    onPress={handleSignaturePress}
                    key={signatureRefreshKey}
                  >
                    <View style={styles.plusCircle}>
                      <MaterialIcons name="add" size={width * 0.13} color="#015185" />
                    </View>
                  </TouchableOpacity>
                )}
              </View>
            </View>
          </View>
        </View>
      </ScrollView>

      {/* Sync Modal */}
      <SyncModal 
        visible={syncModalVisible} 
        onClose={() => setSyncModalVisible(false)} 
        message="Downloading PDF..." 
      />

      {/* Success Modal */}
      <Modal
        visible={showSuccessModal}
        transparent
        animationType="fade"
        onRequestClose={() => setShowSuccessModal(false)}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.successModalContent}>
            <MaterialIcons name="check-circle" size={48} color="#4CAF50" />
            <Text style={styles.successModalTitle}>Success</Text>
            <Text style={styles.successModalText}>
              HSE Inspection Report Generated and Saved
            </Text>
            {requestNumber && (
              <Text style={styles.reportIdText}>
                Report ID: {requestNumber}
              </Text>
            )}
            <View style={styles.successButtonContainer}>
              <TouchableOpacity
                style={styles.successButton}
                onPress={() => {
                  setShowSuccessModal(false);
                  navigation.navigate('HSEInspectionHome');
                }}
              >
                <Text style={styles.successButtonText}>Back</Text>
              </TouchableOpacity>
              <TouchableOpacity
                style={[styles.successButton, { marginLeft: 10 }]}
                onPress={handleViewPDF}
              >
                <Text style={styles.successButtonText}>View PDF</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>

      {/* Date Picker Modal */}
      {showDatePicker && (
        <DateTimePicker
          value={selectedDate}
          mode="date"
          display={Platform.OS === 'ios' ? 'spinner' : 'default'}
          onChange={handleDateChange}
        />
      )}

      {/* Bottom Buttons */}
      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={handleBack}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleSubmit} disabled={isLoading}>
          <Text style={styles.nextButtonText}>
            {isLoading ? 'Generating PDF...' : 'Generate Report'}
          </Text>
          <MaterialIcons style={styles.bottomicon1} name={isLoading ? "hourglass-empty" : "check"} size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      <Toast />
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 100,
  },
  pageTitle: {
    fontSize: 20,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 20,
    backgroundColor: '#F2F2F2',
  },
  section: {
    marginBottom: 30,
    backgroundColor: '#fff',
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
    overflow: 'hidden',
    margin: 20,
  },
  sectionHeader: {
    backgroundColor: '#f0f0f0',
    padding: 15,
    borderBottomWidth: 1,
    borderBottomColor: '#e9ecef',
  },
  sectionTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    textAlign: 'center',
  },
  formFields: {
    padding: 20,
  },
  fieldContainer: {
    marginBottom: 20,
  },
  fieldLabel: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 8,
  },
  input: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
  },
  pickerButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    backgroundColor: '#fff',
    minHeight: 40,
    width: '100%',
  },
  pickerButtonText: {
    flex: 1,
    fontSize: 16,
    color: '#333',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  signatureBoxImage: {
    height: height / 5,
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: width * 0.012,
    justifyContent: 'center',
    alignItems: 'center',
  },
  plusIconCircleContainer: {
    justifyContent: 'center',
    alignItems: 'center',
    width: '100%',
    height: '100%',
  },
  plusCircle: {
    width: '100%',
    height: '100%',
    justifyContent: 'center',
    alignItems: 'center',
  },
  signatureImage: {
    width: '100%',
    height: '100%',
    borderRadius: 2,
    backgroundColor: '#f5f5f5',
  },
  loadingContainer: {
    alignItems: 'center',
    justifyContent: 'center',
    padding: 30,
    backgroundColor: '#f8f9fa',
    margin: 20,
    borderRadius: 8,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  loadingText: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginTop: 15,
    textAlign: 'center',
  },
  loadingSubtext: {
    fontSize: 14,
    color: '#666',
    marginTop: 8,
    textAlign: 'center',
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  successModalContent: {
    backgroundColor: '#fff',
    borderRadius: 10,
    padding: 20,
    alignItems: 'center',
    width: '80%',
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 2 },
    shadowOpacity: 0.3,
    shadowRadius: 4,
  },
  successModalTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    marginVertical: 10,
  },
  successModalText: {
    fontSize: 14,
    color: '#333',
    textAlign: 'center',
    marginBottom: 20,
  },
  successButtonContainer: {
    flexDirection: 'row',
    justifyContent: 'center',
    marginTop: 10,
  },
  successButton: {
    backgroundColor: '#015185',
    paddingVertical: 10,
    paddingHorizontal: 20,
    borderRadius: 5,
  },
  successButtonText: {
    color: '#fff',
    fontSize: 16,
    fontWeight: '600',
  },
  reportIdText: {
    fontSize: 16,
    color: '#015185',
    fontWeight: '600',
    textAlign: 'center',
    marginBottom: 20,
    backgroundColor: '#f0f0f0',
    padding: 10,
    borderRadius: 5,
  },
  syncModalContainer: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  syncModalContent: {
    backgroundColor: '#fff',
    borderRadius: 16,
    padding: 25,
    width: width * 0.7,
    alignItems: 'center',
    elevation: 8,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: 4 },
    shadowOpacity: 0.3,
    shadowRadius: 8,
  },
  syncModalText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  syncIcondiv: {
    flexDirection: 'column',
    alignItems: 'center',
    justifyContent: 'center',
  },
  syncIcon: {
    width: 22,
    height: 22,
    marginBottom: 12,
  },
});

export default HSE_123Step3;