import React, { useState, useEffect } from 'react';
import {
  View,
  Text,
  StyleSheet,
  ScrollView,
  TouchableOpacity,
  TextInput,
  Modal,
  Dimensions,
} from 'react-native';
import MaterialIcons from 'react-native-vector-icons/MaterialIcons';
import AsyncStorage from '@react-native-async-storage/async-storage';

const { width } = Dimensions.get('window');
const { height } = Dimensions.get('window');

const HSE_122Step2 = ({ setActiveStep }) => {
  const [sections, setSections] = useState([]);
  const [expandedSections, setExpandedSections] = useState({});
  const [selections, setSelections] = useState({});
  const [notes, setNotes] = useState({});
  const [inspectionId, setInspectionId] = useState('hse122');
  const [noteModal, setNoteModal] = useState({
    visible: false,
    itemKey: '',
    currentNote: '',
  });
  const [voltageInputs, setVoltageInputs] = useState({});
  const [chargingInputs, setChargingInputs] = useState({});

  useEffect(() => {
    loadSavedData();
    setupChecklistData();
  }, []);

  const setupChecklistData = () => {
    const checklistData = [
      {
        key: 'engine',
        name: 'ENGINE',
        subItems: [
          { key: 'engineOil', itemName: 'Engine oil (CHANGED/LEVEL)', inputType: 'Yes/No' },
          { key: 'coolant', itemName: 'Coolant (LEAKS/LEVEL)', inputType: 'Yes/No' },
          { key: 'airFilter', itemName: 'Air filter / intake pipes (CLEANED HOUSING/CHANGED)', inputType: 'Yes/No' },
          { key: 'fuelFilter', itemName: 'Fuel filter (CHANGED)', inputType: 'Yes/No' },
          { key: 'fuelSystem', itemName: 'Fuel system (LEAKS)', inputType: 'Yes/No' },
          { key: 'vBelts', itemName: 'V belts (CONDITION/TENSION)', inputType: 'Yes/No' },
          { key: 'radiator', itemName: 'Radiator (CLEANED/ PRESSURE TESTED)', inputType: 'Yes/No' },
          { key: 'radiatorCap', itemName: 'Radiator cap (INSPECTED)', inputType: 'Yes/No' },
          { key: 'engineMountings', itemName: 'Engine mountings (WORN/SECURED)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'every2000hrs',
        name: 'EVERY 2000 HRS',
        subItems: [
          { key: 'valveClearance', itemName: 'Valve clearance (ADJUSTED/CHECKED)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'electrical',
        name: 'ELECTRICAL',
        subItems: [
          { key: 'batteries', itemName: 'Batteries (POLE CONDITION/ JELLED)', inputType: 'Yes/No' },
          { key: 'dateEngraved', itemName: 'Date engraved', inputType: 'Yes/No' },
          { key: 'voltageCranking', itemName: 'Voltage at cranking (VOLTS)', inputType: 'Yes/No' },
          { key: 'alternator', itemName: 'Alternator (MOUNTING/CHECKED)', inputType: 'Yes/No' },
          { key: 'starter', itemName: 'Starter (MOUNTING/TESTED)', inputType: 'Yes/No' },
          { key: 'lights', itemName: 'Lights (WORKING/WIRING)', inputType: 'Yes/No' },
          { key: 'chargingSystem', itemName: 'Charging system (CHARGING/WIRES/FUSES)', inputType: 'Yes/No' },
          { key: 'rpmGauge', itemName: 'RPM/Volt/Temp/Oil gauge (WORKING/CALIBRATED)', inputType: 'Yes/No' },
          { key: 'eStops', itemName: 'E stops (CHECKED / TESTED)', inputType: 'Yes/No' },
          { key: 'limitSwitches', itemName: 'Limit switches (FUNCTION TEST/ADJUST)', inputType: 'Yes/No' },
          { key: 'engineProtection', itemName: 'Engine protection cut out', inputType: 'Yes/No' },
          { key: 'horn', itemName: 'Horn (CHECKED / TESTED)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'mast',
        name: 'MAST',
        subItems: [
          { key: 'pins', itemName: 'Pins (SECURE)', inputType: 'Yes/No' },
          { key: 'slides', itemName: 'Slides (NO FREE PLAY)', inputType: 'Yes/No' },
          { key: 'inspectionPlate', itemName: 'Inspection plate (DATED)', inputType: 'Yes/No' },
          { key: 'cracks', itemName: 'Cracks', inputType: 'Yes/No' },
          { key: 'sheaveWheels', itemName: 'Sheave wheels (INSPECTED/CHANGED)', inputType: 'Yes/No' },
          { key: 'wireline', itemName: 'Wireline (INSPECTED/CHANGED)', inputType: 'Yes/No' },
          { key: 'winchCable', itemName: 'Winch cable (INSPECTED/CHANGED)', inputType: 'Yes/No' },
          { key: 'rotationGuard', itemName: 'Rotation guard switch (FUNCTION TEST/ADJUST)', inputType: 'Yes/No' },
          { key: 'rodKicker', itemName: 'Rod kicker (FREE/SECURE)', inputType: 'Yes/No' },
          { key: 'fineFeedRam', itemName: 'Fine feed ram (SECURE/LEAKS)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'drillHead',
        name: 'DRILL HEAD',
        subItems: [
          { key: 'chainTension', itemName: 'Chain tension (IF APPLICABLE)', inputType: 'Yes/No' },
          { key: 'gearboxBreather', itemName: 'Check gearbox breather (CLEAN)', inputType: 'Yes/No' },
          { key: 'headLockBolts', itemName: 'Head lock bolts (INSPECT)', inputType: 'Yes/No' },
          { key: 'slideBushes', itemName: 'Slide bushes (INSPECT/CHANGE)', inputType: 'Yes/No' },
          { key: 'strainer', itemName: 'Strainer (CLEAN)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'hydraulics',
        name: 'HYDRAULICS',
        subItems: [
          { key: 'hydraulicPressure', itemName: 'Check hydraulic pressure (TEST PRESSURES)', inputType: 'Yes/No' },
          { key: 'hydraulicHoses', itemName: 'Hydraulic hoses (LEAKS/CONDITION)', inputType: 'Yes/No' },
          { key: 'hydraulicOil', itemName: 'Hydraulic oil (LEVEL/QUALITY)', inputType: 'Yes/No' },
          { key: 'hydraulicTankBreather', itemName: 'Hydraulic tank breather (CLEAN)', inputType: 'Yes/No' },
          { key: 'hydraulicOilFilter', itemName: 'Hydraulic oil filter (CLEAN/CHECK)', inputType: 'Yes/No' },
          { key: 'rotationMotor', itemName: 'Rotation motor (LEAKS/CONDITION)', inputType: 'Yes/No' },
          { key: 'wirelineDrum', itemName: 'Wireline drum/motor (SECURE/LEAKS)', inputType: 'Yes/No' },
          { key: 'winchDrum', itemName: 'Winch drum/motor (SECURE/LEAKS)', inputType: 'Yes/No' },
          { key: 'hydraulicOilCoolers', itemName: 'Hydraulic oil coolers (CLEAN/LEAKS)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'beanPump',
        name: 'BEAN PUMP',
        subItems: [
          { key: 'pistonsLiners', itemName: 'Pistons/liners (INSPECT/HANGE)', inputType: 'Yes/No' },
          { key: 'ballsCradles', itemName: 'Balls and cradles (CONDITION)', inputType: 'Yes/No' },
          { key: 'seals', itemName: 'Seals', inputType: 'Yes/No' },
          { key: 'oil', itemName: 'Oil (LEVEL/CHANGE)', inputType: 'Yes/No' },
        ]
      },
      {
        key: 'tyresAndHubs',
        name: 'TYRES AND HUBS',
        subItems: [
          { key: 'tyre', itemName: 'Tyre (CONDITION)', inputType: 'Yes/No' },
          { key: 'hubs', itemName: 'Hubs (BEARING FREEPLAY)', inputType: 'Yes/No' },
          { key: 'bolts', itemName: 'Bolts (SECURE)', inputType: 'Yes/No' },
        ]
      }
    ];

    setSections(checklistData);
    
    // Initialize expanded sections
    const initialExpanded = {};
    checklistData.forEach((section) => {
      initialExpanded[section.key] = false;
    });
    setExpandedSections(initialExpanded);
  };

  const loadSavedData = async () => {
    try {
      const savedData = await AsyncStorage.getItem('hse122Step2');
      if (savedData) {
        const parsedData = JSON.parse(savedData);
        setSelections(parsedData.selections || {});
        setNotes(parsedData.notes || {});
      }
    } catch (error) {
      console.error('Error loading saved data:', error);
    }
  };

  const saveData = async () => {
    try {
      const dataToSave = {
        selections,
        notes,
        timestamp: new Date().toISOString()
      };
      await AsyncStorage.setItem('hse122Step2', JSON.stringify(dataToSave));
    } catch (error) {
      console.error('Error saving data:', error);
    }
  };

  const getOptionsForInputType = (inputType) => {
    switch (inputType) {
      case 'Yes/No':
        return ['YES/OK', 'NO/DEF'];
      default:
        return ['YES/OK', 'NO/DEF'];
    }
  };

  const handleSelectOption = async (itemKey, option) => {
    setSelections((prev) => {
      const updatedSelections = {
        ...prev,
        [itemKey]: option,
      };
      saveData();
      return updatedSelections;
    });
  };

  const openNoteModal = (itemKey) => {
    setNoteModal({
      visible: true,
      itemKey,
      currentNote: notes[itemKey] || '',
    });
  };

  const saveNote = () => {
    const { itemKey, currentNote } = noteModal;
    if (currentNote.trim()) {
      setNotes(prev => ({
        ...prev,
        [itemKey]: currentNote.trim()
      }));
      saveData();
    }
    setNoteModal({ visible: false, itemKey: '', currentNote: '' });
  };

  const handleVoltageInputChange = (itemKey, value) => {
    setVoltageInputs(prev => ({
      ...prev,
      [itemKey]: value
    }));
  };

  const handleChargingInputChange = (itemKey, value) => {
    setChargingInputs(prev => ({
      ...prev,
      [itemKey]: value
    }));
  };

  const toggleSection = (sectionKey) => {
    setExpandedSections((prev) => ({
      ...prev,
      [sectionKey]: !prev[sectionKey],
    }));
  };

  const renderItem = (item, index) => {
    const selectedOption = selections[item.key];
    const itemNote = notes[item.key] || '';
    const options = getOptionsForInputType(item.inputType);

    return (
      <View key={item.key} style={styles.centeredItem}>
        <View style={styles.itemContainer}>
          <Text style={styles.itemText}>
            {index + 1}. {item.itemName}
          </Text>
          <View style={styles.itemRow}>
            <View style={styles.optionsContainer}>
              {options.map((option) => {
                const isSelected = selectedOption === option;
                return (
                  <TouchableOpacity
                    key={option}
                    style={[
                      styles.optionButton,
                      isSelected && styles.selectedOptionButton,
                      isSelected && option === 'YES/OK' && styles.okButton,
                      isSelected && option === 'NO/DEF' && styles.rejectButton,
                    ]}
                    onPress={() => handleSelectOption(item.key, option)}
                  >
                    <Text
                      style={[
                        styles.optionText,
                        isSelected && styles.optionTextSelected,
                      ]}
                    >
                      {option}
                    </Text>
                  </TouchableOpacity>
                );
              })}
            </View>
            <View style={styles.iconsContainer}>
              <TouchableOpacity
                style={styles.iconButton}
                onPress={() => openNoteModal(item.key)}
              >
                <MaterialIcons name="note-add" size={24} color="#333" />
              </TouchableOpacity>
            </View>
          </View>
          {itemNote && (
            <Text style={styles.noteText}>
              Note: {itemNote}
            </Text>
          )}
          
          {/* Voltage Input Field for Voltage at Cranking */}
          {item.key === 'voltageCranking' && (
            <View style={styles.inputFieldContainer}>
              <Text style={styles.inputFieldLabel}>VOLTS:</Text>
              <TextInput
                style={styles.inputField}
                value={voltageInputs[item.key] || ''}
                onChangeText={(value) => handleVoltageInputChange(item.key, value)}
                placeholder="Enter voltage value"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
            </View>
          )}
          
          {/* Charging System Input Fields */}
          {item.key === 'chargingSystem' && (
            <View style={styles.inputFieldContainer}>
              <Text style={styles.inputFieldLabel}>V RUNNING:</Text>
              <TextInput
                style={styles.inputField}
                value={chargingInputs[`${item.key}_running`] || ''}
                onChangeText={(value) => handleChargingInputChange(`${item.key}_running`, value)}
                placeholder="Enter voltage when running"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
              <Text style={styles.inputFieldLabel}>V NOT RUNNING:</Text>
              <TextInput
                style={styles.inputField}
                value={chargingInputs[`${item.key}_notRunning`] || ''}
                onChangeText={(value) => handleChargingInputChange(`${item.key}_notRunning`, value)}
                placeholder="Enter voltage when not running"
                placeholderTextColor="#999"
                keyboardType="numeric"
              />
            </View>
          )}
          
          {/* Position-specific input fields for items that need position checking */}
          {item.key === 'tyre' && (
            <View style={styles.inputFieldContainer}>
              <Text style={styles.inputFieldLabel}>Check positions:</Text>
              <View style={styles.positionsGrid}>
                {['LEFT REAR', 'RIGHT REAR', 'LEFT FRONT', 'RIGHT FRONT', 'SPARE'].map((position) => {
                  const positionKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
                  const positionValue = selections[positionKey] || '';
                  
                  return (
                    <View key={position} style={styles.positionItem}>
                      <Text style={styles.positionLabel}>{position}:</Text>
                      <View style={styles.positionOptions}>
                        {['OK', 'CHECK', 'REPLACE'].map((option) => {
                          const isSelected = positionValue === option;
                          return (
                            <TouchableOpacity
                              key={option}
                              style={[
                                styles.positionOptionButton,
                                isSelected && styles.positionOptionSelected,
                                isSelected && option === 'OK' && styles.okButton,
                                isSelected && option === 'CHECK' && styles.checkButton,
                                isSelected && option === 'REPLACE' && styles.replaceButton,
                              ]}
                              onPress={() => handleSelectOption(positionKey, option)}
                            >
                              <Text
                                style={[
                                  styles.positionOptionText,
                                  isSelected && styles.positionOptionTextSelected,
                                ]}
                              >
                                {option}
                              </Text>
                            </TouchableOpacity>
                          );
                        })}
                      </View>
                    </View>
                  );
                })}
              </View>
            </View>
          )}
          
          {item.key === 'hubs' && (
            <View style={styles.inputFieldContainer}>
              <Text style={styles.inputFieldLabel}>Check positions:</Text>
              <View style={styles.positionsGrid}>
                {['1ST AXLE', 'MID AXLE', 'REAR AXLE'].map((position) => {
                  const positionKey = `${item.key}_${position.replace(/\s+/g, '_')}`;
                  const positionValue = selections[positionKey] || '';
                  
                  return (
                    <View key={position} style={styles.positionItem}>
                      <Text style={styles.positionLabel}>{position}:</Text>
                      <View style={styles.positionOptions}>
                        {['OK', 'CHECK', 'REPLACE'].map((option) => {
                          const isSelected = positionValue === option;
                          return (
                            <TouchableOpacity
                              key={option}
                              style={[
                                styles.positionOptionButton,
                                isSelected && styles.positionOptionSelected,
                                isSelected && option === 'OK' && styles.okButton,
                                isSelected && option === 'CHECK' && styles.checkButton,
                                isSelected && option === 'REPLACE' && styles.replaceButton,
                              ]}
                              onPress={() => handleSelectOption(positionKey, option)}
                            >
                              <Text
                                style={[
                                  styles.positionOptionText,
                                  isSelected && styles.positionOptionTextSelected,
                                ]}
                              >
                                {option}
                              </Text>
                            </TouchableOpacity>
                          );
                        })}
                      </View>
                    </View>
                  );
                })}
              </View>
            </View>
          )}
        </View>
        <View style={styles.itemBorder} />
      </View>
    );
  };

  const handleNext = () => {
    setActiveStep(3);
  };

  return (
    <View style={styles.container}>
     
      <ScrollView style={styles.mainContent} contentContainerStyle={styles.scrollContent}>
       

        {sections.length > 0 ? (
          sections.map((section) => {
            // Calculate selected items count for this section
            const sectionItems = section.subItems || [];
            const selectedItemsCount = sectionItems.filter(item => selections[item.key]).length;
            const totalItemsCount = sectionItems.length;
            
  return (
              <View key={section.key}>
                <TouchableOpacity
                  style={styles.sectionHeader}
                  onPress={() => toggleSection(section.key)}
                >
                  <View style={styles.sectionHeaderContent}>
                    <MaterialIcons
                      name={expandedSections[section.key] ? 'expand-less' : 'expand-more'}
                      size={20}
                      color={expandedSections[section.key] ? '#015185' : '#333'}
                    />
                    <View style={styles.sectionHeaderTextContainer}>
                      <Text style={styles.sectionHeaderText}>{section.name}</Text>
                      <Text style={styles.sectionItemCount}>
                        ({selectedItemsCount}/{totalItemsCount} selected)
                      </Text>
                    </View>
                  </View>
                </TouchableOpacity>
                {expandedSections[section.key] && (
                  <View style={styles.sectionContent}>
                    {section.subItems.length > 0 ? (
                      section.subItems.map((item, index) => renderItem(item, index))
                    ) : (
                      <Text style={styles.emptySectionText}>No inspection items</Text>
                    )}
                  </View>
                )}
              </View>
            );
          })
        ) : (
          <Text style={styles.emptySectionText}>No sections available</Text>
        )}
      </ScrollView>

      <View style={styles.buttonContainer}>
        <TouchableOpacity style={styles.cancelButton} onPress={() => setActiveStep(1)}>
          <MaterialIcons style={styles.bottomicon} name="chevron-left" size={16} color="#015185" />
          <Text style={styles.cancelButtonText}>Back</Text>
        </TouchableOpacity>
        <TouchableOpacity style={styles.nextButton} onPress={handleNext}>
          <Text style={styles.nextButtonText}>Assessment Complete</Text>
          <MaterialIcons style={styles.bottomicon1} name="chevron-right" size={16} color="#015185" />
        </TouchableOpacity>
      </View>

      {/* Note Modal */}
      <Modal
        visible={noteModal.visible}
        transparent={true}
        animationType="slide"
        onRequestClose={() => setNoteModal({ visible: false, itemKey: '', currentNote: '' })}
      >
        <View style={styles.modalOverlay}>
          <View style={styles.modalContent}>
            <Text style={styles.modalTitle}>Add Note</Text>
            <TextInput
              style={styles.modalTextInput}
              value={noteModal.currentNote}
              onChangeText={(value) => setNoteModal(prev => ({ ...prev, currentNote: value }))}
              placeholder="Enter your note here..."
              multiline
              numberOfLines={4}
            />
            <View style={styles.modalButtons}>
              <TouchableOpacity
                style={styles.modalButton}
                onPress={() => setNoteModal({ visible: false, itemKey: '', currentNote: '' })}
              >
                <Text style={styles.modalButtonText}>Cancel</Text>
              </TouchableOpacity>
              <TouchableOpacity style={[styles.modalButton, styles.modalButtonPrimary]} onPress={saveNote}>
                <Text style={[styles.modalButtonText, styles.modalButtonTextPrimary]}>Save</Text>
              </TouchableOpacity>
            </View>
          </View>
        </View>
      </Modal>
    </View>
  );
};

const styles = StyleSheet.create({
  container: {
    flex: 1,
    backgroundColor: '#fff',
  },
  checklistTitle: {
    fontSize: 18,
    fontWeight: '600',
    color: '#015185',
    textAlign: 'center',
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
  },
  mainContent: {
    flex: 1,
  },
  scrollContent: {
    paddingBottom: 80,
  },
  headingContainer: {
    paddingVertical: 10,
    marginHorizontal: '5%',
  },
  headingText: {
    fontSize: 20,
    fontWeight: '500',
    color: '#015185',
  },
  sectionHeader: {
    paddingHorizontal: 12,
    paddingVertical: 6,
  },
  sectionHeaderContent: {
    flexDirection: 'row',
    alignItems: 'center',
    backgroundColor: '#F2F2F2',
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 5,
  },
  sectionHeaderText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
    marginLeft: 10,
  },
  sectionHeaderTextContainer: {
    flexDirection: 'row',
    alignItems: 'center',
    flex: 1,
  },
  sectionItemCount: {
    fontSize: 14,
    color: '#666',
    marginLeft: 5,
  },
  sectionContent: {
    backgroundColor: '#fff',
  },
  centeredItem: {
    width: '90%',
    alignSelf: 'center',
    marginBottom: 15,
  },
  itemContainer: {
    paddingBottom: 15,
  },
  itemText: {
    fontSize: 16,
    fontWeight: '500',
    color: '#333',
    marginBottom: 8,
  },
  itemRow: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'space-between',
  },
  optionsContainer: {
    flexDirection: 'row',
    flexWrap: 'wrap',
  },
  optionButton: {
    paddingHorizontal: 12,
    paddingVertical: 8,
    marginHorizontal: 3,
    marginVertical: 2,
    borderRadius: 5,
    minWidth: 60,
    backgroundColor: '#ECF0F3',
  },
  selectedOptionButton: {
    backgroundColor: '#333',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  rejectButton: {
    backgroundColor: '#FF4B2B',
  },
  checkButton: {
    backgroundColor: '#FFD700', // Gold color for CHECK
  },
  replaceButton: {
    backgroundColor: '#FF4B2B', // Red for REPLACE
  },
  optionText: {
    fontSize: 14,
    color: '#333',
  },
  optionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  iconsContainer: {
    flexDirection: 'row',
  },
  iconButton: {
    padding: 10,
  },
  itemBorder: {
    width: '80%',
    alignSelf: 'center',
    borderBottomWidth: 1,
    borderBottomColor: '#ddd',
  },
  noteText: {
    fontSize: 14,
    color: '#333',
    marginTop: 8,
    marginBottom: 8,
    fontStyle: 'italic',
  },
  buttonContainer: {
    flexDirection: 'row',
    justifyContent: 'space-between',
    alignItems: 'center',
    paddingHorizontal: 15,
    paddingVertical: 15,
    backgroundColor: '#F2F2F2',
    borderTopWidth: 1,
    borderTopColor: '#ddd',
    position: 'absolute',
    bottom: 0,
    left: 0,
    right: 0,
    elevation: 5,
    shadowColor: '#000',
    shadowOffset: { width: 0, height: -2 },
    shadowOpacity: 0.1,
    shadowRadius: 2,
    paddingBottom: 30,
  },
  cancelButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingHorizontal: 15,
    paddingBottom: 30,
  },
  nextButton: {
    flexDirection: 'row',
    alignItems: 'center',
    justifyContent: 'center',
    paddingBottom: 18,
    borderRadius: 5,
  },
  cancelButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  nextButtonText: {
    color: '#015185',
    fontSize: 16,
    fontWeight: '600',
  },
  bottomicon: {
    marginTop: 2,
  },
  bottomicon1: {
    marginTop: 4,
  },
  emptySectionText: {
    fontSize: 16,
    color: '#888',
    textAlign: 'center',
    padding: 20,
  },
  modalOverlay: {
    flex: 1,
    backgroundColor: 'rgba(0, 0, 0, 0.5)',
    justifyContent: 'center',
    alignItems: 'center',
  },
  modalContent: {
    backgroundColor: '#fff',
    borderRadius: 8,
    padding: 20,
    width: width * 0.9,
    maxHeight: height * 0.6,
  },
  modalTitle: {
    fontSize: 18,
    fontWeight: 'bold',
    color: '#015185',
    marginBottom: 15,
    textAlign: 'center',
  },
  modalTextInput: {
    borderWidth: 1,
    borderColor: '#ddd',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 10,
    fontSize: 16,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 100,
    textAlignVertical: 'top',
    marginBottom: 20,
  },
  modalButtons: {
    flexDirection: 'row',
    justifyContent: 'space-between',
  },
  modalButton: {
    flex: 1,
    paddingVertical: 12,
    paddingHorizontal: 20,
    borderRadius: 6,
    marginHorizontal: 5,
    alignItems: 'center',
    borderWidth: 1,
    borderColor: '#ddd',
    backgroundColor: '#fff',
  },
  modalButtonPrimary: {
    backgroundColor: '#015185',
    borderColor: '#015185',
  },
  modalButtonText: {
    fontSize: 16,
    fontWeight: '600',
    color: '#333',
  },
  modalButtonTextPrimary: {
    color: '#fff',
  },
  inputFieldContainer: {
    marginTop: 10,
    padding: 10,
    backgroundColor: '#f8f9fa',
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#e9ecef',
  },
  inputFieldLabel: {
    fontSize: 14,
    color: '#015185',
    fontWeight: '600',
    marginBottom: 5,
    marginTop: 10,
  },
  inputField: {
    borderWidth: 1,
    borderColor: '#015185',
    borderRadius: 6,
    paddingHorizontal: 12,
    paddingVertical: 8,
    fontSize: 14,
    backgroundColor: '#fff',
    color: '#333',
    minHeight: 35,
    marginBottom: 10,
  },
  positionsGrid: {
    flexDirection: 'column',
    gap: 12,
  },
  positionItem: {
    width: '100%',
    marginBottom: 8,
  },
  positionLabel: {
    fontSize: 14,
    color: '#333',
    fontWeight: '600',
    marginBottom: 8,
    textAlign: 'center',
  },
  positionOptions: {
    flexDirection: 'row',
    justifyContent: 'space-around',
    backgroundColor: '#fff',
    borderRadius: 6,
    borderWidth: 1,
    borderColor: '#ddd',
    padding: 8,
  },
  positionOptionButton: {
    paddingHorizontal: 12,
    paddingVertical: 10,
    borderRadius: 6,
    minWidth: 70,
    alignItems: 'center',
    backgroundColor: '#ECF0F3',
    borderWidth: 1,
    borderColor: '#ddd',
  },
  positionOptionSelected: {
    backgroundColor: '#333',
    borderColor: '#333',
  },
  positionOptionText: {
    fontSize: 13,
    color: '#333',
    fontWeight: '500',
  },
  positionOptionTextSelected: {
    color: '#fff',
    fontWeight: '600',
  },
  okButton: {
    backgroundColor: '#015185',
  },
  checkButton: {
    backgroundColor: '#FFA500', // Orange for CHECK
  },
  replaceButton: {
    backgroundColor: '#FF4B2B', // Red for REPLACE
  },
});

export default HSE_122Step2;